# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['typeddfs']

package_data = \
{'': ['*']}

install_requires = \
['natsort>=7', 'pandas>=1.2,<2.0', 'tabulate>=0.8,<0.9', 'wcwidth>=0.2']

extras_require = \
{'all': ['pyarrow>=4',
         'lxml>=4',
         'openpyxl>=3',
         'pyxlsb>=1',
         'html5lib>=1',
         'beautifulsoup4>=4'],
 'excel': ['openpyxl>=3'],
 'feather': ['pyarrow>=4'],
 'html': ['html5lib>=1', 'beautifulsoup4>=4'],
 'parquet': ['pyarrow>=4'],
 'xlsb': ['pyxlsb>=1'],
 'xml': ['lxml>=4']}

setup_kwargs = {
    'name': 'typeddfs',
    'version': '0.9.0',
    'description': 'Pandas DataFrame subclasses that enforce structure and can self-organize.',
    'long_description': '# Typed DataFrames\n\n[![Version status](https://img.shields.io/pypi/status/typeddfs?label=status)](https://pypi.org/project/typeddfs)\n[![License](https://img.shields.io/badge/License-Apache%202.0-blue.svg)](https://opensource.org/licenses/Apache-2.0)\n[![Python version compatibility](https://img.shields.io/pypi/pyversions/typeddfs?label=Python)](https://pypi.org/project/typeddfs)\n[![Version on Github](https://img.shields.io/github/v/release/dmyersturnbull/typed-dfs?include_prereleases&label=GitHub)](https://github.com/dmyersturnbull/typed-dfs/releases)\n[![Version on PyPi](https://img.shields.io/pypi/v/typeddfs?label=PyPi)](https://pypi.org/project/typeddfs)\n[![Build (Actions)](https://img.shields.io/github/workflow/status/dmyersturnbull/typed-dfs/Build%20&%20test?label=Tests)](https://github.com/dmyersturnbull/typed-dfs/actions)\n[![Coverage (coveralls)](https://coveralls.io/repos/github/dmyersturnbull/typed-dfs/badge.svg?branch=main&service=github)](https://coveralls.io/github/dmyersturnbull/typed-dfs?branch=main)\n[![Documentation status](https://readthedocs.org/projects/typed-dfs/badge)](https://typed-dfs.readthedocs.io/en/stable/)\n[![Maintainability](https://api.codeclimate.com/v1/badges/6b804351b6ba5e7694af/maintainability)](https://codeclimate.com/github/dmyersturnbull/typed-dfs/maintainability)\n[![Scrutinizer Code Quality](https://scrutinizer-ci.com/g/dmyersturnbull/typed-dfs/badges/quality-score.png?b=main)](https://scrutinizer-ci.com/g/dmyersturnbull/typed-dfs/?branch=main)\n[![Created with Tyrannosaurus](https://img.shields.io/badge/Created_with-Tyrannosaurus-0000ff.svg)](https://github.com/dmyersturnbull/tyrannosaurus)\n\nPandas DataFrame subclasses that enforce structure, self-organize, and read/write correctly.\n`pip install typeddfs`\n\nStop passing `index_cols=` and `header=` on read/write.\nYour types will remember how they’re supposed to be read.\nThat means columns are used for the index, columns are given the correct types,\nand constraints are verified.\n\nAs a bonus, adds clear documentation and early failure to your code.\nAs in, `def my_func(df: MyDataFrameType)`.\nBecause your functions can’t exactly accept _any_ DataFrame.\n\n### 🎁️ Features\n\nNeed to read a tab-delimited file? `read_file("myfile.tab")`.\nFeather? Parquet? HDF5? .json.zip? XML?\nUse `read_file`. Write a file? Use `write_file`.\nAs in: `df.write_file(input("Output path?"))`.\n`read_file`/`write_file`, `read_csv`/`to_csv`, `read_json`/`to_json`, `read_xml`/`to_xml`,\netc., are now inverses.\n\nSpecific issues with Pandas functions fixed, too:\n\n- No more indices silently dropped when writing some DataFrames and formats.\n- No more columns silently renamed when reading some DataFrames and formats.\n- No more blank extra columns added when reading some DataFrames and formats.\n- You can read empty DataFrames, just like you can write them.\n- You can write an empty DataFrame to any format, not just some.\n- No more empty `.feather`/`.snappy`/`.h5` files written on error.\n- Have type-level defaults, instead of passing `encoding=`, `skip_blank_lines=`, etc., everywhere.\n\n### 🎨 Simple example\n\n```python\nfrom typeddfs import TypedDfs\n\nMyDfType = (\n    TypedDfs.typed("MyDfType")\n    .require("name", index=True)  # always keep in index\n    .require("value", dtype=float)  # require a column and type\n    .drop("_temp")  # auto-drop a column\n    .verify(lambda ddf: len(ddf) == 12)  # require exactly 12 rows\n).build()\n\ndf = MyDfType.read_file(input("filename? [.feather/.csv.gz/.tsv.xz/etc.]"))\ndf.sort_natural().write_file("myfile.feather")\n```\n\n### 📉 A matrix-style DataFrame\n\n```python\nimport numpy as np\nfrom typeddfs import TypedDfs\n\nSymmetric64 = (\n    TypedDfs.matrix("Symmetric64", doc="A symmetric float64 matri")\n    .dtype(np.float64)\n    .verify(lambda df: df.values.sum().sum() == 1.0)\n    .add_methods(product=lambda df: df.flatten().product())\n).build()\n\nmx = Symmetric64.read_file("input.tab")\nprint(mx.product())  # defined above\nif mx.is_symmetric():\n    mx = mx.triangle()  # it\'s symmetric, so we only need half\n    long = mx.drop_na().long_form()  # columns: "row", \'column", and "value"\n    long.write_file("long-form.xml")\n```\n\n### 🔍 More complex example\n\nFor a CSV like this:\n\n| key | value | note |\n| --- | ----- | ---- |\n| abc | 123   | ?    |\n\n```python\nfrom typeddfs import TypedDfs\n\n# Build me a Key-Value-Note class!\nKeyValue = (\n    TypedDfs.typed("KeyValue")  # With enforced reqs / typing\n    .require("key", dtype=str, index=True)  # automagically add to index\n    .require("value")  # required\n    .reserve("note")  # permitted but not required\n    .strict()  # disallow other columns\n).build()\n\n# This will self-organize and use "key" as the index:\ndf = KeyValue.read_csv("example.csv")\n\n# For fun, let"s write it and read it back:\ndf.to_csv("remake.csv")\ndf = KeyValue.read_csv("remake.csv")\nprint(df.index_names(), df.column_names())  # ["key"], ["value", "note"]\n\n\n# And now, we can type a function to require a KeyValue,\n# and let it raise an `InvalidDfError` (here, a `MissingColumnError`):\ndef my_special_function(df: KeyValue) -> float:\n    return KeyValue(df)["value"].sum()\n```\n\nAll of the normal DataFrame methods are available.\nUse `.untyped()` or `.vanilla()` to make a detyped copy that doesn’t enforce requirements.\nUse `.of(df)` to convert a DataFrame to your type.\n\n### 💔 Limitations\n\n- Multi-level columns are not yet supported.\n- Columns and index levels cannot share names.\n- Duplicate column names are not supported. (These are strange anyway.)\n- A typed DF cannot have columns "level_0", "index", or "Unnamed: 0".\n- `inplace` is forbidden in some functions; avoid it or use `.vanilla()`.\n\n### 🔌 Serialization support\n\nLike Pandas, TypedDfs can read and write to various formats.\nIt provides the methods `read_file` and `write_file`, which guess the format from the\nfilename extension. For example, `df.write_file("myfile.snappy)` writes Parquet files,\nand `df.write_file("myfile.tab.gz")` writes a gzipped, tab-delimited file.\nThe `read_file` method works the same way: `MyDf.read_file("myfile.feather")` will\nread an Apache Arrow Feather file, and `MyDf.read_file("myfile.json.gzip")`reads\na gzipped JSON file. You can pass keyword arguments to those functions.\n\nSerialization is provided through Pandas, and some formats require additional packages.\nPandas does not specify compatible versions, so typed-dfs specifies\n[extras](https://python-poetry.org/docs/pyproject/#extras) are provided in typed-dfs\nto ensure that those packages are installed with compatible versions.\n\n- To install with [Feather](https://arrow.apache.org/docs/python/feather.html) support,\n  use `pip install typeddfs[feather]`.\n- To install with support for all formats,\n  use `pip install typeddfs[feather,fwf,xml,xlsx,xls,ods,xlsb` and `pip install tables`.\n\nFeather offers massively better performance over CSV, gzipped CSV, and HDF5\nin read speed, write speed, memory overhead, and compression ratios.\nParquet typically results in smaller file sizes than Feather at some cost in speed.\nFeather is the preferred format for most cases.\n\n**⚠ Note:** The `hdf5` extra is currently disabled.\n\n| format   | packages                     | extra     | sanity | speed | file sizes |\n| -------- | ---------------------------- | --------- | ------ | ----- | ---------- |\n| Feather  | `pyarrow`                    | `feather` | ++     | ++++  | +++        |\n| Parquet  | `pyarrow` or `fastparquet` † | `parquet` | ++     | +++   | ++++       |\n| csv/tsv  | none                         | none      | ++     | −−    | −−         |\n| flexwf ‡ | none                         | none      | ++     | −−    | −−         |\n| .fwf     | none                         | none      | +      | −−    | −−         |\n| json     | none                         | none      | −−     | −−−   | −−−        |\n| xml      | `lxml`                       | `xml`     | +      | −−−   | −−−        |\n| .npy     | none                         | none      | ++     | +     | +++        |\n| .npz     | none                         | none      | ++     | +     | +++        |\n| .html    | `html5lib,beautifulsoup4`    | `html`    | −−     | −−−   | −−−        |\n| pickle   | none                         | none      | −− ️   | −     | −          |\n| XLSX     | `openpyxl,defusedxml`        | `excel`   | +      | −−    | +          |\n| ODS      | `openpyxl,defusedxml`        | `excel`   | +      | −−    | +          |\n| XLS      | `openpyxl,defusedxml`        | `excel`   | −−     | −−    | +          |\n| XLSB     | `pyxlsb`                     | `xlsb`    | −−     | −−    | ++         |\n| HDF5     | `tables`                     | `hdf5`    | −−     | −     | ++         |\n\n**Notes:**\n\n- † `fastparquet` can be used instead. It is slower but much smaller.\n- ‡ `.flexwf` is fixed-width with optional delimiters.\n- JSON has inconsistent handling of `None`. ([orjson](https://github.com/ijl/orjson) is more consistent).\n- XML requires Pandas 1.3+.\n- .npy and .npz only serialize numpy objects.\n- .html is not supported in `read_file` and `write_file`.\n- Pickle is insecure and not recommended.\n- Pandas supports odfpy for ODS and xlrd for XLS. In fact, it prefers those.\n  However, they are very buggy; openpyxl is much better.\n- XLSM, XLTX, XLTM, XLS, and XLSB files can contain macros, which Microsoft Excel will ingest.\n- XLS is a deprecated format.\n- XLSB is not fully supported in Pandas.\n- HDF may not work on all platforms yet due to a\n  [tables issue](https://github.com/PyTables/PyTables/issues/854).\n\n### 📝 Extra notes\n\nDependencies in the extras are only restricted to minimum version numbers;\nlibraries that use them can set their own version ranges.\nFor example, typed-dfs only requires pyarrow >= 0.4, but Pandas can further restrict it.\n[natsort](https://github.com/SethMMorton/natsort) is also only assigned a minimum version number;\nthis is because it receives frequent major version bumps.\nThis means that the result of typed-df’s `sort_natural` could change.\nTo fix this, pin natsort to a specific major version;\ne.g. `natsort = "^7"` with [Poetry](https://python-poetry.org/) or `natsort>=7,<8` with pip.\n\n### 🍁 Contributing\n\nTyped-Dfs is licensed under the [Apache License, version 2.0](https://www.apache.org/licenses/LICENSE-2.0).\n[New issues](https://github.com/dmyersturnbull/typed-dfs/issues) and pull requests are welcome.\nPlease refer to the [contributing guide](https://github.com/dmyersturnbull/typed-dfs/blob/main/CONTRIBUTING.md).\nGenerated with [Tyrannosaurus](https://github.com/dmyersturnbull/tyrannosaurus).\n',
    'author': 'Douglas Myers-Turnbull',
    'author_email': None,
    'maintainer': 'dmyersturnbull',
    'maintainer_email': None,
    'url': 'https://github.com/dmyersturnbull/typed-dfs',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
