# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/03_reward_model.ipynb.

# %% auto 0
__all__ = ['RewardModel', 'RewardLoss', 'LitRewardModel']

# %% ../nbs/03_reward_model.ipynb 3
from typing import Callable, Union

import torch
from torch import nn
import torch.nn.functional as F
from torch import optim
import pytorch_lightning as pl 
from transformers import AutoModel, AutoTokenizer
from einops import rearrange

from .utils import load_yaml

# %% ../nbs/03_reward_model.ipynb 6
class RewardModel(nn.Module):
    def __init__(self, checkpoint: str, dropout: float = 0.1):
        super().__init__()
        self.tokenizer = AutoTokenizer.from_pretrained(checkpoint)
        self.model = AutoModel.from_pretrained(checkpoint)
        
        config = self.model.config
        n_embed = config.n_embd
        
        # custom head
        self.dropout = nn.Dropout(dropout)
        self.reward_head = nn.Linear(n_embed, 1)
        
    def forward(self, input_ids: torch.Tensor, attention_mask: torch.Tensor):
        last_hidden_state = self.model(
            input_ids=input_ids,
            attention_mask=attention_mask,
        ).last_hidden_state
        
        output = self.dropout(last_hidden_state)
        output = self.reward_head(output)
                
        # output = rearrange(output, 'b 1 t 1 -> b t')
        # for eacb item in the batch
        # choose the hidden state of the last token as a reward!
        reward_scalar = output[:, -1, 0]
        
        return reward_scalar

# %% ../nbs/03_reward_model.ipynb 10
class RewardLoss(nn.Module):
    def forward(self, chosen_rewards: torch.Tensor, rejected_rewards: torch.Tensor):
        assert len(chosen_rewards) == len(rejected_rewards)
        batch_size = len(chosen_rewards)
        
        # maps the difference between the rewards to a probability
        probs = torch.sigmoid(chosen_rewards - rejected_rewards)
        return -probs.mean() / batch_size

# %% ../nbs/03_reward_model.ipynb 12
class LitRewardModel(pl.LightningModule):
    def __init__(
        self, model: Callable, loss_func: Callable,
        lr: Union[int, float] = 1e-3
    ):
        super().__init__()
        self.model = model
        self.loss_func = loss_func
        self.lr = lr
    
    def training_step(self, batch, batch_idx: int):
        chosen_input_ids, chosen_attention_mask,\
        rejected_input_ids, rejected_attention_mask = batch
        
        chosen_rewards = self.model(chosen_input_ids, chosen_attention_mask)
        rejected_rewards = self.model(rejected_input_ids, rejected_attention_mask)
        
        loss = self.loss_func(chosen_rewards, rejected_rewards)
        
        return loss
    
    def configure_optimizers(self):
        optimizer = optim.Adam(self.model.parameters(), lr=self.lr)
        return optimizer
