
import random
import math

import numpy as np

from optproblems.base import Individual, BoundConstraintsChecker, TestProblem
from optproblems.continuous import double_sum, EllipsoidFunction, sphere


class F1(TestProblem):
    """Shifted sphere."""

    bias = -450.0

    offsets = [-39.3119, 58.8999, -46.3224, -74.6515, -16.7997, -80.5441,
               -10.5935, 24.9694, 89.8384, 9.1119, -10.7443, -27.8558, -12.5806,
               7.593, 74.8127, 68.4959, -53.4293, 78.8544, -68.5957, 63.7432,
               31.347, -37.5016, 33.8929, -88.8045, -78.7719, -66.4944, 44.1972,
               18.3836, 26.5212, 84.4723, 39.1769, -61.4863, -25.6038, -81.1829,
               58.6958, -30.8386, -72.6725, 89.9257, -15.1934, -4.3337, 5.343,
               10.5603, -77.7268, 52.0859, 40.3944, 88.3328, -55.8306, 1.3181,
               36.025, -69.9271, -8.6279, -56.8944, 85.1296, 17.6736, 6.1529,
               -17.6957, -58.9537, 30.3564, 15.9207, -18.0082, 80.6411,
               -42.3912, 76.2776, -50.1652, -73.5736, 28.3369, -57.9905,
               -22.7327, 52.0269, 39.2599, 10.8679, 77.8207, 66.0395, -50.0667,
               55.7063, 73.7141, 38.5296, -56.7865, -89.6477, 37.9576, 29.472,
               -35.4641, -31.7868, 77.3235, 54.7906, -48.2794, 74.2714, 72.6103,
               62.964, -14.1446, 20.4923, 46.5897, -83.6021, -46.4809, 83.7373,
               -79.6611, 24.3479, -17.2303, 72.3404, -36.4022]


    def __init__(self, num_variables, phenome_preprocessor=None, **kwargs):
        self.is_deterministic = True
        self.do_maximize = False
        self.num_variables = num_variables
        self.min_bounds = [-100.0] * num_variables
        self.max_bounds = [100.0] * num_variables
        bounds = (self.min_bounds, self.max_bounds)
        preprocessor = BoundConstraintsChecker(bounds, phenome_preprocessor)
        TestProblem.__init__(self, self.objective_function,
                             phenome_preprocessor=preprocessor,
                             **kwargs)


    def objective_function(self, phenome):
        phenome = [phene - offset for offset, phene in zip(self.offsets, phenome)]
        assert len(phenome) == self.num_variables
        obj_value = sphere(phenome) + self.bias
        return obj_value


    def get_optimal_solutions(self, max_number=None):
        return [Individual(self.offsets[:self.num_variables])]



class F2(TestProblem):
    """Shifted double-sum."""

    bias = -450.0

    offsets = [35.6267, -82.9123, -10.6423, -83.5815, 83.1552, 47.048, -89.4359,
               -27.4219, 76.1448, -39.0595, 48.8857, -3.9828, -71.9243, 64.1947,
               -47.7338, -5.9896, -26.2828, -59.1811, 14.6028, -85.478,
               -50.4901, 0.924, 32.3978, 30.2388, -85.0949, 60.1197, -36.2183,
               -8.5883, -5.1971, 81.5531, -23.4316, -25.3505, -41.2485, 8.8018,
               -24.2222, -87.9807, 78.0473, -48.0528, 14.0177, -36.6405,
               12.2168, 18.1449, -64.5647, -84.8493, -76.6088, -1.7042,
               -36.0761, 37.0336, 18.4431, -64.359, -39.3692, -17.714, 30.1985,
               -18.5483, 9.6866, 82.6009, -45.5256, 5.1443, 74.204, 66.8103,
               -63.4704, 13.0329, -5.6878, 29.5271, -0.4353, -26.1652, -6.6847,
               -80.2291, -29.5815, 82.0422, 77.177, -11.277, 32.0759, -2.6858,
               81.5096, 64.077, -26.1294, -84.782, -62.8768, -37.6355, 76.8916,
               53.417, -25.3311, -38.0702, -84.1738, -11.2246, -83.4619,
               -17.5508, -36.5285, 89.5528, 25.8794, 68.6252, 55.7968, -29.5975,
               -58.0976, 65.7413, -8.8703, -5.3281, 74.0661, 4.0338]

    def __init__(self, num_variables, phenome_preprocessor=None, **kwargs):
        self.is_deterministic = True
        self.do_maximize = False
        self.num_variables = num_variables
        self.min_bounds = [-100.0] * num_variables
        self.max_bounds = [100.0] * num_variables
        bounds = (self.min_bounds, self.max_bounds)
        preprocessor = BoundConstraintsChecker(bounds, phenome_preprocessor)
        TestProblem.__init__(self, self.objective_function,
                             phenome_preprocessor=preprocessor,
                             **kwargs)


    def objective_function(self, phenome):
        phenome = [phene - offset for offset, phene in zip(self.offsets, phenome)]
        assert len(phenome) == self.num_variables
        obj_value = double_sum(phenome) + self.bias
        return obj_value


    def get_optimal_solutions(self, max_number=None):
        return [Individual(self.offsets[:self.num_variables])]



class F3(TestProblem):
    """Shifted rotated high conditioned elliptic function."""

    bias = -450.0

    offsets = [-32.2013, 64.9776, -38.3, -23.2582, -54.0088, 86.6286, -6.3009,
               -49.3644, 5.3499, 52.2418, -13.3643, 73.1263, -8.5691, -20.4915,
               -60.1487, 16.0884, -78.3319, 70.0387, -6.8521, -64.797, 65.4005,
               -26.0233, -33.8757, 51.5893, 27.6427, -69.4485, 25.5123,
               -59.0782, -66.5481, -51.2733, -81.776, -71.6572, 37.081,
               -63.4248, -64.7785, 31.5299, 18.5387, 9.8342, -0.6037, 1.7346,
               70.1605, -82.0391, -42.7368, -83.593, -85.0255, 41.1773, 4.1649,
               -13.4505, -0.31, -38.7944, 71.2702, 65.532, 8.7753, -55.4691,
               -20.6252, 22.2901, 13.6798, 65.6745, 75.8418, 27.8926, -15.0616,
               -17.3036, 57.9346, -86.6326, 65.0596, 47.3884, 29.166, 65.5435,
               3.4643, -39.814, 18.2261, 77.0446, 62.1882, -11.4, -10.6218,
               70.1276, -40.8673, -24.4451, 52.1398, -10.5136, 29.2399, 2.1705,
               44.0863, 81.7943, 80.0466, 88.3266, 16.6098, -50.2573, -71.6993,
               71.5368, 61.4273, -3.6739, 77.9428, -22.3294, 64.7634, -74.2823,
               14.1899, 37.8473, -77.7129, 28.9959]


    matrix2D = np.array([[-0.9838884637416794, 0.17878336308515416],
                         [0.1787833630851541, 0.9838884637416795]])

    matrix10D = np.array([[0.17830682721057345, 0.05578633058716659, 0.4759190557666973, 0.24551129863391566, 0.31998625926387086, 0.3210200144836385, 0.027787561319902176, 0.2666400104677562, 0.41568009651337917, -0.47771934552669726],
                          [0.6351636285946867, 0.05009142383664624, 0.20110601384121973, -0.6807688241663351, -0.049934546553907944, -0.04639942342458296, -0.19460194646748039, 0.18961539926194687, -0.019416259626804547, 0.10639981029473855],
                          [0.32762147366023187, 0.36016598714114556, -0.2363565509404495, -0.01856685401744485, -0.24479096747593634, 0.44818973341886903, 0.5351863573361957, -0.3120692519053052, -0.13863719921728737, -0.20713981146209595],
                          [-0.06478321058798428, -0.49424101683695937, 0.13101175297435969, 0.03161517193119454, -0.1750610791487186, 0.6890803934491838, 0.010544234469094992, 0.21948984793273507, -0.16468539805844565, 0.3904855051851341],
                          [-0.2764804478537137, 0.1138311450612022, -0.30818401502810994, -0.3595940710443874, 0.2644625803470219, 0.0286167883791575, 0.47528027904995646, 0.4099399404977017, 0.4113104336891543, 0.2289934518888688],
                          [0.15454249061641606, 0.54899186274158, -0.1838202994179226, 0.3394446190390916, 0.285961887742557, 0.12833167642713417, -0.25495080172376317, 0.394607523020371, -0.3452464027000741, 0.2959031832336851],
                          [-0.05190797769001451, -0.1445075780970033, -0.46086919626114314, -0.05368796481836808, -0.36317793499109247, 0.027439997038558633, -0.21422629652542946, 0.5054514889308478, -0.09806471701908984, -0.5634699101856451],
                          [0.5014298935446065, -0.5313365904845752, -0.37294385871521135, 0.2337086643138151, 0.4432753766248853, -0.16972740381143742, 0.2036414896333169, -0.023717523924336927, -0.07180545586295492, -0.007333217845033976],
                          [0.10441248047680891, 0.04306422614936954, -0.41675972625940993, 0.016522876074361707, 0.0017437281849141879, 0.2959494487903076, -0.5119748773936874, -0.3267981976235789, 0.5825310659093351, 0.13204141339826148],
                          [-0.2964590765763169, -0.031303011496605505, -0.0780091540821166, -0.41548534874482024, 0.5695940357244347, 0.2909519840034815, -0.18560717510075503, -0.24653488847859115, -0.3714902508547979, -0.30015617693118707]])

    matrix30D = np.array([[0.10164957261764876, 0.28907203592317565, -0.28676502138024335, 0.1218656189849876, 0.005564508600170918, -0.22303152232011716, -0.1947278015668383, -0.1187111843571265, 0.13820926976639555, -0.03782856975350359, 0.3145503195435482, 0.04487983403959616, 0.12570115974584642, 0.21140455372781417, -0.39649477833532365, 0.21679987185764366, 0.1734433978156879, 0.05205633230714463, -0.2286556716275806, -0.1724263688476806, -0.018765349832665177, -0.2450977619662135, -0.042909543123452715, 0.08520392393757856, -0.0028073675525156255, -0.2694346492544198, 0.204261606401297, -0.07730857500700167, 0.12110828066102342, -0.04171193079104835],
                          [-0.06356688128738831, -0.06366532526403493, -0.09861259546241607, -0.10014364585581643, -0.14493349141275022, -0.041766780198875544, 0.36929813921174637, 0.11774688957741393, 0.19505102110413464, 0.10671307555259535, 0.1088859352715337, -0.22768051668524378, 0.019400190759083166, -0.1133194666552539, -0.08575427185145576, 0.1868394788471477, -0.2154077357793683, -0.27527136459375956, -0.24996069817518474, -0.043423348482698, 0.21024671926158417, 0.0006398686130767, 0.39221941156458107, 0.11354160173324693, -0.376237262492889, 0.011874606989760618, -0.10259435399326221, 0.17356454384129819, 0.10525158160810666, -0.1991314037147829],
                          [0.1716067566418046, -0.039406430050305384, 0.11335061164399883, -0.4007898331836075, 0.0025266241235494162, -0.00865890279827147, 0.1396817679586794, 0.05395086229230797, 0.453574017608888, 0.27416554845722513, -0.20120102931824305, 0.10786018168385161, 0.11215507852777834, -0.08522175632536345, -0.036565694622393584, 0.07681600872811145, 0.08386159939400979, 0.25854807960750176, -0.010666415655920922, -0.05266368870433495, 0.046050815312167294, -0.2874335398245715, -0.022690225101958856, -0.036380368816751676, 0.28497320149377114, 0.20056353219635795, 0.27387241616454305, 0.20574639118739677, -0.07649882043529228, -0.07038372924617202],
                          [0.2775954381088784, 0.03720353496167518, 0.01886152913696575, -0.2621111205488722, 0.49253846010095254, 0.10261612034343869, 0.22224353035271752, 0.2319628589073994, -0.1643220295757958, -0.2509741577638973, 0.30930716388123813, -0.13324284074970702, -0.07505331839007812, 0.04315272602362225, -0.20505081318573026, -0.007245074369464138, -0.03476662397080396, -0.1603320333170557, 0.08740311868819384, -0.1010145017960481, -0.03696687551134546, -0.015208663498307698, 0.0032052127606547872, -0.14309051818892518, 0.2689989534380106, -0.021838249567058324, -0.1580018044174444, 0.009339956251623809, -0.19639985053639275, -0.1996604797546255],
                          [0.09871960697666698, -0.2267964611200932, -0.25254146449215054, 0.05280126298394851, -0.1933448392228727, 0.21559967509050282, 0.29372216194092204, -0.1611901846261491, 0.15819279073280304, 0.08625079760905448, 0.11385968059614064, -0.27752413275486176, 0.18174263574348862, 0.2523333098544508, 0.038988626915871225, -0.08929704696802197, 0.16819065608949466, 0.27870318713320774, 0.23598261494637288, 0.2449714908714476, -0.1932522476640285, 0.018121744120277616, 0.002403366684444846, -0.14803728856015086, 0.06941494521425752, -0.27338951414303636, -0.23466893106435363, 0.003976031572932356, 0.13804596199533947, -0.11192640635625034],
                          [0.013549989414998383, -0.15897918291945926, -0.2576996867333399, -0.1784741533120154, 0.10885546020927407, -0.02183302426781088, 0.011850247261564915, -0.4127826328926849, 0.020744877014688398, -0.21183337363018903, -0.09655020970752953, 0.45390162748620316, -0.24632751256734814, -0.02967255646941305, 0.025745054803161548, 0.008597760929578901, -0.4171224385941253, 0.12446362871527113, -0.061058126254324656, 0.14976678374594685, 0.03424716176167472, 0.024894537273179276, 0.10477176440902719, -0.14765843280100496, -0.026208389465230564, -0.21326752653532252, 0.08590846352226907, -0.021833389903042136, -0.008177048146396568, -0.25723522049513087],
                          [0.18507347206915797, 0.15758035573018975, 0.032477251416465375, 0.4513071535613241, 0.19946842670449538, 0.37564752754982783, 0.1214312485110197, -0.19712551151710206, -0.13251075393884892, -0.14209059076039393, -0.2882611491007309, -0.18497202703143684, 0.1268081208097443, -0.18806334431661748, 0.03705938920470933, 0.08334864078395424, -0.027389917097839694, -0.023383972318880147, -0.10674018659465959, 0.07779746056373545, -0.01173616083176247, -0.22937486599785303, 0.0688547189862504, 0.0628942659530501, 0.08653140694309171, 0.16356035494155557, 0.23597247253073309, -0.0028318894910640846, 0.21650296150916884, -0.26430986667164025],
                          [0.13594932566906506, 0.3074191368514748, -0.15350247009863083, -0.14464531358364677, 0.04569261220518883, -0.05189948196742329, -0.05023133376759738, -0.26647001189789654, -0.00302927455868819, 0.1414678206033217, -0.14946656543309728, -0.13255052878573606, 0.254605021271442, 0.24464274994716803, 0.05335814642380131, -0.4806845568085278, -0.10429863563017264, -0.044814506124222976, -0.19163684292503938, -0.04257621258985102, 0.2755575497651694, 0.2672161378422087, 0.06221164286156204, 0.2569063146581813, 0.1622822489440353, 0.08913301921926425, -0.1249001362000181, -0.038948084012320354, -0.1427961021242354, -0.07269754815671875],
                          [0.03906726157411894, -0.3037293537719967, -0.17810842626246476, 0.04551997284638125, -0.0475207899802671, 0.17515992233757294, -0.011875763424877415, 0.10366344641108381, -0.012206184268571521, 0.2808511082903948, 0.31488849823259546, -0.008302221147605102, -0.09358833954572465, -0.2773088560924207, 0.0722588515120634, -0.02523184139648953, -0.23915419607575888, 0.01580488647934725, -0.3394620056100733, 0.03470271487542195, -0.11175215790045064, -0.010080015898204227, -0.24868686049393682, 0.36285510582611696, 0.306165344066221, 0.017401570953453942, -0.0029360361797943135, -0.28626680055921755, 0.028433878851802184, 0.045232477160892424],
                          [-0.26023600115933015, -0.21499575729520948, 0.1400048219411101, -0.09209391350394246, 0.19762492844673585, -0.1160680960363133, -0.1434911020093422, -0.24003309876225815, -0.03076489776643003, -0.15151291610765563, 0.1348069540937481, -0.10061335421716092, 0.15837154489540492, 0.019952736779194315, -0.18012745424870008, 0.01279656937980069, 0.08468879502374074, 0.23027670832848535, -0.3435080486516305, 0.23464834876862828, -0.06202851577285202, -0.14119879636686303, 0.23594051722401366, -0.13255055751508624, 0.04088273372406207, 0.4212886268641687, -0.2766448363731282, -0.07165403009795077, 0.08322672312454074, 0.1375581333515522],
                          [-0.11453132803044509, 0.0005016103467944411, 0.06333082983936206, -0.190127227640041, 0.03199469160152175, 0.037475410260706395, -0.10651833891466217, 0.2586572448986541, -0.25243252362585583, 0.1818138537310842, 0.05982141932356695, -0.03827781356067774, 0.3251954662837682, 0.11502545808617184, 0.10909906763059674, -0.06221718876058416, -0.22015238873256468, 0.11693860325273177, 0.07170143938917405, 0.06353150239828748, 0.3285670598500912, -0.25477116399848937, -0.1137288063660733, -0.2803643931188687, -0.19616703427550175, -0.06479602378440377, 0.14803341992172783, -0.39103954522623335, 0.18421575628361483, -0.20567728172193261],
                          [-0.30659016189985566, 0.13312718600651358, -0.08903401020698419, -0.26771359515466603, 0.30206030171609105, -0.07760238932793143, -0.015530031921647092, -0.1053433173399142, -0.12306870860450268, 0.03887631912384898, -0.057248130601958505, -0.07901605313688591, 0.0668144887425574, -0.10461884702108182, 0.41576604583960663, 0.35093232220868503, 0.2746140185508066, 0.006127619227398179, -0.07501588089849591, 0.09983092779737897, 0.02525310705863986, -0.005485543132883931, -0.2397536368923103, 0.245111971309485, 0.017417557421154497, -0.22523540216036722, -0.16543629204672444, 0.19938153032978617, 0.07670402454959387, -0.1707781703667471],
                          [0.006689533991120046, -0.030659953050316126, 0.10272965845804866, -0.06721648766793795, 0.14210487621784515, 0.1355886412891005, -0.011894510681984643, -0.12056119870765801, -0.0020096507361536207, 0.25051991430019005, 0.053355200123323884, 0.10019225153719218, 0.4015622771308912, 0.17081089509279365, -2.6119138618403325e-05, 0.2672923120943625, -0.13258462573464175, -0.3126419986029899, -0.10852024420513201, 0.0854968851463823, -0.38834820528282776, 0.41292322461932435, 0.0063405624124030216, -0.2106430730674752, -0.02910727101394996, 0.009332353221005443, 0.28247031266947714, 0.03171318701282695, -0.03942522499589698, 0.09869812010307308],
                          [-0.3260660043651632, -0.06853195731760066, 0.04554446822403763, -0.11555168208661482, -0.17825174102936656, 0.05871209551117389, -0.11765576068605013, -0.19746247788481788, -0.16204447647757614, 0.079117267317619, 0.2750498617494267, -0.1581581983694536, -0.13278630206979702, -0.15796297078398455, -0.06808686684726778, -0.3712864017036526, 0.24920682322750803, -0.22500859368947737, 0.036929236094847336, 0.07940052830938431, -0.05044498730149946, -0.08006463613957981, 0.1047415815056818, -0.08204999381930998, 0.0974309100049117, -0.044897124011706036, 0.41381288616302425, 0.1961680257619389, -0.11894214294232994, -0.29121555285078454],
                          [-0.03374360023766475, -0.039881061790095325, -0.09792181375516726, 0.06863479065354666, 0.3395842706518788, 0.05714479700355888, -0.0911782130051627, -0.20044401863158703, 0.40432500907917257, 0.18809302975576175, 0.12077786335462118, 0.038280318129714316, -0.17521362545123614, 0.12450303785453637, 0.2856748653621252, -0.05573720949136751, 0.11466442561705872, -0.35885530134843846, 0.28373847564034077, -0.08629316234503127, 0.0031540189437001685, -0.24461095883440617, 0.16002934713033515, 0.015339169328660411, -0.1075616664377728, 0.1525273789423709, -0.04889714347341972, -0.336329439262963, 0.018316757659536213, 0.06182989319898719],
                          [0.14240466715456365, 0.15321457769660388, 0.37285641983131007, -0.1570967543312308, 0.21981357096489298, 0.07514334794736817, -0.2284877619182089, 0.01714165231048845, 0.13215246272666265, 0.08292559321432781, 0.06504573770111124, -0.21087097725917006, -0.19185388072821907, -0.06359297493233657, -0.11947725369276378, -0.10561979878742121, -0.18288882151273664, 0.15135418938215822, 0.10826355082329937, 0.32237181825723915, -0.04194545141862241, 0.07307412581521902, 0.19921301547571402, 0.24798156337561747, -0.08376938149158719, -0.3387612275872482, 0.1023734890135393, 0.05908728748477116, 0.3016341077486798, 0.19936630448367115],
                          [-0.020694916791897004, -0.07944697259110753, 0.08048592896466096, 0.25362252509829203, 0.09791435814140367, -0.19941627515551485, -0.06248136911318807, 0.09004818729689022, 0.047491466802618267, 0.20719020857347664, -0.08348172160319763, 0.03686344936894379, 0.031196842419712864, -0.2682155810599555, -0.021514495727349932, 0.15227077446092127, 0.23222531207789465, 0.10479941207418143, -0.0001491910378031755, -0.014117832287631294, 0.23854539677645265, 0.3242745514048472, 0.41029197491955305, -0.17292072214480383, 0.3222683108411213, -0.2556994832218574, -0.03781754758268757, -0.2707052167722046, -0.10379129395551044, -0.17573957645103017],
                          [-0.18605460547463779, 0.25057635463170563, 0.006761729421830952, -0.03002742798534898, -0.0906983516624831, -0.01086111465067783, 0.3466793860849267, -0.07682037436277776, 0.051795188478884625, -0.14409782286993017, 0.12304688755604394, -0.23778574893246746, -0.301659094813294, 0.21241680363756868, 0.0849027994478177, 0.26136020343724486, -0.027256101625377265, 0.15920276021915677, -0.005789509652101111, 0.22819303431099655, 0.20150668235139793, 0.18442046276866697, -0.06757870156923138, -0.007779634101181586, 0.10202752769307531, 0.17086491478664886, 0.3485715522838689, -0.29048194696238494, -0.17725434398512266, 0.1615758013298807],
                          [-0.33330180733706594, 0.0939206948762525, -0.10888451131033554, 0.2821805067660851, 0.3374348277643299, -0.27984333600756145, 0.24412147571491902, 0.17486113420742802, 0.1214913683178583, 0.2770188399139099, -0.22394849868648964, -0.028510431226916117, -0.11802827287860457, 0.023800046077563572, -0.2721862063209091, -0.2567039239372421, -0.1434977653278118, -0.03148335113170741, -0.06739983333460163, 0.22633940659978288, -0.14400590013832085, -0.06449333263849208, -0.2700692189111661, -0.1389346106574317, -0.02190877868697272, -0.05551564376087152, -0.024198146629657243, 0.09122949398093559, -0.03640203350558166, -0.03206738744403439],
                          [-0.033260249656886946, -0.03713528807899055, 0.18905207148017175, -0.03286215638527545, -0.15683447682801296, 0.25163980730298163, 0.03809467570092754, -0.013505876543970436, 0.12582566384208957, -0.0840280956576722, -0.10034574705428417, 0.21028987367517912, -0.05758921228458939, 0.14808233845212124, -0.1494276862199406, 0.015515074946079845, 0.15981333614687732, -0.44844163190452635, -0.21576106481219698, 0.2574727206149297, 0.37915675510110086, -0.07169174320701732, -0.19383285347918833, -0.14222720000830874, 0.2686276537393518, -0.18293549650767138, -0.16961512048474714, 0.009633724619305787, 0.21851274403303259, 0.13312409337606884],
                          [0.3755078366572995, -0.20161612858498879, 0.22590527596234802, 0.0867452504671401, 0.013850166150773292, -0.21681091111820583, 0.0384516188556211, -0.10986826346319642, 0.04407898310635957, -0.05263268535654733, 0.05030574186504403, 0.023822550637571747, 0.13579980757140755, -0.09089752597225279, -0.014883710422609884, -0.00783958875197413, 0.18090769592980158, -0.07545111855007587, -0.05672588501540177, 0.4051081312093054, 0.08789836277095082, -0.0736520115889272, -0.2252182970327358, 0.1635857856616217, -0.36051966406400604, -0.05309862539264366, 0.03346527066261012, -0.11910223620024862, -0.448521018902321, -0.13632591005447317],
                          [-0.025110797493557085, -0.28733379317079755, -0.13434839908223567, -0.03572457045497909, 0.1322172750750852, -0.21795248281946067, -0.03940086744230657, 0.024793438341622155, 0.32391621194604914, -0.4395111080750659, -0.02016420836888503, -0.2308378490143518, 0.2205180912573041, -0.22497419475504515, 0.0239884912323955, -0.15280905866121272, 0.0011287165960036211, -0.08657409249422608, 0.08050672599579121, -0.11988786632273904, 0.1419587543238365, 0.2838814329357093, -0.25556836173554304, -0.042281551362847156, 0.011271735359420557, 0.01824747043604259, 0.25134409934105456, -0.02270791486558549, 0.30181572712174365, 0.06278649622909739],
                          [0.0815837455518731, 0.10775112443770757, -0.3314584065739228, 0.20099897680649959, 0.04363765613024238, 0.09739816376380893, -0.3161684882592524, 0.09067123378413507, 0.08226068819300537, 0.0444647412793935, 0.24250263043499515, -0.07574396541395291, 0.09442225207667716, -0.14516037821599304, 0.15188640305850426, 0.10483638283446395, -0.1970794461725917, 0.019448535803295412, 0.09411734035171354, 0.2868321910732192, 0.3037207491323617, -0.05554637441643755, 0.013649751163702376, -0.2436180982993806, 0.06852955823481943, 0.08331120191434356, -0.02323413916270817, 0.42197164577658186, -0.23211003632725546, 0.2126713386215258],
                          [-0.08411349716480196, 0.2671738104200829, -0.32942793492683603, -0.22935439089071982, 0.022102650718827326, 0.4116021160804735, -0.041438272573231454, 0.23057935150834066, 0.14100790046220504, -0.11635412696170282, -0.1223182623802988, 0.15192671828995424, 0.0494117079395017, -0.31375226619414054, -0.16242170003691905, -0.12926642975958613, 0.29312047759968773, 0.06509623895781569, -0.1006076157365267, 0.16610591592807022, -0.12756791269583487, 0.13242178052066697, 0.06788380788555118, -0.03480220337328866, -0.27412193609556623, 0.02409964652358143, -0.01391780977515977, -0.24694317827549023, -0.11064527956806586, 0.05959117490215722],
                          [0.07804624697550466, 0.3842475033746546, 0.05837658938969228, -0.04209996901270768, -0.16476055047659377, -0.2554931971495647, 0.2511831390962274, -0.2067569559471814, -0.061874033881479766, 0.07591049244840525, 0.2744561257246464, 0.24240850402356406, 0.15810527439614874, -0.406159213924635, -0.052019555916896505, -0.021263245792077553, -0.0523481734866822, -0.09057086765340154, 0.3107763986385857, 0.1584450254972078, -0.051654891412967766, 0.030184317955887333, -0.07816291345334235, -0.03511262115683164, 0.09556458922803548, 0.19593754388065976, -0.16686639297545844, -0.0769273143220218, 0.28483342779841636, -0.007094215849415299],
                          [-0.22478593316709744, -0.10690753323625946, -0.14034481301019194, -0.10766894140971613, -0.07000181268975715, -0.05209842258237794, 0.039134752347044895, -0.03808097030304609, -0.11809915343001645, -0.13532422253864415, -0.2542695029461662, -0.09427426205798985, 0.3026142977152861, -0.07266164938985886, -0.2202473626090201, 0.1253057089844455, -0.18954925252662602, -0.20028344319988667, 0.3256438745811935, 0.09610257663324431, -0.04326174625410054, -0.28343383100775704, 0.19018992705637697, 0.2946258652086023, 0.21652746759735045, -0.18068242850777214, 0.033090098143782956, -0.13829217905437288, -0.2539045292194694, 0.26185486656293894],
                          [-0.06365124443059178, -0.12075946721838979, 0.03678448248091081, -0.029902582436935995, 0.1378510257617648, 0.27044524288469585, 0.027655093015208545, -0.40475575314803686, -0.0971153979333453, 0.27508335573136006, -0.010475237362260963, -0.11715666011176244, -0.0651778605148483, -0.18199678584566278, -0.40911206402015826, 0.133071143212234, 0.012679223495304051, 0.1260038260375165, 0.17741010018177125, -0.3120296114922535, 0.30753458154358654, 0.12053521161179297, -0.26530282198230454, -0.046099076232067604, -0.20628115953462534, -0.00031984675745712316, -0.08294173897394426, 0.02409072068473203, -0.08264078804184766, 0.13289415927179837],
                          [-0.09258002020007425, 0.24959117063361386, 0.27461608686658096, 0.004029815199549951, -0.13671282070528148, 0.04244901895828142, -0.008529854215737903, -0.17603562754195543, 0.26619379098571355, -0.1358914415275283, 0.01772992480330396, -0.24075271123044695, 0.06999100337946608, -0.23917148064949065, 0.19747604518219758, -0.059404321029556735, -0.22795601049700706, 0.0008454535553885466, -0.23168778326475864, -0.24381374562821848, -0.1822390964907852, -0.16964726751324385, -0.10706404807877484, -0.3148920617579128, 0.01814552248979811, -0.3038182951305896, -0.17675099554628118, -0.12509130295376725, -0.26427576494710864, 0.05343054637229896],
                          [0.18442315678624885, 0.036065642439805944, -0.1515866151190366, -0.1868196236156065, -0.2151018293597755, -0.1470663497172954, -0.3892524916386423, -0.019877627554174545, 0.007209294998808069, 0.09663377273367546, -0.2892342065038359, -0.37250304146667473, -0.2629035120026147, -0.01725631438983024, -0.16234791697498255, 0.24157090656377067, -0.07110144164010855, -0.19399427715565953, 0.0634826103416546, 0.14192436053873175, -0.16252711085120092, 0.05468965462113791, -0.1407952587932556, -0.15640731411060593, 0.062021408267310514, 0.19017265007815504, -0.14199057163087, -0.14796318049637583, 0.060261052484320406, -0.27351513921215437],
                          [-0.32626134390303263, 0.061165891213855725, 0.22603679973647378, 0.1549110585925226, -0.061053932058101106, 0.21289760696624113, -0.19865846008834215, 0.10235370088684803, 0.33580669335073227, -0.14945338682590917, 0.14397656792139552, 0.13906972085266758, 0.13441459194040084, 0.14004187857964312, -0.15628802435917502, 0.10101794957554246, -0.18083245532989659, 0.09022662091878751, 0.22127295163290212, 0.020387632665325006, 0.012700334596552097, 0.14966814660972003, -0.1084752699177321, 0.26690818873297156, 0.05316152267086041, 0.1322473405003913, -0.14066573725697004, 0.02521076152518014, -0.11541944854629883, -0.46733198884685845]])

    matrix50D = np.array([[0.1374921239644807, 0.11620641238747896, 0.29622984347321335, 0.14717319493126776, -0.2869269656303789, 0.12067109509769522, -0.06108664698851719, -0.18956839632920866, -0.06915312830961531, 0.004704936865753106, -0.014454094347701483, -0.03913046461827609, -0.06821329625867326, -0.20481122961852874, -0.18073616626530875, 0.0031452249614259196, -0.03575554675627914, 0.020950470053892153, 0.11294742869496059, 0.00890703105111641, 0.029434134169277405, -0.08802729041434089, -0.0535946531001928, 0.34242466555415957, -0.10833793926599444, -0.1611309612935178, -0.14142586316935538, 0.11195557749019834, -0.054681396531274795, -0.08982765800147972, -0.12572643384705237, -0.08045620645991348, -0.15622886904146507, -0.19149236143723505, 0.14717964722662946, -0.1466349811035666, -0.05988348398208523, -0.05333982818068158, 0.1431020047247136, -0.03250046573368938, 0.2261802770861211, -0.1105901547603094, -0.06430662806745263, -0.31774375412228123, 0.2311018288059341, -0.15653293665734805, -0.06964504893630709, 0.0593688078479851, -0.021080137447729994, -0.03887987159203763],
                          [0.024158413309536317, 0.14152765985453006, -0.1393617335668575, 0.326408411507377, -0.10830375323257624, 0.23111052970350845, 0.009413356882596856, 0.08980795666091525, -0.13155714307688335, 0.023313780687825192, -0.11134733671708735, -0.15082669453914926, 0.007633171470992013, 0.09958168366491209, 0.16032132607792188, 0.14360226768722653, 0.16600892207153334, 0.12767526763097045, -0.0758912417502087, -0.1296277300259371, 0.20998106482951215, 0.22768561546705346, 0.13044283115593303, -0.0769343211921715, 0.02455803535802115, 0.17328400284674841, 0.18877011094557278, -0.34733872793913484, -0.0009042741365653881, 0.09532836123909019, -0.15277010733270172, -0.01282985878897174, -0.1656425747074998, 0.16921734671009117, 0.14058001746363105, -0.2075301767431376, 0.09449346078434114, -0.156032152503374, -0.03548921688616584, 0.23006050507641124, 0.02867698294484518, 0.02603664280740564, 0.07929585139396957, -0.056538663079506446, 0.1074034333815209, -0.13620737024671262, -0.08621012010918017, 0.06442669989797885, 0.03266795817181725, -0.07713843849070327],
                          [0.4360601801408968, -0.21875495765111594, 0.15060597455015912, -0.0035749583050595913, 0.010719648697299647, 0.05760819155178119, -0.19228955895644434, 0.029516903365539086, -0.004344890490402425, -0.06592000249464444, -0.005299039022139178, -0.0013939318178723401, -0.1704646883783843, -0.021415705796698926, -0.02479259731543426, 0.05202615324015692, -0.11996749160134997, 0.023506651444489466, -0.3382163546282686, -0.041963907092947936, 0.027636394442371197, -0.04565328030045615, 0.08391438529603644, -0.032181161454000906, -0.13859439473554275, 0.28049837219209783, -0.16880658307136706, 0.01749371887435438, -0.05567952843077275, -0.12706047027394549, 0.10432335893247105, 0.17054614368905915, 0.16161150888336365, -0.04408510979805637, -0.10944686649496031, -0.14798313411754555, -0.042299020023011676, 0.009433492765636221, -0.11001372675028455, 0.2522700314985749, 0.062363306186182584, 0.06603040636416153, 0.042011360419535425, 0.06781265987243575, -0.02186175371760594, -0.042531739023569944, -0.03356317056273459, 0.19564076117111032, -0.049930188944878835, 0.3667564488663167],
                          [0.19793826733247166, 0.10221434177807967, -0.11921610593164685, -0.21950225040961566, -0.07430699148690856, 0.2262313066196156, -0.010523182391099187, 0.04466153089692594, 0.04032464860538328, 0.11678669156857949, 0.1397734799678145, -0.24144611627992135, 0.23313959751071758, 0.2122512931512456, 0.01925420763304312, -0.03653316059474689, -0.16530874620146913, -0.08288772089711872, 0.0577022245224529, -0.20697985426681317, -0.1683421337350696, -0.02433533462492371, -0.04880347681314115, -0.21487662286642972, -0.09870675515082906, -0.18522495571462183, 0.07860385768380979, 0.18169633160029394, 0.1191020379465064, 0.04488185878720272, 0.08733277494663784, 0.03539753494987104, 0.13622961113393553, 0.0017997441678148853, 0.13572187339631525, -0.10692578570436861, -0.06439722615244588, -0.14573278652415517, -0.1044806858559825, -0.04693251690739316, 0.03850719357919044, 0.12037290417084076, -0.08201875684329339, -0.12610536961930907, -0.17077554636916922, -0.3877682321917242, 0.04145394667703017, 0.09050413058554505, -0.20839061224770947, -0.10198887586821408],
                          [-0.031084137652741036, 0.10913904553216508, 0.061359267410751386, 0.01822908318077594, -0.05507724882237775, -0.01454281057039299, -0.01830917787050708, -0.126579536422708, -0.1259598675574144, 0.10189955467076839, -0.03069412086211235, 0.22060979912034792, -0.05408790573190157, 0.006216534710514693, -0.2551947820664214, 0.04509038280330157, 0.27025001977481533, -0.15068399297723653, 0.3472568858409702, -0.046391607821416186, -0.003438019330944425, 0.18169282006470103, -0.009893638204881868, -0.029861953787196107, -0.2415395078252435, 0.29433211107850615, 0.06898865784417613, -0.022744436288289746, -0.0019995765287704847, 0.02280066683039151, 0.18696428395897752, 0.1380386958379012, 0.1110909424708327, -0.11946727566779002, 0.16186321780296378, 0.1622965374180136, -0.11213950866583546, 0.07276413445390861, -0.23110259516476103, 0.241855275203866, 0.048913769046922066, 0.1006228606519433, -0.06083326769234195, -0.05415976291632559, -0.09085288015300089, -0.04281615600026826, 0.07943600032707104, -0.31230124087332906, -0.13167736839594046, 0.028587696171626088],
                          [-0.17972787138385787, -0.02082207808565187, -0.12924077342895757, 0.09330549299290931, 0.0541802549269417, 0.20155164940320183, -0.052367445473402566, -0.10977628784933711, 0.048526963490563416, -0.1921218322929618, 0.0694133366740844, 0.24758137826067222, 0.03613838781852391, 0.06255937845968064, -0.1678971667803711, -0.012300317746980581, -0.0008843290016572996, -0.27334343185371607, 0.011408822209026144, -0.3150824307756473, 0.1236474851665063, -0.009460702233531201, 0.11918699476291654, 0.15477572421072316, 0.32006262630462834, 0.10746616310391116, -0.023061115239086532, 0.15661077789087655, -0.04602659453802866, -0.0673874694482454, -0.18623314627796617, 0.07421690234018626, -0.058815732843707946, 0.06416560472742448, 0.2021609778038831, 0.030586922243066098, -0.09096569633682479, -0.004878053327253105, -0.00024600469912725996, -0.18908488667825357, -0.0112312417479179, 0.21920444838012776, -0.17932072408103922, 0.2365312089332057, 0.06823019763366443, -0.06801280657707631, -0.04063683682705466, 0.21383752821106572, -0.0936811837026591, 0.1893488758723808],
                          [0.10241020066333675, -0.10676991174069746, -0.028496303590960326, -0.054492302340552086, 0.003104677023586857, 0.06976521390336334, 0.34387049695752375, -0.0574715098690741, -0.0981808554632042, 0.12867977566735186, 0.3341727514785653, -0.2000648886047981, -0.14629538296446998, -0.1768559396027662, 0.03111049749810964, 0.062308965346578725, 0.21579298712397085, -0.13044123802208588, 0.10844456763158242, 0.11721785615066312, 0.0558836096315775, 0.01696523124451452, 0.09177441225466773, 0.178725877473389, 0.2280625756973569, 0.09920679353891901, 0.004682610090884683, 0.2402885506863267, 0.06435148844207567, -0.09656691071678418, 0.10293215585822287, 0.1530195082175369, -0.10441767946168024, 0.13795171056036384, -0.29635628004593084, -0.12968664395781757, 0.22522816480902766, 0.03488942897942544, -0.16375134721101056, 0.05435796138013811, -0.02976490806945961, -0.033530719461079515, 0.0039250384426245425, 0.08501472448370129, 0.0891158212197072, -0.07062733650988355, -0.05709711106264185, 0.004607342432040253, -0.1962890644090435, -0.17955888675263204],
                          [-0.0830982050418944, -0.0551160976532148, 0.2592434607251344, -0.13421030374294643, 0.06234725135056294, -0.024717894497700593, -0.15281311334686729, -0.03643068194582431, -0.19663657129313813, -0.04880060279316908, -0.08441374540663209, 0.13367607211652144, 0.13673698914513494, 0.18827807165026175, 0.23120650637665804, 0.10990994450534011, 0.12667855396157848, 0.1084249903458292, 0.1228959944273864, -0.2659675797834193, 0.18344365171939342, -0.09955856407841343, 0.023887903074297966, 0.08551584298513731, 0.21059956542380232, -0.06828856234676754, 0.023054270490740693, 0.08044018650664204, 0.13467208964202695, -0.026481595129309442, 0.006688421576910688, 0.11483880312855758, 0.08042920252768859, -0.13084337681832617, -0.25658893522697307, -0.08298441789337227, 0.11054113621417365, -0.11118445504094696, -0.08589354685563837, -0.10406950844405413, 0.10739127466526145, -0.05611577188722653, 0.15169808296124085, -0.08032351284825615, -0.1274701834520033, -0.2054484803666361, -0.21876132541406954, -0.2656747677996773, 0.23549675436092718, 0.1454299562688092],
                          [-0.1759944316054039, 0.040868183852561905, -0.16417531622016548, 0.021591351403191593, -0.040926787972743274, 0.33176105588067734, -0.10786190109635253, 0.031904908245736906, 0.03798316079505036, -0.017629925696529127, 0.11465147700829059, -0.013963776054906894, -0.08678471750657239, 0.22135271793430883, 0.08480152663828996, 0.052423253934914635, 0.03367208398217381, -0.07993546352933445, 0.03856991149254942, 0.42973804243275643, -0.1911995957436632, -0.1397210760230327, 0.25890113099419954, 0.035894749642442075, 0.12266102556773109, -0.1143503969882896, 0.14714569841144373, 0.053963314769949845, -0.12368396244955282, -0.041220615116026815, 0.09506211470776839, -0.03775647882833812, 0.07728624224412001, -0.3542727328154775, 0.015222122029378275, 0.013500102998258047, 0.05671403929760104, -0.048853058869893565, -0.06580828506520822, 0.13401405860165858, -0.08812668588227238, 0.13263817371830455, 0.14252956608831896, -0.15415503805309594, 0.18862775497341364, 0.04174503314360933, 0.028853459612087125, -0.026913952886134587, 0.10299597024380772, 0.2388262637627845],
                          [-0.013537266381015626, -0.013458252457162959, 0.08177860134710639, 0.16459053177849162, 0.35705296725635316, 0.15059566598235638, -0.14398641835239162, 0.08321894659089639, 0.3682966710618022, -0.015762419273935836, 0.055580177513836565, 0.14728464114866674, -0.07802450771550662, 0.11830087672377279, 0.013651782971881904, -0.11436663720636205, 0.23829352712165108, 0.17417325822127444, 0.049664807274514236, -0.02854637742258987, -0.07870909748480782, -0.02268746651950747, -0.19142695614721433, -0.026489648616313158, -0.14932582865150115, 0.05061829771176761, 0.014104343285946699, 0.006808039844286887, 0.01616365352465852, -0.3504016276447838, -0.11152024221899735, -0.011507697068746558, -0.0027975458744665467, -0.11246171324704693, -0.008084343724593834, -0.19919118245055445, -0.02218888699610857, -0.1743193833779996, -0.11290623603134028, -0.12169253508750975, -0.13186777781042153, -0.21472624262674056, 0.10167570409912428, 0.08162981132155679, 0.10985065285797287, -0.03223670170045778, 0.03363647076932151, -0.057710406143305845, -0.24554683527783946, -0.1721741828708583],
                          [0.08053530933811258, -0.10762581005891031, -0.19439011661676758, -0.12655863732945669, -0.005543313423752209, -0.21991460029694862, 0.1355200563817616, -0.18338766613962484, -0.01084757323707693, -0.12235722191493868, 0.06783176119880253, 0.070138521216297, -0.22288510891388333, 0.011653972646282105, 0.09389676008280487, -0.14629282398363047, 0.1149967503777612, -0.22218909528261171, -0.021650724831380603, 0.10169685474108929, -0.028058817773881615, -0.21746508421517705, -0.18674421099990737, 0.011944897141629728, -0.08245962934048832, 0.23716743286860886, 0.18162262514074726, -0.12477472071302181, -0.0323580557287842, 0.14299681847254228, -0.19803641397245053, 0.1730913153604051, -0.07089577311238471, -0.22280929111663397, 0.14178886324986614, -0.08846652403259885, -0.0773728871096186, -0.04651267635081815, 0.05792392891029696, -0.08498924274812712, -0.1043242960779492, -0.0644626452300548, 0.17291889762499008, -0.08934112161378942, -0.1581770979311065, -0.2825200994698403, -0.0965729091597996, 0.1813047897278307, 0.22594788463134496, -0.1094582307814836],
                          [-0.03743580928997017, 0.1968649406569915, -0.10694803707774472, -0.05113926341389138, 0.19425725859483448, 0.0875580066356316, -0.09614398075573807, 0.14967638935920716, -0.0989151334347381, 0.2142451799688381, 0.22628240767754532, -0.19019064289142254, -0.1126273068259643, -0.3044391966922787, 0.0768468307287102, -0.0542372280720733, 0.13173650653218613, 0.06779010124919706, -0.1100004025871449, 0.010028279670043156, 0.1330581176468593, -0.14217904462078393, -0.06843252096237475, 0.07513189353467153, -0.0594750908097868, 0.06335923778126903, 0.013813728607414772, 0.06793958298299041, 0.023329385950774235, -0.13021946066005877, -0.07470952783412133, -0.1617869302289446, -0.12853046992510667, -0.028814475889556553, 0.10264847200784061, -0.13928692494319925, -0.24914072624873806, -0.07206767043721686, -0.01572035200362956, -0.09060176137053466, 0.17537506060800126, 0.3093367040175351, -0.04741256582722909, 0.0433945286356908, -0.3139188963696999, 0.14520852045427468, 0.041715790074307216, -0.19928071359279206, 0.18914299864144635, 0.13040758338052175],
                          [0.15598111057100744, 0.03474587676445346, -0.13728386360850897, -0.09846074661095994, 0.26731696605144395, 0.07560528863181909, 0.11763035288639814, -0.2631137659264421, 0.040264525038723735, 0.22075018322051, -0.13577931773047988, 0.11685611297441181, 0.2291139891741589, -0.10159623555926654, -0.04226630095299814, -0.05398388670953539, -0.09997866808922062, 0.32936393044819495, -0.13415021567849678, -0.08439945284481906, -0.19852834329651375, -0.06305747532773476, 0.24150059788851416, 0.26693536338186624, 0.1303197690011481, 0.15042982176275915, 0.17055997666766087, -0.11950775144559243, -0.27064381858751574, 0.1395555579285587, 0.13944946183889453, 0.04768579727316196, -0.009475494004672399, -0.15951407406637758, 0.05601603209548495, -0.11120423596350525, 0.05780943383138413, 0.06718930260663482, 0.07926861383418216, -0.04308420090375259, 0.00650837353868584, -0.07059990302712225, -0.15376920622458212, 0.002105337862939953, -0.06547035079888057, -0.006518373699839084, -0.06973830970260589, -0.12342648066504748, -0.08711412723344567, -0.06327784728021081],
                          [0.004703146158482409, -0.11571404492141765, 0.042223404479169994, 0.13154295702505375, 0.08966275851895726, 0.03189001575631992, 0.17678218970443899, 0.09368775746972206, 0.09539439484654119, -0.30250723930001533, -0.14952900151532034, 0.05039928569221834, -0.24382295428653433, -0.03329730577786959, 0.014212707621607234, 0.16250228890294346, 0.05718782007964886, 0.17948668526564496, 0.17057184481865664, -0.03280694430186141, -0.2038046252173096, -0.16367098979130407, 0.10798675968921964, 0.061145475653125815, -0.18776028329236874, -0.13623718896015005, -0.16926652809799697, 0.061115897135113006, -0.1070564825743838, 0.12582279495096937, 0.13596179752198437, 0.15110222375359977, -0.014359164579094716, 0.19470475211653182, 0.16279078895935992, -0.08345772147643393, 0.2966769004789913, -0.11619188202392404, 0.05255398818674221, -0.09848900484866507, 0.0849281307442802, 0.3836151834331776, 0.03958398524175511, 0.014501122911371005, -0.15234779743658058, -0.0670502001031025, 0.11432751940663602, -0.05966104599273439, 0.08979160382414238, -0.08623234914623434],
                          [0.004647140782736005, -0.005295060143125603, -0.07692625270832537, 0.06515514587529855, -0.0016140320256810864, -0.05255587294849551, -0.03252564706366486, 0.022948253621819754, -0.15749788064396236, -0.19066441666502973, -0.2785404183125994, -0.07035447960817612, -0.02079903458654361, -0.04938686664574205, 0.08172579132388648, 0.12320016529238845, 0.07200235896855232, -0.17613180026677425, -0.009052178071724613, -0.15214512305265665, -0.25561722298438744, 0.12166482814665379, -0.0012341417859283686, -0.025671336475752315, 0.23965862143673844, 0.04294723502112701, -0.06597834023248351, 0.1341355855006902, -0.08078875198732474, -0.09539821415124095, 0.014313486563715337, -0.013433626318530926, 0.018653019592736162, -0.1926052390255822, -0.25463579749945386, -0.2564571106358162, -0.3250609675737128, -0.10311993116890265, 0.18446585887316963, 0.17515572410660935, -0.07706212351518829, 0.011586384478892176, -0.10006896299488122, -0.03861875605593768, -0.05292061201421966, 0.05323373669902538, 0.37527010834208313, -0.038783795466979874, 0.05101595722518299, -0.24885806305619437],
                          [-0.02798033634849482, -0.44863139760927045, -0.13550105375614302, 0.041333284997318386, -0.10577139310849866, -0.22116954508269543, -0.1472022719280727, -0.1643346794638675, 0.051706489550777876, 0.015664236863734818, 0.26619784165189386, -0.18071866433561715, 0.2560719968372815, -0.09216695085519731, -0.07139133700182274, 0.14214192181100854, 0.24079828829275216, 0.10960028785723994, 0.08300900719899121, -0.13909316254090548, 0.033255075839511386, -0.1124580097504491, -0.10511922297232319, -0.14070006113399655, 0.022272294529382817, -0.16912646643638476, 0.03542885148550816, -0.1130180603523481, -0.28331550623440044, -0.09716547425887266, -0.006821271499676338, 0.054628040580113404, -0.08403575751920021, -0.05478404601463932, 0.047401685360651785, 0.09853497762999418, 0.023918322514207082, -0.23332009766811646, 0.18078896459916935, 0.17411735988984225, -0.025334252011496163, 0.0839580309030984, -0.035153933720795214, -0.02001413861415017, 0.03056590061024335, 0.0591680760400162, -0.011057738884213993, -0.057830666170592265, -0.1173015495257044, 0.12478681756175458],
                          [-0.10227205187641014, -0.030310567955987168, 0.3272509893328024, 0.0901059290019317, -0.08278593214993724, -0.008781745257571312, 0.029658284938480217, 0.2715866521807663, -0.06806545056352499, 0.150175922469357, 0.049165121878548296, 0.14559476498026744, 0.05894316884501152, -0.12188172069280524, -0.09471699156503355, -0.26074167533873704, 0.06819920771054397, 0.030233891265781607, -0.09674749767630735, 0.08782792605655868, -0.0016112591301294511, -0.16011061153611214, -0.06538532228012948, -0.07433604483298833, 0.10798873477418418, -0.06609754724345951, 0.26463432802537035, -0.14452841764330343, 0.09244749829734071, 0.10005621702153428, 0.2749551100798633, 0.23233445066637287, -0.06645058004438816, 0.06991236125466646, -0.01255203431745453, -0.1395211090856693, -0.11416203740820424, -0.02226778842612427, 0.20471680569729397, -0.02448963004424036, -0.12819662534938098, 0.09780566122537668, -0.11954866459269371, 0.1487918696656096, 0.204961479950904, -0.2789325079551441, 0.2284491109012968, -0.015025419050954966, 0.04102723782197799, 0.10620697242744546],
                          [-0.009917321126126939, 0.08500499599738157, 0.12572148220186938, 0.36480752117641924, 0.07640227562159944, 0.02063462239121614, 0.020130585708442523, -0.12108116672707808, -0.16860486182904968, 0.16006492212194884, 0.21226451080844, 0.2021772188602623, 0.08904154331954044, -0.02556562454335465, 0.13164753597974363, 0.15479513627312197, -0.030006654172207586, -0.06278228355492238, 0.048383986194628824, 0.11045693149903875, -0.03131118872409057, 0.1420963655170613, -0.22500590813269955, 0.06806694615301773, -0.04703146014102175, -0.06864501886595992, -0.16350076570041433, -0.1387039768972694, -0.1345074425760927, 0.10808228745442366, 0.0752473634437382, -0.20566822684634878, 0.18302287144970675, -0.08438975115739854, -0.1820010358726909, -0.05550026520068833, 0.09059554497102779, -0.061227774941920256, -0.020336432700042324, -0.09508072123496254, -0.31479578820069704, 0.1378172617580644, -0.15242801556840713, -0.02503509044458085, -0.2829454675744026, -0.08082200820217317, -0.06567170933701427, 0.2690881892495536, -0.011237052891356208, 0.050313190045519214],
                          [0.20264770279422162, -0.16279485516923747, 0.010404090373437067, 0.016506108844246258, -0.046551791780283965, -0.07529029614955532, 0.29215680392686405, 0.15705076026721304, 0.09493912574191361, 0.13803118500362493, -0.0968050430079206, 0.2960501473053837, -0.004286691379975896, 0.10894452030857511, 0.04991549662567343, -0.2052132746816391, -0.055533788996897836, -0.01415120111051266, -0.024052737373851506, -0.0016818631668601028, 0.36610702016547236, -0.06883624941407208, -0.03559292107273929, -0.04125798839620006, 0.24561223364799148, -0.05121990100435601, -0.046270372489705044, 0.031477181864546976, -0.060717562605734264, 0.07474574495407021, -0.018706178993238815, -0.09739272446172241, -0.016040179227890854, -0.1025711852660334, 0.06931953155183274, -0.022236478111709427, -0.017928816426036803, -0.21892593399740928, -0.20211761149539226, 0.032449245129397994, 0.05387439157338122, 0.17198843777936898, 0.04451586329021549, -0.3736075804149899, -0.033566451882346596, 0.24071957209956313, 0.18531372384654854, 0.0035667511114824543, -0.15987642387824222, -0.05057271738127865],
                          [-0.07537798346430212, -0.09769712415918122, -0.18538291582634508, -0.03971216633585666, 0.13975008891308877, 0.1755067762573327, -0.15391619544765725, 0.0031137681786751795, 0.22802649877686462, -0.0038801558279965586, 0.01876621159572644, -0.010677454079390195, -0.12316361843760976, 0.001629669469175378, -0.15316872950638644, 0.25263962440559173, -0.19819985725711906, -0.11983751584107066, -0.07327040464663631, 0.14239357772537706, 0.3828268180325954, 0.12142251361881508, -0.2151732447232034, 0.03148734067005056, 0.008309436504821494, -0.0192109420212605, 0.00953368222145878, -0.03968527168738797, 0.05420325870651958, 0.10385382501129171, -0.02096234941283189, 0.2164702964831803, 0.08008828581309134, -0.004114842378529619, -0.07603999160944258, -0.15227366548577936, 0.17019904488204562, 0.15536787786231882, 0.2704273561853973, -0.0018318775501165712, -0.008119612471361974, -0.06897783905616543, -0.21238814649179855, -0.2222023194477731, -0.08331787795388895, -0.10354693546224994, 0.13736278868474808, -0.26677527064235684, -0.010779809179358354, -0.00941988926046905],
                          [0.1461241459973719, 0.07257516717227365, -0.21042273377928544, 0.16929337342774176, -0.04676092431621773, -0.10990508548761271, 0.10407036447024301, -0.24545089652487315, -0.11238234981621081, 0.27653159339284933, -0.272421400127879, 0.13172491694289085, -0.0832778470988651, 0.035403534079689096, -0.047171772963530895, -0.05672686407814816, 0.07259965821032023, -0.07242684692525532, -0.01394905159528548, 0.1768969711588013, 0.06293019811968241, 0.037245669472052746, 0.24249652170842886, -0.18411937258079994, -0.11745412023860409, -0.19046502355017153, 0.024261266124505916, 0.1414704779276214, 0.18243459221624297, -0.23012342203274122, -0.12722573420065672, 0.06375344260241654, 0.06624668030010482, 0.05535328587416735, -0.06954739295823148, -0.037938011629028795, 0.05240779972852544, -0.19777627077125878, 0.3352994237156567, -0.13449062299434544, 0.015166017758611398, -0.007117702536826086, 0.09355130971181144, 0.15411049260219967, -0.10660757341672641, -0.0565285792656695, -0.013971342636861344, -0.08876146455377122, -0.1379510018284186, 0.20744135682941364],
                          [-0.11979933520952238, -0.07207598275878427, -0.01953253193969577, -0.09181184453305237, 0.29268246700311307, 0.22883650935960287, 0.027239296752990423, -0.20205357442000937, -0.0641565979767727, 0.0772714439078058, -0.07367931438938922, 0.07244169639992531, -0.15980820979979654, -0.23444645203572373, -0.07187167796215671, -0.1694552902032788, -0.022522640446682264, -0.011598246259612015, 0.017542432013882375, -0.2146804685331852, 0.01747874727713187, -0.047842955031377206, -0.02085461172841982, -0.15380184161996155, 0.02587199344920569, -0.34975146042961497, -0.17532515103509508, -0.14678591757636958, 0.18021644946784574, 0.09500227817352948, -0.024755018680761805, -0.01885030918126153, -0.04402177513543062, -0.06573844589680497, -0.05058041027060586, -0.007488879353534751, -0.08037260851061725, 0.20795497049326675, 0.029917224336063704, 0.3777288426949057, -0.12735252069112815, 0.17026264920629278, 0.26488530527650594, -0.037036394320562056, 0.06643707277650558, -0.041770381432797016, -0.1615896436887009, 0.0698588699163445, -0.054410638948826195, -0.08010207730008424],
                          [0.022284849849896326, 0.0433793364808565, -0.18553923944821618, 0.07269621508529431, -0.13152985585579866, 0.0432250022802366, -0.01267930160509415, -0.05786815710684336, -0.015853779977703753, 0.06074612669669791, -0.09254697677848206, 0.06528726867862196, 0.10393752278635404, 0.0999570451806761, 0.1049695196290865, -0.014379298479230217, -0.22150668345224853, 0.10511393063166458, 0.03115140384654522, 0.14666409101156544, -0.16286327395905634, -0.10067410877820934, -0.3234667146499723, 0.10568898256833593, 0.12306616925501082, 0.05119028817476767, -0.34671419189692676, 0.006038666349734644, 0.22281545195758473, -0.010328951716107131, 0.01556048686542452, 0.21587500039205182, -0.45427116926046224, -0.010152096718740558, 0.07932442258203716, 0.04623842812681746, 0.012759812072083168, -0.1534541281871768, -0.16615296357387294, 0.2083153648683784, -0.11977243889326672, -0.0473734997357221, -0.09315539874414622, 0.1784055045552669, -0.05421570873268043, -0.0200761555542628, -0.009686978592292313, -0.17973796482031373, 0.017279720173087872, 0.11846519104399574],
                          [0.06446923729494101, -0.09504665247730137, 0.126840563805456, 0.09006773430141567, -0.16498680942499658, 0.16155955210693485, -0.005460234666600379, -0.01880642632729621, -0.04421522787197616, -0.05780330138778503, 0.06241495409882547, -0.06277183918272318, -0.1343214309872556, 0.0163860127554471, -0.14989574610316644, -0.24119937020452983, -0.07533919031187176, 0.17283775041772023, 0.12332297080114236, -0.045839333404342236, 0.0640765682212555, -0.07752726732877575, 0.12223705551191062, 0.033772336740209935, 0.06312941861080228, -0.16031701531218476, 0.21282653737777543, 0.08154108866898879, -0.0026356163542421264, -0.07864519841591443, -0.27674323943703116, 0.04579024265493613, 0.17898075943094593, 0.0009377965235339561, 0.08904501726353763, 0.013750167411162948, 0.1241334351808789, 0.004638356993389197, -0.09950690604097902, 0.2666810779199912, -0.23876762035435278, -0.22916468751148408, -0.23650817263955268, 0.1543983395663433, -0.3770261924721972, 0.1208753500582816, 0.10454031686591006, -0.009490053326783194, 0.2266228551129204, -0.07879719999816555],
                          [0.09511841702382788, -0.0660788318583924, 0.09351103258268868, 0.0587833430830885, -0.09688372038213874, -0.1735002118606512, -0.017780041436987335, -0.07713954345118186, 0.10392914092647347, 0.07002071920107458, -0.04253566216997253, -0.2227918758467281, -0.07741609158973416, 0.2804181564868337, 0.0837460990980419, 0.007830814374502532, -0.025286209445468444, -0.07968755681132558, 0.03123331046073344, -0.05754012133287277, 0.07351673859201684, 0.0729655950074155, -0.14192000468252677, 0.3590914784599049, 0.16636873897990825, -0.06702644748207168, 0.08089608625272032, -0.09582995831756486, 0.12416553613512595, -0.12169027834777546, 0.2433724852436284, -0.17699138426694364, 0.11256679106532286, -0.07730859637390627, 0.32070916661443183, -0.16418434198587348, 0.016498622521431614, 0.25228099637472473, 0.15255026665680516, 0.04875563174812195, -0.07008212772997147, 0.14626224205803803, 0.27327742939402766, 0.2465883800649849, -0.09024768046216727, 0.056716476799721374, 0.027590103286361307, -0.07230531751421614, -0.11156922003002125, -0.019790632988596218],
                          [-0.045831458489914066, 0.24740296817835278, 0.08823576020351372, -0.1947413568714431, 0.1743753409524092, -0.2072687552544093, 0.3065260889357993, 0.14288316435501225, 0.14238587088177107, -0.19830608869559285, 0.12377195095778214, -0.054794439460776674, 0.03432244340109095, 0.03666415130034315, -0.10275393206817969, 0.058155598504403425, -0.0573771995599095, 0.042784963022963175, 0.022475038838029947, 0.10392323487171112, 0.06534099342380523, 0.25476371694098654, 0.14889654711090697, 0.20285422973704567, 0.029745904261100055, -0.14938586418448496, -0.09874745745412766, -0.15766178172891446, -0.01164679252829034, -0.17295291867817664, -0.002308592290796273, 0.20327636735396232, 0.09571628136522681, 0.03554367359293003, 0.07196127270704201, -0.0072421142678351985, -0.27095602134876307, -0.2959014489484061, 0.09666664984607606, 0.20503965128168172, -0.14183690957898556, -0.011299545776996229, -0.050023347630253186, -0.08225628484011285, -0.04786718111249976, -0.06296384793917333, -0.22692886791973346, 0.013219231865326544, 0.021550786680442573, 0.12177424078005156],
                          [-0.04761365536146662, -0.16436643824470532, 0.02821560148106892, -0.08388402546252578, 0.14218173494186873, -0.07926765869580214, -0.1883250233029576, 0.10152099760819593, -0.3568557541880029, -0.14089858245784984, -0.09724822948587511, -0.028883956616452103, -0.012429102045988093, 0.09374723409685942, -0.3702618445226094, 0.031450394324213156, -0.11850451634073753, -0.01884117968764085, -0.158992452419291, 0.07249350477714374, 0.04887535321964524, 0.10484922358500937, 0.05143916354053632, 0.08276596090253849, -0.15306910074706753, -0.01713679546273266, 0.2518957367690015, 0.0893405577160713, 0.14804812740684192, -0.04647677681099182, 0.14969279072374952, -0.10251057822234164, -0.35108210075049506, -0.13847801164363915, -0.011560535128676126, -0.02672478077410759, 0.13886339820943158, -0.19546464470370617, -0.06640125519711035, -0.08539196984493716, -0.1984201646540744, 0.12556867216088605, -0.014540227767386879, -0.05557759643610085, -0.08391919883535692, 0.10193762592911566, -0.17057864984899623, 0.07735593101284566, -0.12312337096042099, -0.16597724336110636],
                          [0.047751227210036204, 0.032658411494503437, 0.01647013998156814, -0.048261258156358545, -0.046661573237947215, -0.07222567090867016, -0.19457673401040218, 0.04688937168515387, 0.0926235058246809, 0.21033040994373103, -0.03260508846560848, 0.10688561667784023, -0.30829394443396496, 0.06688723374756891, -0.03718020785118166, 0.18796216328620516, 0.06373860579495602, -0.307950712382529, -0.12562055994708343, -0.09734769882504625, -0.2400914775372804, -0.31084624996736515, 0.012441864549729773, 0.15190545554459164, 0.08970506880159303, -0.13211161683574862, 0.08903100754229219, -0.2194992124140553, 0.06424256300602381, 0.15160289741283983, -0.015989718380103206, -0.10141743986879947, 0.1218734765841365, 0.26886231035796265, -0.04885461649335814, 0.003179063543532045, -0.044707179781985754, -0.2628707479040124, -0.07206682778075893, 0.08752899391672216, 0.04128666950565137, -0.0839237524453737, -0.20650120225844962, -0.013915048612073861, 0.029395579302540345, 0.12199657535401577, -0.22694112115816495, -0.12346225111827525, -0.08702538280834138, -0.09492300058471989],
                          [-0.28409293331248386, -0.012930928939058213, -0.12444406313932728, 0.12680565615185818, -0.06550317957596899, 0.006183584512791723, 0.09215315707481604, 0.020866578366759116, 0.003504660277758028, -0.14402068955701425, -0.009006930867670635, -0.05349053078222559, -0.06300102502468047, -0.12985706966372038, 0.24755139850999214, -0.22031158445162036, -0.22753059043466742, 0.07039336300047618, 0.20932029708374933, -0.16316194627589267, -0.027987072493243084, -0.12006903138739035, -0.10325543682587698, 0.07740135616422375, -0.1381991631168025, 0.2504055630290222, 0.20221466803464613, 0.04433639479526045, 0.24672071540964552, -0.020300097046438717, 0.1575567880536051, 0.005872058595578596, 0.17804431975768337, -0.06667032554224925, -0.11785807743493226, -0.05680330350863375, 0.07052315403326932, -0.13858345651897608, 0.21591851697209205, 0.12049089671072341, 0.11579090583744259, -0.005058441761966774, -0.13078331951377123, -0.13107616818975437, 0.0007503238906834832, 0.20269442536965862, -0.17936112998170364, 0.10415692835247328, -0.2415956288981101, 0.14617773721218724],
                          [-0.21564838732047467, -0.04464114229275476, 0.033072435874516797, -0.21219578416004847, -0.2935550570059296, -0.08330755674324218, 0.03462343429321226, -0.24816051933675828, 0.22343996696975546, 0.19434423031071338, 0.1569700891158693, 0.20204652928519198, -0.08140578139920594, -0.05717588764498723, -0.019468399870769947, 0.1594298299172208, -0.27104383880436017, 0.10453990277191355, -0.026191879335008942, -0.10328237763868177, 0.019146406918168473, 0.138647654555919, 0.13214366250559442, -0.13141913189702695, -0.11300098925398844, 0.10081730506298227, 0.002613945884217858, -0.06771375198987829, 0.1047785724525005, -0.17475252288909823, 0.03021380003155114, -0.07091101398714247, 0.009709474342839482, -0.09343110633999933, -0.04295553076611749, -0.2724794349968118, 0.050901612168309913, -0.163634629377763, -0.11263052516700896, -0.02216572799635468, -0.02571491522495654, 0.20214017464786874, -0.010143868515846656, 0.14204841832894435, 0.15842655341100526, 0.036728510973636906, 0.06736441267479902, 0.01291588561774926, 0.27653163522905316, -0.19671949716905776],
                          [-0.059244032398172324, -0.13280207634191657, -0.03058407508834027, -0.33095815289574193, -0.030108179977903066, 0.07524352102528373, -0.08285398817543406, -0.07195721155234483, -0.3683946798226929, -0.01912109655863658, 0.13275775995632402, 0.10267561418981974, -0.11650114314138289, 0.184206898421966, 0.11779199106434769, -0.30171767066916105, 0.003925486495763255, 0.10073657950020448, -0.02485367930139345, 0.05168588139243844, -0.1546974993045953, 0.27688154637372603, -0.14411161859577287, 0.20371465561096838, -0.09707344766667533, -0.008061856373000895, -0.08278866890506592, -0.07607894114015407, -0.16346528863496762, 0.046559975827814626, -0.26246054221334325, -0.01831072377249786, 0.1217393423906023, 0.2703201981443506, -0.0006758845213250125, -0.1382112002308176, 0.053738895505905986, -0.06948291169378125, 0.08568910091909525, -0.031481442886958436, -0.002287763642926289, 0.09546275830028499, 0.012114637623849423, -0.039606322692301264, 0.16056054131000194, 0.030267797484188865, 0.16633585807038048, -0.1471888643015032, -0.13336472692596815, 0.09415742915838955],
                          [-0.018027447396159198, -0.16248497511143783, -0.27255244136851686, 0.197757438370414, 0.049779052421899006, -0.1878915690843561, -0.2079801635749245, 0.1611444048172155, -0.04865822844975581, 0.17564950077340497, -0.02188620377872107, 0.04123997745865841, -0.010501889368554264, -0.12814296389759652, -0.1572727908304814, -0.07727674487915548, -0.18051807387961957, 0.027975467029148056, 0.27581769067316453, 0.00909043085518992, -0.19200765412311396, 0.1456978339390387, 0.027552670918928098, 0.08160873241823213, 0.16569870297371403, -0.09576897763273033, 0.05735942646276047, -0.11894097792241111, -0.005473215213674037, -0.23199055423198015, 0.06101328471719898, 0.19136480349884405, 0.05351810774522551, 0.21125853634609307, 0.007510030439170076, -0.16904925196146472, -0.07450562208500733, 0.15088762213771617, -0.24348731809499624, -0.1305311565390782, 0.12726244798726924, -0.06324651998996451, 0.1505378536066525, -0.25417443922936855, -0.052070245738834314, -0.0387021651416259, -0.029002845203943778, 0.15612487048197443, 0.16042362474872973, 0.05954669952301503],
                          [0.11363157489202275, -0.1407899370642752, 0.07344109410090559, -0.15150538481605327, -0.03773005273863293, 0.2122983360456805, -0.04960207200570184, -0.0829220227504401, -0.07777279267905068, -0.11652789012033608, -0.041937301218968055, 0.04402861199880704, -0.1971471942699836, 0.061450547204326814, 0.1957487969110485, 0.09979940793351173, -0.16910845095108448, 0.09639564202114065, 0.3460580665469725, 0.20506443651997142, 0.1738121301449954, -0.07041717933434904, 0.09498571847451363, -0.10988168550084358, 0.05285031187220663, -0.03441431354087211, 0.07845721824244062, -0.3291213669892272, -0.12362935198341125, -0.14981460023849075, 0.11340500770003607, -0.05072323217379922, -0.1876019810033974, 0.04194087844721887, -0.11340535473228731, 0.11822944981574282, -0.3434138079159768, 0.04154870582576719, 0.032943752408787434, -0.15256236258799516, 0.13640001102005303, -0.005334367244859921, -0.1244081791151819, 0.15751514838799438, -0.13206645131505115, -0.058726854778653755, -0.023593066491890375, 0.06542882520845711, -0.19358137019456037, -0.10643193992060962],
                          [0.2585994662987746, 0.2519243449408773, 0.12210575931318997, -0.13183403096134047, 0.19719555330961203, 0.0021688429326270765, -0.07151766086506114, -0.0733410508432938, -0.07968557043075065, 0.12569414422929182, 0.04041219564195972, -0.01769423964873476, -0.13862964185462337, 0.007689600190379189, 0.23713304095492066, 0.07520431051178758, -0.05544955572203296, -0.1337949112014219, 0.15166516620863799, -0.2468824528969151, -0.023508896848733356, 0.023101595117974685, -0.059428712486528604, -0.10549759999585936, 0.011234715829253804, -0.01456346063573024, 0.12897020021480368, -0.022135192757662608, -0.051702921294854605, -0.23087990841245962, 0.08978044606134977, 0.2261479850808003, -0.1869424175528443, 0.02326660316295725, 0.15081840678501637, 0.16905610608693608, 0.24248361089970066, 0.00178768851999016, 0.14817927652844642, -0.05559537550429741, -0.22896777296785, 0.00031706471909264994, -0.08669616667531241, -0.17975846601015968, 0.1738973737442043, 0.2534700646231551, 0.15261572609379204, 0.1251936801655581, 0.16045999269730266, 0.08535412270143948],
                          [0.011499269199745877, -0.11840074965542705, -0.07148856111976791, -0.0974348577468137, 0.13419605880166158, 0.15386432808492292, 0.12739132998242214, -0.033364021245004365, -0.00146013227273833, 0.1025870738388921, -0.19340181197404438, -0.23649313353186707, -0.08575914542364693, -0.13611227248557414, -0.21730359495667576, 0.09450077809023925, -0.0635254661849532, 0.03382606350975267, 0.1629413117459578, -0.09248517275456494, 0.06694193812180094, -0.004813638514348097, -0.11874326826044121, 0.18904908549576493, 0.021761998152086043, 0.11243272035250045, 0.005225866681574768, -0.11192132573261072, 0.0885015044458164, 0.09414912812655442, 0.054090187368817585, -0.24833414512842614, 0.14074989607183042, -0.003936871017069263, -0.06335053189000012, 0.23705107929229044, 0.00547909990148945, -0.3713263483163232, -0.08563561633165899, -0.11000401227517399, -0.09425703961750993, -0.14701849954710333, 0.16455379188340127, 0.09728436644638734, 0.12288538855127167, -0.1574609099785832, 0.3187867038144821, 0.08445557839512705, 0.14076413396044124, 0.2277118222700476],
                          [-0.18016077571580622, -0.023489029555641416, 0.1085980636443699, 0.03555839941830249, -0.03442141679678791, -0.07577179376076208, 0.01064223480664714, -0.12726718102414047, 0.09860543480462872, 0.09280023989547678, -0.11288182793283265, -0.3337751504615306, -0.34974948379917675, 0.13776001806630558, -0.07468076992936672, -0.2287000658273846, 0.05008345041090584, 0.046204425717634744, 0.03394263270923437, -0.08193209451962148, -0.008790338842657447, 0.12838105952050935, 0.06295457677424221, -0.2935525112172386, 0.14850846641364673, 0.08513322587346096, -0.21262991801276, 0.07527616748964053, -0.20805650865683042, 0.1652849246610912, 0.16980569374938204, -0.0647233365994843, -0.12660011052012984, -0.016953822301357488, 0.044042184975088744, -0.18277697855048014, -0.04792036078269078, -0.010299292888234278, -0.06710390726852725, -0.20287476204865612, -0.18491891068359004, -0.09311716842428873, -0.18158906171086495, -0.12112473487620477, -0.025613990478127777, -0.02465137084432279, -0.20018916386041788, -0.10110215722228692, -0.008694278569278952, 0.21009969914644316],
                          [-0.32663311758185626, 0.1837771614696603, -0.11426970987818001, -0.1471286856340909, -0.01955080164378242, -0.1045810087851186, -0.04435436481899158, -0.090342847228477, -0.1158455867796421, 0.07875337678018333, -0.20194054930904662, -0.023556984426997968, -0.056342990177656656, 0.14082143799216804, -0.03220784438013452, -0.03997314832857609, 0.1601240635035542, 0.14209852038232212, -0.1946657106695917, 0.04773267309541836, 0.1853839998356553, -0.2795641822872285, -0.08027660914206391, 0.06262812162107771, 0.019445717932179918, 0.025621805900752064, -0.13995092056390285, -0.014154470824254771, -0.15438609721337432, -0.37097423484893943, 0.1823992247328356, -0.020740242361716142, 0.10131399943153355, 0.1286288928285722, 0.01506167853524858, 0.09753854898698386, 0.1455859457434597, 0.058443667608151825, -0.06377772977104057, 0.1655981405649254, 0.056331386918214835, -0.0023404988212128042, -0.14902884105999312, -0.10881748741770644, -0.0658730774133826, -0.2123731985594028, 0.13257411256632035, 0.27697110532975217, -0.03645743095663995, -0.1302785545623915],
                          [-0.08226477075127513, -0.06610406242781616, 0.17875299249669982, 0.06339237573364594, 0.12344390637897781, 0.11117948474048868, 0.10419281949672343, -0.31826688144810966, 0.06534906920381664, -0.18514827514551396, -0.06508726986377773, 0.05029010468495427, 0.20108450915631568, -0.12502205452239512, 0.0968203297825337, -0.05454642665491973, -0.1622071919487721, -0.30692656735805557, -0.1138335589500706, 0.12557399033048214, -0.010853513638100363, -0.010407147386859007, -0.09148154695690416, -0.08311012046741016, -0.03730059504800653, -0.0006092612079759172, 0.22625380706407452, 0.11787925236278035, -0.1843208572396973, -0.15964260207214534, 0.1657416426984277, -0.19430185630430905, -0.10138368944150151, 0.3992366083094025, 0.10227790821109341, -0.1315751044305682, -0.02016899913003069, -0.09362812662452799, -0.11477090454916292, 0.14261167825269544, 0.03180921798794127, -0.11946611312706508, 0.1348889574914507, -0.09359413387011088, -0.09539364682285556, -0.04027946734627746, 0.008144899932257055, -0.1744333216007825, 0.10090666522087485, 0.09871307968857791],
                          [0.1936964141163099, 0.15786366222039652, -0.03854690044390554, -0.07338010770216379, -0.26902412615194576, 0.12644747329470857, -0.12098215701053215, -0.07613786777421783, 0.254015285486226, -0.11492815798845181, -0.14048780077631776, -0.06822939623144136, 0.1699492552675107, -0.024392447887171632, -0.17697863600255398, -0.17324326726203906, 0.131258787519932, 0.04495976327505718, 0.08725901550320966, 0.019523225918879637, 0.006253776044521463, -0.10751244279691836, -0.03960086921421233, 0.099492800691808, 0.018625132091404182, 0.1725125944470292, 0.027038930237053137, -0.0987550541069362, 0.01992043726511263, -0.09369042414185158, -0.04644278850798229, -0.021291851038143666, -0.006997217708685669, 0.22484997309945357, -0.3545011651059813, 0.0478174788831026, -0.030204314006861743, 0.06718406796604244, 0.07197113173852929, -0.09804163517722064, -0.3575087657031427, 0.3506014699724004, 0.11006633987550997, -0.1936681631847768, -0.046533846673323775, 0.05584692460266066, -0.11607217928470011, -0.0754463417511927, 0.04807203304544699, -0.0192906390629548],
                          [-0.07929646566179914, -0.024131943134984696, 0.12582617207303282, 0.054931709755634835, 0.02341409787446219, -0.19074740381237762, -0.19624220732200603, 0.04450128455800215, 0.19645348567275872, 0.05336050466425386, -0.17138831246962943, -0.16293313081091684, -0.048177103838996316, -0.25151296812630236, 0.2894469229514785, -0.20311293531160063, -0.15178711916919216, -0.19108086548920603, -0.06915322716194712, 0.02397256881234711, -0.010874271834806123, 0.1946341183081051, 0.15527429117595068, 0.13733672894442767, -0.008007327156932373, -0.0482811538880768, 0.07117369623238862, -0.07447071582494068, -0.10621116198756136, -0.055026952022834384, -0.22100692651516274, 0.0492447759339225, -0.06246400822586071, -0.1685287479051006, -0.08056708658771615, 0.2708000782564378, 0.1937326881355113, -0.056673141132189804, -0.18192408532984838, 0.03709681507583911, 0.01210015183948856, 0.22105094678401216, -0.08241799342325198, 0.04366155885389273, -0.0338673223262576, -0.2992518023808649, 0.0675107604972628, -0.11466795584337794, -0.16956752447927406, -0.04693479776750572],
                          [0.02215417936558937, -0.23700563613233214, 0.06010046765521834, -0.16310130836806877, -0.031408931592276375, 0.2595226779260457, 0.1287681653449252, 0.28780370777650444, 0.11273826191503608, 0.21618115578363808, -0.17192769854256224, 0.0822997449700466, 0.06751037821728618, 0.038644939268686104, -0.055819188741912426, -0.11443626325712276, 0.14382123016657639, -0.1807486180253677, 0.06475384010085908, 0.03431771849775224, -0.1624945471265394, 0.23523102270610663, -0.1558826952913545, -0.030906308592279122, 0.041374893412133795, 0.10574483059993499, -0.07196998513804927, -0.009917249812857886, -0.09991089523851686, -0.18955167709302317, 0.03174122405771145, -0.12596744872284493, -0.06312037310397176, -0.08084516656034242, 0.01505930175099559, 0.08029412509140024, 0.18589782902600013, -0.029662696510233552, 0.27542958084838143, 0.10498007657995231, 0.224467439408358, 0.024344229995458662, -0.11351601029610171, 0.06816881650747565, -0.11567750476797482, -0.024096942498628038, -0.2905929683450007, 0.1016042464083039, 0.24798053697732395, -0.049060188668656725],
                          [0.117485875742302, -0.1742263700157319, -0.039523550516705336, 0.2326690836228456, 0.23000745494369593, -0.07904735660146998, -0.08375224044361802, -0.15160808611672605, 0.05423435486902463, -0.0060755217967054555, 0.2748489149921174, -0.007477985922424215, 0.0018896184511461287, 0.28488956348513267, 0.042779414938904796, -0.23487675710853126, 0.013346337293573906, 0.004502343861048539, -0.006451313989706766, -0.014273951013467873, 0.007872470082591764, 0.017372905005961514, 0.2814784295999547, 0.09653986965414604, -0.06082032039046738, 0.11781286404198853, -0.15536461415962535, 0.003352685370431033, 0.28368925985582366, 0.0050305186282704525, 0.13556350605282916, -0.18850244928356324, -0.08628218884021455, 0.11339849434876942, -0.06450100454543274, 0.22156146227138607, -0.09995004273863217, -0.019030976079067535, 0.1635839002167289, 0.0946487545374452, 0.08764348459917995, 0.033532443491782755, -0.18310681589696534, -0.1704369967316346, 0.07570264272770301, -0.15405883899986195, 0.10901889214260184, -0.058226193381651664, 0.26397104212173567, -0.1527272836102222],
                          [0.021016252703078657, 0.018730978498865198, -0.1856759009821403, -0.033795543647475876, -0.06845313507613399, 0.01511294928142724, 0.04410130615724667, -0.06488169694619712, 0.20271444222950594, -0.008571461029878196, -0.08587411297312474, 0.040681203888842464, -0.021640384597376112, 0.13433240468108334, 0.08973119923317396, -0.10747469758788238, 0.21916868979647214, -0.056831603471104025, -0.02442815121332346, -0.1472227501013003, 0.11588858194345866, 0.16276709642471726, -0.051329066007981136, 0.25781119760963866, -0.3455824544029788, -0.3593786185007559, 0.16807434823317002, 0.04760555896253815, -0.08574888457599904, 0.1537097295293726, 0.18144497084994407, 0.07268955781378895, -0.2323286792347175, -0.032775077432262295, -0.27834269397049244, -0.0970568893991682, -0.10084826947610602, 0.0648810976620571, -0.17300445557046654, 0.030552943643573534, 0.10724672403322834, -0.061503824341675915, -0.13430480111179183, 0.11252623188130886, -0.006793270719517277, 0.09826560689137259, 0.0867418464564225, 0.18282482762056274, 0.12903782166271388, 0.18135951511415335],
                          [-0.0727535919010361, -0.16183602664093952, 0.1112478038040491, 0.03805610137215801, 0.1687848844186108, 0.06099122346181113, 0.06420413035490029, -0.06581468274680878, -0.11512594462320881, 0.11772734210380074, -0.2052088123444955, -0.2923844671321434, 0.3171234970622314, 0.10644973368375446, 0.02808683760428652, -0.07202851267135628, -0.008898536625503625, -0.26823166700837753, 0.12299472111093442, 0.13459393685856288, 0.12833548421361418, -0.2120413142800408, 0.08577908944459547, 0.04487690389317603, -0.22866002140176947, 0.027280237382639665, -0.18608192334618395, -0.17719046885378723, 0.02632481328764226, -0.008276356178608923, -0.11829731973633312, 0.24994751055352157, 0.1719145533753768, -0.037452192260440914, 0.053305147804555725, -0.21085940819728902, -0.049218361706061454, -0.010756828993144436, -0.1370830280243069, -0.10476419302659684, -0.02085931916277154, 0.09267956470713444, -0.16723184212137557, 0.08439455378780303, 0.17324215130049656, 0.23832474945303386, -0.03896710138276134, 0.022763864863716076, 0.10446700694892812, -0.12101098595669657],
                          [0.09720159269182294, 0.10841833403298232, -0.13624375188268173, 0.17698214309337945, -0.061351741786586374, 0.04174314403393002, 0.3023380380540203, 0.013882815557234918, -0.16156112008365664, -0.23223264908701946, 0.13963279920148813, -0.055907270422838586, -0.07011469101335359, 0.07816312614794925, -0.19074655242640437, -0.1334449770184675, -0.1208409421236679, -0.1227146253581876, -0.20004271829896875, -0.2070717002226672, -0.07570648913967468, -0.061573138051664186, -0.12853236279629024, -0.10316151589218846, -0.0010616151429416396, -0.12379712775128712, -0.01032770608186732, -0.3709475843409666, -0.07803469291215918, -0.2767777002662802, 0.06132994551908956, -0.031734770352870596, 0.05231954875905161, -0.20270451143021284, -0.16519518913955308, 0.06299409587383632, 0.1515844272413884, 0.0706966082596919, -0.11451985410643505, -0.1599000178215602, 0.15590809524007287, 0.017321141139035236, -0.06008123110270013, 0.046407821228668066, 0.07717176995003759, -0.0226351489610591, -0.014144778808848188, -0.3035778967083723, 0.04910942019657426, 0.028904307705884376],
                          [-0.20908849913630198, -0.2947515144974091, 0.0948853672388619, 0.11029663547054476, -0.16051641480865497, 0.22642588201992864, 0.23708772807256337, -0.007523672885391238, -0.021679160797194723, 0.1780507056975991, -0.03633746349812419, -0.03936299729683905, -0.07458981366533024, 0.03842507369155097, 0.1422599725511691, 0.1965625971679531, 0.029290436322201373, 0.07475537965367304, -0.33555806266122845, -0.1388835930304606, -0.13080378953325475, 0.060670403101143834, 0.01884127770920282, 0.1384832830556403, -0.133416109180038, 0.0004836371660627683, 0.05201391815645144, 0.03762015600154396, 0.09112114299576436, -0.15187405262604933, -0.03676508503593242, 0.2236256723464851, -0.005226413708043102, 0.03916576159211493, 0.06213747287034201, 0.28451519920513507, -0.2816502344814292, 0.10510670295032723, -0.010786124795432976, -0.18842548394896658, -0.18208206253427361, -0.021976881257699338, 0.06694813586628483, -0.24466757354438212, -0.11642987247514375, 0.03895589143637633, -0.004587966511757268, -0.05831687653243597, -0.037883966213458825, -0.019799949403995952],
                          [0.014560075441365328, 0.053534940325775805, -0.19106677319123508, 0.05787295763666406, -0.1579173630594444, 0.14592166853454525, -0.21084038458428464, 0.12885676431607962, -0.08306336860463227, 0.010145524007035845, 0.08713398944902832, 0.07402091938056717, -0.013137604336071994, -0.1440211730602026, 0.09498268474775096, -0.20248689405413645, -0.09226570369991138, -0.12615303773729578, -0.04619145479153911, -0.1010815252416784, 0.23220760604298915, 0.024054897356661782, 0.10642200192428831, 0.09122348140273788, -0.09325412428049033, -0.019853125415257297, -0.22160129340773502, 0.0868595074936123, -0.2753834925608185, 0.056057952176785275, 0.3483824315346287, 0.1577499875778439, 0.12367379154053017, 0.0025769395829934086, 0.07652566306385111, 0.10867048133324264, -0.0416448414211638, -0.23045030953498, 0.10052220851210074, -0.07702063461780613, -0.14111081710534515, -0.18217329605421945, 0.2558673848749977, 0.13823128542531715, 0.004684871910373344, 0.035504257341361666, -0.11617856125572094, -0.11517897054597628, 0.06763811625952221, -0.2876156812251252],
                          [0.1404932460080464, -0.09393306280279555, -0.10088292433092293, -0.11283324329527794, -0.08025000738432389, 0.07835843602207514, -0.1059967540744111, -0.16232378197660308, 0.002136832725109487, -0.2631641653825679, -0.09014767138056007, 0.041139381797628445, 0.034029237870025486, -0.3153245424519863, 0.16285372220955688, -0.057518835806209395, 0.32369790194839265, 0.06521673446992549, -0.14618614374500338, 0.15991961333474233, -0.010110480877169984, 0.2056600500983576, -0.03335997574071633, -0.05277015985173178, 0.16224873159890585, -0.12982269519749948, -0.039596821064528304, -0.1118173344290603, 0.3561704945578816, -0.03948108178116315, 0.2066993265795226, -0.02195997355752005, 0.20826507754837933, -0.051530627874275876, 0.2645263010334456, -0.004957615915925493, 0.05519428463423369, -0.06150966552884851, -0.15451662206262684, -0.17298818411626035, -0.02541999354132042, -0.02746877104380231, -0.25753886675851445, -0.059263852646767624, 0.051240278181066544, 0.05169052973072976, -0.046261239239293865, 0.02618746937354447, 0.04058523918618707, -0.007330001215441655],
                          [0.053238366937102544, -0.14414374998355547, -0.19128345242809458, -0.1212795175582309, -0.062041159030865975, -0.08814459542309903, 0.10336350598402722, 0.31082192075728254, -0.06472458891173684, -0.022819088099331548, -0.05248674875249396, 0.14645376882186434, 0.06030541801995156, -0.11934156115475718, 0.1582952845187834, 0.1012946508842709, 0.003517954495788954, -0.08603448404039625, 0.13114034685357234, -0.12472932174833458, 0.06384787152920729, -0.16223942876329497, 0.2105646303917635, 0.053586191487235806, -0.22236430925114486, -0.003241012849387476, -0.10435150589357728, -0.0013066738801647104, -0.0003316179608114471, -0.1379393637722831, -0.03010882000161903, -0.3432268535498913, -0.07574716172741648, 0.004990345160459931, 0.10853708375898011, -0.17111803679093807, 0.02452039834594816, 0.2368390219860048, -0.016938300152503274, 0.03493391264355569, -0.3805392642339704, -0.13249024614915106, -0.16730236714838873, -0.05210888523940232, 0.10323688295412355, -0.20701012895415336, -0.02452176932185337, -0.16487382581194274, -0.031760194386503245, 0.16674806887198618],
                          [0.16216385616248383, -0.04688831214518566, -0.12379361574513689, 0.13283777983750128, 0.10081533369703505, -0.07880115103294315, 0.059813576587519526, -0.00637048819416764, -0.07690158979252128, 0.03020980107938395, -0.1437867665874132, -0.013291502466450925, -0.12423459480516749, 0.03873862157813516, 0.061022413253352496, 0.06834768198356243, -0.11123009623270576, 0.19958552967932905, 0.02335948850884561, 0.06284773627534035, 0.09913562714566883, 0.026569338243576993, -0.29799215731854467, -0.02945768220431876, 0.006387965774914891, -0.059680324910581, 0.22305144612002062, 0.33979324690258955, -0.15373829686581042, -0.02869269866115274, -0.0291025811334324, -0.026628403759817686, 0.20913166445505835, 0.04275570397121452, 0.00861121504002218, 0.12739298903716487, -0.19538391870889987, 0.028519054953177806, -0.08172613813082646, 0.06773111201814995, -0.07289695649383057, 0.2348354686228396, -0.1791243236258637, 0.10667147621586978, 0.3751749819156953, -0.11924100069186716, -0.2927652021376737, -0.10684861782185531, 0.17911088040947595, -0.11649462309081006]])

    def __init__(self, num_variables, phenome_preprocessor=None, **kwargs):
        self.is_deterministic = True
        self.do_maximize = False
        self.num_variables = num_variables
        self.min_bounds = [-100.0] * num_variables
        self.max_bounds = [100.0] * num_variables
        self.matrices = {2: self.matrix2D, 10: self.matrix10D, 30: self.matrix30D, 50: self.matrix50D}
        self.ellipsoid_function = EllipsoidFunction(1.0e6)
        bounds = (self.min_bounds, self.max_bounds)
        preprocessor = BoundConstraintsChecker(bounds, phenome_preprocessor)
        TestProblem.__init__(self, self.objective_function,
                             phenome_preprocessor=preprocessor,
                             **kwargs)


    def objective_function(self, phenome):
        phenome = np.array([phene - offset for offset, phene in zip(self.offsets, phenome)])
        phenome = np.dot(phenome, self.matrices[self.num_variables])
        assert len(phenome) == self.num_variables
        obj_value = self.ellipsoid_function(phenome) + self.bias
        return obj_value


    def get_optimal_solutions(self, max_number=None):
        return [Individual(self.offsets[:self.num_variables])]



class F4(F2):
    """Shifted double-sum with noise."""

    def __init__(self, num_variables, phenome_preprocessor=None, **kwargs):
        F2.__init__(self, num_variables, phenome_preprocessor, **kwargs)
        self.is_deterministic = False


    def objective_function(self, phenome):
        obj_value = F2.objective_function(self, phenome) - self.bias
        obj_value *= (1.0 + 0.4 * abs(random.gauss(0, 1)))
        obj_value += self.bias
        return obj_value



class F5(TestProblem):
    """Schwefel's problem 2.6 with global optimum on bounds."""

    bias = -310.0

    offsets = [-5.5559, 7.947, -1.538, 8.3897, 7.7182, -8.3147, -2.1423,
               -2.4392, -3.3787, -7.3047, 3.058, 6.7613, 2.3444, 5.6514,
               1.0491, -0.8324, 1.3039, -0.0651, 0.0424, -6.5176, -8.6977,
               2.7053, -1.4842, -8.8158, 5.6475, -4.5999, 3.6337, -6.4068,
               4.8867, 8.2225, 6.6873, -5.8862, 2.5925, 0.8027, 7.5525, 4.2621,
               3.5091, -2.6055, -8.4063, 6.1947, -6.5024, -8.144, -4.8444,
               3.1572, 3.9624, -8.4969, 6.2642, 1.1448, 3.9132, 3.614, -8.4785,
               0.2955, 3.5597, -5.5854, -8.9173, 4.0627, 8.387, -3.768, 5.9001,
               3.8212, -6.4771, 6.8886, -2.4951, 2.5007, 1.3866, -7.3488,
               -1.5349, 2.9223, 7.1813, -4.799, 8.3061, -8.7911, -6.3035,
               -6.1222, 5.7116, 7.4369, 6.1738, -7.1118, 3.7062, 6.1274,1.3696,
               -4.5894, 4.5844, -1.2133, 4.18, 0.3337, 7.3355, 8.16, 1.4422,
               7.9909, -8.1183, -6.8285, -2.9079, 0.2028, -6.2375, 0.1294,
               0.1474, 2.6981, 0.4233, 6.1942]


    A = np.array([[-89.0, -28.0, -73.0, 49.0, 50.0, 81.0, 21.0, -22.0, -35.0, 74.0, 62.0, 18.0, 16.0, 6.0, 94.0, 58.0, -83.0, 70.0, -5.0, -16.0, 80.0, -67.0, -7.0, -20.0, -27.0, 75.0, -9.0, -24.0, 73.0, 18.0, 82.0, -44.0, -20.0, -3.0, 72.0, 52.0, 80.0, -11.0, -31.0, 19.0, 91.0, -49.0, -30.0, 64.0, 90.0, 35.0, 72.0, 16.0, 27.0, 36.0, -46.0, 74.0, -35.0, 36.0, 63.0, 46.0, -73.0, 73.0, 38.0, 89.0, -68.0, -43.0, 98.0, 57.0, 70.0, -15.0, -59.0, 2.0, -29.0, 55.0, 31.0, 29.0, -59.0, 31.0, -11.0, 18.0, 47.0, 87.0, 72.0, 44.0, -55.0, -55.0, 93.0, 57.0, 95.0, -24.0, 15.0, 2.0, 60.0, -79.0, 33.0, -25.0, -63.0, 36.0, 93.0, -87.0, 37.0, -33.0, -61.0, 24.0],
                  [8.0, -23.0, -1.0, -98.0, -94.0, -53.0, -85.0, 92.0, 57.0, 77.0, 41.0, -39.0, 96.0, -61.0, 97.0, -62.0, -84.0, -99.0, -74.0, 15.0, 61.0, 87.0, -77.0, -71.0, 64.0, 12.0, -76.0, -18.0, 27.0, 15.0, 24.0, 58.0, -68.0, -32.0, 26.0, -65.0, -45.0, -76.0, -42.0, 5.0, 97.0, 5.0, -22.0, -53.0, -48.0, -61.0, 77.0, 82.0, -18.0, 58.0, -81.0, 79.0, -91.0, -6.0, -58.0, 26.0, -80.0, -50.0, -63.0, -14.0, 33.0, 82.0, -35.0, 16.0, 9.0, -24.0, -73.0, -50.0, -5.0, -5.0, 24.0, -76.0, -32.0, -7.0, 28.0, -63.0, 13.0, 73.0, -59.0, -98.0, -83.0, 56.0, -69.0, 14.0, -7.0, 47.0, 85.0, -72.0, -29.0, -3.0, 79.0, 91.0, -87.0, -27.0, -17.0, -48.0, -11.0, -58.0, 49.0, -26.0],
                  [-28.0, 49.0, -9.0, 80.0, 50.0, -53.0, -9.0, -83.0, -71.0, 3.0, 65.0, 31.0, -39.0, -42.0, 53.0, -79.0, -83.0, -69.0, -94.0, 41.0, -21.0, -17.0, 88.0, -63.0, -97.0, -62.0, 45.0, 89.0, 15.0, -27.0, 96.0, -79.0, -99.0, -96.0, -49.0, 97.0, -38.0, -33.0, -3.0, 66.0, -8.0, -32.0, -8.0, -39.0, -52.0, 82.0, -26.0, 50.0, 13.0, 93.0, -94.0, 30.0, 43.0, 21.0, 13.0, -53.0, -30.0, 94.0, 72.0, 59.0, 75.0, -89.0, -65.0, 34.0, 2.0, -58.0, -53.0, 8.0, -18.0, -22.0, 54.0, -58.0, 41.0, 52.0, -79.0, 49.0, -33.0, 93.0, -5.0, 91.0, -15.0, -59.0, -39.0, 24.0, 27.0, 98.0, -28.0, 26.0, 41.0, 50.0, -46.0, 38.0, -93.0, 0.0, 57.0, -73.0, 27.0, -38.0, 69.0, 25.0],
                  [33.0, 59.0, 87.0, 40.0, 37.0, 68.0, -54.0, -97.0, 21.0, 75.0, 75.0, 46.0, -25.0, -31.0, 65.0, -96.0, -46.0, 77.0, -98.0, 3.0, 46.0, -48.0, 53.0, 65.0, 45.0, -43.0, 22.0, 96.0, -53.0, 99.0, -26.0, 97.0, 91.0, 83.0, -33.0, -93.0, -73.0, 32.0, 8.0, -81.0, -92.0, -24.0, 63.0, 12.0, -75.0, 77.0, -51.0, -86.0, 35.0, -34.0, 13.0, 39.0, 92.0, 79.0, -82.0, -12.0, 7.0, 0.0, 75.0, 91.0, -65.0, 24.0, 43.0, 69.0, 71.0, 40.0, -2.0, 44.0, 4.0, 89.0, 91.0, 50.0, 30.0, 64.0, -2.0, 87.0, -65.0, 98.0, 25.0, 12.0, -97.0, -70.0, -69.0, -52.0, 86.0, -48.0, 19.0, 29.0, -9.0, -71.0, -45.0, -48.0, -54.0, -24.0, -68.0, -40.0, 71.0, -49.0, -47.0, 81.0],
                  [39.0, 35.0, 8.0, -43.0, 82.0, 67.0, -58.0, 22.0, -98.0, 68.0, -20.0, -29.0, 11.0, 70.0, 77.0, 72.0, 24.0, -14.0, -83.0, -82.0, 96.0, 12.0, -78.0, -65.0, -26.0, -74.0, -83.0, -49.0, -91.0, 35.0, -99.0, 16.0, -68.0, 83.0, -10.0, -72.0, 7.0, 5.0, -79.0, 35.0, -69.0, 81.0, 22.0, -12.0, -73.0, 39.0, 72.0, 24.0, 4.0, 61.0, -47.0, 87.0, -31.0, 89.0, 1.0, 91.0, 64.0, 56.0, -39.0, -54.0, -99.0, 71.0, 93.0, -2.0, -48.0, 48.0, 65.0, -81.0, 53.0, 28.0, 35.0, 13.0, 92.0, 82.0, -63.0, -49.0, -85.0, 26.0, -51.0, 23.0, -54.0, 93.0, 45.0, 13.0, -92.0, 34.0, -67.0, 49.0, 24.0, 21.0, -83.0, -5.0, 69.0, -58.0, -20.0, 70.0, 87.0, 31.0, -79.0, 61.0],
                  [51.0, 80.0, 4.0, -28.0, -46.0, 22.0, -44.0, 33.0, -57.0, -39.0, -37.0, 74.0, 47.0, 60.0, -83.0, 84.0, -77.0, -95.0, -47.0, -43.0, 17.0, 0.0, 11.0, 68.0, 4.0, 36.0, 56.0, 52.0, 26.0, -44.0, 33.0, 83.0, 79.0, 41.0, -22.0, 10.0, 71.0, -59.0, 48.0, 98.0, 84.0, 91.0, 63.0, -95.0, -25.0, -52.0, 59.0, 88.0, 64.0, -92.0, 90.0, -23.0, -72.0, -45.0, -23.0, 22.0, -76.0, -2.0, -74.0, -81.0, -41.0, 10.0, 25.0, -5.0, 78.0, -52.0, -70.0, 92.0, -19.0, 2.0, 74.0, 44.0, -36.0, 1.0, 66.0, 61.0, -14.0, -59.0, -3.0, 0.0, 17.0, 66.0, 19.0, 56.0, 2.0, 37.0, 81.0, -48.0, -89.0, 24.0, -100.0, -45.0, -4.0, 64.0, 64.0, -19.0, 46.0, 67.0, 64.0, 53.0],
                  [-18.0, -48.0, 13.0, 50.0, -24.0, -89.0, 35.0, 21.0, 64.0, -70.0, -24.0, -18.0, -20.0, 6.0, 28.0, 94.0, -22.0, -62.0, -84.0, -11.0, -36.0, 53.0, 57.0, -78.0, -30.0, -93.0, -44.0, -56.0, -62.0, 89.0, 6.0, 89.0, -22.0, -73.0, 92.0, 84.0, 51.0, 73.0, -77.0, -22.0, -74.0, 96.0, 83.0, -32.0, -60.0, -43.0, -58.0, 31.0, 95.0, 1.0, -32.0, 32.0, 0.0, -96.0, 53.0, -46.0, 62.0, -22.0, 31.0, -47.0, 73.0, 75.0, 63.0, 47.0, 19.0, 89.0, -58.0, -25.0, 58.0, 93.0, -11.0, 90.0, 40.0, -5.0, 17.0, -69.0, 10.0, 87.0, 3.0, 93.0, 0.0, -46.0, 22.0, -5.0, 94.0, 43.0, 15.0, 39.0, -50.0, 65.0, -23.0, -4.0, -22.0, -83.0, -66.0, -68.0, 81.0, 85.0, -76.0, 7.0],
                  [-2.0, 35.0, 27.0, 36.0, -44.0, 3.0, -84.0, 33.0, 71.0, -71.0, 56.0, 66.0, 49.0, -70.0, -86.0, 67.0, 77.0, 93.0, 80.0, -5.0, 98.0, -82.0, 58.0, -38.0, -70.0, -32.0, -46.0, 40.0, -7.0, -65.0, 63.0, 10.0, 58.0, 36.0, -6.0, -12.0, 41.0, 79.0, 31.0, -15.0, 46.0, -12.0, 54.0, 90.0, 43.0, 4.0, -85.0, 23.0, -59.0, 52.0, 93.0, -71.0, -61.0, -3.0, -81.0, 5.0, 54.0, -58.0, 70.0, 48.0, 56.0, -37.0, 21.0, 94.0, -44.0, 31.0, -62.0, -83.0, 49.0, 12.0, -96.0, 71.0, -4.0, 76.0, 97.0, -39.0, 62.0, 80.0, 33.0, -52.0, -34.0, -23.0, 7.0, -84.0, -69.0, 99.0, 38.0, 24.0, -30.0, -84.0, -94.0, 13.0, 6.0, 33.0, -21.0, 12.0, 67.0, -11.0, 33.0, 3.0],
                  [19.0, -28.0, -47.0, -94.0, 53.0, 52.0, 29.0, 59.0, -44.0, 14.0, -15.0, -35.0, -70.0, -42.0, 44.0, 46.0, -72.0, 34.0, 4.0, 35.0, 47.0, -61.0, 9.0, 39.0, -69.0, 14.0, -72.0, -12.0, 9.0, 91.0, 73.0, 14.0, 24.0, -4.0, 42.0, -14.0, 4.0, 80.0, -30.0, 76.0, 29.0, 25.0, 40.0, 16.0, -12.0, 44.0, 57.0, -80.0, 51.0, 43.0, 14.0, -59.0, 73.0, -64.0, 91.0, -88.0, -93.0, 96.0, 56.0, -21.0, -96.0, -26.0, 82.0, -42.0, 13.0, -48.0, -15.0, -23.0, 45.0, 35.0, 81.0, -3.0, 28.0, 27.0, 34.0, -70.0, 49.0, 18.0, 14.0, 90.0, 100.0, -66.0, 23.0, -95.0, -8.0, 98.0, -63.0, -76.0, 4.0, -58.0, 57.0, -43.0, 34.0, 50.0, -16.0, 97.0, 71.0, 15.0, 63.0, -55.0],
                  [16.0, -26.0, 52.0, -4.0, -65.0, -11.0, -97.0, -80.0, 18.0, -64.0, -93.0, 69.0, 56.0, 91.0, -44.0, -71.0, -49.0, 30.0, -32.0, -83.0, -34.0, 40.0, 83.0, 15.0, 26.0, 82.0, 6.0, 57.0, 80.0, 68.0, 2.0, -88.0, -7.0, 7.0, -15.0, 55.0, 96.0, -6.0, 80.0, -72.0, 78.0, -41.0, 84.0, -7.0, 2.0, -72.0, 50.0, 31.0, -5.0, -86.0, -19.0, 75.0, -26.0, -4.0, 6.0, 57.0, 56.0, -66.0, -30.0, -68.0, 92.0, -19.0, -23.0, 57.0, 78.0, 23.0, -67.0, 66.0, -36.0, -89.0, 23.0, -7.0, 71.0, -55.0, 49.0, -54.0, 89.0, -71.0, -85.0, 90.0, -21.0, -42.0, 70.0, 9.0, -75.0, -92.0, 86.0, -11.0, 66.0, -10.0, 39.0, 88.0, -4.0, 88.0, 73.0, 15.0, 83.0, -50.0, 94.0, 60.0],
                  [54.0, 36.0, -1.0, 84.0, 83.0, -66.0, 66.0, -53.0, 1.0, -91.0, 73.0, -71.0, 23.0, 42.0, 41.0, -34.0, 36.0, 82.0, 56.0, 42.0, -5.0, -2.0, 87.0, 25.0, 40.0, 24.0, 77.0, 52.0, 4.0, -96.0, -64.0, -70.0, -96.0, -26.0, 50.0, 88.0, 83.0, -57.0, 69.0, -93.0, 19.0, -4.0, -70.0, 100.0, 84.0, 83.0, 56.0, -34.0, 37.0, 33.0, 0.0, -96.0, -69.0, -12.0, 98.0, 81.0, -26.0, 69.0, 69.0, 61.0, 24.0, -19.0, 0.0, 78.0, -32.0, -70.0, 46.0, -12.0, 22.0, -34.0, -10.0, -32.0, 55.0, -9.0, -60.0, 22.0, -87.0, 39.0, -45.0, 9.0, 3.0, 7.0, 73.0, 47.0, 8.0, -41.0, 82.0, -41.0, 94.0, -14.0, -56.0, 8.0, -33.0, -51.0, 34.0, -48.0, -48.0, 41.0, 7.0, 60.0],
                  [21.0, 58.0, -18.0, -30.0, 51.0, -29.0, 91.0, 16.0, 28.0, 62.0, -75.0, 45.0, -68.0, -62.0, 18.0, -38.0, -65.0, -67.0, -41.0, -39.0, -30.0, 76.0, 58.0, 11.0, -49.0, -86.0, -45.0, -93.0, 41.0, 30.0, -16.0, -48.0, -7.0, -67.0, -39.0, -96.0, -92.0, -28.0, -4.0, -95.0, 99.0, -36.0, 85.0, 64.0, -33.0, 38.0, 17.0, 96.0, 64.0, 12.0, 35.0, 90.0, -22.0, -28.0, -37.0, 58.0, 48.0, 23.0, -52.0, -2.0, -12.0, 41.0, 24.0, 20.0, -1.0, -99.0, 8.0, -49.0, -49.0, 43.0, 37.0, -59.0, 57.0, -61.0, 47.0, -31.0, -43.0, -87.0, -22.0, -80.0, 72.0, -45.0, 1.0, -61.0, -13.0, 92.0, 22.0, -2.0, 90.0, -77.0, -33.0, 34.0, 85.0, 94.0, 55.0, -26.0, 89.0, 80.0, -54.0, 85.0],
                  [99.0, 47.0, 2.0, -55.0, 100.0, -22.0, 5.0, 38.0, 88.0, 47.0, -92.0, 50.0, 96.0, -34.0, 11.0, -7.0, -17.0, 98.0, 94.0, -58.0, -94.0, -2.0, 83.0, -99.0, 18.0, -27.0, -74.0, -43.0, 43.0, 18.0, 68.0, -27.0, 42.0, 21.0, -73.0, -87.0, -94.0, -98.0, 73.0, -96.0, 95.0, 25.0, 27.0, 51.0, 74.0, 25.0, -54.0, -10.0, 44.0, -3.0, 41.0, 6.0, 78.0, 65.0, -64.0, -75.0, 54.0, 86.0, -92.0, -93.0, 41.0, 70.0, -45.0, 92.0, -83.0, 20.0, 39.0, 30.0, -13.0, 90.0, -45.0, -84.0, -45.0, -69.0, 8.0, -17.0, 18.0, 43.0, 98.0, -23.0, 28.0, 35.0, 34.0, 66.0, 16.0, 54.0, 40.0, -17.0, -93.0, 39.0, -69.0, -2.0, -47.0, -95.0, 80.0, -64.0, -25.0, -42.0, 28.0, 26.0],
                  [18.0, 69.0, -63.0, -2.0, -43.0, -18.0, 55.0, 82.0, 21.0, -89.0, -95.0, 63.0, 60.0, -65.0, -5.0, 7.0, -67.0, -95.0, -34.0, 40.0, -25.0, 10.0, -55.0, 49.0, 21.0, 37.0, 64.0, -5.0, -15.0, -85.0, -67.0, 94.0, -57.0, -53.0, -40.0, -81.0, -82.0, -69.0, -79.0, 97.0, -69.0, -30.0, 93.0, -18.0, 64.0, 24.0, -53.0, 35.0, 64.0, 21.0, -77.0, 33.0, -63.0, -29.0, 41.0, 59.0, -83.0, 59.0, -49.0, 80.0, -41.0, -6.0, -70.0, -78.0, 73.0, -21.0, -21.0, 59.0, -66.0, 86.0, -34.0, 36.0, 35.0, 25.0, 52.0, 27.0, -28.0, -89.0, 65.0, 16.0, 1.0, -82.0, 81.0, 70.0, 93.0, 44.0, -79.0, -91.0, -82.0, -80.0, -54.0, -68.0, 53.0, -88.0, -92.0, -78.0, -8.0, -91.0, 43.0, 68.0],
                  [55.0, -62.0, 77.0, 97.0, -87.0, -7.0, 15.0, 55.0, -22.0, -89.0, 18.0, 27.0, -97.0, 60.0, 6.0, 77.0, -45.0, 33.0, 9.0, -73.0, -23.0, 54.0, 64.0, 52.0, -96.0, 78.0, 40.0, 65.0, -82.0, 66.0, 53.0, -78.0, 15.0, 83.0, 81.0, 13.0, -23.0, 42.0, 95.0, -85.0, -7.0, -42.0, 57.0, 66.0, -92.0, 54.0, -38.0, -24.0, 26.0, 41.0, -97.0, 3.0, -81.0, 12.0, -13.0, 29.0, -12.0, 52.0, 39.0, 84.0, 19.0, -26.0, 11.0, 77.0, -83.0, 96.0, 71.0, -87.0, -35.0, -78.0, -9.0, -90.0, -64.0, -97.0, 39.0, 36.0, -84.0, -45.0, -26.0, -62.0, 2.0, 43.0, -93.0, -80.0, -86.0, 86.0, -12.0, 13.0, 38.0, 74.0, -10.0, 47.0, -36.0, -32.0, 48.0, 99.0, -3.0, 31.0, -13.0, -36.0],
                  [-28.0, -33.0, -97.0, -67.0, 98.0, -44.0, 89.0, -26.0, 9.0, -22.0, -8.0, -78.0, -11.0, 29.0, 93.0, -88.0, 56.0, 3.0, -35.0, 6.0, 72.0, -10.0, -22.0, 33.0, 88.0, 80.0, -38.0, -28.0, 36.0, -40.0, -79.0, -49.0, 62.0, 59.0, 87.0, -44.0, -48.0, -45.0, -55.0, 29.0, -39.0, -32.0, 63.0, -29.0, -37.0, -10.0, 19.0, -26.0, -5.0, -90.0, -30.0, -9.0, -1.0, 52.0, -58.0, 59.0, -68.0, -98.0, -50.0, -31.0, 71.0, -17.0, 13.0, 93.0, -28.0, -63.0, 82.0, -70.0, -11.0, 66.0, 62.0, -73.0, -59.0, -4.0, 31.0, -82.0, -71.0, -72.0, -99.0, -12.0, -78.0, -20.0, -76.0, 35.0, 94.0, 36.0, -89.0, 1.0, -38.0, -22.0, -39.0, -77.0, -89.0, -37.0, -24.0, -31.0, -95.0, -71.0, -12.0, 89.0],
                  [92.0, 35.0, 96.0, -13.0, -6.0, -47.0, 16.0, -25.0, -7.0, 27.0, 42.0, -19.0, -41.0, -72.0, 76.0, -61.0, 84.0, 6.0, -87.0, 31.0, 36.0, -50.0, -67.0, -82.0, 84.0, 71.0, 21.0, -64.0, -28.0, 1.0, -63.0, 53.0, -72.0, -54.0, -40.0, -89.0, 21.0, 100.0, 31.0, -82.0, -1.0, -13.0, 25.0, -36.0, -2.0, 21.0, -59.0, 44.0, 33.0, -18.0, -13.0, 5.0, -27.0, -33.0, 8.0, -59.0, 35.0, -71.0, 68.0, 95.0, 20.0, -13.0, -1.0, 75.0, -94.0, -71.0, 90.0, -31.0, 10.0, -53.0, -24.0, 50.0, -79.0, 79.0, -48.0, -43.0, -1.0, -92.0, 77.0, -11.0, -35.0, 21.0, 15.0, 70.0, 85.0, 60.0, -58.0, -64.0, 60.0, -31.0, -60.0, 28.0, 58.0, 45.0, -63.0, 6.0, -34.0, 77.0, 92.0, 20.0],
                  [82.0, -24.0, -36.0, 42.0, 60.0, 56.0, -56.0, 51.0, -95.0, -33.0, 57.0, 67.0, -82.0, -23.0, 93.0, 85.0, -33.0, 40.0, 67.0, 53.0, 36.0, 69.0, -88.0, -51.0, 44.0, -62.0, -77.0, -66.0, 45.0, -63.0, -2.0, -2.0, 37.0, -80.0, 86.0, -66.0, -13.0, 51.0, -84.0, 51.0, 55.0, -70.0, 76.0, 88.0, 50.0, 10.0, 18.0, 72.0, -16.0, 10.0, 9.0, -30.0, -89.0, 13.0, -2.0, -12.0, -23.0, 73.0, -88.0, 43.0, 7.0, -57.0, 90.0, -4.0, 3.0, 84.0, 4.0, 0.0, -37.0, 62.0, 8.0, -12.0, 18.0, 83.0, 1.0, 97.0, -70.0, -99.0, 98.0, -86.0, 45.0, -67.0, -11.0, 90.0, 67.0, -49.0, 26.0, 71.0, -50.0, 93.0, 97.0, -58.0, -69.0, -17.0, 11.0, 22.0, -96.0, 90.0, 6.0, -7.0],
                  [-92.0, 75.0, -12.0, 32.0, -32.0, 58.0, -57.0, 33.0, -74.0, 42.0, -28.0, 8.0, 83.0, -29.0, 32.0, 31.0, 73.0, 60.0, 26.0, -91.0, 59.0, -3.0, 28.0, -9.0, -31.0, -17.0, -90.0, 29.0, -68.0, -33.0, 32.0, -33.0, -88.0, -22.0, 70.0, -24.0, 36.0, 39.0, 74.0, -80.0, 85.0, -77.0, 26.0, -54.0, 78.0, -59.0, 7.0, -2.0, -37.0, -64.0, 53.0, 62.0, 26.0, 48.0, 51.0, 6.0, 72.0, 25.0, -99.0, 3.0, 31.0, 33.0, -31.0, 60.0, -30.0, 99.0, -45.0, -66.0, 35.0, -15.0, 84.0, 6.0, 5.0, -13.0, 80.0, 80.0, -51.0, 63.0, -50.0, 52.0, 46.0, -50.0, -52.0, -1.0, -49.0, 36.0, 37.0, 68.0, -53.0, -54.0, -77.0, -21.0, -73.0, -40.0, 87.0, -35.0, 47.0, -19.0, 92.0, 55.0],
                  [31.0, 29.0, 77.0, -33.0, 51.0, -2.0, 54.0, -86.0, 45.0, -32.0, 92.0, -61.0, 5.0, 39.0, -25.0, 43.0, -65.0, -48.0, -21.0, 82.0, 8.0, 58.0, 79.0, -75.0, 91.0, 33.0, -8.0, -78.0, -37.0, -42.0, -41.0, -26.0, 44.0, 9.0, 79.0, -92.0, 85.0, 9.0, 56.0, 47.0, -20.0, 39.0, -97.0, 16.0, -30.0, 10.0, -17.0, 56.0, -40.0, -10.0, 67.0, 89.0, -54.0, 84.0, -8.0, 43.0, 84.0, -47.0, -50.0, -41.0, 43.0, 71.0, 75.0, 73.0, 69.0, 87.0, -73.0, -56.0, -74.0, 32.0, -36.0, -67.0, -97.0, -28.0, -62.0, 36.0, 8.0, -12.0, 64.0, 66.0, 60.0, -58.0, 76.0, 19.0, 15.0, -3.0, 85.0, -35.0, 46.0, 62.0, -100.0, -85.0, -93.0, 17.0, -28.0, -76.0, 98.0, -15.0, -28.0, 0.0],
                  [-53.0, -19.0, -70.0, -96.0, 6.0, -61.0, 75.0, 78.0, 34.0, -10.0, 9.0, 88.0, 51.0, -42.0, -87.0, -80.0, 19.0, 46.0, 35.0, 65.0, 21.0, -90.0, -38.0, 82.0, 75.0, 64.0, -55.0, 33.0, 29.0, 19.0, 86.0, 72.0, -33.0, 96.0, 29.0, 93.0, 28.0, 31.0, -40.0, -14.0, 34.0, -99.0, 77.0, -62.0, -46.0, 60.0, -77.0, 34.0, -68.0, 78.0, 11.0, 8.0, -3.0, 57.0, 49.0, -39.0, 0.0, -75.0, 59.0, -31.0, -55.0, 82.0, -34.0, 30.0, -2.0, -3.0, 69.0, 80.0, 6.0, -62.0, -10.0, -86.0, 86.0, 90.0, 1.0, 81.0, 62.0, -26.0, 57.0, 24.0, 72.0, 28.0, 63.0, -21.0, -79.0, 0.0, -18.0, -64.0, -22.0, -37.0, -67.0, -25.0, 50.0, 88.0, 46.0, 42.0, -93.0, 1.0, -68.0, 81.0],
                  [82.0, 71.0, -52.0, -22.0, 12.0, 20.0, -61.0, 67.0, 62.0, 56.0, -31.0, -51.0, 36.0, -59.0, -76.0, 64.0, -26.0, 74.0, -40.0, 90.0, 68.0, 54.0, 32.0, -51.0, -65.0, 98.0, -44.0, -53.0, 72.0, -18.0, -66.0, 24.0, -24.0, -32.0, 65.0, 69.0, 77.0, 47.0, 63.0, 37.0, -44.0, 43.0, 92.0, 80.0, -66.0, 83.0, 57.0, 67.0, -28.0, 25.0, -39.0, 23.0, -22.0, -39.0, -96.0, 21.0, 4.0, 12.0, 7.0, 55.0, -18.0, -64.0, -7.0, -14.0, 3.0, -72.0, 91.0, -42.0, -79.0, -60.0, -57.0, 96.0, -30.0, 1.0, 46.0, 11.0, -43.0, 0.0, -69.0, -44.0, -62.0, -59.0, -82.0, -61.0, 65.0, 0.0, -75.0, 17.0, 90.0, -94.0, -73.0, 9.0, -93.0, 8.0, -18.0, 75.0, 63.0, 76.0, -25.0, 33.0],
                  [83.0, -51.0, 19.0, -73.0, 39.0, -12.0, -40.0, -68.0, -82.0, -47.0, -76.0, 47.0, -29.0, 42.0, 28.0, -11.0, -74.0, 23.0, -32.0, -10.0, -24.0, -29.0, 62.0, -3.0, 92.0, -6.0, 74.0, -2.0, -19.0, -53.0, 62.0, 16.0, 23.0, -43.0, -23.0, 18.0, -58.0, 5.0, 41.0, -28.0, -99.0, 38.0, -81.0, -48.0, 18.0, 35.0, 56.0, -91.0, 12.0, -77.0, 93.0, -50.0, -11.0, -79.0, 71.0, -30.0, 1.0, 90.0, 38.0, 50.0, 1.0, 60.0, -54.0, -77.0, -5.0, -86.0, -63.0, 85.0, -46.0, -38.0, 42.0, -81.0, 54.0, 53.0, -60.0, 92.0, -46.0, 83.0, 68.0, 49.0, 7.0, 64.0, -64.0, -9.0, -90.0, 63.0, 44.0, 81.0, -97.0, -82.0, -85.0, -41.0, -50.0, 93.0, 0.0, -63.0, 7.0, 56.0, 20.0, 57.0],
                  [29.0, -92.0, 90.0, 29.0, -39.0, -42.0, -89.0, -86.0, -79.0, 68.0, -99.0, 94.0, -73.0, 36.0, -85.0, -34.0, 73.0, -65.0, -25.0, -25.0, -67.0, 20.0, 37.0, -39.0, -77.0, -44.0, 98.0, 84.0, 78.0, -18.0, 92.0, -64.0, -64.0, 95.0, 69.0, 79.0, 31.0, -11.0, 89.0, 39.0, 0.0, -31.0, -37.0, 52.0, -71.0, -91.0, -18.0, 29.0, -99.0, -33.0, 63.0, 82.0, -23.0, -62.0, -75.0, 48.0, -95.0, 83.0, 53.0, 5.0, 74.0, 55.0, 94.0, 30.0, 55.0, 4.0, 3.0, 34.0, -79.0, 94.0, -96.0, -34.0, -13.0, -69.0, 11.0, 60.0, 45.0, -24.0, 78.0, 36.0, -65.0, 30.0, 46.0, 28.0, 34.0, 56.0, 24.0, -7.0, 69.0, 12.0, -1.0, 88.0, -8.0, 71.0, -24.0, -67.0, 70.0, -98.0, 61.0, -81.0],
                  [-22.0, -8.0, 67.0, 40.0, -39.0, 47.0, 46.0, 86.0, -30.0, -59.0, -41.0, 35.0, -85.0, -75.0, 0.0, 59.0, -12.0, -97.0, -5.0, -32.0, 37.0, 3.0, -19.0, -74.0, -42.0, -7.0, -30.0, -25.0, -7.0, -51.0, 32.0, -21.0, -5.0, -56.0, -5.0, 67.0, 81.0, -64.0, -84.0, 90.0, 89.0, 16.0, -92.0, 17.0, -98.0, -1.0, -21.0, -99.0, -18.0, 8.0, -46.0, -1.0, -92.0, -10.0, -97.0, 98.0, -72.0, 62.0, 15.0, -57.0, 24.0, 43.0, -87.0, 85.0, 29.0, 80.0, 31.0, 33.0, 48.0, 0.0, -98.0, 82.0, -86.0, -12.0, -29.0, -94.0, -69.0, -20.0, -37.0, -8.0, -96.0, 93.0, -18.0, -69.0, 11.0, 53.0, -46.0, 16.0, -81.0, 57.0, 47.0, 74.0, -86.0, 18.0, 73.0, 35.0, 22.0, -89.0, 22.0, 30.0],
                  [1.0, -20.0, 63.0, -45.0, -91.0, -82.0, -13.0, -45.0, 83.0, 42.0, -83.0, 27.0, 61.0, 51.0, -74.0, 21.0, -68.0, -48.0, -83.0, -37.0, 12.0, 74.0, -43.0, -86.0, -2.0, 37.0, -41.0, -65.0, 23.0, -41.0, -52.0, -36.0, 12.0, -42.0, 70.0, 61.0, -23.0, -27.0, 77.0, -84.0, 22.0, 53.0, -99.0, -36.0, -11.0, -87.0, -86.0, -62.0, 97.0, -51.0, -51.0, -52.0, -15.0, -29.0, 100.0, -20.0, -41.0, 50.0, -53.0, 71.0, -32.0, -43.0, 83.0, -51.0, -26.0, -36.0, -90.0, 76.0, -35.0, -91.0, 38.0, 0.0, 45.0, -87.0, -25.0, -8.0, -88.0, 18.0, -37.0, -44.0, 33.0, -96.0, -24.0, -69.0, 79.0, 58.0, -97.0, -64.0, 46.0, -59.0, -40.0, -56.0, 3.0, 75.0, 2.0, -23.0, -89.0, 34.0, -73.0, -72.0],
                  [20.0, 9.0, 59.0, -44.0, 14.0, 62.0, 63.0, 28.0, 81.0, 93.0, 33.0, 28.0, 85.0, 14.0, -18.0, 92.0, 81.0, -37.0, -98.0, 44.0, -52.0, -11.0, -88.0, 85.0, -70.0, 83.0, 7.0, 60.0, -26.0, -16.0, -75.0, 11.0, -47.0, 47.0, 19.0, -1.0, -80.0, -30.0, 94.0, -95.0, 28.0, -30.0, 62.0, 81.0, 35.0, 27.0, 69.0, 63.0, -4.0, 17.0, -26.0, -25.0, 18.0, 53.0, 16.0, -37.0, 17.0, 80.0, 85.0, 20.0, -81.0, 85.0, 45.0, -17.0, 15.0, -71.0, 15.0, -37.0, -30.0, 47.0, -96.0, -59.0, 65.0, -5.0, 51.0, 79.0, -3.0, -65.0, 84.0, -61.0, -26.0, 56.0, 52.0, 56.0, -5.0, -55.0, 61.0, 2.0, -74.0, -8.0, 98.0, 96.0, 33.0, 4.0, 69.0, 64.0, -35.0, 75.0, 46.0, 77.0],
                  [-57.0, 8.0, 20.0, -63.0, -41.0, 48.0, -40.0, 57.0, -48.0, 89.0, 22.0, -66.0, 24.0, 7.0, 31.0, 41.0, 56.0, -77.0, -78.0, -51.0, -65.0, -4.0, 12.0, -27.0, -44.0, -76.0, 86.0, -89.0, 12.0, 88.0, -31.0, -69.0, -22.0, -27.0, -68.0, -41.0, 99.0, 65.0, 58.0, -29.0, -42.0, 13.0, -8.0, 27.0, 83.0, 28.0, 38.0, 61.0, -4.0, -34.0, 50.0, 31.0, 59.0, 58.0, -21.0, -69.0, -99.0, -28.0, 69.0, 35.0, 30.0, 6.0, 23.0, 80.0, -2.0, 43.0, -44.0, -38.0, -97.0, 81.0, -21.0, -16.0, 42.0, 77.0, -11.0, -85.0, -41.0, 21.0, -74.0, 59.0, 26.0, -74.0, -29.0, 58.0, 96.0, -23.0, -40.0, -52.0, 91.0, -65.0, 2.0, 52.0, 56.0, 72.0, -35.0, -32.0, -5.0, 19.0, -75.0, -78.0],
                  [-1.0, -9.0, -30.0, -2.0, -25.0, -90.0, 5.0, -39.0, 32.0, 9.0, -64.0, 81.0, 22.0, -99.0, 93.0, -98.0, -23.0, 54.0, 57.0, 42.0, -87.0, 47.0, -73.0, 52.0, -71.0, -15.0, -19.0, 75.0, 77.0, 79.0, -30.0, -92.0, 39.0, 1.0, -6.0, 2.0, -37.0, 96.0, 61.0, -38.0, 1.0, 9.0, 73.0, -73.0, 18.0, 87.0, -14.0, -99.0, -42.0, 0.0, -60.0, 10.0, -30.0, -94.0, 14.0, -43.0, -47.0, 56.0, 97.0, -25.0, 34.0, 81.0, -91.0, -49.0, -45.0, 87.0, 19.0, -89.0, -12.0, 9.0, 68.0, 84.0, 28.0, 93.0, 7.0, -6.0, 68.0, -15.0, -50.0, -38.0, -13.0, -97.0, 86.0, 79.0, -74.0, 63.0, -47.0, 11.0, -34.0, 4.0, -69.0, -68.0, 37.0, -12.0, 75.0, 98.0, -79.0, -19.0, 99.0, -7.0],
                  [-30.0, 89.0, 92.0, 83.0, -60.0, -49.0, 36.0, -34.0, 91.0, 61.0, -69.0, -39.0, 69.0, 21.0, -73.0, 76.0, -7.0, -9.0, 69.0, -81.0, -82.0, -64.0, -26.0, -99.0, 42.0, -95.0, -4.0, -10.0, 92.0, 97.0, 4.0, -63.0, 27.0, -78.0, 75.0, -45.0, -86.0, -74.0, 74.0, -95.0, 81.0, 62.0, -10.0, -11.0, 61.0, -91.0, -86.0, -90.0, 69.0, 42.0, 10.0, 26.0, -8.0, 4.0, 19.0, -55.0, -94.0, 56.0, -66.0, -65.0, 27.0, -79.0, -6.0, 2.0, -93.0, 79.0, -50.0, 17.0, 18.0, -61.0, 43.0, -17.0, -77.0, 18.0, 88.0, -31.0, 38.0, -78.0, -35.0, 55.0, 61.0, -89.0, -61.0, -36.0, -69.0, 31.0, -57.0, 59.0, 58.0, -84.0, 30.0, 42.0, -53.0, 88.0, -33.0, 39.0, -46.0, -25.0, 86.0, 76.0],
                  [-71.0, 38.0, 82.0, -54.0, 26.0, 51.0, -84.0, 12.0, 44.0, -61.0, 70.0, -60.0, -74.0, 82.0, -51.0, 36.0, -5.0, 22.0, 88.0, -69.0, 68.0, 85.0, 13.0, 100.0, 20.0, 26.0, -38.0, 12.0, -92.0, 40.0, 56.0, 45.0, -60.0, 62.0, 4.0, -23.0, -4.0, 42.0, -34.0, -93.0, 72.0, 44.0, 0.0, -75.0, -16.0, 7.0, -21.0, -55.0, 92.0, 6.0, 51.0, 33.0, -81.0, 11.0, 55.0, -27.0, -92.0, 91.0, -49.0, -66.0, 7.0, -63.0, 58.0, 20.0, -35.0, -97.0, -95.0, 95.0, 12.0, 18.0, 24.0, 39.0, -92.0, 38.0, -37.0, -48.0, -99.0, -64.0, -35.0, 22.0, 93.0, -20.0, 97.0, -76.0, -10.0, -63.0, 80.0, -79.0, 56.0, 87.0, 14.0, -97.0, 0.0, -66.0, 37.0, 81.0, 71.0, -67.0, -3.0, -60.0],
                  [42.0, 96.0, -68.0, 6.0, 37.0, 26.0, -11.0, -25.0, 6.0, 36.0, -29.0, 53.0, -75.0, -8.0, 77.0, 36.0, -34.0, -96.0, -68.0, 8.0, -64.0, 23.0, 38.0, 64.0, -85.0, 45.0, -42.0, 38.0, -34.0, -4.0, -16.0, 50.0, 88.0, 25.0, -18.0, 67.0, -89.0, 1.0, 36.0, 54.0, -66.0, -77.0, -55.0, -16.0, -62.0, 3.0, 16.0, -92.0, 43.0, 99.0, 36.0, -66.0, 99.0, -97.0, 14.0, 28.0, -98.0, 34.0, 0.0, -19.0, 42.0, -49.0, -61.0, -32.0, -47.0, -86.0, -47.0, -79.0, -8.0, -94.0, -15.0, -74.0, -65.0, 80.0, 83.0, -10.0, 64.0, -36.0, 86.0, -67.0, 35.0, -15.0, 17.0, 76.0, 88.0, -34.0, 28.0, 55.0, 84.0, 27.0, 67.0, 13.0, -82.0, 55.0, 40.0, -78.0, 80.0, 30.0, -51.0, 34.0],
                  [54.0, 7.0, -14.0, 44.0, -32.0, 97.0, -48.0, 56.0, -31.0, -93.0, -5.0, -57.0, -94.0, 45.0, 51.0, -86.0, -8.0, 77.0, -57.0, -32.0, -64.0, -39.0, 53.0, -98.0, -69.0, -27.0, -5.0, -46.0, -87.0, -66.0, 94.0, -96.0, 80.0, 70.0, 0.0, 7.0, -73.0, 11.0, 73.0, 82.0, 9.0, -89.0, -84.0, -51.0, 70.0, 75.0, 80.0, -94.0, -35.0, 7.0, 61.0, -54.0, -64.0, -64.0, -21.0, 72.0, 15.0, 9.0, -21.0, -56.0, 68.0, 23.0, 18.0, -4.0, -16.0, -55.0, -23.0, 42.0, -61.0, -1.0, -6.0, -7.0, -12.0, 63.0, -69.0, 42.0, -5.0, -64.0, -10.0, -41.0, -89.0, -68.0, 50.0, -31.0, -49.0, 30.0, -21.0, 65.0, -31.0, -6.0, -70.0, 97.0, -8.0, -83.0, -77.0, -40.0, -31.0, -16.0, 70.0, -51.0],
                  [-44.0, -21.0, 3.0, 72.0, -56.0, 55.0, -32.0, 54.0, 2.0, 30.0, 17.0, 11.0, -51.0, -36.0, -56.0, -92.0, 55.0, -24.0, -69.0, -45.0, -49.0, -80.0, -90.0, -81.0, 57.0, 11.0, -51.0, -99.0, 74.0, 5.0, -21.0, -91.0, -96.0, -42.0, -65.0, -54.0, 7.0, -65.0, 43.0, -64.0, -56.0, 65.0, -98.0, -42.0, 33.0, 86.0, -37.0, 33.0, 55.0, -44.0, -83.0, 23.0, 27.0, -49.0, -7.0, -2.0, 6.0, 62.0, 22.0, 81.0, 3.0, 82.0, 16.0, -32.0, -63.0, 73.0, 53.0, -88.0, -45.0, -53.0, 27.0, -78.0, -3.0, -74.0, 78.0, 63.0, 8.0, 17.0, -54.0, 4.0, 55.0, 84.0, 98.0, 48.0, -93.0, 95.0, -69.0, -82.0, 51.0, -89.0, 6.0, -16.0, 38.0, 66.0, -73.0, 0.0, -51.0, -52.0, -59.0, -99.0],
                  [-11.0, -39.0, 6.0, 12.0, -35.0, 88.0, -41.0, 13.0, -49.0, -79.0, -9.0, -44.0, -44.0, 79.0, 50.0, 17.0, -53.0, 37.0, 82.0, -99.0, 10.0, -67.0, -94.0, 29.0, -50.0, -56.0, -48.0, 31.0, -88.0, -84.0, -55.0, 37.0, -5.0, 55.0, -97.0, -13.0, -38.0, -55.0, -75.0, -85.0, -81.0, 63.0, -35.0, 69.0, 25.0, -66.0, 83.0, -67.0, 28.0, -49.0, -97.0, -96.0, -92.0, 28.0, -16.0, 18.0, -49.0, -1.0, 38.0, -5.0, 18.0, -20.0, 51.0, -60.0, -99.0, -62.0, -94.0, 48.0, 2.0, 93.0, 96.0, -48.0, -36.0, 78.0, 58.0, -80.0, 98.0, 21.0, 30.0, 95.0, -47.0, -11.0, -59.0, -39.0, 9.0, 90.0, 68.0, -67.0, 79.0, -95.0, -98.0, 19.0, -24.0, -97.0, -95.0, -2.0, -74.0, 93.0, -92.0, 24.0],
                  [90.0, 8.0, -78.0, 31.0, 39.0, 81.0, -68.0, -55.0, 14.0, -81.0, 22.0, 66.0, 69.0, -61.0, -55.0, 67.0, -15.0, -39.0, 95.0, 18.0, 74.0, -33.0, 27.0, 21.0, 41.0, 5.0, -45.0, -37.0, 13.0, -20.0, -86.0, -2.0, -36.0, -61.0, 48.0, -1.0, 52.0, -25.0, -24.0, -56.0, 82.0, -28.0, 14.0, 56.0, 53.0, -48.0, -11.0, 19.0, -37.0, 75.0, -100.0, -45.0, 3.0, -100.0, -23.0, 83.0, 93.0, 58.0, 0.0, 5.0, 68.0, -4.0, -62.0, -18.0, 93.0, -29.0, -19.0, -80.0, -78.0, -70.0, -89.0, 22.0, 82.0, 42.0, 2.0, 3.0, -24.0, 2.0, 74.0, 77.0, -26.0, 86.0, 77.0, 46.0, 51.0, -13.0, 39.0, -52.0, -67.0, -8.0, 47.0, -2.0, 53.0, -67.0, 39.0, -71.0, -99.0, -31.0, 37.0, 46.0],
                  [-29.0, 66.0, -100.0, 89.0, -38.0, -12.0, 73.0, 12.0, 70.0, -82.0, -28.0, 31.0, -52.0, 19.0, 4.0, -43.0, 89.0, -9.0, 85.0, -67.0, -33.0, -52.0, 86.0, -57.0, 64.0, 64.0, -28.0, -13.0, -15.0, -60.0, -95.0, -26.0, -77.0, -17.0, 90.0, 65.0, 92.0, 25.0, -52.0, 7.0, 34.0, -87.0, -53.0, -78.0, -71.0, -57.0, -9.0, -16.0, -91.0, 89.0, -84.0, 60.0, 0.0, -96.0, -48.0, 94.0, -62.0, -43.0, -28.0, 86.0, 28.0, -8.0, 5.0, 3.0, 5.0, -80.0, 30.0, -88.0, -68.0, 52.0, -32.0, 84.0, -32.0, -81.0, 78.0, 65.0, 91.0, -29.0, -60.0, -4.0, 34.0, 36.0, 39.0, 72.0, -30.0, 49.0, 20.0, 53.0, -12.0, -69.0, 48.0, -95.0, 6.0, 51.0, -6.0, 33.0, -18.0, 27.0, 44.0, 80.0],
                  [76.0, 63.0, -16.0, -33.0, -43.0, 91.0, 80.0, -63.0, -81.0, 76.0, -8.0, 99.0, 40.0, -11.0, -16.0, -33.0, -64.0, 99.0, -86.0, 52.0, -73.0, -70.0, 37.0, 45.0, 24.0, 30.0, -22.0, -50.0, 93.0, -52.0, -97.0, -87.0, 60.0, -96.0, 99.0, 43.0, 38.0, -56.0, 24.0, -91.0, 49.0, 55.0, 96.0, 14.0, 32.0, -33.0, -24.0, 3.0, 82.0, -41.0, -47.0, -64.0, -54.0, -81.0, 2.0, -3.0, -69.0, -9.0, -51.0, -78.0, -39.0, 77.0, -6.0, 3.0, -88.0, -87.0, -58.0, -45.0, 11.0, 17.0, -46.0, 67.0, -90.0, 83.0, -68.0, 7.0, 73.0, -92.0, -88.0, -2.0, -14.0, -36.0, 72.0, -14.0, -25.0, -29.0, -67.0, 3.0, -91.0, 54.0, -77.0, -24.0, -42.0, -47.0, 90.0, -96.0, 61.0, -88.0, -51.0, -61.0],
                  [22.0, 61.0, -19.0, 25.0, -100.0, 18.0, -84.0, -94.0, -87.0, 26.0, 97.0, 46.0, 63.0, -43.0, 67.0, -44.0, 4.0, -69.0, 3.0, -15.0, 71.0, 24.0, -5.0, -12.0, 27.0, 19.0, -40.0, 54.0, 82.0, 63.0, 62.0, -83.0, 55.0, 68.0, 49.0, -44.0, -90.0, -86.0, -15.0, -99.0, 55.0, -65.0, -56.0, -72.0, -34.0, 14.0, 8.0, -4.0, 93.0, 52.0, -42.0, 35.0, -2.0, 26.0, -59.0, 45.0, -98.0, -72.0, -47.0, -73.0, 86.0, -3.0, 14.0, 15.0, -42.0, -71.0, -80.0, 66.0, -70.0, 77.0, -82.0, 4.0, 89.0, -51.0, -83.0, -43.0, -88.0, -96.0, -47.0, -72.0, 3.0, -36.0, 5.0, 81.0, -79.0, -22.0, -84.0, 95.0, -51.0, -46.0, -62.0, -51.0, -85.0, 70.0, 91.0, 58.0, 24.0, -80.0, -18.0, 87.0],
                  [-23.0, -93.0, -46.0, 81.0, 67.0, -59.0, 18.0, 37.0, 43.0, -83.0, 0.0, -39.0, -83.0, -57.0, -100.0, -14.0, -62.0, 47.0, -64.0, -91.0, -69.0, 82.0, -76.0, -24.0, 43.0, 32.0, 54.0, 24.0, 58.0, -88.0, 7.0, 9.0, -48.0, 99.0, -77.0, -11.0, 56.0, 41.0, -80.0, -29.0, 29.0, -96.0, 77.0, 24.0, -7.0, 75.0, 40.0, -38.0, 34.0, 68.0, -17.0, -75.0, -95.0, -64.0, -60.0, 6.0, 0.0, -37.0, 71.0, -2.0, -85.0, -52.0, -64.0, 90.0, -20.0, -85.0, 88.0, 16.0, 11.0, 69.0, 38.0, -53.0, 77.0, -27.0, 77.0, 33.0, 5.0, 98.0, 78.0, -44.0, 61.0, 41.0, 40.0, -72.0, 46.0, 86.0, 11.0, 79.0, 20.0, 45.0, 8.0, -32.0, -19.0, 42.0, 71.0, -62.0, -95.0, -39.0, -58.0, 59.0],
                  [-45.0, -36.0, -95.0, 21.0, 99.0, 37.0, -53.0, 33.0, -21.0, 59.0, -50.0, -63.0, 86.0, 29.0, 88.0, -1.0, -51.0, 14.0, 70.0, 55.0, 98.0, 12.0, 14.0, -33.0, 40.0, 49.0, 66.0, 31.0, -87.0, 45.0, -51.0, -25.0, 56.0, 19.0, -58.0, 73.0, -21.0, 37.0, -22.0, 14.0, 58.0, -25.0, 44.0, 9.0, -99.0, 99.0, -73.0, -35.0, 95.0, 63.0, -68.0, -57.0, -59.0, 96.0, 37.0, 49.0, -42.0, -66.0, -31.0, 69.0, 89.0, -43.0, 85.0, -75.0, 31.0, -41.0, -69.0, -73.0, -21.0, -62.0, -17.0, 83.0, -59.0, 67.0, -19.0, -15.0, -72.0, -30.0, 69.0, 49.0, 64.0, 72.0, 11.0, -1.0, -20.0, -91.0, -57.0, 37.0, -63.0, 98.0, -91.0, 83.0, 48.0, -33.0, 42.0, 75.0, 2.0, -11.0, -49.0, 92.0],
                  [91.0, -30.0, -45.0, -7.0, -83.0, -52.0, -71.0, 77.0, -9.0, 84.0, -15.0, 33.0, 60.0, -83.0, -51.0, -80.0, -91.0, 8.0, 57.0, 98.0, 65.0, -10.0, -10.0, -55.0, 91.0, -83.0, -30.0, 56.0, 72.0, -75.0, -20.0, -92.0, -96.0, -18.0, 51.0, 81.0, 5.0, -20.0, 30.0, 60.0, -40.0, -23.0, 84.0, 75.0, 89.0, -80.0, 24.0, 8.0, 98.0, -62.0, 65.0, 63.0, -36.0, 46.0, -81.0, -27.0, 4.0, -60.0, -71.0, 37.0, -93.0, 45.0, 36.0, -97.0, 25.0, -97.0, -16.0, -13.0, 88.0, -61.0, 70.0, -55.0, 41.0, 76.0, 32.0, -45.0, 1.0, 30.0, 86.0, -25.0, 75.0, -67.0, -77.0, -23.0, -50.0, 36.0, 30.0, -2.0, 96.0, -60.0, -31.0, 43.0, 7.0, -52.0, 87.0, -87.0, -91.0, 19.0, -46.0, -53.0],
                  [94.0, 93.0, 59.0, 75.0, 56.0, -15.0, -67.0, -28.0, 15.0, 22.0, -62.0, -90.0, 46.0, -15.0, 84.0, 55.0, 45.0, -37.0, -78.0, 52.0, -99.0, 48.0, 84.0, 20.0, -53.0, -51.0, -56.0, -60.0, -7.0, -13.0, -47.0, -52.0, 31.0, 89.0, 12.0, 62.0, 81.0, 95.0, -89.0, -83.0, 60.0, 41.0, 73.0, 72.0, 33.0, -50.0, 77.0, 36.0, -34.0, 50.0, -5.0, -72.0, 4.0, 20.0, -44.0, -25.0, 11.0, 91.0, -57.0, 47.0, 14.0, 22.0, -89.0, -73.0, -81.0, 26.0, -13.0, -28.0, -9.0, 78.0, 1.0, -3.0, 26.0, 20.0, 16.0, -21.0, 25.0, -88.0, -6.0, -9.0, 6.0, 43.0, 76.0, -41.0, 57.0, 89.0, 20.0, -48.0, -18.0, 52.0, -24.0, 58.0, 78.0, 27.0, 96.0, -21.0, 88.0, 72.0, 64.0, 52.0],
                  [61.0, -91.0, -48.0, -43.0, 31.0, 48.0, -30.0, -92.0, -53.0, 43.0, 31.0, -12.0, 99.0, -52.0, -68.0, 19.0, 58.0, -69.0, -88.0, 10.0, -13.0, -19.0, 56.0, 33.0, -73.0, 92.0, 98.0, -19.0, 90.0, 87.0, -13.0, 52.0, -27.0, 80.0, -81.0, 91.0, -61.0, 83.0, 62.0, -12.0, 78.0, 77.0, -56.0, -76.0, 45.0, 48.0, 50.0, -31.0, 30.0, -81.0, -52.0, 14.0, -36.0, 87.0, 6.0, 94.0, 0.0, 82.0, 98.0, 75.0, -77.0, 36.0, 19.0, 36.0, -35.0, 44.0, 90.0, 17.0, 96.0, 63.0, -100.0, -9.0, -76.0, -77.0, 96.0, 41.0, 25.0, -17.0, -50.0, 30.0, -61.0, 85.0, -79.0, -79.0, -28.0, 34.0, -13.0, 92.0, -66.0, 99.0, 78.0, 15.0, -11.0, 59.0, 51.0, -59.0, 59.0, -36.0, 44.0, 45.0],
                  [29.0, -9.0, -50.0, -82.0, 23.0, 3.0, -58.0, -23.0, -44.0, 98.0, 56.0, -59.0, 75.0, -50.0, -73.0, -49.0, 75.0, -98.0, -34.0, 93.0, 3.0, 58.0, 36.0, 47.0, 54.0, -78.0, -16.0, -68.0, 8.0, 65.0, -15.0, 21.0, 79.0, 90.0, 10.0, 6.0, 60.0, -59.0, 78.0, -51.0, -99.0, 86.0, -14.0, 78.0, -53.0, 63.0, -94.0, -49.0, 89.0, -79.0, -55.0, 46.0, -67.0, 93.0, -77.0, -67.0, 75.0, 80.0, -37.0, -16.0, -69.0, 8.0, -65.0, -68.0, -12.0, -60.0, 57.0, -15.0, 91.0, 50.0, 77.0, -86.0, 34.0, 40.0, -18.0, -66.0, -14.0, -72.0, 57.0, -13.0, -60.0, 13.0, -38.0, 96.0, 7.0, -64.0, 84.0, -69.0, -60.0, 65.0, -73.0, 32.0, -65.0, 38.0, -58.0, 63.0, -69.0, 20.0, 87.0, 65.0],
                  [-27.0, -37.0, -41.0, 0.0, 46.0, -29.0, -72.0, -55.0, 23.0, 57.0, -24.0, 93.0, 85.0, -96.0, -38.0, -42.0, -68.0, 29.0, 90.0, -35.0, 59.0, 10.0, -46.0, 65.0, -31.0, -42.0, -73.0, -32.0, -19.0, -39.0, 92.0, 82.0, 78.0, -96.0, -64.0, 82.0, -45.0, -45.0, 75.0, -88.0, -67.0, 99.0, -7.0, -71.0, -28.0, -48.0, 53.0, 6.0, -96.0, -92.0, -4.0, -34.0, 65.0, 3.0, 11.0, -45.0, -38.0, 66.0, 84.0, 41.0, 21.0, 16.0, -64.0, 21.0, 16.0, 75.0, -77.0, 4.0, 25.0, 15.0, -5.0, -35.0, 72.0, 67.0, -74.0, 35.0, 52.0, 74.0, -89.0, 63.0, 94.0, -64.0, -18.0, -43.0, 73.0, 27.0, -35.0, -31.0, -46.0, 39.0, 19.0, 73.0, 14.0, -57.0, -1.0, -59.0, 51.0, 98.0, -27.0, -71.0],
                  [50.0, 48.0, 22.0, -1.0, -14.0, -82.0, 38.0, -3.0, 87.0, 61.0, -42.0, -64.0, 14.0, 30.0, -75.0, 32.0, -96.0, -39.0, -45.0, -70.0, 60.0, 6.0, -77.0, 61.0, 42.0, -28.0, -97.0, 0.0, 72.0, 47.0, 16.0, -55.0, 91.0, 34.0, -59.0, -18.0, -17.0, 61.0, -27.0, 13.0, 35.0, -96.0, -34.0, 20.0, -9.0, -17.0, -78.0, 69.0, -40.0, -94.0, -85.0, -92.0, 13.0, -93.0, -89.0, -4.0, -10.0, 86.0, 48.0, 80.0, -63.0, -4.0, 34.0, -38.0, -46.0, 62.0, 6.0, 79.0, 17.0, 42.0, -5.0, -80.0, -35.0, 77.0, -7.0, 44.0, 16.0, 65.0, 30.0, 29.0, 7.0, -91.0, 93.0, 14.0, 86.0, -23.0, -95.0, 37.0, 18.0, -41.0, -98.0, 14.0, -47.0, 0.0, 60.0, -66.0, 69.0, -50.0, 14.0, -53.0],
                  [74.0, -38.0, 17.0, 17.0, 2.0, -43.0, -20.0, 25.0, -56.0, 60.0, 49.0, 30.0, -88.0, 1.0, -50.0, 92.0, 10.0, 16.0, -62.0, 32.0, 32.0, -33.0, -19.0, 77.0, -81.0, 25.0, -63.0, -5.0, 63.0, 35.0, -22.0, 43.0, 49.0, -31.0, 37.0, 55.0, -90.0, -4.0, -89.0, 30.0, -92.0, -7.0, 30.0, -6.0, 9.0, 50.0, -42.0, -25.0, -60.0, 13.0, -73.0, 27.0, -15.0, -18.0, -15.0, -72.0, -76.0, 87.0, -82.0, -4.0, 99.0, 25.0, 29.0, 69.0, 39.0, 64.0, 80.0, -6.0, 14.0, -88.0, 86.0, 80.0, -89.0, 46.0, -89.0, -98.0, 54.0, -39.0, -64.0, 92.0, -39.0, 19.0, -89.0, -4.0, -21.0, 5.0, 32.0, -76.0, 16.0, -77.0, 36.0, 58.0, 40.0, -14.0, -75.0, -11.0, 59.0, -36.0, 71.0, 79.0],
                  [71.0, -80.0, 79.0, -68.0, 59.0, 42.0, 8.0, -53.0, 71.0, -67.0, -77.0, -10.0, 46.0, 0.0, 6.0, 64.0, 22.0, 70.0, 62.0, 67.0, 43.0, -21.0, 73.0, -92.0, 10.0, -92.0, -90.0, -6.0, 69.0, 8.0, 90.0, 94.0, 64.0, 99.0, -57.0, -98.0, -11.0, 97.0, -33.0, -93.0, -12.0, 8.0, 75.0, -30.0, -96.0, -64.0, 10.0, 9.0, 10.0, -96.0, 56.0, 15.0, -4.0, 96.0, -19.0, -35.0, 91.0, -29.0, 92.0, -97.0, -3.0, -21.0, -32.0, 34.0, 67.0, 69.0, 23.0, 19.0, -79.0, -44.0, -69.0, 64.0, 63.0, -7.0, -62.0, 63.0, -15.0, 36.0, -88.0, -94.0, 45.0, 71.0, 95.0, -13.0, 10.0, -11.0, 74.0, 11.0, -3.0, -58.0, 98.0, -53.0, 15.0, -88.0, -94.0, -84.0, -48.0, 62.0, 40.0, -34.0],
                  [-30.0, 85.0, 48.0, -60.0, 34.0, -13.0, -87.0, -96.0, -88.0, 17.0, -28.0, 98.0, -23.0, 8.0, 37.0, 41.0, -92.0, -92.0, 92.0, 15.0, 65.0, 14.0, 53.0, -38.0, -62.0, -5.0, 0.0, -97.0, 91.0, -97.0, -76.0, 62.0, 55.0, 55.0, 23.0, -60.0, 73.0, -39.0, -55.0, 54.0, 85.0, 58.0, 16.0, -21.0, -72.0, 9.0, 50.0, 16.0, 85.0, -26.0, -48.0, -93.0, 33.0, -96.0, -11.0, 46.0, -65.0, 26.0, 1.0, -57.0, 91.0, 32.0, -58.0, -27.0, 7.0, -100.0, 55.0, -32.0, -8.0, -65.0, 64.0, -19.0, -18.0, -96.0, -96.0, -70.0, 36.0, -74.0, 6.0, -35.0, 13.0, 64.0, 58.0, 79.0, -50.0, 2.0, -20.0, 86.0, -90.0, 26.0, -90.0, -96.0, -52.0, 66.0, -43.0, 64.0, 1.0, -44.0, -12.0, 91.0],
                  [-49.0, 45.0, 19.0, -27.0, 25.0, -58.0, -90.0, 18.0, 44.0, 23.0, 82.0, 85.0, 54.0, 9.0, -17.0, -52.0, -3.0, 2.0, -54.0, 29.0, 79.0, 26.0, 42.0, 91.0, 24.0, -49.0, 18.0, 6.0, -87.0, -66.0, 78.0, 93.0, 39.0, -53.0, 66.0, 95.0, 2.0, 23.0, 57.0, -83.0, -8.0, 88.0, 8.0, 24.0, -15.0, 73.0, -48.0, -20.0, 62.0, -65.0, 54.0, -62.0, -95.0, -19.0, 15.0, 89.0, 99.0, 49.0, 62.0, -51.0, 24.0, 26.0, 41.0, 6.0, 75.0, 76.0, -46.0, -9.0, -86.0, -72.0, -87.0, 14.0, -46.0, 55.0, 19.0, -19.0, -84.0, 29.0, 60.0, 5.0, 69.0, 15.0, -64.0, 13.0, -54.0, 52.0, -86.0, -41.0, -65.0, 51.0, 59.0, -20.0, 4.0, -81.0, 22.0, -43.0, -74.0, -79.0, 50.0, 66.0],
                  [95.0, 17.0, 49.0, -96.0, -24.0, -14.0, 29.0, -79.0, -14.0, 21.0, -82.0, 35.0, -97.0, -78.0, 68.0, 52.0, 53.0, 12.0, 12.0, 18.0, -1.0, -62.0, 8.0, -31.0, -28.0, -8.0, -50.0, -44.0, 83.0, 2.0, 50.0, -10.0, 85.0, -31.0, 44.0, -49.0, 90.0, 0.0, -99.0, -21.0, -99.0, 9.0, 33.0, 17.0, -32.0, 3.0, -49.0, -43.0, -1.0, 78.0, 93.0, 32.0, 46.0, 23.0, -71.0, -68.0, -15.0, 94.0, 17.0, -24.0, -40.0, -28.0, -86.0, -9.0, 37.0, -3.0, 4.0, 52.0, -74.0, -12.0, 75.0, -70.0, 9.0, -63.0, -91.0, 51.0, 21.0, 76.0, -58.0, 53.0, 81.0, 3.0, 73.0, 36.0, 51.0, 79.0, -72.0, 59.0, -6.0, -97.0, 43.0, 95.0, -12.0, -87.0, 68.0, 27.0, -78.0, -9.0, -1.0, 77.0],
                  [24.0, -60.0, -40.0, 74.0, 74.0, -45.0, 29.0, -9.0, 72.0, 18.0, -85.0, -29.0, -63.0, 86.0, -4.0, 84.0, 16.0, -33.0, 78.0, -47.0, -89.0, -25.0, 7.0, -15.0, -29.0, -96.0, 74.0, -13.0, -4.0, -30.0, 82.0, -99.0, 32.0, -7.0, -23.0, -87.0, 54.0, -45.0, 5.0, 3.0, -91.0, 55.0, 25.0, 3.0, 21.0, 80.0, -81.0, -74.0, -97.0, -27.0, -94.0, 99.0, 4.0, -22.0, -90.0, -14.0, 69.0, -69.0, -52.0, -71.0, -74.0, -58.0, -58.0, -91.0, -39.0, -81.0, -64.0, 75.0, 99.0, -84.0, -6.0, 83.0, 57.0, 24.0, 50.0, 42.0, -73.0, -47.0, 80.0, -33.0, 50.0, 87.0, -25.0, -95.0, -48.0, -22.0, -71.0, 20.0, 13.0, 99.0, 23.0, 19.0, 80.0, 64.0, -26.0, -95.0, 97.0, -7.0, -38.0, -23.0],
                  [-81.0, 12.0, 99.0, 38.0, -60.0, -19.0, 34.0, 20.0, 9.0, -79.0, -11.0, -73.0, 79.0, 18.0, 12.0, -46.0, -76.0, -80.0, 2.0, -87.0, -15.0, -83.0, -49.0, -12.0, 87.0, -22.0, 16.0, -91.0, -15.0, 77.0, -87.0, 42.0, 30.0, -14.0, 62.0, -89.0, 53.0, 51.0, 56.0, 85.0, 4.0, -100.0, -81.0, 37.0, 72.0, -8.0, 7.0, 66.0, 9.0, 65.0, -10.0, 5.0, 10.0, 49.0, 58.0, -98.0, -93.0, -47.0, 46.0, -72.0, 91.0, 91.0, -76.0, 22.0, -23.0, 49.0, 65.0, -59.0, -83.0, 36.0, 28.0, -96.0, -65.0, 98.0, 8.0, 4.0, -7.0, 82.0, -54.0, -82.0, 67.0, 86.0, -7.0, 56.0, 28.0, 68.0, 14.0, -20.0, 41.0, 10.0, 78.0, 84.0, 85.0, -9.0, 29.0, -68.0, 100.0, 5.0, 93.0, -38.0],
                  [97.0, 54.0, 43.0, 29.0, -71.0, 9.0, 25.0, -43.0, -72.0, 82.0, 85.0, 13.0, -37.0, 2.0, 86.0, -56.0, -21.0, -51.0, 25.0, -43.0, 15.0, -41.0, 8.0, -84.0, 27.0, -89.0, -13.0, -63.0, 63.0, 16.0, 42.0, -92.0, -18.0, -23.0, 9.0, 55.0, -7.0, 76.0, -75.0, 65.0, -67.0, 88.0, 40.0, -15.0, -67.0, 15.0, 91.0, 91.0, 35.0, -24.0, -13.0, -46.0, -99.0, -86.0, -79.0, -23.0, 51.0, -58.0, 48.0, -71.0, -85.0, 56.0, -31.0, -15.0, 64.0, 67.0, 99.0, 63.0, -43.0, -57.0, -18.0, -66.0, 37.0, -100.0, -20.0, -66.0, -38.0, 68.0, -24.0, 64.0, 54.0, 27.0, 94.0, -38.0, -73.0, -56.0, -35.0, -88.0, 17.0, -31.0, -49.0, -71.0, -52.0, 56.0, -77.0, -93.0, -38.0, -50.0, 28.0, 28.0],
                  [46.0, 83.0, -75.0, 79.0, -32.0, 15.0, -8.0, 62.0, -61.0, -50.0, -45.0, 67.0, 5.0, -53.0, -20.0, 12.0, -43.0, -9.0, -62.0, 29.0, -25.0, 88.0, 7.0, 65.0, -99.0, -41.0, -32.0, 80.0, -6.0, -74.0, 23.0, 2.0, -13.0, -86.0, 41.0, 9.0, 23.0, 86.0, -34.0, -84.0, 8.0, 24.0, 94.0, 49.0, 37.0, -2.0, -59.0, -2.0, -56.0, -31.0, 85.0, 35.0, -88.0, 21.0, -17.0, -52.0, -98.0, -99.0, -89.0, -55.0, -73.0, -73.0, 16.0, -11.0, 69.0, -50.0, -63.0, 7.0, -79.0, -83.0, -15.0, 27.0, -46.0, 82.0, 55.0, -13.0, 33.0, 0.0, 32.0, -42.0, 14.0, 6.0, -26.0, -2.0, -53.0, 90.0, -74.0, 86.0, 16.0, 17.0, 97.0, 99.0, 83.0, -35.0, 40.0, 4.0, 17.0, 90.0, 42.0, -30.0],
                  [-26.0, 12.0, -51.0, -71.0, -2.0, -49.0, -1.0, 23.0, 79.0, 84.0, -18.0, 78.0, -56.0, -4.0, -93.0, 38.0, 35.0, 15.0, 67.0, -38.0, -41.0, 92.0, -41.0, 93.0, 3.0, -87.0, -98.0, 58.0, 54.0, -77.0, -33.0, 26.0, -48.0, 76.0, 58.0, -11.0, 12.0, 10.0, 23.0, -80.0, -17.0, 98.0, 57.0, -55.0, 10.0, 35.0, 75.0, -66.0, -8.0, -58.0, -32.0, 12.0, 38.0, 30.0, 67.0, -88.0, 20.0, -95.0, -9.0, -65.0, -34.0, 62.0, 85.0, 59.0, -30.0, -9.0, 80.0, 33.0, -80.0, 24.0, 48.0, 60.0, 57.0, 65.0, 24.0, 18.0, -14.0, -88.0, -19.0, 24.0, 62.0, 78.0, -62.0, -92.0, -42.0, -34.0, 78.0, -43.0, 0.0, 78.0, 89.0, -68.0, 47.0, -82.0, 23.0, 73.0, 67.0, -27.0, 48.0, -74.0],
                  [-52.0, -11.0, 26.0, 93.0, -65.0, -1.0, -83.0, 59.0, -55.0, 80.0, 58.0, 10.0, -25.0, -89.0, -63.0, -22.0, 45.0, 3.0, -87.0, -53.0, 90.0, 6.0, -5.0, -91.0, 84.0, -30.0, 22.0, 29.0, -38.0, 39.0, -75.0, 27.0, 99.0, -75.0, 31.0, 56.0, 93.0, 14.0, -45.0, -100.0, -42.0, -93.0, 44.0, 25.0, 28.0, 40.0, -21.0, -36.0, -29.0, -30.0, 12.0, 47.0, 80.0, -95.0, -97.0, -80.0, -12.0, 56.0, 87.0, 54.0, 40.0, 91.0, 96.0, -88.0, -31.0, 49.0, -89.0, -3.0, -85.0, -79.0, 59.0, 92.0, -6.0, 54.0, 92.0, 0.0, 42.0, 90.0, 35.0, 59.0, 33.0, 63.0, -25.0, -32.0, -28.0, 33.0, -40.0, -27.0, 100.0, 73.0, -93.0, 65.0, 95.0, -93.0, -17.0, -10.0, 26.0, -5.0, -87.0, 87.0],
                  [51.0, 12.0, 57.0, 50.0, -43.0, -71.0, -20.0, -84.0, -77.0, 92.0, -85.0, -18.0, 76.0, -40.0, -19.0, 23.0, 88.0, -17.0, -10.0, -8.0, 72.0, 83.0, -68.0, 1.0, 80.0, 12.0, 20.0, 13.0, -2.0, -48.0, -67.0, -1.0, 87.0, 68.0, 12.0, -34.0, -12.0, -88.0, 97.0, 5.0, -37.0, 41.0, -71.0, -35.0, -72.0, 97.0, -31.0, 80.0, 99.0, 89.0, 35.0, 3.0, 2.0, -90.0, -28.0, 73.0, -15.0, -97.0, 54.0, 81.0, 45.0, -58.0, 33.0, 55.0, 71.0, -43.0, -13.0, -30.0, -12.0, 64.0, 35.0, 62.0, -67.0, 13.0, 93.0, -66.0, -92.0, -52.0, 83.0, 66.0, -100.0, 49.0, 87.0, 11.0, -89.0, 43.0, -50.0, 0.0, -78.0, 77.0, -79.0, 44.0, -25.0, 84.0, 82.0, -67.0, 28.0, -49.0, -21.0, -13.0],
                  [66.0, 6.0, 100.0, 24.0, -10.0, 1.0, 46.0, 53.0, 82.0, 48.0, -41.0, -49.0, -100.0, -56.0, -58.0, -44.0, -17.0, -23.0, -88.0, 75.0, 37.0, 52.0, 47.0, 82.0, 32.0, 11.0, 29.0, -95.0, 48.0, 48.0, 10.0, 92.0, -32.0, 77.0, 24.0, 47.0, 50.0, 55.0, -23.0, -89.0, -69.0, -15.0, 25.0, -29.0, 89.0, -92.0, 99.0, -21.0, 22.0, 87.0, -47.0, 88.0, -84.0, -64.0, 28.0, 94.0, -58.0, -84.0, 65.0, -86.0, 73.0, 76.0, 39.0, -73.0, -52.0, 94.0, -67.0, 1.0, -37.0, 1.0, -61.0, -38.0, 84.0, -52.0, 83.0, -94.0, -65.0, -18.0, 64.0, -44.0, -42.0, 97.0, -39.0, 91.0, 58.0, -65.0, -76.0, -36.0, -8.0, 86.0, -35.0, 25.0, -35.0, -5.0, -50.0, 9.0, 22.0, 20.0, -90.0, 91.0],
                  [-40.0, 4.0, -65.0, 58.0, 85.0, 70.0, -34.0, -40.0, 69.0, -36.0, -44.0, 41.0, 6.0, -75.0, 52.0, 86.0, 29.0, 56.0, -28.0, -55.0, -42.0, -35.0, 57.0, 67.0, 98.0, -3.0, 7.0, -32.0, -99.0, -64.0, -22.0, 72.0, 21.0, 22.0, -83.0, 7.0, -18.0, 12.0, 7.0, 79.0, -52.0, 77.0, 62.0, -86.0, -77.0, 11.0, -7.0, 28.0, -82.0, 85.0, 30.0, -90.0, 1.0, -67.0, -85.0, 40.0, 2.0, 41.0, 12.0, -12.0, 55.0, 89.0, 59.0, -65.0, 91.0, -38.0, -44.0, 37.0, 79.0, -61.0, -39.0, 75.0, 32.0, 0.0, -26.0, -74.0, 35.0, 16.0, 42.0, 61.0, 65.0, 20.0, -96.0, 51.0, 78.0, 2.0, 6.0, 16.0, 22.0, -81.0, 75.0, 69.0, 38.0, -2.0, 33.0, 15.0, -66.0, 40.0, 97.0, -20.0],
                  [-91.0, -13.0, -93.0, 64.0, 10.0, 33.0, 97.0, 27.0, 63.0, -1.0, -83.0, -30.0, 9.0, 85.0, 26.0, -9.0, 91.0, -30.0, -58.0, 63.0, -72.0, 38.0, 36.0, 47.0, 14.0, -63.0, -91.0, -32.0, 5.0, 27.0, 8.0, 89.0, 78.0, -65.0, 56.0, 65.0, -43.0, -82.0, -85.0, 92.0, 44.0, 20.0, -3.0, 67.0, -22.0, -44.0, -85.0, 81.0, -71.0, 89.0, -49.0, -94.0, 83.0, -22.0, 85.0, -87.0, 53.0, -49.0, 1.0, 97.0, 31.0, 62.0, 41.0, -19.0, 30.0, 57.0, 19.0, 47.0, 15.0, -21.0, 17.0, -63.0, 4.0, -80.0, -42.0, -55.0, 30.0, -31.0, 87.0, 34.0, -47.0, -19.0, -29.0, 97.0, -35.0, 84.0, 83.0, 98.0, 46.0, 57.0, -89.0, -38.0, 74.0, 25.0, -34.0, -42.0, 31.0, 33.0, -45.0, 81.0],
                  [10.0, -54.0, 67.0, 51.0, 84.0, 5.0, 57.0, 59.0, -31.0, 88.0, -86.0, -38.0, 70.0, 41.0, -67.0, 11.0, 77.0, 98.0, 41.0, -16.0, -87.0, 13.0, 49.0, -17.0, 70.0, 56.0, -25.0, 12.0, -60.0, 11.0, 73.0, 46.0, -27.0, 61.0, -93.0, -13.0, 60.0, 0.0, -37.0, 59.0, 67.0, -7.0, -85.0, 87.0, -26.0, -49.0, -85.0, -29.0, -18.0, 56.0, 67.0, 27.0, -17.0, -16.0, 17.0, -8.0, -98.0, 35.0, 69.0, 27.0, 67.0, -36.0, 15.0, -43.0, 3.0, -52.0, -81.0, -32.0, 97.0, 23.0, -91.0, 71.0, 57.0, -35.0, 53.0, 90.0, 46.0, 92.0, 35.0, 70.0, -57.0, -92.0, 94.0, -83.0, 14.0, -14.0, -65.0, -81.0, -76.0, -74.0, -77.0, -93.0, 70.0, 16.0, 84.0, -69.0, -63.0, 89.0, 23.0, 79.0],
                  [-38.0, 3.0, 36.0, -44.0, -60.0, 6.0, 32.0, -12.0, 81.0, 24.0, 41.0, -85.0, 46.0, 56.0, 35.0, -61.0, 23.0, -84.0, 85.0, 55.0, -76.0, 59.0, -59.0, -83.0, 89.0, 47.0, 30.0, 1.0, -67.0, 15.0, 91.0, -76.0, 96.0, 18.0, 45.0, -28.0, -76.0, -94.0, 50.0, 90.0, -36.0, -38.0, -23.0, 27.0, -84.0, -31.0, 82.0, 12.0, -12.0, -62.0, 49.0, -82.0, -16.0, 100.0, 43.0, -62.0, -96.0, 12.0, 0.0, -26.0, 92.0, 33.0, -19.0, -46.0, 81.0, -47.0, -1.0, 20.0, 71.0, 86.0, -27.0, 13.0, -66.0, -90.0, -51.0, 90.0, 90.0, -16.0, 33.0, 80.0, -50.0, 40.0, 58.0, -27.0, 22.0, -27.0, -14.0, -10.0, -77.0, 84.0, 67.0, 79.0, 0.0, -2.0, -28.0, 62.0, 60.0, 40.0, -19.0, 30.0],
                  [90.0, -30.0, -97.0, -56.0, -3.0, 11.0, -6.0, -86.0, 59.0, 70.0, 92.0, 39.0, -50.0, -39.0, -78.0, -19.0, 18.0, 3.0, -3.0, 42.0, 58.0, -67.0, 2.0, 79.0, 49.0, -1.0, -35.0, 55.0, -22.0, -75.0, 55.0, 83.0, -91.0, -61.0, -12.0, -74.0, -54.0, -95.0, 68.0, 15.0, -66.0, -93.0, 36.0, 64.0, 47.0, -11.0, 42.0, -8.0, -50.0, -97.0, 88.0, 81.0, 23.0, -63.0, -86.0, -91.0, 78.0, 74.0, -92.0, -88.0, 99.0, -22.0, 8.0, 63.0, -23.0, 38.0, 63.0, -47.0, -70.0, -3.0, -96.0, -36.0, 40.0, -79.0, 14.0, -56.0, 0.0, -40.0, -12.0, 85.0, -36.0, -1.0, 99.0, 12.0, -71.0, -84.0, 85.0, 64.0, 45.0, 25.0, 11.0, -54.0, 37.0, -36.0, -28.0, 56.0, 77.0, -44.0, -92.0, -67.0],
                  [-20.0, 28.0, -24.0, -76.0, -55.0, 66.0, -13.0, 34.0, 18.0, 6.0, 34.0, -9.0, -61.0, -66.0, -6.0, 34.0, -35.0, 71.0, -24.0, 85.0, 4.0, 83.0, -80.0, 6.0, 25.0, 34.0, -32.0, -36.0, 35.0, -69.0, -29.0, -7.0, 2.0, 34.0, -66.0, 52.0, 60.0, 88.0, -38.0, 37.0, 58.0, 58.0, 83.0, -18.0, -51.0, 37.0, 70.0, -18.0, 34.0, 16.0, 38.0, 71.0, -7.0, -5.0, -15.0, -26.0, -93.0, -57.0, -26.0, -95.0, -19.0, -24.0, -36.0, 67.0, 2.0, 70.0, 49.0, -4.0, 90.0, 43.0, 57.0, -93.0, -44.0, -43.0, 80.0, -6.0, -28.0, 91.0, 41.0, 85.0, 0.0, -85.0, 25.0, -70.0, 12.0, -9.0, 65.0, 18.0, -26.0, -33.0, 29.0, 2.0, 17.0, 81.0, 28.0, 90.0, 8.0, 67.0, -23.0, -6.0],
                  [-45.0, 88.0, 79.0, 32.0, -41.0, 31.0, -77.0, -49.0, -7.0, 91.0, -61.0, 59.0, -56.0, 13.0, 40.0, 60.0, -75.0, 86.0, 11.0, -20.0, 56.0, -3.0, 72.0, 39.0, -60.0, -75.0, 27.0, 17.0, 33.0, 33.0, -63.0, -14.0, 61.0, 59.0, -11.0, -44.0, 59.0, 30.0, -77.0, -40.0, 61.0, 33.0, 21.0, 70.0, 88.0, 21.0, -80.0, 6.0, -98.0, -14.0, -61.0, -41.0, 27.0, 80.0, 33.0, -37.0, 65.0, 74.0, -64.0, 44.0, -88.0, 1.0, -36.0, 81.0, -14.0, 61.0, -56.0, -57.0, 76.0, -58.0, -49.0, 16.0, 0.0, 39.0, 86.0, 42.0, 68.0, 94.0, 69.0, -68.0, -96.0, -66.0, 93.0, -23.0, 96.0, 4.0, 72.0, -63.0, -37.0, 62.0, 63.0, 89.0, 58.0, -68.0, -19.0, 71.0, -19.0, 58.0, -18.0, 34.0],
                  [-72.0, -78.0, -81.0, -90.0, -25.0, 75.0, -3.0, 70.0, 28.0, -25.0, -82.0, 44.0, -76.0, -32.0, 97.0, -13.0, -69.0, -9.0, 17.0, -75.0, 24.0, 86.0, 76.0, 21.0, -36.0, 35.0, -49.0, -91.0, -61.0, 75.0, 80.0, -14.0, 84.0, 38.0, -37.0, -73.0, -27.0, -55.0, 98.0, -10.0, -3.0, -49.0, 75.0, -100.0, 21.0, -48.0, -93.0, 27.0, 37.0, 71.0, 79.0, 43.0, 16.0, 41.0, 26.0, -39.0, 83.0, 65.0, 5.0, -46.0, 64.0, 10.0, 75.0, -80.0, -59.0, 68.0, -88.0, 21.0, 19.0, 35.0, 28.0, 53.0, -96.0, 46.0, -20.0, -94.0, 70.0, 91.0, 94.0, 61.0, -50.0, 24.0, 82.0, -84.0, 66.0, 63.0, 19.0, 47.0, 5.0, 67.0, -49.0, 42.0, -65.0, -62.0, -94.0, -96.0, 72.0, 79.0, -71.0, 92.0],
                  [37.0, -70.0, 62.0, 18.0, -34.0, -1.0, -73.0, -70.0, -14.0, -80.0, 88.0, -67.0, -38.0, -13.0, 90.0, -36.0, 31.0, -74.0, 18.0, 8.0, 7.0, -24.0, 11.0, 7.0, -71.0, -99.0, 66.0, 63.0, -58.0, 72.0, -91.0, 57.0, -12.0, -21.0, -29.0, 81.0, 4.0, 35.0, -67.0, -61.0, -63.0, -48.0, -94.0, 40.0, -92.0, 96.0, -3.0, 65.0, -29.0, 78.0, 50.0, 15.0, 86.0, -92.0, -12.0, 70.0, 13.0, 12.0, -79.0, 45.0, -18.0, -50.0, -54.0, 78.0, -40.0, -44.0, -28.0, -27.0, 95.0, -14.0, -46.0, 43.0, 87.0, 17.0, -96.0, 19.0, 59.0, -55.0, -7.0, 99.0, 39.0, 83.0, -81.0, 36.0, -58.0, 83.0, 59.0, -99.0, 31.0, -34.0, -70.0, 90.0, -50.0, 74.0, -30.0, 58.0, 25.0, 10.0, -41.0, 8.0],
                  [-29.0, 5.0, 28.0, 2.0, 14.0, -83.0, 85.0, -47.0, -2.0, -80.0, 28.0, 79.0, 60.0, 66.0, 59.0, 96.0, -85.0, -37.0, -95.0, 11.0, 25.0, -17.0, 41.0, -38.0, -13.0, -34.0, 64.0, 45.0, 15.0, -63.0, 18.0, 74.0, -37.0, -19.0, 60.0, 50.0, 56.0, 59.0, -37.0, 79.0, 84.0, 10.0, 72.0, 3.0, 63.0, 81.0, 55.0, 38.0, 94.0, 22.0, -84.0, 6.0, 37.0, -92.0, 43.0, 99.0, -16.0, -69.0, -32.0, 8.0, -93.0, 58.0, 86.0, 93.0, -66.0, -76.0, 71.0, -69.0, 54.0, -58.0, -87.0, -40.0, 79.0, -73.0, -42.0, -51.0, 61.0, -21.0, -68.0, -14.0, -72.0, 12.0, -1.0, -98.0, 26.0, -36.0, -11.0, -90.0, -16.0, -55.0, 22.0, -91.0, 45.0, -19.0, 12.0, 21.0, 64.0, 53.0, -50.0, -25.0],
                  [84.0, 24.0, -28.0, -60.0, 34.0, 17.0, -73.0, -96.0, 10.0, -95.0, 77.0, 57.0, 17.0, -77.0, -48.0, -79.0, 41.0, -74.0, -53.0, 40.0, -67.0, -63.0, 42.0, 19.0, 87.0, 44.0, 15.0, -29.0, 84.0, -19.0, 81.0, 73.0, 98.0, -72.0, 12.0, 98.0, -36.0, -43.0, 30.0, 6.0, 92.0, -72.0, -98.0, -11.0, -38.0, 80.0, 3.0, -38.0, -88.0, -21.0, -19.0, -74.0, 41.0, -10.0, -17.0, -66.0, 27.0, -14.0, 46.0, -56.0, -15.0, -48.0, -61.0, -51.0, 59.0, 17.0, 1.0, 71.0, 1.0, -95.0, 80.0, 58.0, -75.0, 18.0, 88.0, -3.0, 22.0, 6.0, 33.0, 47.0, 75.0, -97.0, 27.0, 16.0, 68.0, -26.0, -91.0, -100.0, 48.0, -100.0, -14.0, 11.0, 81.0, 76.0, 24.0, -95.0, 60.0, -56.0, 46.0, -4.0],
                  [-86.0, -4.0, -43.0, 95.0, 55.0, 11.0, 53.0, -27.0, -4.0, -78.0, 62.0, -72.0, -89.0, 97.0, 97.0, -60.0, -19.0, -19.0, 32.0, -57.0, 7.0, -82.0, 29.0, -91.0, -8.0, -29.0, -20.0, 95.0, -10.0, 7.0, -55.0, 1.0, 99.0, 66.0, 30.0, 82.0, -52.0, -45.0, 71.0, 77.0, -63.0, 74.0, 69.0, 21.0, -11.0, -20.0, 94.0, -94.0, -87.0, 72.0, -83.0, 11.0, 38.0, 9.0, 90.0, -27.0, 57.0, -55.0, -53.0, 59.0, 42.0, 21.0, 11.0, -97.0, 31.0, 14.0, -54.0, -81.0, -3.0, 90.0, -22.0, -55.0, -96.0, -45.0, -36.0, 29.0, 85.0, -67.0, -68.0, -88.0, -67.0, 74.0, 49.0, -25.0, -10.0, 29.0, -64.0, -23.0, -28.0, 11.0, -88.0, -79.0, -43.0, 50.0, 7.0, -6.0, -28.0, -48.0, -58.0, 14.0],
                  [62.0, -18.0, 36.0, 75.0, 51.0, 84.0, -1.0, -10.0, -87.0, 73.0, 48.0, 21.0, 92.0, -90.0, -14.0, -5.0, -26.0, 8.0, 18.0, 34.0, -49.0, 46.0, -50.0, -13.0, 87.0, -95.0, 98.0, 6.0, 94.0, 81.0, 53.0, -78.0, 90.0, 8.0, 2.0, -91.0, -71.0, -69.0, -20.0, 74.0, 87.0, 95.0, -67.0, 22.0, 44.0, -75.0, 45.0, -84.0, -88.0, 43.0, -52.0, 65.0, -99.0, -17.0, 52.0, -44.0, 29.0, -20.0, 80.0, 92.0, -66.0, 74.0, 83.0, -22.0, 27.0, -49.0, -43.0, 89.0, 59.0, -42.0, -76.0, -45.0, -97.0, 97.0, -52.0, -8.0, -70.0, -58.0, 72.0, 60.0, 29.0, 56.0, 36.0, -49.0, -97.0, 89.0, -67.0, -68.0, -51.0, -72.0, 38.0, 94.0, -3.0, -71.0, 36.0, -8.0, 52.0, -95.0, -89.0, -54.0],
                  [26.0, 14.0, -15.0, -73.0, -79.0, 51.0, -98.0, -76.0, 38.0, 2.0, 72.0, -68.0, -52.0, -68.0, 38.0, -27.0, 9.0, 25.0, 91.0, -3.0, -48.0, -71.0, 84.0, 40.0, 23.0, 82.0, -96.0, 35.0, 84.0, -61.0, -38.0, 76.0, 60.0, -61.0, -7.0, 79.0, 32.0, -47.0, 79.0, 93.0, -60.0, 49.0, 9.0, -96.0, -88.0, 97.0, 21.0, 44.0, 68.0, 74.0, -38.0, 22.0, 5.0, -53.0, 29.0, 37.0, 1.0, 34.0, -73.0, 7.0, -28.0, -82.0, 56.0, 33.0, -38.0, -65.0, -88.0, 2.0, 79.0, 97.0, 17.0, -97.0, 23.0, 45.0, 8.0, 25.0, 50.0, 25.0, -65.0, 10.0, 64.0, -10.0, 90.0, 50.0, -12.0, 2.0, 0.0, -97.0, 92.0, -19.0, -94.0, 37.0, -23.0, -41.0, 16.0, -60.0, -55.0, 9.0, 85.0, 5.0],
                  [-68.0, 61.0, 60.0, -72.0, 65.0, -81.0, 72.0, -93.0, -61.0, 64.0, -64.0, 7.0, 33.0, 52.0, -81.0, -98.0, -66.0, 63.0, -11.0, -32.0, -19.0, -51.0, 25.0, -9.0, 55.0, 43.0, -51.0, -63.0, -57.0, 49.0, -13.0, 5.0, 12.0, 81.0, -86.0, 33.0, 36.0, 67.0, -58.0, -73.0, 15.0, 99.0, -52.0, -42.0, 46.0, 16.0, 50.0, -66.0, -85.0, -93.0, 22.0, -81.0, 50.0, 71.0, -32.0, -86.0, 25.0, 22.0, 95.0, -56.0, -93.0, 21.0, 49.0, 24.0, -57.0, -42.0, -26.0, 30.0, -10.0, 89.0, -11.0, 6.0, 29.0, -32.0, 50.0, 77.0, 68.0, -19.0, 67.0, 89.0, 19.0, -47.0, -32.0, 99.0, -51.0, 59.0, 94.0, 6.0, -40.0, -73.0, 25.0, -64.0, -65.0, 71.0, -50.0, 66.0, -22.0, 35.0, -26.0, -95.0],
                  [43.0, 57.0, 54.0, -4.0, -45.0, -41.0, -96.0, -73.0, 78.0, 73.0, 70.0, 34.0, -61.0, -75.0, 77.0, -41.0, -13.0, -67.0, -46.0, 82.0, -4.0, -39.0, -32.0, -95.0, -76.0, 76.0, 69.0, 35.0, 42.0, 46.0, -85.0, -45.0, -92.0, -94.0, 11.0, -33.0, 86.0, -68.0, -19.0, 91.0, 24.0, 74.0, 44.0, -3.0, -76.0, -31.0, -91.0, -11.0, 5.0, -84.0, -79.0, -71.0, -78.0, -29.0, -46.0, 60.0, -5.0, -85.0, 83.0, -30.0, -74.0, -31.0, 57.0, 90.0, -80.0, -16.0, -38.0, 16.0, 14.0, -70.0, -66.0, 27.0, -59.0, 15.0, -100.0, 39.0, 7.0, 75.0, 33.0, -33.0, 91.0, -5.0, -33.0, 50.0, 80.0, -93.0, 69.0, 63.0, 20.0, 44.0, -26.0, 84.0, -90.0, -13.0, -21.0, 14.0, -32.0, 32.0, 55.0, 80.0],
                  [98.0, 3.0, -67.0, -87.0, 22.0, -89.0, -58.0, -87.0, -29.0, -3.0, -26.0, 9.0, -82.0, 23.0, 58.0, 47.0, 6.0, 37.0, 49.0, -67.0, 9.0, -12.0, -5.0, -46.0, -24.0, 27.0, 16.0, 9.0, 93.0, -81.0, 76.0, -10.0, -61.0, -29.0, 67.0, 43.0, -24.0, 43.0, 17.0, 68.0, -3.0, -14.0, -53.0, -42.0, 96.0, 69.0, -64.0, 18.0, -75.0, 82.0, -36.0, 96.0, 4.0, -65.0, 98.0, -88.0, 91.0, -54.0, -5.0, 17.0, -95.0, -48.0, 45.0, -81.0, 95.0, 62.0, -89.0, 44.0, 70.0, -33.0, -31.0, 97.0, -33.0, -91.0, 10.0, 62.0, 36.0, -8.0, 10.0, -40.0, 98.0, 85.0, -88.0, 53.0, 8.0, -11.0, 96.0, 8.0, -49.0, -61.0, 96.0, -63.0, -49.0, 59.0, -78.0, -60.0, -97.0, -63.0, 87.0, 34.0],
                  [4.0, -80.0, 12.0, 5.0, 69.0, 67.0, -3.0, 6.0, 75.0, 25.0, 35.0, 19.0, 62.0, 0.0, 99.0, 53.0, -80.0, 13.0, -51.0, 26.0, 51.0, 28.0, -4.0, -93.0, -64.0, -5.0, 27.0, -34.0, -5.0, -90.0, -38.0, -40.0, 16.0, 39.0, -16.0, 31.0, -90.0, -82.0, 57.0, 28.0, 9.0, -90.0, -54.0, 0.0, 46.0, -71.0, -6.0, 41.0, -63.0, 60.0, 28.0, 96.0, -54.0, 21.0, -45.0, -34.0, -89.0, 100.0, 25.0, 92.0, 19.0, 43.0, -4.0, -59.0, -62.0, -87.0, -71.0, 69.0, 18.0, -24.0, -23.0, -50.0, 80.0, -29.0, -25.0, -80.0, -14.0, -84.0, -54.0, -4.0, -61.0, -5.0, -37.0, -10.0, -48.0, 65.0, 64.0, -42.0, -87.0, 65.0, 49.0, -30.0, -52.0, 40.0, -17.0, -55.0, 13.0, 31.0, 64.0, 22.0],
                  [-39.0, 31.0, 82.0, 31.0, 40.0, -49.0, -22.0, -31.0, 96.0, 82.0, -60.0, 14.0, -70.0, 81.0, -88.0, -40.0, -36.0, -68.0, 15.0, 79.0, -60.0, -53.0, 45.0, -35.0, 94.0, 0.0, 0.0, -65.0, -3.0, 9.0, -70.0, 66.0, -77.0, 15.0, 9.0, 14.0, 3.0, 96.0, -76.0, 72.0, 71.0, -56.0, -40.0, 34.0, 19.0, 10.0, 100.0, -48.0, 82.0, -18.0, -44.0, 93.0, 64.0, 16.0, 72.0, -61.0, -14.0, 19.0, -5.0, -3.0, 20.0, -40.0, -59.0, -9.0, 77.0, -31.0, -50.0, -5.0, 50.0, 78.0, 6.0, 71.0, 33.0, -4.0, 22.0, -88.0, -96.0, -76.0, 97.0, -60.0, -70.0, 64.0, -62.0, -12.0, 37.0, 48.0, 74.0, -63.0, 89.0, 29.0, 42.0, 85.0, -12.0, -63.0, 20.0, -28.0, -71.0, -12.0, 31.0, -74.0],
                  [-24.0, 74.0, 95.0, -22.0, 64.0, 9.0, -34.0, 66.0, -16.0, 30.0, -100.0, -59.0, 32.0, 69.0, -34.0, 96.0, -25.0, 63.0, -84.0, 50.0, 100.0, 77.0, 23.0, -55.0, -88.0, 41.0, 70.0, 19.0, 14.0, -51.0, 27.0, -12.0, 55.0, -31.0, -95.0, -75.0, -11.0, 12.0, -73.0, 28.0, -48.0, 68.0, -36.0, -4.0, -68.0, 42.0, 59.0, 41.0, 96.0, -23.0, 54.0, 91.0, 6.0, 10.0, -15.0, 96.0, 54.0, 35.0, 23.0, -68.0, -42.0, 41.0, 2.0, -96.0, 64.0, 18.0, 52.0, -76.0, 57.0, 72.0, -37.0, 25.0, -5.0, -97.0, 75.0, -77.0, -47.0, -86.0, -62.0, -47.0, -41.0, 10.0, -17.0, -78.0, -37.0, -69.0, 8.0, -55.0, -82.0, -65.0, 92.0, -90.0, -30.0, -38.0, 89.0, -98.0, -74.0, -77.0, 31.0, -31.0],
                  [77.0, -28.0, -34.0, 74.0, -86.0, 60.0, 92.0, 83.0, 43.0, 54.0, 60.0, -82.0, -43.0, 9.0, -94.0, -64.0, 42.0, -92.0, 73.0, -7.0, -2.0, -52.0, 0.0, -5.0, 69.0, 76.0, 22.0, 1.0, 76.0, 83.0, 78.0, -86.0, 0.0, 68.0, 91.0, 6.0, -64.0, 85.0, -43.0, -18.0, 90.0, -29.0, 53.0, 36.0, -13.0, -89.0, -8.0, 100.0, 18.0, -3.0, -44.0, -73.0, 98.0, -70.0, 13.0, 99.0, 87.0, -19.0, -6.0, -9.0, -36.0, -31.0, -90.0, 39.0, -53.0, 12.0, 22.0, -57.0, -45.0, 38.0, 68.0, 38.0, -4.0, -19.0, -90.0, 46.0, -52.0, -27.0, -57.0, -37.0, -63.0, -56.0, 75.0, 58.0, 81.0, 74.0, 54.0, 32.0, -34.0, 30.0, 82.0, 21.0, -74.0, -36.0, 17.0, 98.0, -55.0, -73.0, -34.0, -2.0],
                  [7.0, -74.0, -52.0, 90.0, -42.0, -9.0, -98.0, 85.0, -2.0, -48.0, 28.0, 35.0, -41.0, 24.0, 11.0, 23.0, 90.0, -74.0, -91.0, -63.0, -90.0, 54.0, -99.0, -14.0, 1.0, -74.0, -54.0, 14.0, -60.0, 79.0, 72.0, 89.0, -94.0, 57.0, 0.0, 90.0, 74.0, -25.0, -76.0, -59.0, 28.0, 39.0, -85.0, 57.0, 38.0, -61.0, 93.0, 76.0, -83.0, -29.0, 37.0, -69.0, 36.0, 95.0, 3.0, -12.0, -40.0, -99.0, 66.0, 18.0, 26.0, 91.0, -53.0, -4.0, -16.0, -70.0, 12.0, -21.0, -11.0, -90.0, 55.0, -79.0, 65.0, -63.0, 55.0, 87.0, -8.0, 42.0, 65.0, 63.0, -34.0, -64.0, 15.0, 70.0, -12.0, -69.0, -96.0, -99.0, 65.0, 28.0, -42.0, 82.0, 12.0, -13.0, 1.0, 46.0, 84.0, -31.0, -29.0, 26.0],
                  [-28.0, -86.0, -69.0, -62.0, 67.0, -92.0, 14.0, -48.0, 62.0, 77.0, -24.0, -62.0, 6.0, 28.0, 63.0, -46.0, 86.0, 72.0, -92.0, 89.0, 83.0, -33.0, -90.0, 94.0, 32.0, -38.0, 31.0, 79.0, -17.0, -28.0, -95.0, 52.0, -83.0, -72.0, -90.0, 78.0, 12.0, 62.0, 85.0, -44.0, 53.0, -23.0, -23.0, -18.0, -20.0, 53.0, -12.0, -100.0, 44.0, 96.0, -43.0, -13.0, -17.0, 56.0, 58.0, -90.0, 15.0, -11.0, 51.0, 82.0, -71.0, -74.0, -3.0, 94.0, 99.0, -17.0, -12.0, 20.0, 65.0, 86.0, 66.0, 24.0, 11.0, 91.0, -63.0, 40.0, 72.0, 48.0, -58.0, 33.0, 38.0, -5.0, -58.0, 86.0, -12.0, 45.0, 9.0, -63.0, -88.0, -5.0, 86.0, -49.0, 70.0, -7.0, -89.0, -25.0, -20.0, -30.0, -14.0, -92.0],
                  [27.0, 69.0, -69.0, 24.0, 76.0, -27.0, 20.0, -14.0, 46.0, -38.0, -16.0, -47.0, 66.0, -26.0, 63.0, 40.0, 82.0, 16.0, 38.0, 66.0, 17.0, -29.0, 32.0, -36.0, -25.0, -38.0, -37.0, -78.0, 2.0, 2.0, 65.0, -60.0, -65.0, -26.0, 7.0, 19.0, 63.0, -47.0, 66.0, 87.0, 11.0, 10.0, 92.0, 38.0, 73.0, 87.0, 23.0, 12.0, 21.0, 67.0, 2.0, -78.0, -52.0, 80.0, -13.0, -6.0, -71.0, -75.0, 63.0, -2.0, -39.0, -61.0, -97.0, -3.0, -46.0, -86.0, 71.0, -47.0, -31.0, -66.0, -8.0, -47.0, -42.0, 40.0, 32.0, 60.0, -10.0, -54.0, 82.0, -65.0, -35.0, 45.0, 95.0, -43.0, -87.0, -32.0, 0.0, 5.0, 78.0, -2.0, 18.0, 99.0, -24.0, 10.0, -72.0, -36.0, 47.0, -96.0, -49.0, 1.0],
                  [72.0, 35.0, -41.0, -63.0, 68.0, -76.0, -63.0, 64.0, 41.0, 94.0, -91.0, -53.0, 87.0, 100.0, -9.0, 26.0, 2.0, -40.0, 89.0, 99.0, 18.0, 51.0, 16.0, -30.0, -79.0, 74.0, -91.0, 86.0, -16.0, -3.0, -41.0, -59.0, 46.0, -25.0, -53.0, 56.0, -17.0, 24.0, 67.0, 44.0, 100.0, 17.0, 22.0, -96.0, 46.0, -59.0, -21.0, 50.0, 50.0, -7.0, 70.0, 88.0, -29.0, -21.0, -20.0, -21.0, -54.0, -26.0, -37.0, 31.0, -7.0, -17.0, 81.0, -53.0, 55.0, 79.0, -14.0, -7.0, 66.0, -24.0, -48.0, -26.0, 45.0, 79.0, -56.0, 49.0, 75.0, 10.0, -52.0, -87.0, 10.0, 25.0, -53.0, 88.0, 53.0, 95.0, 55.0, 57.0, 66.0, -12.0, 94.0, 23.0, 21.0, -64.0, -89.0, -96.0, 1.0, 64.0, 98.0, -42.0],
                  [79.0, -99.0, 5.0, 60.0, 3.0, 28.0, 67.0, 48.0, 8.0, 69.0, 47.0, 60.0, 22.0, 5.0, -5.0, 68.0, -91.0, 68.0, -26.0, 89.0, 90.0, -44.0, -30.0, -59.0, -76.0, -39.0, -16.0, 97.0, 86.0, 16.0, -54.0, -38.0, 96.0, -36.0, 36.0, -72.0, -3.0, -46.0, -84.0, 42.0, 25.0, 57.0, 98.0, 89.0, 61.0, -69.0, 56.0, 0.0, -69.0, -45.0, 84.0, -67.0, 33.0, -75.0, -32.0, 99.0, -60.0, -43.0, -43.0, -57.0, 60.0, 2.0, 96.0, -57.0, -6.0, 32.0, 30.0, -10.0, 47.0, 28.0, 31.0, -67.0, -6.0, 22.0, -37.0, -29.0, 30.0, -30.0, -55.0, -77.0, 39.0, 12.0, -74.0, 32.0, -4.0, 7.0, -69.0, -58.0, 20.0, 97.0, 26.0, -86.0, -99.0, 13.0, 25.0, -83.0, 56.0, -52.0, -79.0, -53.0],
                  [-70.0, 13.0, -61.0, -9.0, 58.0, -28.0, 72.0, -40.0, -20.0, -40.0, 8.0, -42.0, 99.0, -2.0, 44.0, 9.0, 14.0, -36.0, -20.0, 37.0, 87.0, -67.0, -96.0, 24.0, 93.0, -65.0, -16.0, -85.0, -26.0, -88.0, 28.0, -71.0, -83.0, 53.0, 3.0, -89.0, 65.0, -74.0, 95.0, -83.0, -21.0, -10.0, -44.0, -79.0, -47.0, 50.0, 5.0, 52.0, -64.0, -33.0, -12.0, -37.0, 5.0, 18.0, -9.0, -27.0, 11.0, 94.0, -57.0, -79.0, -68.0, 93.0, 5.0, 24.0, -77.0, 57.0, -11.0, -1.0, -82.0, -38.0, -68.0, 90.0, 100.0, -75.0, 36.0, 49.0, 6.0, 56.0, 58.0, -12.0, -39.0, 27.0, -74.0, 44.0, -57.0, -33.0, 45.0, 74.0, 18.0, 30.0, -16.0, -19.0, 36.0, -71.0, -65.0, 95.0, 11.0, 57.0, -65.0, -31.0],
                  [-3.0, -86.0, 59.0, 21.0, 54.0, 93.0, 51.0, 86.0, -87.0, -68.0, -1.0, 69.0, 19.0, -58.0, -70.0, 52.0, 86.0, -46.0, 77.0, 42.0, -75.0, -61.0, 94.0, -91.0, -59.0, -19.0, 85.0, -34.0, 3.0, 78.0, 19.0, 24.0, -14.0, -37.0, -22.0, 76.0, -15.0, 33.0, -36.0, 96.0, 32.0, -60.0, -51.0, -9.0, 69.0, 16.0, 7.0, 2.0, 56.0, -23.0, 17.0, -76.0, -9.0, -77.0, -63.0, 35.0, 46.0, -75.0, 31.0, -90.0, -4.0, 87.0, 38.0, -34.0, -33.0, 85.0, 16.0, -40.0, -77.0, 95.0, 46.0, -76.0, 34.0, -29.0, 60.0, 92.0, 81.0, 49.0, -81.0, 28.0, 95.0, 29.0, -10.0, -90.0, 83.0, -33.0, -65.0, -88.0, -81.0, 13.0, 43.0, 28.0, -78.0, 14.0, -24.0, 11.0, -69.0, -83.0, -33.0, -52.0],
                  [59.0, -12.0, -55.0, 21.0, -52.0, -87.0, -41.0, 77.0, 7.0, 74.0, -56.0, 30.0, -9.0, -60.0, 62.0, -29.0, 44.0, -8.0, -34.0, 37.0, 80.0, 99.0, 19.0, 27.0, 95.0, 72.0, 66.0, 37.0, -97.0, -71.0, 1.0, -24.0, 38.0, -45.0, -4.0, -76.0, -80.0, 33.0, 85.0, -47.0, -18.0, 27.0, 54.0, 59.0, -22.0, -47.0, -67.0, -90.0, 42.0, 92.0, 10.0, 54.0, 38.0, 34.0, 28.0, 44.0, 15.0, 81.0, 98.0, -89.0, -1.0, 69.0, 5.0, 12.0, 44.0, -57.0, -71.0, -30.0, -68.0, 87.0, -24.0, 48.0, -25.0, 100.0, -53.0, 22.0, -100.0, -95.0, -75.0, 28.0, -4.0, 65.0, 30.0, -93.0, 24.0, -32.0, 21.0, -59.0, 98.0, -40.0, 68.0, 23.0, 12.0, 99.0, 56.0, 76.0, -21.0, -46.0, -36.0, 72.0],
                  [-8.0, 97.0, 88.0, -82.0, 43.0, -81.0, -32.0, 38.0, -33.0, -77.0, 39.0, -40.0, -88.0, 65.0, 1.0, -86.0, -78.0, -50.0, 32.0, -43.0, 39.0, -91.0, -19.0, 75.0, -5.0, 75.0, -68.0, 66.0, -7.0, -40.0, 71.0, 63.0, -6.0, 8.0, -39.0, -28.0, -59.0, 99.0, 69.0, 73.0, -92.0, 59.0, -77.0, -81.0, -32.0, 77.0, -1.0, -50.0, 91.0, -54.0, -88.0, 95.0, 21.0, 36.0, 54.0, -57.0, -92.0, 72.0, 13.0, -9.0, 96.0, -30.0, 44.0, 5.0, -14.0, 20.0, -80.0, 66.0, 37.0, 23.0, -5.0, 45.0, -69.0, -52.0, 49.0, -3.0, 74.0, -24.0, -40.0, 59.0, 49.0, -5.0, 45.0, -6.0, 63.0, -41.0, 85.0, -71.0, -93.0, 24.0, 87.0, -44.0, -68.0, 57.0, 51.0, 82.0, -31.0, 53.0, 14.0, 47.0],
                  [-87.0, -81.0, -12.0, -73.0, 0.0, -36.0, -4.0, -97.0, -68.0, -84.0, -16.0, 82.0, -16.0, -84.0, -18.0, -62.0, -44.0, -79.0, -96.0, -90.0, -60.0, -22.0, 45.0, -2.0, -30.0, -91.0, 65.0, -39.0, -67.0, -93.0, -37.0, 44.0, -2.0, 91.0, 24.0, 24.0, 37.0, 43.0, -59.0, -68.0, -74.0, 45.0, -79.0, 75.0, 45.0, -44.0, 80.0, 68.0, -71.0, -36.0, 91.0, 22.0, 62.0, 99.0, -91.0, 32.0, -65.0, -63.0, 92.0, -97.0, -78.0, 77.0, 23.0, -98.0, 4.0, -91.0, 88.0, 74.0, 38.0, -2.0, -57.0, -66.0, -22.0, -14.0, 0.0, -47.0, -86.0, 46.0, 71.0, 69.0, -94.0, 17.0, -3.0, -97.0, -96.0, -74.0, -15.0, 25.0, 46.0, 18.0, 34.0, -33.0, 97.0, 60.0, -22.0, 93.0, -21.0, 65.0, 36.0, -7.0],
                  [90.0, 9.0, 57.0, -55.0, -93.0, -96.0, -91.0, 89.0, 75.0, 24.0, -46.0, 21.0, 4.0, -81.0, -68.0, -43.0, -44.0, -65.0, 3.0, 92.0, 8.0, -22.0, 86.0, 85.0, 87.0, 62.0, 54.0, 35.0, -25.0, -92.0, 58.0, 96.0, 58.0, -95.0, 49.0, 40.0, -47.0, 22.0, -48.0, 2.0, -64.0, 20.0, -35.0, 76.0, -71.0, -10.0, -82.0, 63.0, -24.0, -17.0, -96.0, -61.0, -74.0, -22.0, -45.0, 70.0, 68.0, 54.0, -30.0, -79.0, 92.0, -71.0, 87.0, 45.0, -29.0, 63.0, 92.0, 60.0, 73.0, -7.0, -36.0, 67.0, 97.0, 52.0, 41.0, 37.0, -80.0, 65.0, 83.0, 23.0, 28.0, -48.0, 54.0, -14.0, -60.0, -88.0, 44.0, -8.0, 50.0, -2.0, -59.0, 67.0, 47.0, -97.0, -35.0, -99.0, -33.0, 35.0, 76.0, -23.0],
                  [-42.0, -49.0, -77.0, 50.0, 83.0, -49.0, -65.0, 92.0, -60.0, 0.0, 54.0, 31.0, -27.0, 95.0, 46.0, -2.0, -64.0, -58.0, -46.0, 68.0, 10.0, 10.0, -7.0, -7.0, 89.0, 25.0, 89.0, 84.0, -18.0, 14.0, 69.0, 22.0, 60.0, -53.0, 45.0, -25.0, 57.0, 64.0, 22.0, -42.0, 4.0, 71.0, -92.0, -18.0, -32.0, -69.0, 64.0, 73.0, 5.0, -7.0, 24.0, -31.0, 66.0, 49.0, 71.0, -62.0, 57.0, -63.0, -95.0, -13.0, -83.0, -77.0, 30.0, 73.0, -26.0, 41.0, -86.0, 2.0, 61.0, -25.0, -93.0, 89.0, -82.0, 83.0, -63.0, -16.0, 46.0, 51.0, -41.0, -46.0, -22.0, -91.0, 84.0, 74.0, 23.0, 53.0, 79.0, -94.0, -97.0, -84.0, 82.0, 12.0, -35.0, 24.0, 19.0, 76.0, -3.0, 73.0, 12.0, -99.0],
                  [-31.0, 75.0, 30.0, -69.0, -53.0, 15.0, 98.0, -44.0, 62.0, 63.0, 46.0, -63.0, -82.0, 10.0, -87.0, 63.0, 0.0, 79.0, -65.0, 92.0, -29.0, 44.0, -35.0, 81.0, -49.0, 59.0, -21.0, 32.0, -5.0, 18.0, 57.0, -6.0, -77.0, 37.0, 67.0, 67.0, 10.0, 75.0, -94.0, 12.0, -97.0, -80.0, -57.0, 88.0, 23.0, 18.0, 69.0, 31.0, -80.0, 100.0, -16.0, 22.0, 63.0, -53.0, 62.0, -87.0, 53.0, -5.0, 30.0, 90.0, 11.0, -86.0, 10.0, 21.0, -86.0, -41.0, 58.0, -78.0, 48.0, 83.0, -52.0, -85.0, -40.0, 45.0, 39.0, -9.0, 11.0, -29.0, -62.0, -6.0, -25.0, -90.0, -58.0, -37.0, 49.0, 29.0, -83.0, 83.0, -12.0, 41.0, 98.0, -68.0, 99.0, -84.0, 46.0, 22.0, -24.0, 72.0, -81.0, 34.0],
                  [-72.0, -96.0, -92.0, -66.0, -56.0, -85.0, -12.0, -20.0, 64.0, -24.0, -56.0, 47.0, 8.0, -37.0, -24.0, -96.0, -4.0, -50.0, -23.0, 39.0, -68.0, -43.0, -30.0, -19.0, -42.0, 34.0, -64.0, 65.0, -60.0, 27.0, -47.0, -28.0, -22.0, 34.0, 73.0, -29.0, -72.0, -25.0, 20.0, -67.0, -59.0, -89.0, -20.0, -39.0, 23.0, -92.0, -35.0, -96.0, -70.0, -68.0, 58.0, -2.0, 73.0, 23.0, -26.0, 92.0, -63.0, -19.0, -51.0, 96.0, 26.0, 10.0, 91.0, 0.0, 93.0, 74.0, -27.0, -10.0, 82.0, 99.0, 81.0, -55.0, 32.0, -75.0, 16.0, 46.0, -88.0, -47.0, -50.0, 63.0, 97.0, 2.0, -20.0, -38.0, 34.0, -22.0, 93.0, -43.0, -11.0, 11.0, -16.0, 86.0, 32.0, -54.0, -97.0, -9.0, 18.0, 49.0, -89.0, -66.0],
                  [8.0, 8.0, -63.0, 92.0, -68.0, -84.0, -28.0, -50.0, -40.0, -63.0, -48.0, -97.0, 44.0, 73.0, -80.0, -57.0, -75.0, -32.0, -96.0, -61.0, 39.0, -93.0, 37.0, 17.0, -74.0, -2.0, 26.0, -72.0, -2.0, 85.0, 47.0, 87.0, 9.0, -82.0, 8.0, 56.0, 20.0, 71.0, -46.0, 69.0, 54.0, -73.0, 45.0, -79.0, 5.0, -27.0, 43.0, -73.0, -16.0, 71.0, -67.0, -67.0, 39.0, 9.0, 4.0, 36.0, -20.0, -40.0, 19.0, 27.0, -60.0, -77.0, 93.0, 73.0, 16.0, 94.0, -56.0, 12.0, 82.0, 39.0, -89.0, 75.0, 59.0, 80.0, 77.0, -41.0, 9.0, 44.0, -48.0, 17.0, 58.0, 43.0, 44.0, 28.0, 49.0, 76.0, 65.0, 41.0, 96.0, -85.0, -36.0, -17.0, -85.0, 43.0, 73.0, 20.0, 51.0, -61.0, -93.0, 94.0],
                  [-19.0, 60.0, -22.0, 26.0, 0.0, -99.0, -22.0, -47.0, -27.0, 91.0, -33.0, -9.0, -94.0, -43.0, 65.0, -85.0, 91.0, 39.0, 90.0, -4.0, 95.0, 80.0, -57.0, 98.0, 10.0, -43.0, 40.0, -74.0, -14.0, -87.0, 89.0, -2.0, 26.0, -96.0, -37.0, 84.0, -47.0, 25.0, 81.0, 40.0, 95.0, -33.0, 12.0, -90.0, 56.0, 94.0, 33.0, -96.0, 36.0, 69.0, 87.0, -33.0, 41.0, -19.0, 9.0, 50.0, 39.0, -83.0, -84.0, -94.0, -17.0, 66.0, -3.0, -34.0, -24.0, -23.0, 57.0, -12.0, -66.0, -70.0, -3.0, -64.0, 15.0, 25.0, 23.0, 23.0, 38.0, -63.0, 70.0, 13.0, 6.0, 46.0, 30.0, 93.0, -96.0, 54.0, -45.0, 33.0, -13.0, -86.0, -53.0, 52.0, -53.0, 66.0, -82.0, -59.0, 13.0, -98.0, 40.0, 13.0],
                  [6.0, -40.0, -12.0, -46.0, 1.0, -89.0, -26.0, 49.0, 25.0, 70.0, 26.0, 26.0, 30.0, 35.0, -64.0, -42.0, -52.0, -28.0, -64.0, -60.0, 43.0, -13.0, 19.0, -4.0, -9.0, -41.0, -34.0, -60.0, 26.0, -45.0, -72.0, 62.0, -43.0, -93.0, -80.0, 87.0, -75.0, 66.0, -19.0, -20.0, -56.0, 30.0, 7.0, -5.0, 51.0, -29.0, 72.0, -70.0, -62.0, 49.0, -45.0, 51.0, -18.0, 31.0, -66.0, 69.0, -1.0, -59.0, -11.0, -95.0, 80.0, 4.0, 98.0, 71.0, 78.0, -78.0, -78.0, 61.0, 29.0, 97.0, 12.0, 85.0, -28.0, -19.0, -40.0, 31.0, -22.0, 70.0, -27.0, -75.0, -100.0, 94.0, 65.0, -89.0, 68.0, -72.0, -53.0, -74.0, 56.0, 45.0, 98.0, -84.0, -60.0, 70.0, 30.0, -80.0, 8.0, -84.0, -73.0, 11.0],
                  [-86.0, 44.0, 27.0, -40.0, 99.0, 55.0, 83.0, 80.0, -80.0, -44.0, -98.0, -67.0, 92.0, 73.0, 12.0, -32.0, -3.0, -97.0, 37.0, -13.0, 96.0, -77.0, -82.0, 93.0, 39.0, 64.0, 88.0, -5.0, -95.0, -30.0, 10.0, -1.0, 51.0, 30.0, 73.0, -88.0, 41.0, 88.0, -22.0, -51.0, -57.0, 48.0, 97.0, -50.0, 66.0, -18.0, -58.0, 79.0, -93.0, 88.0, -28.0, 42.0, 39.0, -74.0, 12.0, 70.0, -98.0, 15.0, -88.0, 82.0, -20.0, 46.0, -86.0, 11.0, -14.0, -90.0, -94.0, -74.0, 74.0, 71.0, 11.0, 51.0, 9.0, 2.0, -23.0, 55.0, 63.0, 3.0, 87.0, -26.0, -36.0, -88.0, -59.0, 68.0, 28.0, -34.0, -22.0, -44.0, -17.0, 31.0, -64.0, -24.0, -33.0, 47.0, -50.0, 88.0, -75.0, -70.0, -94.0, -39.0],
                  [-7.0, 84.0, 48.0, 29.0, -89.0, -97.0, -46.0, 74.0, -98.0, 76.0, 14.0, 52.0, 18.0, -79.0, 3.0, 9.0, -80.0, 19.0, -96.0, 42.0, 53.0, 11.0, -83.0, 42.0, 49.0, -22.0, -89.0, 11.0, 62.0, 20.0, -66.0, 43.0, 82.0, 68.0, 37.0, -30.0, -78.0, -11.0, -76.0, -97.0, 93.0, -45.0, -89.0, -14.0, -58.0, 79.0, 24.0, -31.0, 87.0, 43.0, -10.0, -27.0, 0.0, 99.0, 60.0, 61.0, -93.0, 24.0, 50.0, 79.0, -79.0, 44.0, -49.0, -71.0, 42.0, 28.0, 39.0, 3.0, -85.0, 44.0, 1.0, 42.0, 34.0, 23.0, -31.0, -74.0, 76.0, -15.0, -33.0, -87.0, 95.0, -89.0, 85.0, 30.0, 1.0, 23.0, 21.0, 13.0, 49.0, -21.0, 26.0, -98.0, 79.0, 15.0, 37.0, 21.0, 93.0, -80.0, -70.0, 59.0]])

    def __init__(self, num_variables, phenome_preprocessor=None, **kwargs):
        self.is_deterministic = True
        self.do_maximize = False
        self.num_variables = num_variables
        self.min_bounds = [-100.0] * num_variables
        self.max_bounds = [100.0] * num_variables
        bounds = (self.min_bounds, self.max_bounds)
        preprocessor = BoundConstraintsChecker(bounds, phenome_preprocessor)
        TestProblem.__init__(self, self.objective_function,
                             phenome_preprocessor=preprocessor,
                             **kwargs)
        self.offsets = list(F5.offsets)
        for i in range(num_variables):
            if i + 1 <= math.ceil(num_variables / 4.0):
                self.offsets[i] = -100.0
            elif i + 1 >= math.floor((3.0 * num_variables) / 4.0):
                self.offsets[i] = 100.0
        self.A = F5.A[0:num_variables, 0:num_variables]
        self.B = np.dot(self.A, self.offsets[:num_variables])


    def objective_function(self, phenome):
        values = np.dot(self.A, phenome)
        max_diff = abs(values[0] - self.B[0])
        for i in range(self.num_variables):
            temp = abs(values[i] - self.B[i])
            max_diff = max(max_diff, temp)
        obj_value = max_diff + self.bias
        return obj_value


    def get_optimal_solutions(self, max_number=None):
        return [Individual(self.offsets[:self.num_variables])]
