
import numpy as np

from optproblems.base import Individual, BoundConstraintsChecker
from optproblems.base import TestProblem
from optproblems.continuous import griewank, rosenbrock
from optproblems.cec2005.helper import generalized_schaffer6


def f8f2(phenome):
    """Combination of Griewank's and Rosenbrock's function."""
    result = sum(griewank([rosenbrock(phenome[i-1:i+1])]) for i in range(1, len(phenome)))
    result += griewank([rosenbrock([phenome[-1], phenome[0]])])
    return result



class F13(TestProblem):
    """Expanded extended Griewank's plus Rosenbrock's function (F8F2)."""

    bias = -130.0

    offsets = [0.2471, -0.8497, 0.5629, 0.2673, 0.6874, 0.1395, -0.2764, 0.4172,
               -0.5007, -0.5204, 0.3347, -0.1486, -0.806, -0.6658, -0.3413,
               0.5511, -0.032, 0.7507, -0.4674, 0.7704, -0.1435, 0.3436,
               -0.3479, -0.2553, -0.6601, 0.703, 0.0566, 0.6381, -0.3636,
               0.2631, 0.0015, -0.5821, -0.1722, -0.4897, 0.2803, -0.6424,
               -0.9881, -0.8055, 0.5302, 0.7047, -0.9234, -0.4617, 0.0438,
               0.9782, 0.0832, 0.302, -0.969, 0.2856, -0.6621, 0.1814, -0.6094,
               0.4327, -0.7442, -0.2942, -0.908, -0.325, 0.5898, -0.6485,
               0.2407, 0.7544, 0.4963, 0.6119, 0.0229, 0.2758, -0.7977, 0.0727,
               -0.066, -0.356, 0.1402, -0.1662, -0.6885, -0.9176, 0.484, 0.4022,
               -0.8278, -0.1385, -0.1059, 0.8147, 0.4007, -0.558, -0.7805,
               -0.6397, 0.8605, -0.5285, 0.8931, 0.6873, -0.0634, 0.9984,
               0.5468, 0.7913, -0.7351, -0.8013, -0.6145, 0.6804, -0.6249,
               0.2594, -0.0397, 0.114, -0.2449, 0.5812]


    def __init__(self, num_variables, phenome_preprocessor=None, **kwargs):
        self.is_deterministic = True
        self.do_maximize = False
        self.num_variables = num_variables
        self.min_bounds = [-3.0] * num_variables
        self.max_bounds = [1.0] * num_variables
        bounds = (self.min_bounds, self.max_bounds)
        preprocessor = BoundConstraintsChecker(bounds, phenome_preprocessor)
        TestProblem.__init__(self, self.objective_function,
                             phenome_preprocessor=preprocessor,
                             **kwargs)


    def objective_function(self, phenome):
        phenome = [phene - offset + 1.0 for offset, phene in zip(self.offsets, phenome)]
        assert len(phenome) == self.num_variables
        obj_value = f8f2(phenome) + self.bias
        return obj_value


    def get_optimal_solutions(self, max_number=None):
        return [Individual(self.offsets[:self.num_variables])]



class F14(TestProblem):
    """Shifted rotated expanded Schaffer's F6."""

    bias = -300.0

    offsets = [-73.6029, -23.5497, -21.7737, 18.1344, 32.7235, 34.4135, 80.9458,
               48.1471, 19.1511, 10.1028, 30.8276, 5.4233, -55.6725, 26.2495,
               -28.9147, -22.6582, -31.8386, 8.8951, -86.4539, 26.274, -89.4711,
               34.9452, 16.9669, 88.1296, 88.5252, 64.4829, 9.885, -0.6096,
               82.3252, -2.7587, -47.9628, 57.1884, -19.7915, 70.5187, 68.342,
               -74.5351, 48.6429, -65.1319, 18.2676, -51.4645, -14.3595,
               78.5988, 8.3485, 18.7453, 83.6457, 7.1207, -19.4943, -77.7495,
               -34.3259, -50.355, -41.2909, 69.5211, -14.1954, -20.524,
               -27.0148, 63.0445, 61.0951, -7.7856, -22.6848, 72.1543, 12.9292,
               -33.1681, -60.6627, 77.6392, -49.8472, -13.8409, -72.5563,
               64.6183, 54.3257, 28.1504, 22.6017, 45.2104, -64.7017, 79.9052,
               -50.3032, 83.4179, -88.8969, -45.8726, -40.305, -71.5232,
               -42.6497, -32.7813, 15.9176, -57.2241, 35.4867, -39.4787,
               53.8841, -88.9035, 64.009, 66.8546, -16.7644, 64.9093, 63.48,
               -62.6336, -30.0712, -22.2469, -24.9448, 11.3499, 47.4943, 55.384]


    matrix2D = np.array([[-1.4056330576581519, -1.6797625184089098],
                         [-2.2209090940411649, -5.1976570199697059e-1]])

    matrix10D = np.array([[-0.18991744426769389, -0.005598698866628199, -0.8105063221720983, -0.8421574184332983, 0.24717798776489058, -0.5680637122620588, 0.925222911687658, 0.0005385340265248797, 0.23922309010707596, 0.559329597049149],
                          [-0.5849886573325191, -0.5118320741701468, -0.3603516984144615, -0.5195683661022106, -0.2624383937776598, 0.8516042398313597, 0.1915401621783247, -0.5286088308437045, 1.1148833615542866, 0.1164478005367021],
                          [0.6087769145798748, -0.30344695967890417, 0.2130706903915233, -0.2576704858070898, -0.2944039775981049, -0.8315773535025596, -0.37567149498877905, 0.920947816023411, 0.9801592858921598, 0.7824623506773289],
                          [-0.04405619530668686, -1.3587374627929365, -0.7265939735631874, 0.23271308093695908, -0.34441013087785666, -0.2335564355432666, 0.16601741129456543, -0.6011527443386572, 0.36395003267714854, 1.4040023208854677],
                          [-0.24794866064306814, 0.1540773020726714, 0.36412794070560633, -0.10869314057360066, 0.22449421437434025, -0.660949798426504, 0.14030470312052284, -1.2474833289025289, 0.5891311542267833, -0.24322201470823795],
                          [1.2172160563083443, 0.7267250187724629, -0.18987943437849253, -0.1517960966071571, 1.0128243477255854, 0.012144768049060478, -0.12890323390296288, 0.35061825512104, -0.036579719047451435, -0.1300418990286989],
                          [0.47448271063154346, -0.743620182345349, 0.4012069247158402, -0.8782981927791382, -0.11343026015895796, 0.30506688225626666, 0.6245294189259485, -0.958260372966208, -0.6427490281387698, -0.5758545882066424],
                          [-0.15476758558573708, -0.5269573890289317, 0.18620618579268733, 0.30852477992180527, 1.1919382410176234, 0.027550913032174815, -0.26708915757679247, 0.4910329570569554, -0.1174596603932355, 0.029100749528743156],
                          [0.024889824915382136, -0.9994434550063793, -0.7880276777919281, 0.2649512877591258, -0.2932068503958782, -0.30169050790391716, 0.10555277693059643, -0.1131618742881271, 0.4870529445485998, -0.8492027499983884],
                          [0.5008345979942692, 0.037950146579611266, 0.45260062856627253, 1.0101800214227374, 0.33546493344741213, 0.3621928681623734, 1.2876382423603074, 0.2510224174014867, 0.015581772505550484, -0.0462021604735075]])

    matrix30D = np.array([[0.3375298323788134, 0.06274349137579888, 0.0636981169729991, -0.3816696975937898, -0.013302559930396748, -0.145564470084809, 0.30412025130830056, -0.7210820715623202, -0.4476854127973274, -0.2674576643672251, 0.1880917007787116, -0.1384328893825545, -0.571071009339713, -0.902351770832736, 0.07028518791333825, 0.17458739519753844, -0.101229975176826, 0.15714297144652684, 0.17831585191466412, -0.04124922052608053, -0.04487827730707412, -0.21652840631352116, -0.6038699647524824, 0.31573497104070986, 0.3232451810506717, 0.5314398437933591, -0.38325546740040684, -0.041353235156597314, -0.19300906858300532, 0.4141537935014383],
                          [0.7742715167625139, 0.3028795643091619, -0.3564281176778083, 0.15114789491430763, -0.6780590679429641, -0.44959507007764016, 0.15988117617263253, -0.17145997616788194, -0.20524947405174318, -0.20550242019323733, 0.2807544903473642, -0.07049099713322435, 0.30022240293713043, 0.7765803174093536, 0.6840491561867478, 0.16700608900464836, -0.16698987550865838, 0.19553400139079008, 0.20047657163075294, 0.058816326063288904, 0.01682897688804791, -0.531039866439993, -0.048638641405760075, 0.355724781225609, 0.023097522751532783, -0.8033254075293895, -0.18938299070906514, -0.250040324205261, 0.08514178956704267, 0.31207797373007645],
                          [0.3752727779327349, 0.19220856873991127, 0.3175774898364107, 0.19632003781171348, 0.4948542231399113, 0.8434212551181173, 0.1862367143732219, 0.0003633675303486017, -0.24497126884330295, -0.28636745363948435, -0.8987085134007647, 0.09955915859116472, 0.24237898935419813, -0.1148754301435829, -0.12485696129940083, 0.7206360867492647, 0.19897404779410416, -0.04823791654567371, -0.11637446992566887, -0.8088144548455982, 0.04612195097720366, 0.08035000440863062, 0.46830906407626344, -0.029337029005820678, 0.028829157170233367, 0.04908778626088928, 0.21013072922260673, -0.07769249251286925, 0.2877111037892098, 0.4503410700666805],
                          [-0.03648059168579122, -0.399490580784493, 0.37112209613555464, -0.29069795932004966, 0.09385011394229481, -0.46991194577185547, -0.07356334660697986, 0.65675417304681, -0.3429936594706532, 0.17650393243444698, 0.2768782338792548, -0.06746767316622047, 0.2014517278485906, -0.5744968155652443, -0.20065850002595656, 0.3846533659252828, -0.4558392056965028, -0.17966026702409044, 0.5865542785985519, 0.30939126845528203, 0.42048250947770277, 1.0309626659007103, 0.06752597183765215, -0.2667164534321976, 0.36997419883956006, -0.2500153118589375, 0.08990480573956108, -0.65301908874955, -0.24619810914449172, 0.20206863286017737],
                          [-0.9147841083287073, 0.2879369985673521, 0.1831109078321392, 0.6736852514299823, -0.629677007689922, 0.7298887206531776, 0.42852749795697537, 0.07563652547527544, -0.09866464725777713, -0.35410441277975957, 0.4014920146765881, -0.1479284773896145, 0.0873853176911443, -0.4619404023810726, 0.7691865837804069, 0.007115432956970091, 0.1379917666016369, -0.40421713805276693, 0.09217514441444744, 0.06939147949883744, 0.2914649485704439, 0.11970680531151524, -0.017179315484363633, 0.3487082515785465, 0.0024409739511734377, -0.033590337485364025, 0.12282720322302318, 0.019687946473899723, 0.3136070117139377, 0.14140614782104738],
                          [-0.32813553525190947, 0.12490722296525465, -0.11371312841650599, 0.04636275919949402, 0.3541226523489026, -0.544541404236522, 0.15327451194831115, -0.1509594478723485, -0.2588323321049916, -0.042140262900239214, 0.42994396292744147, 0.05202586626765401, 0.4801507077845146, -0.09432558645590752, -0.1089280805786371, -0.4091146236702788, -0.8896318294324925, 0.12561704654918648, -0.3198564332632582, 0.10116039657362567, -0.3041889624091651, 0.4701254930633137, 0.21471531027353158, 0.19881952847058654, -0.2992427807681833, -0.46808869131510333, 0.19440957331041225, 0.0750723593308214, 0.07420625947415492, 0.6146574963597755],
                          [0.42020586546261646, 0.34076383803243127, 0.556694699377, -0.2514219215878162, -0.41849137960102445, -0.35859054059408735, 0.36923735991323614, 0.33306120652412347, 0.15790147737227658, -0.19329689451285348, 0.13407601221038984, 0.8530350839108866, -0.3507735773416468, 0.010864803201, 0.21129452880717112, -0.09810318022116944, 0.7863482675364248, 0.2169946625529789, -0.46920613154424357, 0.4029243618649613, -0.10203500628894945, 0.27797958050265614, 0.7964496840013087, 0.5244294013155741, -0.25922666120497884, 0.4958935663484082, 0.04661576803885991, -0.2745272311561289, 0.2516436419849411, -0.17161727486882125],
                          [0.23099500327735029, 0.0662165703317118, 0.3179187998831603, 0.047148831267809936, 0.5008126952971776, -0.05006975425899191, -0.4693269312878927, 0.42182706083135113, -0.2578434111055493, 0.35013420634725656, -0.19185617542664785, 0.16889058861123504, 0.8962784851977145, -0.5535776984733215, 0.5758671965742926, -0.24708108482647428, 0.5773981487801989, -0.2635817203271533, -0.12555095235286554, 0.045837076010761424, 0.05576135943044598, -0.3115269556060708, -0.34989819639348974, 0.017620861620264577, 0.10884072545097653, -0.08061351607913235, -0.2849887679472038, -0.08707572687706852, 0.0401483633997017, 0.053343969809192204],
                          [-0.4627987678666829, -0.7816836850909911, -0.13934379034597602, 0.37255199259023875, -0.22599087943628848, -0.10647328614367432, -0.4201495870700444, -0.20569735752833854, -0.30155002136867315, 0.1779616206122602, -0.701772432398147, 0.6891284677370144, 0.057349941438958135, -0.12660592072048887, -0.05680723183792084, -0.12201405662301303, 0.2625780174616034, 0.44282430733674494, 0.3211065879612928, 0.07497618004967588, 0.025936773749760257, -0.03068489633961858, 0.002668349231600184, 0.1387204229596921, -0.5619717442492322, -0.22135965234067373, -0.559940523591063, -0.22910343768698377, -0.29737508486443814, 0.37244507477668926],
                          [0.27739418123190784, 0.19967570434769757, -0.38562712581036646, -0.04971903048407664, 0.6898282381462536, 0.2823016259146069, 0.1570429496142992, 0.35079901009081765, 0.21409247736375703, 0.5825055464809651, -0.3579520713869547, -0.03380403023664721, -0.09291153578344351, 0.028304897226369006, 0.48418013914368735, 0.5317996038814284, -0.10900549105377373, -0.2598295986117351, 0.5828971202747144, 0.6341881766172949, -0.2199486224854303, 0.16475251810835906, 0.007494184895589395, 0.30248808567078256, -0.029984105334022, 0.04731684402886319, -0.34417396511384046, 0.738468766909104, 0.39434143588453563, 0.5326368987603627],
                          [-0.39871290175631247, 0.6545986674315583, 0.37848279612788366, -0.3783567611930276, 0.3034201150636166, 0.25073257897393875, -0.7158479406505922, -0.051373597340970384, 0.06072860394258302, -0.4191918825251367, -0.20308114819938589, -0.12845776974379813, -0.8418488172366169, 0.7677711842613896, 0.5910418612663394, -0.1513902564523408, -0.37305470712746286, 0.13049846402422963, -0.29429638110371087, 0.19638563987715582, 0.3235307315207371, 0.1596765316684111, -0.17966974145559395, 0.6116132293849914, 0.37556660863072955, -0.2532616398357708, -0.08025937184682871, -0.10295535422382981, -0.09446897779279781, 0.2249374107111487],
                          [-0.27733523775584534, 0.4981212739838658, -0.9040205071594238, 0.29425360339186696, 0.31509804041265466, -0.07694110460480705, -0.10836421761606496, -0.31938109595403164, 0.16700978124750093, -0.0717272792658114, -0.10055765468873534, 0.7800971465724618, -0.45201732323663024, -0.29081691851926883, -0.18951810548157966, 0.18522488352025748, -0.2776335132433423, -0.10554069062416369, 0.43832845762375805, 0.48239298427364763, -0.585885118234503, -0.407216006826867, 0.08783434700316192, -0.5362238316604614, 0.2361545105429738, -0.032927296240029304, 0.48801834464290933, -0.9093155091643131, 0.2705780917385158, -0.17228751771295184],
                          [-0.35026377624814176, -0.5191136298544367, 0.041814361593600115, -0.4593264024621773, -0.11804006099384445, -0.2869781082919137, 0.47928820532573396, 0.7578354415552735, -0.061163676576846224, 1.024547504054853, -0.4108435057199059, 0.021753817352954622, -0.18234010765326775, 0.05286336380705399, 0.016680507281286623, -0.2367963220971287, -0.02601018386879642, 0.7893085171581321, -0.09037815815230166, -0.13705584027320167, 0.0025818296618910497, -0.6034480680391126, -0.2673916395652133, 0.21726005489867134, 0.3964592180201004, 0.11110446816055203, 0.8421908426688614, -0.21495598328636478, -0.269663329444035, 0.35662584596733815],
                          [0.1469492478636134, 0.332361175830377, 0.6802455170011739, -0.5377830002331848, -0.03083229417499596, -0.2565515855115268, 0.5078675213932998, -0.2916019101498528, 0.24329183056550963, 0.49283777472399687, -0.3199272044948286, 0.02211042510359062, -0.7201187802440863, 0.2869659529472252, 0.2418799987709168, -0.20252203380480127, -0.2802789165790325, -0.568504548220147, 0.38649805216471256, -0.29111002870290287, 0.15889083266854442, 0.3959533769650681, -0.1967576903655079, -0.30468535448697637, -0.9300550483718429, -0.08865978032973716, 0.45283423921179716, 0.1273463933220339, 0.19366432006906098, 0.09342884499041255],
                          [-0.173896242924032, 0.40975324798010737, -0.015085645735000939, -0.23515911860776026, 0.48174516381590055, -0.3862947345751654, 0.4977788360905092, 0.2767016861636895, 0.2918108566293055, -0.509610478043906, -0.08102400788542456, 0.03485657883033094, 0.3005121969177384, 0.06696082107963269, 0.16218781530162818, -0.3801589258967074, -0.4950060629547128, 0.37146574272729915, 0.13258908400708272, 0.23918429333240376, 0.0960757478221621, -0.5075817138371813, 0.028531116680339336, -0.16342484901956492, -0.560207276473936, 1.0608644326333703, 0.04838037051258804, -0.13601647788224058, -0.25036741814475094, 0.21568940762576205],
                          [0.5231809915555192, -0.00951053448989514, -0.9314825426968341, 0.1607106470342237, -0.533651643510215, 0.6207373326198545, -0.10420039344829869, -0.19671408577827695, 1.0324487696036364, -0.29382855540618547, 0.02971300313159091, -0.14514741296775854, 0.3297579654701213, -0.2883200427988319, -0.06975292082695836, -0.6023827964288274, 0.08152931546818591, 0.03951769941626822, 0.009408578274874915, 0.03229220260390411, 0.6810578991746964, 0.29052627575070844, -0.15939945847807285, 0.16244848319933122, -0.09604613615629795, -0.2703200463919698, 0.6353291455100376, -0.41818073494151614, -0.0790378627197314, -0.10203654382947691],
                          [-0.045952245890551476, -0.20084832493310145, 0.21346913715663204, 0.36577188773240227, 0.21340841952108297, -0.3740575359651276, 0.8541950663383082, -0.029821046483487726, -0.2802754405486632, 0.02692780105076585, 0.08721417365348706, -0.20907987050061907, 0.1101254211267892, 0.34351841019068424, -0.5297822670697593, -0.4400764176511861, 0.3504851110360528, -0.2145997484253341, -0.0458370579286653, 0.26214210989211656, 0.6889365460669724, -0.13636904101752165, -0.08969114791542207, -0.6299677977603639, 0.5963350699603904, -0.3874639394456691, -0.9256894465902593, -0.2855784662288512, 0.359252247812872, 0.3205598670999693],
                          [-0.05407900717889332, 0.30901025983762354, 0.23128233171919976, -0.06735405940389379, -0.4802859590918669, -0.5884975018264222, -0.023316996340643884, 0.06419696171245155, 0.9535612193692651, -0.11250959666872201, -0.42756394856974167, -0.5593131832663752, 0.17309447730086294, -0.32774163845043974, -0.08818534183914815, -0.09425623362906395, -0.15885610932681785, 0.6316088036417242, 0.4987994723873175, -0.2988266192736495, -0.22751096641344862, -0.27661306487609055, 0.3875166973889187, -0.3770188705780694, 0.5380340691109297, -0.4427908091014706, -0.18230966113183744, 0.42303543059595466, 0.19520020797092882, 0.502931901140279],
                          [0.41957934384951506, -0.1637236145154945, 0.2019901184626331, 0.9017071667391443, 0.21291090353989195, -0.3000298939647383, 0.2711308511573576, 0.6671592622023593, -0.07467135293695293, -0.31888313076076236, -0.5356379196918157, -0.21955440704646934, -0.05556254601316861, 0.1347840966419646, -0.7779715378862413, -0.10987781896885693, 0.06902315191255792, 0.5909096032305277, 0.39940428410301243, 0.0982253668099056, -0.04922097924739846, 0.5207382619298911, -0.4135492352002125, -0.40583491990312104, -0.5026014614377597, -0.16248217172625964, -0.09348839457750524, -0.2378213840132282, 0.1727410462088588, -0.28736537747835],
                          [-0.08557311304966217, -0.6058749308411604, 0.09596323720819382, 0.443270323634493, 0.031132243287128103, -0.22744267253699338, -0.5760366128606915, -0.1662693188417899, -0.1903435093367724, 0.13368369784845888, -0.4814201233900204, -0.5090455347175888, 0.4845578262666549, -0.007051058408878746, -0.028645660044352372, 0.6880589282274618, -0.36722974330452635, 0.030952435855909566, 0.05945273340937935, 0.3281684066836705, 0.06135250070401392, -0.5207377302224129, -0.10006387652889195, 0.6498107247235859, 0.005843583904903804, 0.40920238763682204, 0.07816671803530112, -0.11244826149581458, 0.7963052109919482, -0.5251784362004432],
                          [0.10415771342089057, -0.1727496050948579, -0.3227733341726266, 0.1574588751500625, -0.3891235400814336, -0.5614791715068235, 0.060777852721861654, 0.5946457556019736, -0.37291682023290534, -0.3425006352354232, -0.6202321463940236, 0.1834509802888794, 0.1267395119964736, 0.37212970516953964, 0.19247949317941826, 0.3011734317638609, -0.3343940062707806, -0.7002476326627656, -0.5542336818533097, -0.09609993004699033, -0.3015855020300577, 0.10078576277802774, -0.13643136215452578, -0.1437732889555569, 0.6743158845057899, 0.369129582793294, -0.4337845500077763, 0.2686172970415085, 0.07104702703701712, 0.20426800061715272],
                          [0.2334144888599089, 0.5317238913457153, -0.2956042529210083, -0.22147795079975435, -0.5192230496207726, -0.016483704030897384, -0.17990175399509098, 0.003956830949332343, -0.15968905616532522, 0.35389466711712886, -0.039499346547104304, 0.5369345281671375, 0.37769938695973915, 0.23792637540681574, -0.35111293696014284, 0.02188824625457761, -0.5529325566550066, 0.18888716768271327, 0.020409005102658525, -0.21774406896791237, 0.7748966836991966, 0.5267427606793781, -0.49291323652907126, -0.02240012557311958, 0.3377110080129067, -0.1741849906280443, 0.08535645981163184, 0.41457716136745304, 0.35674961958321855, 0.34624771310162206],
                          [-0.13336931299196075, 0.32179803942158103, 0.49967748185492333, 0.3682310143534618, -0.22190487517370452, -0.06842111042381906, -0.22838492398408392, 0.1505181032621511, 0.06506035959409856, 0.5548350534083474, 0.6614529079040404, -0.5635298143851342, -0.29925207585450214, 0.13415278846898396, -0.3904196741160039, 0.42551203258215486, 0.28592767383659345, -0.21941666797113982, -0.08977986592559799, -0.4505596437040868, -0.2887846374674396, 0.12539427584973126, -0.22494015128661868, 0.725935477541561, -0.02978066978501868, 0.26488129583148384, -0.09036824242617847, -0.6808880342826722, -0.30415433215467624, 0.8923864701233948],
                          [0.30375689668918765, 0.19425093197511786, 0.7330030139849023, 0.2354883110995077, 0.03195892644179795, 0.24525809119386502, 0.09685572404060047, -0.6428265068536505, 0.6023515812388063, 0.6287298634678442, -0.4803408461007882, 0.07079814314671551, 0.2437665169432524, 0.22757301666037325, 0.5048784117643349, 0.3275224474296488, -0.20898105147898546, 0.5651402956564291, -0.2879681306893284, 0.07049287550313812, 0.025578123475355784, 0.4768057184334622, 0.2204941308070314, -0.4512885775065991, 0.32899804207611066, 0.1611486138333136, -0.08570061550394042, -0.08647528910211173, -0.36887549947330966, -0.07523039656874181],
                          [-0.14856061738663645, -0.24619716648213358, 0.6183492321015802, 0.45108005646141736, 0.36907303134325425, 0.051480906595633105, 0.12444988269954924, 0.304211967470923, 0.962403940221372, -0.11020765527372353, 0.44038741554085864, 0.8495419147066039, -0.05019078246450156, -0.5273503376011149, -0.2679040812166617, 0.1049650741176635, -0.5590011311662733, 0.23184955675745023, 0.36946513149056165, -0.18609205315958022, 0.35424180927710835, -0.4835334516400682, -0.5807076057561821, 0.5817588975277239, 0.25847877117650797, -0.1499874406652067, -0.25233121369354106, 0.7038382485339378, -0.19880225593769332, 0.059782292948078915],
                          [-0.29826855084537496, 0.005547717759123244, -0.5343361566879246, -0.3425565661100969, 0.8271895717451807, -0.3687273054007548, 0.3020522462629295, -0.18273155453302578, 0.012887237724834785, -0.006924388505181095, -0.486802169796696, 0.03144379215939405, 0.2617667301402404, -0.1420513269379291, 0.20698141300164866, 0.0030224641221819754, -0.03820664547178732, 0.2961049547332606, 0.5863768499096672, -0.716352107025, 0.026918689204340443, 0.6581290315699154, 0.014207113321751785, 0.21076250437176952, -0.02187017389358367, -0.02488407615549957, -0.25702763122268885, -0.22585542736039302, 0.06638751883507651, -0.4220234215664128],
                          [0.028222904334379864, -0.30532521244911304, 0.13685914180391628, -0.2527211289547533, 0.3388144960672168, -0.1758758321002492, -0.20953118730079673, -0.01015104991487667, 0.20762520196404544, -0.11936745034193355, 0.8790257229743874, 0.24140826382934044, -0.2192955802682027, 0.3407261998524924, 0.211485542835781, -0.07361071845617911, -0.03196535033686417, 0.12077812999709997, -0.23975749253496959, -0.7048503725344866, -0.09816042345777422, -0.06489064800516695, -0.0029328537746811933, -0.5516941400051366, -0.06016991136397311, 0.3381546306931629, -0.35215460899229134, -0.3003764852283933, 0.9409792474130629, 0.4831315567973405],
                          [0.21048138181288423, -0.3607889649163684, -0.08962228854947313, 0.35489613495750494, 0.11251149002863087, 0.5862890564286569, 0.16052084312961226, -0.04345941961205818, -0.17977690995091944, 0.36331469265418803, -0.026926290913898594, -0.2515400488083467, -0.5653516877755107, 0.233985777665291, -0.04668618329039237, -0.38692095572848995, -0.8756243592959183, 0.08849304147133759, 0.45457569883715443, -0.448128758049083, -0.12192979320964423, -0.41188739836089255, 0.7824217225435128, 0.42735236975794055, 0.6927083295705568, 0.1542367662641297, -0.7009759239554423, -0.07362820742060144, 0.1915803802453185, -0.43824101560329026],
                          [0.3138618590390267, 0.1381827141774622, -0.3860937375912319, 0.34556474473272847, 0.4833615605820018, -0.7520635859190808, -0.21210710752267165, 0.3004892285424099, -0.7859274937018786, 0.18213463052171727, -0.0407255193145239, -0.13351860242682206, -0.2764978310546308, -0.15307856644134224, 0.5743500393284625, 0.5738027974775745, -0.026600822000319463, 0.04479388513888803, -0.35503771391025657, -0.5875402205755466, 0.5037207212187844, -0.07095598643198867, 0.26732927193807143, -0.11445190830944826, -0.02216763457009811, -0.3286887440789256, 0.40414409226867104, -0.051352094933992815, -0.4185030167005709, 0.02207597372770559],
                          [-0.08001592608278307, 0.44483910511398406, -0.26809138345210737, -0.31603665975776457, -0.40500556272316385, 0.40660540433757447, 0.20432398393022283, 0.5243173685768345, 0.35220987544971, 0.18342912694978894, -0.4308112191308184, -0.1025553662248487, 0.07661561955299324, -0.6011006659652751, 0.07450335205139771, 0.411205322062162, -0.24591955528864118, 0.1434972224322226, -0.872804321603206, -0.03548625441044369, 0.13353173738243349, -0.2757900963688434, -0.32698513440632615, 0.06506289981974661, -0.5939385643176319, -0.2138361613669488, -0.929605152100944, 0.018117709856617557, 0.4453720818580381, -0.2551190257514694]])

    matrix50D = np.array([[0.32397691027305126, -0.2936431082609151, 0.030541058894425856, -0.2422425325995264, 0.25964135544524347, 0.11450932197408345, 0.1401092777555897, -0.19213904678568144, -0.5868727624825523, 0.06926278209472139, 0.1303080675310439, -0.17913170231094067, -0.1774195828502948, 0.16044241585266167, -0.004059343120983066, -0.24339005727289248, -0.19796578266924922, -0.011617215840432692, 0.14423796499633357, -0.06002344559962039, 0.3564557396214197, 0.22463012395816262, 0.3201217558465226, -0.563716876469897, -0.2877554060658339, -0.05919962450940408, -0.06773244986578589, 0.13562641812638374, -0.16086080797758798, 0.3031622860619994, -0.3692526954836028, -0.008866658589060213, -0.13599246832217018, 0.11441599792110431, 0.3846804122567686, 0.4759617198598991, 0.29611200380803004, -0.12050212059896326, -0.37767430618984543, 0.15985991885507592, 0.08711267487555002, -0.5258443227357014, -0.08445138609459751, 0.22169424173418903, -0.00255747548602586, 0.11964077798313023, 0.2531503996847524, 0.17931803179957628, 0.2803414389373369, 0.3242702303031809],
                          [0.0679935608924171, 0.2689357335728322, -0.16771706642399412, 0.2761579489066069, 0.3992502365685539, -0.20643310899650816, 0.2368209272420583, 0.7337487522087921, -0.3559385675175405, -0.65093941844893, 0.19763123028093393, -0.09168755129420374, -0.25440163013543365, 0.3895055311922778, -0.23141361036771715, 0.1577646815003589, -0.33815640789893475, 0.42650500815439, 0.13167506157672434, -0.12549961567254084, -0.05608620407308822, 0.23602468047397707, -0.179400881024066, -0.05560832487551409, -0.6136792272892835, -0.23721139112993733, -0.10481566978173469, -0.13087177071826447, 0.28546734722103523, 0.1927687596806915, -0.32908016791920713, 0.045791242366262, 0.2882170336698725, -0.041098633876316674, -0.4065876575862268, 0.13197258409822932, -0.22028965750044896, -0.23144805245139183, 0.20852175082579433, -0.22780513606861877, -0.04087581791780609, 0.24995758227822767, -0.058483205077989, 0.03785439240041574, 0.007276874104538155, -0.33602002972380407, -0.2664081080325936, 0.3277754160114806, -0.06667031974032142, -0.43242131864674854],
                          [-0.15586415193180944, 0.04735137311759721, 0.06573756919985885, -0.011002948134856468, 0.22768517275652328, 0.676183710112375, 0.18516798349627478, 0.149842825124789, -0.0832851747623066, -0.053903019926188254, 0.1256533302461045, 0.6269137821639893, -0.10430807343820804, 0.09912891420625089, -0.44306853345267017, -0.45364515211916157, -0.3313992538276289, -0.145024526181335, -0.6599343392307612, 0.08785441307639402, -0.3293748453699597, 0.36535363710188157, -0.26871287653844717, -0.24088827380971448, -0.05666272684450284, 0.5528976059109364, 0.23076655599673634, -0.013286176749724397, 0.16529165264335277, 0.11569514917814658, -0.059241052192645305, 0.038529425460292234, -0.32410569324503363, 0.22981045087191665, -0.31630117597276547, -0.022416260579633515, -0.1097241951620803, 0.43831039899474566, -0.3576163010315645, -0.22203592760403074, 0.11943919636903208, 0.405433793359223, 0.4318778520821694, -0.16560213355959863, -0.29931347489508464, 0.44958895597770876, 0.22330815160262105, 0.47030076605240256, -0.06960463882704852, -0.23651624418494865],
                          [0.3062125502619084, -0.4534991901015876, 0.15670934364488637, -0.2369991924869512, -0.16614802125580058, 0.13314115537415905, -0.08320733041084204, 0.23506063691362242, 0.4017125656748858, 0.05290621436760881, -0.13119123260189292, 0.5146705537068857, -0.2231286619884548, -0.31671204331922753, 0.03666579173502935, -0.05193153149792079, 0.3582332055897515, 0.044728240393773655, 0.24575482206156685, 0.08852283459177149, 0.034333899936470076, 0.2284182551286699, 0.1348868262776797, 0.0639464036920535, -0.44525447619115655, 0.36883767571447956, -0.10492608307205015, -0.13161172075843214, 0.34070458695990063, 0.2789127533681659, 0.060488354811971554, 0.2467309388776523, -0.20492444846597002, 0.20627469673003826, 0.10366844974350603, 0.34091911778001294, 0.31876223111543445, 0.16765058230805602, 0.3767307259635297, -0.16721086779383507, 0.550601906217574, 0.06424964724489889, 0.35325742864207677, 0.09478650787402183, 0.13973608917651106, -0.576077227825549, 0.2336084524870056, 0.4807637414984893, 0.1445815682066602, -0.36580631527710405],
                          [-0.17815693097428145, -0.10841500203729193, 0.2294799189779049, 0.2962482940641338, -0.2825484616228266, -0.24534742557465003, -0.4466280387650824, -0.015651736067279088, 0.1805515362431991, -0.016058680382845823, 0.1427698308252779, 0.05017033456281584, 0.28994536971905155, 0.15427634896346015, -0.11216715004786004, -0.35285927359759733, -0.2906858623955336, -0.19118636505899295, 0.041014396563072854, 0.05645614121555422, -0.4478384855829484, -0.1606777596367316, 0.3785758631677552, -0.09863014475657504, 0.11066935181196494, 0.22132523263786097, -0.09971083417810556, 0.42485991924400374, -0.32361236121356546, -0.09759819619379168, 0.3193928119898154, -0.14093420109486332, 0.14305034669760502, -0.29365623401278945, 0.07249184805370876, 0.22155566560616768, 0.018140945495610506, 0.05572617618702757, -0.3535840820712265, -0.4961432948798424, -0.18657237800578905, -0.001120232855174345, -0.17936756363163217, 0.45571331383889524, 0.34239736342468474, 0.19693221279414574, -0.2911044522678472, 0.23724648591707892, 0.7572586301809598, -0.31214103500588175],
                          [0.023269259262679004, -0.14295215185495844, -0.3330237276652427, 0.03208429992209828, 0.30993660712327886, 0.4780832934855034, 0.07890421014096918, 0.30718062624338016, -0.2013054191029408, 0.4129229351383594, 0.0975442549857021, -0.0070593107494656675, -0.014944233084077298, 0.5753328917795542, 0.009123255750094341, 0.0857906715863238, 0.6046435478649945, 0.1304993181770018, -0.24150596263783225, -0.005004819097890257, -0.049266944439768284, -0.43755488970352246, -0.16634534795784203, -0.16426641714936244, -0.13144844171129916, -0.033232229202171185, -0.32742713240001253, 0.4362535645390082, -0.028845595604504354, -0.0625844798369868, 0.13356805166401742, 0.2861594502000094, -0.0873204913448217, -0.16499006024001375, 0.5351229835708343, -0.2888593059574718, 0.19607730704835064, 0.12457454167565878, 0.56768123422707, -0.21773672781810674, 0.18521141531643973, 0.7106149723745881, -0.09892602084662988, -0.20103271428036257, -0.29558777485641896, 0.29277489673548873, 0.006042185903240484, -0.2927393605606532, 0.1797836928213912, -0.3119576988588374],
                          [0.14558116250741754, 0.3283389409488613, -0.3440556293863024, -0.46897656899449713, -0.11002521535009671, -0.08380008067538422, 0.14309313990386333, 0.19486324103401537, 0.2138367252974957, 0.4455878768233825, -0.2928475478374887, -0.3038425820408686, 0.43895715059957036, -0.6314850728530368, -0.4245702618558579, -0.4053386627217472, 0.057954084691841184, 0.17885311078985833, 0.1798563832748947, -0.3107022465614065, -0.13141002716621616, 0.3678610961794793, -0.09155824261284241, 0.09315557120298289, -0.12286230375133962, -0.3843591835793633, 0.014440857888299201, 0.1828760895905881, 0.10942399032694193, 0.11631080233752707, 0.22628605356579967, -0.5661961248438325, 0.07811528454022379, 0.027106404481116442, 0.24919639744494324, 0.26500514305068434, 0.5194793798302735, 0.3494569443779628, -0.029726418434712135, -0.3483004220795613, 0.5142506926786666, 0.342608196775908, 0.006758332892132791, -0.10609559144771222, -0.314167042691269, 0.24876566104812126, -0.329637054983244, 0.17326146362922873, 0.08893853084202268, 0.17922273187710236],
                          [0.754016822659674, 0.00592181877615602, 0.4564272695308298, 0.31127141147508247, 0.4697219985824977, -0.20414762180924648, 0.09447029585427187, 0.26669146261539156, 0.4510408835700628, 0.11302176041966373, -0.07596272233059818, 0.32354498166568424, 0.3540527532064912, 0.07067893407607925, 0.2988995354917276, -0.0665353453208129, -0.3886290220175512, 0.26044270899616695, 0.24960643887606174, -0.21034772748300629, 0.10900399120710862, -0.20171992870789912, 0.3076128550860964, -0.1845786727409521, 0.13988668164226517, -0.22012887928533975, -0.4028225027924817, -0.054262073129150115, 0.3498507163856964, 0.06838695357393697, 0.2969444965524744, 0.09543185120456843, -0.021723985748858525, -0.1182428539395533, -0.20407972220055204, -0.3446359241283783, 0.15029721374861388, 0.1778783471159134, 0.13108760317708723, -0.2775422755278204, -0.8885142918692758, -0.20946388493818874, 0.2886365370196074, -0.2417275268233362, -0.1795398739656435, 0.018179788098747426, 0.031067996546779945, -0.09553044487862468, -0.3346318992227741, -0.3036208902089364],
                          [0.14773621674775286, 0.024919949796253896, -0.12152800818042783, -0.2877974926580601, 0.37172704263656103, -0.4637820877146326, 0.2016607649434802, 0.025822084751485126, -0.26144817838179807, -0.004404666902472332, 0.25312159635121967, -0.05764988031500462, 0.1378310449547902, 0.08395159969917958, 0.18373797976768044, -0.1449468809454031, -0.2753259585312015, -0.04888420004591705, 0.3403653304661512, 0.14121547852585148, 0.5002351195001006, 0.35752381129947725, -0.2265225878968179, -0.524993709193335, 0.016981120556557334, 0.0718509545379352, 0.13999141620321234, -0.03414118422102056, -0.3233697688989249, -0.014524995082875525, 0.009609155715016634, -0.21308578160132838, -0.03444779191051212, 0.047908912607793404, 0.39216752693389345, -0.7344853357092284, -0.3709259152980765, -0.016319903505016716, 0.1107946755147166, -0.20304720977022772, 0.1893489929638577, 0.49217694009858765, 0.7107072852059376, -0.07955805154816271, 0.4604681179100967, 0.002660394671870963, 0.009155463782389225, -0.22884098320440258, 0.19077072591489788, 0.2912340053675071],
                          [-0.2463365697890391, -0.21128556057079953, 0.02661571599677539, -0.06877937075018885, 0.3100695925447799, -0.01526252536590916, -0.13011371746451664, 0.018194937665093894, -0.3054304634046148, 0.017188298732608437, 0.01122216614147159, -0.031064928803235543, 0.18537200951541874, -0.3192922872905495, -0.16360604859640704, -0.2975779070604563, -0.39230879959130327, -0.29315397265927906, 0.013393114797500008, -0.18040991847380172, 0.19754473917433363, -0.06514155414995583, -0.4404236176763727, 0.3086136859633508, 0.02941574927429339, -0.09897348399493983, -0.3421447617383321, 0.054687492924307274, -0.13764354179193028, -0.015465156611765306, 0.10799474908548921, -0.519641921915005, -0.1855980321417438, 0.018360291016117115, 0.21640827661606887, -0.1376112405165008, 0.22572586415372897, -0.6326527933226159, 0.13917720929922484, -0.08631356112773413, -0.20891403907026482, -0.06086340720464752, -0.022084666761174945, 0.07204145140940656, 0.24771811509016634, -0.6385166124715005, 0.17544109936364077, 0.2477750451805451, -0.0036360794314131833, -0.5533889564938084],
                          [-0.11685731196699478, 0.2663113938731957, 0.10896484528054333, -0.06198049929777504, -0.029056863183616, -0.045318030552030306, 0.4234635149348497, 0.013833893978250921, -0.3132447024613347, 0.2682081033355069, -0.11384866719860477, 0.262880544605648, 0.3681029509300358, 0.3938598193575643, 0.020809752283060586, -0.2878330035677149, 0.19818716597306046, -0.07352266633310474, 0.39103043326861964, 0.3125529921886903, -0.1611434661610574, 0.14330253985456143, 0.16378223826564514, 0.4064258815777235, 0.2739218766481246, -0.8569186664002624, 0.1693239645540904, 0.125849554728838, 0.23129306343473421, -0.43356744679487436, -0.12173578071216329, 0.5768822212658196, 0.11250282044529013, -0.1130072018048843, -0.2551370758116056, -0.07000503680286191, 0.20178830796602834, -0.34556707329721725, -0.12938808927241152, -0.336431211259312, 0.17400681953783712, 0.16902134762977375, 0.126438898598458, 0.127686269571348, 0.32119752483861014, 0.03448014171102267, 0.3896418813113085, 0.4215807271736637, -0.10851650840430643, -0.19928350138470244],
                          [-0.024705802486232352, -0.12152320733213129, -0.4573172152173218, -0.37319804748443247, 0.21482833992107986, 0.22708422386235474, 0.2597067619861658, -0.08991807347161966, -0.1537521522385987, -0.2295698918386874, 0.4935183674253109, 0.3360248778821521, 0.2198950703488239, -0.38787980281413087, 0.1836700344627613, -0.009644553999040503, 0.0775602447248523, 0.42567797717137223, 0.010961504073930714, 0.5321292701983846, 0.4333119471493844, -0.3004855655305269, -0.6659820573477234, -0.176429051538236, 0.21648999891626408, -0.06464480557994585, 0.25419150064449936, 0.12634446131848479, -0.11501086596233374, -0.31173607665122743, 0.7718464753198706, 0.4500913133516377, 0.54715135312894, 0.2552473004574125, 0.23164661825779742, 0.24649161812535156, 0.07389555329608131, -0.0015664474006338684, 0.1393052792637351, -0.501725773427366, 0.1061114730323093, -0.2485250666233723, -0.07399562437748058, -0.1703206482571162, 0.05647483475299126, 0.16271123315338976, 0.1614742273045123, 0.15191091319554048, 0.2235322002351506, -0.4045481655644463],
                          [-0.03455956756885048, 0.006739147910386073, 0.15377749692922155, -0.22646366000767873, 0.3100534124534554, 0.4500143574904157, 0.5092903957784256, 0.27261893878497384, 0.7004747179324613, -0.13722879664226653, 0.04554181752328666, -0.4496583887924451, 0.08100161972947775, -0.11192132206044524, -0.04024643790357551, 0.2969726570253557, 0.17629126678740634, 0.4198232134608214, 0.17000766672308487, 0.5208656370198169, 0.18780670062097873, -0.09601033432619868, 0.22470524394463012, 0.16009268289460918, 0.16713263461479283, 0.19181266372785422, -0.3271225831001034, 0.12325794779671106, 0.034370899704101926, -0.03615331766491135, -0.2528717277299076, 0.06841255334966675, -0.5965097415480116, 0.1637351287844202, -0.3871901427789416, 0.3303732465823094, -0.26677191072089523, -0.21224766221971939, -0.21202542699410304, 0.1979321167642018, 0.41308542462662257, 0.21848059889586674, -0.02775249317130354, 0.22589124062087265, 0.43218979131794616, 0.0007166716078402802, -0.3846244555206318, -0.2029186044045051, 0.5421321254414366, 0.024049205459776665],
                          [0.021765336567917324, -0.15680546410228852, 0.10518990346854212, 0.2953043589716132, 0.29792983483707564, 0.2495419158314025, -0.2166485972101586, -0.1899305146726589, -0.20570281222514392, -0.1226719190754255, 0.014751842277164154, -0.41161203400379004, 0.5295544210371347, 0.20824458038447957, 0.21048614112519068, 0.5241839059852149, 0.04470340784663285, 0.06831355517459996, -0.17039303687279256, -0.057598629519298825, -0.40106188685259203, -0.066557713080267, -0.5244086916841039, 0.17437935961410925, -0.25554755834914866, 0.1362264366205147, -0.11122470941665671, -0.2484027424441892, 0.561688388781723, -0.091734653713691, 0.19937586945047633, -0.33773987302376623, -0.3427683033582074, 0.2982187860959107, -0.5118160511045569, -0.21330071197697414, -0.07688562197193352, 0.32079115508940564, 0.08923982433119444, 0.05651870735667304, -0.04727103128158912, 0.11553897693208312, 0.23407003099056334, 0.5957817034883812, -0.16706405506653302, 0.23117836645917578, 0.28596288611986276, 0.4451660468619188, 0.27909026823997396, -0.06798915583824192],
                          [0.25467127890016283, -0.23685649357466376, -0.38221404668227177, -0.14382566671912334, -0.11574263017143264, 0.07770311166960955, 0.17895617220257776, -0.45322397683209753, -0.2443924742900186, -0.2586457631602463, -0.23136946120060087, -0.038796322872587, -0.0248643978899471, 0.35553614203444456, -0.2704622386003048, -0.20248741994839406, 0.4611148268092026, -0.3684802839345853, 0.012724480288368704, 0.07283152916049215, 0.06100596224092308, -0.23748157131399367, 0.07293994055701593, -0.2440841967339234, 0.3996269134162763, 0.11231180262329608, -0.15819383801872908, -0.31756631902955335, 0.22505047720753882, -0.1248019615421643, 0.02832131757510237, -0.22555084582596271, 0.37298298301322336, 0.09242193550751333, 0.021342402436446847, 0.3283794452341422, -0.27089033562110093, 0.2322859267089788, -0.175414334086325, -0.013494281216956637, -0.3134360687945272, 0.03238981534091273, 0.6267993767955471, -0.12005904137010617, 0.070844867688156, -0.36920304912696694, -0.7204168289345704, 0.3601594713280228, 0.10551219958028048, 0.24058214633378233],
                          [0.3421641502026581, -0.3538431959980894, -0.23573239955142106, -0.09224768256735942, 0.13673808292881193, -0.6572807704277775, -0.3173411075884596, -0.052139452048714216, -0.08339481369447689, -0.0003694415324779071, 0.32676861463840473, 0.07791236865998868, 0.053477110028767955, -0.07512785869094987, -0.14582446466085489, 0.4272180530858794, -0.22150145103461835, 0.3017607078550834, 0.11411131863834964, -0.24299286365205608, -0.18660815870772213, -0.28182229769942113, 0.05851987479974538, 0.3886979139735834, 0.1638502630121564, -0.1888851240933327, 0.3081360119642289, -0.4011268467227843, -0.07597395381354881, 0.2967366969866387, -0.3486498635892983, 0.12350255755253081, -0.28320211073421914, -0.2847073029684747, 0.5435064333028856, 0.348337636187772, -0.08158691883185866, 0.08070839927059652, -0.5167257114352038, 0.09821945134731291, 0.08659442772726243, 0.5138714118440583, 0.12899040564226355, -0.28673599259529947, -0.3887131810544835, 0.2549085867234362, 0.001685978468618293, -0.06501007633949836, 0.2069328475012695, -0.006694797843050715],
                          [0.2389544918985937, -0.28102868281399596, -0.17043187548126876, -0.04124369250323039, 0.09506145696111482, -0.26053150633757755, 0.23648315375952433, -0.18167733767627853, 0.21329617181910412, -0.23991559879123092, -0.359195426394645, 0.03435071492607059, 0.04025509968166412, -0.4405006717044003, -0.28516354284625295, -0.10213995554826952, 0.45946194654766515, -0.032398967197821586, 0.3962524632948791, 0.00944433129216542, -0.24140413317412196, 0.08359416653291629, -0.3356301359445065, -0.002263238606219733, 0.04168573285592814, -0.05632131084727879, -0.09443714934211053, 0.025042139249153016, -0.6506924883511495, -0.09734855196878983, -0.3778281345036389, -0.01563656260722202, 0.08032561575260441, -0.3445755373485002, -0.2942066569968706, -0.33307693653427417, 0.10385590551651594, 0.29311085635718337, -0.30181088456035826, 0.21713494653770374, -0.09719574274263519, 0.43465059870673195, -0.3688208245841363, 0.12398549621317678, -0.5706793746045905, -0.11961163618538437, 0.16537299695910126, -0.21661968054262853, -0.1718946070165784, -0.18141082363699157],
                          [0.032626828048189865, -0.25670397249434673, 0.2916809453457959, 0.025869017155108528, 0.16701879635359118, -0.2745570432906414, 0.09651783993717561, -0.07603977055528378, 0.23495664651246165, -0.28855580297314165, 0.05147997518149588, -0.40090017228535507, 0.31912048046977587, 0.23622650732339168, 0.19956100135108024, -0.05799509731942684, -0.3573704142354331, 0.1045834236656035, -0.41525721344726285, 0.039861070865491136, -0.017021101403833266, -0.17292024326354113, 0.21040580205661866, -0.28211890728219263, 0.48854094213540583, 0.26085324784854896, 0.18184996903871953, -0.6038899697857579, -0.03303085579469582, 0.06464281286818346, -0.11208645856175997, -0.007599123118119163, 0.5436712778015969, -0.27849372357824587, 0.035909568017556445, 0.42697693071718745, -0.18893457965541888, -0.08802456393866376, 0.4677161788814444, -0.31335476778188875, 0.5757142407863671, 0.4321133152744399, -0.21622446867010792, 0.3435443146541028, 0.15232097936628286, -0.2962686492354645, 0.09990514585236432, -0.11785988384750426, 0.13868325723881747, -0.056760717120902186],
                          [0.3578809044491193, 0.4917838007514219, -0.009682817084517226, 0.05898338979580791, -0.2748434251334089, 0.11453839377556796, 0.09433279801171146, 0.26243835187444325, 0.00502373371545882, -0.4515647938136144, -0.23933226507629882, 0.12490410886795236, 0.008518407337553692, -0.14928257258396968, 0.16811443257781072, 0.7075700301491331, -0.16634632227395843, -0.6671807798833191, -0.33450182343662066, -0.1296367795462818, 0.20660804271717675, -0.5033740870177339, -0.2518515959721578, -0.4536403175035468, 0.17105880683320107, -0.46551179492185807, 0.004388291910366898, -0.3009296348517466, -0.0538944750532194, 0.3965736149785571, 0.5713413562747627, -0.047848414498081165, 0.03138803403225218, -0.16446394278880577, -0.09251258242748249, -0.431558703089838, 0.1295142744603407, -0.08134599020898577, -0.41416959424015976, 0.18703257679432259, 0.3656421616677943, 0.10704205260260691, 0.10961483426775367, 0.18788505369949027, 0.04486011881372501, 0.24035293177079334, 0.04827430866393457, 0.152015008730184, -0.15541171552707841, -0.22763744574993125],
                          [0.7817005835063673, -0.16298262265420396, -0.12412856342192949, -0.23523143042291692, 0.18795486967238953, -0.0511180436177054, -0.3295433045293285, 0.24360544722194705, 0.05649341845955206, 0.015974332852163944, -0.1933162015755603, 0.17597416651128678, 0.4553850601798133, -0.11047557301381233, -0.12903837452104275, -0.11522463961301072, -0.09641833346609027, -0.28541129448553326, -0.4272127586593766, 0.5067568210846152, -0.6542004595061384, 0.7966763566752733, 0.29545013215219373, 0.2243977738236599, -0.018993627588939573, -0.22685942357971947, 0.4096378988866675, 0.1629361707238432, -0.034663295368358556, 0.24413668799884258, 0.10311525668973776, -0.2533356255630351, 0.43094890844373707, -0.10934781545572557, 0.061068432180295944, -0.26865603317021247, -0.16252088913649584, -0.35745896188228127, 0.3294025101625107, 0.766806283917625, -0.07193769210052751, -0.3160126776777664, 0.07434064501114895, 0.11103931217752368, 0.10893056830902845, 0.06461488497287636, -0.004144232463247512, -0.1941222695216615, 0.2142927197744959, -0.12318138799592289],
                          [0.30818322154592936, -0.46522601069197345, -0.30241987693079864, -0.366218158570839, 0.1017465356287602, -0.038864843435560424, -0.3090746853368868, -0.059957219137704604, -0.18813175104927055, 0.3411746293346833, 0.47082627164453855, -0.215441552307794, -0.5767775313953861, -0.1590523197198709, 0.06680637861246418, -0.06680869980809954, -0.265276774739169, -0.4025093764990326, -0.3756107742778124, -0.7495927492291401, 0.009943601953045075, 0.13094822751148008, 0.29146534227608584, 0.3850542570794388, 0.17612946659246953, 0.23563210223670286, 0.2820218589727051, 0.2553533647678363, 0.17711105743689481, -0.009203926714528455, -0.05633505114547079, 0.33397079335771956, 0.0925811046286277, -0.020448622633040647, -0.6447094058749637, -0.1553164221238831, 0.14373386166480584, -0.283432094544106, -0.3208060764908073, 0.02666912760330393, 0.41096440595653183, -0.13522740646092524, -0.13505503736582147, 0.36058366723550017, -0.15046277896185284, -0.40994394252275607, -0.24281417946143533, -0.07941934122225881, 0.13106084871221102, -0.5996924102207343],
                          [-0.3687970066908685, -0.15699949231807986, -0.2896323246976527, 0.011925253412678534, -0.3321079021415415, -0.20093158137991848, 0.19253395072148136, 0.3475551537014937, 0.29236105888659847, 0.21386017680590508, -0.30220901499723174, -0.012400270378145976, 0.13825188765074284, 0.7312353085968964, 0.7075445357521708, 0.128590163297447, 0.1305486249624658, -0.5167752616031288, 0.3958416180914786, -0.04508120956318456, 0.032053420232495994, 0.23180656314724787, 0.35882930693412235, -0.044340284277642805, -0.1139935132718265, 0.1634283870662636, 0.1513578858578529, -0.24568868958218507, 0.12354184148186753, 0.01010336379726598, -0.18974963698556158, -0.5264361513608381, -0.0036811087659817526, 0.6715970454987725, -0.10221672326457258, 0.2877273876646457, 0.05604790203770497, -0.1824137191414638, -0.01274024359559954, 0.07017035293643901, -0.682261534984952, 0.3765050306825546, -0.04297850780868516, -0.2183323161480819, -0.07871184447701984, 0.48604529979383904, 0.28486714969797494, 0.06405295848167951, -0.2166015365450955, -0.3240856527337768],
                          [0.2820342741221696, 0.025942588854736645, 0.13498921398298205, -0.250915815540449, -0.02147547070793991, 0.19953603038165857, -0.06120947640447304, 0.07646177942419255, 0.18880123049353764, -0.4109647525929359, -0.08114650354741057, -0.5371852008616361, 0.1638148506448442, -0.06355789936340413, 0.29107568762909913, -0.28943841062596254, -0.3122512769650914, 0.029356776625066024, -0.41899159766434, 0.4528940503627292, 0.17605985986981942, -0.3471572254767026, 0.03488936277532738, 0.3682675842351206, -0.6804915841944603, -0.06963114278552333, 0.28247826682985555, -0.053343153148856855, 0.039590117940636715, -0.3483910452793132, -0.06391497220218886, 0.289705415901836, -0.04098891685861597, 0.10773093071445132, 0.4506618373983425, -0.3513067140268469, -0.09188299280689485, 0.09758222074246056, -0.1919538426728179, -0.25526376008528934, -0.35956711457400253, 0.0713449304673408, -0.23010171189304185, 0.16442623171628262, -0.02469256321995998, -0.08339020042694749, -0.20407755727017976, 0.3355277949355261, 0.01994658998783961, 0.16023222341035348],
                          [0.1550048209264011, 0.438528134212955, -0.16401454948423, -0.6186550378610582, 0.2407835426973056, -0.3524243664596816, 0.12955596402603056, -0.3107748860021094, -0.26189196679122984, 0.14678124193326836, -0.09118362350799673, 0.22745927798078364, 0.49362083823351527, 0.22332719931046927, -0.5243002708258909, 0.4420242801184692, -0.09545782575529428, -0.30769368093690463, -0.07504601838478381, -0.12646697900572573, -0.20805637367633198, -0.39948087130067517, 0.2830868513109486, -0.22944424217492182, -0.5151936069762683, -0.04903693257202696, -0.03911005240205086, 0.394534270187238, -0.49796007812556514, 0.3149858102011144, 0.35998578181775465, 0.25390327391095824, -0.49388876393376757, 0.17155229001713432, 0.03674628553722341, 0.3675558662094451, -0.36189122680826713, -0.2860809183027937, 0.4486353312798933, -0.018912597736454928, -0.006573616754372531, -0.2050534187293047, -0.05842209559665364, 0.23691340453484763, -0.16712108466639158, 0.06880543827789691, -0.016783184829305582, -0.24179236669324286, 0.2543010198057103, 0.19306799122298351],
                          [-0.05667920077128435, 0.47913144257537194, 0.08071598486456293, -0.22927485805267636, -0.3194947396834868, -0.23567464317071685, -0.019775364962298668, 0.2652589788383577, -0.40228295270346426, 0.1499255512413924, 0.023780370982586323, 0.12568532356825057, 0.6123001534442516, -0.077227967253947, 0.0262329830343324, 0.05996973146038561, -0.2563402969318256, 0.20080369575353138, -0.08911898840314797, -0.108067574836173, 0.23232194080292684, 0.04538700823867684, -0.3444191126921379, -0.631432442240296, 0.09952286187187495, 0.38000010376509685, -0.287389538608626, 0.04118523043837081, 0.3070496687978054, 0.187439609595336, -0.14277653370892376, 0.24210254880979834, 0.27330358985517034, 0.11836394409931078, -0.5042673366779319, -0.15719177572617263, 0.5927816803604358, 0.15802846198532752, -0.01749635395228306, 0.23523490866677996, -0.5691167790420346, 0.2451390369462848, -0.6374482035189367, -0.3282463645812911, 0.250619425090395, -0.36128856677911936, 0.35153640342654807, 0.43890651439080874, 0.41076708947780305, 0.10930884582288958],
                          [0.003572912382717862, -0.221055751555288, 0.06034007714254647, -0.06572709048018999, -0.28659805665323496, 0.12546957663423688, 0.4431478775446367, -0.11867239383997866, -0.18774891224834275, 0.36532142888321556, -0.01927676014305474, -0.2912955350620936, -0.32429155381293645, 0.1580737227346115, 0.42943197471084915, -0.27593186052834856, -0.09503273996824348, 0.5413627533724804, -0.5562349955265906, 0.29261275890684946, -0.35116492826766954, -0.09492471262716944, -0.4681038999349498, 0.011519923412244, -0.5278407523225042, -0.10082865876620753, -0.06637003729158779, 0.041277377509555055, 0.2815172250204333, -0.37732955099049176, 0.14929833735343515, -0.4483851618624511, -0.259624529763985, -0.23766023513109558, -0.11120880176826226, 0.1453146158149074, 0.28559814581717374, 0.12162089393967604, -0.4216920962403514, 0.12647540738796986, 0.10042076468907524, -0.04853265718855618, 0.3453011802454126, -0.02741610394852643, 0.023516576627719835, -0.19992634514532265, 0.4992800693955726, -0.3977442114939688, 0.2527348423441627, -0.3405097385662369],
                          [-0.05299262579650165, -0.21306450103661334, -0.08697490182757078, -0.364373674387887, -0.05589393161421558, -0.26088713865895585, 0.22501057928772117, 0.31774173129808725, -0.004299948251492047, 0.19144629003981123, 0.4556475020269424, 0.17330384584386246, -0.1533885721096193, 0.05542672068059594, -0.19391667949329305, -0.09640762008998685, -0.3885427419674046, -0.20785386833711494, 0.0376221410790544, 0.24980704530843298, 0.1920721764165766, -0.3471292075638447, -0.22835152222459484, -0.11698918968123204, -0.007800675510508984, 0.10101232493394968, 0.16945670932945167, 0.11599260029710806, -0.11418882426922894, -0.6716697269589423, -0.2086525104644385, -0.1755652768986757, -0.19091464154411025, -0.32850102289003913, -0.42002034212548456, 0.045601196711857264, 0.1334745178292971, 0.3207134029048939, 0.5044395500981355, 0.1731679346204012, 0.15682687226036224, 0.06425157279779299, -0.07421658694356609, 0.3491560531238662, 0.2096723764598118, 0.2725522351059314, -0.1722894572489507, 0.19220454871149256, -0.6904476231367649, 0.10155012345529744],
                          [0.43232018421006907, 0.1731565552632071, -0.2630248829375316, 0.21660244583518315, -0.45267086904413, -0.10910198815028574, -0.07687567203195919, -0.6160945436259914, 0.15188507910019633, -0.5835174424945824, 0.2436615715589424, 0.06104557945923922, 0.13364749153456498, 0.18839537538989604, -0.00010055383164530099, -0.3525814129928268, -0.06926379239018368, -0.19422006072980286, 0.11987583982576651, 0.23590692771742341, 0.07549449733451177, 0.40901986767890136, -0.049219640940702074, 0.3612942924554794, -0.07947354079106816, 0.03560218006504484, -0.5284528086367337, -0.07751362087550744, 0.4368369068066155, -0.3674888364687341, 0.004896147235614552, -0.21857608940852175, -0.4552374293737628, -0.266239951286937, -0.2124276432311314, 0.05821792989467301, -0.08903327981528023, -0.2370812546252365, 0.0021778024473647142, -0.06122119369770547, 0.30527466942915227, 0.02065973194215365, -0.5637756475264603, -0.17454318279227332, -0.0498786770019809, 0.12412580189405487, 0.19140411770565685, 0.1415755574608639, -0.07536134084435676, -0.09713338313963077],
                          [-0.3798186744107998, -0.3527016117209821, -0.13608097201911637, 0.05389392314083086, -0.22355123499917925, -0.11457650556091223, -0.045171603579379875, 0.39592152687053983, -0.0851262778200848, 0.1102632767142542, -0.2880403945594029, -0.5653274951214622, 0.2910696385391556, 0.4287437235503299, -0.2658295771700675, -0.024687940706627016, -0.07040099087600739, 0.14445944849821277, 0.03185761631375037, 0.2376092506945504, 0.08983051119283594, 0.10747472125958714, -0.0744565873959727, -0.12964768138496557, -0.0010896700293673576, -0.008960775624301257, -0.2716091808076373, 0.2052868084658707, -0.2714414814606765, 0.41978028667855966, 0.4222644797289709, 0.39313512489108987, 0.05405119028159318, -0.641851820926154, 0.02232428711849862, -0.03795401892533697, -0.6635758104680473, 0.09400368538842199, 0.02523696243712642, -0.07044106737167011, -0.2784969817546758, -0.48052351477727234, -0.011949703670131444, -0.38984784075125, 0.02541357841551201, -0.09480295101471947, -0.1084800453177107, 0.2420675927701143, -0.06867617821467314, -0.15875332637457312],
                          [0.36503412055925416, -0.191725229580527, 0.4034651039342419, -0.22813202309751363, -0.6833075169353392, 0.026777417414285864, 0.2653947852536335, -0.24615339075237486, 0.04476625691223826, -0.35616328727281277, 0.4504084784155492, 0.002490746582235996, -0.055992606889216395, 0.228251529180132, -0.03458857105232208, -0.18576673027929078, 0.07965687418068712, -0.013021775882128092, 0.6256794891522464, 0.4125768874319032, -0.4517722282200026, -0.09794372113639077, -0.4059135817998599, -0.22963723267275815, 0.022287522695533808, 0.0533344173068083, 0.45133283728826107, 0.06181606063574933, 0.185228452856943, 0.6980532100083366, 0.25221372769967515, 0.000538865565497218, -0.7171902241528857, 0.34503965593610947, -0.31801920332587297, 0.08127246887917472, 0.22498219212743942, -0.021774131818256284, 0.22652830477013758, -0.4978174995368755, -0.1267933772964796, 0.021640233132436938, 0.2957636663708505, -0.37237607858000177, -0.21405373854662052, -0.174214027924308, -0.31282847555076265, 0.25064910953559427, -0.03958385720363628, 0.3291509448191081],
                          [0.002554784020017964, 0.09215760119921876, 0.23957785193419454, -0.20801098972870316, -0.20776552084909097, -0.0069529987973039505, 0.38552262584116537, -0.01400504751755275, -0.6084529449964203, -0.47988868123241973, 0.13478635940270423, -0.06629917996618048, -0.0444416907819189, -0.5355187445596448, 0.3816547386441232, 0.04945764226271763, -0.24452633917890507, -0.22680949227410485, -0.1649638470855273, -0.48432363852017307, 0.08097215823809119, 0.11677998061798787, 0.015648445438399834, 0.2615731883593416, 0.16322685622207764, -0.5094145779706963, 0.013036572181013225, -0.049390359389983526, 0.005899822428530207, -0.1802076545373564, -0.25509180438320544, 0.2383088851074806, 0.02134243105331982, 0.022239554635211656, -0.05284669096281055, -0.0009022098882834895, -0.02018554735968635, 0.4218902420658207, 0.4647604186126482, 0.6577404122609182, -0.24908006088866808, 0.06843546674362157, 0.1888130417374941, -0.2445610278135437, 0.438273872433543, 0.20651262632944933, -0.0032107019282176623, 0.15061744910511374, 0.34719090273239117, -0.3229194315399113],
                          [0.12407369572757251, 0.26089646498349783, 0.1042950538397554, -0.2411818652141362, 0.07851948955835139, 0.2750066357591126, -0.006989325954393066, 0.49573724400068264, -0.041602988608384545, -0.458057551126581, -0.17490962399636031, 0.2705860051334031, 0.07833662485270775, 0.5698475127805812, 0.0036248008816198085, 0.13969766915806914, -0.09288251088805695, 0.22076190607423968, 0.7330509353948516, -0.001768883017222681, -0.1540742620009062, 0.2992332197876751, -0.19829990064603947, -0.1465335041278793, 0.18095498762454393, 0.09916127714364059, -0.07671914799454015, -0.04694522336946632, 0.3050659088886819, -0.7358254788989376, 0.21739468353410857, -0.10956397938929112, -0.14041746163307847, 0.29172770159141287, 0.43355771016890166, 0.09622194548117965, -0.15750586750337417, -0.01652654496882436, -0.22343856705071055, 0.33872454102320765, 0.17051720166494483, -0.25982594431438466, 0.11304299852977324, 0.5435376728920018, -0.398638834555487, -0.47147992730266153, 0.034171507811660506, 0.10475473901069225, -0.17547879801926655, 0.012137414047840512],
                          [0.05181634274792895, -0.3541955747056454, 0.30188267671830776, -0.1028809725972118, 0.18718896928656598, 0.29710594722546246, -0.08713005142175667, -0.15142718194795746, 0.16798333440490493, 0.9345537334733276, 0.17897724891458122, 0.013093191280041732, -0.1489836006845215, -0.2534087523713508, -0.28044686979207145, 0.03168401145460391, -0.3915647494375953, -0.49324008626118493, 0.26067137100231746, 0.1579167415220996, -0.2893874903525422, -0.12044848419185764, -0.32806350991575645, -0.34075051997362377, -0.6607171372218745, 0.012742283448161723, -0.2504094237146955, -0.24425827124613797, 0.07747353311101042, -0.2630755114557348, 0.07418147916511461, 0.0663296292562, 0.493054958146037, 0.2652603990521508, 0.11070932075070941, 0.11379354938396853, -0.010631520127835352, -0.34537173022320644, -0.2903651473899017, 0.25200277624610057, 0.46152516432603935, -0.2827976327137795, -0.50671429515121, -0.39097972406408527, -0.104930679419896, 0.03532621291601818, -0.03620235672642558, -0.2712446405638479, -0.03895842336014977, -0.2988453927016074],
                          [0.21121930438392594, 0.027043060507820668, -0.11845628636350108, -0.1495023088407434, 0.16874100415562115, 0.17029597132863786, -0.2862558282373142, 0.11004778187126832, 0.0032404562891350103, -0.21136818314480668, -0.34583980633213934, -0.11993160559464082, 0.353725194667821, 0.1625841152231146, 0.026559209301454383, -0.41585166989274575, 0.35419755507866396, 0.10543253455734966, -0.19257006028723989, -0.5689518294813016, -0.15494646409362603, -0.17000343104701585, -0.7058256046315996, -0.3600261192951937, -0.3066475469649883, 0.215894852297486, 0.34835218804938567, 0.26969490854364686, 0.18845766744858952, -0.21666913680400401, 0.3512603230312015, 0.49701515503358545, -0.1446421600804269, -0.004935002374739733, 0.03407926587573547, 0.32159043198136955, 0.3400458458186997, -0.25599090901672555, -0.009536967281677425, 0.45599274771958825, -0.44909963008864157, -0.02922719176005746, -0.058138690113110285, 0.18796268086363851, 0.38147240872471994, 0.3322048302416977, -0.10067908172937173, -0.3536362102472368, -0.11878130909910421, -0.09853894523911647],
                          [-0.010637204545603143, 0.06156876433509545, 0.26005143289765165, 0.3406402333247717, -0.23936707184888587, -0.12924993557907247, -0.4805842140471323, 0.25338961202924903, -0.29985881790517993, 0.08209176276568034, 0.1721814509458322, 0.14438278856501904, 0.30943008635643243, -0.2336504810353731, 0.6027665705057227, -0.15860460249338146, 0.31507857454294586, -0.015975901448400522, -0.09193532122827733, 0.20214099136507596, -0.11935067970850058, 0.24918905387981363, 0.06069554089233587, -0.3660584606896004, 0.010232363749126009, -0.0832249817500994, -0.050245677595059345, 0.8623184370497469, 0.17905561431454675, 0.14146890326461198, -0.07969721817554477, -0.2626999185172572, -0.0385786808464679, 0.2535808913947394, 0.23179724578503472, 0.19280354871236344, -0.8751050402180799, -0.22966623564939223, -0.47282627214050077, 0.18093717049226468, -0.17136397307927312, 0.43802187214242033, -0.4660319079280367, 0.03568468476274136, 0.006116689557842329, -0.21741576178581798, 0.04931963948540358, 0.025965405256808655, -0.5294290067178741, -0.22854913319503617],
                          [0.05257327069991356, 0.2764649132151136, -0.05388478087004054, -0.13341674439147494, -0.0059651313390318645, 0.26222200757287845, -0.5213510310452731, 0.021499744230975046, -0.3071487073040528, -0.2768010774467647, -0.5280052599329371, -0.029731125221674128, -0.09786817456598948, -0.010351613931637693, 0.5262673090180199, -0.16704835948012908, -0.15547203701835302, -0.3655899797747222, 0.16439447457525042, -0.09268619294870528, -0.37761178883909013, -0.3643403120951899, -0.33081711275866366, 0.1919924340101239, -0.1637592941812907, 0.09312457517199871, -0.5160844512979699, 0.0672161330476135, -0.10939214560775126, -0.03140752326298542, -0.21317352551719043, -0.04026448574114799, 0.2612335661196821, -0.22107471380443344, -0.3055049687111776, 0.18653389033883805, 0.06649668122285216, -0.43581422708631906, -0.08816223916951955, -0.22830250680262804, 0.06264766435098904, -0.027728328286104824, 0.2693799672437889, -0.15930487339332938, -0.6050087508389724, 0.27623631727752573, -0.22845423428808898, 0.31406960314016563, 0.28019333452362877, 0.42493861110910325],
                          [-0.2697734688640779, 0.2881370243334, 0.7171149598116858, -0.27280125515123343, 0.43916487161608875, 0.14634477225400938, -0.0696099083421152, -0.05528675427941985, -0.11619675622895663, -0.04378884952270539, 0.20220971744984798, 0.18720736283298428, 0.27466214331351824, 0.24257830313812898, 0.41688190662640723, 0.011257877771629354, 0.6588329960227699, -0.18761546318831981, -0.14740030941841994, -0.13383217704765824, 0.028656009036290522, 0.5189240491176936, 0.031869452184677974, 0.07848831768643656, -0.2533435323273363, 0.21037080960524784, -0.10396433963331404, -0.3300897278466195, -0.40965229828865485, 0.04361007003709216, 0.1254596991525098, -0.2770181475645553, -0.021824550271034516, -0.39959115295125996, -0.07937387032815049, 0.1271493915607208, -0.16857898246415237, -0.04348183250554214, -0.12517383322774325, 0.011750593859921464, -0.3228359253286009, -0.060760278612402754, -0.039245597261610476, -0.4737997613019418, -0.28035847559124966, -0.023605657295881797, -0.8349827889944352, 0.2644610301058561, 0.1566302640172758, -0.18703560695307941],
                          [0.19107572814693502, 0.12565219823344062, 0.08528637542707576, 0.31575044594045026, 0.14708251359801836, -0.2316833506356493, 0.564080384520561, -0.10066035702400668, -0.27988258079895056, 0.17800748907163033, -0.47810989701508105, 0.0994113152900845, -0.21214539653989814, 0.21708653379127013, 0.03769045676118262, 0.2505418916880581, -0.2748529678294146, -0.0544323313908074, -0.3064597932373567, 0.25089393566742857, 0.04207985673396912, 0.2946407948347115, -0.7187617479917954, 0.6164647983349687, 0.1289515682942916, 0.38928859971558116, 0.00024146668262098152, 0.19469113728784193, 0.277886634102873, 0.3836752987445998, -0.1818653939320371, -0.061194558703503696, 0.035973334025484895, 0.05664269637707746, 0.31857415484640644, 0.10090176909168924, -0.1661388431909187, 0.3766476120114459, 0.02609573328168771, -0.18096390185965638, 0.20325005169796695, -0.25423098225325486, -0.25617667438218017, 0.23647941006741977, 0.05032294690272291, -0.08016967917468223, -0.19067308272098477, -0.13708620589436812, 0.2082651411347023, 0.016995884104812994],
                          [-0.227916309838602, -0.30696840250268437, -0.4431007440446358, -0.10400921693828474, 0.6688255146158322, -0.08500930654717491, -0.19159179546949653, 0.39637483435539944, -0.24035445600496821, 0.00979398938917393, -0.2641392427625042, 0.1585634142206677, -0.29882523877151523, 0.03876337543093675, 0.585569296239151, -0.025913325851227475, -0.12592948389142156, 0.08609385612342096, -0.21950334190134807, 0.19047700691219516, -0.16149588479258103, 0.2827721251646676, 0.37513751279204305, -0.17969433137873483, 0.31344686810953737, -0.2517431375837434, -0.18982648423022555, -0.2340785694374732, -0.06932398438739719, -0.451741480564841, 0.18713763448557286, 0.008485256381730002, -0.6608037475601884, 0.34188449852172254, 0.02445807314886242, -0.177971283961045, 0.022460716620618108, 0.6853454479165844, -0.10271136718074332, -0.08535182528354185, 0.1718392234771049, 0.09849977948196288, -0.5224198315084768, -0.011918850228076936, 0.3903140409994602, -0.19399835023424994, -0.04588379809084946, -0.1372243009084217, -0.17771852787770417, 0.4014971846880661],
                          [-0.43642692834143615, 0.3857000907838976, -0.12026057784896325, 0.30016776391301747, 0.15441823512560943, -0.21035021850763372, -0.5089733755345045, -0.051470862906178576, -0.03330936227500053, -0.02335253561930966, -0.01134642256320498, -0.1205527025978543, 0.38255844546569034, 0.15906510240803046, 0.4184834340105293, -0.3967525420090674, -0.19578985197441443, 0.06829193021148507, 0.07196528098240482, 0.6931740105383886, 0.13534575117377384, -0.040892666088280606, -0.07370398934038372, -0.09074727661168772, 0.08283847212694978, -0.26820992419005607, 0.33710435502130714, -0.42588784945080094, -0.5364112369674269, 0.056337860718191204, -0.3169439512222694, 0.4596913616427222, -0.2699983132934961, 0.3780869768297391, 0.2000138454217616, -0.04789506497887107, 0.3441102320573355, 0.1792747341236356, 0.3329648825501552, 0.23740134023557616, 0.2954360380560094, -0.03579695938110446, 0.419626613902673, 0.26842917503272273, -0.35415240061868186, -0.14308778083206644, -0.5211294974367717, 0.05155341544068101, -0.06729088506789146, -0.13522534905055125],
                          [0.04386290170274419, -0.030015973622569074, 0.230447759052138, -0.2454715976477281, 0.32544809556755416, 0.1340898561832826, -0.6663908266775905, 0.17886953296683972, -0.5408182194756982, -0.11801836962918102, 0.010525127534048573, -0.0403873236142891, 0.009716181834596938, 0.03600020045802258, -0.5249081752748246, -0.21490015571365523, 0.25137628452240757, 0.050083732049918055, 0.16013401950400744, 0.37548586879091767, 0.48195564268263485, -0.3042483832167948, 0.5241196317378116, 0.43624455438459964, -0.34046540223916627, 0.25205317848706466, -0.1319742895395443, -0.21244138626591336, 0.12986677177852682, 0.09155950740224614, 0.19103030151152386, -0.2670997715506575, -0.021509751697052545, -0.196823868431602, -0.3627425711217929, 0.21798612442308732, -0.2667896970870727, 0.1600149121080229, -0.02227792498028865, -0.616677045697508, -0.11274009332692567, 0.35032151588494176, -0.2338633954830153, 0.13162829438093884, -0.038207943249174425, 0.25592884422088724, -0.10675012669842335, -0.8502817094367158, 0.09372151209036918, -0.08947160308487913],
                          [0.5256775345001266, 0.2168518022325412, -0.18056330926660938, -0.3404935942854697, -0.1330904425162624, 0.5091396233888706, -0.11783363435234168, -0.17722519242292442, 0.03664616396556088, -0.0186711772620909, 0.26880746903364655, -0.31659629123200506, -0.4363834759319545, 0.4167856477403476, 0.36279785064307946, 0.25531835993829105, -0.4214858828407437, 0.347307027248227, 0.17666410286477058, -0.052776945829624514, -0.21409446403293844, 0.11927632987974879, -0.1342108557715787, 0.5900257242493855, 0.05684820290140852, -0.04114075214415989, -0.27642268951898763, 0.1684056293893434, -0.2596878310055264, 0.6154588019687189, 0.2777902054905606, 0.03701442717236067, 0.02837940386602497, -0.11967886019346596, 0.11427025048288421, 0.0890149564366238, 0.13861455989507393, 0.44615276323401915, -0.0806693177165882, -0.0003212624445014661, 0.19794441753460493, 0.3036341823099765, 0.020077437226937507, 0.2378961515825428, 0.5593108748622568, -0.23167966388343345, 0.1124977892762665, 0.11609235536111533, -0.4194564860636659, 0.06885041197052413],
                          [-0.2302450362262233, -0.11353918599330565, -0.2701628939123228, -0.05658360456768857, 0.024761848395809735, -0.20684960814069905, -0.23731018792378133, 0.5881682668550592, 0.09795762648606503, -0.486538078737359, 0.49076694996847864, -0.02318629709018958, 0.4396539480553894, 0.043382930648559616, 0.09747137608296405, -0.027956474390239745, 0.32296761238854554, -0.3117549872087597, -0.31836922403880774, -0.018868739565307036, -0.004074484898462356, -0.1261568015803155, -0.14289298560862645, 0.001894633777753986, -0.022516183213851758, 0.04722086400031285, -0.19929907602280164, 0.3869862481875929, -0.01430095028800369, 0.2183264389980227, -0.2681776470903505, 0.06304468740403708, 0.3475026317193079, -0.08265666408236985, 0.1372342399466151, -0.27564064905529484, 0.46256277130191087, 0.33415857878498356, 0.09553478586866088, -0.36496585544586413, 0.385375477021354, -0.2951810439816278, 0.3714120961648371, -0.5392346396196452, 0.05389738602425793, -0.301644535546707, 0.18337065219719698, -0.6963667543666325, -0.10017115342007801, 0.24940386132179707],
                          [-0.5893317433839286, -0.23005939871767955, -0.047689130766300276, 0.17773302409573866, 0.11549410880573688, 0.053451763916639965, 0.22412387254380314, 0.08114180443540837, 0.29272800792023734, 0.029990849963488195, 0.6057149369447367, -0.12757313352894786, 0.11364541619835482, -0.1431908671297521, 0.14304059212383585, -0.035336279349609576, -0.14205077565117763, -0.36948450105743125, 0.05035905527727404, 0.18875997343804427, -0.08180933267534198, 0.563052767191073, -0.28593720208491763, 0.17367672899921488, -0.3701089771716893, -0.010606595438968365, -0.5131113354482344, 0.20647710082792647, 0.3653586577751298, -0.08017984259587371, 0.4071165655136601, 0.4950803015407727, 0.12717799366994348, -0.1766931099369038, 0.09498206317542157, 0.13074338187176435, 0.1373533183490928, -0.310559592752105, -0.38541700236667065, 0.02997632557516215, -0.3872193902643263, 0.679792103992042, 0.18385270820986777, 0.34145545767969687, -0.29906974856515567, 0.03910759201321474, -0.14751617677771114, -0.16293992444285726, 0.15575661807215926, 0.4473200145497854],
                          [-0.02486477761935898, 0.0905939979394151, -0.0930087163371224, 0.6230690327137215, 0.1301087025316014, -0.11978067818501055, 0.30963663478686293, 0.07069021485287974, -0.053858512154458, -0.36319273653169154, -0.17816287580292892, -0.13724454363960598, 0.11759874114327236, 0.05147374653397575, 0.3458231801417736, -0.30963971457463535, -0.01986936461413378, 0.004062894002846533, 0.1935152636492727, -0.1833975598158804, 0.21944479524031024, -0.26755945215016264, 0.4262512941647567, 0.46664579348135904, -0.19799571098290347, 0.2567443569631369, 0.4642071210859123, 0.39196636043578753, 0.14571360378517761, -0.0026643970398511907, 0.6904044828168912, -0.39060392927362564, 0.1685033495932154, 0.1498479843966996, -0.17258152148065367, -0.13798900549723395, -0.025261264351451124, 0.019606732016813413, -0.10224993848747471, 0.6626722431165497, 0.2472027858505941, -0.0431142044069425, -0.09058461041857627, -0.23902219929273943, -0.3683833072229844, -0.24289659819386808, 0.47263426116350093, 0.22750086494357286, 0.1309235149844046, 0.40412942179692496],
                          [-0.19459256507199005, -0.2451610467054431, -0.12510167386976015, -0.3018952283185278, -0.38368114740088827, -0.0423183648978639, 0.34946173539677666, -0.289012196808126, -0.6012364065120547, -0.327478282373159, -0.33684370855355833, 0.1347754169709336, 0.3010673962794075, -0.3319648370214194, 0.2588356045204205, 0.2724031950446369, -0.10699691154541834, 0.4193587138821891, 0.14906089420762472, 0.15622706852202534, -0.5607091033193786, -0.07168698614732906, 0.3814747148702975, -0.11310033125780578, -0.25014430915100777, 0.20844674749525932, -0.473695169630533, -0.3211744752897471, -0.26901796989869037, 0.14097302320696042, 0.07286373257080489, -0.10751401369753946, -0.28616281634223756, -0.10004176557738262, -0.04767160963684687, -0.04796351370855834, 0.4707144054473952, -0.5102593567390673, -0.10313679550575779, -0.09109401019388516, 0.48378468366724503, 0.22494546683306518, 0.017549946757635562, -0.13133427213322568, 0.16494735952145256, 0.28340413363736017, -0.5824914803653346, 0.024113951955529754, -0.32396632564073946, -0.16753978179964424],
                          [0.5241599569170763, 0.04319944889456292, -0.03705316832633285, 0.8142030792244265, -0.20636904506366918, 0.5090525098025525, 0.07839508855373746, 0.290248018471825, -0.2991547500742936, 0.0204429518750274, -0.023468727277708865, -0.2691014026380276, -0.06213830794217705, -0.21212171812451874, 0.12203312653529999, 0.4602600717329894, -0.14668964741863902, -0.2590603927663207, 0.0915182751441396, 0.3093327220522848, 0.5258179996090356, 0.19050786947413656, -0.11211466605514461, 0.21372774137340378, 0.030070446813509122, 0.04128786315764661, -0.05848080760551277, 0.1581401714025653, -0.2027595601793097, -0.5557075975226569, -0.14502705642114924, 0.27138768929875073, -0.11236183777746929, 0.3479808891260209, -0.17766943575623248, 0.29762268338713493, 0.24679122623069263, -0.019298421490261848, 0.017689210834573976, -0.2934936542969116, 0.14158040580820777, 0.23468218611282726, 0.15898493092553123, -0.14254192223163017, 0.026557583133537968, -0.09492692269053588, 0.1423927093778428, -0.16577110275818452, 0.041969894531982035, 0.11125473257665061],
                          [0.3357220835732336, 0.3157274106694554, -0.5614365928921001, -0.01281888411221432, -0.05619871226629489, -0.08456188446617113, 0.7069555678458488, 0.34337588727579965, -0.08702224095240405, 0.21672646297120096, 0.3242500347162012, -0.26746986720187577, -0.0333120020325509, -0.5220852734335818, 0.11635765004843734, -0.2706271761138452, 0.27400881421217105, -0.21770183740323554, 0.5591743434468861, 0.23247081032335043, -0.6926855396975323, -0.3359926642586294, 0.03852984670534437, -0.1841385702970911, 0.01755326724749561, 0.49005501471135043, 0.10556520681351451, -0.32619315386593206, 0.1053905865386873, 0.17443502919148388, 0.10250021475422363, -0.043579754266374886, -0.2141414509906018, -0.18689969788389812, -0.04497471178827943, -0.2598286803306744, -0.11605201183943242, -0.4301329862615527, -0.15618119155935467, -0.1846490566475455, -0.12023090163161054, -0.5059462394343432, -0.055825580215487836, 0.026474217405676313, 0.16411358188337324, 0.013889500652237291, 0.2625470258341287, 0.1640324392526895, -0.12021433527468521, 0.17548845387588966],
                          [-0.08182862339316894, -0.23522349787427518, -0.5606602741571629, 0.3665886760553592, -0.16107595112546957, 0.6125236849809548, 0.15483049122528114, 0.16047072004167778, -0.07459920123064266, 0.25901194802935645, 0.042202016111651065, 0.6539056896563338, 0.5787593609113029, -0.38008337766269, 0.15008222146974218, -0.18079684655954809, -0.09658296105973568, 0.00525579799448327, 0.28618416807322883, -0.11209534276114856, -0.16473339996749117, 0.12183959877114732, -0.0341548367147853, -0.23873780240971762, -0.3290643870856043, -0.24849715112706544, 0.3748913780830617, -0.2166654728375817, 0.037694228037001655, 0.17558195510446883, -0.35342579944192354, 0.04765890337902064, 0.01595681268780145, -0.6462161113995958, 0.009656428950703816, 0.124357915129388, -0.2504161074313905, 0.030468143630985323, -0.09846500774489782, -0.3985328169437611, -0.3386368813036149, -0.08038496612864547, -0.044227523991770205, -0.008816660736991567, 0.00859568034418301, -0.31540594946558154, -0.27886598510012983, -0.3025119719009806, 0.2121456995126574, 0.21850810902468948],
                          [0.6447956330499757, 0.09715105782104946, -0.011609374772605133, -0.1120216374136983, -0.3701699043230495, -0.08369764077278005, 0.2051823258515576, 0.488917619050295, 0.059512592694019734, 0.05026305894615609, 0.12043432189170217, 0.35654906142776654, -0.09963856032702134, 0.15574416409822095, 0.19174071938571371, -0.6221045670357648, 0.3134020202555958, 0.24027481550699786, -0.16917966239115417, -0.32030137297874134, 0.03181380779422503, -0.1069666045602961, -0.4546620659269732, 0.21216407952242544, -0.13834524086675049, -0.30892285427424737, -0.5675162918897821, -0.3546508273928601, -0.8493191696513456, 0.42202919673845835, -0.07846122184956558, 0.0022087415764172552, 0.20897620680451956, 0.42942565250436465, -0.21145249219907408, -0.15992702943087356, -0.4866256343012095, -0.18570358656498323, 0.04894614080482184, -0.012019352188174535, 0.04660252767997484, -0.033164580087755566, 0.09047162893014762, 0.3188297285926045, 0.08222668645038622, 0.14957552203034127, -0.10100788437239974, 0.24121526880821145, 0.2517274568681591, 0.4783945964130829]])

    def __init__(self, num_variables, phenome_preprocessor=None, **kwargs):
        self.is_deterministic = True
        self.do_maximize = False
        self.num_variables = num_variables
        self.matrices = {2: self.matrix2D, 10: self.matrix10D, 30: self.matrix30D, 50: self.matrix50D}
        self.min_bounds = [-100.0] * num_variables
        self.max_bounds = [100.0] * num_variables
        bounds = (self.min_bounds, self.max_bounds)
        preprocessor = BoundConstraintsChecker(bounds, phenome_preprocessor)
        TestProblem.__init__(self, self.objective_function,
                             phenome_preprocessor=preprocessor,
                             **kwargs)


    def objective_function(self, phenome):
        phenome = np.array([phene - offset for offset, phene in zip(self.offsets, phenome)])
        phenome = np.dot(phenome, self.matrices[self.num_variables])
        assert len(phenome) == self.num_variables
        obj_value = generalized_schaffer6(phenome) + self.bias
        return obj_value


    def get_optimal_solutions(self, max_number=None):
        return [Individual(self.offsets[:self.num_variables])]

