
import numpy as np

from optproblems.base import Individual, BoundConstraintsChecker, TestProblem
from optproblems.continuous import rosenbrock, griewank, ackley, FletcherPowell
from optproblems.continuous import RastriginFunction, WeierstrassFunction


class F6(TestProblem):
    """Shifted Rosenbrock's function."""

    bias = 390.0

    offsets = [81.0232, -48.395, 19.2316, -2.5231, 70.4338, 47.1774, -7.8358,
               -86.6693, 57.8532, -9.9533, 20.7778, 52.5486, 75.9263, 42.8773,
               -58.272, -16.9728, 78.3845, 75.0427, -16.1513, 70.8569, -79.5795,
               -26.4837, 56.3699, -88.2249, -64.9996, -53.5022, -54.23, 18.6826,
               -41.0061, -54.2134, -87.2506, 44.4214, -9.8826, 77.7266, -6.121,
               -14.643, 62.3198, 4.5274, -53.5234, 30.9847, 60.8613, -86.4648,
               32.6298, -21.6934, 59.7232, 0.5063, 37.7048, -12.7993, -35.1688,
               -55.8623, -55.1823, 32.8001, -35.5024, 7.5012, -62.8428, 35.6217,
               -21.8928, 64.802, 63.6579, 16.8413, -0.6205, 71.9584, 57.8932,
               26.0838, 57.2353, 28.8409, -28.4452, -37.8493, -28.5851, 6.1342,
               40.8803, -34.3277, 60.9292, 12.253, -23.3255, 36.4931, 8.3828,
               -9.9215, 35.0221, 21.8358, 53.0677, 82.2318, 4.0662, 68.4255,
               -58.8678, 86.3544, -41.1394, -44.5807, 67.6335, 42.715, -65.4266,
               -87.8837, 70.9016, -54.1551, -36.2298, 29.0596, -38.8064,
               -5.5396, -78.3393, 87.9002]


    def __init__(self, num_variables, phenome_preprocessor=None, **kwargs):
        self.is_deterministic = True
        self.do_maximize = False
        self.num_variables = num_variables
        self.min_bounds = [-100.0] * num_variables
        self.max_bounds = [100.0] * num_variables
        bounds = (self.min_bounds, self.max_bounds)
        preprocessor = BoundConstraintsChecker(bounds, phenome_preprocessor)
        TestProblem.__init__(self, self.objective_function,
                             phenome_preprocessor=preprocessor,
                             **kwargs)


    def objective_function(self, phenome):
        phenome = [phene - offset + 1.0 for offset, phene in zip(self.offsets, phenome)]
        assert len(phenome) == self.num_variables
        obj_value = rosenbrock(phenome) + self.bias
        return obj_value


    def get_optimal_solutions(self, max_number=None):
        return [Individual(self.offsets[:self.num_variables])]



class F7(TestProblem):
    """Shifted rotated Griewank's function without bounds."""

    bias = -180.0

    offsets = [-276.2684, -11.911, -578.7884, -287.6486, -84.3858, -228.6753,
               -458.1516, -202.2145, -105.8642, -96.4898, -395.7468, -572.9498,
               -270.3641, -566.8543, -152.4204, -588.3819, -282.8892, -488.8865,
               -346.9817, -453.0447, -506.5857, -475.9987, -362.0492, -233.2367,
               -491.9864, -544.0898, -73.4456, -526.9011, -502.2561, -537.2353,
               -263.1487, -205.926, -465.8232, -9.7565, -16.914, -452.9397,
               -430.6107, -162.3137, -69.0204, -302.4082, -527.8932, -489.4274,
               -74.2562, -6.8285, -546.3631, -415.188, -322.1499, -145.3822,
               -527.0019, -422.1779, -254.2542, -495.7289, -286.1023, -196.1644,
               -104.9383, -388.449, -57.9322, -53.9958, -114.3882, -50.9434,
               -508.0961, -77.0278, -242.9642, -11.5865, -506.949, -222.1662,
               -303.8325, -359.4041, -156.0715, -252.9711, -345.7062, -133.0654,
               -30.1874, -306.6162, -160.6673, -210.5113, -186.2518, -550.3127,
               -519.0305, -320.776, -471.1067, -159.1724, -44.0751, -84.1251,
               -538.0389, -444.1993, -590.9338, -216.8007, -523.2731, -317.2,
               -393.6671, -548.4339, -416.0403, -380.3826, -109.542, -167.761,
               -345.5127, -544.8631, -357.8218, -598.2117]

    matrix2D = np.array([[1.3401073866086286, 1.2082401821285099e-1],
                         [1.9287305900841805, -2.7596402279496588]])

    matrix10D = np.array([[-0.0736966258253135, 1.5747490444892893, -0.06437794220716994, 0.6320184873093958, -1.2455211411481415, -0.3534118742809838, 0.3503169101851909, 0.628864797589927, 0.6859363235501234, 1.3975663076173925],
                          [0.6370001612307905, -1.3833836770823484, -0.24437874951092337, 1.6992995943357547, 0.7175744713750285, -0.07775380057027045, 0.49291080765053624, 0.1139284717810019, 0.48163647386641817, 0.28150613437207017],
                          [-1.4466181982194921, -1.1273816086105013, -1.066572484895932, 0.2190008893433219, -0.058130776006865136, -0.09918784192608603, -0.1246583157252458, -0.5054737280836883, -0.2102019141964088, 1.15099849872843],
                          [1.0410802679063424, 0.47577677793232626, 0.9643015456796787, 0.015636976117984064, 0.20539698111678034, 0.2583978003982166, -0.5171036180189703, -1.544901458983435, -1.4560361158442292, 0.9987790406073044],
                          [0.2626027294463596, 0.9294754074143687, -1.2953100028930926, 0.6651202964256139, -0.2795778170165599, 0.8406053769811276, -0.5292282960772916, -0.8604022007291047, 0.4950316276918325, -0.637653768929581],
                          [0.0813078894779547, 0.800623274265925, 0.07229461867918849, 0.44874698427975906, 0.17959858022699743, -1.363480069396921, 0.0752579439965767, -1.2486791053473751, 0.6814352640703267, 0.13558980136836016],
                          [0.0879135166538627, 0.21022739728349416, -0.15708234535904123, -0.3518219655045403, -0.06419016021376184, 1.5082748057903228, 1.1168462803089814, -0.367730422251357, 0.26828021681357744, 0.49698836189165707],
                          [-0.9540623537861275, 0.39879009060640763, 0.5802263024350377, 0.24831174649263604, 1.1781385394000925, 0.5313480974528408, 0.7825724045032703, -0.38166809840963106, -0.48082474351369503, -0.6207653363651408],
                          [0.27628599479864874, 0.36188284466692094, -1.0302756351623272, 0.7234864486780912, -0.37379075361566066, -0.079223639600377, 1.6221551897070494, -0.008288043678169736, -1.0881497169330046, -0.19204701133595675],
                          [-0.30035568486304853, -0.50758053487595, 0.3114345484062782, -0.2544490030739615, -0.7798852810230192, -0.6826283999943626, 0.5593266552193551, -0.7957905012142242, 0.47071685181799255, -0.8001926849489549]])

    matrix30D = np.array([[0.5251956125907735, 0.6296248117148668, -0.08160606132683026, 0.03576049930628756, -0.1553809151095137, 0.26008955030180714, -0.3817517462497128, 0.21907209306890707, 0.5049366645678821, 0.09395282774631639, -0.5679868982066928, 0.002631118205619362, 0.10712583383256184, -0.5641080362479242, -0.02086877363977522, -0.21376215799535148, 1.085753263316051, -0.08680871270488333, -0.39111385376560204, -0.18250247739704042, 0.06336234409970146, 0.026592938973198463, 0.2863405301962532, 0.19237895948477204, 0.048978143151076375, 0.6093511482033286, -0.22905064518848176, 0.2892090775588169, 0.1315788265531045, 0.03621036120804725],
                          [0.027197296268476768, 0.3236770289166518, 0.2673564112582096, 0.04490858064014018, -0.0880245095974015, 0.49123258896754796, -0.47376233220537367, 0.4872276971467088, 0.14197769748752107, 0.13910203365026141, 0.3526886776484709, -0.46586058664048363, -0.05666207653031261, -0.5720999577327093, 0.6026729252785665, 0.4064302843438666, 0.05328432741089904, 0.025918104459188657, 0.0582263897014344, -0.014864624187302571, -0.46895664227060835, -0.04079452826979109, -0.11341181189342649, -0.14813997915679597, -0.7283915422256599, -0.3586587295924857, 0.32472054085304086, 0.10670492326843771, 0.31823580636284665, 0.011459610657431932],
                          [-0.14283926063879585, -0.40697507884813094, -0.06296791964078344, 0.2855067197834561, 0.31648661178877774, 0.311487663126304, -0.27654956860625796, -1.2237448785371914, 0.2546045133306948, -0.13999525690124606, 0.05380306150937515, 0.37893910566199, -0.2399949100281412, -0.5462967745846887, 0.017506325549621105, -0.05243914629521984, -0.25287880152376124, 0.32110768760177216, -0.15055115975099517, -0.02675417714829102, 0.0961702569022476, 0.280470642988422, 0.28478479816634583, -0.1543526405465699, -0.7505105486761497, 0.8889779657967395, 0.18846345208432838, 0.0317734278667209, -0.3152303688792176, 0.391634897912328],
                          [-0.19230305653243795, 0.14412829188588933, 0.07619417177065936, -0.1735258103004624, -0.3728410683405971, -0.053093367817076687, -0.1397705027851209, 0.05641305057750666, -0.11263300539607736, -1.371097087246573, 0.7914080842341531, 0.14224298162107657, 0.28660234332283036, -0.6364182269034965, -0.17168309990813768, 0.521765472124468, -0.22936753909544647, 0.36141846228003754, 0.0627538701542238, -0.46508534830244314, 0.013055775943231198, -0.6446459739259691, 0.29729051219327046, -0.21303821930909408, 0.14106349740288346, -0.3228430275958782, -0.013172326455183443, 0.17639421427199017, -0.020193518969203666, -0.23697642187071966],
                          [0.017941156920697556, -0.0032613324153474795, -0.09394169873306508, 0.11138582919894903, 0.22267275228606281, -0.22487764497325685, -1.4220234187488923, -0.15779176485867222, -0.5935258145256564, 0.6750086913684697, 0.09045301187679394, 0.29018722319490176, -0.03162887306957824, -0.03336932732541356, 0.3866895113369479, 0.40423892540887746, 0.516351910191089, 0.18744417647614436, -0.9211491723069192, -0.0937791913117578, -0.12190006994453413, 0.27895027675192247, -0.19191409061439396, 0.1874423066223706, -0.11894635990235042, -0.22361731431964088, -0.4767800989241227, -0.4877780386976937, -0.6446228290582775, -0.17834097178379926],
                          [0.5241809392674807, 0.030104693742323985, -0.4487326935770162, 0.04052747876674157, -0.2269121128941838, 0.2063553399950617, 0.3375736925571363, -0.3136295062851913, -0.2381786368368246, -0.26992406601823105, 0.024883493432015787, -0.1820422531185993, -0.14787231062767067, -0.14467488789128286, 0.7379856995277947, -0.3228021904941481, -0.07131674250992479, 0.5760285021060951, -0.4461450842771441, 0.29144817483112656, 0.27372504229943323, -0.3912637911086641, -0.10936764493327546, 0.23627053879857174, -0.0047688691507444415, -0.14908304885126428, -0.13942634074538363, -0.40742945170888095, -0.04016305926434399, 0.2770992063121178],
                          [-0.1796315543498434, -0.3088259643977865, 0.16679046990687899, 0.676797215229488, -0.018504702022676345, 0.3568901053467492, -0.301103213619431, -0.11203540763789063, 0.46762898807555625, 0.3583278615749358, 0.24746066931208252, 0.26161381404301665, 0.4196098501352797, -0.8054289580967843, 0.340098493985431, -0.3538830052249477, 0.09850168325751024, -0.12856460639098805, -0.238443988061706, 0.06674413315483059, 0.0024370893511487437, -0.014624272930326206, -0.48064300877141364, 0.3281587261778145, 0.6740142384725395, -0.22289988626631865, 0.788199459862219, -0.3592567513698649, 0.3050968654002365, 0.25165714367765074],
                          [0.09017013784265451, -0.8073109999262168, -0.4365423396918477, 0.5984269321316447, -0.3586460110002811, -0.24710040484269682, 0.27569166091336883, 0.25760155278258223, 0.042680494023697044, -0.2629769458735662, 0.11471795844732986, -0.10392750185280501, 0.20677423941458503, -0.07210854161974037, 0.132044115878351, -0.4591713295737204, 0.14669987662114944, 0.03764141140554433, -0.3751300930960975, 0.4771299522479577, -0.09275930437942279, 0.06292591063541508, 0.3774518737867685, -1.0524395533410724, -0.048504602512376184, 0.7971027738260442, -0.7710853944447905, 0.11665560980862894, 0.07665521220783406, 0.03415172583637216],
                          [0.08481666108717764, 0.46235575219599717, -0.2554204500148909, 0.2968492070400785, 0.6062353968056093, -0.06288103304333757, -0.3498241731087342, 1.310754720345192, 0.4404081678951473, -0.22553199823789682, -0.012023184035191885, -0.10277496886755634, 0.7073512297543338, 0.517995489698905, -0.47671762987877364, -0.5953885908893226, 0.2647045600959649, 0.15084743008189475, 0.2259857512634269, 0.01866193905803066, -0.013051438350340533, -0.00364924285578231, 0.1484433903970785, -0.15948137169610951, -0.029584625828204394, -0.01740202444250005, 0.7359850272711348, 0.050960442077867665, -0.3396572199706991, 0.08024814215779577],
                          [-0.08703123128238717, -0.10182327665839208, 0.05466654682530931, -0.33065207081640063, -0.05491894040507877, -0.20479308033662338, 0.055533135117925, -0.02454725881948226, 0.12836466398224441, 0.01586373962728287, 0.13852561709358163, 0.22125720882632596, -0.5320446490831986, -0.2706316408254332, -0.08329046992078805, -0.356926025588531, 0.09244654815357126, 0.030424519200944772, 0.12246842590966991, 0.32439545553902455, 0.0824425275067621, -0.7170473436454663, -0.6163496360524862, -0.6074038300236714, 0.15683576642407607, 0.004147799561720685, -0.036427157973224746, 0.9397390072144407, -0.10206438193992133, 0.12421845076027337],
                          [-0.19963377842023827, 0.09759470942597367, -0.7253774879313741, 0.2237484821228248, 0.05746747772940647, 0.3839780343624228, 0.6927019583379127, -0.45083891971688544, -0.29816461409904826, 0.20598181571831803, -0.6519368752866852, -0.0215909662301006, 0.2871145104402745, -0.22178249626481297, 0.3757960766047793, 0.13292198860936325, 0.43053757946168203, -0.7366880956589341, 0.5508167821362803, -0.1822455639728697, 0.5651905394544006, -0.008219891690000482, -0.2045578193547179, -0.7389761191619963, -0.22786647844105837, -0.6992413851789548, -0.5419644068892996, -0.5032195101873232, -0.03553834259751016, -0.1055147411172392],
                          [-0.20893926557003645, 0.3163613681239083, -0.09946609727345554, -0.2421841400286103, 0.3536296128089201, 0.00013058167800751792, -0.21240729902817237, 0.20667632229657842, -0.01733790919429791, 0.019089642335327804, -0.03690553184108504, -0.42743589890699607, -0.5665140481648028, 0.4701636432644331, 0.03265240315687935, 0.5196675684858377, 0.7619684355576732, 1.061948414608741, 0.3582700099700933, -0.1517202249461459, 0.9405493836112241, -0.1220093285575887, 0.8068319465947688, -0.34984163644832483, 0.15674611272462183, 0.22757829405084617, -0.2646025116539486, 0.30937502450864124, 0.4011328835535284, 0.48569499692733825],
                          [0.04882908154654262, 0.07617315349068944, -0.14761081845068474, 0.4438901544071871, 0.42244517870184206, 0.0725102736139313, 0.014684225471323432, 0.7686363193400053, 0.033390537951981804, -1.0510738883424307, -0.36363312671337084, 0.6658594300598614, -0.8423657315506895, -0.09764248838073249, 0.04854737303310692, 0.3583514670106889, -0.5959573031193156, 0.25625748610364163, -0.3294838306312295, -0.018967240790369676, -0.34256961159765487, 0.39400672042039264, 0.14341508539865483, -0.7481023787309903, 0.5652753260336568, -0.15227199107324685, 0.12292735377135154, -0.3610813730577095, -0.4743226182630074, -0.009737474140454045],
                          [-0.05335775509247778, 0.008827148671438675, 0.21030427826190828, -0.4777249708539189, -0.5352961266500237, 0.06727573686421368, 0.20177662334784804, 1.075247668850319, 0.3384522149121312, -0.39927766332883585, -0.5957970154833612, -0.14535988587691653, 0.41369825233198726, 0.08257940382783094, 0.42731944332457267, 0.30184617077225445, -0.10166539987082789, 0.6584574181898547, -0.18299140513336967, 0.05363158760405134, 0.5524878837980216, -0.08495859196848445, 0.29577858255352657, -0.02830078261197758, -0.15482186962457775, 0.6609440762704086, 0.08713346977742532, 0.14196405262271278, -0.5104835365628128, -1.0556064992354912],
                          [0.055474357068774696, -0.2175593879293049, 0.491313018636015, -0.20858129798407596, 0.19030794698042097, -0.27265814228570556, -0.0899259059016074, 0.008254514421487934, 0.22817487587074314, -0.5321020598762006, 0.07256933829688463, 0.24912444351799437, -0.1130286362066933, 0.2668234613908604, -0.1693900667484152, 0.05272898742725138, -0.10999418523245638, 0.1817616992423075, -0.3533820645739874, 0.08782930473168324, 0.30277587147671964, 0.5140482296347775, -0.31511395034124356, -0.17003306715357008, -0.33984380475288284, -0.0418572205753124, -0.0840286337896464, -1.149052347563724, 0.3652433940148505, -0.5172605807355659],
                          [0.19089315156505332, -0.3515748393505331, 0.23433987108224003, -0.055036159031292074, 0.11228386172919293, 0.17928383935856776, -0.3110614614692487, -0.7745118811943349, 0.18393786738635443, 0.10942716885446327, -0.2515559356546049, -0.23792434311914115, 0.038173157847961776, 0.23411381660414032, 0.13773552925947255, 0.08258914601449341, -0.15237297029250385, 0.42640468897547906, 0.4822724586449902, 0.11404166941428694, -0.5452648474860727, 0.2929168028156773, 0.44637835262499403, 0.30185539052419164, 0.02729973957172667, -0.6005543170715507, -0.9084720332930788, 0.41684134740964707, -0.2889616893789568, 0.18472780389261786],
                          [-0.443659317804456, 0.20161997706402612, 0.810352086315364, 0.13999675637604053, -0.09076733299918537, -0.024145092836682057, -0.4262901090412367, -0.18124238397636042, -0.0010927829116524064, -0.5766929622365579, -0.3962120376311297, -0.022553539088515966, -0.09532489557636975, 0.2268995045765963, 0.03580828794179378, -0.5057916203447834, 0.0002901929300335606, 0.14693801984442958, 0.11252773442183153, 0.4400567697598135, -0.8692838858950027, -0.043109398038224486, -0.09579911611195302, -1.0836855674880084, -0.09085837605787367, -0.24696263697799142, 0.07044504026949695, -0.4393350174497034, 0.027517518972710996, -0.46099955282192784],
                          [0.03901669712899452, 0.5681818582732804, 0.18618175597017259, 0.4794360604177564, 0.03308728334416186, 0.11737822896360332, 0.04700328967037003, 0.2819817441611866, -0.6083504385589359, 0.4457684776834928, -0.050984788598342545, 0.21432351006331557, -0.10784378269401643, -0.005804743619719485, -0.7949634901905451, 0.04290120798746404, 0.04413551712078642, 0.7157251571814833, 0.1416111418079355, 0.33573863068689414, 0.0893792908417187, -0.35581894390084906, -0.12566700503283249, -0.18513340945886422, -0.19652048268055347, -0.23880366715697232, -0.21086550609957921, 0.567269403650762, -0.026717566551968223, -0.3124184080762039],
                          [0.2678154138097963, -0.34112256052479367, 0.2325212427265422, 0.15844493977223365, -0.6443344964268435, 0.034948104428968764, 0.3168584315577158, 0.6361513756681318, -0.43118155782809375, 0.5058144552370181, -0.023805332218941794, 0.5722788658584614, -0.33677525969988037, -0.011958602405631676, 0.6979946172534979, 0.09424292604166938, 0.3741556204762604, 0.0812971020594356, 0.36809524768471985, -0.37822814889765527, -0.14180993618470203, -0.03467363752483, 0.2976276016491256, -0.053317672022776715, -0.23399180423087013, -0.17830847107435255, -0.037431131385841115, -0.16556211424110664, 0.21672322583982717, 0.3817377909156938],
                          [-0.5801147369884342, 0.2587088293197146, 0.08273373836020612, 0.2516432670212754, -0.10879187325032068, -0.450765922923288, 0.3453985928644022, 0.043012684211313705, -0.23408200093718884, -0.0802515173766273, -0.6558286684326794, 0.007670310376122003, 0.8386006392285189, -0.25376100378943967, 0.15527457737903413, 0.22158367738035356, 0.229869231994624, -0.09714392332911286, 0.049006457836016114, 0.18696642019098078, -0.6072646931787915, -0.28567358369746837, -0.5088812410185157, 0.1718087163437044, -0.46119218791441374, 0.1765770903968061, -0.5882232330623944, -0.23286550667880862, -0.3394578527958249, 0.5727874398475595],
                          [-0.22540793402711115, -0.06755359977714916, 0.33876832577160637, 0.6791509184377688, 0.2775346640119923, -0.1696747935394242, 0.2544620165300436, -0.2508865810497295, -0.017941377104392423, -1.2393201589385745, -0.4617760773527162, -0.2840540221038478, -0.25423169925847056, -0.09099793397662476, 0.7746308130496928, -0.013204070466857364, 0.24102227106093604, -0.7495484468179708, 0.17833236495344146, -0.020287503377335096, 0.4920794792333479, 0.02224793023825241, 0.007621685097736311, 0.5442641972624087, -0.17563066341779757, -0.584042195481135, -0.05791419779062022, 0.18969436631833472, 0.06900938138803524, -0.1608672071777576],
                          [0.24454616550640165, -0.3603871977814697, -0.3347519327476256, -0.048546056667370927, 0.21962829161232106, -0.08302076189815602, -0.46659496465189526, 0.38964634647453283, -0.4825698368684642, -0.4017965422323238, -0.4957713116877271, -0.02801738359802306, -0.2252444744073479, -0.11298989621975267, -0.2828587802788923, -0.39415652829519715, -0.4786892371592048, 0.18227794740612688, 0.22418836468775957, -0.369586641044146, 0.07692342239686746, 0.503629925428362, -0.9086942265148878, -0.1668864973563786, -0.0636358736862869, 0.365179951842247, -0.7587599483128067, 0.30145209924222166, 0.5055587724640189, -0.1939932620531713],
                          [-0.7793575853267664, 0.15860956365391224, -0.5055383818572619, 0.2413527235561757, -0.10941797990557137, 0.617687649159074, -0.15908559350750945, 0.027986467033818678, -0.17237273393170366, -0.09657659488631037, 0.5718379054279766, 0.17145589966877633, -0.3333234071494628, 0.06662918915242626, 0.05730679503895292, 0.214524624101122, -0.280107409653574, -0.2419132307760144, 0.3754431057649062, -0.11162878866297524, -0.2956444793778426, 0.11499169541906058, 0.03411001256969341, 0.21288727749536326, -0.2604206337828247, 0.3174429890874937, -0.6020402286019954, 0.1380972699793407, -0.20708907898121062, -0.22319403115599873],
                          [0.013544895722588504, 0.38990467589143474, -0.09922837603565184, -0.08460693650167266, -0.7253303157375289, 0.2509908514915258, -0.2321162244872553, -0.2885289278431979, -0.43731065979480066, -0.50169445880165, 0.15400604862456319, 0.1335122369063184, 0.4101173817185761, -0.04768197794918107, -0.3707672276134215, 0.10051823135374476, 0.207850018281615, -0.23209860178099315, 0.02182364955165532, 1.103226543145372, 0.38907323874142796, 0.2300062065235771, -0.39568074870871445, -0.035327255347893875, -0.13016040982050547, 0.02265190739421924, 0.15965751762220737, 0.5077492837970494, 0.1605329720700767, 0.2324611053813426],
                          [0.3332634666187155, 0.10850832158239503, 0.033109729840135646, 0.4766110201439036, 0.17367758763344615, -1.1594173563425136, -0.08617781256170608, -0.3572096185876832, 0.5398765358227895, 0.2395033052034622, -0.14227289711360322, 0.09541418671673987, 0.2219249480952825, -0.01729510776220258, 0.7594007249168333, 0.22018323608165088, 0.29781876974398264, 0.07687181491615469, 0.6127790354872965, 0.27989038789886195, -0.16625488269305133, 0.4111816498252515, 0.21609348696113853, 0.23317676941428825, 0.40244045407168405, 0.3706134828646342, -0.14051205940218423, 0.2072009066932428, 0.1589753231665995, -0.21484113318248707],
                          [0.08034989150019184, 0.3650717615600979, -0.23016886275056048, 0.31623497884122836, -1.1875160076152353, -0.9131206765501281, 0.3518992820112484, -0.048899431950184626, 0.22277174429508992, 0.06063956292328349, -0.2439465042883474, -0.12827771886081601, 0.10168334728227843, 0.04612259921139138, -0.5495672987703597, -0.057541844653563025, 0.24522085984280917, 0.4541572527005129, -0.3516806666282396, -0.00010796984274126797, 0.651375904544969, 0.589916441439836, -0.06672246577358562, -0.18224949082926373, 0.12897605496341666, -0.3838531130539134, -0.13499424049674427, -0.07936105990645868, -0.8848799250849371, 0.5233267562621462],
                          [-0.35724817146191507, 0.40695286082904986, -0.3204541115341793, -0.2766956107522639, -0.24204270975494563, -0.7649465200981684, -0.050034052953182036, 0.2470871839939605, -0.028640593839252486, 0.1984592493281509, 0.08354570799198618, 0.13445055377377782, 0.0729484731480335, -0.39318407362561464, 0.22815167109001178, -0.3623496741357807, -0.562722394118617, -0.17767738544618106, -0.34060539319279787, 0.3794645286617106, 0.46281014029219136, 0.09639711119533802, 0.5294717088372223, 0.2594933741076368, -0.03908377537790259, -0.24919808723230227, -0.43544162318520363, 0.21720689370639346, 0.29156816248001793, 0.278415197982219],
                          [0.23800147045098194, -0.23800978666381364, 0.4836215425000708, 0.007081465145223219, 0.2315396888962164, -0.20798543870793243, -1.255522560781906, 0.338567368180303, 0.022975744746418024, 0.23884978465158732, 0.16228958069295288, -0.7870323586904364, -0.2705350533376601, -0.2590803004251703, -0.4615189592917698, 0.1562991672090882, 0.05964969907678848, -0.0024157167244675377, 0.2675238349410481, 0.3423763502853791, 0.32858683338745404, -0.5754947794282538, 0.5852589976775637, 0.08213968788037487, 0.249009470866771, 0.24300044494268533, 0.26289797069890863, -0.39645477447068467, -0.41291979677661855, 0.28439373819468894],
                          [-0.10845807355856903, -0.0795765853171458, 0.4496908727322971, -0.6892826976093493, 0.2584474740863176, 0.3671446035178645, 0.2511874276544932, 0.14216128987646925, -0.14789133480526645, -0.2297615159007828, 0.31476217644132587, -0.21687803131052097, -0.06472015875408321, -0.16262540403474884, 0.30468341208199023, -0.6979495385209046, 0.07722720472422526, 0.46592056813958793, -0.141067475385309, -0.10744787372995435, -0.051380310356971, 0.5493708461973117, 0.024672344704677288, -0.23164502167731096, 0.08449024456735708, -0.12426487299083105, 0.2146558082648994, -0.35652633111324666, 0.000300786526311591, -0.4349791745712771],
                          [0.02598111150273699, 0.3139990289974612, 0.9647048740679679, -0.2877943927102681, -0.3155208258945394, -0.11263330906743362, -0.14041001139620138, -0.22897479410224464, -0.5196786607506801, 0.16135823650573577, 0.2518553018859645, -0.17335088497743176, 0.1440182772131403, 0.02560028913101942, 0.5560824759988701, 0.007991501858669755, -0.4610280064229542, -0.24902158398041724, -0.05700926791737188, -0.035233910206498645, 0.09726571052612544, -0.2616110145562515, -0.14908422355415274, -0.23905281038540271, -0.29305586801624206, 0.46858863763195496, -0.23308970639748922, 0.10269595907675749, 0.044926224514281465, 0.5411420486006493]])

    matrix50D = np.array([[0.03805238582592365, -0.43696522446755887, 0.2876283757701406, -0.14869274056332188, 0.03527072442321661, 0.20153437554176168, 0.1546049578976795, -0.041434493050330506, 0.0855799281830358, 0.1778174650253942, 0.24749131169371122, -0.15351362489354337, -0.13322755622984614, 0.22517618235558157, 0.1384720750224243, -0.011211783075269807, -0.31007910166543234, -0.09288599954347347, 0.10178608522296399, 0.26456084567950183, -0.10827983979737434, 0.39885859430242515, 0.1489584610451438, 0.21894599254620026, -0.2335218432347528, 0.2572025445745421, 0.030456519403245782, -0.1139996562066478, -0.4537397346600716, 0.05320516610255535, 0.3559036954966776, 0.11815299135043948, -0.28939429939818784, 0.17168079937684816, -0.5064506499811106, 0.13453602446365875, 0.25866005870451725, -0.012286209738872431, -0.2044864288325436, 0.2895927371195766, -0.06268866563016105, -0.015143485385491573, 0.1149776707951348, -0.036922330767178206, -0.8009220927624823, -0.3168493590433961, 0.18099064353703162, -0.05856580514982062, 0.32973285539775815, 0.7260187523631976],
                          [0.10483118491397679, -0.7016161325108624, 0.5788856811675729, 0.11273066675798235, 0.3714587333766673, 0.12166793305665767, -0.0994832278881795, -0.028661130532041865, -0.18841400963133142, -0.27375724851867456, 0.06449338405954462, 0.2239560963074523, -0.34755797741734, 0.10403583261102882, -0.13903753734337293, -0.03406589714378792, 0.0789963832743978, 0.5698649459909362, -0.11158798578254522, 0.6585035870075159, -0.16872502137168863, -0.4174399534449043, -0.31027283942258943, -0.08170289868979794, 0.25643196252442274, -0.0028582940991851062, -0.4617188022136089, 0.1686847420519254, -0.2481681275545601, 0.22054796697411588, 0.36590893217132875, -0.24993900613595893, -0.6117326932313103, 0.5439120961358611, 0.01474822914878918, 0.3662450527031218, -0.026500637815512273, 0.17639275602164597, -0.3421477371151082, -0.001011591404090333, 0.041799457106334585, -0.02307029676241453, 0.07203716809530952, 0.007759897182725116, 0.13059758256829576, 0.05624980387645762, -0.24194422712113342, -0.4671400652650947, -0.07218596975983896, -0.21004569294290537],
                          [-0.18455080271112254, -0.11104936450110339, -0.10301386204654245, 0.1781044584131509, 0.2548130232869315, 0.11598484453476976, 0.39069653520549996, -0.07260123959250551, 0.20660271887208112, -0.30075594794992366, 0.67133425324697, 0.12775639641107542, -0.3498975514232699, 0.2141441963395171, -0.01929215628455121, -0.10330742256790355, -0.11785953324489111, 0.1594017937106509, -0.3897754961924303, 0.31934205237817587, -0.12246524315088281, 0.019033085998140156, 0.09457216134160527, 0.04826853259855388, -0.10970094379956734, -0.3527536354529695, 0.7292999174425409, 0.6169046191164174, 0.11595450357233406, 0.0940829260413159, -0.3007902010107965, -0.1885074673587857, -0.07379842429457081, -0.05581408808828675, 0.12590147108279018, -0.3631218516518986, 0.3837938615199975, 0.20314614552826238, 0.1109872978642691, 0.272471276615514, 0.7527658091919687, 0.36391661874827314, 0.05024336614475427, -0.35614053327549583, -0.5628721492853384, -0.05653245789090275, -0.14927071174807938, 0.13646650127722346, -0.03808706092394868, 0.2874170662307121],
                          [-0.06159988423185619, 0.0497166950007648, 0.03507107056183669, 0.6177999040912435, -0.20385820090850573, -0.08180545967404192, 0.040387936762386834, 0.06355032426569573, 0.44146421474422587, -0.1960889534410645, 0.16727996399203565, -0.3084195093552953, 0.6477097757079836, -0.20508436438159533, -0.7899025590316698, 0.12405504233503391, -0.22211220052248784, 0.3050367112825268, -0.09918965201079825, 0.014299725023131862, -0.4715944792150509, -0.8014749839593728, -0.1018937322429752, -0.06939205429417955, -0.15533872204187027, -0.11397460323414813, -0.13799898415652123, -0.0718975297131855, -0.32097036708663945, 0.2359024842169943, -0.07837440840187636, -0.20333036634115031, 0.09753810417573859, 0.16865391451309683, 0.09656015833355455, 0.3900302434675154, -0.05500232376956077, 0.282377227862466, 0.1639262928001534, -0.5963224365156607, 0.16481858374670166, -0.098701408731816, 0.3349254600886834, -0.2982009097353532, 0.4701625783707275, 0.014565500548764715, -0.12015321711742508, 0.578013295728248, 0.08633011301924662, 0.03823536556685021],
                          [0.34851833485177225, 0.10635653285319673, 0.4614431262631558, 0.1135508891524141, -0.16277886332029823, -0.37352011922297573, -0.3046494370383343, -0.28403376674707115, -0.45982332115861235, -0.12000929161650169, 0.06267713164451899, 0.1568288220151769, 0.3242033603786481, -0.19726587369996004, 0.015544682426063243, 0.2280827593098117, -0.14493213910654476, -0.1102535862724764, 0.04601882378284166, -0.28940305217462897, 0.4264565370512193, -0.08768000969555205, 0.2648439061906195, 0.8191823033006288, -0.6457186964688421, -0.22878764164171153, 0.0049936526806072315, 0.10949617895135975, 0.35320675250981326, -0.2087673532921263, 0.2049281093963706, -0.007739265922655639, -0.5901771680441557, -0.031785724789853316, 0.0862352489425974, 0.29539392120830876, 0.08016090911031189, 0.24128680950641768, -0.23151836876805956, 0.27364556704637255, -0.26490260092394236, -0.07076089872222335, 0.3627779402112506, -0.6661567584652878, -0.2111228311750712, -0.5212039321489559, 0.07956930002069701, -0.1278592211007395, 0.0067887318840787955, 0.25969519065039526],
                          [0.5019721268860146, -0.19839718079069063, -0.13729838153194918, 0.26271391194317667, -0.12347431887536726, 0.21989127711987225, 0.16514419425584068, 0.1472309459567653, 0.28522342748182733, -0.41580904082561726, 0.05543943566619926, 0.2818072084726279, -0.24648737391529343, -0.3526516178923162, -0.08692816419816018, 0.04272233194769391, 0.5231405525812528, -0.17928981414072928, 0.021102063666770594, -0.01155095893993045, 0.04015043560599363, -0.6953882230613104, 0.01579628401043725, 0.07497558528925549, -0.10461169278457914, -0.2385604939220793, -0.3349339577018021, -0.005734698541364076, -0.2703750489956148, -0.22382084558133405, 0.2810021377669517, 0.40534568320982567, 0.5022394111139843, 0.07654234430903842, -0.08770559788731522, 0.13248108099156947, -0.025505260882648942, -0.2685851279314038, -0.43188238398883094, 0.28798067577245806, -0.15747599632904316, -0.03377274532043483, -0.5435730133138172, -0.1060192625434286, -0.1281496563296399, -0.6212791497013652, -0.018632105753595595, -0.04169933065818392, -0.22830112729741103, 0.4650121443258204],
                          [0.13328796434400506, 0.2941160289117185, 0.4896529598689584, -0.17351866407237695, 0.5827296833257647, -0.2420845991190483, -0.18980847476945292, 0.5841631010674185, 0.1286166879784828, -0.17424420799462295, -0.3271775938268777, 0.02041935217455854, 0.1336370575967614, -0.39913476127604813, -0.35028175923448884, -0.15533783991027497, 0.08987353219235794, 0.21189642576306647, 0.22477690248024754, 0.19299896782239165, -0.4901746243724058, 0.1322831804038558, 0.41508587664797886, -0.43273244730158195, -0.37960573688121285, 0.09470551509369678, 0.2021325814939548, 0.06820474301946604, 0.084626943976536, -0.08772549656964053, -0.13168864646855163, 0.2932983146447695, 0.023998513952184837, -0.12880808543214514, 0.41316409837978163, 0.3266752523874673, 0.19089327714203905, -0.35468141673399306, 0.10651357242925996, -0.009533192600780626, -0.026589096543573502, -0.17459328526095286, 0.09673595549476781, 0.006920032976895486, -0.31305334100734483, -0.006339646267278612, 0.4176845938464934, -0.2969913702205492, -0.2943928105313143, 0.20667692918759736],
                          [0.3344165397934471, -0.015604020615921501, -0.07773531782783215, -0.2667151186501513, -0.15133314482388469, -0.244389356828077, -0.6894640832649944, -0.10022150777325782, 0.004180524888508947, 0.21004043903715539, -0.05272263667116706, -0.10992283459278526, 0.1127665172468062, 0.02698843420953793, 0.0030537867827769086, -0.027430026223700613, -0.025598332112073796, 0.27944560193585594, -0.21421401226538503, -0.33428867566773623, -0.4526699269300205, -0.457583379266466, 0.12607752745435982, 0.07242629112594327, 0.44386585296435355, 0.14104485222592317, 0.24085176847332812, -0.053263335472907736, 0.06528898625588352, 0.1533966095671161, -0.11658321794192475, 0.0667345140341391, -0.01630815847855932, 0.31843649493607357, -0.1141857473290002, -0.30520144020465234, 0.18017019674154294, -0.15905419707166366, -0.22310450008157762, 0.5455963998004559, -0.0182027149799573, 0.43096282435973204, 0.5210299067065922, 0.08790434561247462, -0.06671675371348298, -0.20992651905318366, 0.4958440845470096, -0.06571967430024181, -0.4884932581698237, 0.040976269671047796],
                          [-0.1553063205400706, 0.0036121906621980194, 0.9462714246752105, 0.17132013218030398, -0.5925404111458936, -0.3232362489180799, 0.4537230545312134, 0.32329071848984825, -0.21268860934896286, 0.3550041595739707, 0.06419022372768227, -0.12781906587323855, 0.23116271798559562, 0.019921633140118196, -0.11404264785519985, 0.05864148434385411, 0.04759517583124024, 0.882510889831274, 0.16093290672007765, 0.013922986171365314, -0.41569780604010986, 0.27763693304632264, -0.0005086217572310439, 0.0027138853077701875, 0.1835605094772778, -0.0590022240962141, 0.054165404009816315, 0.1102533301068517, 0.135785146578327, -0.4022983991235872, 0.379258947791967, 0.2615705603234854, 0.05394038098861114, 0.11719953080774619, -0.015049346662356104, -0.6573357800323547, -0.2519146983442405, 0.06419737257643592, -0.13386832164906312, 0.39458436867923363, -0.23148439760292205, 0.18484470067538986, -0.04389391386626605, 0.02421115750276368, 0.4031764535973427, -0.40505904800951315, -0.4840763663047157, 0.5187021197613539, 0.0629060411295903, -0.46379561316408535],
                          [-0.14511601633933296, -0.1897377212252728, 0.5512103373359587, 0.16638989665020273, 0.04980223462456009, -0.062224717225179506, 0.048431254260380065, 0.14038854872917392, 0.37780124207096666, -0.1733396642073849, 0.019410466717521085, 0.18486840035643978, 0.19088036818381993, -0.17485563994181974, 0.3910949008033944, 0.29837425210193336, 0.12196484894182112, -0.11991397085129195, 0.9832805790416969, -0.4220306791895844, 0.20653643591828116, -0.06265900978034303, -0.3365360242176527, 0.037660827072197325, 0.02883559952751216, -0.0021118827176742558, 0.8925989024508734, -0.1051389857122281, -0.38323174122500403, -0.00988093184794084, 1.3533343320373947, -0.36101236484629184, 0.04106165178107583, 0.20121870314435766, -0.020734468679918266, -0.1224926746459504, 0.3210886739533343, -0.054309763612803874, -0.07115511148664301, 0.11307488827012327, -0.537677182645803, -0.26864346025404146, 0.5086806841769054, -0.23219121852966015, -0.0810631411898914, 0.016872359352529842, 0.1914142708970802, 0.338409505485992, -0.24101018562853968, -0.04480055092711441],
                          [-0.4162562892170454, -0.0683775616576616, -0.07390696352887793, -0.2521153637406893, 0.2507325978865825, -0.23495853545990383, 0.13621501531962169, 0.18988066394911188, -0.050916374773613236, -0.23548154665094218, 0.030894748067212888, 0.3024402791326987, -0.0449237070359937, 0.2622855004241555, -0.13817204666466626, 0.11280839285153138, -0.3175440170956745, 0.052831511252225474, 0.26538269907346873, -0.11235604664294632, -0.05522675293505036, -0.07843522036598738, 0.04528138466262343, 0.38393444981582014, -0.2831104894974813, 0.02273295207287512, -0.508287891368306, -0.433555357803186, -0.3164643622273821, 0.21120374345474802, -0.27441539054318387, -0.5552077649470106, 0.5977187100543251, 0.08534383123999499, 0.21642852020546444, -0.07275991485385236, 0.3588587337566475, 0.2641108613216125, 0.0848302094878881, 0.6744417138561466, -0.32417328776321985, -0.19637316609495487, -0.6580530480559873, -0.11853774683840956, -0.26473820898857936, -0.31526497216282134, 0.3352423010324977, 0.06728507211196792, -0.12834329543414355, -0.3473178084861174],
                          [0.10189067479417527, 0.5554194677921841, -0.15952755713513378, -0.18599373207389383, 0.20943637306507887, 0.11713573534595, 0.16779708855167508, 0.06743707279229626, -0.45933665248627775, -0.09360670831847481, -0.22086106883329165, -0.727464642947015, -0.049200859708198624, -0.10651633739744179, -0.6516375149923693, 0.1792740380524273, -0.18224039720700058, -0.1763319988633798, 0.29586388042288314, -0.11144085988665567, 0.049331778000612475, 0.20999342997691287, 0.12177485885569801, -0.06202233185379811, 0.32761812017263187, 0.3234116893359278, -0.2916260482391238, 0.1779992361320372, 0.14169369282108202, 0.02600808869010134, 0.2810921792060069, 0.3111352037188155, -0.2413954372245814, 0.18603838696197555, -0.02582765267304575, -0.04138649426443882, -0.15331718217743892, -0.09024651897663397, 0.356651044982822, -0.087354138678369, 0.44040818146336264, 0.4601177866982192, -0.21896590183505005, -0.2592660457991389, -0.04124225040691659, -0.12830094172619963, -0.19568383031196718, 0.24235568282216396, -0.5992444405983532, 0.08252408035366376],
                          [-0.22678256846316314, -0.16018150099884623, -0.14774897625714267, 0.3442396006139959, 0.01815357335244738, 0.1150556250596121, -0.2839469993227681, 0.038251474110170804, -0.4141111510537462, 0.15590356669351305, -0.29331181694444436, 0.034054882506305825, -0.2246658074761541, 0.0071338232656331015, 0.4672461103820459, 0.4631028243921213, 0.13420387935381203, -0.3925974787761873, 0.20424976954908852, -0.057636395854129906, -0.7938821416256131, 0.07526398956646871, -0.32891493728585025, 0.05530965330825435, 0.41437953359515645, -0.43383860583458067, -0.3515089450056282, 0.11225012559608138, -0.07450938455482398, 0.08504774975302626, -0.12465511089634619, -0.020876459940285495, -0.15052735847848447, -0.3378247668047277, 0.23357158715670928, -0.23770484680017434, 0.4097442979690514, -0.2551491436039209, -0.02059951105635029, -0.2930786195307674, 0.1312895278296448, -0.15957628650472394, 0.7097973004769165, -0.023776790351721984, 0.04859241740632748, -0.15842805826812184, 0.2534750329985254, 0.25440133660899983, -0.11293732556232516, -0.07931704315692391],
                          [0.24345414985629368, -0.24190142879133986, -0.006312642946034497, -0.1627484156026207, -0.09158234807798289, -0.08055739869333851, 0.1320503784603428, -0.18378541316057884, 0.044895601782560954, -0.05544125436770456, 0.10662170139723046, -0.5180805012577647, 0.09192419696536153, 0.07421519225472031, -0.2588855219179257, -0.08189277834702187, 0.052283257935579724, -0.1315326505107462, 0.03377714815366334, -0.1989427924532881, -0.11950177093527897, 0.4182544038690565, -0.511732998825023, 0.42312696735674993, -0.06489202195947776, 0.29175752273381095, 0.09978352429374325, 0.3583224729416667, 0.424763908519062, 0.5144434322339011, 0.02957330395837553, -0.22209145735049995, 0.10819961890907752, 0.09708240670174297, 0.6803327782576237, 0.04265700606987002, 0.0680327540107269, -0.17263493480020356, -0.4616340968676106, -0.3053150579450835, 0.2705144870512716, 0.10212274861445078, 0.31580852935495646, -0.19314054308941592, 0.06109374643979484, -0.17650526346379863, 0.6623144637594208, -0.3185695906046357, 0.42426683280892435, -0.11747045032933155],
                          [-0.2998047383609179, -0.10087852590167024, 0.38110986607680064, 0.24822091865824225, 0.34857304276853324, 0.309202725887305, 0.10589285434902958, -0.16580792712654574, 0.8542337809580152, 0.20258582031400996, 0.006188962619603545, -0.022859027903744777, -0.10175476925458467, 0.4516869773952825, 0.03513336383777924, 0.2444822454547213, 0.5370317930486733, 0.6283199792611716, 0.2935413008703123, -0.6338870440966571, 0.4673801157248158, -0.14699981996097033, 0.35501028084127473, 0.09218387177623816, 0.35705588152262774, -0.03901657298628439, -0.9172536986296165, -0.025420516565827486, 0.42738251586279136, -0.43380045836658293, -0.0536534350307672, 0.23758084003571894, 0.2429806467048634, -0.025587815658438103, -0.04799765436566957, 0.027389000356303712, 0.387879059408713, -0.24705058129582327, -0.10833405828248385, 0.05470611328161086, 0.3134179181647861, 0.37765921592600005, 0.17654737947323249, 0.30845900126204306, -0.15234630194346616, 0.2412099525374433, 0.10365756412816704, 0.23056667135349015, 0.3221502710972189, 0.02162357971436454],
                          [0.062225864345649314, 0.010017022754826688, 0.24559145179860206, 0.3424312216080492, -0.10152375900910764, -0.11682715847206906, -0.3781519066689608, -0.060110436588514844, -0.3081665923206254, -0.0668227779337467, -0.022483882897840365, 0.3302254266879249, 0.08399827150703443, 0.2260159098032801, -0.12567179986831278, -0.08944247896692119, -0.09804458511509836, -0.20245248341742117, 0.018297292915326474, -0.24580847046597953, 0.0512417073527931, 0.38066727798002187, 0.11156732375561278, -0.6001918165720783, -0.22699439175598315, -0.007056717158254563, 0.16119091509929467, 0.2475671931723466, -0.14053207449787236, 0.25195587393343205, 0.03843084790593941, 0.24699319265447153, 0.18781835398005176, -0.3338634426341214, -0.15825270407514092, -0.20064522889523556, 0.4751442516401977, 0.2997525775410838, -0.8707317456639325, 0.048023164021787274, 0.2516061276095334, 0.13404808062145257, -1.0092987140101382, 0.12109820766670422, 0.10739846681032013, -0.2882472430254693, -0.009874529212155678, -0.11427608910146862, -0.16409187347142568, 1.0035848163048608],
                          [-0.3208456865912992, -0.06757767799686322, -0.11493522484669175, 0.6155491340993239, 0.05794050945062908, -0.4039571402771479, 0.5244314679105847, -0.26255422479200563, 0.31263965326484966, 0.3316002552509834, -0.5371865990904016, -0.27187959835211245, -0.14558288906544387, -0.5107061190347327, -0.08653500655556047, -0.1194375802354228, -0.04731681046414874, -0.35458413924196125, -0.48729799890331515, 0.05559427978312384, 0.02670603322651881, 0.11629942581863288, 0.27875238230449934, -0.05959109018948214, -0.4870181241888812, 0.0047308267262781796, -0.06647262725269025, -0.1348889625937178, 0.0642239111284258, -0.1188489364450456, -0.0022802310616081415, -0.29192797100242907, -0.04401752553493955, -0.16917420566239338, -0.0100447652378679, -0.16638408912929, -0.4604019985548172, 0.2194379504065991, -0.1415436667194023, -0.32174352986456717, 0.5452664186747029, -0.12988359063329072, 0.0006482091611702068, 0.08536191846012142, -0.26195983824823266, 0.05310858190825693, 0.19437987648270547, 0.6116634897784823, -0.10899018941462489, -0.5937910782998408],
                          [-0.15156992299820887, -0.6173955316919715, 0.25756231016667247, -0.17590899631518006, -0.14379899918434663, 0.23212348444333586, -0.21201279602140893, -0.21768352731739277, -0.14376018741422236, 0.15771788680501878, 0.39088961552529566, 0.2829031846848314, 0.036740064193359345, -0.3899016630132894, 0.08239943834523054, 0.32780114939468635, 0.07761452917299325, 0.2810871829555107, 0.26879974891889336, -0.6065461936441251, -0.1373922415863002, -0.270832251651826, -0.014509349242903789, -0.060659137187506484, -0.526753282172554, 0.33368357693769857, 0.21320763311610236, 0.6393980310395309, 0.08541395250112248, 0.3001386895376288, -0.12530767244569127, 0.8226101052915608, -0.1635572194538505, -0.3842246600183955, 0.21146001566959444, 0.3169747616011556, 0.05068677329122142, -0.27511490600035743, 0.14433011225320197, -0.06093244157462926, 0.2868430449929779, -0.060479947260701494, -0.3993755960419534, 0.280463825475281, -0.19426790033546626, -0.11166175421506001, -0.01858793441741143, 0.3123213606966744, -0.4596978343837397, -0.8070331157655319],
                          [0.16459233171036267, 0.124634194530781, -0.1192338929739212, -0.09144667822205074, -0.2792389984011045, 0.4246586814178188, -0.2999166867918805, -0.4376607124094483, 0.02326578013799844, 0.0753375188590675, -0.20129788779080635, 0.24233167825033747, 0.006010470108957388, -0.009427230593903227, 0.028759253293415186, 0.05776010551406913, -0.09079395625302854, 0.2809891103202227, 0.1904625267868349, 0.42578661553720215, 0.19824811974770407, 0.41735030843010124, -0.07218221751975215, -0.25107607330383735, 0.12308567039591523, 0.09511631713722822, -0.03375297947482464, -0.27920749211555307, 0.3187104865529833, 0.05504439707426499, 0.052713521107788715, 0.011447046857792922, 0.07494494445378917, 0.33585573609256714, -0.6874680722638916, -0.2615325375191319, -0.30733871932379864, -0.20343683418979183, 0.33147598041431425, -0.20745146840176643, 0.007483954666241062, -0.46028757589510305, -0.27214194985594264, -0.5995825773273377, -0.1540938645216806, -0.355046713452376, 1.1006955362590967, -0.5544756215618218, 0.0005632400699910296, -0.27663854267731497],
                          [0.057100450664282806, 0.2126661003583206, -0.284685209672551, 0.05571180157337276, -0.3925231808259111, -0.20739818626126522, 0.36408635626539776, -0.725294878498131, -0.4245710432354764, -0.24820839538745376, 0.02184780621403595, 0.22488825500371323, -0.5176914033256075, -0.12019875442591899, 0.04194991780461576, 0.5431568064980209, 0.1772776361281366, 0.20793385251936164, 0.17335652653715145, 0.32956700356947477, -0.09879838854017474, -0.18897903657711165, -0.21296705922921563, 0.787701179706905, -0.22474404191305064, 0.19340704405543382, -0.08360152742526161, -0.1316582172460811, -0.2047377668857125, -0.06567113557528781, 0.47113445625048844, 0.5350154205095607, -0.25788006219002824, -0.16175690031015957, 0.2188008908361652, 0.11807148643285088, 0.5212061948993496, -0.06734876185001787, -0.4163408545298346, -0.1241795548397322, 0.23748705452529467, 0.20761470042829205, -0.10452344722762184, -0.09401503010192366, -0.07410126663211958, 0.29926217654046783, 0.46374631424383433, -0.26469214463664126, -0.10414296665966995, 0.01337260537056161],
                          [0.10736436189947347, 0.07209567285054941, -0.05506581025851331, -0.26368133819710254, -0.45804504830221765, 0.0693115656806773, 0.5032387686017936, 0.26954319941225174, -0.3268001390281444, -0.1687859532765649, -0.1972006069308241, 0.2357565605059224, 0.6398735039365198, 0.22865729659745113, 0.010483399871943504, -0.21659575010986667, -0.22978723584483274, -0.06729774700374797, -0.14716879413333675, -0.2082955468624024, 0.09812154069383591, -0.005184222491366254, 0.10854803393904415, -0.4614083768588506, 0.0016513028803329358, -0.6987217187295005, 0.30830181808971985, -0.7595001469835387, 0.17253852198453562, 0.42079661744881697, 0.11166309856994726, 0.18044388568966283, 0.18406442078101648, 0.12513571095242434, -0.289430458794969, 0.27697236419919463, 0.10135376506895726, -0.4467368484142714, 0.4842912920849427, 0.0026305709384700547, 0.30203155478695287, 0.12480711097712495, -0.148297593585423, 0.398292938350622, -0.023521860193883474, 0.1555848359369632, 0.20704569050559307, -0.17535815377929273, 0.27829795084112735, 0.04966347394371569],
                          [-0.29868369497462666, -0.2358413130386855, -0.23919499685391105, -0.1291846261342792, -0.19283439261703195, -0.39145309290766084, 0.006015389658665882, -0.38014062212535654, 0.17236947446924492, -0.45761103378314005, -0.004591685261890015, 0.5301287476230739, -0.648118311942133, -0.7676176362208973, 0.08371851530279405, -0.011358819030697494, -0.06605799393120307, -0.1814164739205332, -0.10400185656021968, -0.33022770423572023, -0.06869974561265248, 0.3178503324015528, 0.44395918324652517, -0.0620509865808135, 0.4189683500753665, -0.06200256220048817, -0.5539779292089321, 0.14731771411653447, 0.06949119605632884, -0.40971060582075797, -0.07834967009641829, 0.12021050123285618, -0.0434044413763218, 0.5270599945386306, 0.08050107764330296, 0.3556432769551731, -0.2916501310998927, 0.26721492089236654, -0.18881752713495412, 0.2280214761512651, -0.20020154950143457, -0.15587304459398124, 0.27629570237233914, 0.11589094184023886, 0.03438787273210451, 0.2911538101955855, 0.21788567304538262, -0.15557818490712774, 0.06536348759884515, 0.07315620827130162],
                          [0.5720426881174291, -0.37408074312631295, 0.5788407604097876, 0.24712030779471406, -0.16701228822174652, 0.014158235868567258, -0.219687574255693, -0.09411326141356874, 0.11432622985916369, -0.09880061772870359, -0.04153837604445342, -0.0766204878088828, -0.024380409611013357, 0.3275844714274347, -0.4583723532211306, -0.012073084475628926, 0.3102078059695432, 0.004045059444133897, 0.07328469063588014, -0.09350235857351127, -0.6099017272690006, 0.6253124266293597, -0.07216177803354681, 0.3890747009830244, -0.045245377577034146, 0.07662994841245074, -0.008631210538253876, -0.11696580509829152, -0.38375268753853053, -0.27767742648322574, -0.30395999047847944, -0.5378317512982106, -0.09606903920355309, 0.24253828912318134, -0.2944046575992731, 0.027385453933609073, 0.17913224608091824, -0.29064449623574934, 0.053329821442911894, 0.2085745521694417, 0.14447476502483594, -0.2100762283668778, -0.13251860212667171, -0.1129048890596004, 0.2153328410489768, 0.7161596610883288, -0.3055278860851549, -0.06564268420177756, -0.3834140879731684, -0.05620570105534477],
                          [-0.06380925275387336, 0.12241186386590451, 0.22562358403460733, 0.40354915869149227, 0.049629288725892756, -0.17083995532730906, -0.38880557264746596, 0.1820328563056075, -0.7917072265526366, -0.08806458518765961, 0.06329080782127958, 0.00898654475308753, -0.6447611327306731, 0.14593857839706473, -0.10638270137420533, 0.012412401931832169, 0.2267450671497012, -0.09073378788132037, -0.36340522002350856, -0.1778370461521577, 1.234693903465352, -0.25724989522499686, 0.15934513393940428, 0.2740903608638073, -0.07482484823026507, 0.3707749178649854, 0.11794136531749533, 0.31421249539742374, -0.05356645510232498, -0.11219503177430842, -0.025200519719975494, 0.11317833033297664, 0.028696925697534383, 0.2717489906987425, -0.20319304546179193, -0.07165914132491105, 0.30486326841261185, -0.6062079580109132, 0.3247945249045219, -0.025932069290657646, -0.3122447975650112, 0.16251011959092906, -0.03697571165001618, -0.05159901663958318, 0.4706089034286523, -0.03447996718148019, 0.19948660170748875, -0.01753099518681731, 0.8991388018524373, -0.21231995004292062],
                          [-0.07330129292270737, 0.05730252854257464, -0.23106332305154687, -0.14950488184937336, -0.20668727703343728, -0.054874960463536175, -0.29142030177523476, 0.3320731464223642, 0.16767269715829927, -0.09843575548856497, -0.1666686602856982, 0.5059795526906545, 0.05543699387886429, 0.16771726509720492, -0.3758384797161968, -0.0065175446169282005, 0.00018875784866835785, 0.49986950943110575, -0.26987171827939294, -0.20651164574799757, -0.3740198688061141, 0.3385418049956094, 0.3642672209475073, 0.2408128135996878, -0.518347606260944, 0.39882951811822653, 0.07924348913002043, 0.5916642110309858, 0.49034271284319547, -0.21738559998143472, 0.5478985530797114, -0.2929362450161456, 0.006459944675479616, 0.4174111346573358, -0.01847040686793699, -0.05120659007587556, 0.2835105954903688, -0.22866323834471558, -0.2072046637161395, -0.1514549669009214, -0.28138151747352375, 0.14040434802025314, -0.020906140687823503, 0.24642551988318262, 0.1600975078698443, 0.17739788970128334, -0.015605573267136415, 0.178741952244825, -0.03933748024218745, -0.04240240931955104],
                          [-0.3222986887122009, 0.38916895210719216, -0.555508278654762, 0.2094408664547181, -0.2406999450414222, -0.23446735417120404, 0.058205417339312226, -0.10386087447339108, -0.45393037063479474, 0.2973827735342331, -0.1055864863987161, -0.11695396517599457, 0.2913007583388519, 0.23595351651812208, -0.7799720052732648, -0.30258669830587975, 0.35991860582164176, 0.10447286445871784, 0.2915223101888283, -0.25297271309690145, -0.22292008644467104, -0.45521613816740064, -0.01240924807890434, -0.1470038124362313, 0.311567006043194, 0.31372895607972057, 0.37584975363431716, 0.0005833693958408666, -0.16825328507965356, -0.05505426638483749, -0.19927619993851353, -0.6559882026421221, -0.26546192108928723, 0.03564437725114174, 0.20906977812186833, 0.04920317518018665, -0.27204459003596915, -0.0854966307507292, 0.04347927930771363, -0.13610429238216395, -0.1588381978200916, -0.04528736401788344, -0.2777967268529637, 0.08529786202526168, -0.6071160149282281, -0.16525200223976702, -0.562098959448713, -0.147664942146306, 0.27369056170535394, 0.24867634437647354],
                          [0.577953885317215, -0.09106218378555221, -0.17666967996293167, -0.16993732643809342, -0.5766020492717326, -0.3584550679944625, -0.32798305714675363, 0.07499141041388444, 0.21626544478411752, 0.11060463809026866, -0.4173787300338519, 0.0010593482394206275, -0.0005856398122577342, -0.29403318621276114, -0.2124023273787954, -0.4473689885660676, 0.10471128417129398, 0.17454953359769287, 0.20337322160491927, 0.4426415726685595, 0.4127380296061156, -0.4827018033189484, -0.2568267234214668, 0.1262347135903954, -0.14871815463006888, -0.29582777513741537, -0.1140686807740627, 0.6188662887444208, -0.29322522457317796, -0.11337244080040315, 0.22689305480866903, -0.0906003661707618, -0.15310310721319204, -0.010838202212164147, -0.07722246895362464, 0.06478137434727567, 0.6344604267485869, 0.7559679935213122, 0.1086881167160777, 0.3231669396371168, -0.025960724296534125, -0.32210059156544746, -0.2645222895435054, 0.2703031748702043, 0.06759141233632368, 0.08961007993717313, 0.01761408162557046, 0.611754572680239, 0.15260469684981834, -0.08080187263858223],
                          [0.11307496735967953, 0.15074960576658852, -0.2948737885158986, 0.06581958935929196, 0.2495826085544439, 0.1329267960863252, -0.28584499386903756, 0.040554192710225166, 0.08002258598677522, -1.0475116683525416, 0.04391787892488825, -0.12741785083409682, 0.1406916458064814, -0.14030390115988492, 0.021502200325171705, -0.30860113015669793, 0.28793984699546193, 0.18614517545852474, -0.25169118563954884, -0.04231257706834903, -0.09875059683730783, 0.398460415527548, -0.6212080600410093, 0.16180286621090786, -0.11119721507368277, 0.0344773657869108, -0.10734564597286873, -0.40024282968117986, -0.23412262916404794, -0.23078220648524975, -0.2868789342948393, -0.12544708416138012, -0.25229820086258975, -0.22999359590924057, -0.20813881391225075, 0.3040508623690138, -0.16952054785563367, 0.06501066865733417, 0.02731332353209532, 0.08716314164093474, 0.14874652910879488, 0.9443395888195854, 0.06259272474303469, 0.1474305701903723, -0.03136262050330801, -0.76634636745167, 0.05395560155271519, 0.25836705577046615, 0.5320266876293153, -0.3543688444031803],
                          [0.3742188223223157, 0.23774521528486417, -0.06504308824314198, -0.06515977084975533, 0.005008458982703729, 0.311548001533244, -0.06871543093446851, -0.2632394254085519, 0.3667359903826425, 0.010029893302420942, 0.3617770723089057, -0.0682174418394283, -0.7619469759596454, 0.556318091512958, -0.32259089775646327, -0.27248031844504095, -0.4406191855474333, -0.06007291516538939, 0.4673918187928599, 0.05644630762721236, 0.0868478990621616, -0.3279214870719393, -0.12512172818402928, -0.9290843483085615, -0.43304224601643954, 0.11698820592901764, -0.2711649464968398, -0.1930978867988724, 0.0562117857491118, 0.12715229727420607, -0.050891698334357534, 0.01951673052951869, -0.547551385042132, -0.16187084101254218, 0.2868825023887508, -0.2738416205644007, -0.17673986638759598, -0.41433974907674503, -0.08863284684148869, 0.031980727095413844, 0.18485947421947066, -0.32284776138978766, 0.27035498465495983, 0.2842076519669699, 0.21060530319358117, -0.11102378386935609, -0.02734531550853666, -0.01041734621367342, -0.16114078466931328, 0.18935418973157125],
                          [-0.36683076453699776, -0.12619379350894896, 0.08472164140234066, -0.2771611056913561, -0.04071175708895358, -0.06908974391448046, 0.027251482097147495, -0.2757170429098016, -0.04356676688889526, -0.24429543540441037, 0.17288625289558984, -0.03666332401465478, -0.03297993012867942, 0.4653464598390539, -0.16231008811072328, -0.16492568214791672, 0.32838681505687595, -0.1663308053889196, 0.5196814511703353, 0.2444955183315024, -0.3519225172561871, -0.3616004034606635, 0.347080891953133, -0.133849126523749, -0.06083408883453145, -0.044260801388124806, 0.3004771885341632, 0.5192197098152602, -0.05047881411186106, -0.41979701039713335, 0.39063547450487, -0.10270910902434678, -0.34146709448929413, -0.2131475337295137, -0.1203242192441367, -0.04312538149311592, 0.11774972034345509, 0.35480549282018337, 0.3492029135437031, 0.015788339397940457, 0.20733546751841003, 0.21544373849342624, 0.006092483570007973, 0.08838705637583114, 0.031757601558188894, -0.016293727333215196, 0.288456997141136, -0.6572304858445027, -0.19357613704063792, -0.658391282499013],
                          [0.3401207939869326, 0.2208449157867163, -0.02323226737823692, -0.14616764641661162, 0.03565674653393306, 0.12017330748770817, 0.3952895845006816, 0.24926867373630662, -0.23646982328133487, 0.9824386966161585, -0.136216529315826, -0.19168715061642871, -0.3714216365373221, -0.3173221524340645, 0.39878930306337945, -0.16644500334998777, -0.20012601956651016, 0.19112101907119422, 0.10008830633486426, -0.3768131710449666, 0.01860603394298517, -0.3682086666167436, -0.05397579446762604, 0.0654056388137753, -0.3605169420425447, 0.12989623005977985, 0.028068082216767543, 0.01176569509857758, 0.6737134494837314, 0.052009360647413905, -0.4981297645296856, -0.26400733446113583, -0.051370265120647345, 0.12181555257360918, -0.05931729072309488, 0.3121543509734224, 0.25744332344866994, 0.3143879958259941, -0.29312986832672794, -0.038421885503901694, -0.19772638660398406, 0.44716362383291985, -0.10189535660541041, -0.4628264639655723, 0.11079117861699954, 0.02938649262580934, -0.15681120903045462, 0.0876156074563671, 0.054924626572939686, -0.25700865200060996],
                          [-0.07148722242074732, -0.3070724072479076, -0.46113896788525865, 0.291292990962442, -0.03960842460605823, -0.025603698122717806, 0.28872689018402253, -0.12377265071411273, -0.04695231743476724, -0.06071692147051918, -0.313952109654941, 0.6345919131389679, 0.5941352052574163, 0.1486385835556845, 0.09224085246716403, -0.11992700771874985, -0.01923240788978853, 0.03104963039799001, 0.2712266181163513, 0.04457677993549325, 0.465994767402929, 0.09327306815036605, 0.6744238852826367, -0.11698199955826591, 0.02632671281035647, 0.3228185253373628, 0.290772728688503, 0.25003366741186867, 0.06753636133597031, 0.07147731318883316, -0.027178797551161666, 0.039068517349304256, -0.12460883799158269, 0.07809316275962737, -0.1691553866096256, 0.2916477709834325, 0.1718295525552336, -0.06765854360675551, -0.1731999897100989, 0.1260673299487263, 0.3434873578038119, -0.14739891690978676, 0.6766898261105093, 0.12211166695618614, 0.06278648852285953, -0.127297110957933, -0.6299362431977316, -0.012022401433342027, 0.3867058070836041, -0.3894865862716485],
                          [0.16746306790487805, 0.1078212739183369, 0.050434777087704226, 0.0285772018783096, -0.008179294803447321, 0.8867276203221859, 0.05983949103629487, -0.2858239638864642, -0.3861377135206087, -0.22079906147233277, 0.1920347284152267, -0.328334665557723, -0.07997164085347093, -0.39320974396050923, 0.15288856585945568, -0.17258431693795934, -0.34994569182421115, 0.24075984282548266, 0.05996292667575264, -0.009689285340477293, -0.07028528847349348, -0.1774070355706699, 0.17598976496681093, 0.5238284302076827, -0.006637791518700694, 0.05780450988255853, -0.042626230447022434, 0.09046057320215158, 0.2521167947676831, -0.10834082293613845, 0.04283202719192873, -0.247910304687774, 0.640965978524942, -0.07141237368556642, -0.13332830244101812, -0.3661492898147485, 0.2693322258067063, 0.027152919208661138, 0.2564645713274194, 0.48118802200515554, -0.2600088789436492, -0.29055840802615657, 0.19862862302144652, 1.1583689299977686, -0.1339689539482607, -0.49197944827203627, 0.11614520812471504, -0.20959894171732613, -0.20552578993006054, 0.23549486104212322],
                          [0.05546932385600891, -0.30364727905565264, 0.5211010653833346, -0.13025800091859385, 0.07856124553805724, -0.3268210968292885, -0.01576592230155001, -0.284701948538991, -0.15992062493211598, -0.05721621574549369, 0.2063688487062191, 0.2372314792549392, 0.25239996356588995, -0.17386738970515198, 0.0063914814588736995, 0.49670853929407915, -0.03666162209540248, 0.04772161252284151, -0.20435076692228246, 0.09934023954929493, -0.028871849724336342, -0.1015139766838997, -0.19427413079337888, -0.42005626808927904, -0.5334929525826821, 0.40110883081835536, -0.3337660505460138, -0.5686831950082878, 0.06514240232565384, -1.1908161820222236, -0.10274131332246289, -0.04178157639942153, 0.05255626568968899, 0.5750814632501936, 0.37536190010147913, -0.4453284764309293, 0.16138307041656982, 0.10635164866368002, 0.705864485419143, -0.14713830042736026, 0.14402292615595347, -0.23365530210936078, 0.04718603983968284, -0.20652881935191458, 0.06680504148014453, -0.22394291862006982, -0.010028009526221236, 0.07492510970932727, 0.02489054677848982, 0.3399603505476075],
                          [0.1020804582589203, 0.5060973976476492, -0.17089457504948602, 0.0944821601832546, 0.47875398452900775, -0.35912446339511206, 0.5437453866545414, -0.06125586675349452, -0.026182342066228197, 0.2387497877557988, 0.10224366493618162, 0.09664285661036545, 0.1505302489490463, -0.09658124676004505, -0.2007650211598484, 0.27978875359884675, 0.04886491636519064, 0.1711995289277692, -0.17871578633589144, -0.1483315162900144, -0.023207109063953766, 0.11976595240573593, -0.3896540501008171, -0.6172187425138788, -0.3280492269533965, -0.3768764507592722, -0.24395198549717576, 0.19877480138972103, -0.11911650762768958, 0.11296816353384528, 0.4580956021029789, -0.3240805707903043, 0.1143747001169617, -0.0675124952259381, -0.45432479346309484, 0.36371272144251454, -0.31643773413598947, -0.12394302490003785, -0.41880837772441604, 0.465221404189728, -0.2561725623919863, -0.5173230893131866, 0.15632517369361623, -0.551537442057429, -0.3684810064489554, 0.10252036509190315, -0.28084667613695763, -0.5103414706666576, 0.09456765858473926, -0.08432325634304468],
                          [-0.319932564049145, 0.14081999833962563, 0.1384723448089354, 0.38726015838875605, -0.03887026415552638, -0.10240691957659594, 0.031070347947030817, 0.1954690132781685, 0.479650738983971, -0.1429193029308721, -0.016654081062057728, -0.25388376365665305, -0.2373583658227508, 0.09657300397252198, 0.707193816451285, -0.09811649549778721, -0.005954321805839501, -0.050963977820767906, 0.15299012261513042, 0.1574212918199768, -0.26036906590451836, 0.06750106966567461, -0.06197466265133112, -0.037781351314248555, -0.1911897041648567, 0.5319571137735595, -0.6658310935318703, 0.34449152987420023, 0.28271369508380473, 0.13016298429202133, -0.26192213676949927, -0.1005502074832561, -0.3929203658302784, -0.454024323874663, -0.8362281593662534, -0.12211978217439852, 0.11030275540485587, -0.0392032210497945, 0.4976332720486971, -0.101751541087296, -0.08114890166427417, 0.003376227127936629, 0.08848185627979249, 0.20321368727427608, 0.16271877922659564, -0.3281743499806351, -0.07900481523662828, -1.0688028807853724, -0.18502974413501838, 0.45351337110379286],
                          [-0.5914143103673524, -0.2265210625620185, 0.21214191869794605, -0.1932932538381661, -0.17892291744966793, 0.16731997748940233, 0.005326136252910907, 0.08489940087215271, -0.13488268747871004, 0.039964383364003424, -0.30806316052795013, 0.3262602370557069, -0.03313159378188398, -0.053377905848170704, -0.22253001341329273, -0.13383661933170682, 0.013649988584819564, 0.115110406105287, -0.27276915678267666, 0.003965357620138513, -0.0474641451832749, -0.48326008702560397, -0.16085894630347727, 0.5897625057812886, 0.2737704291452081, 0.42901413153321344, 0.22423555760121663, -0.36990755442103035, 0.14538658419082945, -0.16955835496286042, 0.23719223670413106, 0.3073634584938824, -0.06499147761416485, -0.41847926473651653, -0.0345358288395728, 0.21638588599695546, 0.07253367072172068, -0.17284563946092393, -0.2108601167296994, -0.061975977951014, 0.6642794115287244, -0.4378908793842747, 0.3182874251037591, -0.3955509475540829, -0.16421099589184432, -0.17683754905960355, -0.07967058289491452, -0.12287703894611567, 0.19454260217828254, 0.21710606847423017],
                          [-0.05040287446171415, 0.5986297281636457, -0.0846834946168808, 0.3820227457978761, 0.07376982701880458, -0.13856681592309522, -0.1514053820980311, 0.14355727385279551, 0.04890394793621735, 0.01397214151114077, 0.5351497968871931, 0.08781053454291064, 0.10566005142230367, 0.3198112915292079, 0.5358493634812602, -0.15458011294207477, 0.03673011323225048, -0.23292962138976353, -0.12575002291363502, 0.040911089329165706, -0.13655948948354343, 0.26551669833099995, -0.3089063374805578, -0.05107806154049223, -0.05322395091721269, -0.1671742086329588, -0.1262802542332465, -0.04562802932985076, 0.41479680039374184, -0.19724435807046092, 0.15246686605119758, 0.054798010406118806, -0.124143506375871, 0.10838056096197213, 0.131448571288968, 1.1407691646448275, 0.22179768608313585, 0.12015191399234111, 0.06955898232697756, 0.3604854344247145, 0.01849473727987146, -0.49831392695033977, -0.29598264168833355, -0.0737471545703375, -0.09142155669069094, -0.06529215401419818, 0.17317380053770126, -0.023831408363532078, -0.09521955418312789, 0.3121687751012601],
                          [-0.33120248209446734, -0.30413056118618403, -0.025468086695268486, -0.4617619307019855, -0.1894877110949593, 0.2325242691975896, -0.046559880995415644, 0.22664042728521033, -0.3369947252939814, -0.18637306591245337, 0.441799587660781, -0.1500787488522241, 0.0376759778387712, -0.13590829692971623, 0.3340645081836385, -0.37548243040642665, 0.12279818651600688, 0.38039801707898235, -0.06245834976966317, 0.03028745045228255, 0.12934265196886072, 0.2623144506199107, 0.024634049826657244, -0.03206485694916335, -0.4482650397646065, 0.07066115843066027, -0.20488268339097107, -0.0911264374899772, -0.313153886418549, 0.16520343188119102, -0.035200357303384756, -0.39286129393642305, 0.3373316809327602, -0.2855408955381998, -0.004112814290184412, 1.1054300766032394, 0.04155882592367749, -0.4359281844558761, -0.09778549739116878, 0.12268358373086871, 0.34065425983455583, -0.050109881087771535, 0.4253902437204669, -0.18850551041764477, -0.030048630633612933, -0.17647838389908624, 0.0648492866588751, 0.5673917897466884, -0.6342577922359767, 0.0739378409453638],
                          [-0.40218090147187424, 0.054267643955361114, -0.042265170844644886, 0.2300531278150063, 0.19144475271947184, 0.30016555704834297, 0.24044155919657162, -0.16903983479700244, -0.049910954715620036, 0.05360993723420457, 0.05022603777559711, -0.2984838227689926, 0.1620058144347377, 0.32276253168087427, -0.3589977636500304, 0.027602676267607547, 0.5006260250049807, -0.32249150044172376, 0.012943654630066286, 0.1398061444801397, -0.09489382256098057, -0.024306137598815208, 0.07270719663045135, -0.1989457658038086, -0.37372522790629076, -0.012253153191483545, 0.13581169080888303, -0.07645075251102236, 0.0457392640355495, 0.5435391429674303, -0.18938771027987214, 0.378083216003557, -0.23732038758747862, 0.704772498603475, -0.31994546363260107, -0.11734908724660992, -0.08677143505650745, 0.4261429631473246, 0.016156096334698153, 0.6089809205604293, -0.1894988647414741, -0.1262014424464547, 0.6189414430548754, 0.06135420242628188, 0.1917268166038485, 0.21473927067719514, 0.48585572993573256, 0.08868602139441244, 0.10019852299130183, 0.36121211799188335],
                          [-0.0509271782358919, 0.06513665119831494, 0.40069557489882573, -0.21637680287711758, 0.142936248915357, -0.15952653916535658, 0.10186545919159681, 0.07246469983720329, -0.38446919298010446, -0.057888235366998324, -0.31558638219921364, 0.2180577274807049, 0.2564824650748623, 0.05764741143181967, 0.04857660173588751, 0.13669066808100658, -0.14923638865799738, -0.854298498443115, 0.23602224985803288, 0.5383847690340786, 0.13987518979871547, -0.4871488066045942, -0.5110650797639444, -0.22555487120173673, 0.19035136202969297, 0.4941234875489759, -0.0857061823356981, 0.3370315823532333, 0.1602209600575866, -0.0967483092136465, -0.9144529357304536, -0.19454879503685238, 0.0939692709084495, 0.11616954968104816, -0.5273166009594192, 0.19265231140640543, -0.23141518968393218, -0.13981260607385965, -0.6384112364514675, -0.3853718818567395, 0.10226830969732746, 0.35800177135159034, -0.23777232956216562, 0.05484521031116487, -0.6549065727103168, 0.2071517932070334, 0.07471999018955423, 0.5978930214540176, -0.05827761517388208, -0.03274354835696984],
                          [-0.10694291408704908, -0.6320557478270096, -0.23392936695914085, -0.26082702963664417, 0.6825912640275553, -0.06403128817762026, -0.08774365747856847, -0.19319831650888905, 0.170500320302985, 0.025263569970606956, -0.1848830490415595, 0.12293668322368616, 0.2668194469973175, -0.009651165619591824, -0.10478501115382365, -0.5777789124009185, -0.3763207465339995, 0.16141955971001998, 0.3832615959862458, 0.021151858226874685, 0.12512451667494245, -0.19824215509313164, -0.5339737145675708, 0.08362790237758981, 0.2343509786011175, -0.5839242807473631, 0.5419877882087092, -0.25812210843688504, 0.30252846730195787, 0.020581624712660482, -0.04753588960909908, 0.26094885626995534, -0.34147904542713303, 0.11535884670494394, -0.22621139300851031, -0.0905620272650468, 0.08953574388837521, -0.07765844379801089, 0.06795039401545985, -0.23063463931825942, -1.352717731457719, -0.01526080763831351, -0.10804733267072862, 0.04650790604293209, 0.224211998609484, 0.20584964427955063, 0.05586432298035153, 0.09548366707020471, -0.4084935553460939, 0.011210277110263211],
                          [0.13973940761210815, -0.016825424777885085, -0.24214905569042855, 0.3877269098841391, -0.4818917144387956, 0.0411712986916446, -0.47185674243137876, -0.14085837447611638, -0.16389811300710752, 0.2107831900099674, 0.0024177478815045265, 0.29971062078978605, -0.26014979297215896, 0.18105491203823235, -0.06704994514319867, -0.5880680597224401, -0.0068248506972384375, -0.012713447046193068, -0.09639588241693435, 0.12797373602918594, 0.2060827559678057, 0.05029439802670042, -0.1251081727498456, -0.2928658124561131, -0.3089862175733467, 0.053625862660643224, -0.20787013178721275, -0.17884108155042294, 0.010654340121936305, 0.0021719698497950184, 0.22481262009589228, 0.08979716535186447, 0.3409371592489734, -0.04190182091704181, 0.07971014765044743, -0.13879617580404838, 0.39157194143147517, 0.11675216528774351, -0.20831556019010772, 0.047869571700907235, -0.6597442726882146, 0.25735458597109334, 0.5774637526277644, -0.006725239009087855, -0.7556984920273963, 0.2084542660207638, -0.867448440046151, -0.1842680719948943, -0.28770993128996575, -0.5518060278844358],
                          [-0.22510330758891767, -0.21800258299697284, 0.2530881506333026, -0.11759192198838338, -0.3536679301084247, -0.039855571453015215, -0.02883635751956678, -0.06298900612146367, 0.1782738644286694, -0.3652303390216149, 0.5021515515770804, -0.09060536533484478, 0.042352229309138005, -0.29562101390032075, -0.4955445236317986, 0.43882862681373397, 0.10594881305637503, -0.2267814011160597, -0.16424386256225273, 0.3788176520151914, 0.15811628615855597, 0.04676651297916668, -0.06043046875284661, -0.21400146384065727, 0.6066802733667185, 0.22059911871344737, 0.08233523575378822, -0.43714611993267205, 0.47789404124264284, 0.1518594955350015, -0.028428199631776802, -0.5277026193797065, -0.1783386544623167, -0.5657218832162665, 0.19486526285100916, -0.1363297014721808, 0.3824596264932283, -0.09578094493859884, -0.07222396225827475, 0.17171335499248122, -0.015808629061163027, 0.39568333561855845, -0.02653036943846051, 0.3979030361481099, -0.12460619393940527, 0.040983383451755936, -0.09759280971170171, 0.13527926158809442, 0.041385263961392235, 0.0863741173487552],
                          [0.3687956776366994, -0.09380589499884168, -0.17792143727081763, 0.3799549724032234, -0.1338152617318102, -0.2412358365218323, 0.3412131794931279, 0.8088087106052757, 0.2349551658618128, -0.025088357009543632, 0.24611984203726034, 0.7065600055809411, -0.20085646207873234, 0.20017129444350587, -0.4854570038872409, 0.1867003213644003, -0.07705020408331771, -0.023834353681199228, -0.11830348593720137, -0.10584371588379837, 0.136739823431868, 0.151421973858571, -0.3254335252636582, 0.17745368871651218, 0.054541134435212385, 0.26593534075888076, 0.19298475963350425, 0.11837366954207591, 0.08989715696018102, 0.12369397532221915, 0.031190889405750676, -0.017968724505745066, -0.22524779258855362, -0.22550027071145184, -0.264515391781326, -0.14785401167535772, 0.048172916819018606, 0.36307121737418996, 0.09014851939854605, -0.6470179494251904, 0.16577089336657932, -0.4958475076707615, 0.25606818909313167, 0.2846883965892807, -0.15679763351263565, -0.11021594189354794, 0.28742456731871086, -0.05818178663052926, -0.4017833092542494, -0.07562989208557813],
                          [0.07208385036054867, -0.139732290864554, 0.006276851687832734, 0.18812477041979445, 0.21666846288006922, 0.23067170918820257, 0.26912978483488526, -0.0390061365901562, 0.20987341242136712, -0.4779986071500052, 0.4122089839857979, -0.024994081666790284, 0.7796100822298815, -0.3369474605385623, 0.0875376405093261, -0.6384933024860157, -0.07213574820679738, -0.2428735772893499, 0.16790282561413017, -0.09085187556905838, 0.05585630053465192, -0.2226396312247312, -0.07951476159267803, 0.2885807369853336, 0.19586917302268317, -0.1921946660573251, -0.08325431404578061, 0.08531641726491176, -0.10703953226305188, -0.008769357637639271, 0.05173353019050296, 0.048581208233902835, -0.16804936006560237, -0.04615739198538191, -0.08017932604259985, -0.8313795908074592, 0.38730467390257234, -0.10853390193788555, -0.05771486614606948, -0.07475875565016085, 0.06980678153500867, 0.07204097278941216, -0.6496935663884624, -0.1687534249691857, -0.18809435057822746, 0.7499903422731594, -0.061787463766932504, -0.028703981067911107, -0.08636351935838261, -0.4258773296621623],
                          [0.17372424465397893, -0.4124464576009152, 0.009550492630217865, 0.4382745600431152, 0.11511838484051717, -0.05028465966988873, 0.23702878699366847, 0.13934370847465438, -0.1940052711343048, 0.6284884919583242, -0.13070467837482133, -0.006248731766055164, 0.09602134257919039, -0.09646042320355017, 0.09155973865124024, -0.21682111912516513, -0.10930351356396878, 0.25660215850603174, 0.3775928953733348, 0.0933613200157692, -0.1531207168326497, -0.6105778008909745, 0.04229704398231854, -0.0182150784767697, -0.11358721156221864, 0.13246228344447544, 0.21597555468693708, -0.05489838898220358, 0.624191132352557, -0.4784755900004361, 0.07064165309945997, -0.15504013128570857, -0.010696383826201441, -0.6457035070977213, 0.061220572248960575, 0.17856713857776244, -0.36818468925784237, -0.35263574401215414, 0.3738470312459257, 0.4024725210532777, 0.14702993464519362, 0.2169271993492267, -0.20487546407788151, -0.08404842074804227, -0.14830203580729542, 0.3819346977846699, 0.7471070294805356, 0.3741654119497776, 0.039738622560757664, 0.4049027381604331],
                          [0.12428926072645675, -0.21279921756594813, -0.03276206287421362, 0.029438180190151698, 0.06120442579862403, 0.18504128480523574, 0.07059370155481053, 0.08600114743878332, -0.16664430544762604, -1.0839411712418408, 0.02151825053936631, -1.0109915928799962, 0.05318683813644566, 0.17828599931282282, 0.16225838479860497, -0.030095769805606198, 0.17684321805634243, 0.10636908921038064, -0.0921212031944871, -0.7122587492114364, -0.042012195608097595, -0.11733379271622535, 0.057686169950701624, -0.5335947201617022, 0.4423295645669872, 0.15555114360235103, 0.24921246970079655, 0.19706282381620602, 0.19151755257853176, 0.5587601983886485, 0.012330809276312693, -0.16746902256726773, -0.1594852786243851, -0.4342397711657715, 0.0007485910777358829, -0.23144083607664603, 0.06962322259516693, 0.13044542786890753, -0.19597975792823946, -0.2745312077739771, -0.1325405805844991, -0.7903582212683012, 0.3225953443391168, -0.807993156317992, -0.14856787377072872, -0.051710637161252355, 0.19106536578489394, 0.0008497288133912884, -0.21745110867493234, -0.09005469979227243],
                          [0.1530044621701435, 0.24034241299126516, 0.2987924796971381, -0.35851588366748416, -0.006789267275917617, 0.3489751727002909, 0.6998101513554452, -0.18622324209707874, 0.15891987285371642, 0.09458424438959685, -0.19339080096633618, 0.0233554321280748, 0.5161733169976186, 0.07836531566424855, 0.43427531232799693, -0.19709434043357912, 0.12160068468117213, -0.5269581270919731, -0.33863135347908585, -0.26139667880033135, 0.23591008399170882, -0.1128967964657461, 0.22826058973079985, 0.05610743388841332, -0.009111760052658815, 0.25815759915569875, 0.018436551666044285, 0.16947697517648375, -0.5730572544300757, -0.14576946682730624, -0.29706401025384127, -0.2636114935153813, 0.01373136855262064, 0.023009819353948673, 0.2769818240541256, 0.5228220269933622, 0.07619646381371928, -0.2219010412767814, 0.0037837001647927247, 0.028063841531047195, -0.7600631857060789, 0.20390042605522798, 0.04179679975826947, 0.07770206167996078, 0.28798495717372535, 0.13577190453572333, 0.1130430620825544, 0.15348608014937842, -0.33000986264526544, 0.04510172009118008],
                          [0.39625998640374177, 0.13395906069480462, 0.1205749238220996, 0.006873969420211499, 0.1950359818203938, -0.5085558899751274, 0.19831021367767016, -0.5419213332055012, -0.2129390945179338, -0.017461932065041236, 0.4812518048886693, 0.6106838118621513, -0.22665519559906963, -0.32092257445807365, 0.05450193699479053, -0.5128266275330882, 0.1481377715640537, -0.6451634514694378, -0.01714673415028008, -0.4853720238555376, -0.3013262971452737, -0.10038783415103249, 0.38361631524714285, 0.14490581864963398, -0.05470694609596706, 0.10818175384380216, 0.04879084916110706, -0.021326750128830736, 0.3851526442132556, 0.6904222754922017, 0.2846049373501818, -0.005587038978743714, -0.04859893666778773, -0.12144523556206918, -0.3391612688078727, -0.06629282940436936, -0.11105310391881432, -0.025741746881222716, 0.47661555862899074, -0.2367993083247634, -0.18459958431867218, 0.38203601137307275, 0.1700307865725909, 0.059971287524476904, 0.2099814761949747, -0.27733819668430637, -0.4570826215310686, 0.671980687053017, 0.20346782486651688, 0.009550712167201532]])

    def __init__(self, num_variables, **kwargs):
        self.is_deterministic = True
        self.do_maximize = False
        self.num_variables = num_variables
        self.min_bounds = None
        self.max_bounds = None
        self.matrices = {2: self.matrix2D, 10: self.matrix10D, 30: self.matrix30D, 50: self.matrix50D}
        TestProblem.__init__(self, self.objective_function, **kwargs)


    def objective_function(self, phenome):
        phenome = np.array([phene - offset for offset, phene in zip(self.offsets, phenome)])
        phenome = np.dot(phenome, self.matrices[self.num_variables])
        assert len(phenome) == self.num_variables
        obj_value = griewank(phenome) + self.bias
        return obj_value


    def get_optimal_solutions(self, max_number=None):
        return [Individual(self.offsets[:self.num_variables])]



class F8(TestProblem):
    """Shifted rotated Ackley's function with global optimum on bounds."""

    bias = -140.0

    offsets = [-16.823, 14.9769, 6.169, 9.5566, 19.5417, -17.19, -18.8248,
               0.8511, -15.1162, 10.7934, 7.4091, 8.6171, -16.5641, -6.68,
               14.5433, 7.0454, -18.6215, 14.5561, -11.5942, -19.1531, -4.7372,
               0.9259, 13.2412, -5.2947, 1.8416, 4.5618, -18.8905, 9.8008,
               -15.4265, 1.2722, -0.4592, -8.2939, -5.9257, 15.6065, 2.0942,
               -13.7829, -13.0051, 15.1424, -14.6214, 8.6143, 18.0522, 17.5778,
               -7.681, -4.5826, 2.0896, 4.7818, 9.4115, 10.168, -2.8787,
               -0.4077, 3.0359, 14.4928, 8.2224, 1.8095, 3.6144, -3.4366,
               -10.7086, -3.1311, -9.4393, -5.0748, 17.5458, 17.4368, 5.4145,
               7.5412, -15.6611, 17.6698, -16.519, 11.1525, -12.1781, 17.7685,
               10.6649, 8.7514, -19.7875, 1.3069, 9.7793, -11.6669, 0.2654,
               7.9918, -4.955, 3.5881, -7.7524, -16.2979, 9.0324, -0.6257,
               8.8854, -2.7198, 9.4307, 9.08, -0.658, -11.0244, 19.8484,
               -0.1346, 17.8195, 11.7214, -10.2953, -0.778, 12.8435, -8.9002,
               17.6856, 6.1183]

    matrix2D = np.array([[-6.1741686291391119e+1, 7.4653970506018482e+1],
                         [1.5005440432658538e+1, -1.9763239081787880e+1]])

    matrix10D = np.array([[-18.785768809450733, 33.616954860437176, 26.882113915382682, -10.43306419742936, 0.9448928940824758, -3.353896433259691, 3.5352127339076374, 7.394276941396782, 7.790908752641235, 2.0912921099835673],
                          [-0.38080580880289006, 10.420967284673154, 9.347291913115678, -20.926490513724943, 11.4259031588907, 1.1056372574995397, 36.87928250597037, -1.910339780463472, 7.56116849320932, -9.743066435789972],
                          [-12.341688082549489, 6.3621993552738045, 8.248429939792464, 8.089256366417835, 0.06923450661901143, 2.5786241359574857, -0.4973402186181861, -2.062722095484327, 1.4302051477457656, 15.52200376094467],
                          [-17.00654251044467, -12.679306064055677, 51.65812051951116, -3.97661207806369, 3.9349384750136576, -30.777202564613845, 6.146597147627116, -11.404959107806402, 12.694206030880832, -9.395143228117439],
                          [-5.484749154282662, -13.643476981518553, -20.8125786035429, 12.48063177681885, 0.8449782059956992, 24.830393330514045, 33.83850518570256, -17.003569707093064, -5.293964367404844, 26.065703095424336],
                          [11.422878520470586, 10.221461943150496, -5.999478987400263, -8.935891602574142, 3.340791625151446, 3.924548854255449, -6.760571785727851, 14.016300477765046, 2.3533969357952147, -15.957358828479556],
                          [14.1069797350053, -0.689797572922294, 25.92835826668488, -30.138271725378775, 12.953067028884863, -17.125782201118525, 19.122903237509483, 3.8502101712160597, 14.44987126033593, -37.768641488073015],
                          [1.8171620273851625, -4.52289774299815, 2.596064824368431, -3.077970366333548, 3.666238380627702, -3.1422719671052084, -1.93910379576585, -1.132846020949443, -1.4593971192280721, -4.385065305078107],
                          [17.05963501513677, -40.88734304067851, -9.04136854737176, 9.207816613351653, 2.483559081696921, -3.135238286666343, -0.5159708434485237, -10.448164970351954, -3.97908386413912, -5.010151763870892],
                          [-2.100410473384162, 4.285743492212902, 18.13880373052391, -5.569156622351854, 0.02041492876416795, -5.5315683071808275, 1.7507462325577925, 2.018382353850689, 8.96737078655512, -3.5936542419629482]])

    matrix30D = np.array([[2.511190264603399, 8.257741094092125, -4.576972024334265, 11.663051871013478, 1.8392711991609043, -3.930477981638431, 6.381869129516557, 5.554163663846647, 3.792259995720394, -8.119579509114208, -6.749873116676933, -4.16744839268803, -5.247229107097693, 4.380091021522936, 1.6362652066486465, 1.5457444553413353, 6.729651169519672, -10.841704025647012, 5.27133366266056, -7.315142457952093, 5.586455189183544, 0.13436020980535218, -4.757951297029205, -7.494064226166734, -5.71422876353246, -3.4949745886606784, 16.86892010541546, -6.058600435639238, -11.353403083342686, -1.0923576208896637],
                          [-1.1762520815425976, 0.19467231905937696, 9.262321627779777, -4.877620455960017, -3.0138253051512947, 4.138941347418219, -4.381399078195643, -1.095599959738453, 3.3625539508337665, -2.415178728944297, -6.861613142354518, -0.005940467288075457, 2.135933445957106, -0.30727574965743903, 9.502954614655595, 3.789220991977143, -2.7696427577007503, 2.672716708411572, -3.374116812054481, -2.0411067751984415, -3.340247070143979, -0.8374990943521036, -5.259049047845686, -1.5214938770854254, 3.5602450658168063, 0.5878490497775958, -8.24965720380312, 1.5778811846183591, 2.56392435402041, 1.537958604481863],
                          [-1.3085812283158484, -2.0503713510983976, -3.8749598247741446, -2.1360189905937013, 0.4253742810704534, 3.043654044666869, 2.8524638400066973, -4.609692730685587, -2.6769899949340137, 0.018654380314467155, 4.1843245155883, -2.9414549211011054, -9.0702547942116, -2.890167502434176, -6.5170875304560765, -6.458220473167814, 5.2147578439709585, -5.6728283981570335, -1.2918267757780038, -0.4252377582507283, 2.004256913177237, 7.010207042020759, 3.2458529989643554, 0.9758521951937688, 1.4063626039048713, -2.052462316198574, 2.872078581244381, 2.445851054572349, 1.108924529459803, -0.2606291991295358],
                          [-0.6913013228225544, -1.9945323148620686, -8.604754485993125, -2.9761072632153596, -2.5462817170506593, 0.1308927281969895, -2.0952635102573707, 3.3569684373927915, 0.2599788677493233, 5.401823223841656, 3.0612958034992652, 1.403978462555566, 3.459625904755164, 0.8751447813065708, -4.037867043038509, 2.6981413548750095, -1.0464805632232836, -0.1621208184980227, 3.972120681919849, -0.36009588611547283, -3.5904310229755434, 1.1357447862030143, 6.085551336331358, -0.3065977505986462, -2.3495830369958446, 3.903939972739528, -1.4818186582438302, -2.878395159465752, 7.258503538363769, 4.796644771310021],
                          [-0.44423208195583835, -3.5713856787861915, 13.286045625307453, -12.839633153682575, -4.489679872696604, 7.869013641543565, -1.965105477391064, -8.636636350365924, -0.09482264984524189, -10.673493697992319, -12.53156986402604, -1.8549791851385953, -4.845831775303889, -3.3031932998435436, 14.504105531464209, 2.5900447684339514, -6.135686158996087, -4.270904315841696, -6.861683086853831, -2.744870140116196, -5.102676391316569, 5.943125003510186, -8.766966677691988, 2.000662359536807, 0.48220557102433964, 7.450887387478847, -10.45013256174983, 3.625933262996068, 4.179177543735097, 3.9824767971390695],
                          [-10.464299208072154, -0.18582026098543591, 6.5520413966727755, -5.911945191669922, 8.168937620030086, -10.982208830632558, -3.00135846456146, 2.9120659976777534, 2.671937505436929, -3.2565588882474192, -1.0109819456145726, -8.18916926806801, -12.001606012082647, 0.5651405467043745, -6.816130902986662, 7.034212706943163, 12.40601107576485, -14.537675875626526, 1.4015463474018615, 0.07076147778121572, 5.173018588042309, 17.500331178638167, 3.0976732548896924, -4.026635320023784, 3.0078191708339106, -6.986787465181084, 6.3941959158811414, 3.687826212758699, -10.500866576776128, -13.151879796730649],
                          [-5.781123825040053, -4.695427919915561, 9.384139425829048, -7.575211154097536, 8.514278328525183, -5.720630545110835, -3.828493595935516, 5.90998861394398, 0.14890776917348506, -0.6311855790951664, -7.572076478194538, -3.770557393047349, -8.178577514209056, -5.098256869553833, -12.07105725371895, 1.2487379324442383, 0.47854419252377683, -8.678825875969533, -0.9462098586300893, -0.7897275684569901, -8.64978321329506, 2.485419664365408, 12.153488342644684, -3.0622423287838485, -0.17037027582981643, -2.0935247896287046, -5.15404262324049, 4.5082775046079915, 1.6106478189308102, -3.274879887486953],
                          [2.268818310206689, 7.300452401745146, -4.331575628327129, 10.74428874720899, 0.1504927377755603, -0.5227038319103562, 6.686183057924437, 3.7822199735751276, 2.188704155720839, -8.966284106856413, 7.937092834912006, -1.3156741545363397, -3.3952200221430955, 1.5598471755009025, 2.7888347922477204, -2.138994752636062, 8.323858134206022, 1.1033496914596266, 2.0809958580689525, 3.6719079154589567, 10.73018786831511, -0.7865749211442614, -12.077147983799486, -3.2004833008354185, 1.9646808590108427, -2.6570004171458312, 12.549412268552729, -5.048053145262655, -8.92560318109584, -3.2910786331928614],
                          [0.04105519419198197, 4.198164286592344, 0.380892004577647, -8.23942251831774, 4.779957612796264, -3.4787851409838986, -4.340229693544734, -1.2696168576985805, 6.146810236016467, -3.0718648059925666, 0.9713379129568289, -5.896120164000506, -0.7575508318583819, 0.8782009342897784, -1.5325189430622428, 5.844371522790389, -0.837014728066397, -3.9072787371408775, -2.030173016623314, 3.202676786182734, 1.470137096717735, 6.858775309836105, -5.7516870726842155, 2.2611091220924573, 5.446233047509159, -0.9945752140316289, -4.198066474981203, -0.9162837190624441, -4.341766414322003, 0.6600030953405235],
                          [-4.758805234887069, -1.249096368305273, 5.230241683696193, -19.44236789411332, 3.990165558570265, -5.21678532927235, -4.960118855176376, 2.1136647271764106, 0.36524674000405544, 3.6709416597019735, -1.9142880398717106, -6.094333718753953, -2.6829676276599868, 1.227422326532798, -9.433645672535167, 4.9664378753052905, -3.616907241245708, -0.14555234532874778, -2.2803947884135107, 2.265409089297554, -5.317818474692691, 4.356958717932082, 9.75139488271248, 2.1844180242408706, 4.551729219908339, -3.2039375193038246, -8.161877539527412, 9.881964507838315, -0.9426793939999769, -4.0907597726080605],
                          [5.895714456877837, 5.673958378336709, -7.551189095277389, 8.6763663921392, -8.74212847286295, 7.876941413449155, 6.186314942878492, -1.3852922593783206, 2.3393505014942435, -7.303402814099851, -0.6203459067214555, -3.2148004432772503, -0.5285586783061472, -0.32765172515905244, 6.60017678920534, -4.030480514390237, -2.74819741682516, 1.1317199597948133, 4.041615304275, -3.807276819902737, 7.174795405973731, -7.105910692661075, -16.16824790535485, -8.490742807482299, -1.205275569694925, -3.2137834518973687, 10.19460083362542, -8.1537702193513, -3.34614228867147, 6.852576635102674],
                          [5.050755100515472, -1.0150841665383323, -4.488951762662663, 3.0719093620471365, -6.3092147908051945, 4.607171586689359, 0.5187719859312746, 2.5762076049942544, -2.5389515408876373, 3.8711704280020434, 0.86099787829018, 9.333192517131101, 3.898241746958841, 2.045637143082407, -4.632172666627585, -3.251052343878506, -10.340844584872428, 9.938362935470167, 2.681875064667896, -0.3092887196066297, -3.553804690623363, -11.909220170048043, 3.407795649838816, -0.8522349929092184, 1.4705436942360726, 0.018651383236501812, -0.4009399370789993, 1.0066041993408876, 5.546567833598792, 8.936287451220219],
                          [-2.296736642073308, -7.507685960269931, 3.878804185286892, 1.927870783876082, 8.606052457582582, -4.902250427177561, -0.36941525186262203, 3.7711058984310473, -4.497236148553519, 6.92064424996612, 6.555606653106257, 3.979481264635631, 0.5516700172892737, -1.2503803722341893, -11.577926205347564, 0.9894936246678699, 4.643449205576342, 0.039903009834914564, 2.717889467266028, 2.628829050189848, -1.391097858187011, 2.040534828877561, 14.571897255676616, 1.6457997025263595, -0.6011288484691789, -0.8404047965826955, -1.4631979926819447, 3.202376802222665, -1.6711768384467316, -3.3759766251534],
                          [-3.9375156766066337, -3.139929919295693, 7.0632092358776095, 1.8092484750688675, 8.870247836692679, -7.116279757514643, -0.5615698009579931, 1.6076237660478159, -0.4429803377018967, -5.392629763518957, 3.2533129344769782, -0.5391049139516102, -7.500926900397016, 0.9805730851429574, -5.634247758036995, 2.236908157606175, 3.147822704551946, -5.9377624757314145, 0.8872390596601577, 3.778662703078195, 4.579851636624116, 9.68224407023739, 0.005007990083924427, -2.0666085879540494, -2.8401007471138193, 1.0726704708603498, 6.107120400203828, -2.014818439278602, -7.165823950693442, -9.272074819149603],
                          [-2.680240010855875, 0.35600434995900165, -0.4144569076217877, -0.06776461827814266, 5.320643303727243, 0.7378754378835478, 1.2136684358984866, -3.145814327985767, -2.507697183673514, 2.536104038824088, 3.558431317438912, 2.1311591533309513, 3.764201252276733, 0.8162043973711717, -4.632956423959657, 0.18412028912752174, 1.2174333740701588, -4.036068485529014, 2.258556949877698, -0.9100825423666752, 3.7541872563123806, 7.457650756177197, 1.4568472563765222, -1.6086122502502214, -0.7576487719636653, 3.790322347324032, 0.8870400990512606, -6.027791111032598, -0.32819170272693354, 0.8660027666095153],
                          [6.578208237675698, 0.8184102932913206, -9.288863917296439, 22.14413550757907, -5.223269252930489, 1.8041411263480054, 6.924767448007314, -2.7767801206174636, -2.4943841856277307, 4.743636200841757, 2.6908221523406097, 6.088855334101281, 0.9810394501311365, -0.12367295029956667, -0.1401318287401328, -13.797914058713273, -1.914030716314594, 1.3560533260143395, 2.4858409618477904, -2.669304324669153, 2.2893697932797594, -12.59957022859263, 0.93847348962723, -3.000750651813473, -6.748464919761018, -2.9989918132723914, 7.50927404965129, -4.712789813740673, 2.5839742623096855, 4.259407321635679],
                          [-4.302432019312597, 1.0302191609738225, -0.03540140103404428, -3.8646036591002857, 11.181713062447539, 1.7991288421541245, -4.632382202673897, -2.780991517575134, -2.5313169687422117, 10.165545659183975, 1.550622641761506, 5.402782921171298, 7.0667343369424325, 3.5594650532243692, -7.838510786198628, 0.8727945279310535, -6.729399256798964, 1.851943659950421, -3.1730230733872498, 0.30347745630898443, -7.294328632117977, 0.4312433576422381, 12.108553118909567, 2.7855123137319424, 3.951422589853138, 5.216571541524427, -14.01807799640241, 3.6806182504814076, 10.312528175880152, 9.072426201364811],
                          [-5.5329864922490914, 1.3681576267420914, -2.9962789113758914, 1.1805877383771748, -1.1727360989475766, -3.0334405374569635, 0.24082744014884844, 1.2745548563169782, 3.6640706145869864, 1.450239057462672, 0.43774922918318027, -5.139121137322196, -2.1842397317007856, -1.7226736633847843, 5.003903006525603, 1.555677376370723, 4.592628073610951, 0.031529946446351376, 1.9620386750575327, 1.3115193273334391, 2.97144042923956, 4.3586285713805735, -3.0209104071865602, -2.026289674016738, -2.179100342988682, 0.19956140445455484, -0.6422687649699342, -0.4843021746912696, 0.8676971938854262, -3.4282566958973373],
                          [4.749688683748538, -3.0130962354663238, -0.8964734715061581, -5.404408901826174, 3.2519001244819705, 4.700501169041437, -0.48584192208462884, 2.1290198545775536, -5.743986607455017, 5.05072704099386, 5.290426760843522, 10.017074714718186, 4.109783454482344, -2.171588999232304, -9.143973237995885, 2.1580967522563212, -4.288300573083361, 8.440796859622568, 0.6229337082016434, 0.3668577705895741, -5.372666211859802, -6.422211641960874, 12.41021882450194, 3.7493570416711957, 2.7699126830365564, 3.5082222436746777, -5.740173772592829, 6.606652001961278, 5.630677773801195, 6.253155297611987],
                          [4.559034813600619, 0.1913350266480245, -9.72469888506327, 16.676435885179576, -11.815244699143054, -2.963619615334862, 2.644195062564698, -3.5045347535842035, 1.5394738508529109, 3.2479533092428405, 4.136178478443613, -1.4150422310097437, 0.18077736159480107, -2.5167947201866085, 6.160828524988144, -6.794575512958723, 0.07581772972899481, 0.7273671762874099, 1.6683055420721837, 1.2534583279579286, 7.270581180009739, -5.020680731171347, -10.710260791368778, -3.482314899976028, -3.159956937857408, -5.264233710547034, 7.244641579264396, -10.641144576600027, -0.028015125221679434, 0.21956976936330955],
                          [-2.5513085043810344, -5.624763316256503, 5.8227501198454705, -18.83984795424589, 2.287665332808314, 3.202650930764413, -0.14315718142819273, -1.9393371050172241, -7.297486396599838, 4.714811455331595, -2.285923849103187, 1.5719145123582468, 1.1886888386113303, 0.6651250486650224, -3.62112489105088, 7.315055227559002, 2.5920811268356303, -3.6049680612829635, -1.2692515127885735, -4.603628537664914, -6.1868646030843815, 8.62584019419564, 15.640620254332642, 9.339712043883617, 0.7005649246134704, 8.033185101455041, -8.984569662115023, 8.756788125840803, 3.0002946387562526, 1.5527019436374203],
                          [-3.861290375479539, -4.675448372631536, 9.251215593601152, -23.607840924556356, -8.695565550492798, -2.9890761291947134, -4.7394354006804384, 2.8356784578772074, -0.09816665388344539, -12.599368055638564, -10.371683803332736, -6.880266107606954, -13.323520170751154, 0.8366430981369344, 5.174040794619552, 10.49263721553011, 6.8020531332912935, -6.116168624020712, 0.6121682022083461, -1.5616787444124285, -2.961017400369882, 14.555633724082611, -2.408921040101399, 4.164763222501917, -1.2885371130833283, 0.5249215999155405, -0.4197438099013884, 6.847306921165118, -2.4178117806412915, -7.702891475003848],
                          [4.161137142387705, 1.4820377177923658, 1.2871131760474233, -11.780576083757202, -13.704438843176279, 2.2529706444658775, -4.628978648000722, -8.44852567895537, 4.501241923053804, 3.2180822100676387, -9.096692746700437, -6.359766478822383, 2.458766877052462, 2.7472396634682528, 11.352826871439879, 1.6900717568947712, -4.9446475840448105, 1.2874782451208633, -5.514465163301222, -0.43632046338437597, -2.460885613501708, -2.7668726158968946, -5.244256957948294, 3.839764805067356, 1.9744867737957825, -3.8504675955016556, -5.719184031653332, 3.1125551947104624, 2.1919986794855943, -0.13026077572473954],
                          [2.1053106927251957, 0.544816987896791, 1.8463460394236781, -0.5097848054883325, -8.19139781918348, -0.22684979135170835, -3.0851797887818924, -5.699211118481947, 3.293594390108838, -2.909444711189409, -7.764958328853763, -1.1668970305075335, 3.020313808115756, 6.799166924533074, 10.7746541697421, 1.5250393026865812, -7.554231411053715, -0.5870131079195986, -1.3007596553023062, -4.080716282648349, -2.186465568681216, -2.4064847064680794, -6.958047179121224, -3.2357537230417077, -4.304227928343334, 0.20637053734784094, -2.6250272141353848, -5.715311685851154, 1.3755895996309002, 7.31396280189546],
                          [4.4606143748925815, 3.1952976197848404, -8.190318873976533, 17.720929604353678, 1.0799035076752885, 4.032046572169589, 6.846170634331792, -2.990923968785385, -1.0138487919520105, 4.210550185586089, 0.7647110258199106, 6.307647053631859, 6.791811206494187, 1.3588577626377345, -1.6246138143018776, -10.150256933841442, -8.955086516370987, 2.41129571860055, 2.9918358631703246, -2.0841054852597667, -0.38788283603335283, -12.037911290804784, 0.7847814450043312, -3.5125859319946096, -6.3580512246788015, 2.354270933909405, 1.7629785519170058, -7.508929712507494, 4.153956520027669, 8.430589071222315],
                          [0.11299545143656423, 0.3719367861398056, -7.025568330781, 9.769013689448464, -0.640793257766108, -11.677067837314976, 1.4515727489278767, 1.1715021644690176, 1.6926476580912793, -0.881205326796837, 4.268282553052323, -5.637459471007516, -4.8974500013531115, 0.17173589307977707, -0.9211330468045001, -2.4905148715845815, 9.607046944744821, -7.206797204798768, 4.349647727744751, 4.621339140117586, 10.832359506208546, 5.100626759149631, -4.076131311364439, -0.9327754879353387, 0.5026551214022198, -8.6858164471001, 15.294166113376837, -5.994248698329602, -11.694453405846096, -11.801285777570579],
                          [3.672120136902007, 2.6266279533884895, -6.295270647326517, -1.069859934951232, -3.8113302994051375, 1.5949006885529526, -0.8226494216103423, -2.9961023548979657, 0.011410712161871278, 3.8049739555840176, 0.29095969601553967, 0.15441256642165688, 6.0091236927948035, 1.0488409418576379, 0.11420533472469993, 1.041139886749876, -1.7664882512232245, -1.2322615086785684, 3.2960713266398476, -2.373608140335303, -1.250158758539599, -0.7208731345537661, -0.6156773187005177, 1.4032590478706066, 3.0055294423481325, -1.215626145907597, 0.04209426947719365, -2.2318793590146617, 1.6113608788934293, 6.469735186892669],
                          [-5.818552929715837, -0.9169295712328542, -0.366611990060092, -5.1299503247235405, 6.905064130414796, -6.5445798712234575, -7.16220048221164, -1.2183927219650177, 0.02838697214804719, 13.304659303178061, 5.212031101975595, -0.1984145756217106, 2.043938045553383, -0.29548032145709496, -9.738766031154933, 4.373893439532715, 5.133821298601127, -6.818797238217159, 2.032166191824439, 1.2990684156080132, -5.2824866779081745, 8.79704452443048, 12.849891231025046, 2.7871568041542045, 6.2258818054938425, -4.429387659772509, -7.4036051874247, 3.4789466856660187, 1.6328914814136246, -1.1057519935283844],
                          [3.4868785913811906, 2.6339386298341982, -3.8345442598175135, 6.338661537296378, 0.08440220487435036, 0.891537452927858, 7.386752241464462, -3.1433412659900224, 0.17569569189932993, -1.189840684944929, 2.063745656499299, -3.873741753209196, 7.631037421790815, 0.8704075199916941, 5.4169231361090375, -4.796174802688587, -5.441484022053353, 7.113220057014851, -4.021007101784125, 1.0933378235540314, -0.7237134159105555, -11.778856908851662, -8.947505317275446, 0.37169797000665405, -3.3683497769665385, 4.657443083154591, 0.453955538023004, -1.7407016169440057, -1.6522054174095602, 1.0293176939756639],
                          [2.46194490699664, -0.3367320063335661, -0.5524038255785771, -8.452671777634976, -5.633844858472623, 4.461702367049373, -3.7791579697242024, 2.769245822243623, -1.006803537946152, 1.372993145527169, -2.883759917737196, 3.9353447413494282, 1.5888078809409378, 1.6663327424869052, -3.3451072476151555, 9.018878681488209, 4.966343941267382, -2.378406115718362, 4.984115491643569, -6.369451753840427, -4.810457553048149, 4.695631510723671, 8.870216904135045, 1.8512664237737275, 0.4785318619045305, 2.2841714202034744, -1.6997915076850245, 4.11585412102773, 2.079681507808699, 6.064422880045283]])

    matrix50D = np.array([[-2.1811729470950274, 2.9343564593939337, 4.115868920389109, -3.699243032052193, -3.2153101374188258, 3.466985078119013, -0.6862618534939295, -3.084613681096596, -2.9511902151453384, 2.60811761203987, -4.431835015004433, -2.284255953725203, -0.5843836481216494, -0.8193417696129652, 2.137061933426771, 1.0373706119516755, 7.5991505142347675, 5.19990971156951, 3.940847773856171, -3.473293828180321, -0.8811360505478198, -1.041472294238417, -1.3534110192071724, 5.682131780710389, -1.4005685069829434, -3.900897548022731, -0.18457739778169152, -4.037073412285883, 4.983088925346097, 3.958850069799029, 4.180161179070768, 1.0122371670224608, -5.269827562868856, -2.82767367998703, 8.227462079383727, 3.22926048906839, -1.3614573837420272, -3.3216232412179694, 6.9254427035927995, 3.713525692777241, 2.5439551347493143, 2.123106668913829, 0.14836095771699104, 2.613422926573869, 8.165032551034889, 0.18339867069126023, -0.17417279953330608, 3.3158315165344896, 10.718605428354687, 6.181634271976038],
                          [-3.146032968746549, 3.1294922933112357, 1.740677109763696, 3.397203334189256, 4.5168742879293475, -1.8609839352718112, -6.679475029698681, -0.8535043008684563, 2.3779878701480146, 4.202498967326826, 4.8786324869313775, -1.9753722953629818, -3.853222689639001, 8.56383645944542, -2.9484667067260713, 2.7848063190247965, -2.8848079593625933, -3.3911068408602145, -4.484321668112352, -8.421310645639629, 0.5439715267996306, -5.8404408232180876, 2.1837124162774164, 5.152470168369714, 9.45734772743931, -1.4234140156734756, 3.0934687334904583, -6.1834767902113175, 4.490992868094745, 1.5931733939682113, 9.66862972774296, 3.821108036270328, 3.6139427870159593, -1.615815153538791, 8.981652111896324, -9.896807563693285, -4.348002208900189, 2.141893014013037, -3.994941665346619, 7.059345453624859, 0.1018448762254195, 6.402955711346134, -1.3818998210997158, 4.68943512026946, 13.255764721669074, -1.6568005393764331, 3.896226569381653, -10.470762325715043, -8.57905089206593, 2.5761360022745916],
                          [0.42092447161147023, -5.94499360803197, 5.353384024490412, -1.520895188651907, -2.4662566080313746, 6.7097387253174485, 2.9842599043161804, 5.676840340769628, -7.747287791669881, -3.3580475312997002, -4.675635397896646, 7.522214307060975, -5.425833277570155, -10.288365006862724, 1.9199120248387, 1.4816911786796343, -0.7359914505350693, -0.8726149136349934, -0.14933500276098174, 3.578156201711868, 1.004877265514155, 1.1356479301293319, -2.206239758080679, -5.155702083635134, -7.668930896686096, 4.065752170058, -2.466361619636282, 2.9849967410942635, -7.0452187924416165, 3.438215123310483, 8.10135441240087, -9.030541675212532, 1.0220345349838247, 2.4098582775893456, -1.8982396462010536, -3.1848219163948635, -1.2656966222586525, 0.28133821108493495, -1.129073041219713, -0.8384858965900704, -2.1178873821856947, -6.198268918031194, -0.48227245553389453, 1.858702689779468, -2.4756399620641543, 7.758983423942538, -11.645653640910256, 12.877148071357308, 6.048154446600138, -2.4050185670380984],
                          [-5.548591661662151, 1.4988371369325841, -3.7020228016958536, 5.716584305144308, -2.067584692938686, 5.3360268949678265, -0.14268843872951417, -0.2482230846139939, 7.414772941403687, 1.3220465654098885, -10.20975088226796, 6.666630560155863, 1.7613534272086786, -2.760155904510411, 7.409651873226508, 7.590190906084985, -8.479285044319877, 4.78160149765015, 4.433770526692685, 4.073693867006923, 8.844358689338508, -7.168942365797874, 2.823978716391375, -0.5338954150425028, -0.4330711715212993, 4.959760679710093, -0.12601007487442842, 1.434558896770936, 4.156348116759847, 2.650810399317543, -3.581070516483921, 2.3805374293217896, 0.7648845350983506, 2.040457267963802, -1.1368984134823383, 12.028363466745473, -1.7310494862505517, -12.791032788711671, 5.801221198374832, -1.7204361114288758, 3.543006027700408, -6.077907694286926, -6.013416016706121, -0.49971293783458065, -3.3628939141447862, 4.06556940891059, 0.5735579869591081, -0.8361723538384161, 9.865894604449979, -0.11954359393107708],
                          [-0.6393556675002479, -0.395148961384013, 2.306998961226599, 0.38565463471003636, -0.48100849595698203, -10.006064984240512, 5.281141672670887, 2.8969528632672796, -1.1553189554396566, 3.96822030619333, 6.602972839117979, -5.878245606100543, -2.706804386855406, -1.550004581247472, -1.8180339237987786, -4.987310393964732, -0.5992701209444397, -0.04548101685038856, 0.19451656749203491, -5.882822677564405, 1.2275571169403832, -5.758686879292456, -0.4390237726479448, -0.29378632584375614, 6.135906497066182, -3.5563400831757206, -7.769409776815131, 0.2689306852542025, 3.005401559824679, -2.4796462950666323, 2.8353779597578197, -1.737705705417878, -0.44880067347915487, -3.640406676719827, -3.9622276542118477, -6.191629235803958, 1.9937579250643571, 4.338122797131882, -5.312970029936395, 5.289505851958587, -8.2473207910618, 3.9230927641429822, 6.118076535061491, 3.932044800936584, 4.057257306936264, -6.24637934603994, -0.11449017914685404, 6.900646311836585, -7.5269146850081485, 0.013336916884733001],
                          [6.4199236103241395, -6.031679781300002, -3.0689611155570096, -12.03565576937457, 0.3080971282463147, 5.477333288639311, 1.095985001303241, -2.365040760587699, -3.421881660615524, 5.333877726243997, -0.7710445175225198, 7.16994386601804, -0.21532439705948636, -1.5155100065161986, -3.4264779237483, 5.453997338861766, -5.679577642473422, 3.4076879723740348, -3.605223512688283, 1.8296723066525042, 4.647272642489544, 8.363940824165896, 5.133827461016357, -5.399536406002441, -2.848603597442412, 4.320239708074139, 0.3048833545788893, -1.4931740649216831, -11.396159484269791, -11.050907808768368, -3.4500513068044203, -2.6872707236169235, 2.13869144179758, 0.9913571624991522, -3.41362704954137, 0.41839573707648653, -1.3607422533909588, -1.3469137572609498, -12.553468395131823, -2.2509397155238924, 3.2940950763667654, -6.747381001425922, -1.7735676906362765, -0.09518380671162534, -3.3611071114495554, 1.1564853700262774, 0.64482717276426, -2.1970747979822134, -2.58514040063312, -8.773745346668058],
                          [1.8571626252641125, -3.896195962415534, 4.318979930911289, -0.3436678686024677, -1.2001599209008278, -6.500996753080366, 0.6968975033021122, -4.092053304788806, -4.9770519699926945, -1.7931007136851806, 7.552026745593997, -1.2320174440434515, -3.261105076849482, 3.3580617911869943, -5.691882785872963, -2.607676508873757, 0.6417849722819073, 0.4966473430712988, -5.355456353895686, -0.9523248460764899, -3.3894884630314155, 1.4540986965034808, -1.3254603447624527, -4.784289914273652, 0.1280712095243286, 1.8543231540116825, -0.13360613750237954, -3.0166364343414975, -1.2021482652910873, -3.273123643299089, -4.747620097705959, 2.0177207465402893, 2.561868224075271, 1.6416245764163035, -1.8590560400133138, -1.400969900942947, 1.6307229309117868, 0.859724554923492, -4.339439023031213, 0.44583309959060413, -1.3997426098970047, -0.4764252217506567, 1.2367873743967683, -0.01938701305529442, -4.400158218566679, -4.70680719831311, -3.638903889338451, -0.9760994115091898, -9.07397069455538, -1.3626293782393137],
                          [4.053026828745031, -1.8843489971519274, -2.3805543620199363, 4.548726162559596, 0.6023017954392615, -6.014388155895238, 3.0112680421170923, -7.106690997902299, 3.4975876325327815, 3.01693964594721, 0.41662935500322246, 7.90686932333833, 0.49479567736764113, 2.6004782294206312, -2.6409157885778805, 1.6288352667874704, -3.6741027469992296, 5.242591500805451, -1.6227706873430148, 4.649448647384096, -2.315641426620347, 3.3583851911673923, 5.4832028243714666, -1.8295180150881392, 5.959925687812954, 5.5431539001469545, -0.6460295458500606, -1.396644170106836, -5.020056344910143, -2.2982430911342306, -0.042345186263867934, 4.091549272494531, 4.348338946170486, -2.6524411093421163, 1.1606237709014178, -1.4442959328932927, 1.6004049208147142, -2.0059519933056813, -3.6052625257588526, -0.27532575056845415, 0.636279316289715, -4.716105111271755, -11.999199145830987, 4.551675629779344, -3.218495725168904, -6.48951616368812, -5.896749595237101, 3.2342669680385336, 4.143264034874555, 2.0569888761010273],
                          [-7.758145205996301, -0.09864514354040516, 5.986161829486599, -2.730906126742084, -2.042546237883511, -11.055593896832166, 3.229199172946432, 9.325007228472904, 2.3999742043158965, 3.7539013503070917, -1.63195399831212, -2.8447975642204724, -6.277077069926233, -4.570795568432281, 3.1626102967095453, -0.7208736734090309, -5.782668189322478, -1.565469717436058, 6.9813370538633155, -9.829933061443045, 1.9124855997478838, 0.5434477503346897, -5.100207601409711, 5.28389823670128, 11.433632185874972, 6.478063329602996, -6.146128245727442, 1.8831015014938597, 11.09958446702332, 2.2748443725911285, 15.868910588507415, -8.998974839152757, -4.039660910124327, -4.653993378255957, -3.898731187953098, -0.33485010648169183, 3.452066455171445, -6.363555764766472, 2.606163357920814, 5.424047072079213, -13.151951400348695, 5.219406649966909, 8.842189672977241, 3.6065330162966633, -1.3613786812808266, -6.6357170484124826, -1.9388374295440722, 10.45091379871981, -0.9812144477376891, -0.18786742896715825],
                          [0.4520209060806526, 5.4849844534018075, -7.91592598071636, -6.787117604614177, -6.282921174198018, 0.16634816179134576, 2.545857363053987, -5.0360008254541455, -5.182171435115663, 0.8915451143179376, 3.999838207270551, -9.454515686289552, 5.859294808650545, 3.8131092872238375, -4.748774690410636, -6.314000752202052, 2.9788044158495817, 1.5394282026260078, 4.101168311351286, -4.896940022659707, 4.96168550470985, -0.4391278495440873, 0.3999062923052982, 0.8344370992045818, 1.932385727212599, 0.6115970705289023, 0.6443172454420197, -1.0514187685791894, -2.975450672524505, -6.457284944824272, -5.038885822799668, 3.4568074781503704, 0.34599218967219286, -2.245504128301364, -6.9626339733592655, 5.987286548613562, 1.1455220373026789, 0.9351796361536809, -3.2308752791539885, 0.6702408409897777, 0.6791724862043859, 1.7112236941099024, 7.610043144980796, -5.147482684471794, -0.0916181332299297, -7.014120011383329, 1.397656244679942, -6.161307233334175, -5.8794340019326174, -1.9577365367461372],
                          [2.945221760539681, -0.16841047050692226, -6.619543558158152, -10.483330187872827, -3.5203936810447933, 13.386515666614596, 2.8242791307770636, 2.312044162067516, -11.150066758189546, -2.331066594054133, 0.35289234276077974, 0.5896895313422283, 2.866867801641331, -6.63545171594341, -1.1100856004113255, -4.294665014571258, 10.146693790965454, 0.07784213611533011, 0.9672209496644921, -3.201645882090357, 2.311402327409733, 0.4340826011826098, -0.7624217948383426, -4.625087487006639, -10.837328955784463, -7.913600138740039, -2.275600550639252, 4.617039430331057, -13.819975736283016, -9.59711749218468, -1.1725990333911551, -5.5343187201878745, -3.185700396458093, 0.6997298052078389, 0.11336298441676203, -2.5719425115846173, -3.4855449230365085, 10.38275371978039, -5.894915098290808, -0.10528830039306365, 3.606456218981716, -2.6267824248071747, 5.124729800156526, -6.648432584315818, 10.405809744053737, 4.267669334232777, -5.885054852795926, 2.0036724768435787, 16.843430701429007, 1.5940306368542807],
                          [-3.447185146419533, -5.845827361078772, 1.5767831454852228, -1.628179404923817, 4.5003201166809115, 7.692370920544149, -8.614032388489267, 2.2385027115885, 2.19126665943233, 0.8762897679096376, -0.28533344190825527, 5.197097820233922, -1.798835033261539, -4.450460033904172, -1.3978834256068484, 7.29406937162884, -1.956637195813758, -0.29874478102508534, 2.4293886225911665, 3.639923933632406, -3.6193124115116913, 0.43950290722549135, -1.1421990750310587, -4.367508933846327, -7.400901769659149, -4.054004326935708, 6.446247217538327, 5.7907859386822595, -0.4055721751540252, -0.10844480949466892, 1.3536147317934903, -2.7504152583989, 1.458325532075612, 7.616838514793935, 4.742746335165853, 3.3902099531238594, -0.31178474788494065, -5.170237623535597, 3.407643579606333, -4.407911489251694, 2.584267185096307, -2.040375817603933, -0.3797804431300529, -5.317872887661521, -2.3778464167760696, 2.894376734749014, 0.6088592129296643, -5.057972551074084, -3.4085259037061792, -2.815561664170687],
                          [-7.7648573102733325, -3.9886174419911367, -0.6107340097687931, 0.851278159122319, 1.9416266479746624, 1.797100224593785, -6.230551928875448, 6.755460478497137, -0.623002118625796, 2.640523337666294, -2.22192262693304, 4.282714519954316, 5.122887123269534, -2.9765687516604387, -0.6913427609231, -0.8919165624051173, -0.027940592086149327, -5.515011025570046, 5.586272810762366, 8.46139312973962, -4.22393741135074, -0.2331366568161425, -5.879518130259961, -4.731741900691931, 1.3398979010309098, -3.7642924432716036, -0.687710000659835, 12.127702789885397, -3.9789516092269355, 0.9176554490052975, 0.43587187859050164, -7.306824835077798, 3.6732037234119104, -3.4030965505660595, 8.709846988968565, 0.8633310999762357, 5.255571467013947, 0.9470106279107262, 4.692166697897378, -5.202403336309225, 0.48528953995282326, 0.5120679778701752, -1.4948052992167593, -5.386854776470432, -2.8166399279894603, 2.348168233273688, -6.271554715004595, 0.916088323121388, 2.090717132197386, -0.9814541607618463],
                          [-5.166695855475358, -0.4169799371911622, -8.431980091351804, -1.5359251339555937, -0.7060997930076678, 4.3411468586802435, 2.5693535095295426, -2.1741121756241477, -0.025601575798851493, 0.229826127217435, -3.0333713159000033, 1.7581813262472767, 3.279355619944728, 6.176542784737349, 3.967476239476274, 1.9930925350362685, -5.639338926234684, -3.2639128219396136, 2.425416905348931, -7.245514156681493, 7.550411186062073, 4.556787302399477, 1.2160168858636817, 3.6533847080841886, 7.064107959814116, 6.323564761635758, -4.659630031009278, 2.020750094116856, -3.0950243999330023, 0.0028615647045670788, -0.780742367872568, -0.6763081432892453, 6.9137124094557505, -7.729027091903827, -7.020219701465321, 11.280960267225364, -1.733936905109346, -0.8198306662862853, -5.535790229728325, -3.7694170090805503, 3.955451497862544, 0.7252371351734664, 12.112121062258026, -1.6292463152138104, -12.1246158869862, 1.5134431872318124, -0.6425477739459599, -0.445452271900995, -1.221829218234247, -3.8520664828652507],
                          [-0.3356107034515458, 7.070762482043146, 1.4674652802508992, -3.073003391918617, -1.7899452610931232, 3.7843178892477134, 7.5227842168339665, -3.598700513363549, -0.23960636280281908, -8.23072754784012, -1.4360818021181296, 0.3825492908381993, 3.1555565857999137, 5.313543977642622, 2.3174352584767632, -3.3923085197173277, 1.7168732819697403, 1.9680148487080624, 2.517820557216366, -1.7986745586928874, 10.280379572180225, -2.4684372910962145, -0.024687419393669696, 1.4975380011231458, 5.838898418169408, -1.12377953110247, -9.083965070920504, -5.015969594967138, 2.4071537595928914, -4.655822796402647, -3.6669377196913517, 0.8022807818769685, 1.956032076247054, -6.91789829832093, -4.770833652659844, 5.15672312713272, 6.089442317459058, 2.7980394027582123, 5.110687008511366, 3.2817455472998662, 2.8675389997881764, -3.5534898996502053, 3.0061364793976972, 0.023239826044004548, 3.834792216648338, 0.6073833139402914, -6.717092741167902, 2.641388370766523, 14.859522089716151, -1.3818286610858928],
                          [-2.4715190174895514, -0.3741491967729305, 5.192128787520017, 11.902324987797133, -1.975623030233649, 3.8802234767123687, -4.091101690747431, 4.741586955760147, -1.1816388064867063, -4.254820166495917, 0.18842639024118932, -2.1615241000693577, 3.140376514168083, -1.1378960048445477, 4.393788287800161, 3.8270455221371686, -1.3872753499230983, 6.6712419743230695, 0.4947029090090313, 11.25846658733468, -7.47331846020905, -0.5519197303061179, -0.6323292535074885, -2.1943458835630745, 1.5646354080719969, 2.718682339091797, 1.0985072863938565, 8.04994905205497, -3.2467377916294953, -2.003812785811227, -2.447302379013395, 3.581522991684621, -2.7361527621152018, -7.057474200802721, 8.156584641720125, -0.9157876443647931, 4.38501170431038, -2.6733941033412845, -0.7118621226729018, -7.076580371311905, 2.6203999193069722, 3.657052283260198, 3.1961418178421535, -3.050028911677487, 0.9945046237781756, -6.475589334286181, -2.888189622344789, -4.889122191815344, 10.517692480682424, 3.6544895486961035],
                          [0.2966641740690427, 2.972020146940701, -5.601747425992043, 4.703421085611974, 3.4620074421966605, 3.5171457983801537, -1.9168368402805152, 1.681193014828391, 4.319205986891581, -1.0797859518581636, -0.5381700478764406, -1.0473814671837853, 2.196216343147865, -0.13761400379487618, -3.257839014623869, 4.3506291168241455, -1.429208602362036, -8.940591002263865, 0.08439943472765958, -2.8940684720691636, 5.366669114599036, 0.8817725501648869, -0.005675247275083416, -1.949538929503743, 1.8064237267480887, -3.4297707079803463, -1.0160893452890365, -1.6731593068488397, 2.292924443517089, 2.2366880983195068, -1.9950526222917837, -3.8941073367627848, 1.5433138519138578, 4.219794635283504, -2.8077699305944703, -0.2585112812674179, -0.5656667537754381, 4.854000505538401, 0.03148720879996677, -0.13155036246132684, -3.3517562233580347, -10.092338530555633, -6.348889044206083, -0.390613742125233, -1.9485004835642112, 6.984319636923957, -2.299902812781629, 3.360208597170524, -1.562052080517028, -3.9491968415013727],
                          [2.604849475518966, 2.996419014300198, 6.7571854933163005, -0.5791636729125363, -5.083314989918003, 2.210484799293937, 0.4704157075504124, -3.1851851483527565, -0.7772209442809673, -4.935356036554414, 1.2263777645695213, -3.2808145377925495, -1.9961238317497771, 4.651102091818415, -2.660756345966619, 1.8688055512071997, 3.6941558977895466, -2.5805269534045476, -0.3994318969670085, -7.037872639318338, 3.71503563612875, 5.1561449210203465, -4.607371850104148, 2.4696447169066404, -3.941115754499477, 4.632772875669402, -3.744699284871324, -7.886755064230768, 5.163064128616017, 1.873427712768038, -3.167387430539451, -4.965045258467493, -5.464858716682307, -2.2776037495174153, -7.510206824332493, 6.130565504956207, 0.44183439158525706, -0.4466886302475638, 13.167555564095549, -1.6110353072594135, 2.9690065476329246, -3.1525897823014577, 9.051298436170116, 0.48407480310665285, -0.5424486984668193, 1.2888974804914455, -1.902150320154089, 4.586485591154747, 1.7225792820365322, -3.982493188051307],
                          [1.823167328597172, -7.000155627164851, 11.485846713961731, 5.157703997763007, -3.056847000912098, -12.857634279969027, 1.1507675097475594, -5.946108190024723, -3.418825493403208, 0.4042330473602586, 4.42675726829446, 2.986865240081597, -6.373299762691733, 1.6452847894751246, -4.426745308845719, -7.4033665499888865, 6.887289121409692, 3.3994482733247073, -4.031865898608008, 4.98584814151622, -5.794477646861708, 6.148156745227488, 1.6320192313311395, -3.1650980492201355, -6.461737370694111, 4.420829733370735, 6.373578955704776, -2.7479316152368085, -2.049901196388903, 0.4863043078424782, -1.3283266997575174, 4.7051895884164265, 2.6455191310692143, 7.874560549224978, 8.129486077502234, -2.0120994132679395, 7.776118258019894, 3.1297747186355043, 3.637112899535665, -4.203055081616009, 0.8866370828477309, -0.33405741942572004, 1.6585541300317976, 2.58458505153186, -2.514912807914422, -5.090842893243917, -2.915363102886176, 9.274154835289698, -2.1433748385601166, 2.703592446254095],
                          [0.7148992372566041, -2.1127479967924248, 2.819226186200852, 15.083412800243428, 5.858635638811004, -12.193307869141913, -2.2980760121405526, 0.8350690049113392, 13.307217935755807, 5.450910386895031, -5.6232547903206695, -1.4160964517954446, -1.0532352336436606, -0.45792992077246425, -1.5575426441628615, 1.1586986869065132, -9.821042438028712, 1.4155711333073322, -3.8610147568957203, 6.762235566315358, 4.452531812525799, -2.773383373093771, 3.590282395910417, 2.3942673732200257, 4.308519991514673, 0.831933669548809, -7.382428637507665, 5.487486106643706, 7.61014272572821, 9.6732360999421, -0.6255612741270518, 6.036624998757761, 0.44004844455150227, 3.2612946240293885, 0.8920215049334704, -1.3337392183961578, 3.488886292044135, -5.57599082198951, 4.551560150324263, 0.39469173233064625, -7.388287826591474, -1.6127223291856254, -5.105163865516342, 11.711372588296905, -1.4527042164639599, -1.1184455382710803, 8.270268538062371, 6.126825496482823, -3.834939129010669, 2.6592264504272167],
                          [-0.4631667452085943, -0.03893505898397499, 5.0926383662392345, -5.535435156652205, -1.6637932192880718, 2.052893831911966, -5.101681315459833, -3.2865504806885566, -1.3709507625760056, 4.945734323961552, 1.5966689520840194, 12.913033973754501, -4.749087050406579, 3.8821847932506928, -0.8777478003681706, -4.399677329989611, 4.520201418185038, -4.9031942705015945, -3.3830107711856834, -7.455341336651463, -5.590103593334205, 7.752767793470071, -0.09851605613593559, 4.048216931087332, 8.972422592407016, 6.090275508947961, -2.8386386213715555, -8.910665218498886, 0.989015630447135, 0.6499832840253119, 13.614123652336918, -5.734687589730214, 7.178019742949145, -10.337801894096472, 5.289032555499993, -2.040249444815198, 2.820267936006093, 3.8389248466026653, 7.139391126960829, 0.11834501838615374, 4.956042564832433, 4.4893226629868215, 4.19077504550896, -4.824322878217921, -4.147208188304087, 3.136689547227092, -9.860652236231967, -3.85500453857066, -2.561058103153544, -3.137845757832527],
                          [-5.283627383889158, -2.8815824446201304, -4.75525779025986, 0.2541289386164237, -5.315254684937005, -1.7732955853119183, -3.3832699892849956, 0.24565239991822888, 0.7457761402092832, 1.4254129194103693, -2.1002117738144994, -0.9616919940007992, 5.379027578958338, 2.248959142423047, -2.8360578756252077, -6.440043884612458, 0.6644282292492604, -5.235809374787982, 0.2932892856152399, -0.24523623765379485, 1.4275301439657597, -0.9467103615013388, 1.7431212265818727, 1.8356930849449866, 6.335122556553479, 0.5250094834024639, 0.055469285030803964, 4.640985695643711, 0.8122408981303887, -1.7079888440406188, 0.0347161537061787, 1.1109002416769145, 2.0538453507277583, 1.1298240660938865, -0.7424945146654961, 4.022752617964072, 1.158801747159414, 6.016620153516945, -9.402362746905348, -5.653329762691624, -4.060575731849419, 0.5309238164996919, -2.593307026193766, -5.459400357621605, -9.739434649053505, -0.005351353562476825, 6.520664610402798, 0.16299133627589169, -11.356343237597859, -4.9852545048266474],
                          [-11.864604689855875, 1.2931772291353982, 1.8402193139550544, -4.0420036107493145, -4.784681302511323, -3.411563813851747, 1.0361860511857974, -4.449407901220091, 0.5757515737792475, -8.281015668699938, 3.741836144215253, 9.494203920370188, 3.8523623603054937, -1.1894869422299352, 1.4969706842247945, 4.730834866484068, 0.5715405098379014, 6.886395860217168, 7.334866785751459, -11.10630701029329, -0.6662774956088793, 4.664797092176286, -1.6591584193504847, -8.360138544650505, 0.6731880839007108, -2.1103057569457193, -0.29684150640653134, -2.9474073860189267, -11.307584464248698, -3.5290929926976924, -1.8871154018317107, -0.014395350558063708, 5.141540132380333, -2.5803666163110925, -3.5595145924380014, 2.3892136920461464, 1.2323426962656616, -1.1293367538657562, -0.8395194788604922, 1.5687649791348799, -10.661161604438734, -8.009871610410707, -9.32323786583082, 3.16821696921555, -0.9671743846185062, -2.8043411756205976, -3.8517426574980598, 1.9754748283029675, 12.424173141005145, 9.609867301474749],
                          [-0.609145487283926, 3.920370570377799, 5.080009507777729, 2.67390957499106, 1.4726100965536593, 11.465176844260533, -9.622535907803519, -2.3900179874056766, 3.1914345348994457, 1.9593850652052527, 1.8856400531655928, 2.707993467827896, 0.7175264632278954, 5.235983894839305, -0.0608204340525528, 9.606320262132009, -5.972705483680759, -2.117178258861364, -1.369905668377933, -0.5072388563724199, -0.5710179217649324, 3.0991708649284835, -0.565217764436233, 0.053129043372111676, 0.4292479583799981, 5.087041038517961, 2.507807475379049, -8.891876599524606, 1.0985935053638622, 2.9140235225006403, 0.205707178295021, 0.38651931155674696, 0.7987602400688287, -3.420309061987196, 4.455122694391554, 3.3541720032232973, -2.5872391415994143, -3.7634664008684475, 7.371719351603081, -0.6429271586451295, 10.241206956976981, -1.2359996887085538, 4.151051386832631, -0.9419797750121672, 0.31100167548783997, 4.3136476411300455, -0.9857463770004277, -14.070926816813452, -6.252940800704606, -1.3363497104103585],
                          [4.624800658849354, -1.2524760020368615, -7.607406002544551, -3.696828236507681, 7.319220817921557, -3.4790223475317212, 2.015524763954684, 2.0109770930561854, 6.606087803535055, 3.428792249836022, -8.62016432185275, -1.6607298168228533, 0.05133511769944629, 0.46878596438881803, 5.948994232013246, -4.61001138723171, -0.9899204813675028, -0.9448356934205876, -2.6990540467456308, 9.455891427871933, 0.8068382536121658, -5.44524523759695, -2.3628600196975618, 10.451615342775106, 3.4968965886778793, -2.7495603977089553, 0.7216644829180932, 8.38953236021362, 9.28008822811596, 1.040064465099916, -0.09211449439873122, 9.326689730218412, -4.601057710257338, 0.7818771068610951, 2.4293097352377404, -0.03713151286166905, -1.513863101674014, 1.1435936714118213, -2.514296328054986, 7.053437023206941, 5.073411212780613, 8.85044600504158, 7.389688052310512, 3.6052820490442676, 1.5750021373981287, -1.574088984762509, 11.364142452818664, -1.2978070116701115, -1.3199166352026312, 3.5385938834896074],
                          [7.223291493432808, 5.249662727717799, 1.1690868011797662, -2.2570424549795765, -0.9377622382260503, 1.1410435340221745, 6.090137931558849, 1.3513911604453406, -5.276794664284258, 1.6400969368819516, 4.079979729132871, -9.107419343479028, -1.2362925665078228, -2.15888584382871, 1.5141297306234942, -4.143688453179083, -0.4203744735352457, -8.08167464776116, 1.0400159844335737, -4.2568787738202625, -8.346852720680939, 4.01035737566575, -8.914510518343235, 3.0335858381663106, -4.869074038723837, 8.12047337437504, 8.683557163295676, -9.181776689810423, -2.631331366893564, 4.2257572245871735, 7.132716936295488, -1.6654522241389318, -1.7686805018121088, -1.2045078168461507, -7.730627862337559, -1.597062824664289, -7.628711836278891, -0.8467145792865671, -1.0190526590363962, 3.6015207206278856, -0.5927571675758252, 1.2651963939317568, 12.243831858067077, -0.6710019659620572, -8.296782276305072, 2.2991724990553846, -4.388240533664004, 2.49890980876668, -17.497943493979708, -1.9748765615384682],
                          [3.2482035377926533, -0.11360715756968091, -1.8984048369512379, 15.771424739141601, -1.9433240718891318, 6.7137369308291825, -9.080026201528154, 1.2427786628590862, 4.13670479721046, -10.813704787528748, -9.440997087580083, 4.709415803693509, 6.764846676209681, 1.9438841676975795, -1.6371246734583782, -0.7413567151472653, -6.724450548267685, -5.206825992659921, -0.3690555146974183, 3.186970195179023, 9.639433602953982, -0.6584078974543475, 0.040418196140975256, -8.40374194556481, 3.7292160037932995, -3.273734903841818, -8.51474535036405, -1.0645944417862012, -3.6723742264320207, 2.248802786852708, -6.610378694861948, -5.23230156059568, 12.172854077272651, -0.8940565688309353, -4.3992500270224655, 1.30416295858556, 10.120035637873093, 0.5384021385134983, 0.6773942545576657, -1.5626914563724617, -9.121790940964058, -15.940553727030816, -19.383172551488634, 6.882270784546835, -6.260184693451385, 7.559219167223596, 0.18132310651882325, 4.452160154375648, 5.625121287592664, -5.412065506847717],
                          [4.521815152956495, -1.997626239724765, 2.288505211554657, 6.274776221538393, -2.0375932259009786, 2.3240718320982694, 1.4280380970982047, 1.094167516916545, -1.1936966576547732, 0.15299980628741502, -2.3714094633943184, -3.305805520769915, -0.9323048157634451, 5.042060534491876, -2.7876138570258937, -2.220702120170048, 4.333493786465977, 1.9403092840782226, -5.535747578155356, 6.070489206263932, 7.460384553819056, -3.864086615132058, 1.7046632306183704, 7.4729659877609285, -1.7871737800820866, -3.8887950945094913, -8.148414234843996, 3.796280482207626, -0.3593808066006916, 0.7722580089701175, -4.250451348035778, -0.17631708414996788, -2.227868216900043, -2.9863965024167642, 1.5825340967736146, -3.4970339880237886, 5.228879367212694, 4.20360366725904, 6.352170111828589, -2.418091482591064, 1.4388084911073515, 1.3041393066293936, -0.6126319861828604, 3.4190714876403665, 13.234967252039171, 1.6362229353310938, 2.0117773741629037, 2.4410743748650603, 6.07631674315405, -4.96765319671934],
                          [-5.006075473927838, 3.8910696112122323, -6.213580348038314, -5.043286249891092, 2.542587108808039, 2.3106009695968837, -4.788059466019111, -2.5081472661268895, 2.9492448139053717, 4.407961117673411, -1.4449455642902527, 1.7098228682941903, 2.426274757426432, -3.5068298951140933, -2.8480672455229143, -1.156414756160164, -1.2910585251787643, -2.871027927086192, 6.631954386074968, -7.577605860275853, 1.2810422924042646, -3.09163505904, -4.6834169470694285, 4.238553965051812, 8.809380914530212, -9.822900810860883, 1.4697269795885957, 0.5598277020313284, -3.3792589843769654, 1.3741830411820128, 12.995788828396325, -1.6836500489523063, 4.323540498472866, -2.628511137734258, -0.47270668610698363, -5.381673689702437, -2.3589705431286205, -1.7127784200082679, -3.099763227935753, 6.672970789247825, -4.521932905081756, 1.4423982895076934, -1.312653880810287, -0.984746477459216, 3.9746217089109956, -0.730035179024899, -5.33777781756063, 3.284195719485357, -1.9800747633150213, 3.1908812830548667],
                          [2.924563563446964, -0.6615404564545664, -2.6579254760369238, -7.540285641317331, -3.8999487845468064, 7.877266718301883, 5.415160585217243, 5.302067750915484, -6.083963574865066, -2.030456411063037, -2.2481528778092925, -2.84846460366086, -5.947933482981221, -6.802819170463317, 6.054228154955106, 0.26026630080810154, 3.0497587066230203, 4.208955468132993, -0.14468448854997834, 4.757807583334452, 0.41258810378904043, 1.3298790354625238, 2.299904809427065, 0.5629137027026857, -12.362773524800987, 1.9343543085093158, -0.7557969585025783, 4.238642285426375, -0.23422784917485506, -5.2223952719710915, 0.5025550796840096, -0.4176331604508269, -5.4529813380921315, -1.0249446838745901, -4.973360658243648, -1.2656690805203732, -6.657511086637963, -4.067144880523533, -0.8078368479243752, -1.5338509568089023, 4.26002273015782, 5.233693539566429, 13.322172693987124, -6.751527232380944, -0.7690612083618874, -0.10519818034962253, -1.6713589914411011, 3.5969339894882433, 3.742111239743085, -3.629964468761452],
                          [0.9113773642501531, -3.6210332517370016, 14.241630266536351, -6.926264140794485, -5.10471321935514, -1.7846386272103967, 0.7240346476040619, 6.961831161365289, -12.635271780228175, 4.891827148330499, 9.178470822491063, 3.2308648586104027, -8.260259747294075, -11.93920262850562, -1.0965073313317306, 1.9446710198374646, 0.5723661328250215, 0.5605656400734523, -5.775552193932657, -9.057178648333261, -3.1693047528076375, 4.037337810890823, 3.7202610335281685, -5.772107249462473, -5.749624390571425, 7.666280089271778, -2.786656058967296, -3.062734592276578, 2.012208815432985, -2.053520851866058, 7.02172400200585, -13.535790325351622, -6.174564206087252, 1.8310542479302925, 0.934033007644389, 3.560105578443347, -3.7642835567307604, 0.7810501180444476, 2.0075537193234183, -4.456113293952613, 3.520351884643091, 3.5270633984557067, 4.048194654023782, -8.555035324138384, 2.558766982314706, 5.921721378607101, -3.5900407521810274, -8.458983140138368, -1.8355560229854886, -3.5177948510632],
                          [0.6854795120335183, -2.2502047043382203, 1.7831275504998552, 1.9411558038793562, -1.0440876525015121, 3.552443286850047, -1.5982213829068863, -2.7260932888435803, -3.275880015394614, -4.764120330024997, 1.7923104617288688, 0.4746033989514234, -3.122079367356412, -5.8879626786565575, -0.2823475091377744, 0.05449566039261076, 7.52439877467375, 3.9221978300612674, 0.07851395323077122, 0.9819369580247101, -0.2897894697972315, -5.194944821372528, -1.1720220919006277, -1.6715306238821022, -12.917972005547458, -8.993176806184293, 4.49520563718794, 1.858662199186156, 2.0900750761897027, 6.263338264611445, -9.361248440637484, 1.7113808094457479, 0.7352317657035127, 7.53905482785725, 0.8452722491124968, 5.07025615627525, 2.0558549190727593, -1.5513559122863239, 5.458414767851083, -1.1278514003213678, 8.152296802584521, -0.8956448684520244, -0.8210502387074676, -0.21239148671962768, 4.835271059639872, 0.8199709405491493, 3.1926171359625988, -1.406477236126461, 7.050846621833501, 4.8882608769529545],
                          [-2.0470858357196002, 5.985359430961768, 7.864947959888306, 2.866671365905823, -1.1455139204546754, 7.781243407711388, -1.5134098276644659, 6.863745019409546, 1.7679186909836935, -4.325533779529048, 4.172716234065462, -5.9992565295736675, 3.753374056991287, -1.3756271110986649, -6.680405989386838, 2.366771514129348, 5.627125821641753, -2.6281524403773453, -4.714801084009349, -2.398571052275288, 4.305154529520176, 3.9388455363878228, 2.4401452993795565, 0.7012215319302735, 2.140791878072148, -1.3083384079893992, -6.012670902732997, -6.789543323611117, 7.042832779559848, -3.5814138811771463, -2.6745528167303143, -10.353236057017385, -6.450730316118697, -8.538843530726357, 0.49691989815334353, -4.637704456360085, 9.432168954807135, 4.113865353732008, 11.638457495582875, 1.659411015158106, 0.40499168077447395, -0.7001270144490425, 4.073739915432173, -4.666290006693564, 16.725469244254835, 4.158991134259826, -2.7053126486781087, -6.417059358893197, 7.357053438917173, -7.96042107834951],
                          [-3.904848272335747, -0.997554706150052, 2.316045336166676, 2.165143256118896, -4.702523777426779, 1.4080616982404106, -6.1132765438429475, 3.404680511396124, -0.9244697523735782, -0.6792265448614909, -5.172740592780766, 6.07773192260721, -9.208693330591057, 3.3315828367554863, 11.131483248308074, -6.68025108642636, 2.049786996576351, -5.23285550740413, -1.3309024963264615, -5.557423590374933, -7.654127605906107, -2.420969126034774, -4.148070576184457, 15.78699897876955, -1.3158149517992754, 9.438475786855733, 8.353895546381676, 1.1044711104871845, 5.521417387626931, 13.548170700023649, 14.82176516905205, -9.7533097975171, 2.740014580145752, -0.8974942985420497, 1.7621548289113351, 0.9334988669338351, -2.1583978621138757, 1.0693802548829192, 2.0228323267816886, -4.226508394192648, -1.596461096634323, 8.367646712463195, 3.3001333336157384, -4.223432909244873, -4.384762305657151, 7.351020919473545, 1.8139250561338154, -0.4468691234317062, -6.488416886643297, 2.822829373681529],
                          [-0.5437563621629338, 6.669389093400888, 5.869566992845138, 5.179873461263168, 5.952740229940506, 3.90173625725557, -0.6111394358628743, 3.690767444090001, 4.143198444693691, -0.2350012374708395, 6.082656865745282, -11.501158185333246, 1.402147679728306, -1.1526976730907434, -1.252506357352794, 11.7411811342163, -10.579997350257871, -6.472521659550002, -0.009868391516406394, -5.082513932460701, 4.092051286613483, 4.229357374836312, -1.4843329236657192, -3.8095155897199757, 3.523829400800777, 1.5362544657659443, -4.664479914607826, -9.740017621595317, 7.531418276850748, 4.601167575909021, 0.510907576060006, -0.11140180347287947, -0.9412526455770422, -0.0018377614684163927, -7.744088245511848, -1.1321012799039716, 0.33975750232331403, -5.69239372323967, 8.740875190196945, 5.8487383280699135, 0.4802519700309729, 0.8364430541587852, 11.569829516144816, 5.63603090788927, 1.2400290286083848, -0.14270298143000693, -1.4931441815354058, -7.508316193515885, -8.639828418481507, -3.985599261304754],
                          [-9.72291781762564, 1.2237304109344873, 6.148368342592813, 3.939058988632471, -4.145625539246775, -5.256404772720555, -6.6738814439588, 2.698296937695829, -0.4066488445696607, -5.075652194824366, 7.42344221158818, -0.4267633139422773, -0.28220900854520287, 4.985378828368066, -4.9328409110036615, 1.8659285219734565, -2.1950293765366014, 0.650153698946152, 3.3641320694535586, -5.670119325051978, -3.7486536837587865, 2.008212300590448, -6.806385315783814, -6.377172600119692, 6.620724696636512, 5.621411981557269, 0.7846906439363995, 1.5464900057469269, 1.502911219586237, -0.08919401934811493, -1.7349149820646252, -2.125938297819102, 1.364561269251386, -0.8916013565838941, 3.761554897913852, 3.584861672297175, 3.470928774246678, -1.6450903687955318, -0.5329289877178147, -3.877137371295684, -7.263127184424221, 0.8636667502817988, -3.824419441906762, -3.899336017724008, -2.8364142591114496, -6.781417269724397, -1.7525249594663577, -5.817664829840026, -2.9432796646677586, 3.0960411284463025],
                          [0.3805045200676939, -5.345562043781661, 0.5304024682936985, 5.7697357225548345, 5.052523568578367, 0.19165633815135438, -6.724608904079022, 5.626862418841577, 4.1882614785918815, 1.6002600103811742, -4.982440358440862, 4.405006767015237, -6.878332794317458, 1.71852321855718, 0.5298525832113472, -1.805932870664268, -2.4507232813548026, -10.173870886818232, -5.614142113022878, 0.6318135591126652, 4.65804281454644, -1.5310469728543996, 0.4449301748665034, 6.35552894171907, 4.312667956107846, 4.044576953955262, -0.00038464125773766, 3.691528448865336, 5.538054792492429, 4.127416292093811, 5.28009711816537, -7.771313210660411, 5.197804810959809, -0.9843544151482001, 1.9255710821344236, -2.7672785643822246, 4.657131332172481, 2.0292189455876155, -0.06668132747437584, 0.4518782544667682, 3.5823045899542505, 7.932430970860311, 5.346736149615849, 2.392189601099287, 0.8337009519332111, 5.570146499109889, 0.6033441087182615, -0.7493118618084055, 1.2079289274754368, -5.862543383940459],
                          [-1.588519685346939, 6.019334101751538, 0.05943524720424844, -4.336747947774172, 5.703167520140951, 11.496493813664879, -7.2232014127725535, -4.380028196710877, 2.220065227590723, 0.8541927990375752, 4.197350391722202, 7.1327431956937914, -0.6784050770763976, 7.168963639251453, -0.915032037135938, 10.265630202689072, -2.6419301200347234, -5.145078235826476, 4.920701421402401, -9.595383007466937, 1.225281516268294, 2.322202681686451, 1.0671953252994828, -3.9542860975042893, -1.5118921397147604, -0.10543192696646136, 6.739149884849524, -9.905550854807561, -2.513036953150417, -2.808100567110336, 9.252723368545489, -1.6941512287245895, 6.531638778689662, -3.81655591105161, 5.225560665800852, 2.4559789267451624, -3.28547314365578, -3.2329134447745633, 7.452718498748747, 0.9320565797164476, 12.095851696408237, -0.6661536264273874, 0.16160608304896412, -10.869892129182173, 4.920456445485888, 4.288386850873101, -3.5807483215187728, -18.399048415752503, 1.2479363940882788, -4.392677892800085],
                          [6.576980755374807, -3.817283120944696, 3.6144348261607253, 1.25455643102756, 0.7874168149434687, 2.56571703923686, -6.5728692430321916, -6.048548336079516, -0.5724505224868729, 0.3285094823280336, 1.4983128792702884, 11.594123608525646, -6.206472505449357, 5.162413383956713, -1.3732901818005883, -12.609669494575687, 2.731838485509834, -2.3478559361583677, -9.333101469165285, 0.8710603347217136, -2.849554396397379, 4.198250058679359, 2.249244264258431, 6.214245002409882, -0.23214910239485753, 0.005560459352722147, 0.6643590150501335, -1.9743352519942505, -4.379567946480713, -2.849940621030974, -3.59203928530271, -0.7388519538636487, 9.683845758932506, 0.8315470254450996, 2.3784012699751163, -1.1148602251591961, 13.420108909579122, 9.384772601475525, 2.017849338795863, -4.569416615469086, 2.4579962313736043, 3.79577195479515, 1.5273576982323585, 1.23680831591004, -4.755477734851177, 4.7510510362434655, 2.8675528505130896, -4.325489332025419, -3.1605542469573757, -2.5925006789101124],
                          [9.763948768847811, 5.399890726884157, 2.593545879264744, 0.050228925314554355, -3.5704079329211145, 2.5105983824106786, 5.06658380837357, -11.859278114366289, -6.001305178151099, 4.092967319822938, -1.030555157664345, 0.2992203417361152, -3.5985417258008243, 2.048597065406492, 4.642379471385132, 0.5465826416925581, 4.200839270205723, -1.6926356212579248, 1.3103883272353416, -10.461983576264512, -2.1537708054285405, -2.5601801047754678, -0.15639064382078688, 9.094385975924691, 0.9799103882175776, 6.5488574538713085, 3.0722999030760136, -24.39824943569817, 7.499992915621075, 9.884166470220247, 8.367654727469247, 7.793433805959974, 1.5227405811752073, -0.9925865970400235, -3.625084881617455, 2.7636643285163833, -8.901714332268426, -3.4256976753560293, -0.3281955113916797, 11.456029522853832, 13.064310615478087, 5.161243203301019, 7.37988379933762, 10.148480210180699, -4.138569251456429, 1.4189451741830217, -4.6542448210871665, 3.3554074355571086, -5.532492438584681, 5.934990929321272],
                          [1.256086643194157, 3.748095602495039, -2.3597838546770027, -1.2198874515935918, 6.832083045995509, 3.6728319500474833, -5.365765411611757, 4.052720279458622, 9.388993633847134, -7.842219266890498, -1.0147629080890554, -7.7027588097553705, 0.6969670864882398, 4.467401874666447, -0.12383150548770665, 5.701986290884865, 1.6117398685088868, -15.407472883280505, 3.094958294068748, -13.651248190397858, 7.572194091822549, -1.8101384319925056, -6.9435166283158605, 0.25838585494815974, -2.1633605422122106, -10.371603058474584, -0.3564826199148949, -9.641595380203825, 10.280203185378966, 2.4719064670615847, -7.437819337947104, -2.0654157176475225, 1.459368247190576, 3.560459665122085, -11.380357918840678, -1.0569040341228426, 0.2104297566815154, -1.1793326939799451, 8.883249623835821, 6.399355949367497, -4.033574242871201, -0.2511440953979337, 2.3619245680111423, 5.062117590677898, 0.857618693013085, 10.492943089231279, 11.878573928061357, -0.13247286585771947, -2.709191782088733, -1.8531959741784358],
                          [-5.628013814850081, 6.419933190758643, -0.35136759438281145, 7.587268207298724, -2.7011328318836587, 2.0525403772057835, -1.642128554708027, 2.196902814960854, 3.46735156628651, -6.649810218244905, 4.530309249770316, -14.04036803100767, 7.264192700782642, 2.151117367212941, -2.193703771566058, 3.0230355859883598, -2.231749769181103, 1.9151407766026578, 7.588728468106296, 1.4818966799918902, -0.6510052536610877, -1.498323494523489, -5.392852410676906, -9.729456534810199, 2.950831554135977, 1.0200568097679046, -7.153027538954618, 7.49126954219451, 3.081877422318111, -5.68607374525046, -13.680596615006635, 5.458992261307202, -8.175078967397862, -4.5391751516199985, -2.5075480098218303, 13.21615224742184, -0.6672821788488292, -7.606416932873506, 1.9716005334633557, -6.257032988973288, 0.9871199037845915, -0.43455458680818687, 6.34670872586778, -7.176417674220998, -7.536503669925861, -9.449988484064999, -3.5896272815322288, -1.2169515334565801, 1.9952126530026333, 0.11544739515188046],
                          [-5.469395830676059, 1.2156973866629002, 3.690924001381172, -2.1284987551514423, -2.038590975867799, 3.488786581904695, 3.1573557746189196, 2.2576668151494013, -2.4075731583422995, 1.14473676059797, 3.154178812811831, 1.8444041168750858, -0.5102791345084937, -0.9241086865366346, 3.2809830536487006, -0.8954704704197505, -0.10156190762986268, -4.712164445394172, 4.340563070300797, -5.927451484391119, 3.2200819590737764, -1.9586602184162627, -0.37951247789383175, 1.2196956069057814, -2.666593986688123, -0.017941404003013382, -1.267534718134236, 0.41706657231754585, 1.1431844653597631, -0.28513250194585416, 0.8695535943251562, -9.390484215754125, -0.5452355849435295, -0.7112885621890541, -1.6209518290447742, 7.585140945120939, 1.3099798793191837, 2.485569062293139, 3.5034586494512667, -8.32267110095633, -0.5676984362428678, -0.6095653317773287, 6.773551773503053, -10.185775204643706, -3.149433702782298, 7.301656058202745, 0.7296034462720584, -0.834095833397025, -5.623730344001863, -3.114410210045099],
                          [-6.028836376910797, 1.4041361112661825, -2.8918156556254138, -0.7696573220697478, -1.9413208463813305, 8.024264154820942, -5.019960724403169, -1.7331120734002465, 6.582262282595707, -11.03489861923301, -5.718233321493844, -2.4012631941320297, -0.08022012011724847, 3.606734685045539, -1.706405400853632, -1.9391515896099292, 0.7638979504800721, -7.12987019282421, 3.9320209123199263, -9.448715067704056, 8.337081613173835, -6.314976916495122, -5.337029829944319, 1.0739310541773825, 1.1182700682884594, -8.534405146429632, -4.080518113096706, 4.171457581148636, 6.164954686468538, 0.5168437461297662, -10.233609730527087, -2.6319073802948103, -0.6943423347699996, 0.28940269950678454, -8.375986520575022, 10.741783550875775, 2.794340649895594, 0.11444763998506403, 6.15591927340057, -4.756620168609091, -1.1200248975153018, -3.015342650657699, 0.4059129460405317, -4.9225802610799025, -3.9762174234851613, 6.081375986505937, 5.711175177294205, -1.2298026755462972, 4.801375020729377, -5.90389747474669],
                          [0.5755493459736083, 2.9920388580556545, -9.756116867464158, -4.119055823440516, 3.8445471822259187, 6.50478401936015, 1.7123968853122686, 6.994900153919273, 0.11796137920554614, 4.6772474786599165, -0.5471940619350727, -6.022705878062047, 7.249536144998662, -4.777342107429387, -2.3340682378966724, 10.897315692690242, -3.6833129256146435, 3.759847740556386, 4.06587607374866, 0.2890318027546456, 3.747992343801341, -4.618922888758259, 2.0480664604422367, -6.298640305677413, -1.3685839932796777, -6.589080036299509, 0.47542229228687616, 1.5480290144566127, -0.48825926286882976, -2.6882070864134677, -2.758578752798082, -3.347758302127655, -5.622919257653036, 1.7669434875395582, 2.2115234779658426, 2.6371829556482367, -2.270654681320067, -1.2216523744440202, -0.22959277072205322, 3.0091249940667333, -1.0553655512878621, -8.198402591117997, -7.995567010433049, -3.363482267267138, 9.590339809452482, -1.6633279403572538, -1.0412283635366795, -7.9086528224990404, 7.129455533999805, -0.837456609482065],
                          [-11.187530538769568, 0.630810190428023, -2.759974308606879, 3.7036770529267153, -5.883614778901836, -2.499057944031979, -5.640751361033074, -5.4361570016775005, 1.6334013031852888, -0.962370276939658, 1.4745606162929215, 3.035705654309438, -2.0100837088012367, 2.9466730138352766, 0.3632741684341759, -4.641080469173048, 3.057746517215692, -0.7426199690697959, 3.410357226285049, -10.39903702555496, 2.3836938843324744, -8.37976225803057, -2.857248016109013, 1.7403251870901058, -1.0062561128401437, -5.396482739362099, 2.25150939484525, 5.8738357681186075, 3.9957326504158863, 4.431915673549435, 1.6142605860516943, 2.409464527489214, 4.232770747758966, 6.881597154279073, 5.793632751146091, 9.509433003843975, 0.8473093964248938, 0.7979640750790963, 9.839753699541209, -2.849894852529039, -0.877208465531441, -0.6092555779378057, -11.412863731510758, -2.6881156658676932, 3.1929214846490566, -0.4231513091716755, 2.480362365907237, -5.083336511459714, 5.341372096034704, 9.298094540801738],
                          [-1.1922644977235153, 1.664943177267965, -8.055031004795968, 3.8863404177485052, 4.231497969382219, 6.6728212422053215, -4.0101552381793395, 2.8725927716487085, 5.804976890358814, 4.061648346706289, -2.242645468526189, 0.7689306033302883, 10.83568617985439, 0.6081688997139723, -4.193097304452987, -2.104479080360873, -3.501760356200079, -4.555027070187989, -0.3855772924777729, 10.673072900747322, 5.058095842079301, -4.658274280974422, 0.07216261928393437, -2.527432258082749, 5.533668081171907, -7.305935597634586, -3.412382578177124, 8.834718682078787, -4.606096672267631, -4.107566418298149, -9.4420608391916, 1.5820995419468473, -0.6848207767808343, -2.6692653232871404, 4.405914029701284, -1.1280201243051007, 1.7516415070616231, 4.2808215354346375, -5.673001626557937, -2.691631912511261, 1.795019749745806, -3.489162488851455, -3.817459076408926, 0.029202250098223376, 3.509321010444394, 6.253545337896118, 2.266226371087243, -0.4871645783124728, 1.4760307952025506, -2.7455338128198696],
                          [-1.9236930676805444, 0.5334727865912501, 2.642576424744253, 10.18373925173281, -3.1015755760221153, 3.6717862357760964, -0.6352045934877949, 0.40435575291637255, 3.9936016715896927, -13.07307043530768, -0.5791706645257125, -0.07870951259414803, 0.6539420564951574, -0.7005331158336581, 2.5871128632274423, -2.7047200384278294, 0.5835562612724838, 0.09740547674449665, -0.49985598115295604, 4.929562390898208, -4.30822825460183, -4.480771388814152, -5.4291159672731135, -6.1419288668499625, -4.508681651034717, -0.6803694437410875, 4.191241863650265, -0.2682791704202379, -3.1820724778687075, -1.123679405784985, -8.864069947263221, 4.440500540162377, 1.6089085173259314, 4.67166362675439, -6.385369590918131, -1.380328848112248, -1.715891342310521, -2.6690320245975765, -2.8297853586656214, -3.1377818016519, 1.4211068644766816, 0.08758558468820568, -0.9618366860852454, -0.8778244915336448, -4.9985851453463965, 1.320309768048014, 1.4888412512279494, 1.241748829847679, -0.8803753336999063, 4.1716045498325025],
                          [4.180131676266284, 2.363437805155407, 2.3399858095859893, -1.75333120205849, 2.017759704991109, -4.348264055828835, 3.4201917058537656, -9.058390521099339, 1.002351500005565, 15.233811801270729, 1.4088366250897322, 4.173515628024621, -11.586411070026838, 6.3409353673306255, 3.1339091026425394, -5.008189315379027, 1.8332748391652287, 6.377046155555555, -0.9061468979698286, -1.8456893308726998, -1.6186946873876835, 7.004833893084493, 6.3972304624453935, 16.69993814539928, 14.256929306696106, 7.671136364775898, -5.045065943348272, -8.518907842382301, 13.043568810082991, 4.536888013386354, 10.888449458163436, 9.692299792502995, 1.2667109970270194, -8.66123613562407, 4.706993635295026, -0.11947532857666382, 4.671683362500049, -1.1267970096951485, 7.905438784293059, 2.0108207299483687, 12.867478412992957, 13.919255423627142, 19.024072399674235, 3.9317713006015724, -1.3293143596875021, -13.751226117225041, -6.795895544669303, 1.6588519969148052, -5.500518103261737, 4.526800745126955],
                          [4.8226272946892665, 3.630520411815047, 5.105311339338726, -2.98219906478478, 6.432380718748004, -0.3865018515832935, 2.497087538255677, -2.0813140765654885, -1.7011705585334647, 0.49181371616529923, 2.5627033347020114, 2.0268677963792987, -0.4622145768506662, -7.721923812278549, -1.251280231308756, 6.716276893312436, -8.172454986612744, 4.864231259256777, 0.6286309674104807, 2.1047673864785104, 4.432796244033694, 9.04012473470419, 1.3214975272110767, -5.794680844166123, 4.427403733520335, 3.4532083684495904, -10.753778198933023, -4.328561930441647, -4.8756664351269725, 1.900209559665922, 5.695301342941311, -6.389827501512811, 1.6806163434316892, -6.393054551740946, -5.312620780730849, 0.3344884498137905, 0.008913492937367654, -9.030623671508087, -0.3350194668411707, 2.242348396351255, -4.385648739149432, -7.512130510640039, -5.2793628569447835, 4.455216985338258, -0.7551303327490257, 3.0295722150422826, -10.364432900818068, 5.057151515056781, 5.413861058875989, -0.8448901523146453]])

    def __init__(self, num_variables, phenome_preprocessor=None, **kwargs):
        self.is_deterministic = True
        self.do_maximize = False
        self.num_variables = num_variables
        self.min_bounds = [-32.0] * num_variables
        self.max_bounds = [32.0] * num_variables
        bounds = (self.min_bounds, self.max_bounds)
        preprocessor = BoundConstraintsChecker(bounds, phenome_preprocessor)
        for i in range(0, num_variables, 2):
            self.offsets[i] = -32.0
        self.matrices = {2: self.matrix2D, 10: self.matrix10D, 30: self.matrix30D, 50: self.matrix50D}
        TestProblem.__init__(self, self.objective_function,
                             phenome_preprocessor=preprocessor,
                             **kwargs)


    def objective_function(self, phenome):
        phenome = np.array([phene - offset for offset, phene in zip(self.offsets, phenome)])
        phenome = np.dot(phenome, self.matrices[self.num_variables])
        assert len(phenome) == self.num_variables
        obj_value = ackley(phenome) + self.bias
        return obj_value


    def get_optimal_solutions(self, max_number=None):
        return [Individual(self.offsets[:self.num_variables])]



class F9(TestProblem):
    """Shifted Rastrigin's function."""

    bias = -330.0

    offsets = [1.9005, -1.5644, -0.9788, -2.2536, 2.499, -3.2853, 0.9759,
               -3.6661, 0.0985, -3.2465, 3.806, -2.6834, -1.3701, 4.1821,
               2.4856, -4.2237, 3.3653, 2.1532, -3.0929, 4.3105, -2.9861,
               3.4936, -2.7289, -4.1266, -2.59, 1.3124, -1.799, -1.189, -0.1053,
               -3.1074, -3.9641, -4.3387, 3.0705, 3.3205, -3.8178, -1.498,
               -4.3807, 2.711, -3.7956, -2.3627, 4.0086, -1.3728, -4.4362,
               -2.9183, -2.2457, 0.3065, -0.8924, -3.2364, -1.2521, 2.7198,
               -3.9787, 3.0678, -4.24, -3.958, 3.9479, 0.2203, -2.6124, 2.6498,
               -2.3256, -1.5383, 3.476, 2.4462, 2.4575, 3.7409, -2.4887, 3.8555,
               -1.1426, 1.3389, 2.2323, 2.3137, -4.337, 3.926, 3.5905, -1.2858,
               -2.0113, 2.9087, 3.9278, 1.0812, -0.7461, 3.474, 2.3036, -3.3781,
               -0.4491, 0.9402, -3.0583, -4.2165, -2.3604, 0.8364, 1.4773,
               -2.7292, -1.5904, 0.7696, 0.5164, 2.6576, -0.5427, 1.0358,
               0.6926, -4.2775, -1.5911, -3.5775]

    def __init__(self, num_variables, phenome_preprocessor=None, **kwargs):
        self.is_deterministic = True
        self.do_maximize = False
        self.num_variables = num_variables
        self.min_bounds = [-5.0] * num_variables
        self.max_bounds = [5.0] * num_variables
        self.rastrigin_function = RastriginFunction()
        bounds = (self.min_bounds, self.max_bounds)
        preprocessor = BoundConstraintsChecker(bounds, phenome_preprocessor)
        TestProblem.__init__(self, self.objective_function,
                             phenome_preprocessor=preprocessor,
                             **kwargs)



    def objective_function(self, phenome):
        phenome = [phene - offset for offset, phene in zip(self.offsets, phenome)]
        assert len(phenome) == self.num_variables
        obj_value = self.rastrigin_function(phenome) + self.bias
        return obj_value


    def get_optimal_solutions(self, max_number=None):
        return [Individual(self.offsets[:self.num_variables])]



class F10(TestProblem):
    """Shifted rotated Rastrigin's function."""

    bias = -330.0

    offsets = F9.offsets

    matrix2D = np.array([[7.5503055799402896e-1, 1.7429941863385081],
                         [-1.1777173706375654, -6.9870721936486357e-2]])

    matrix10D = np.array([[-0.2589526173816388, -0.10175942956744435, -0.2467921802793384, 0.4233663827866598, 0.5558797701356326, -0.47324508893696443, 0.571002027664067, -0.23833191325746064, -0.4030177332510947, 1.0929187891467511],
                          [0.1727956598795052, -0.19981637292749274, -0.0109707074922486, -0.44498865221943396, 0.6667713291853621, -0.5710999692623439, -0.006927243578817618, -0.4404074152035126, 0.755934104360985, -0.24817003612007793],
                          [0.7150985095277524, 0.5709269013379215, -0.7943727204338437, -0.040574079212326905, -0.6170016526442229, -0.14442817339295375, 0.5550029362709729, 0.06129204781869642, 0.4233758670549419, -0.2927588827125809],
                          [0.3745987809361404, 0.30414535752990945, 0.6185412159588087, -0.18253713882553899, -0.0859517765845402, 0.6820591632005888, 0.4896598986892495, -0.2540924983568257, 0.20647249378240112, 0.45247309637109323],
                          [-0.19101408337939288, -0.14013677463582844, 0.762136466858636, -0.2639533407035719, -0.23210209939619258, -0.42312555008356156, 0.8025248946761377, 0.48181973827203645, -0.27594074125143486, -0.5713797199684697],
                          [-0.10592640806749581, 0.5351290496366163, -0.14380270301318246, -0.9172161678617142, 0.1845341278344554, -0.007557005713686518, -0.29109770529581835, -0.39219703336260764, -0.8207887702395809, -0.3870474670005376],
                          [0.47632734580863423, 0.428470330673616, -0.07978335716005022, 0.694017388540791, 0.7687920521548371, 0.08663876644798557, 0.1688976490783195, 0.16681970926268003, -0.5179511251437844, -0.1844689884612331],
                          [-0.5404347166798317, -0.5034831281373731, -0.3453511374197012, 0.2508474437008598, -0.17847360081965177, 0.20845804127948184, 0.37880938613374354, -0.8987902876823304, -0.05080961545822102, -0.5985297222778116],
                          [0.6299937952931762, -0.8809435292940254, -0.2654366721814134, -0.8511438032434496, 0.11409151085517524, 0.5882034236265717, 0.14751162215097532, 0.3686186805034061, -0.19226941922360685, -0.2584171192993318],
                          [-0.7370289025382166, 0.1398667329577477, -0.03902959296976058, -0.3870764135580965, 0.8173984016799755, 0.5266397733013177, 0.14884568436925888, 0.4470626637365287, 0.40258059679430486, 0.01783877880451383]])

    matrix30D = np.array([[0.06335258949852296, 0.05672305076252325, -0.02695070941427732, 0.5036772915089572, -0.17229433871272504, 0.18640203659987595, 0.007339703197867826, -0.3309891648916064, 0.04266518911929274, 0.07016071371497765, 0.0032679132554803312, -0.14972020339323092, 0.10373777561442425, -0.2659773961279128, -0.09578172975496194, 0.03160639180911339, 0.2379225189603566, 0.5918541116066094, 0.3364007304371468, 0.23392627475841743, -0.5839410727306553, 0.3540500180951431, 0.15298918062807598, -0.030256480178875365, -0.2539723815560839, 0.31572678634248547, -0.3515077666736317, -0.3048008899075772, -0.060246527470463704, 0.14784275286950338],
                          [-0.15303824114632544, 0.183585351002781, -0.14343088966796802, -0.030368486493833252, -0.24045786963278276, -0.03447197622003516, 0.07075250855636125, -0.33818894801085264, 0.574022354733769, -0.29999811349627287, -0.011816642073004084, -0.2028278864622005, 0.12148286246059095, 0.27628216666834293, -0.3630176651145949, -0.43656278888976596, 0.25449693389248257, 0.01866727951428067, -0.08797734177188053, -0.12612561251090862, 0.14350306337753108, 0.1647168697868917, -0.006177744203272412, -0.8468317153538225, -0.2765566614212845, -0.07228260858773611, -0.04007650347414116, 0.004383259418973454, 0.019246963531342157, -0.4171391203449824],
                          [-0.09461622991205118, -0.1685735084816444, -0.13820485006442146, 0.3339834665393658, -0.0458230892810814, -0.15126745263475783, 0.10470289793845276, 0.27633450351527145, 0.3134570150665945, -0.014216228422573023, -0.21149308945846634, 0.24789011794141738, 0.06168276281323435, -0.1011537070746861, 0.676510330993916, -0.2376149255236887, 0.2242221281147384, -0.10478403606959068, 0.30501541776493385, 0.40184550037919614, -0.031998325404569206, -0.08581137280623216, -0.2679153025970472, -0.11954616459090261, -0.040723833688222015, -0.4050394856509804, 0.015019639187367363, 0.13667079048804404, 0.28899443907879996, -0.2633435086169651],
                          [0.07973133030861902, -0.2902974686640666, -0.15711033378260056, 0.14805619823469293, -0.31403841896355017, 0.14830673685132398, -0.09641563354406664, 0.3911162579920498, -0.003990584756873014, 0.10173782181061546, -0.17007538709175166, -0.1592719793922661, 0.1883511036137417, 0.2972786497313627, -0.24984743261669734, 0.35776550824778963, -0.16545432898652054, 0.10990465300888526, -0.0996333887328015, 0.5365011152239183, 0.6679368922565107, 0.1607290132825583, 0.030180406715580595, -0.008371438932845567, -0.186049998692845, 0.17200654086355155, 0.5065133265488687, -0.01821378264611957, -0.5131022761715736, 0.0913534135133195],
                          [-0.3909786914772958, -0.4051631354570161, -0.29280344754806076, -0.03335001867839192, 0.16773031217578363, -0.2174663849548531, -0.028288010081692985, 0.6093403284631902, -0.12905128754630954, 0.015383241083884502, 0.2667713217812024, 0.07126257881898007, -0.08825633441407976, 0.20016851943143352, 0.21404662174010528, -0.28957335795260486, -0.22523641420841328, -0.012098621720922725, -0.4318595557910643, -0.10205590950971773, 0.1760178756410006, 0.8895991774260724, 0.29692315512112943, -0.23234762222452876, 0.26719466953478455, -0.018717538884440096, -0.3940794109323505, 0.07100412954210301, -0.5006836101778509, -0.07000845703250935],
                          [0.11640775059901108, -0.12411817286987235, 0.001670826828277877, -0.11655231072970242, -0.28712159807233895, 0.38324047856083815, -0.02317078434872634, -0.2324167393940222, 0.3942032485586294, 0.18017436277032778, 0.1154463424105523, -0.2813506977146114, -0.47636575565475026, -0.32774731322460937, -0.15241662374533727, -0.4760273994959571, -0.2608210668461727, -0.3428501171256769, 0.4188170865298606, 0.1359071771738531, 0.5000528739873007, 0.2248427993635855, 0.7584568413669648, 0.14197748119565942, 0.23124559527700234, 0.1881757454971244, 0.2215402738719409, -0.5377913031133401, 0.1614593901532991, 0.32648117861603754],
                          [0.2807602660555922, -0.20459814784556232, 0.5017108059447317, -0.09981461706697946, 0.371303634468467, -0.2847352808745651, 0.5037057270302671, -0.0310741881991538, 0.13922571090137803, 0.6209383878283083, 0.366021957072266, -0.18911637623628827, 0.03131100013513387, -0.2699425011341873, 0.08179040045016489, -0.3025613643955995, 0.30798895655272296, 0.057762327049527204, -0.025984640435035258, -0.08403212917549135, 0.004871402503692779, -0.13369198165259943, -0.22501015701437, 0.26722614828920804, -0.31452111240625147, 0.13111373390155406, 0.06364423359379096, -0.1361161576388387, -0.6162277604532115, -0.5223194390707014],
                          [0.21809792855351498, -0.015659253162891282, -0.03139629164532687, -0.29831526548691295, -0.6348013423829633, -0.08040903669775848, -0.05764498554378858, 0.19627871330339058, 0.18165656462608568, 0.11810278280167409, -0.0188863168144262, 0.33756700632674586, -0.3502494533027744, -0.42796866596963046, -0.2195897933184021, -0.1761629111633228, -0.14218717912062934, 0.22157310812540337, 0.06149377035521008, 0.20108102491589547, -0.08712108154447262, -0.2958898723889339, -0.25061272832655124, -0.02506364781253062, 0.18925836759833997, 0.05042484020763617, -0.14985557590469015, 0.4605369829616045, -0.1994223892735207, 0.3048349230240463],
                          [-0.04977155926838571, 0.09413283247490722, 0.2365075674600621, -0.033102298276168195, 0.14871067841765573, 0.16679259138580066, -0.7250131283541154, 0.19344718089168617, 0.16077444297823196, 0.4742932220190952, -0.457138015764315, 0.42216078662862944, -0.4217573313848444, -0.23850947864798056, -0.0836670106815892, 0.040603511075240524, 0.2161632396553658, -0.1376829372368324, -0.5626568078696313, 0.12858720436076043, -0.23135843593208363, -0.10064553874378594, 0.2975063249203477, -0.016698629430390778, 0.05183192762508454, 0.282538598642444, 0.029728283251851895, -0.18897748034632633, 0.057359157117103, -0.3345247359484555],
                          [-0.1392563327519083, 0.2220909895496909, -0.12012828708841183, -0.3161483737713023, 0.05118300596101006, 0.2754022567695218, -0.35925174441493535, -0.44297397481374373, 0.15011794783112334, 0.30199433865476166, -0.4704996606672661, 0.27273902679492884, 0.4228919491565102, 0.024057006386366618, 0.5458374155056723, -0.2454770930791788, 0.04600394398326692, 0.053530859107174464, -0.19432644915130357, -0.03881931406297027, 0.2529774513980233, 0.15403952082127018, -0.26799915887122333, -0.031286434673164984, -0.263206349961535, 0.1871480077284608, -0.24730717839944527, 0.20301608532527807, -0.28864922104713764, 0.6311149821166675],
                          [0.22074266351601457, -0.11266075532698061, -0.0722415820193467, -0.03592274427381795, 0.10159862684335982, 0.20799823630206574, 0.2198823861339132, -0.6066004552788643, -0.21393148333055445, -0.06265884287402183, 0.04600059272218315, 0.09278488783400352, -0.1125126425644487, 0.6481651539049006, 0.18531037045724286, -0.690760359023849, 0.17023209807096473, -0.1758342088164826, -0.14739114115086022, 0.021884252222916525, -0.5167467136681062, 0.08112659157833388, -0.18607182988440127, 0.03537864607385664, 0.052475296638563124, 0.5255169247366382, 0.4733049609096269, 0.028836894787041958, -0.05063412794434164, -0.07237978284709261],
                          [0.15227857012773682, 0.35890907545504747, 0.40311576678653227, -0.670794634047272, 0.08453881452617241, -0.11878067788422383, 0.1671073484415434, -0.09594153342794279, 0.17337352202551856, -0.17717436234025222, -0.05894552278360876, 0.11168668267598318, -0.06191622975281925, 0.5639565381375082, 0.41791622486727664, 0.07461322437330618, 0.14658483315164927, 0.324616404220787, 0.07552473311911745, 0.5198295716318212, 0.2303303496134756, -0.33161492836201656, 0.5117039074965118, 0.28547087590118064, 0.20073840807203575, -0.1472559207574819, -0.4625575469730994, -0.06032745209744485, -0.49896731024407665, -0.01971324233877178],
                          [0.3144352921471098, 0.09155170128060593, 0.24131250171541102, 0.3082364310107559, -0.015751811615472695, -0.10837694969979236, -0.33368256804806223, -0.09773665439564928, -0.23415834339896408, 0.19895365656634859, 0.11866077269974197, -0.7063793761419398, -0.2966729518045654, -0.03892519635968982, 0.4071865300175129, 0.10879528190943742, -0.09314848229345235, -0.2537308295168225, 0.17769258797233684, -0.011904351194733081, 0.014004256163295823, -0.080122743851896, 0.16405392512187764, -0.40277077092674984, 0.18563791255097423, -0.4159919240106178, 0.12529767835121203, 0.4274814162256622, -0.08678716160086684, 0.1357413038260568],
                          [0.6245639189801953, -0.021194385261232473, 0.15790651976032902, 0.020046793068165814, 0.013485287207486447, -0.19143050922350266, 0.053053619287120934, 0.1648025816684513, -0.02558107879906038, -0.5204849608828906, 0.27366106496207987, 0.37268447698550033, 0.5056389902510714, -0.5060716527836658, 0.17622109812935938, -0.16088276699725435, 0.09115420832801799, -0.3193721151009111, 0.1007622086850195, 0.17199096873026187, 0.11170952761795268, 0.15276479657982292, -0.07280382811968383, -0.3044514302215227, 0.34296871062700884, 0.33763415872847674, -0.03258919790417299, -0.3946276729967768, -0.13173293396858948, -0.23186690614308192],
                          [-0.33469228494985137, 0.20799931761035834, -0.0645235389515046, 0.4183383075968552, 0.06416955506782063, 0.1752731715255125, -0.11082704894603775, -0.4374198408666582, -0.39351853787706437, -0.28604100205000355, 0.5691018990593724, 0.10181603584935292, -0.45485482259097904, 0.06360362112520623, -0.10655964113618122, 0.01978208862729827, 0.07772065824803223, -0.029510223386463082, -0.12437913293723712, 0.2578126861040272, 0.5760449905641809, -0.2258142264336958, -0.40572768097048034, 0.157649041106536, -0.011341289128143667, 0.14851350034808614, -0.05552788406565953, -0.355072688281002, -0.22404728666046494, -0.12539346485283065],
                          [-0.19850895807711058, 0.4532639745313272, -0.10398033683866753, 0.03263380125003023, 0.09131401942401332, -0.5690381765576734, -0.2728648985674325, 0.40988738750653836, -0.0826698599947914, 0.1900962970736503, -0.09192692333289024, -0.08817260622682042, -0.2556802496385996, 0.25677859500848793, 0.12486970262940503, -0.23789170778049062, 0.24286433093315454, 0.26459619311082727, 0.22474512771605368, -0.02629623262854933, 0.006466976876191574, 0.24451032686765417, -0.21189052231900604, 0.002285070964737742, 0.3211765782645637, 0.2757981310844412, 0.2769877200223868, -0.5189171588724941, 0.1805248049110005, 0.23591694484557954],
                          [-0.31942635721679, -0.1815973160218126, -0.062138905606841405, -0.23095094676068476, -0.16355391317440895, 0.06274372372512839, -0.44562776381459746, 0.17784844982906634, 0.3624980631029926, -0.17975683881766333, 0.5555543164936071, -0.42018934556316945, 0.3933513494297034, -0.003876469010719, 0.4677439829364408, 0.11218926829095412, -0.2191750791691326, 0.31003503866519394, 0.012568335546410068, 0.06671256009424531, -0.1537048742007257, -0.7070505180084038, -0.21904334332760392, 0.14204759439403283, 0.21992215382554028, 0.1742363538768358, 0.20309131798586313, -0.33370216189057095, 0.0406825364530199, -0.3337944617495834],
                          [0.05287868724949856, -0.30608131674192995, 0.4045471731319519, 0.17044314492885823, -0.03628647352207799, -0.17340755399328106, 0.16075335121374462, 0.23993018676500177, 0.7538989376567938, -0.2758321242550274, -0.2919563906967567, 0.045478124446842676, -0.26024227411294654, 0.2930355566173637, 0.21107229602275898, 0.12362281077672166, -0.18897873891249867, -0.22110317637406918, -0.17747628664673934, -0.14851477409108343, -0.08124872456294821, 0.14961261809138654, -0.4781945922972805, 0.20600894669061368, 0.1998160608534948, 0.04570236137589734, -0.18436806734899275, -0.19619834388626295, -0.027134280656839656, 0.2167352344151729],
                          [-0.12098851840808815, 0.10045906754921452, 0.20471084225037173, -0.47266092521609576, -0.3382958675302724, -0.2948153380803361, 0.0006533732309131589, -0.07305947030990889, -0.41371927088380983, -0.31262153628757894, -0.2671883689737768, -0.3758017865369472, -0.04449075092351616, -0.2055455065566696, -0.03882714296576443, 0.19309451680858764, -0.269420481281664, -0.36344740384626795, 0.11646121573376635, 0.5594932561274332, -0.13855118758774485, 0.4816455983468774, -0.5654091116812393, 0.06813489514455655, -0.3042719372650152, 0.02615675459410987, -0.42093466523717066, 0.020777143271129575, 0.3257254219854145, -0.1533228280168162],
                          [-0.44930782843651795, 0.2447266797743762, 0.16737635815761187, -0.13430001529183805, -0.2745339036748697, 0.03947903414068177, -0.006373820025268782, -0.10505299872164929, -0.31079040895401294, -0.693513467167076, -0.5263914827840172, 0.007256974836705082, -0.09118557685987336, -0.43556940780065884, 0.1255004414086437, -0.25015968690685886, -0.13159571769917278, 0.21365691024452527, -0.3316308490718226, -0.4853067570023412, 0.17371913880855583, -0.1018403477721275, 0.2615530928599693, 0.16760356630961928, 0.0753858634221938, 0.01274344384846103, 0.5481716392540166, 0.416295471616993, -0.30820534457037735, -0.27904132047576524],
                          [0.05560324052577517, 0.37622417785064627, 0.17167152742138964, 0.38887755533328106, -0.5057137836073247, -0.15936721545761376, -0.056673554795662454, -0.2780198894532408, -0.12444596303043737, 0.4403390016264669, -0.0725275949272298, 0.3320846739161083, 0.5397465992622228, 0.5639719705526293, -0.3761532885509307, -0.17956544126243673, -0.21205781403078797, -0.06661455426643254, 0.23029648388321358, -0.10021971123949601, 0.011754999771520605, 0.18169814370133233, -0.1838657144098942, 0.5152012651450554, 0.3386999648397855, -0.33692492698433935, -0.159312800094762, 0.14537981887358953, -0.13639381352982102, -0.3949701191524413],
                          [-0.44998772224654765, 0.23838811137117805, -0.370224416055436, 0.152073074162235, -0.052672803829473036, -0.3783361561885239, -0.3215140778529922, 0.08906623257389529, 0.22208756809702468, -0.1731079380303258, 0.2656389517221004, -0.053273720087529494, 0.2018674832982632, -0.14449839225557942, -0.341758341589261, 0.30548865658921287, 0.5902168590120951, -0.6809519090119021, 0.1667522501576729, 0.18663305009546455, -0.22857089144903744, -0.08134277871682119, 0.43154350737026487, 0.3947700808589092, -0.06165851533681578, 0.28455104184414504, -0.1679119845437667, 0.27370661419920417, -0.365698226482665, -0.14846258804785598],
                          [-0.5093047682318179, 0.05514526879284861, 0.5573652541862978, 0.19116309158266145, -0.5549307057681351, 0.17263562082753675, 0.22195319353503512, 0.39231597399574086, -0.2627061543489674, 0.31531135515250497, 0.04683663340920785, 0.3174625459631497, 0.15587527714923358, 0.08491312910978416, 0.18147885992927765, -0.007952704227990853, -0.3377388592569701, -0.3527333644172903, 0.15223884149437325, -0.3803699002140566, -0.07683068819108986, -0.24148856814584913, 0.1063726411341036, -0.29896451116624967, -0.5469365920450053, 0.21206819416691797, -0.03630748598851463, -0.27154233102382935, -0.014664628992897813, 0.19382418530329498],
                          [0.11154886613263536, 0.6437149806999989, 0.11664996244848248, 0.3676809924091751, 0.29895796311783873, -0.06051388504408904, -0.012162057399845444, 0.3649781966606572, 0.06095497193092535, -0.17668123997779805, -0.21884683182836137, -0.08895448914261682, 0.10689520492332445, -0.15144390319388643, -0.36392522884452105, -0.5314576859632357, -0.6228583061558366, -0.15538277682004253, -0.30211357438769637, 0.4387267234803507, -0.33767207313498454, -0.2877550921672262, -0.12075393220669384, 0.22962491263881035, -0.25275962117475137, 0.13047689624678388, -0.008527935223278924, -0.10974660424916488, -0.07592440991435138, 0.2726091483131775],
                          [-0.27158074399957677, -0.34871063311664535, 0.0198012108654956, 0.06017236866260716, -0.016669316871492347, 0.09795650859553666, 0.3190251208859632, -0.036543774787308074, -0.3543901690267282, -0.156283210442647, -0.6094463957983466, -0.42925901468284816, 0.10292567605791395, 0.056079043347752426, -0.2171925114493125, -0.23356235871036382, 0.470258177144928, 0.15721644654023753, -0.03337048004528811, 0.2842382453893308, 0.11465876057709301, -0.2226097629053654, -0.10579486570029173, -0.0386911739168872, 0.8390292886798072, 0.2202690762214671, -0.37348346016359796, -0.3205073207436112, 0.3090520870949553, 0.10119829169347228],
                          [-0.2848376671325233, -0.3867652985650254, 0.28954064586952777, -0.10278504879082706, 0.5797936359249891, 0.6190299382678774, -0.3522683245500386, 0.21653752181435718, -0.21158684823484045, -0.23313749375995335, -0.019952272259973924, 0.11023535324932031, -0.1272288313234855, 0.10022619920907408, -0.28013781424922346, -0.10797535911704169, 0.25895903046357466, -0.03863229363855216, 0.3754011106268033, -0.0237847472355025, -0.17760365037457235, 0.11032452920669877, -0.3208676138855241, 0.2815400607111184, 0.0746602227885565, -0.26088730658023945, 0.2652751212490844, -0.09960258785335363, -0.5573809172298311, 0.17963455059211758],
                          [-0.47819406439647205, 0.4535402810748514, 0.021414164562559734, -0.2726232008994901, 0.13093585523562906, 0.32285996093030656, 0.3890794019536873, -0.02627496526152142, 0.26691099541019464, 0.13572777118037122, 0.039988310336496545, 0.058593329395910436, 0.16183477874626123, -0.1563044061236729, 0.16745559631137608, 0.38328718243274174, -0.4747636886775456, -0.04709468544943507, 0.12140831206618286, 0.24901239537734782, 0.04364697971315729, 0.19765303749138097, -0.10376854275295036, -0.3749844302317146, 0.42764442225019855, 0.6124177516044889, 0.19730113740988187, 0.2322955303254538, -0.07876537526371702, -0.33827090875675925],
                          [0.14725587741674584, 0.6225315611759047, 0.12702206769352406, -0.055254918241399345, -0.4617839929343705, 0.6690884728432455, 0.1878079771484699, 0.6052734499425834, 0.1993987643264288, 0.06756678885622325, 0.19079690602654734, -0.08872102668962367, 0.10752288316770534, -0.038803464307073085, -0.04771504332165819, -0.1971112461682327, 0.5972408430904755, 0.026290758226777032, -0.4165790241992454, -0.020965391894001477, 0.1345228083415431, 0.4299963860747021, -0.07089836187056582, 0.31470217812769646, 0.2689460613442491, -0.1685757783824816, 0.21330591105377716, -0.026906071922239864, 0.31643202123696357, 0.022194189195356026],
                          [0.35903925081536275, -0.07214806221911754, -0.1564836135387715, -0.024302181008234645, 0.3497926609272494, 0.52274171546098, -0.36769968731431857, 0.3839189282123319, -0.011020491967528034, -0.10393713144084128, -0.17755040448546563, -0.1381720442205483, -0.09278080257034885, -0.18251032954153196, 0.07033965947159586, -0.05716092128350151, 0.06261900184717739, 0.1562349132621388, 0.4773873664624826, -0.38393596527767937, 0.016053069972589793, 0.050648948855668306, -0.09876341932159212, 0.14423735208049157, -0.17159220321086766, 0.6623412204492125, -0.4250457604765382, 0.5838788233673984, 0.0904921288625179, -0.4032949259989454],
                          [-0.06839276629159381, -0.2598313462332782, 0.6167922709991274, 0.2562527715425454, 0.23655696114094105, 0.030866034264359594, -0.23933241275518116, 0.09868947711597731, 0.2817545408123683, -0.14920906302395515, 0.45726924806653785, 0.032197046147603534, 0.38316766942320973, 0.2889527397297365, -0.17165722981895515, -0.09162247668781096, 0.03899888505412967, 0.4637568330553473, 0.01369751444235539, 0.09268275159648072, 0.3184102981049872, 0.25510245540510856, 0.004474388575249632, 0.018541196836009775, -0.18698962512229167, 0.17459770017003354, 0.257186471540121, 0.5741096605396063, 0.5359206271361305, 0.2261381342792902]])

    matrix50D = np.array([[0.3373835192931274, 0.13906932070819356, 0.06800774490004415, 0.11998048305300532, 0.3621259592624327, -0.0436902934820657, 0.0318555841506871, 0.09989189649193427, 0.021799517544264107, -0.4456333799225406, -0.1430543384483614, -0.22536656033121194, 0.10448534415176626, -0.10493671204873271, -0.12860219812548052, -0.12244140882849876, -0.002199290360842906, 0.3161055009692792, -0.16689454853708396, -0.2290142900209269, -0.25026793696128585, 0.21652784795613994, -0.2713553889006185, 0.01072816552075803, 0.10589908325656514, -0.22721428283640355, 0.050956378238045275, 0.055747624516526734, 0.005091939490173785, 0.08110423799474288, 0.1084414302207522, 0.13914002791022398, -0.17956816049035496, 0.014241991464983108, -0.07701621674560376, 0.3296217236794687, 0.3216523020475158, -0.029053657793769356, -0.5584054680420424, 0.051067228767165164, 0.2480565143224723, 0.09781896961676104, -0.18406269719997984, -0.1039319754210096, 0.46065483591932765, -0.47350912548247154, 0.16570079408307986, -0.07304271628121582, -0.032739086079605476, -0.2170391898443492],
                          [0.007183580494949354, 0.1616794420251248, 0.01662606938519924, -0.12551868369401617, -0.2697518811455078, 0.1037540018068636, 0.10074365417326034, 0.14712389122499597, -0.08879605670191772, 0.1321334593568519, 0.18589822845738294, 0.27864294908191656, 0.2853077430824709, -0.08906878941650194, 0.1259743085952822, -0.09405521699102318, -0.08324508758731355, -0.044895630409162515, -0.3254776244831676, 0.17075321100746613, 0.08604323276362859, 0.19298046228498664, -0.11197995955904927, 0.25043050770653896, -0.5023970705459162, -0.16941369270304396, -0.015285705684176278, 0.4102972880295854, -0.5069956127355857, 0.13875769586237027, 0.016166981161205622, -0.09606873358370667, 0.02167514857924844, -0.5184082353661467, 0.061083720670806294, 0.1711798668830768, 0.0694487355185805, 0.4064081723459493, 0.09578059720160434, -0.2653218384660224, -0.3761434536167523, 0.1006014059922122, -0.48072902883185364, 0.11557343539851866, 0.25804910396375924, -0.005511768726759857, 0.32197843447492575, -0.1319195902757912, 0.020329872150803265, -0.09476458662080914],
                          [-0.0403007135073426, -0.19958092943886074, -0.055341397824141714, 0.2754756088105067, 0.5448092907684446, -0.017608229730915182, -0.1807824627890471, -0.08455549408644865, -0.6153807810418837, 0.4691214810327137, -0.10771931256739335, -0.23524149672074848, -0.10655701364854459, -0.12887782507788662, 0.19504320267659192, 0.03812890905012854, -0.03530598252156836, 0.35092370724717387, -0.15612262725121104, 0.0321868384326514, -0.1789589749509919, -0.06846108675806117, 0.2300838580851028, -0.09414859095320934, -0.055674073202425955, -0.1256936505680836, 0.02001965566910398, -0.30401103445590894, -0.3318512234676945, -0.14610868307295757, 0.272069582631002, -0.14113672689222542, 0.16480194423974723, -0.2883836276868385, -0.09157912403855223, -0.016070401621420077, -0.17121578127291018, 0.2096124415358873, 0.35478554803386353, 0.1430050936169271, -0.00245829951239453, 0.31316219616484886, 0.03822795711316402, 0.1808274425772522, -0.18911558224155733, -0.35139060878995626, -0.032228452698490014, 0.29587207600428567, -0.02110322467479086, -0.061387049473319126],
                          [-0.03172202020877081, -0.34575143456308355, -0.2649465197458069, -0.07118452568485359, 0.08939984761719431, -0.020320262383746138, -0.0369145674933185, 0.33300867532067857, 0.006065235776139329, -0.08443501427162153, 0.17840818141799608, -0.2961378484956935, 0.24525138060205873, 0.03639734843660774, -0.12249911196203908, -0.018173533361726585, 0.017961799587943045, 0.057923546788187494, 0.4724240048113291, 0.14033933201363363, 0.3784225595603269, -0.02410724911347519, -0.07132183768288858, -0.1254414186104404, 0.08772716205821167, -0.49161729270862736, -0.0729823444401647, 0.11534634167531348, -0.2582937150743161, -0.17789460018716083, -0.03284307414805702, 0.347396723235777, 0.1431961021914653, 0.26425581423101757, 0.03298070344011024, -0.1239339295935893, 0.2169307121421237, -0.010128297783724278, 0.30328515473758594, -0.08739886743495097, 0.10235577120027589, 0.1450925340028965, 0.05879940409114386, 0.2647802494094542, -0.11171227039287002, 0.08549359299404072, 0.28601530434938044, -0.25869095380212964, 0.18013511663068227, -0.5570771220580059],
                          [-0.1544967846846514, -0.12593829380180524, 0.19028393250453046, 0.051888250934198604, 0.26667291736560655, 0.041805946085775134, -0.34703720515115344, 0.09258907643567406, 0.20792353403134978, -0.20722787509846444, 0.25991479611068197, 0.10361502054387703, 0.06838761541558047, 0.038537958936266825, 0.1644106285352211, 0.058884636378640304, 0.23943878041337185, 0.10211759973684437, -0.037827215280554966, 0.19345226195104115, 0.0026329796681910426, -0.07297493768895193, -0.24025922638432756, -0.05427234910663771, -0.49683400540372485, 0.07419937493081834, 0.3989211675756301, 0.021627262649977635, -0.0832267427875985, -0.23581274331963595, -0.08311861796571776, -0.05327857329766859, 0.2504159565874328, 0.21895724105435116, -0.10930319328787813, -0.010747789973670844, -0.24541014530857053, 0.03272789983197835, -0.37743335071647527, 0.15368170470366188, -0.44731367340450223, 0.11831167911731696, 0.14427377118866674, -0.022463994823916145, 0.052920491712713014, -0.02814438472858065, 0.38180087585595507, 0.37368744230981643, -0.1317805675102409, 0.2592650888740298],
                          [-0.1609089356996852, 0.13874269616989107, 0.2057422528354205, 0.14994284355480747, -0.20313753821187447, 0.04116045442528293, -0.2711296095287403, 0.1716096121246699, 0.0639513738764305, -0.06327283585115616, 0.010166185308539612, 0.16765800121101357, -0.05764853746234467, -0.13764905635469832, -0.1744292082939623, 0.3590831172910128, -0.07915237361326372, 0.05610993817544515, 0.18881482252490767, 0.23340505547554724, -0.18827594961752947, -0.07482884331737745, -0.07207742495405789, 0.06811483202449206, -0.2553301135198392, -0.22322109966007825, 0.20842332903939403, -0.5523816601948118, -0.2543307837540263, 0.5093022568124725, 0.2669748469524698, 0.48684565847804434, 0.013056828313052165, 0.04106322058744743, -0.38560740908141106, -0.17825828450496123, -0.06479685998720713, -0.21766719101132254, 0.19227767316936764, -0.2047051829646626, -0.059590639240950885, -0.13180102761014315, 0.12453204879940362, 0.1170559700367221, 0.0025737388016710597, -0.3344985937569884, -0.4008494543171022, -0.09106795542114436, 0.3080538520290426, -0.06245531841385379],
                          [0.22010469345616757, -0.12452946439049815, 0.20037120602321828, -0.28763398955190733, 0.02596718872239332, 0.16873170420525335, 0.15251659071697546, -0.07487790749762582, -0.5787256094575088, -0.02658733765899331, 0.005803605267763742, -0.07394005701745851, 0.262886119118516, -0.042057473538564906, -0.01985843730468299, 0.03884264785290799, 0.38056784603767746, 0.02372904118503303, 0.24465443272896373, -0.049482647596728525, -0.0003979776749304863, 0.1375293643389871, -0.18668305731769574, 0.4004433920964403, -0.052391182038639024, -0.025741330934027203, -0.25471666680235683, -0.11472016512302913, 0.18276147603899748, 0.22299689892880678, -0.7025878328316443, 0.168935630698198, 0.18967894462405746, 0.019450343980259285, -0.1340327088584428, 0.12743310694600501, 0.24185510387279954, -0.14208097913637413, 0.10720764308062129, 0.252185734954862, -0.18544433787432815, 0.024628567329749412, -0.071283655654004, 0.22018132576030797, 0.21868705270583297, -0.014749181090520251, -0.2081939934735324, 0.29935910250093967, -0.0979709639923922, -0.03015441094759571],
                          [0.23896084612427484, 0.06511429659601359, -0.20763141257295817, -0.08969721752071041, 0.35971736739946825, -0.07159396077527257, 0.03802568873332908, -0.13195790005881813, 0.3524566018828827, -0.01189431922021627, -0.1750472072715964, -0.04418328966513527, 0.1529669140935806, 0.11792930950781154, -0.023363491045359484, 0.15417524432797083, -0.09802992595737337, -0.10207637240841864, -0.00531033900413852, 0.03935331967094708, 0.09045145702804598, -0.4118044614071613, -0.2080349653679518, -0.23751589921071536, -0.07432900631332293, -0.1534140632265737, -0.191002177590906, -0.06713732561560276, 0.06500763886572963, -0.20369029921232165, -0.05697651731810648, -0.20003594614404777, 0.3552690417977187, -0.38485505682696114, 0.10191951343998372, 0.4841268494086146, -0.42115543891701757, -0.527882939705085, -0.05508784592801775, -0.07463515684948306, -0.0513079080988089, -0.29367620329874466, -0.47919139579823883, 0.3470817402450935, -0.0840464768700495, -0.2287513609130381, 0.04969005707516154, -0.0560578527322851, -0.10519138574796581, 0.06106581673192077],
                          [0.1328352766333937, -0.10692229800091937, 0.27455553674682426, -0.17737045663398643, 0.08595854785828445, 0.022684120297932563, -0.27104324205009894, -0.1224784517512627, 0.22522133080663786, 0.18765935458267477, 0.2017817910287456, 0.15646314515291718, -0.310370249341148, -0.4559317563116451, -0.16501525130236888, 0.04481593620938962, -0.1349631485787034, -0.08551531559673026, -0.17034466802128306, -0.013084930341981531, 0.022032332528700557, 0.05811652790296997, -0.12847748896961728, -0.4079191822319009, 0.3307310848418589, 0.177124949795676, -0.23663405576477692, 0.1196531801762584, -0.1005771900753828, -0.2716379349630249, -0.2998453260068354, 0.14744030474198902, -0.41736986210626187, -0.02664488110149827, 0.3727344212414948, -0.05345493834487139, -0.0897563297211676, 0.2482043080022186, 0.02209941537264298, -0.3313691932019308, 0.05044426744520046, 0.08809087335474738, 0.36485208318342505, 0.15182148053760536, -0.030543014618174474, -0.22711200514221258, 0.03332035048969569, 0.20602046259926363, -0.12741659199997094, -0.19900057333676888],
                          [-0.08815328820597262, -0.1252918879978267, 0.2530828313337426, 0.02957280731151965, -0.08681965874835075, 0.20253321140197586, -0.01690824446928517, 0.17116878316664982, 0.11108553157819329, 0.3679976437186111, -0.11033546969557968, 0.127328290917389, 0.20406782284173053, 0.30633567041967813, -0.15942994188861434, -0.12777916757663468, -0.5785772873927331, -0.004395824856251729, 0.2500900418965425, -0.17765724671835803, -0.5839294698969034, -0.22716487619960524, 0.08190454514795147, 0.13254835532721754, 0.11585044657045762, -0.019887615546712742, -0.00516599077204043, -0.17190666647499578, 0.4140250804329272, -0.15926653208185676, -0.01863382537729448, 0.20356747556449886, -0.020467994667503958, -0.15777765937181287, 0.19844154446737267, 0.23480812871166581, -0.10648433900251586, 0.1198885949280369, 0.07197327454715627, 0.04212231886531407, -0.10928776031004592, 0.24235214711136732, 0.16631963889103737, -0.09226057487689464, 0.0666900350376677, -0.20402493673491923, 0.3136795015086501, 0.1449735370408585, 0.1883745265535729, 0.024015058275895895],
                          [-0.5432348567321881, 0.2678899518683061, -0.24576322996884561, -0.010792858439386144, -0.27787447634582935, -0.41436541708037533, -0.1902736732873071, 0.2304840562779109, 0.2994783275338204, 0.18993861111839158, -0.0863768980386414, 0.003939284781979441, 0.07710299487696363, -0.10942186851478743, 0.1273970709474518, 0.1194066759281144, 0.45886394637255973, 0.08934356745788584, 0.056005453723715314, -0.22300656029655544, 0.023074789583709867, -0.04698416856503934, -0.0340645196632256, 0.07515641098496058, -0.1477011426991658, 0.11796125145213317, -0.5183641525846864, -0.11947432446790121, -0.1674507933355812, -0.3379044758035879, -0.02313047301848511, 0.1116949132389792, -0.3696085590997422, -0.33657940747863746, 0.05848972901403073, -0.07129863986236079, -0.07444599789364383, -0.1619919422592056, -0.1503639865508262, 0.3536271662609215, -0.20950721704131953, 0.0999483538863784, -0.14811483022035354, -0.2284426796055682, -0.057618442190366555, 0.02192858992306645, -0.09073661353684895, -0.12238400369438128, -0.011758173157772263, -0.005011968166286795],
                          [-0.01887587277049011, 0.18305139423801897, 0.08529476025496299, -0.189567229162249, 0.07625520500284302, -0.1073638175733399, -0.052095787114188025, -0.1852975005282686, -0.1920707192195667, 0.21393856060763983, 0.299302013600471, -0.01080623375284457, 0.2338684985370396, -0.06960588980285207, 0.45330581590762054, 0.2571864510490401, -0.39368018694624335, -0.11694733443035371, -0.19912862775677678, -0.0005360640173190292, -0.056284957738679635, 0.15165213979886782, 0.46226222347559265, -0.27252135627777035, -0.03140067779817127, -0.17479331442625246, -0.2168255964567562, 0.1615617614844675, -0.15796787718556893, -0.05339359793627088, -0.2647144067827666, 0.041399725638138846, -0.08047993024210505, 0.5245081483762781, 0.0893962983351343, 0.00820716277759868, 0.3770058341928896, -0.049243457372258374, -0.4409807873100397, 0.19891674225988348, -0.09465824720207264, -0.2123043774609504, -0.2091205001777403, 0.10333007708883996, -0.15750755776108913, -0.3331010831344908, -0.0440519932020945, -0.1606371072228149, 0.10582800826460437, 0.014250826432573943],
                          [-0.10572423835433134, 0.3387636057589707, 0.20434250193755144, -0.22316048926600932, -0.03285682772163329, 0.07327920700516118, 0.28408707062998517, 0.029113671147464805, -0.09406593306017509, 0.26667931678696816, -0.17317906644734074, -0.009388970229233112, -0.24690558477726865, -0.13030955283655982, 0.4134438756609175, -0.13234835974797915, 0.17398002420641803, -0.017342194375531837, 0.24842742358301245, -0.35412916145564227, 0.23912079387896387, -0.01868241389487505, -0.26473892592809556, -0.12172248451150576, -0.2042369476117963, -0.22352543050247708, 0.23829536170232166, -0.14129975231674374, 0.2784573477012477, 0.045835779113055815, -0.3705720483371794, 0.01497725042162712, -0.3642159003974945, 0.19772970400719658, -0.008903788150132014, -0.13108425122655892, -0.011866065010853704, 0.20339804387676277, 0.08668769016623248, -0.04068684208486756, 0.016950832587909182, -0.10824273193132383, -0.4030154441539725, -0.14755608121105482, -0.25685937383207635, 0.17946415215177192, 0.399254564739529, 0.39484802205680064, 0.17139949513156688, -0.3168138920553271],
                          [0.22484647106987687, 0.0012813023948767118, 0.0528749253720917, -0.24361523322007253, -0.16164937421045456, -0.33475328606454513, 0.035761880842887894, 0.16664930270866407, 0.010249493012148494, 0.27682975883439115, -0.13534016223011183, -0.46675857927898523, -0.0029205596078775487, -0.15971059085466632, -0.09168598158361803, -0.3893402086155039, -0.31718017469526166, 0.15030192552934382, 0.41755321767522874, 0.29886641438650946, 0.00258908850051748, 0.2982974805498983, 0.23564726148582166, -0.4216817333873416, 0.012279889061681806, 0.02986655753735626, -0.011983156121187825, -0.20982976081639546, 0.16063499472722392, 0.23471651106225996, -0.05037332712549685, -0.1399409518504912, 0.05089516866416405, 0.005595308881454143, -0.117079676557033, 0.07649150654234932, -0.10397636996623968, 0.2157151966553524, -0.010337228554286584, -0.0013317446569580367, -0.440169873553966, 0.20495760922657008, -0.2849390913034384, -0.14535852222140871, 0.4985677618163062, 0.43031556166180424, -0.21157314835845195, -0.12625627623976227, -0.19868499184234076, 0.1951282118054817],
                          [-0.10213186306891642, 0.10325441325663184, -0.33122189348001263, -0.10868814586912902, 0.02161707291103359, -0.06775173761026823, 0.22000185645441622, 0.045206358236523206, 0.09707156705436579, 0.02641092717690095, 0.1562574851439919, -0.2825797672918798, -0.1130921797353524, -0.07687831394736185, -0.13304269878561562, -0.4047922709694816, 0.19441306498369573, -0.017094117796106516, 0.15954969434029026, 0.5720712100878927, -0.1562994828617427, 0.41453175949017873, 0.07488033484343823, 0.08477163768709649, -0.3385102726232517, -0.36155619562498664, -0.2703892287737755, 0.06492829981131625, -0.045766584396978435, -0.20421173021687106, -0.05571588131612078, -0.04588745553652718, 0.1233735630140673, -0.18941659108750697, -0.03566130693383948, 0.1123555600792966, 0.08160378687090798, -0.11364853680841544, -0.13310954026623126, -0.417943594131908, 0.3764890179657605, -0.1552381259397994, 0.18389103409162783, -0.3305955715821107, -0.19214701062142173, 0.06507937452551438, -0.09088801428172454, 0.32949378153993536, 0.16030960733433433, 0.17438863148545558],
                          [-0.14628876516403108, -0.13619464488671504, -0.10488131550163819, 0.10841926181767422, -0.14463152659657258, -0.3584640451701166, -0.36849650713730997, 0.11943376489509351, 0.04479661513194851, 0.14365946585486192, -0.22406787931998923, 0.3036597333326543, -0.15840341926021334, 0.2912483175524652, -0.2056491379910195, 0.33054237081502064, 0.2544566544339761, 0.25185315618958515, -0.14260603178192743, -0.03338044464828349, 0.10015319923866789, 0.28890426202422725, 0.42941868371312136, -0.36576727656606556, -0.09326397248928639, -0.0834580469135061, 0.23543199349051339, 0.23251922281682472, 0.3071522817127304, -0.05297812166757548, -0.10261814544016112, 0.17478442825659538, -0.06978755680517032, -0.21144464220895443, -0.32318349944451924, 0.12690603368414807, 0.16149498333819612, 0.1707348909927908, -0.020835714164276486, -0.048413350605976985, 0.11541668295414839, -0.32300972824776397, -0.1580358882018432, 0.09500410568656997, 0.2234747043706421, 0.09597649774653878, 0.15074461135402567, 0.3812439937796277, 0.13553259259672334, 0.004044004238349627],
                          [-0.2238250532802295, -0.09660933320678215, 0.43593481542373497, 0.24833043371887617, 0.19642664452597874, -0.2849747513415732, -0.15084048264348657, 0.3195736238407932, -0.1885141825635279, -0.20583997107391963, 0.1789960067804888, -0.009329533871344668, -0.31495761415018236, 0.26785635742210645, -0.15732256842882036, -0.47206370537644315, 0.14296516988336705, 0.011052187499876468, 0.21996590465888735, -0.07989594236676821, -0.12349109859014343, -0.19785258827107347, 0.021913333354489983, 0.04586701061077043, 0.2376886232230504, -0.12126095764031505, 0.03727027537167672, 0.31758915535293364, -0.08157346910098541, 0.022664879783453495, -0.13191157714742804, -0.14515763788895406, -0.32334905270186154, 0.05882683257615838, 0.1286862854817868, 0.4420864885705286, -0.09478139135698342, -0.3338747031808579, 0.08361756086710484, -0.23730683449588885, -0.10857360707407009, 0.1758439179519455, -0.3570878663413517, 0.012031635084034313, -0.2223065663119016, 0.04259525455194402, -0.4595574896734526, 0.05661393704709964, 0.12750002828441115, -0.08975717064861205],
                          [-0.1470131639787292, -0.29570378497678995, -0.08359442844999523, 0.30704254931036395, -0.10528421531434701, 0.5964295197311227, 0.022112807195352135, 0.058576350427541224, 0.16103738080511742, 0.1503940032181242, -0.06888878824872131, 0.3976716576109001, -0.16624999396483825, -0.24992813976581058, 0.053699587021824924, 0.3478185774950601, 0.37348414115927264, -0.41751907682892403, -0.12429341198576836, 0.1702411730226821, 0.05979047429219256, -0.013065129453446307, 0.24539940656557596, -0.20193212823065324, 0.3833608944239984, -0.07753352607430034, -0.07880118085510517, -0.3947458365659748, -0.10394433187295041, -0.13137801632362386, -0.12735189130213256, -0.10502031708140128, 0.2498239939643152, 0.13044554599398647, 0.0758380082500044, 0.382573352346948, 0.04233837228950271, 0.2514006166192301, 0.11136342446479729, 0.10555844932047127, 0.1133709949431384, 0.40526640936839975, -0.45518111674278905, -0.33263493651233655, 0.20621240668421628, 0.019613668131380244, -0.16204730681729135, 0.01092829119597995, 0.14184186880317284, 0.15261879584088262],
                          [-0.270563445777818, 0.33210133477608306, -0.17076946378461227, -0.07620817936507619, -0.08274699415877197, 0.43018576967814054, 0.2039181898061242, 0.0019830751694653198, 0.3224532937106059, -0.20176396644181332, -0.16521490173924114, -0.292802090566444, 0.21328998901136026, -0.16071009713394283, 0.1540897178329292, -0.14269091798422254, 0.1103519188222904, -0.003048883833015813, -0.22398373979614183, -0.08428167372685427, 0.06131530123785613, -0.22348883309392809, 0.3514647301483127, 0.0818150659199004, 0.1698717176285517, 0.039526155936056434, -0.025100725402871307, 0.18697179310818315, 0.24031130320478192, -0.0713420366332993, 0.008760598483205102, 0.40900411775135925, 0.16854136148085688, -0.014837583298906613, -0.3334542803369754, -0.032814024965909086, 0.03200247130138011, 0.16112184614892183, -0.28044889168607284, -0.5178952381084129, -0.10752468010977716, 0.3763767272533413, 0.07466362579491538, 0.4412750826173908, -0.2935747634628986, 0.07610637150013189, -0.18929692452503138, 0.10696388479061383, -0.22599555611471112, 0.037065922318926546],
                          [0.08656332541311491, -0.013849377997761769, 0.0024712546514067974, -0.3365706501792631, -0.08835466488053345, -0.11310772224776623, -0.00177199170587505, -0.3585484389634847, -0.0833027093533972, -0.007781772145417504, -0.07066239963337814, 0.29575929476700774, 0.1699644714932603, 0.15142976692500557, -0.06247904329411532, -0.041421532907136197, 0.2516759005980578, -0.2464187671180838, 0.18897744140283096, -0.21667623015389204, -0.09788367710915916, -0.039690638598137123, -0.022796899665719052, -0.29020004117364817, -0.10554625186254986, 0.10762783645753053, 0.14271767588096965, 0.0855736886487496, -0.03211970808575038, -0.285453089684684, 0.04641311972679777, -0.28285529840263945, 0.1619879001792918, 0.034959029112574404, -0.4048590230544273, 0.0676105679706042, 0.17353265171924223, -0.2768173834269658, 0.3458311176674309, -0.45930661238379683, -0.08521491223813876, 0.0732343634340353, 0.25051888990378157, -0.45010190023482405, -0.06428370543078211, -0.19207504393210617, 0.011201196421901335, -0.18149949556294165, 0.23331439569137014, -0.24576114878944116],
                          [-0.01596238100110713, -0.4211898436256823, -0.2040929998116716, 0.22597211088473854, -0.4480474529154241, -0.020800355939702572, -0.055294947726346654, 0.27766724681907, -0.3978448330066849, -0.24341824758261227, -0.1759091369231418, 0.2727659727627112, -0.184677192540601, -0.29505373621660935, 0.3544120762383758, -0.0946945615944141, -0.2919200092563876, -0.07492658198438197, -0.0012025713015780693, -0.14374118243089046, 0.14959710342495597, -0.02039437748112541, -0.11123898648368821, -0.04702116367381904, -0.3796487613184994, -0.1785808986869667, -0.4162611318246397, 0.037524671640931505, 0.08531831393025448, -0.26967783312248367, 0.14650847100047515, 0.07831765579521428, 0.034560720582811684, -0.12116488986263822, -0.15230142381531778, 0.04593159442931089, -0.35568064333538363, -0.17762598149145212, -0.32337846429050515, -0.1936832702442054, -0.16064961697250796, 0.012400332194922306, 0.08669478925207982, -0.03448833496930944, 0.12581153141489562, -0.12900807875809012, -0.32577703357124826, 0.16974010288081037, -0.15427327562430124, -0.10619846354524051],
                          [-0.33169889510021333, 0.3367691924782706, -0.0403989958186882, -0.2225915417745181, 0.47798420555193494, -0.0791526244074232, 0.07071920962957479, -0.026435066775034398, -0.1087526362883541, 0.018199918805095992, -0.04912290296527066, 0.08909411077487503, 0.37643014444387374, 0.07386777517242256, -0.21371514876530917, -0.16683786721768573, -0.28121753110889586, -0.3938177015644698, -0.042777439398621275, 0.014067348720091997, 0.5386064012960741, 0.013699336734007576, 0.10200920207507193, 0.01985477673146493, -0.1595840435975044, 0.08698858478680875, 0.061270211423418104, -0.1449465712890853, 0.3402742131074425, 0.11164242470373784, 0.22922955851113844, 0.04007905014093227, -0.017222129484205533, 0.0025580139231283407, 0.1840213732292401, 0.03122986588697151, -0.20111007828273963, 0.22769039067039037, 0.21332386295665023, -0.048683302323762175, -0.13167116347360736, 0.018157816800177393, 0.15073141375022325, -0.2546667712029045, 0.29764831829487437, -0.17604477064134963, -0.29438741507348876, 0.25057683491455346, 0.06485463383279616, -0.12923100913204413],
                          [-0.009243026111074068, 0.0960939110626546, -0.30895859918048063, -0.14236373169347494, 0.002069110496348312, 0.016409826159630846, 0.3241785367433988, -0.038078381017469784, 0.10941254258470513, -0.08108905509412676, -0.7403239325724089, 0.17197038831507566, -0.2194679816200533, 0.27038398638404837, 0.06897432673318053, -0.18988789597601288, 0.14088166479408815, 0.05795413474976548, -0.39516882365476425, 0.3703805157364161, -0.12282128131080729, -0.4581566528701881, -0.14987502303499484, -0.04947366440116554, -0.21049752957778045, -0.17754232282356236, -0.30287878984223005, -0.03288037137342596, -0.18269822725214546, 0.05940140651418966, -0.018970094335963583, -0.2552046468085726, -0.2599527673433252, 0.3048780721284509, -0.08111234246843288, 0.12013871120733363, 0.008946425443975035, 0.0853758919790102, 0.15429882146991009, 0.2990947315727599, -0.016998611253479463, 0.019990128006480378, 0.30679542199530696, 0.23563275517246807, 0.19081224164164587, 0.14096932181941116, 0.028210270527385492, 0.11230895270840995, 0.2716238492230242, -0.18464689030573114],
                          [-0.17417381204506535, -0.16759976294281448, -0.4597301513523743, 0.07841279120821025, 0.11788660006728045, 0.247607042085511, -0.11716185822521488, 0.05488944199384331, -0.02086095769524329, 0.2081081668726206, -0.0020660214637191433, -0.36870212327502006, -0.21599811964714705, -0.013530346192532388, 0.03179276572516423, -0.09151323613783782, -0.23262413726582695, -0.07271442515640723, -0.22881874325080948, 0.13161192171962657, 0.3312239739128443, -0.01596205861025398, -0.026712041811269398, 0.28782451068169945, -0.163056761915996, 0.32005257210783583, 0.4282049652692833, -0.24653800792429853, 0.026968354446332925, -0.2208152105877912, -0.10417036499570738, -0.13666486831449956, -0.4394453798485949, -0.11264560011773478, -0.08573422516107344, -0.10233914092372501, 0.42538583360214194, -0.49517359710283976, 0.03576773487519158, -0.04668134033335716, 0.0703002999060876, -0.1129095313935014, 0.07486081877650767, 0.20916965260131523, 0.42316549139176096, 0.10596459781764188, -0.14540952256710643, -0.0145195879207226, 0.17854502942850092, -0.1483170455402235],
                          [0.0476282842777639, 0.0374406087109284, 0.027578165435921516, 0.352759022830466, 0.3269694242052595, -0.15138912676133678, 0.0666524553317486, -0.36968719022935015, -0.06672130256854827, 0.3403737342306224, 0.25479891966534346, 0.19951026200760505, -0.3842656479343536, 0.316970577751769, -0.09932055760617765, -0.10343426896099281, 0.12717184040853144, -0.31514589033909507, 0.12948871587282046, -0.15365227000616477, 0.3128571674225463, -0.25947247628739456, 0.052209732773719014, 0.1548951966700978, 0.07024957615774803, -0.4285016619858548, -0.1505644737990139, -0.08240920428899834, -0.10586202830367777, 0.18734072212586972, 0.0015711233533860728, -0.01848310090789372, 0.13083451388941147, -0.2542854790073059, -0.05363038044200218, 0.00862908315641922, 0.2380882219512663, 0.13259781843642557, -0.40672491762435264, -0.06163921744285174, -0.2767849991835452, -0.22991519224817328, 0.28306678490399806, 0.16001206195833945, 0.05266516826846613, 0.2717028011622471, -0.06899414840251343, -0.28643913650637015, -0.18171309433038174, -0.14425145399181846],
                          [-0.20198280337669108, -0.1394504945393054, -0.1647835301460584, 0.12279562121028906, -0.1271097553343224, -0.06445925385252109, -0.35561015577321653, -0.5695758722429844, 0.11812351364719126, -0.33270361216021765, -0.1312724068330807, -0.06590364553300868, 0.07720870196911785, 0.3226431015181151, 0.194402824316611, -0.1416274866216198, -0.06317842881395068, 0.4816572991964367, -0.33706280032976504, -0.25785543487898976, 0.07944062026288239, 0.07922608230243844, 0.07463930031743156, 0.23552414846567357, -0.019803631195923627, -0.3563584959269095, -0.05317070440147903, 0.003477952513022871, -0.05999098289515479, 0.2677378552095067, -0.6411119657404839, -0.1294145963029442, -0.032074712559031306, -0.12826113612662587, 0.31698618759643066, 0.08888208732311584, -0.3266553223683465, 0.07730696677310865, 0.15120928052235905, -0.22061826637797782, -0.06462918440547924, 0.157951539376253, -0.005274301907654132, -0.33439697394993817, 0.028133701468489678, 0.016387054561571072, 0.11088761041865722, -0.21388409814660322, 0.16510501071945446, -0.012761004578999686],
                          [0.21520208136982186, -0.09547604732021087, 0.24934572064979596, -0.059514583062515704, 0.03117234974386937, -0.37661982700552366, 0.2781222742831714, -0.10142512622737673, 0.3885628024009834, -0.20049263651528304, 0.08085929316981748, -0.018204670969813014, -0.3094047044434138, 0.17834353878435316, 0.2538532053653912, 0.07220965517099773, -0.11019188000038682, -0.07902464093854543, 0.0015378138389771248, -0.06554812851109394, 0.019090944129371446, 0.41227802987814155, 0.6849369405203839, 0.41933343875493945, -0.11449921903478541, 0.16036107190708732, -0.04035981858279364, -0.3199783170936368, -0.2947487327821628, -0.4134060193858455, -0.16492469656579822, -0.3274335444476466, 0.22272520539582277, -0.03868185117904059, -0.012577177866276948, -0.0912763815541705, -0.051394668218643934, 0.048188459087253215, -0.04863526146585065, 0.012137273488365016, 0.08170620463938601, 0.034776092642723425, -0.02424874251418224, 0.37776243663776327, 0.2866230569671816, -0.06381648150496026, 0.17861626931480148, 0.05288626493026403, 0.16205530965816878, -0.26520468596359126],
                          [0.018159205156685856, -0.0878165360002932, 0.3760756894116944, -0.40265178657242323, 0.12847732127759723, 0.22604419541839815, 0.13440008184565877, 0.4463291232417801, -0.0711073388264633, 0.09156807855111151, 0.03809825857348431, 0.18334565177917853, 0.1938609397848839, 0.1535370846439177, 0.2331911867095848, 0.02780010059994424, 0.0892614843299114, 0.26491940944702463, 0.10894809565507665, -0.03554468801295899, 0.30200706655674553, 0.005646756558461646, -0.07958679751312615, -0.42300573267077496, 0.023792435554920194, -0.08709500630400488, -0.010722750872051328, -0.12469616723545299, 0.13351099623258947, -0.05905523537470848, -0.13668452730773822, -0.2941530543216247, -0.09444663779093815, -0.34399881530649623, 0.1801603889082819, 0.017432305533319037, 0.041960676223182856, 0.2590633140899204, -0.3135243244967841, -0.03838225420759657, 0.08818747398629528, -0.033659927545210806, 0.1601253249790722, 0.03412208040381973, -0.09225504717545392, 0.05609021277373629, -0.17534079329476887, -0.1671038906096667, 0.19152644587790632, 0.12739482382207182],
                          [0.3255580859791293, -0.5011426935190678, -0.28704294067998437, 0.189100265536481, 0.08553221378120954, 0.1619916832612629, -0.07135128161581847, -0.0837190247982049, 0.319214372880126, -0.40901224580220347, -0.10258424088709059, -0.11969229054139718, 0.4567998727449098, -0.22840837057355337, 0.2985556552133477, -0.14099846271465538, -0.08982606070557664, -0.29561320392035795, 0.2644755791589128, -0.26326086381998703, -0.29189808078840923, -0.05731908271402649, 0.10867466302274434, -0.08906054829276011, 0.005558476368991476, 0.08330775801020057, 0.3941355536050986, 0.048914518773229804, -0.20329964218005087, 0.0490349324077862, -0.2462469116949446, -0.09333934797375826, -0.30208296967215953, -0.06148199605674423, -0.0911166984278334, -0.2506328612094281, -0.016743716802387093, 0.3059842606042208, 0.11829060448959888, -0.05594306135324385, -0.21536788949873842, -0.5385837273541844, -0.20342083690588436, -0.03013865237052821, -0.18902886337853833, -0.10114536136034227, -0.3440645045089351, 0.042008726225119475, -0.1665317352511337, -0.083086415704041],
                          [0.34582844928051026, -0.029121580717510698, 0.05767190196066319, -0.1818725006068328, -0.19268065762142703, -0.23629341480179808, 0.1334440034865715, 0.3349734473934811, 0.008474121646166887, 0.008979972507851704, 0.00023127840796362265, 0.024495155008326067, -0.0455437617799319, -0.37285597512813906, -0.014954482071340673, -0.1966146642996955, 0.07966557693293838, -0.05149324152679365, -0.5110016565385257, 0.0239872695950347, 0.15840558997114415, -0.20600076999907405, 0.04764780748105878, 0.21447062356176977, -0.08670538359228484, -0.03789109975173006, 0.30175434210550645, -0.3260883956896474, 0.2030207503036565, -0.09992446752236726, -0.003744949461790911, 0.27832532877211713, 0.06994096676421552, 0.2506459480697396, 0.11335240051560054, 0.5238580242533627, -0.01077196307836652, -0.05773586774794211, 0.005532352940311512, -0.009805418507343308, 0.03533641207442817, -0.05207113291962138, 0.40278639377703845, -0.12420324522685422, -0.24470298011101313, -0.2668756222406815, -0.019033247017714177, -0.28705633253827767, -0.2921373734107698, 0.007351997915612702],
                          [-0.08253460201287152, -0.11875267675703549, 0.03513412073693635, 0.22375827427876974, -0.024157127567451326, -0.17435293457981133, 0.057977015683896925, 0.03046042730078951, 0.2613694418447279, 0.41109142684842365, -0.021953917606636827, 0.18474679566605456, 0.3133256134329718, -0.18747214990300334, -0.0631515220718061, -0.3997973766949416, -0.11922652736981412, 0.25167909532554694, -0.10421254237147543, -0.3243757505240211, 0.017164254901687773, -0.2314572002226125, -0.13238259426835566, -0.16649310790726068, -0.2552463306164483, 0.2642357422557613, -0.16808768632484794, 0.17329873893523215, -0.20261872464266673, 0.05798952751655652, -0.13206724587985452, -0.08791390292193642, 0.5160788800350763, 0.17575126755645717, -0.08489853080975221, -0.009851877024462762, -0.017726970651292633, -0.028168438321033426, 0.005433121266069018, 0.1243530749097603, 0.5021495374023683, 0.027017599959271606, 0.05739122912800958, -0.07294730701365962, 0.19234451746415832, 0.20771456386911663, -0.288183650764097, 0.13082085817326136, 0.09561901605009615, 0.022589952998531407],
                          [0.021151851894870198, 0.24388747628392757, -0.14019297823772608, 0.1469293954096281, -0.014380217442678239, 0.2593031315678992, 0.012618133125573505, -0.019634404884082836, 0.05518254513941241, -0.03773544391115278, -0.48666468666296386, 0.06416626851392912, -0.0010622650592205657, 0.053393986825277046, 0.30331368897337735, 0.15088757502997308, 0.10790688801993507, 0.2543959918705865, 0.3635400506720069, -0.10252402475489958, 0.05964866470042387, 0.19814827499540538, -0.0692256399351122, -0.16520929822728495, -0.025014538081910363, 0.2184545774556381, 0.16451908546330357, 0.25062582151856866, -0.4398514211449301, -0.016315647251599275, 0.10911331783979883, 0.4310081395989335, 0.22563938145529916, 0.28851471509698834, 0.5773633474884784, 0.2643450402063053, -0.009246100054140954, -0.06460548647351434, 0.09376543408303802, -0.016715155878796403, -0.2604883890689839, -0.40146920000204617, 0.16555182550245978, -0.02256320290381062, 0.057561274137780914, 0.0633497137001714, -0.1683158959071828, -0.06542025595142521, 0.18170266186399045, 0.058639222712588195],
                          [0.24404904609072747, 0.22566175512202144, -0.11718608051876185, 0.2518560245958363, -0.10040367262703621, -0.4211705540797976, 0.12043801461836995, -0.07604403276141175, -0.3169628045251734, -0.2602627455575044, 0.26992146144591217, 0.1553598617200372, 0.3241013155015261, -0.28914681946075715, 0.15475204082127544, -0.18342633831069022, 0.07593218417052428, 0.08373800980782672, -0.04154058905276486, 0.3816174375236509, -0.032549855761772495, -0.40957039730556194, -0.14329058713674783, -0.07313069386778658, 0.23917141293436714, -0.12098580051163532, 0.14713202104288575, -0.3401813602650634, 0.04419628875106282, -0.4815754628431176, -0.20007509073612068, 0.24156927369646747, -0.0910412125085159, 0.04822572655709093, 0.042571477560760615, -0.21297917856478316, -0.19730712337406808, 0.14965900479625832, -0.15067559556032928, -0.0747425427723379, 0.04349790130889443, -0.1060902543815553, -0.08034214613900528, 0.04976767503361211, 0.2990510967436094, 0.12909304196478227, -0.08296773507484434, 0.09120033558986719, 0.5064564036274083, 0.13281594216982087],
                          [0.058289796205032225, 0.16531132232192874, 0.025264398759267087, 0.26124024833560444, -0.3636378657397076, 0.08413708449946392, 0.173797772270191, 0.20520622456830184, -0.1288686099880333, -0.08554026716600266, 0.3787306880248658, -0.48836818423061723, 0.13154589532063538, -0.1695433087370102, -0.17366988228637606, -0.05076887786859082, 0.13489640440850198, -0.2441909783897505, 0.10860064179154323, -0.32894888537391836, 0.09421570949650504, -0.28320662719282275, 0.3468512697536997, -0.27055357574035527, -0.06801527247866553, -0.06191665102967714, 0.07060928311352, 0.06769840522208352, -0.3939812753507288, 0.22744031722603533, -0.06457471698883374, -0.10498313208122188, -0.3472603612102212, -0.33467023061239226, 0.004027878128086994, 0.11338183146671246, -0.11524237288746476, -0.2681799155004594, 0.2472107575107888, 0.14825605033169756, 0.10719307340212467, 0.06393038540678321, 0.3044809179719814, 0.06895302115085512, 0.2610987731183826, 0.12863435754405592, 0.4803608643584144, 0.10942335930408673, 0.12269682365920047, 0.31922404796385556],
                          [-0.19890680433477806, -0.12259951687192781, 0.1812463897731057, 0.15656734471084882, 0.2043133267328312, 0.14398114499339323, -0.16716539482952725, -0.0682565442041287, -0.07777263574454842, -0.21822811927780494, -0.17297751806450928, 0.045624634671963024, -0.10271149838847039, -0.2015525784416839, 0.08022390218853559, -0.4555092571349837, 0.29697651265785585, -0.35221921198105577, -0.2798284172290807, 0.17581895685294063, -0.16608061095430063, 0.060198351295399376, 0.3793519507188789, -0.22909211679865538, -0.3657921035669136, 0.20812753445326468, 0.016135331439640292, -0.2384991849275556, 0.42874310967668533, 0.29349417127666483, 0.10288068028819293, 0.3218472846738562, 0.0939511439595473, -0.16504196104720217, 0.30451233095044633, -0.03553774335307685, 0.009054293465056682, -0.18912853343050054, 0.014247416328666149, 0.02246268702580376, 0.1223972124632699, -0.06995259993638821, -0.26629608038199243, 0.27054610339030377, -0.0014989123868617777, 0.04921811084566881, 0.12633740063668814, -0.3254498548207688, 0.3588758031403193, -0.34802315489546876],
                          [0.03798623517325016, -0.19561352610199462, 0.174142774933956, -0.18384158235891537, -0.3596029727501584, -0.09858777358839746, -0.029986013438854063, -0.13685408565032334, -0.08703482842744645, -0.17768579612238783, -0.2165515844917978, -0.12583585241470963, -0.10863040454497805, 0.3199150361210887, -0.3379433973517997, -0.505391259550405, 0.07263620160784669, -0.1025997671084827, -0.3190587547987544, -0.19790317250997397, 0.23983862459845595, 0.1233410078004098, -0.16131434209292506, -0.291383817988337, 0.00325587029935491, 0.034743607733244336, -0.02273817818806336, -0.19840892339314342, -0.3255167980897431, -0.1606211359864307, 0.023422280583867056, 0.39740596309899684, 0.19753951507135334, -0.021352839907600927, 0.06422564017268202, -0.2221627075419876, 0.34014578410377955, 0.14500556170598544, -0.01107501139108876, 0.13420157206057165, 0.06625946205948807, -0.2855705037983518, -0.2922379837095188, -0.0702680395316427, -0.05742431416780536, -0.1833872952884116, 0.10449487405268182, -0.033143854353530414, -0.16932815981288266, 0.3931026344046785],
                          [0.10510062134489044, 0.2181633176364186, -0.058882749280792485, 0.1666460264998193, 0.1654004216475547, -0.06452866253600462, -0.3869158408605434, 0.017398413699662368, 0.1066461446603431, 0.0902368361354377, -0.07481986267966848, 0.2729931564885803, -0.3859488263493438, 0.05412373175329782, 0.24498609517672637, -0.25325693635313035, 0.04672075564221834, -0.2023900393801156, 0.30122345033339293, -0.11130953046655612, 0.2257700318271018, 0.16735796737546504, 0.007983745538268198, 0.2238201262261664, -0.2656747356387943, -0.13050282550001077, 0.09179062362366838, 0.23739229067954878, 0.23734727434427427, -0.15014396547101144, -0.15929637331247623, 0.06846271535570618, -0.15550954492553698, 0.3321590624561526, -0.12830387892286882, -0.01706042726745843, -0.0922931375899814, 0.18149304532174496, 0.23212038719245426, -0.1359854305388998, 0.16505303005393995, 0.31312449524710606, 0.0017768850028987634, 0.3288966158480489, 0.28809040238129124, -0.3343443474926623, -0.2481353401124901, -0.24068706547879798, -0.11353284988224877, 0.589265241438486],
                          [0.17674627021237174, -0.1339581027489568, 0.3945341191986686, -0.10601064472280798, 0.19921703864931972, -0.011573291711251628, -0.13513119214337133, -0.1413648490697865, 0.33836440720624006, -0.26381524034874987, 0.00742957337975482, -0.09228536838443926, -0.031996883028855574, -0.5428901614578945, -0.006825533881891291, 0.12099208798379577, -0.07024792735561022, 0.12069459083178853, -0.023183522211221136, 0.016601013416038594, 0.2490664466823686, -0.351685170967758, 0.08237596919616993, 0.27085226289163356, -0.06004650057977874, -0.3031308957333232, -0.2368219660343384, 0.29020439609590165, 0.23579033816109854, -0.19315014888438714, 0.17393025825709948, -0.10203380546428525, -0.010024670288915322, 0.01750538354025787, 0.09186930276540584, 0.031033847251318733, 0.5938698782782434, 0.05010692756476677, 0.40637619951222637, -0.0959070972234154, -0.02230591919580468, 0.08049446598304158, -0.019877904395624327, -0.061644122554989716, 0.17285370135769082, 0.35828754480199054, -0.00731606396039753, 0.03785178275451791, 0.4131379137389314, 0.24519467963807257],
                          [0.042597780573791516, 0.0709624550837246, -0.07094548347785135, 0.35426844652264233, 0.18011077401135253, -0.4467040352718849, -0.05977296253440255, 0.40074686229343226, -0.05307439476068388, 0.09201924514813813, -0.38130773216506075, 0.07474378403554852, 0.406887776657896, -0.29157137174318176, -0.06679614387851673, 0.12060569288036205, -0.11336103304804275, -0.2528693705945276, -0.1286023193241732, -0.25492151682403624, 0.03920024949526378, 0.27447035358385324, -0.21087972235860136, 0.09538927136799627, 0.05545830910589283, 0.014411447319863619, -0.16720334699222952, -0.26216453097265224, 0.018954191414825745, 0.01616268283880047, -0.21059838305817727, -0.25783481416644993, 0.3219156392260238, 0.18926200621090036, 0.31556572444761377, -0.10514183551560102, 0.24918796512777575, -0.18279913892755345, -0.05761322713495583, -0.34628660622513574, -0.22647256365136412, 0.20712743483631135, 0.1670930440467961, 0.023694059713099794, -0.1713841079202473, 0.3186783647380116, -0.010774957045007178, 0.05646855773185213, 0.15580625746719298, 0.28984253675196836],
                          [-0.24381139493572107, -0.18870722759099318, 0.1275096388923865, 0.12800500270170675, -0.28869177913917443, -0.1633365248827843, 0.11072913116150876, -0.06828469552430061, 0.39085692622439455, 0.14344633974841278, -0.16252930072469823, 0.1392971542893567, 0.07333573651171427, -0.13786558986405312, 0.4196067063952801, -0.13779120810831225, 0.16282854470842853, 0.019360257731351586, 0.17886645552837285, 0.4062978909021966, 0.2687213419482133, -0.08309479366011248, -0.11330425313887309, -0.02170239054863026, 0.4851055243352806, 0.05986410972574857, 0.1693748360000612, 0.19316929708076752, 0.13297541660661838, 0.14914830672675086, 0.160362257169137, -0.1946166803251007, 0.07439427135397561, -0.14091461158287077, 0.041463223903992305, -0.015031587228552787, 0.15735193571443643, -0.14187215197552022, 0.11162164943819938, -0.03702209880142698, -0.2593246186347032, -0.13550324689104234, 0.03159598025649857, 0.19054362799920024, 0.0659393486121595, -0.5258147281388471, 0.27639118860410855, 0.30894520233435585, 0.0807355151600734, 0.052813432983917954],
                          [0.07869078860494765, -0.058517622743293904, 0.4625396463564499, 0.40021530585265325, -0.2854874814586982, 0.0034453313947334576, -0.14125698481884016, -0.07139869728329966, -0.23862850901347804, 0.13525537344771382, -0.07749312365241874, -0.5402851292097992, 0.4574642341165849, 0.4590689941249678, -0.13652973916911862, 0.40089506974137246, 0.37382119805679404, -0.07200591038013138, -0.00752670057999917, 0.2642878177409044, 0.23982179202921752, -0.21545810969634188, -0.04031553980899694, 0.06720584732891284, -0.12870142899246984, 0.19028211887829177, -0.16744999710183156, 0.034802419663061156, 0.18180337987389417, -0.06595621640719992, 0.14723948358205013, -0.06065845176699502, -0.08242645228252335, 0.1406629777230778, 0.11394198022241256, 0.029756440441704737, -0.21440644832327907, 0.2678629321925485, -0.10819948574978999, -0.33467201804606334, 0.06186934759977562, -0.25852728232448136, -0.2264524394319831, -0.1090519871851207, 0.19879509894421116, 0.02772763530230974, -0.013619788891832598, -0.12861365569722422, 0.027641603831620773, -0.14542590694160573],
                          [0.16719064455255306, -0.15118812197111683, 0.09496011520608735, -0.10488459494843987, -0.007794269181680955, 0.19655440133142665, -0.18686632318663993, 0.19397176722539738, -0.10814630963494802, 0.19038030152940677, -0.32899592932952537, 0.21724940368875256, 0.07801188234270191, -0.21001186560656354, -0.24863905376692585, -0.2700273925636868, -0.120418699172484, 0.2744097398395678, 0.12917895161259227, -0.10823083053180477, 0.37120177797944875, -0.0741167350305068, 0.41403414824312246, 0.4067162258021796, 0.2021868778275822, 0.23513470887665575, -0.46867792728268826, -0.08538296581295236, -0.09948660290943354, 0.03701511512348484, 0.08846463340766661, -0.056862195590793266, -0.2898066216643363, 0.2386270031659796, -0.2942914089777672, -0.05360501189809313, -0.19009817507760315, -0.0536924518611515, -0.19646681885378403, -0.19468130416728543, -0.19909145103510073, -0.3306420965924028, -0.19916497965771401, -0.15817520990412098, -0.14342521261723437, -0.14727496121144945, 0.42544810634431235, -0.14814814232853818, 0.29773943396882957, -0.06422400853015445],
                          [0.05622486761376219, 0.2624744332691627, 0.1980129974293737, 0.33211962922156685, 0.0683413269231278, 0.5273937105219527, -0.2022328505290443, 0.1936440768784303, 0.12286308384687693, -0.0653978277109837, 0.22820095464265558, -0.06918798513215747, 0.07799090484843033, 0.0031118424168245923, 0.011938585606943799, -0.24983927678797252, -0.0002941998560829773, -0.018229241997202432, -0.3233750458974871, -0.15904674375284272, -0.0650276452052258, 0.4932918987120901, -0.3080144974808853, -0.13722632617202862, 0.19781803263307599, -0.3758431444388701, -0.3815716221986971, -0.06122480829675031, 0.1420118233258, -0.41158679979119717, 0.057256352502577, -0.19770308458599023, 0.13714651646743004, 0.08811728565962419, -0.3926276866488673, 0.053489475060886145, -0.15785760734015014, -0.19894282330886595, 0.4828534754414199, 0.332067124439549, -0.41081411364683684, -0.3308523121787276, 0.020593666306494, 0.14935694051923565, 0.0688093699492045, 0.026537349073199747, 0.009945381086385357, -0.15788835401222145, 0.0872649489574538, -0.06565187796700556],
                          [0.0982682485809841, -0.011091615932820598, -0.16278863413371053, 0.20906960311549191, 0.0876063170453712, -0.06686723093115556, -0.04947961128494197, -0.0015498064489273267, 0.05081930526766669, -0.22514517109682114, 0.17251467005878496, 0.38763428346525536, 0.1367402000349151, 0.23043458900650102, 0.3665193348166673, -0.14194898718552257, -0.1585002675966416, 0.24089133977366783, 0.19467973820509535, 0.04101353076328564, 0.20422740443384213, 0.11761805555567911, -0.03241030900991068, -0.2072430376633118, -0.19892566590762983, 0.3804037880115191, -0.23245014649832532, -0.32112363587029813, -0.05852060814729824, 0.13844113138296382, 0.261165837810352, 0.1041834585482338, -0.4773214664994714, -0.01049727150189202, -0.057646400439177126, 0.3427865636374169, 0.5015710431188097, -0.07726402609374348, 0.2569431821980865, 0.0023257958733098327, 0.068901614159041, 0.06298897922697116, -0.033302114415136704, -0.0867128875897747, -0.04877868054420474, 0.1481315456747613, 0.1601850868102127, 0.18108217271977423, -0.4734507264500594, 0.12157873370479301],
                          [0.09034614857204532, -0.11795645217978964, 0.39846009311216274, 0.22038372400994427, 0.36879882301112055, -0.18282355908697212, 0.20148577602885287, 0.25058650121741155, -0.011684299472810297, 0.04064443888092639, -0.23555100138466487, 0.010606276755596132, -0.08672725439594282, 0.08629012505893992, 0.2895801646758018, 0.07086594064258772, -0.02608095103619959, -0.1971310668930995, -0.3072610487999675, 0.23478571103438725, 0.14847246028001948, 0.00025954840214829564, -0.14518344349349735, 0.027695162781267327, 0.12615727945300656, 0.3630490104818813, -0.09012478203911971, 0.2668962291037598, -0.20880784926196952, 0.05773496539099687, -0.27277221123222145, 0.34712818708799936, 0.022052626147550415, -0.2546264797391342, -0.4014351023422581, -0.35533511557755215, -0.09521438995103836, -0.22738199376673957, 0.16347069363277703, 0.04321743826244473, 0.16579927734965869, 0.07394314321815144, 0.09163049298034913, -0.5822124872760737, 0.12945690260578024, -0.018337604887917063, 0.10407205171919276, -0.07351518442287651, -0.1240733327810088, 0.01091413062827025],
                          [-0.2134851600547287, -0.629360211488789, 0.052312556718824565, -0.4193561410120912, 0.13410462733398903, -0.0887601953057345, 0.015087702415577116, -0.28940510204401104, -0.12119789984022906, 0.2661935867785129, 0.03947508885034066, -0.06579010301225122, 0.23638891616766383, -0.10707593140594907, 0.30806744191249086, -0.051148063734140936, 0.09943788124932373, -0.16392942502362276, -0.2588205956193407, -0.1791956517789932, -0.20581644142714808, 0.17531241088227328, 0.060634223287570885, -0.06482125461576373, 0.22298367088305876, -0.29657514518882677, 0.03672558217897366, 0.01703476546106779, -0.1851825328013248, 0.0247461182312569, 0.30989387859627215, 0.1638541264743641, -0.38011073409296103, 0.19885224592781775, 0.08021817026008714, 0.1260959469000171, -0.2808986365251977, -0.14264434100172696, 0.05975602532071618, -0.15923254244394114, 0.0642509188828865, -0.10379337540859147, -0.03215904869960913, 0.20825425203039397, 0.2455865168551081, 0.3796824234339448, -0.026312935314828347, -0.0776080818367445, 0.004554781641177376, 0.23460226130715642],
                          [-0.18704124809635572, -0.150942236826911, -0.21413024069055087, 0.1937748797286983, 0.28245656417688003, -0.23313577669442054, 0.6871165431649776, 0.043655624212557254, 0.11129243110975527, -0.10284281465334887, 0.23326057939987913, 0.08735212192263292, -0.17697212808409288, -0.1216852751065959, -0.23466696609062418, 0.15414278027569547, 0.06744446166800469, 0.43951595075698197, -0.009472795547364396, -0.01070766667287554, -0.07741577080078559, -0.03870565006227407, -0.06902564288082028, -0.16769714720915394, -0.258698127928493, 0.011379929349126933, -0.21410729274139698, -0.05986523313743757, 0.1521628749650116, -0.15785845207774155, 0.08859560304526137, 0.20049447296353365, -0.1630866767573826, 0.1711100415868988, 0.20483767461395075, -0.14783820743685377, 0.09536632362078287, 0.035588288872376625, 0.21094490995690648, -0.2773440143207655, -0.3659791881745856, 0.06623775913988335, -0.11920675808315893, 0.06729408616160124, -0.005379091761450289, -0.25269144849062775, 0.07253385678791975, -0.23289752832901667, -0.0674241866505637, 0.16161114900495702],
                          [-0.10991149188970176, -0.1617470502034914, -0.31737922020216547, -0.21046559092441525, 0.0960116593522906, -0.10258987733641664, -0.04679053205832048, 0.19172015382059288, -0.16162803718997149, -0.1482913991428109, 0.419182565641926, -0.25765942209360243, -0.2785874976405963, 0.2442043017905999, 0.2784442357062204, 0.10283103448684579, 0.005529426079583008, -0.2775955645822767, -0.16084168174483815, -0.20319499663553792, 0.041226368999512306, -0.0019505600014220281, -0.18408082624135666, -0.02229331285954528, 0.18776257917454747, 0.3277644456014356, -0.32644736688401427, -0.1695491893632849, 0.07022926960788368, 0.15011808470969246, 0.0680284996268099, 0.06358279130396402, 0.29045445093046784, 0.00022101575739690309, -0.3051733831146184, 0.4385601445152102, -0.0015934693843859473, 0.5078482678603585, 0.14419932660617304, -0.34227575429716217, -0.05725076101623752, -0.1991507386658782, 0.13374746741841598, -0.12768987096014192, -0.07704348334339414, -0.08967483719460292, 0.17654320377613866, -0.08483189709942866, 0.31987447271258185, 0.13497253849423269],
                          [0.3694257084442933, -0.14382386720046228, -0.04894087372550821, 0.24918534500290684, 0.07246292079393336, -0.025552296229068957, 0.046380947386968595, -0.3634502153040025, 0.1993711318986433, 0.21312826323518713, 0.0635085612250678, -0.17750716446103415, 0.1360324329956153, 0.07840429446842015, -0.06712281039475501, 0.12282166065171997, -0.20910905044695646, 0.041012272926494814, -0.09226172938490158, 0.03412879889013179, 0.4926800169248435, 0.2986528596593524, -0.29173333628649006, -0.14449263724029127, -0.12923172284394374, -0.0588803521979157, 0.11813721853905966, 0.15076046531620274, 0.10699996563690162, -0.07351874269327739, 0.15682149350031338, 0.17744898135310497, -0.2833784874205939, -0.1645998946181832, -0.27029716432425493, 0.34584165614563817, -0.22311960694536978, -0.033254873370233515, -0.1805332900963469, 0.3382205248896027, 0.08970222557653015, 0.4107211601590128, -0.03377625025194068, -0.03993415750625275, -0.3912400910306088, -0.1678474920677066, -0.3008730398136921, -0.007174566932071807, 0.7068678511914565, 0.03958248181667388],
                          [0.5115308149000615, -0.002986900132311461, -0.31895276050061294, -0.15440214720804288, 0.09063718650660078, 0.01190070669391093, -0.05139961542875645, 0.2585920977598268, 0.24608778362827524, 0.16505760097640748, 0.305247918753234, -0.13691143660733518, -0.16913442263525702, 0.4112472464744706, 0.22101085424825356, -0.06288228916894237, 0.42667571293082124, 0.03226909441511776, -0.2933351144384945, -0.1901952786721909, 0.003794841368233937, -0.07361069935704853, 0.10405873890447208, -3.486836746364281e-05, 0.1301219465428949, -0.054861737948663, -0.11682845592655733, 0.0606909483795857, -0.09877480336056087, 0.11889315134013449, 0.3176323143524038, 0.2919906143617574, -0.010892499175846166, 0.029421451852527924, 0.3981315768109407, -0.44793738160395713, -0.0477051786112174, -0.14778145466810144, 0.002529757156148644, -0.09443768205803704, -0.2256458362053124, 0.23040937172954187, -0.057433627210036454, -0.0798803585342289, 0.20518269152079055, -0.14870123957140347, -0.16467154641793266, 0.35199667680895874, 0.15767694465606272, 0.023619925150926695]])

    def __init__(self, num_variables, phenome_preprocessor=None, **kwargs):
        self.is_deterministic = True
        self.do_maximize = False
        self.num_variables = num_variables
        self.min_bounds = [-5.0] * num_variables
        self.max_bounds = [5.0] * num_variables
        self.rastrigin_function = RastriginFunction()
        bounds = (self.min_bounds, self.max_bounds)
        preprocessor = BoundConstraintsChecker(bounds, phenome_preprocessor)
        self.matrices = {2: self.matrix2D, 10: self.matrix10D, 30: self.matrix30D, 50: self.matrix50D}
        TestProblem.__init__(self, self.objective_function,
                             phenome_preprocessor=preprocessor,
                             **kwargs)


    def objective_function(self, phenome):
        phenome = np.array([phene - offset for offset, phene in zip(self.offsets, phenome)])
        phenome = np.dot(phenome, self.matrices[self.num_variables])
        assert len(phenome) == self.num_variables
        obj_value = self.rastrigin_function(phenome) + self.bias
        return obj_value


    def get_optimal_solutions(self, max_number=None):
        return [Individual(self.offsets[:self.num_variables])]



class F11(TestProblem):
    """Shifted rotated Weierstrass function."""

    bias = 90.0

    offsets = [-0.1367, 0.1186, -0.0968, 0.0237, -0.2933, -0.0478, 0.3518,
               0.3579, -0.0586, -0.0375, 0.253, 0.3165, 0.3974, 0.3813, 0.1718,
               -0.2854, 0.0849, -0.18, -0.0942, -0.1684, -0.1939, -0.0154,
               0.198, -0.1252, 0.2211, -0.0536, -0.2085, 0.3883, -0.2305,
               0.1972, 0.1802, -0.2742, 0.3224, 0.2133, 0.116, -0.3053, 0.337,
               0.2232, -0.0619, 0.0743, -0.1337, -0.2071, 0.2378, -0.1349,
               -0.0009, -0.2858, 0.1892, 0.0775, -0.3299, -0.3864, 0.029,
               0.1797, -0.3591, 0.2299, 0.2227, -0.0747, 0.2258, -0.0291,
               -0.055, 0.344, 0.0219, -0.3578, 0.3695, -0.0036, 0.1446, 0.0133,
               -0.3803, 0.0483, -0.0406, 0.1314, -0.3729, -0.1127, -0.1502,
               -0.1911, -0.3004, -0.1845, -0.1738, 0.0771, -0.3186, 0.2763,
               -0.1197, -0.3663, 0.1649, 0.205, 0.0431, 0.0131, -0.1709, 0.3491,
               -0.1357, 0.1072, 0.3958, -0.2918, 0.1065, -0.3617, 0.0621,
               -0.0183, 0.0449, 0.3748, -0.3578, -0.1262]

    matrix2D = np.array([[4.4632867438003154e-1, 1.5894632776519158],
                         [-3.9310341311896009, -2.7966708535050393]])

    matrix10D = np.array([[-1.1110697888144878, 0.36905789544385453, 0.7292055292144803, -0.42847443920418715, -0.32558066659661755, -1.446565452628495, -0.020224085659255202, 1.5503062159856127, 0.9223087681212796, 1.1404581440230421],
                          [1.5092777909404804, -0.4599298468134994, -0.2455426051220394, -0.5551264613775078, -1.8651997436174714, -0.12195709189112197, -1.9679217757126555, -1.4248750959518428, -0.8155634563877943, 0.22223404831489502],
                          [-0.5640279361008805, 0.5604791845701329, 0.4027307660174637, 0.10593056356204295, -1.8718870129434997, -0.9936668670945775, 1.7677849646336035, 1.6106625259748026, -1.0123369759917022, 0.6077665913774026],
                          [-0.1098180630614653, 0.30970829860712834, -1.3245588099086854, 0.9859399464337323, 0.1355979273645493, 0.028150671230047226, 1.001596419139145, 0.08579900652808065, 0.014468896133512495, -1.2914400024570518],
                          [0.8770280237386544, -0.5995510785229491, 0.034166677827547016, 0.4544431333317236, 0.6141138072484364, 0.08713506884733818, -0.37193688015275456, 0.30270728793727675, -0.039512527976833356, -0.44801215556282553],
                          [0.6888221374558707, 1.1470043442495683, 0.8632079673083123, -0.47180326822629304, 1.7206295945742422, -0.9192731070464375, -1.2344873056018493, 0.5867874862240999, 0.7112782132904119, 1.0565811136727492],
                          [0.33995500494567416, 0.4419526642630929, 0.682467208794489, 0.1671188992095813, -2.3623524924091583, 0.0839137754815431, 0.09556465834393763, 0.012213386723430246, -0.23738972474670303, -0.2753469794049502],
                          [1.190079821855058, -0.667054552031223, 0.8711175453115998, -0.7985618910202718, 0.6944240170910183, -0.7035731837431797, -0.8460840934908737, -1.0734520052438552, -0.6888710861789611, -0.7235422746201137],
                          [-0.15363247693593787, -0.04619078947565181, 1.3335277878980332, -0.10031163853521424, 1.2209115415033431, -1.2275314484176902, 0.057371210992301205, -0.11377154788313812, 0.726923760585939, 1.9524121519232884],
                          [1.357557811598952, -0.04304128471937102, 0.7151477814755193, -1.4644509947508315, -2.023762074738498, -0.8382154166978522, -0.14272030990811013, -0.15849819639360752, 0.15425476649857556, 1.2515451419349455]])

    matrix30D = np.array([[-0.24415797617133117, -0.3629778598966794, -0.025771631863859217, 0.8229781816198142, -1.0016940502240026, 0.4296448224812279, 1.2473170610679825, -0.9378241839672766, 0.5806047982406501, 0.4115684796126129, 0.37378270762270127, 0.014477980320051913, 0.24111989003425846, 0.14208416160447723, 0.6431648703781747, -1.1960480734219814, -0.09541787405348977, -0.8826430338513224, 0.06157200009026991, 0.6518593178226029, 0.7356482715010267, -0.6524663204047787, -0.7148187616183787, -0.0856485125356522, 0.2175717453141174, -0.17759969600204073, 0.49023742257745073, -1.1831424839930096, -0.4984723253132853, 0.08744869210767466],
                          [0.0696863657945216, -0.5060813445994665, -0.02640078047539199, -0.4650825928424804, -0.011660254464842694, 0.30319144433961687, -0.01832001593989361, -0.1919892233594309, 0.2549400776258536, -0.1838723446922793, -0.2207368747600526, -0.1471054543307868, 0.3790018422866155, 0.030801483548504927, 0.32612886899231947, 0.26917928891695997, 0.030286949485795633, 0.6111944500481038, -0.23125092522758592, 0.756828249889362, 0.12289225796269292, -0.2085936114532866, -0.30575995026671166, -0.6076561993016952, 0.020547539028165016, -0.5948950904386973, -0.6989480504774146, 0.642317402028593, -0.658582668007117, -1.9310950228018156],
                          [0.6182210089859005, 0.08469883581782425, -0.715202750657864, -0.10689864320172354, 0.8815208048157552, -0.6723141944035139, 0.7025831889123358, -0.06336560769413899, -0.6505734238934142, -0.7964669005257073, -0.8748124658096694, -0.06356032275692644, -0.6081389795514863, 0.1349529532597709, 0.7214354566989988, -0.2404961135873736, 0.3903573578567388, -0.14117460619677125, 0.03135977518895493, -0.3465466148174292, 0.7252225612181439, -0.4676549382078047, 0.8072019777726889, 0.10333733388275301, -0.12363244332318712, 0.3693626729122148, 0.30991477446959703, 0.6825302395308693, -0.2316375646385486, 0.851419827678896],
                          [-0.1422838783275414, -0.3257611522236677, 0.3234042190680949, -0.40373848491187536, -0.07012446435116082, -0.4265850351231416, -0.06950293016609707, 1.0906290705010628, 0.23524253461138142, -0.8654284266933125, 0.21280070313115437, 0.1491382594227291, -0.03776545746276128, -0.8863933276646458, 0.5951933305720567, 0.24912543166106357, -0.412791413712175, 0.3214314670028321, -0.8335840905602045, 0.08044128467646398, 0.6412983804127074, 0.4608569967948416, -0.01970430080415135, 0.41110938370644307, 0.09111542346756432, -0.021878890404958315, -0.33055924958789107, -0.2536892077543866, 0.5609010686904733, -0.5673424974952329],
                          [-0.09454067257351423, 0.3349336031145085, 0.12510790781408673, 0.33934822087672445, -0.4303393713754486, -0.8500612921392777, -0.025830986147262775, -0.40261099282305074, 1.3795092249145322, 0.5830275612023639, 1.1723017382685115, -0.016084363194653832, -0.0010036934128044225, -0.45743031433085934, -0.028457788133284423, -0.04277097041381193, 0.6047959449548794, 1.0427550925924531, 0.08522772251130126, 1.026658128702627, -0.08823109167864493, 0.24176307315030804, -0.7529916756307705, 0.11613938699010794, -0.014876858620700373, -0.006399828109041203, -0.2019787407243364, -0.18088576354318964, 0.08089254517733155, -0.469265558941737],
                          [-0.02706813178554679, 0.010367219958836382, -0.25176931579503253, -0.25864110296971743, 0.02301718133034092, -0.43348529635746236, -0.006277106501408065, 0.13444457815819574, -0.5876226746740735, 1.1474269948293025, -0.17545466311197006, 0.473790154766074, 1.2007604078438672, -0.400960316564303, 0.3690009002692958, -0.3916827657161646, 0.48466106801406295, 0.5984772531795615, -0.20795982758734022, -0.5481712905551321, 1.0893371609991616, 0.9654213308832109, -1.1198944283147472, -0.5474807206091883, -0.494708735356575, 0.16557749200590863, 0.2643722481563944, -0.15870046483433192, 0.2135013665235056, 0.7130167160764739],
                          [-0.1662241926079159, -1.2937837601970033, -0.7849925628404485, 0.02255721697866861, 0.6998250452627367, 0.7923992827929015, -0.411549379614634, 0.005090237399313233, -0.19341858683544877, -0.45537301700736194, 0.6471173330826264, -0.6088699292679896, 0.35127180530483265, 0.025735231672505632, -0.06672835810901795, 0.8314206052184767, 0.18348978146728573, -0.2998819553296103, 0.20082892415721063, -0.5066416177807921, 0.5207418925992354, -0.738665481897798, -0.5104686404804015, -0.703701006453205, -1.200344615546625, -0.8060702357234216, -0.61024035085568, 0.3889180943616861, 0.05119932449950286, 0.6030731095801309],
                          [0.4559605864787219, 0.06437070571039213, 0.0954508689021255, -1.172840718633678, 0.17030688937141256, 0.466203952601758, 0.1406663641948111, -0.5770107384930494, -0.5185141761187327, 0.3544363696063443, 0.8930812142930364, 0.6462453641177555, -0.5310094565243813, -0.17800308964589293, -0.13379282610831977, -0.7528059077336271, -0.523235559882417, -0.47384741749215636, 0.44576835183928004, -0.3521099055382947, -1.2040114067658438, 0.30561033288239126, 0.4081217979936844, -0.02125373669832332, -0.2933905883878369, -0.04149665210875168, -0.008385243789877539, -0.2751327552923185, -0.1719910657294616, -0.3603111537158193],
                          [-1.2440209840194787, 0.18600960770413574, 1.5234473666150032, 0.6372164775914188, -0.7067872152820598, -0.09329682343981056, 0.07585319313081089, -0.27762936942960015, -0.27387269933041775, 0.4976916727982114, 0.5403836394843771, 0.14583962497295858, -0.4568172818788001, 0.040819854314474185, 0.35291684408409224, 0.09428835515473148, 0.3983850391736472, 0.4771255845270982, 0.25665398047825816, 0.44297030841999746, 0.379349438498325, -0.6483238267390634, -0.26597158107661395, -0.21937504315999423, 0.19820201974656781, 0.40810758258273117, 0.12344383052121652, 0.16004699151926552, -0.05003344532078717, 0.42265739101167593],
                          [0.5327905475470192, -0.7741849463651709, 0.26048305828149754, 0.6212894752480851, -0.2913208153869774, 0.845185856246884, -0.10573428432942601, -0.3004137020149608, -0.4536701213382787, 0.22962473845124998, -0.457878535088495, -0.9725776524066423, -1.026993523470228, -0.4136579543649682, -0.627761941014733, -0.005544649381651433, 0.5058052147512437, 0.1347004478560244, 0.46560180304523746, 0.3450436361359998, -0.9143593219139656, 0.5127974649718243, -0.06607415948686027, -0.04345874251862052, 0.5390462832936525, 0.06026415360873097, 0.7679484648287777, -0.7379514651587145, 0.07561107374386902, -0.01562365117071193],
                          [0.4478297965226008, 0.0999372699709477, -0.5234063910810418, -0.872493322683741, -0.17719934572116405, 0.05782265577492497, -0.052638119993973706, 0.28218855249259145, -0.35388264534215946, 0.7627195192894962, 0.34422778396797515, -0.1915018321957579, 0.2268120165814829, 0.43229900056073445, 0.22214070583925408, 0.41833846519962187, 0.2056888889144615, -0.37209036973458254, 0.46906911895840403, 0.8129271057756465, 0.20073669151979084, 0.23659176560377926, 1.1261787683959885, 1.4602709066505466, -0.0913765673262805, -0.8348264212257537, -0.9021955852991717, 0.08628224684233214, -0.696961286196059, 0.023611597817413846],
                          [-0.5778564377478782, -0.6153264061834051, -0.8410162575408124, -0.9259924765522334, -0.4792657867150168, 0.21231911561910727, 0.40160655422791613, 0.13319937655222414, -0.04519478467524819, -0.05888734609940097, -0.7433985581376058, 0.18428925253474931, -0.9222513386443089, 0.01687410426574945, 0.6615177068742466, 0.2592292786482728, 0.38873811329265606, -0.43793222473801374, -0.2300815633299828, 0.16586790473874363, -0.5430030083555014, 0.8535491016961871, 0.09983163092450519, 0.320509586584195, 0.13648222161540094, 0.6179855743217844, -0.8657952698899307, 0.1262854667032729, 0.6196451981288785, -0.5756246829443674],
                          [0.1780090514368573, 0.6574734686427509, -0.5801947140711495, -0.7113338505590743, -0.4209520387912745, 0.25759585985235073, 0.022386498096828543, 0.6875259851288092, -0.43206066674191973, -0.758127913170196, -0.6208648485819169, 0.7908604597118424, -0.7011277750841167, 0.6153544401736631, -0.7033573990562646, -0.3331012710105753, -0.19973395104155903, 0.36732247236949345, -0.8158990536790423, 0.07563657764321154, -0.14864927432559338, 0.4800229358933217, -0.32834323283387906, 0.4880931367066433, -0.046595345418194226, 0.31076297981222206, -0.11109418622486184, -0.704264617960512, 0.3458887455449835, -0.0011835147063002423],
                          [0.7788269169810891, 0.03379124395898091, -0.3887918326880264, -0.2772408139636585, -0.07656631597596125, -0.19631417493581454, -0.1740326702622334, -0.06961840922790455, 0.17141868920448045, -0.4377362641367542, -0.9560225923254281, -0.8292947219590668, 1.4125041074085158, -0.7650885589975412, -0.49523865569782155, -0.18380684115102158, 1.1170161797395441, -1.298380594809119, -1.0035360505485005, 0.2483028595307139, 0.0781388253071057, -0.42973199404826823, -0.5362836690691362, 0.2756137073824725, -0.09883587540870221, -0.2081774291903833, -0.5464783617389075, 0.06781646805721118, 0.16110809625603603, 0.23336329088558727],
                          [-0.6435097293659593, -0.606502859133339, 0.04177825214010064, 0.21250431217774393, 0.5689115788681713, 0.03099418612443204, -0.3869478691586266, 0.12658210424927868, -0.11440768154296067, 0.8530207796244793, -0.5687189077909988, -0.13535436802249456, 0.3720052427841296, 0.35360604020162834, -0.24827547805902822, -0.5769989931793598, 0.6682802147000084, -0.20797080431693934, 0.29357588352513614, 0.5054932944840316, -0.12740960649776967, -0.09942136234681635, 0.14278639744026145, 0.5806161476108449, 0.9328754175459328, -3.068204709107125e-05, 0.23751516869107858, 0.9032266296767456, -0.1888603565942328, 0.18883150858078682],
                          [0.5327651915724533, 0.11195771764132577, 0.1914076749842824, 0.5637726294423818, 0.7874770028352207, 0.4465234938374055, -0.2527589313564717, 0.7273644530371607, -0.15610547189212395, -0.09396708782256444, -0.10523121780701541, 0.21785417693363623, 0.4208338883501319, 0.44788139957492934, 1.0604521266570879, 1.4189801892530718, 0.4900696834276783, 0.5119411730390904, -0.12097198691340771, -0.07231393999358521, -0.36252072428382437, -0.2554817969260664, -0.28460459085668954, 0.49535873392421426, 0.27167118141506147, -0.22218683825173638, -0.26178475352298325, -0.22993556394813985, -0.3821433949677236, 0.23778269965655247],
                          [-0.7637447809131526, -0.44933701303800133, 0.31795139453301974, 0.5659841850677332, 0.15510223529664321, -0.547738949151706, 0.25622005577236345, 0.3712071531098663, 0.01324634420612504, -0.10212088736513447, -0.05810162962687612, 0.0904074585972174, 0.33777073318738343, 0.11816762945116313, 0.02761671369606078, 0.1292341058595661, 0.15820285689850766, 0.17729412156136612, -0.4353652355474657, 0.5947876043219285, 0.19891762637294358, 0.22722860406830364, 1.2601299736336238, -0.4163298561004527, -0.47272179469913467, 0.18402680878158123, -0.6939273599354413, -0.7261535952592272, 0.3563209223214253, 0.7190777882321077],
                          [0.027248949595568644, 0.15166546537934147, -0.4111693140488253, -0.20230784507415903, 0.15801529091832245, 0.1059294921252284, -0.11567588084304466, 0.6088528846905249, -0.3412131452444802, -0.15729499384376586, 0.7401055317297539, -1.5006289832915671, 0.13090767568871237, -0.37272686872380556, 0.6370798781282889, -0.33667335330587517, 0.5508130036005932, -0.4176674376320644, -0.1442216731275609, -1.0288483850076697, -0.04725689571527415, 0.6061425281688448, -0.26582812921124604, -0.34010816358942053, 0.6740955997168462, -0.14153986785434847, -0.4746644753735187, 0.06934473290581603, 0.00014184921564734665, 0.4433300034159752],
                          [0.08419707796852648, -0.3314780915307308, -0.5748777413971959, 0.5582140861395108, -0.00771133694632169, -0.5624791612969371, -0.6393558368233327, -0.40183976093129536, -0.08292280339767255, 0.20319471472109374, 0.1627920682833453, -0.30801587220003646, -0.5770041704285909, -0.7735049322123501, 0.46182980542289415, -0.5250504125375844, 0.33806078139556, -0.03377008172498841, 0.41281565364577233, 0.1673081250170544, 0.18663200040080657, -0.2619323744164921, 0.35228362288863574, -1.1778287360994388, 0.006762231985964498, -0.7460690295791756, -0.5899610518113729, -0.589398749856204, 0.5215467417902745, -0.4720957938921199],
                          [-0.931199005996854, 0.43277972981622004, -0.6788644747795615, -0.18769300919624224, 0.9225276939240815, -0.2492459271661735, 1.2561591206849212, 0.017013643162545916, 0.004329514934235869, 0.18705846081209077, -0.043203736666849755, 0.6081602203988467, -0.3117414351751388, -0.12080026150693676, -0.20085128126327123, -0.3495641693767574, -0.2617431378699667, -1.0742483229978674, -0.31149570727587006, -0.2049854923084885, 0.412238149142494, -0.09172510483652617, -9.379873505933889e-05, 0.2851107181556532, 0.04338034859583835, -1.0063737694040293, 0.07775418686371369, -0.36254146913912216, -0.5520777213772112, -0.3733820589476754],
                          [0.04116705063387595, -0.24425211387516635, 0.868177624936142, -0.6310643735648179, 0.6965422027113577, 0.40319728431999535, -1.0508692395121153, -0.20790409002778718, 0.32719278565624244, 0.573022016022216, -0.1386032997236148, -0.01693461916220912, -0.4045826507260844, -0.09919495365883366, 0.4564017588058512, 0.8900608375745283, -0.6118728617372957, 0.13113611503760472, -1.1100834310585534, -0.3459208220777488, -0.30042198432214434, 0.18991392015661507, -0.12291308082799066, -0.16891773536567184, 0.09902588442792665, -0.1295197016468267, 0.7799472999406502, -0.2589849200518236, -0.2705059566437447, -0.041066080450706505],
                          [0.06364293256789, -0.029462535446426047, -0.43772044622958267, -0.3771028741371682, -0.19824965325213048, -0.26796733841199005, 0.14799726274422473, -0.5091895974909051, -0.8310202978017072, -0.2993304957719206, -0.3525293201505264, -0.47513281224425963, 0.2773403104635961, 0.005756957893099445, 0.7089630701153238, -0.04874928997518104, 0.28860260664614024, 0.09587941102473566, -0.4337145058932092, -0.1576319633091992, -0.6596716147693944, -0.4045145460914585, 0.3059928841445649, -0.050113121531001566, 0.10969143426704225, 0.30054854913719564, 1.047377155881221, 0.2154051997932506, 0.05567285281411036, -0.673115503345183],
                          [-0.2446941577971199, 0.106801747558545, 0.5177811430675091, -0.7425229033260294, 0.08758867199750504, 0.1488061243987806, 0.16402967563046653, 0.5419463641830184, 0.7326939633320102, 0.027811516346864403, 0.011000864954076438, -0.32341278575120824, 0.2897436489179927, -0.12186786919688203, -0.6324240007689849, -0.09420751707073916, -0.4826010926339242, 0.565034246460138, 0.44977967161758536, -0.5184709981086193, -0.04829432448780157, -0.0304127641931348, -0.1939220555449293, 0.1293952584865357, 1.0007001700968348, -0.644818117165531, -0.24188732014255082, -0.08002082361010655, 0.08083068399885046, 0.5675133480218297],
                          [-0.04064469007650939, -0.8053740565771232, 0.06269135922206256, -0.16980811851286837, -0.3231364263252641, 0.8015624492193028, 0.6080184848683853, -0.44126427156650444, 0.2524892845855758, 0.0827563726106704, 0.6239258604598814, 0.0832768086476268, 0.25416569089101493, -0.0869651286872917, 0.5578373622386232, 0.4751502036558199, 0.1535095715398947, -0.109541964830443, -0.35704762816262214, -1.0162460336743973, 1.1701009734426944, 0.053127030907076905, -0.1743631787913957, 0.23589406161222792, 1.4550702556977255, 0.17956079554205673, -0.18126194795380912, -0.35973459359790766, -0.42567330831406897, 0.1407599647811832],
                          [-0.4231030504583378, 1.3993791734006384, -0.2974867876655841, 0.15602781829287796, 0.29199620055090375, 0.6858788413037628, -0.16350822491602326, -0.3550392847780144, 0.33905640188520225, -0.21439719070100224, 0.3079795587172005, 0.5687468450030202, -0.005316770326502202, -0.7585241434126971, 0.4545479908160448, 0.7891296191404112, -0.2754348270399245, 0.254161382530106, -0.14951441180599565, 0.6424193617583918, -0.5081042102635959, 0.1780925717035237, 0.47903610264167507, -0.2953055281360557, 0.3634409958852889, 0.6295044167232462, 0.14806437233862593, 0.2591894360442388, -0.7264873881286057, 0.38278716069019314],
                          [0.3921856376731301, 0.3017296264325326, -0.08937841704803298, -0.33128521745664474, 0.7075639990998909, 0.6524794994457755, -0.3556210203445219, 0.3915628693526461, -0.4060738297958019, 0.2623372115106654, -0.6226757627498926, -0.686829509408306, -0.8802850053493994, 0.0891112829587712, -0.368448384696812, 0.40038493018851656, -0.5686153365648299, -0.7330852401360398, -0.8171702320310947, 0.4262549255179605, -0.054719377308353336, 0.3733440317259621, -0.0034952930501937624, 0.6310827664020321, 0.39341772826956545, 0.7843883618126585, -0.5528061449161896, -0.7256171706424286, 1.2888700744945383, 0.1619270450131655],
                          [0.2550723175369013, -0.28724568573356946, -0.006174842350617299, 0.012864444322135571, 0.24471340126827526, 0.45123251816548665, 0.3652990125826973, -0.5631916381032893, 0.3067845475760742, 0.29658312359131167, -0.22885718615661893, 0.3757514854834658, 0.7064767555106136, -0.5348913902315591, 0.2979528592573472, 0.4979861846667213, -0.1035696209638984, -0.7355431110717402, -1.1668152496168631, -0.15749648875652353, 0.5150036159505347, 0.3642258119005273, 0.027052424742704917, -0.2753461499806025, 0.2786051523968783, -0.6492345136837528, -0.05729164898573211, 0.7817979032165872, 0.3088327898252882, 0.767521393507543],
                          [-0.7203579536098543, 0.12186760463050267, 0.6993165883356993, 0.25470286360719924, 0.07156037889248763, 0.5083236268538709, -0.7738843723158384, -0.8706267296971979, -0.2997035074548644, -0.1646895384064018, 0.2318690664960127, -0.300834802890036, 0.14743182359894935, 0.3398984557528355, 0.16568236949094028, 0.053229818872275306, 0.48674262643751776, -0.2025190794921917, 0.6839931409948092, 0.8778118040515414, 0.012419916881345049, 0.585798656306177, -0.6983453607754033, 0.2625647706455376, -0.5724090617650777, -0.2929380226445725, -0.6314847989197571, -0.3433025839911423, 0.6251478033818892, -0.06194723002124032],
                          [0.48494634843398743, -1.0646641154737155, -0.3162425757708595, 0.6599352254560598, 0.27911403881285113, -0.6383838192620968, 0.10451100176490347, -0.6300073511955803, -0.07816628738603076, 0.17361435117137894, 0.5116148182745501, -0.1799805841887142, -0.34195261491435486, 0.1361304235158068, 0.6363577531380599, 0.46254787581617707, -0.39831779952313334, -0.5021469245000705, 0.07294492797704483, 0.5205235336992341, -0.10450885208855465, 0.052059739437135814, -0.2350561490106614, -0.2517601437397039, 0.29275707838129317, -0.4409220071562644, -0.41161368768391854, -0.38709462245053244, -0.6485935288927744, 0.2291863966962973],
                          [0.44300444268054223, -0.03421267135612949, -0.7968740372230199, -0.2948702246276172, 0.33819048231434223, -0.45248335923393934, 0.0002829197632219019, 0.4559788937173712, 0.1115694082096472, 0.01740552441265769, 0.5874992458516929, 0.2635337857730487, -0.09293931815122562, 0.5520651770187974, 0.39362461781370917, 0.7172616178908198, 1.2713997332235305, -0.7345751921321013, 0.9308900908704225, 0.7152942154835733, 0.04501525040916676, 0.18001372272860333, -0.5112250025957518, -0.6931834769358324, 0.007719485547783567, -0.022186014617412653, 0.5222714167490485, -0.491943461735535, 0.3329814936821886, -0.09154425206730374]])

    matrix50D = np.array([[-0.27163801068156107, 0.134326839234945, 0.03644900300543208, -0.11413094815132077, 0.2263409209429326, -0.3973082804398856, -0.7816452795600626, -0.5628899944908553, -0.3440553128033295, -0.14186577911031234, 0.8559711418383728, -0.0560609663832308, -0.0099298035559868, 0.9284266082636905, -0.15201787290154384, 0.4147842098540234, 0.4052611597164367, -0.1314249690303096, -0.3719943998467434, -0.22707437015487214, -0.5471400068952263, -0.49604739543703236, -0.68652250664034, -0.3647069056761487, 0.5587977909085883, 0.2662847149570991, 0.0029666659417890984, 0.6560888783996701, -0.11187457476215706, -0.31708284312063134, -0.03625944021780954, 0.44407781017686254, -0.5620969347004696, 0.1787104625854561, 0.3335775859875112, -0.7346302975237299, -0.4643804650528027, -0.8438207198400376, -0.5512293529710054, -0.012430810253594715, 0.35460400003967624, -0.29881740842849014, -0.14278355325734995, 0.7087140907010712, -0.004884391692982326, 0.3706438637757279, -0.5658818489305324, 0.33182104529294465, 0.44519255515879785, 0.2435759195770585],
                          [-0.085093704242022, -0.28097824523892256, 0.029879182573724017, 0.04034237720150322, 0.27018972037837463, -0.009696182491643203, -0.6148349888642042, -0.20401906002637074, -0.32662458432620867, -0.17733372917589732, -0.49395069417596205, 0.4054568559149919, -0.38641364536224676, 0.006642675066339513, 0.37581083998788645, -0.44015796503750704, 0.4178458487804978, 0.5154299597905613, 0.30225155263421355, 0.06860893141489266, -0.5179110918151482, -0.11670621255833233, 1.0210376346131973, -0.23002346470604995, 0.129907820603467, 0.33530908221650896, -0.5096430795450033, 0.01201439264683252, -0.5981366669478432, -0.31473214624168105, 0.22693173088172025, -0.04553067717784868, 0.42532637777326016, 0.33187301320498463, 0.649452433021589, 0.3407382055641518, -0.13942579792688642, -0.3193946624564077, -0.08172632520754163, 0.6047230873021201, -0.7332849394746981, -0.12942485955606847, 0.8577847039988006, -0.14883751476831025, 0.21195145503378388, 0.6910317711220518, -0.3723590919090767, 0.0610293469125648, 0.2878540211309899, -0.3415650096044832],
                          [0.4288730980860097, -0.4376121627819444, 0.1989623952530268, 0.4607159256412678, -0.6123764334376319, -0.6623691163448017, -0.1370749332944478, 0.09238695638173555, 0.13616876690557844, 0.16821661909177466, -0.22021042029430976, 0.45782028938816094, -0.20015009754918353, 0.09510428973141775, 0.03983528692558149, 0.9239997052013248, -0.2734178082699232, 0.16685847649443147, -0.0027330739227713584, -0.18501525847258363, 0.3583334234575581, -0.41460842367855555, -0.11225547882501334, -1.0986036151514518, -0.4501008889268376, -1.1522622984804243, 0.09266117378510258, 0.058084111422417956, 0.33776199307306287, 0.3762032374715932, 0.09169757621906319, 0.622121324406941, -0.5820108673412216, 0.2967714793541981, 0.4413892856711498, 0.34769276502975527, -0.8971659376775095, 0.46119145022776703, 0.3509324832423293, -0.28370452257441814, -0.6662438596531937, 0.15170296491256, 0.27780548557922596, -0.5794925995707136, 0.5065145110759797, -0.5313923177793151, 0.1759530778142564, 0.30371155533432476, -0.2764462147100667, -0.3895803856453015],
                          [-0.09886692710126613, 0.47976170343721863, -0.19698127701562126, -0.18875111329737243, -0.22430178256810787, 0.054620608312376764, -0.10403970383037182, 0.18280988987284394, -0.09901759785984456, 0.9916810809225691, -0.8492336555881118, 0.4146593176907769, 0.7189670167054909, 0.4131834601375626, 0.5080298633429509, 0.41167591829306843, 0.38392309880232683, 0.4824560137172402, -0.23644043237952383, 0.8251086766462665, -0.022662415476808865, -0.6779450959920524, -0.6630448946785822, -0.32651864088430804, -0.13223176936806236, 0.4532948053591338, 0.6216390397296444, -0.23768439028816002, 0.5907215756388975, -0.005936716154742351, 0.44553579514243613, -0.5416607611029792, 0.12306334689057025, 0.16918689796766181, -0.022942963423251417, -0.06610220773928249, -0.28043696993710704, -0.20335930454641718, -0.10817072857229869, 0.4820661798413363, -0.26964739234491225, -0.14121007995569448, -0.28763180580653136, 0.23740994049431682, 0.29299137331818376, -0.3417217196019238, 0.6569087979565239, 0.1792868770139191, 0.20050895908403932, 0.6611792837376351],
                          [0.32019775990296373, -0.15449187580030999, 0.9284562602551348, -0.36208415842831443, -0.1036838958476487, -0.5584031791695913, 0.014383246724526302, 0.3688986461238465, 0.1869911985035501, 0.19063002424747766, -0.01613249983279609, -0.38156440482164006, 0.11074987116707148, 0.5947680123468251, 0.12689135103372445, 0.28600144862356347, -0.13214778521254078, -0.02331189442774942, 0.5531889136301535, -0.08507454276283949, 0.10812770507838466, 0.32808301858280275, 0.19473481776560778, -0.40640434572202544, -0.6518161897209163, -0.2086872430418071, 0.8868018059305772, -1.2533268872440912, -0.021024466529054764, -0.35952008229713905, 0.9043039991631598, -0.3732677091989411, -0.8588596586701204, -0.68820319206135, -0.10974722236238584, 0.16263384434318184, -0.2239855283596859, -0.7165203439577117, 0.4849956933532484, 0.05970511750201908, -0.543956233363111, -0.09524526355981638, -0.5222266802184757, -0.14070875475663572, -0.7535566659791733, -0.6062994814790916, -0.7776965899442141, 0.17981493404396715, -0.3677473085634842, -0.6705098507895102],
                          [0.6895601686661381, -0.5322885031823197, -0.18747680743055306, 0.3439920172082569, -0.09118098373309874, -0.528685198179235, 0.1612354790212309, 0.08914114275949356, 0.2979470106957422, 0.11383058867964534, 0.28691931061120046, 0.23433400282923414, -0.2027772835297699, 0.12951883411495313, 0.6141409920278484, 0.6221262218869789, 0.022789754577986224, 0.006598501021486014, 0.2509603751600963, -0.03618064445794282, -0.3048488956114277, 0.06759847454818407, 0.10374028739278351, 0.21152080810134063, 0.5341803572726102, 0.02424743177457319, 0.036509270567975885, 0.4559880751376399, -0.9727751408479415, 0.5453436200207229, 0.1929919162930066, -0.08432908960205503, 0.29936076914083243, 0.30619132680766814, -0.192259857278946, -0.8702169695187335, 0.7339874845944215, -0.07979939822830556, -0.11858363305309617, -0.6354773538211406, -0.12676946963566457, -0.02044461888399423, 0.20314467159746874, -0.1854472537187313, 0.30630678856838295, -0.04546442679405707, 0.23313050635554525, 0.7069787674873818, 0.5123433017715282, -0.5587638626014193],
                          [-0.4921521922726888, 0.12384110159991463, 0.2613324708773459, 0.4118735545322543, 0.0017234106099440656, -0.8963046323358481, -0.03924208836451989, -0.0787062171067413, 0.9465790672871417, -0.048966301477460485, 0.0988305285698764, -0.1901814354150181, 0.830194197854259, -0.2911584913428134, -0.17845809350013533, 0.5483081054084799, -0.11927059183280461, 0.13510576445459122, 0.280107477777682, -0.21249359313111676, 0.4262547853859454, -0.5748836219634795, 0.12549981904101065, -0.7653693350610374, -0.09842666081985366, 0.2967103663595074, 0.2865938440194766, 0.2988929736671098, -0.23201237202611272, -0.0897097945847684, -0.26032511130066993, -0.0492624588741998, 0.2219555515917539, 0.35975009554386606, -0.7416360839264658, -0.36401768600768875, -0.39047774214654457, 0.7878738757680592, -0.25848886272019994, -0.35894776449809657, -0.4514435503067529, -0.26471768553244635, -0.21655626174467413, -0.5665100762971144, 1.0589507643304763, 0.15027423467509426, -0.06735934315036995, -0.21489241357833636, -0.5425920894098943, -0.285511996514401],
                          [0.06501812178375796, -0.04437387662763983, 0.07197104112347512, -0.0631438368119771, -0.1610234105232604, 0.2955719826600388, -0.18883406996456872, -0.440157790177782, 0.3125744679393567, 0.09315515565486121, -0.5969183099581022, -0.03786556441960787, 0.1438263373578949, 0.4836539468718789, 0.05267170433904325, 0.6055335613746178, -0.33762609619637163, -0.42931413705275184, 0.2354610000641127, -0.9723833197387936, -0.1882294362825989, 0.0037430038349969275, 0.08639566832986396, 0.16532384635457434, 0.047487235917430023, 0.7071192052136469, 0.4124079626835684, 0.48065163272884603, 0.5632035524971888, -0.25774373800239314, -0.17188400917866176, 0.3065864873599606, -0.017467992745742077, -0.7385966695434025, 0.4232186593703412, 0.1072335445934485, 0.3768941464322641, 0.3498357486536593, -1.0076992866144785, 0.16183544005305223, -0.11506018863650655, -0.30390936241370015, -0.3704283087551841, -0.06574152363573017, 0.25420538131466774, -0.6767904229441261, 0.10435321825299233, -0.3233774517144761, -0.4247381037456346, -0.13734926836801237],
                          [-0.6444716576711594, 0.24654516259775316, -0.1285691340963951, -0.18701633072238094, -0.2308004148086161, 1.035260810309222, 0.20834119764324915, 0.3930150551878284, -0.46442540082176, 0.14105021653716318, 0.22881304228193095, -0.3852779598190716, 0.7809929309744599, -0.2898970745350542, -0.12528163885897306, 0.6106771397239669, -0.03494138736464204, -0.25225452110755375, 0.34798086511456394, 0.2567039718141064, 0.8304819861890145, 0.6656935563698445, -0.24547615973454218, -0.0010456576985708246, -0.31292900763476905, 0.6255177683930297, -0.5477798094040904, 0.026778975679052297, 0.467401244725097, -0.18956425591081066, -0.8842766851888144, 0.40487751117148674, 0.4130451834525059, -0.15050104063864836, -0.14974570993435674, 0.4999164776228832, -0.8060976289937984, 0.17730135990772614, -0.2584462860424783, -0.4818499895779903, -0.43092186949000016, -0.14407225310043154, -0.6303406847164448, -0.016216517875117985, 0.3947928296335764, -0.40020810720718547, -1.0300141668167995, 0.24296460299297157, 0.34062904946688094, 0.7236047182990636],
                          [0.023069003330989588, 0.5990744212809517, -0.3508711866485147, 0.210373970244188, -0.2553399797339283, 0.8924103406992099, 0.17563897213584848, -0.8538595715634498, -0.13057396774372138, 0.2608919344784661, -0.4848749697835003, 0.19157922118546508, -0.0018911477924778673, -0.6793315227758273, -0.4325365152170213, 0.6398748051515205, 0.15398997930722208, 0.021198397283139464, 0.3265835168476392, -0.08033954211532682, -0.161659487019247, -0.6722112604340786, -0.2217252421063347, 0.5057570397681512, 0.17995011024419028, 0.1325737308355106, -0.08638609234795419, -0.13131880903967577, -0.19383508881739642, -0.15196013902427893, 0.0286208672586497, 1.0690769039382406, 0.02144666022307714, 0.24645948164259088, 0.29927633531987186, -0.21422053973576666, 0.08802117814999581, -0.14528846231522236, 0.12782726918948098, -0.22787756763005276, -0.37065778734310684, 0.07926209136190404, -0.2888549957354941, -0.0685316030309582, 0.3433088835380068, 0.4926213884444243, -0.29252447624985295, -0.18614498836164484, -0.32354556478703567, -0.009520236168169993],
                          [-0.41770601921991035, -0.5252737414980924, 0.4877859966934701, -0.19371168622406118, 0.27294283253382695, 0.0025908654486301932, -0.11591668152771449, -0.7240474037271221, 0.7729223721987917, -0.32125273970030627, -0.3575798884865216, 0.45084901284479684, 0.2800988555487168, 0.3912903397491156, 0.26749606448540036, 0.8088631858854213, -0.4844341816027353, -0.241527177607065, -0.20441315891198333, 0.04349874455860166, 0.33812545029732755, -0.808397283771473, -0.19217868272899513, 0.28237808804384157, -0.18403950835215027, 0.1764047764121203, 0.08114257926958286, -0.9699814235126865, -0.6691666045033754, 0.06357308072928648, 0.4858972577100382, 0.18675549990391754, 0.5746874287248869, -0.2759145725731622, 0.1039245714434059, 0.33937026210631493, -0.19945860079593125, -0.8141223077636506, -0.09001087838713535, -0.21324539480701618, -0.220743656704162, -0.49967472259946744, 0.20673328740453245, -0.08540368832234578, 0.3939429834608762, 0.2767301182485269, 0.574744407207605, -0.14620151828973088, 1.0616102898093027, 0.5286435331482083],
                          [-0.1520373851796157, 0.9861076876367627, 0.395414514520239, -0.04111906575210752, 0.18431275657266705, 0.0180967978193346, 0.21344686215858483, -0.18924754265045743, -0.26337570842582525, 0.23786225813283352, -0.2894437045850552, 0.18811562141901486, -0.14405130553780532, 0.4761474972949558, -0.46621879679457984, -0.524428178635576, -0.08099453937477265, -0.22671764667931366, 0.7213535916877675, 0.4509994499423001, 0.309311363661705, -0.5357422427584716, 0.1821654358264098, -0.6908258088629404, -0.10878882962138414, -0.4884614632284607, 0.736689738337651, 0.20313611061868608, 0.04260470204366159, -0.0675076871892809, 0.27606854984532975, -0.3549175922410503, 0.08586952666156035, 0.4308518997582713, 0.2056853950100087, -0.5837999295938531, 0.04123112894544026, -0.9162814570053277, -0.009837615426859597, -0.04630219643378495, -0.3625369406656516, -0.07729140551847058, 0.08487277234088515, -0.8278328138884363, 0.16219523467618732, -0.34636821549774577, -0.32951273568112716, -0.3075161759130672, -0.41933775868282075, -0.06407934527356257],
                          [0.34143065288830393, -0.20820752794137054, 0.1118630070184376, 0.07598176377545887, -0.05038564863184043, 0.17435907567061676, 0.18651799946058714, 0.32874798710603015, -0.27290335748056943, 0.2904345983615972, -0.783487889370724, 0.10454344262021159, 0.8701108060592435, 0.5854816592612707, 0.058569497824150246, -0.08825738759156508, -0.09050308518966294, -0.17325340333123557, -0.3684212524027744, -0.19673792718196212, 0.46652060972500403, -0.10368660655996713, 0.02266419256838436, 0.636607245290315, 0.4385966257513466, 0.2661211026435179, 0.04565493034017593, -0.05531788988851255, -0.1821398911537288, -0.24394811569505964, -0.09650355807649169, 0.20698214899040218, 0.11877600919379298, 0.10809404683036808, -0.8670042067966192, 0.03855537729057154, -0.4804073835889159, 0.36745361719362796, -0.6582026903451045, -0.41000131104232296, -0.17177620770244484, 0.17642202066287005, 0.05712649465571545, 0.0395344944155972, -0.2905358755528477, -0.41886380545540086, -0.08840474976121135, 0.0083796036061538, 0.10295198562982694, 0.0909365411755511],
                          [-0.03525298002298935, 0.37778458689870753, -0.07791208522170291, 0.4379100481888387, 0.027033015811802945, 0.008997230745681577, 0.31411483976190924, 0.36589715636302517, -0.49425972725415307, 0.14580461561127864, -0.30400425053411706, -0.5428825823002196, -0.05066460907458987, -0.08520346354969774, 0.4764164538813473, -0.2917120128830621, -0.7506529339070704, -0.22477977199285615, -0.22000298028367804, 0.3789890640753429, -0.12480431794312695, -0.13175467742183156, -0.6298537900897542, -0.23091446156208123, -0.2498498697832423, 0.7633087594009896, -0.4690955090678429, 0.01701466025248002, -0.026427137430250845, -0.4020892107866464, 0.30177024507196937, -0.22144919686243725, -0.43626079613259217, -0.4114225988789175, 0.035481337342764825, -0.410640174026147, 0.6792346287076635, -0.028537471685508315, -0.07642068453450629, 0.0441368759398425, 0.4787892805926042, 0.5143010355569881, 0.16999180054155494, -0.4289942568198913, -0.6731080903483978, -0.2736910250273937, 0.38001904303362966, -0.3067950028756009, -0.7498278418398325, -0.126325145816593],
                          [0.14083349172705573, 0.09878593841121264, -0.6734612727392814, -0.29211177371126773, -0.5743253362772948, -0.14700074232675212, -0.018091410956992675, -0.5495466219382572, 0.0834063923936143, -0.026358781855923996, 0.6521330509826982, -0.10870970763931698, -0.030178589178039113, 0.1333104842841227, 0.13161335386601894, -0.004988368809634413, -0.2559494530350537, -0.2648540504674506, -0.37727286982789066, -0.45386714312287385, -0.5465991239529847, 0.4979540208925255, -0.19308513812372755, -0.04415691411163704, -0.32301777009642874, 0.33186495418371453, 0.4501341317387706, -0.004435675632171791, -0.4289097692903028, -0.14960568402197663, 0.4493400092977963, -0.15361135432543854, -0.4571659682268078, -0.09545585038258673, -0.5995731835319137, 0.23199082790025882, -0.2303589592008672, -0.011791409578488832, -0.09128790082219505, -0.5177782466537854, -0.4029279081089681, 0.37349410668509225, -0.1621621198547406, -0.3976470675895028, 0.7018356201626861, 0.5177074755399428, -0.12612294628155513, -0.30490578342800617, 0.62655207798885, -0.2763156317533218],
                          [0.33606937685195454, 0.8930662145777841, 0.0959951698064667, 0.10814440431134133, -0.07135060344902244, 0.3541685003269843, -0.24552768147571707, -0.592482141603163, -0.461474231032439, -0.2034105007474815, -0.1407241215203695, -1.0292029456591933, 0.3395723878509296, 0.5445491056820484, -0.3780019119925597, 0.3269771346708635, -0.28085472719831345, 0.7701922685875129, -0.05198079076953544, -0.13302815010818936, -0.2999149863498516, 0.3672391203024783, 0.01792732864299411, 0.22110462022515853, -0.019539269206431124, 0.02265329541491079, -0.2603292900249131, 0.11112958774394509, -0.013325951454995863, 0.8021473737981932, -0.3441901399468305, -0.4560563929913168, 0.21145892687506748, 0.17878072978449544, -0.07676105796431577, 1.0428722944594693, 0.3375461180860242, 0.3297118060985277, -0.1609934300852011, -0.3280681059827658, -0.36037546411489, 0.2778534646345003, 0.8644452282104211, 0.09086325671674518, -0.7067577059661996, -0.3364140701621605, 0.15565848683602904, -0.8254060319597347, -0.05911913954434497, 0.14051825511332827],
                          [0.11406410503060964, 0.1281048732507663, -0.382105857715485, 0.05082608841897042, -0.9054747008614153, -0.21137040806476654, 0.30746781218113867, 0.5049611869906933, 0.6266507908893353, -0.14350517109913496, -0.8623961305639113, 0.5806217875002117, 0.18304334436019082, 0.3767064683834034, -0.05175273627491408, -0.0911893791040687, -0.2221561487786588, 0.34055319164808573, 0.14014115371375652, 0.11510242774675046, -0.08288777144524542, -0.5075849236307302, 0.21139512178413064, -0.4869692430216939, -0.016380878317684318, 0.015201590827706838, 0.04695095943846639, 0.05322956334624003, -0.22359157228293292, -0.016873590936240596, 0.08383642664798599, -0.33823251148105804, -0.3704333925280124, -0.1871178095862827, 0.11971901586713551, -0.014672469472770198, 0.20912543947830148, 0.6795773941488981, -0.6021816447252131, -0.062319330477135323, -0.18373608780256595, 0.33238521108957514, -0.10643243823096656, 0.356132033669895, -0.5583118000462185, 0.7676018495958645, 0.10483110107580537, -0.20844944105818072, 0.2469159466511147, -0.28740816883178183],
                          [-0.10617439446552067, -0.004671666320941431, 0.30431882567568563, -0.45235672607427563, 0.17748038778294975, 0.4572087087620708, 0.1202259048636364, 0.03246825774695677, -0.7722549927291446, 0.0002196980821358302, -1.143531258650208, -0.18824420642600698, -0.18547229242816904, 0.18391335656073154, 0.10960929084331775, 0.2929796229919357, -0.33163375061167727, 0.5916117734150318, 0.7124775743245416, -0.30792421607736786, -0.03666386596047267, -0.1695408250019942, 1.383512356060747, -0.03701090624858444, -0.39264150995073, -0.2057067916284516, 0.2399520553821058, -0.396996537551377, -0.33223634572930805, 0.07253012866938126, -0.7445419729752569, 0.6165630659162541, -0.18281865707681044, -0.35862464355551193, 0.5141315123483242, -0.08755613441853113, -1.0796210729737667, -0.4333538665907931, -0.6852147802272286, -0.052358855521195125, 0.1474484954050343, 0.14722012473689583, -0.09740468854815648, 0.4716719430947822, 0.47280162345010135, -0.2887379060973697, 0.3729882589743659, 0.025469688046870276, 0.11020698952443546, 0.6017202273759226],
                          [0.19811420453417836, 0.05735445401011248, -0.43732338950672844, -0.5374638826800047, -0.18661426591679153, -0.5228290524961192, 0.36951727121751454, 0.3834850551393509, 0.5111848236779262, -0.18004839351813148, -0.1559962413291643, -0.5956018684782193, -0.8917619642490467, -0.26730988072183504, -0.35839124652902143, 0.6529010064971863, -0.192800038367972, 0.10728424500707659, -0.30060488992202244, 0.07680960997785573, 0.2772906911092039, -0.7383658013943706, 0.27638522988822056, -0.2811809534728361, 0.3710468589862853, 0.2646535090809512, -0.36379637311359636, 0.11008090942946207, 0.5613046169844647, 0.3007687074317989, -0.23560244869748873, 0.25241384079803403, 0.38060268889552146, 0.17023937888415053, 0.07634959461363812, -0.33564301297716403, 0.2878033214076312, -0.42783971088604544, 0.06546171203445939, 0.0617476283528582, -0.26613926921079006, 0.05391889144376738, 0.7256024685928164, -0.1606498331208681, -0.13731043635549495, -0.14652476304529158, 0.2805098631795503, -0.3667689669202969, -0.2913731244992701, -0.26757575497925423],
                          [0.6221352032053995, 0.0489905869504261, 0.4799055964241836, 0.08905991963364117, 0.40552983970624096, 0.8387075833081176, 0.14227374196648016, -0.6997916704465414, 0.21457564484724612, -0.04054145020025694, -0.2094626339031336, 0.38057247764085506, -0.621684101716655, -0.13516728487066462, -0.06740461158268016, -0.11555177785468224, -0.31639291585948087, 0.2033690695912951, 0.09883651921497896, 0.3702160665099393, -0.18133276561904185, -0.11831774726386504, -0.12319588449735767, -0.2640819407200278, 0.13156933003431068, 0.4325682839858314, 0.41559404395120103, -0.6804892184746391, -0.31168917444596667, -0.6480258222728935, -0.01473028833942891, 0.39895091943315425, -0.05979889044288892, 0.18519440180776073, -0.026581275504576184, 0.16926832710238154, 0.19359077691332727, 0.7463535449721715, 0.35243099739753575, 0.220503104540904, -0.4848364580742902, 0.3511359892926842, -0.0012922739039671616, 0.40309301442140205, 0.16132844472576008, -0.9982701939095272, 0.7919389795004416, 0.17654870899013525, -0.18363417588290776, -0.17944416941682248],
                          [-0.10197384364898585, 0.3420685238040291, -0.40505444395853746, 0.5238189544221216, -0.1692345610224916, -0.3385353627185221, 0.2947745991284823, 0.11161326565726269, -0.5292693331957565, -0.6006094219242138, -0.04985146941672383, 0.6461291327862403, 0.05846423912547227, 0.058938665791406435, 0.18924359775020574, 0.06732200508501249, 0.6278164253100873, -0.0016926461709172226, -0.5497733118453231, 0.461087978982504, 0.001954957993977982, 0.8917039142558587, 0.15804510583834527, 0.5408366365365358, -0.10534291825406461, 0.3588663609002638, -0.27697933330006225, -0.2022034561074666, -0.10963795387033279, 0.2189008913996849, -0.12227214650540816, -0.36353789602323683, -0.39244477185214505, 0.4904819891998783, 0.4871626520682055, 0.34871119359355074, -0.4231722383821738, -0.2547666127651194, -0.1196155475803692, 0.6033826475810605, 0.3554543387964014, 0.13881099763630575, 0.584553455985842, -0.6354786010838033, -0.07899606251046504, -0.057688201954075854, -0.6022715482106311, 0.008819653271693059, 0.18047150206360105, 0.22689730886784368],
                          [-0.026122951020515442, 0.04043669106913615, 0.6286756750608485, 0.4117625857423363, 0.008775767437249019, 0.03605355261705282, -0.16603117489257502, 0.27408897740555316, -0.961978223155733, 0.047340452574929456, 0.10993460037408598, 0.5170581548733927, 0.5001549870745754, 0.41111540667543905, -0.4728327608424403, 0.17620547847142864, -0.43091627159456974, -0.5435331212173947, 0.6159117857992261, 0.0424140732260095, -0.05338679294185707, -0.29671808736101857, -0.23749868207903924, -0.11643299388810743, -0.21184506169316983, 0.26278576837926065, -0.03980576213966089, 0.016123729139403933, 0.29825256712016096, 0.12964020323757322, 0.017584207295574104, -0.30298044791265333, -0.14242931058384323, 0.9320047390348696, -0.6902586252790086, 0.6557638024512079, -0.04733771972368847, 0.43245335573867877, 0.9521917738935963, 0.06613137526068905, 0.26863259752104346, -0.040347223927889725, -0.43755601783593207, 0.21316869874999883, -0.27081618739408064, 0.13577254134724726, -0.10620794788454292, -1.0695499135580784, 0.21861342314664653, -0.1382168556836637],
                          [-0.8592345123960654, 0.14488081881607784, 0.022012947745329137, 0.1541496681476594, 0.27837590293567405, -0.631949395858818, 0.22982967398972398, 0.15033356028021053, -0.05367423926002092, -0.22437634164202155, 0.3293875264490187, 0.311514830776225, 0.2600161786550655, 0.23440139574701993, -0.1305244363507155, 0.491664620546597, -0.11118736897457432, -0.17316608619323776, 0.26264878604571906, 0.2582588116496416, -0.743068931748436, 0.0013119388035261054, 0.010486208353429158, 0.07773384865085597, 0.27228238285086565, -0.535454032289949, -0.13340915881053894, -0.02901837084967357, 0.33250616831178115, -0.37262941745763256, 0.017087542790429663, 0.2763857134885835, -0.004869045150193114, -0.4232136107392869, 0.05329950426454681, -0.30710493574614073, -0.21454302701745154, 0.4344592481373135, -0.02195751069960937, -0.5782086550270263, -0.8556698117667516, 0.28851263550530654, -0.37081086678090447, -1.0755911763927983, -0.017264688889957996, 0.3315455791682792, 0.042084029227935284, 0.159342454927313, 0.21125651488328134, 0.18282232712586607],
                          [0.04007756645472728, 0.20909619961544199, -0.7757569014639554, 0.8082882461277824, -0.2809932528389761, 0.6336885234723331, -0.7339752640934163, 0.037449571863765206, 0.27242091555664794, -0.03470029721121244, -0.32112084577705474, 0.18144008803544887, -0.6181135616924109, 0.2402570695244629, -0.040436167997766316, -0.12132277253506799, -0.18997048005778175, 0.19647710150451628, 0.0678273406574012, 0.3948541474481347, 0.5058998040994509, -0.39133951642198306, 0.48162790764732, 0.07364061515658939, -0.48586808675333937, 0.1458277765184019, 0.4861933958405464, -0.8182859137627307, 0.20204034767740076, 0.13188040412715715, 0.17793759191278075, 0.27014231592628063, 0.09738408959079455, 0.24161438785251027, 0.1600724372199544, 0.07261881919413991, 0.03743743294340704, -0.14097277452890403, 0.9721064550322568, -0.5563787282377941, -0.48153300316267056, -0.793898557687507, -0.07638871761472978, 0.22621622210921172, -0.6715953501208095, -0.8750080155211699, 0.0032352512491879842, -0.2733960555124563, -0.15190584651788647, 0.2014167131039828],
                          [-0.05519604642350198, 0.08082128615319589, -0.053211960429151106, -0.12311262048318074, 0.14261831068604303, -0.42492845928283646, -0.22606671087884223, 0.9789760337899686, -0.24684155757211823, -0.1905543266249385, 0.22449424645899607, -0.05275659875822236, 0.17863320622458442, -0.04762585091445187, 0.018947191780251067, -0.23367209900145963, -1.1259670736552665, 0.2380327500695647, 0.6672117518824432, 0.2863971152108732, 0.06505703286519504, -0.1226262368847229, -0.2741251026151652, -0.49879664072585667, 0.16345716474684485, -0.054418376630435084, 0.5763709265868713, 0.00803908896395486, -0.08242010489524984, 0.08632279273029603, -0.22125876195892313, 0.45133450489038085, 0.009438677794637819, 0.532871674564852, -0.21468547616749514, 0.5320011696930983, 0.31286568617249594, 0.3759414197742119, -0.24085773137119448, 0.47740952479411963, -0.21532266446505324, 0.5977162874273736, 0.5131948940108471, -0.426862385872453, 0.19430231909877055, 0.0346995922070375, -0.007262688097386824, -0.5271732595391893, -0.01622607022004484, 0.023718257399028705],
                          [0.23123037806340482, 0.5336023144300218, 0.12601156498891541, -0.3788213072693966, 0.16720746726084676, 0.36923655011716283, -0.0054928079785392, 0.4467441336258238, 0.22520723882576538, 0.15999164396019458, 0.6467208903639351, 0.654843420811427, 0.8707301793415243, -0.3166721899801644, 0.4036732460761945, -0.8562218874335806, -0.4593829461213352, 0.2784632825019169, -0.0570141003627557, -0.23972861120730246, 0.05127854687896305, 0.5530119926898409, 0.06664105764374346, 0.34125440937032, 0.7849251266205832, 0.1871681119177575, -0.5327020072205951, -0.47385158810500433, -0.016938717889834143, 0.3182734789131503, 0.14552602418882243, 0.21015691684138307, -0.3404174564670971, 0.06042618391162374, 0.3146327458704267, -0.15457017780478532, 0.004408152919732913, -0.07794673931152281, -0.3991961124461071, -0.7676613797517293, -0.30142450018572187, -0.29651873925893724, 0.24836811945384749, -0.13291891037160317, -0.281441577200869, 0.15774797051772255, 0.11894722182743624, 0.18659868152327497, -0.7911158887807492, 0.7081137816239707],
                          [-0.23488840612043818, 0.20474463487857908, 0.07567414327921991, 0.218788621720901, -0.8222831923238348, 0.2079908102822224, 0.4955217996030414, -0.078615937272715, 0.027106152207762586, -0.8848605675146525, -0.003720643805458314, -0.46039397313858843, -0.17930593123737623, 0.031334462625547156, 0.008372582666408418, 0.26726103435595716, -0.1917398409103322, 0.07852821300505157, 0.025711191625118313, -0.6030537270116658, -1.0462757576958988, -0.11067684139708753, -0.21619679957102814, -0.48068923128385926, -0.610848726017922, -0.17140370772849126, -0.10993478304450435, -0.5287911380870582, 0.43201175393838853, -0.08024737267984512, 0.5517704892851283, 0.3128312952257161, 0.18324138110617685, -0.057914815805984406, 0.01338949506941781, 0.33135438393995065, 0.4455702253041239, 0.7624236431120354, -0.4847399565546321, -0.6264638017292193, 0.42630628307615437, 0.15760897316625283, -0.14153391743571117, -0.7541248361138547, -0.09483715499079268, -0.15873506741458895, -0.057449124839982275, -0.11939648238586353, -0.21508053091556073, 0.20464175533491358],
                          [-0.08304375714953624, -0.7522707446849811, 0.6334689367580428, 0.22193977546500346, 0.05493292939855696, 0.06823999056501902, -0.04842643245325061, 0.24729084304278107, 0.16526080390311085, 0.21468683793815416, 0.3615023026661298, 0.19402180280794157, 0.14032773542773525, -0.18036701715405298, 0.4688819146108374, -0.37239777754976827, -0.5766822408638723, 0.37967603660990545, -0.043427490815311644, -0.851864266471018, 0.21832246420162868, -0.0834500318583295, 0.006520258405468188, -0.3546674173760146, 0.3371284453521877, 0.5289966633349372, 0.31926269807442365, 0.026808524352679892, 0.24326284206329823, 0.04184709988350301, 0.1363206599837775, -0.6344119007786326, -0.6475983447260538, -0.3578271742417658, -0.48498624522776457, 0.17869081352536875, 0.020552869698350462, 0.06416847584637557, -0.9858651590272014, 0.28581907755201913, 0.037916071019877665, -0.12536880465804454, -0.1940263345118302, -1.0334621827502548, -0.15127107616322244, 0.4854235048026877, 0.7428488662081894, 0.30496316617228125, 0.294610335265333, 0.5857884563015645],
                          [-0.6973826763726269, 0.04709731639172182, -0.17372566466337824, -0.6132716903955775, 0.7229789654027607, 0.806230662755707, 0.21675783370167678, -0.27504883671977276, -0.9195595158222067, -0.02384034103826221, 0.38636760580802465, 0.06758564419107727, 0.671580453007786, -0.02624025463958158, -0.04267752930329431, 0.9634972089163077, -0.21410712071394203, 0.00904774053480878, -0.05249702303993162, -0.632179188483016, 0.7682547382424477, 0.06701445249995497, 0.36976259235335596, -0.4184861799055477, -0.16727518200148006, 0.12204066670826293, -0.15791708053605752, 0.34016165210821825, -0.7024578745034715, 0.25745377829554533, 0.531666594216913, -0.57671742697522, 0.0804495528499982, 0.29114231799941265, -0.24156226601026992, 0.3383365266722135, -0.0121845664600512, 0.06466069061178112, 0.465587576092076, 0.2787621890473073, 0.02884650402763902, -0.8869126650858824, 0.41965739468086927, 0.2068082417362656, 0.3387277106377924, -0.3205007081287867, 0.30433000009840205, -0.05434672827935941, -0.1967526596043791, 0.30776633297039346],
                          [0.5074637855366352, 0.8764370706133253, 0.13421251694186334, -0.40786665146806983, -0.2255315194672694, 0.20201006492419696, 0.05903120919481004, -0.304134125344786, -0.20672416834634055, 0.3793109058825446, 0.13979174561946686, -0.26305058599637726, -0.4360429138273909, 0.616539212174837, 1.0179765182772222, 0.34345641137048283, 0.1489209337811236, -0.09796106786203423, 0.18188168122107093, 0.8060293773645723, 0.05774657413259724, 0.12527405721227391, 0.01712019373404067, -0.6434555117319319, 0.4558266483267329, -0.49513392945735607, 0.040535871131377396, 0.01226416538228391, 0.31154153697980114, 0.3482898170222783, -0.40927005511872694, 0.752219635863381, -0.0845010183267556, -0.2864852609415712, -0.16414084035287946, 0.28611013037330574, 0.38397334179941656, -0.09548972309913911, -0.24507582361137836, -0.04224861299889251, 0.29797585681666766, -0.7239450114568708, -0.3108707741125386, -0.5215547657640222, 0.2584314769451262, 0.0914928540497598, -0.33898589514836713, -0.4340115316752908, -0.2209921046047565, 0.4457514209188542],
                          [0.5035614273146879, 0.034923478536269134, 0.12735204141498524, 0.6723335953131582, 0.10793722383991007, 0.2791478148720556, -0.31836667838632765, 0.09185948995979534, 0.8015850252285522, -0.2687919360914592, 0.6238730165364414, 0.5287154566426047, -0.3514270951351097, 0.1559079164603482, 0.7812806608867942, 0.262057187114562, 0.27443994468055644, -0.5324603042788667, -0.03653500226060106, 0.06362126618829259, 0.2759381498743278, 0.17878728447514772, 0.665424120944697, -0.06210775979744097, -0.1918974237745129, 0.20263652228973805, 0.40768211006859106, -0.1455141560657341, -0.01181317469749385, -0.21562604254309356, 0.3505422468434919, 0.3343487344370427, 0.040294373941268884, 0.09926215698569765, -0.2957590829632828, -0.00950966204976926, 0.2879119092167357, -0.17056051394139946, 0.45277524878844655, 0.026400024072150207, -0.1325176225511559, 0.8115372407394355, 0.17870366834226523, -0.0497364903880524, 0.582781525127919, 0.022514595072990393, 0.0629163065345944, 0.6439654102068014, -0.4940545157774032, 0.45037137127784516],
                          [-0.18866032400459687, -0.08736933363001889, -0.13936499389661414, -0.6333861681813828, -0.2917396683840856, 0.180673737442876, -0.416863642357651, -0.05432453425319546, -0.43887916385764325, 0.17242701970631402, 0.08056155419920563, -0.325001232697004, -0.4049470867117689, 0.08750758870860281, -0.13852905088309098, 0.6659326609259931, -0.21486518945771224, 0.1610289840750661, -0.44361309259845777, -0.32075776776193177, 0.6012571748651319, -0.13387991989505105, 0.4836016019151976, 0.1002287080332454, 0.3148465011218596, -0.129557517976188, 0.13993061790959388, -0.06639017866720273, 0.10143556468591666, -0.2274363730679981, -0.4409867900454519, -0.16746165486393028, 0.36057403786711884, 0.38869605065817064, 0.5899896040620687, -0.2529846014694098, -0.25210586312709665, 0.6364158878216719, -0.1601402503957845, -0.38462090952725403, 0.2387223989908719, 0.34570360898045766, -0.2278015008217641, -0.3338346753511582, -0.4192513411924328, 0.025305215322911634, -0.47519181524052145, -0.4587543904523229, -0.09141260375393993, -0.04173382688809369],
                          [0.9963248482789984, 0.31498980318092995, 0.4757149181858846, 0.3359743560447932, 0.6023082830273278, 0.3538558063951036, 0.1671337649062758, 0.596737390002501, 0.4240064062694966, -0.27019064872785326, 0.2511778005250963, 0.23476800665277264, 0.4748158911030591, -0.3059490942573546, 0.12618224890103852, 0.08036577912937934, -0.351661227251231, 0.06988277493012499, -0.38567048009983523, 0.04543069176350358, -0.17353400499072086, -0.3933023075891029, -0.6142902888891373, 0.08229820086859274, -0.09525231059242181, -0.4354580543867038, 0.17936107831999593, 0.4607468637929399, 0.21962637651130051, -0.46548748744169477, -0.052162639577241696, -0.0630256928365292, -0.7552438206191294, -0.3754435066371507, 0.3051602949462422, -0.8140361212330435, -0.5480470050167631, 0.07636149105123316, 0.06505790480586476, -0.20828392923085737, -0.13756943265558097, -0.4675887275999439, -1.1657428805501604, 0.3570546234210898, -0.06703696466720599, -0.10729046962294195, 0.07422773539468647, -0.4061669195505827, -0.13992522284230893, 0.4533838326782005],
                          [-0.3506346837201876, -0.09787873567380169, 0.19295892490770447, -0.22042679208709626, 0.22547294340078408, 0.1267085095927911, 0.22424189116501025, -0.24987309046665482, 0.30467864232867126, -0.0722707021423195, 0.3454821504122172, -0.9440856811754595, -0.04396846879013139, -0.11181511876343075, -0.4098592598744469, 0.5785991715946507, 0.19171566249478725, -0.4593161660526639, 0.33947128222495426, -0.7421984226403575, -0.17507629484873316, 0.7329289130038036, 0.06674070972771604, -0.8496071519957978, 0.3118844677965311, -0.5982837360962358, 0.7653763495933328, 0.4179661592935557, 0.1492965104352316, -0.9381102595281597, 0.623043684584713, 0.009295164452809002, -0.0995749869602669, 0.25123687124081473, 0.5487877981250012, -0.1466090873708568, -0.45605398297599864, 0.1800380279788105, 0.5042906790476966, 0.2157909619500347, 0.15363167542989156, 0.2826434968148127, 0.2965909025708344, 0.15442607396009717, -0.8457525727801335, 0.13418279238871464, 0.16821826844650942, -0.29029450629013953, -0.29650230495035784, -0.8896000254514321],
                          [-0.22034592656522736, 0.6575711901147396, -0.34057863738880423, -0.3044588946879625, -0.3359782624962514, -0.2597541103104576, -0.5038221458682891, 0.11467053999949535, -0.10110653471056381, 0.3120580683278598, 0.25457959037670236, -0.1912779691002992, 0.4253145312180039, 0.4473713733358283, -0.24067423477045452, -0.013387065126805889, 0.131228916730134, -0.7736943853051561, -0.2606301328537418, -0.03644605725055777, 0.3067402961153671, -0.562532130685486, 0.5636413205345274, 0.12813839131247856, -0.3990337648087093, 0.44596907824676846, -0.31876033307853613, -0.028920375811141135, -0.45031247593149515, 0.09465701018534871, 0.01963391836483823, -0.11351013097640052, 0.4229427212673715, 0.33714124543646573, 0.47027613006699853, 0.5196336697675233, -0.2370435469252906, -0.7143881204316707, -0.38832997039012146, 0.13413837742136586, -0.22919228752362367, -0.18765499583739012, -0.3731220877729565, -0.5510504479757143, -0.7310076551543734, -0.33207493722857223, -0.07251875933936804, 0.35938471881842415, 0.36980753872853805, 0.039794199844480266],
                          [0.34094376366506896, 0.15871669819888684, 0.2921662989567074, 0.19276103691497642, -0.3447800138557084, -0.33880685860625726, 0.29042160108095594, 0.3745267879861071, -0.29015970005929936, 0.7216070658062348, -0.5770566161551065, 0.23989484856689697, -0.02822760449823073, 0.17592988653174427, -0.5663660381255565, 0.1710432620915672, -0.2567907033964606, -0.12691310957110602, -0.5561582230281977, 0.2562539688527516, -0.2478562551143312, 0.257247261233691, 0.06759371040409645, -0.25594004693224726, 0.4435758338083674, 0.5506355545880653, -0.7534574820969095, 0.04463940699591452, 0.15559500026445133, 0.5447096646117222, 0.13029989774464998, -0.3013337908124959, 0.36753552884345275, -0.19792866767155315, -0.6142314736600496, -0.10604134132037155, 0.3522579254042862, 0.5369679293656754, -0.019115951093630748, 0.5759580724966678, 0.7646182787330368, -0.25063905799796166, 0.5474844521169036, -0.1910954034570894, -0.3591133951561522, 0.3184972626295098, -0.9698429462941864, -0.036988643219542, 0.037711207340114516, 0.4534184608708016],
                          [-0.3592003072507232, 0.34945621265034293, 0.3017630096371406, -0.6354984540530186, -0.35493335611379395, 1.0425091462351992, -0.9599790801863178, 0.1891264620532953, 1.029848162563226, -0.5116340382623273, -0.6112026689603465, -0.39710063435567844, -0.2165581384040784, 0.05568497037588005, -0.14183720255125798, 0.00688825828901682, -0.46131624208986505, 0.3871348521861414, -0.7623232498538878, 0.6988256499353195, -0.1284457291627474, 0.7224386465537798, 0.2273693459548224, -0.0920937388097065, -0.05659094106041461, -0.9968008564902192, -0.05478449367926245, 0.34413796743838143, -0.08676036122269612, 0.039015320166326424, 0.279696413427358, -0.12035517042377177, -0.30042959771022676, 0.33664069834349203, -0.02157983628839457, -0.14095068186992082, 0.8237663238302937, 0.1384293137458885, 0.44186717093960654, -0.36087396201573374, -0.03455506591632143, -0.37390626457899745, -0.28486587520644935, 0.15689069722404117, -0.21860879204070113, 0.6361776638795583, 0.09612755795350005, -0.052959520765673035, 0.425889314784931, -0.7800732201240195],
                          [-0.12844091760768098, 0.7475189185086785, -0.2921576704033426, 0.5934100130334803, 0.682049612367881, 0.40653971555867713, 0.3274885157671079, 0.333493972489824, -0.07251687961542368, 0.5524426198362536, 0.3001785308233555, 0.22702564871586545, 0.49768831322595536, 0.24581079283910903, 0.10087133687918666, 0.12059299059278358, -0.40940019759792684, 0.5635820637397949, -0.5066429717399782, -0.08478668563485514, 0.3889115125499588, 0.11413835015484339, 0.2227538538198756, -0.029298440488432928, 0.401485513996731, 0.1335609580001156, 0.1483498433670824, -0.02932174338809868, 0.3533155579888241, 0.2707264814608413, -0.02877153202857507, 0.07933597706590138, -0.14178043689383868, 0.08091234389172333, 0.3558187979270886, 0.7329410612508596, 0.06909435421404496, 0.7042021197919721, 0.7764007134662493, -0.7941662259997635, -0.3019836190311929, 0.18227230950282808, 0.18268034737748481, 0.4650619145954926, -0.023005987389326764, 0.4975708459558048, 0.4152381647250453, 0.04291074593290811, -0.2000532513909498, -1.088598925916401],
                          [0.20586772566762573, 0.018373377222856757, 0.05678464144515269, -0.5455318856146388, 0.34101026394896494, -0.5817224829613379, 0.13895769078281403, -0.23467393356247968, 0.36876502096394786, 0.06045732741627219, -0.0014599221011179567, 0.08508152859264638, 0.013218467680340466, -0.06594294695522082, -0.22905191150542642, 0.01935916807327604, 0.02692982070938195, -0.02881200183682095, -0.21445766638190492, 0.6689380109767367, 0.48288900553339703, 0.3986602593999314, 0.2643543479679796, -0.3816811208852836, -0.45786420403007094, 0.5754128324507106, 0.722699226175119, 0.11823818512688845, -0.34417787601945754, 0.6951013635745149, 0.646986749216809, 0.30007222348128426, 0.10822135549283865, -0.19936089361772102, -0.060364289228085545, -0.1400905418039587, -0.061247569089361076, -0.012949965110268274, 0.16190727289398438, -0.5768225758270256, 0.12425192496199716, -0.10180320012018729, 0.15162415198304874, -0.4037039772861033, -0.20639494957613563, -0.3415759807446827, 0.08389172334096695, 0.3935879247826305, 0.09005729201922874, 0.23361063557459305],
                          [-0.22834475109823368, -0.10080082702661702, 0.07832113198960515, -0.5767345570489761, -0.4233924025095062, 0.2102500787440606, -0.06884704980983927, 0.20137025416112378, 0.1903815788195195, 0.06476129209787346, -0.7611159518636367, -0.4512171558142873, -0.03909188049857943, -0.7449750734093424, 0.485969984656723, 0.3296130222589124, -0.2730667953204336, -0.48583726454390663, -0.4302509685141079, -0.035142582117139734, 0.08478101677866383, 0.10238455980564527, 0.033423258224677524, -0.5904698379445529, -0.19594744085548282, 0.31932052204157285, 0.7090265080846203, -0.204949239743339, 0.019465357331784797, -0.03652856293116031, -1.5105323809582576, 0.09060938826259639, -0.29603823587470307, 0.07653006647474843, 0.14062208044537589, 0.26661317672233875, 0.3798275871445923, 0.18318627823058975, 0.25910244859666265, -0.3295240806809115, -0.21725549426753174, -0.12324227568486448, -0.8165640076160668, -0.10002748066358821, -0.18456996212693905, 0.6752945475444667, -0.8029978550902679, 0.9301898706854754, -0.160753711305583, 0.23852436241277153],
                          [0.2737627879408631, 0.09375697731341068, -0.8236719804278254, -0.03182230569315023, 0.5651815219879027, -0.6144000098672959, -0.3307845429372558, -0.5469372072476827, 0.6452963990792743, -0.17001955558407336, 0.13162287878356618, -0.2708579341334115, 0.4557175242534564, -0.1163122751329813, -0.0732811379233919, -0.29730440237098, 0.08028047744427529, 0.16718279175903708, 0.6385757072599514, -0.5300696890828462, -0.5605028269880394, -0.5576585967737498, 0.44828192757643476, -0.07050728583490486, -0.8133362672786782, 0.12728183919904512, -0.9215926552721935, -0.8720081389499285, 0.7447750821618558, -0.5412855768355316, -0.17445429607880666, 0.10694443791912965, -0.32353739879940235, -0.2937380663777035, -0.32522852982229916, 0.561585005866488, -0.6025152423853436, -0.1246753685759863, -0.05896118069037057, -0.5884988441185637, 0.3200065066280484, 0.6723324447994803, 0.721537671277729, -0.6729730865235479, -0.2961751354410577, -0.5655093083517845, 0.11979686409892107, -0.06346205802388871, 0.8674983435186221, -0.15940994387185195],
                          [-0.36236481429582545, 0.5393214656325572, -0.0718078380244277, 0.4845471648412879, 0.4381368225287779, -0.4010830385551783, -0.3887932363917448, 0.37372183091406797, 0.12832149749895846, -0.07713278193962786, -0.25616662508991694, 0.039501266972359214, 0.12602143454921796, -0.059466848360897974, -0.09728301748273127, 0.1216813118325159, 0.6469003952299421, -0.79944644200391, -0.053924621261515, -0.16900057667143334, 0.9305048192950457, -0.36767658238944434, -0.16825626092304327, -0.7698605224858603, -0.3723989490161321, 0.5989341613640461, -0.062482682657104216, -0.5601576260386201, -0.15828885430045248, 0.4709685406784456, 0.16169647238206492, 0.2250069399127026, 0.10865379032068335, 0.48897900566171726, 0.55130504811078, -0.48928901293248334, -0.35353834498528647, 0.03902854810198714, 0.21395848702457826, 0.18234632841696682, -0.6485977257503781, -0.015105641664224134, 0.695471554088321, -0.5279244757414165, 0.15513371634620193, 0.03608088887168786, 0.40688982759110576, 0.10363178517768709, 0.7623121360750993, -0.2137843012966924],
                          [0.09466301016408946, -0.4669454615440216, -1.110656546325695, -0.03168460748598208, 0.2777520075572462, 0.9195927496708519, -0.4778006385369732, 0.45100505541228586, 0.6530081545636752, -0.16301086989147254, -0.3351844910119542, -0.47628290166169884, -0.6079656767999138, -0.04773437964102503, 0.30310177169886754, 0.19999590124435856, -0.27140193005053426, -0.18931298463464752, -0.1562075284992411, -0.4732263055206662, -0.22555283105831833, -0.11065665790595744, -0.4145774089068626, 0.4343491455972791, -0.2678039417004698, 0.19118277188267319, 0.17331319764963984, -0.399322298396102, 0.15728235829572823, 0.7353197917928719, -0.07851564928002376, -0.4900071957382111, 0.3993978631836946, -0.26359466426021044, -0.09485344364614617, -0.5182285469724363, -0.6863224487932507, 0.04660915581831833, -0.22956332821735623, -0.8795527832766731, -0.06622779229246092, 0.20960988617495385, -0.8195747444906929, -0.6459391968409125, -0.5506537792479176, -0.0780014846533772, 0.633391692952092, -0.011965012704532027, -0.579151139196901, -0.8085980712111223],
                          [-0.018716128687263453, -0.2068484256081694, 0.14155160817128246, -0.2290539308998183, 0.6933665586728099, -0.0566241393851843, -0.45827584751240064, -0.48388885293252887, 0.5102537820139141, -0.16447466706981992, 0.16206324906914335, -0.42226653544627024, -0.16857036422739763, 0.32902831832589596, 0.5852747502762171, -0.07866807008550211, -0.7929185216317433, -0.5798610270801637, 0.44892911367137356, -0.17861985315311169, 0.7794615702285304, -0.038800108680097534, 0.024399708626405878, 0.801837277481463, 0.13600155013641255, 0.03403316034953912, -0.47089744602676137, -0.28160028431205947, 0.8630615357479468, -0.06767573290244155, 0.759374647988992, -0.3487565595544601, 0.374884874994439, 0.46996719832581685, 0.5236762317681618, -0.10326227156791679, -0.43257886282853386, 0.2755992721235898, 0.16310496168147148, 0.13460741105518487, 0.03315356026802144, -0.5308601057234874, -1.1070559742816275, 0.3460594947737372, -0.2808501331027445, 0.47339848826966596, 0.17098232647701178, 0.4657303315303336, 0.3436292393040765, 0.05423511241425388],
                          [0.10486816471548979, -0.3276415790621764, -0.1879382728030744, 0.42007484696183806, 0.609547872732935, -0.1222551412919986, 0.8373913594935853, -0.5892688602422977, 0.11664903132287632, 0.4554295938413083, -0.06619108857199589, 0.07487241247368295, -0.24153331758085186, -0.1518963140574025, 0.697572139244728, -0.36459882655706, 0.0035526634341021823, -0.6247562411743297, -0.17252373507877194, 0.7532681548262918, -0.5953227049669215, -1.274930707943646, 0.35598114359285987, -0.17925254130309476, 0.28201593756821897, -0.40886876795582633, -0.5985945488021863, -0.6739472914826616, 0.2899723427470818, 0.5264428711964556, -0.3195544117046103, 0.04493980946242427, -0.773837166448111, 0.3177858468232254, -0.2595928808929745, -0.08322506707452429, 0.8724646807262092, -0.35346573303671613, -0.4939701943804169, 0.3772718312864656, -0.365726426293347, 0.002497761806304226, -0.05379139394483701, 0.5578374524987851, 0.15194409142307794, -0.16063048056143908, -0.15477774057579663, -0.4497894445537028, -0.17762320861156305, -0.26998381593051846],
                          [0.22448780975982424, -0.07949076110867365, -0.438289800371133, -0.021696890390899955, 0.8235228259438562, -0.18295301628911398, 0.8799585104967069, 0.1899545198345368, 0.41575789142691305, 1.0049515027618992, -0.2701186077405746, 0.23056805375008166, -0.6480183659566743, 0.8011352819120827, 0.3464956898016361, -0.28746397070907037, 0.8759024178980583, 0.024114558034364984, 0.018195421786230837, -0.6858387581368254, -0.2593482877279996, 0.21595058321490762, 0.15473405302466336, -0.4151315528645297, -0.04100828017464697, 0.2903891146452775, -0.3219414650889681, 0.5526070154052517, -0.1970153591893241, -0.13487057012174233, 0.13554245085405658, -0.1514390941375672, -0.26962627443316156, 0.5316718245877833, 0.04273199288401847, -0.2104826090193455, 0.6500562868785364, 0.2628860981339528, -0.12969983112033345, 0.29586669133255333, 0.13575785295382614, -0.19098292233696712, -0.10189479108381228, 0.3229704419280089, 0.2112487524739989, -0.44478290177476854, 0.24069866227611045, -0.3285541513528193, -0.18158412516291614, 0.026929587579938734],
                          [0.20314011609178168, -0.15884462870753158, -0.415652974823897, 0.26018235392655664, 0.07875409240247692, 0.18991915794256772, -0.4683771763874668, 0.12832512400751125, -0.10363261871246143, 0.09586523829397225, -0.05096391616610262, 0.06783051698091279, 0.04394216061041992, 0.6468155171695533, -0.39115937427960135, 0.28875257662470255, 0.24096063718718547, -0.1540151504949698, -0.25875536058014637, -0.06444175296548249, -0.2651669182347367, 0.39381898797698967, 0.006700842188755413, -0.09501659449469685, -0.5158497968599519, 0.2000582615841661, 0.059741629146875794, -0.23930824534873785, -0.3065561424361429, -0.33547624525304537, 0.7177041442542905, -0.7902032871951694, -0.11891937021504481, -0.007641047124201231, 0.1742338405473041, 1.155699745581287, 0.03643658620705075, 0.14109380606162633, -0.38414866471810544, -0.5062735916364374, 0.39152619580670367, -0.7551873250180948, 0.20962659146252896, -0.04368942459065895, 0.17495113845938542, 0.5042357795061007, 0.5098906004556297, -0.05154161686422733, -0.4562193659540469, -0.11065786402764481],
                          [0.4874877748468512, -0.022848217132146673, 0.2037051813442085, -0.40487639650957397, 0.18036215539397113, -0.0928262375446875, -0.47368714521357236, -0.6005762600611048, -0.2103124309721142, -0.2720882319644949, 0.10427916694750278, 0.3933577294927192, -0.7068246506109195, -0.20598937268290127, 0.47326679734602073, 0.5665491396708165, -0.4994418197903304, 0.2228358660516532, -0.2077685385879769, -0.01880070280590769, 0.05156725301579768, 0.861451429657478, -0.42080049089361615, -0.15565590883275043, -0.41598660454902736, -0.2788742263655957, 0.19252862433239523, -0.6376302874659537, 0.25602314482423116, -0.4932293535287224, -0.6446416201407679, 0.8959764634345114, 0.36510642276755445, 0.0229572927525218, 0.5240821319200871, -0.5621898539781962, 0.30361035603571296, -1.1047206759719934, -0.5649483651364029, 0.3704461579510216, 0.2844690803203772, -0.22229841849187545, 0.5282752124310237, -0.12372107042596026, -0.9362482317110103, 0.05606218272793535, 0.019695455458168476, 0.4475472396349657, -0.18074278309630284, 0.2472459590268632],
                          [-0.7497403808687935, 0.45181592816045596, -0.49706233107615594, -0.024703700119880614, 0.23540469009330528, -0.7232877775412503, -0.5988330109672428, 0.6806845986443428, -0.5535295965886434, -0.15646016533868673, -0.10529485823809187, 0.23535235295051482, -0.34921504968940364, -0.10915948475300774, 0.4678270246843575, -0.13694744601120534, 0.4505286257664257, -0.03173016202857054, -0.29688933431274156, -0.3386970818137529, -0.1829938726681689, 0.32125804253894397, -0.3250720062563206, -0.6879130484006923, -0.08086963531608603, -0.0764862684221165, 0.1812715844986636, -0.017851440000193543, -0.16307178017054377, 0.21954606866175458, 0.11938598506229833, -0.5005731273599778, -0.7765860901543187, 0.06978301326666433, -0.6957368067196552, 0.5020255202972366, -0.3475974643532, -0.3009715876667809, -0.7088737040040791, -0.42361028298981673, 0.3698439724981827, 0.42648242666049885, -0.5995279206792671, 0.13696550253979686, -0.09357595614136724, -0.7695665829710573, -0.23338658595790435, -0.1945902574150057, 0.33850665579464684, -0.4956200518340297],
                          [-0.047396300021601015, 0.5394573157884913, 0.4621087770604601, -0.01342115732461692, 0.1945082313646345, 0.3320540509974952, -0.015215262131489061, 0.7624707212395234, 0.2988412195537388, 0.6824893056060876, 0.0023465718005811747, 0.7851674713259361, -0.31639904843183575, 0.1366440571840963, 0.6017103556104895, -0.05493549923419728, 0.20513579882032298, -0.2531107669495467, 0.07283002094017892, 0.31909762370549594, -0.22573501394068848, 0.1708977912359225, -0.009914771193765402, 0.847485245470789, -0.43315308303114086, 0.3064604312242441, -0.027705025310524028, 0.26052814479811437, -0.06242601359420183, 0.14747153992278206, 0.36578986353707116, -0.5747808943292101, -0.582507487614319, 0.17138751074382375, -0.7061609999145355, 0.4670165348476047, -0.5624413629809072, 0.22342784579046285, -0.18518607676044604, -0.11506852500730648, 0.09543120377974368, -0.23105148424057514, 0.10021213972148846, 0.29180664695271785, -0.3424999888527675, -0.3336257481879851, -0.22308278296879547, 0.5132728224655192, 0.031173277851889204, -0.13072412646921538]])

    def __init__(self, num_variables, phenome_preprocessor=None, **kwargs):
        self.is_deterministic = True
        self.do_maximize = False
        self.num_variables = num_variables
        self.min_bounds = [-0.5] * num_variables
        self.max_bounds = [0.5] * num_variables
        self.weierstrass_function = WeierstrassFunction()
        bounds = (self.min_bounds, self.max_bounds)
        preprocessor = BoundConstraintsChecker(bounds, phenome_preprocessor)
        self.matrices = {2: self.matrix2D, 10: self.matrix10D, 30: self.matrix30D, 50: self.matrix50D}
        TestProblem.__init__(self, self.objective_function,
                             phenome_preprocessor=preprocessor,
                             **kwargs)


    def objective_function(self, phenome):
        phenome = np.array([phene - offset for offset, phene in zip(self.offsets, phenome)])
        phenome = np.dot(phenome, self.matrices[self.num_variables])
        assert len(phenome) == self.num_variables
        obj_value = self.weierstrass_function(phenome) + self.bias
        return obj_value


    def get_optimal_solutions(self, max_number=None):
        return [Individual(self.offsets[:self.num_variables])]



class F12(FletcherPowell):
    """Fletcher and Powell's problem."""

    bias = -460.0

    alpha = [-2.028, -1.5589, 0.7774, -2.0752, -0.1601, 1.0811, 1.408, -1.6129,
             2.419, 2.217, 0.5244, 0.7151, 1.7674, 2.0937, 1.5349, 0.0433,
             -1.9471, -2.786, -0.1282, 1.2721, -0.1741, -0.012, 1.683, -1.7042,
             2.9255, -0.1696, 2.2797, -0.8142, 1.5232, -0.2444, -1.5589,
             -1.7701, -0.3173, 0.0825, 0.4034, -1.1047, -2.7235, 0.8031,
             -0.2834, -0.6032, -1.6169, 1.5346, 2.7228, -2.5399, -1.4283,
             -0.5822, 0.7624, -2.2955, 1.6961, -2.5261, -1.6655, 1.8325,
             2.4668, 2.3259, -0.1323, -2.6112, -2.3339, -2.3877, 2.598, 0.8058,
             1.4093, -1.2263, 2.8888, 1.1118, 0.7998, -2.0294, 1.2477, 1.1918,
             0.4183, -1.0449, 1.7843, -2.0655, 2.4799, -1.26, 1.6222, -1.8351,
             0.0099, -2.5411, 2.331, -2.6963, 2.4796, -1.9349, 0.2676, -1.6177,
             2.0879, 0.5869, 2.0894, -1.4914, -0.012, -0.4632, -2.3644, -2.1405,
             1.2302, -0.619, 2.7602, 0.5755, -2.6869, -2.3987, 1.945, 1.7011]

    a = [[79, -66, 21, -94, -60, 68, 92, 63, -5, 90, 5, -43, 64, 59, 59, 47, -2, 37, -41, 52, 30, -18, -51, 94, 13, -73, 41, 65, -55, -97, -57, 16, 77, 40, 45, 17, -20, 91, -46, -92, -27, -17, 85, 84, 56, 63, -5, 51, 2, -44, 70, -88, -39, -94, 29, -1, -10, -59, -92, 17, 96, -50, 12, 95, -56, 47, -55, -45, -46, 8, 71, 51, 51, -36, -73, -72, 59, -79, -93, 67, -17, 83, -56, 28, 94, -20, -83, -87, 56, -96, -52, 54, 13, 24, 27, 74, 86, -24, 11, 79],
         [-18, -48, -15, -67, 63, 42, -75, -59, 0, 30, 55, -18, -80, 0, 78, 80, 9, -25, 95, -23, -84, -30, -31, 72, -24, -39, 63, -15, -24, -9, 60, -97, -72, 75, 67, 47, 75, -87, 66, -13, -64, -8, 97, -50, -47, -85, 67, 84, 14, 80, 10, 53, 21, 56, -1, 7, 4, -69, -27, -76, -4, -30, 29, -46, -88, -56, -63, -78, 71, 51, 97, 95, -42, -96, -55, -71, -97, -73, 49, -11, -2, 16, 21, -3, 38, 97, 27, -64, -66, 60, -54, -92, 58, 37, 0, -70, 58, -77, 47, 25],
         [-65, 62, -61, 43, 92, 49, -27, -10, 0, -57, -83, 32, -26, -84, 79, 1, 42, 35, -92, 31, -25, 68, 11, -91, -50, 67, -44, -86, 19, -64, -16, -13, -20, 47, -93, -49, 75, -23, -7, -65, 82, -37, -57, 7, -88, 4, 12, -57, 1, -37, -83, -67, 4, 85, 55, 19, 48, 72, -27, -43, 68, 58, -51, -11, -5, 32, -62, -97, 38, 58, -13, 23, -96, -84, 53, 55, -35, -76, -54, 52, 45, -34, -56, 11, 24, -19, -45, -89, -24, -49, -65, -11, -26, 45, 36, 71, 33, -42, -61, -64],
         [-65, -30, 86, -52, 4, -13, 64, -52, 69, -89, 0, 4, 6, 19, 1, -31, 60, 95, -41, -3, 36, -63, -37, 16, 11, -37, -7, 16, 32, -81, 76, 86, 32, 95, -15, -50, 15, 15, 37, 90, -56, 89, -13, -36, 48, 10, 74, 10, 51, -58, -41, -30, 35, -10, -16, -30, -23, -33, 14, -62, 27, 35, 61, 89, -10, 62, 87, -70, 51, -63, -11, 3, -57, -60, -59, 74, -9, 17, -29, 17, 36, 82, -35, -9, 59, 58, -31, 62, 65, 38, 5, 5, -55, 77, -76, -85, -25, 28, 75, 11],
         [23, 59, -17, 22, -98, 51, -35, 76, 24, -94, 15, -32, -48, -52, -70, 35, -25, 70, -92, 28, 54, 94, 66, 72, -34, 14, -45, 54, 49, -42, -48, 80, -74, -91, 48, -33, 60, 22, 77, -84, -56, 52, -45, 75, -9, 94, 45, 15, -47, 12, -55, 28, -78, 3, -23, -44, 3, -50, 55, -32, -73, -29, -19, 29, -9, 23, -5, 83, 76, 79, 92, 35, 62, 24, -49, 50, 92, -82, 65, 28, -95, 1, 9, -29, -27, -99, -1, 38, -55, -49, -53, 32, -50, -88, 95, 65, 79, 89, 19, 58],
         [87, 5, 63, -18, 97, -9, -53, -77, 46, 78, -80, -30, -5, -17, 40, 5, -48, -10, -51, 22, 58, 75, -65, 15, -33, 41, -51, 67, 99, -28, 67, 53, -66, 81, -91, -17, -32, 23, 60, 51, -28, 92, -96, 25, 38, -82, -64, -84, 61, -22, 65, 66, -69, 55, 35, -45, -23, -19, -78, -79, -69, -34, -77, -90, 54, 63, 47, -70, 65, -54, -53, 11, -20, 25, 11, -16, -70, 68, 84, -22, -33, 78, -98, 93, 5, -36, 59, -24, -42, -42, 15, -7, 88, -35, 12, 14, 23, 54, 76, 39],
         [58, -7, 26, 20, 21, -53, 80, 17, 9, -78, 31, -86, 53, -78, -38, -32, 19, 49, 91, 12, -58, -11, -97, -28, -77, -84, -68, -38, 76, 25, 46, 3, 0, -68, -95, 11, 48, -43, 93, 48, -91, 84, 3, -20, -39, -46, -7, -23, 24, -42, -23, 62, 75, -62, -93, 75, -93, -42, -79, 33, -51, -37, -34, -80, -53, -54, 8, 5, -75, 51, -85, -40, -89, -56, 61, 23, 15, 66, -48, -57, -43, -77, -49, 66, -54, 24, 61, -47, -75, -79, 58, -20, 26, -68, 26, -74, 86, 58, 10, 18],
         [37, -73, -44, 47, -14, -73, -2, 3, -98, 46, -95, -42, -89, -83, -24, 92, -13, -22, 22, 54, -19, -83, 13, -12, -38, -100, 92, 49, -59, 52, -15, -2, 36, 53, 2, -79, 10, -49, 82, 97, -35, -17, -77, 71, -86, -2, 13, -10, -28, 14, 15, -95, 21, -4, -15, 73, -85, -64, -57, -80, -62, -71, 86, -96, -94, -95, 56, 61, -83, -60, -79, -31, 63, -4, -8, -89, -40, -69, 26, 33, 74, 77, 76, 38, -27, -16, 52, -47, 63, -63, 87, 53, 2, 93, 55, 24, -19, -85, -76, -33],
         [5, -2, -5, 63, 65, 97, -28, -23, -26, -59, 53, 13, 87, 72, 54, 42, -85, 97, -13, 58, 14, 41, 30, -60, -2, -71, 95, 34, -33, 26, -39, 50, -28, -12, -11, 45, -35, 71, -71, -94, 50, 31, -10, 7, -17, -41, -80, -77, -43, -32, 20, -40, -89, -58, 61, 70, -19, 3, -5, -46, 22, 63, 30, -33, 66, -52, -40, -67, 14, 93, -35, -25, 4, -51, 35, 17, 68, -26, -94, -74, -60, -74, 74, -3, -76, 11, -32, 68, -47, 30, 77, -9, 10, 10, 70, 59, -39, 97, 27, -68],
         [-85, -51, -71, 94, -58, 89, -19, -80, -12, 29, 46, 58, 68, -85, 31, 45, 50, -77, 13, -79, 66, 33, 26, 70, 25, 41, 49, -72, -33, 86, 61, -34, 72, -28, 20, -31, 5, 12, 85, -56, 42, 94, -53, 73, 11, -43, -94, -12, 21, -29, 14, -42, 10, -80, -77, -80, -45, 85, -96, -77, -89, 44, -26, -40, 28, -56, 60, 93, 90, -2, -68, 29, 83, -53, -64, 74, -61, 45, 32, 70, 73, -72, -9, -63, 30, -51, 57, -23, -53, 64, -57, 34, 95, 70, 16, 58, -69, 74, -47, -78],
         [-76, 41, -75, 25, 58, 20, -35, 24, 31, 0, 95, -57, -44, -60, -9, -22, -71, -19, -71, 41, -97, 96, -15, -98, -78, -46, -44, -72, 53, 61, -47, -72, 34, -46, 43, 43, 58, -60, 51, -76, -63, -57, 15, 10, 84, -53, 3, 26, -31, -65, -93, -2, -33, 84, -56, 88, 88, -34, -87, -24, -28, 94, -70, -64, 100, 61, -78, -93, -75, -32, -57, -87, 56, -13, 95, -93, -6, 25, 5, -45, 48, 20, 72, -57, 45, -92, -86, -20, 6, 17, -90, -82, -98, 98, 75, -57, -65, 4, 85, 15],
         [-30, 21, 35, 91, -94, 36, 25, -75, 75, -90, 69, -88, 72, -82, -56, -11, -73, -25, -18, 64, -75, -29, -42, 71, 2, 67, -90, 63, 63, -66, 3, -54, 91, 62, 67, -94, -15, -97, -45, -80, -80, 18, 36, 17, 38, 27, 64, -17, 65, 34, -89, 70, 45, 30, -72, 44, -83, 69, -14, -52, 70, 78, 49, 66, 20, 62, -80, -5, 58, -57, 81, 16, -93, 81, -32, -50, 39, 40, 42, 65, 91, 58, 59, -60, -99, -58, 33, 73, 98, 46, -54, 60, 45, -65, 88, 3, 92, -63, -92, -2],
         [-92, -13, 67, 44, -16, 10, 34, -64, 55, 30, 89, -12, -31, -60, -68, 75, -30, -31, -1, 26, -7, -53, 13, -88, -69, 34, 52, -98, 31, -18, 82, 15, -99, 59, -59, -86, 25, 10, 73, 24, 74, 2, 66, -7, 52, 79, -12, 13, -63, -69, 65, -99, -73, 30, 84, 20, 77, 62, 9, -34, 88, 21, 66, 87, 80, -74, 2, -60, -8, -22, 83, -62, 72, -29, 37, 5, 18, 88, 32, 98, -77, -24, -82, 76, -67, 17, 30, 65, -78, 58, -31, 15, 36, -8, 47, 49, 55, 41, 12, 24],
         [95, -69, -1, 21, 30, -35, 83, -52, 18, -81, -94, -51, 35, -70, 39, -1, 73, -55, 21, 48, 30, 4, -33, 5, 97, 73, 90, 1, 55, -64, 67, 80, -72, 77, 68, 59, -59, -89, -80, 67, -12, 57, 64, -37, 62, 90, 71, 65, 53, 40, 0, -75, 6, 66, 18, 30, 79, -77, -48, 62, 91, 14, -13, 97, -39, 52, -10, -79, 65, -96, 1, 55, -20, -46, 90, 55, -2, 10, 70, 21, -61, -60, 7, -91, -66, -79, -26, -85, 15, -97, 73, -84, -77, -59, -44, 7, -12, -23, -84, 18],
         [-93, -55, -38, -15, 50, 25, -94, 80, 45, 38, -59, 47, 95, 42, -81, 59, -38, -24, -77, 93, 42, 44, 39, 76, 57, -99, 85, 17, 10, -69, -91, 70, 31, -51, 15, -6, -85, -47, -38, -82, -54, -15, 62, -68, -34, -35, 37, 32, 45, -94, -58, -19, 74, 3, -4, 33, -61, -58, 58, -53, -35, -42, 88, 20, -14, 63, 23, -79, -23, 50, 67, -31, -63, -8, 63, 55, 87, 56, -18, -82, -95, -67, -62, -46, -76, 45, -34, 14, -25, 16, 84, 23, -10, 11, -93, -4, 68, -4, 50, -97],
         [54, 38, 80, -43, -86, -45, -97, 45, -98, -30, -29, 33, 72, -13, 34, 24, 16, -53, 31, -49, -63, -2, -51, 65, 3, 2, -51, 42, 48, 94, -55, -7, -2, -94, 64, -6, 100, -76, -27, 0, 75, -91, -43, -22, 100, -13, 11, 41, -2, -15, 91, -1, 78, -85, 69, -58, -39, 64, -29, 91, -83, 97, -56, 13, -54, 27, 32, -41, -14, 83, 46, -1, 69, 42, 79, 10, 17, -41, -72, -89, -32, -36, 72, 59, 21, -73, -78, 42, -86, -5, -53, 17, 51, 72, -48, -76, -48, -59, -49, 35],
         [26, 80, 15, -96, -19, 83, -17, -12, 71, -49, 29, -69, 85, -88, -11, 54, -77, 81, 44, 76, 60, -74, 22, -81, 26, -94, -46, -50, -21, 87, -71, -92, 0, 37, 13, 85, 82, 34, 17, -54, -59, 8, -43, 72, -12, 6, -96, -72, -66, 84, 38, -17, 92, 34, -77, 77, -69, 9, -25, 25, 9, -85, -19, -4, 75, -16, -65, 18, -86, -65, 10, -31, -58, -66, -54, 85, -99, 46, -13, 17, 42, 56, -42, -96, -34, 89, -27, -14, -40, 65, 40, -30, -72, -39, 86, -80, -26, -92, 19, -49],
         [6, -73, -63, -94, 96, 28, -8, 41, 52, -75, -80, -58, 93, -19, -10, -44, -84, 86, 55, -32, 36, 47, 39, 86, 35, -60, -51, 74, -84, -8, -19, -48, -21, 58, -12, -33, -7, -47, -38, -62, -20, -53, -87, -94, 68, -83, 100, -90, 81, 92, 34, -47, -50, 36, -77, -68, -83, -55, 73, 49, 5, 13, 18, 51, -49, 64, 34, -76, 82, -63, -74, -8, -32, 13, 10, -98, 81, -40, -65, -89, -81, -20, -39, -11, -95, -9, 48, 17, -29, -45, 73, 97, 61, 88, -95, -84, 69, 98, 32, -67],
         [-40, -15, 96, -73, -78, -21, 32, -31, -34, -43, 76, -14, -78, 11, -22, 14, -4, 57, -65, 75, 55, 31, 7, -36, 86, 99, 5, 64, -67, -40, 10, 29, 26, 77, 10, 18, 90, -30, 45, 73, 72, 52, -47, 2, -42, -16, -79, 78, -15, -94, 95, -2, 98, 70, 69, 56, -18, 95, 49, 75, 12, -23, -19, -62, -59, 17, -44, 47, 62, -60, 78, -3, 87, -31, -5, 69, -77, -46, 62, 50, 13, -2, 7, 43, -76, 4, 31, 21, -51, 52, -59, 83, -17, 43, -97, -52, 80, -75, -33, 93],
         [-82, 30, -8, 78, 44, 42, -41, -17, 5, 82, 1, 72, 75, -35, -93, -38, 17, 13, -98, 92, -30, 76, -31, 71, -56, 21, 54, 68, -6, 17, -66, 75, 97, 89, 90, 60, 54, -2, 43, 85, 71, -34, 63, -24, 8, 34, 2, 12, -84, -53, 8, -10, -59, 3, 79, 33, 26, 84, 56, -13, -69, 16, -13, -2, -77, 90, 32, 91, 55, -97, -13, -80, 65, 81, -98, -12, -47, 52, 97, 52, 90, 63, -23, 68, -4, 28, 17, 20, 43, -63, 80, -33, 62, 26, -1, 12, -52, -87, 100, 82],
         [20, 21, -36, -5, 46, 16, 50, -7, 82, 10, -99, 45, 73, -72, 73, -36, -75, -84, 58, 10, -21, 80, -24, -27, -1, 90, 73, 6, 56, -72, -6, -99, -89, 47, -2, 8, 36, -13, -91, 86, -77, -57, 33, -7, 47, 3, 41, -94, 46, 55, -88, -43, -82, -3, -80, -99, -93, -25, 12, 52, -86, 18, 65, 90, 92, 86, -15, -32, -25, -88, -6, -10, -96, -68, 46, -91, -45, -37, 39, 66, -38, 32, 21, 46, -56, 33, 45, 40, 75, 57, 38, 27, -67, -80, -28, -20, 52, 51, 48, 3],
         [-64, 78, 65, 59, -80, 39, -68, 50, -36, 6, 49, 70, 41, 45, 94, 54, 79, -72, -5, 37, -44, 14, -53, -57, 64, 41, 44, -69, 63, 9, 23, -59, -9, -48, 62, -11, -57, -29, -71, 20, 80, -3, 100, 33, 56, 50, 36, -36, -55, 82, -90, 9, 79, 47, -86, -38, -63, 57, -29, 62, 53, -69, 1, -78, 10, -71, -92, -72, 63, -33, -10, 40, 18, 40, -46, 85, 75, 91, 35, 66, -32, 58, 65, 85, -72, 62, -93, 12, -88, 65, 57, -27, -48, 18, -62, -82, -58, 76, 88, -77],
         [-77, 61, 65, -55, 57, -3, -7, -8, -90, 5, -9, -56, -21, -51, 65, 17, -72, -23, 84, -53, -72, 14, 91, 83, -8, -50, -92, -63, -51, 6, -79, -93, -2, -64, -30, 98, -66, -4, 12, -31, 42, 62, -42, -60, 87, 85, -64, 31, -35, 69, 65, -62, -39, 30, -85, 73, -42, -56, 52, -71, -97, 97, 66, 63, 43, -77, -18, -43, -21, -65, 3, -69, -92, 49, -28, 78, 19, -23, 29, -29, 78, -17, -16, -92, 15, 5, -37, -45, 78, -26, -53, 62, -96, -63, 81, 7, 12, 30, 85, 52],
         [-23, -19, 47, -30, 76, -25, 72, 88, 65, 17, 23, 87, -70, 6, 29, -49, -14, 79, 79, 35, -66, -99, -16, -84, 23, -67, -71, -25, 85, -46, -81, 26, 41, 96, 83, -8, -91, 72, -3, 40, 100, 85, 29, -47, -97, 36, -74, -87, 76, -39, -47, 35, 10, -100, -18, -98, 53, 16, 42, 49, 19, -6, 86, -62, 18, -21, -10, 89, -14, -30, 56, 61, -15, 80, -7, 37, -25, -61, 12, 48, -76, -7, -16, 17, -84, 39, -92, -19, -67, -20, -71, 78, -37, 68, -39, -45, 26, 52, -72, -94],
         [-42, -42, -10, -96, -91, -54, -39, -49, 64, 1, 66, 13, 7, 53, 94, -91, 85, -81, 17, 51, 16, 7, -81, 89, 16, -5, 10, -10, -63, -58, 68, -44, 99, 58, 71, -21, 35, 92, -85, -61, -64, 97, -6, 86, 56, 47, 60, 47, -28, -51, 88, 25, -32, -35, -24, -97, -81, 32, -21, 59, -24, -73, 31, 46, -69, -2, 49, -56, -98, -12, 74, -84, 90, 59, 45, -87, -61, -19, -93, 76, 51, 76, -63, 87, -11, 8, -80, 54, -83, 9, 25, -79, -92, 18, 37, 99, -32, 53, -37, 88],
         [87, 52, 15, -85, 53, -45, 30, -11, 17, -99, 80, -56, 65, 70, -45, -56, -73, 51, 93, 5, -44, -56, -30, -1, -69, 27, -25, -25, 81, 29, 66, -24, -58, 71, 77, -74, 63, 97, -58, -26, -27, 26, 90, 46, -36, 58, 50, 83, -60, 28, 83, 37, -90, 36, 95, 54, -43, 40, -4, -33, -18, -59, -28, 62, 93, -69, 44, -75, -64, -22, 65, 69, 14, -36, -90, 37, -76, 41, -16, -31, -8, -92, -86, 77, 13, 34, -5, -48, -34, -87, -60, -63, -24, 52, 50, 99, -78, -61, -95, 8],
         [34, 67, -33, 51, 98, 34, -36, 67, -84, 86, -66, 4, 16, 52, 79, 41, 84, -19, 32, -64, -78, -76, 68, 79, 96, -78, 6, -31, -21, 79, -83, 82, -41, -60, -100, 17, 91, -96, -94, 32, -6, -36, 10, -53, -10, -96, 18, 1, -49, 12, 98, 65, -25, 96, 8, 66, 97, -58, 70, -16, -9, 6, -18, 38, 51, 19, -35, 10, -22, -39, 84, 47, 46, -1, 76, 65, -6, -25, 41, -9, -84, -24, -10, 79, 29, 99, 37, 61, -31, -54, -74, 42, 20, 80, 50, -24, -73, -47, 42, -1],
         [-16, 66, 13, -26, 85, -30, -43, -74, 61, 31, -2, -32, 60, 62, 76, 46, -34, -38, -20, 41, 81, 20, 84, 35, -70, 16, -4, 9, -41, 27, -95, 57, -11, -96, -42, 28, -63, -6, 15, -7, -13, 31, 74, 53, 35, -21, -25, 51, 61, -60, -91, -10, -51, -82, 66, 12, 92, 13, -14, 51, -52, -76, 17, 64, 64, 41, 83, -71, -78, 74, -1, 72, 2, 86, 49, -14, -10, -61, -72, -30, 49, 51, 77, -79, -98, 21, 27, 79, -58, -10, -92, -17, 4, 69, 45, -9, 65, 77, 14, 64],
         [-65, -34, -65, 48, 81, -12, -54, -70, 63, 1, 15, 2, -27, -23, 58, 75, -54, -41, 15, 61, 93, 40, -65, 48, -70, 18, 49, 50, 65, -7, 3, -97, 7, -51, 78, 14, -22, -24, 88, 66, -90, -18, 38, 77, -58, 14, -53, -22, 70, 45, -79, 76, 68, -61, 76, -89, -57, -7, -78, -9, -27, 42, 96, 11, 4, -35, -82, 5, -47, -44, -8, -16, 12, -69, 18, 2, 53, 79, 82, -8, -17, -13, -21, 6, 61, -34, -30, -8, -63, 80, 52, -35, -9, 44, 81, 83, 94, 52, 35, -41],
         [94, -22, 35, -74, 69, 22, 88, 4, -97, -99, -98, -48, -60, -48, 53, -9, -56, -92, 55, 30, -8, 44, -92, 43, -45, 68, 90, 1, -3, -72, -92, -89, 41, 80, -30, -88, 99, 42, -57, -87, 5, 55, 77, -94, 23, -59, 45, 61, 24, 58, 70, -57, 77, 85, 83, 82, -72, 2, 72, -22, -98, -7, 6, -22, 99, -96, 99, -88, 89, -39, 41, -1, -5, 10, 30, -92, 50, -60, -95, -83, -27, 3, -86, 7, -74, -91, 25, 31, -76, -89, 50, -63, -32, 5, -12, -26, 17, -36, 83, 1],
         [-19, 78, 60, -93, 50, 97, 2, -97, 71, -48, -2, 5, -70, 7, 77, 45, -35, -99, 11, -12, -92, 100, -9, -41, -95, -33, -30, -47, 36, -53, -94, -31, -52, -53, -14, 84, -57, 19, -58, -21, 75, 2, -73, -54, -46, 57, -72, -98, 55, 43, -9, -95, 81, -6, -21, -64, -61, -12, -89, -37, -97, 93, 87, 91, 93, -92, 99, 62, 49, 80, -79, -27, -49, -61, -53, -97, 16, 63, -24, 91, -90, -3, -10, -61, -42, 27, -63, -89, -47, 71, 3, -23, -7, 68, 11, -92, -88, -67, 44, 45],
         [-6, -73, -75, 99, 48, 89, -69, 92, -71, 13, -67, -43, -52, -81, 76, 57, 97, -50, 20, 49, -13, -54, -24, -52, 32, 96, -53, 73, 39, -29, 15, -90, -40, 24, -55, -70, 14, 29, 91, 99, -15, 2, 78, 74, -21, 20, 60, 35, 58, -1, 7, 74, -90, 30, -28, -11, -14, -4, -76, 66, 79, -13, 62, 94, 71, -78, -27, 14, 15, -80, -8, -14, 47, 92, 62, 83, 38, -30, 6, -61, 48, 68, 14, -42, -80, 26, 28, 97, 55, 68, -11, 18, 20, 70, 61, -91, 20, -87, -79, -43],
         [-90, -92, 85, -94, -31, -16, -15, -54, -94, -86, 37, 26, 88, 90, -41, 17, 98, 88, 39, 56, 24, -97, -59, -13, -55, -95, -49, -99, -67, 28, 19, -97, 31, -82, -66, -41, 0, 28, -38, -4, 95, 2, 94, -28, 30, 10, -44, 73, 62, -49, -84, 62, 50, 5, 37, 24, 86, -87, 23, 66, 86, 43, 71, 70, 37, 82, 36, -81, 80, 23, -92, -25, 89, -13, 91, 19, -19, 35, 37, 14, 25, -72, 18, 14, 27, -20, -69, -27, -16, -42, 74, -4, 17, 61, -46, 64, 83, -19, 39, -69],
         [94, 47, -69, 42, -2, 51, -18, -100, 94, 3, -17, 70, 45, 99, -21, 61, 82, 0, 33, -95, -52, -67, -62, 29, 58, 59, -23, 30, 37, 97, -95, 23, -95, -59, 13, 71, -29, 40, -35, -76, 39, 37, 63, 9, -16, 3, -46, -99, -18, -59, 30, -57, 25, -41, -97, 88, -86, -46, -40, -21, 46, 31, -16, 57, -38, 11, -84, -53, 31, 14, -51, -7, 21, 7, 91, 44, 21, -89, -51, -38, -99, 18, 19, -90, 60, -95, -23, 51, 14, 47, 66, 75, -9, -3, -48, 2, -28, -52, -63, -81],
         [-24, 96, 57, 85, 71, -51, -90, 37, -96, 9, -44, 30, -10, -80, -40, 49, 18, -59, -89, 97, 67, 15, 34, -51, 4, -74, 61, 47, 8, -77, 37, 85, 58, -11, -36, 18, -73, -80, 73, 69, -63, 86, -50, -25, 52, -47, 90, -64, -63, 72, -97, -70, 47, -71, 29, 70, -9, -96, -30, -80, 39, 83, -97, 16, 61, -90, 35, 97, -63, 46, 37, 82, -72, -57, -64, 12, 12, -64, -51, 29, -92, 5, -34, -60, 46, -45, 6, -79, 95, -53, 32, 10, -68, -45, 69, -80, -31, 90, 87, 42],
         [1, -82, -74, -47, 73, 97, -29, -26, -68, -22, 78, -67, 38, 94, -77, 30, -34, -82, -64, -100, -13, -37, 56, 71, 8, -91, -46, 58, -50, -23, 62, 55, 64, 84, -10, 21, 88, 92, -95, 92, -3, 82, -87, 89, 52, -27, -86, 86, -65, 13, 10, -77, 67, 30, 91, -62, -3, 72, -32, -68, 44, 12, 3, 28, 57, -54, -91, 39, 34, 34, -17, -13, 10, -23, -1, -95, -7, 79, 7, 92, 2, -32, 35, -36, 46, 48, 76, -23, 80, 3, -40, -40, -30, -88, 24, 24, -75, 91, 82, 41],
         [-10, -14, -69, -7, 32, 45, 31, -36, 92, -30, 71, -95, -94, 23, 100, -76, -61, -95, -84, 86, -81, -78, 78, -52, -71, 35, 78, -59, 87, -31, 21, -93, 90, -14, -98, 12, 2, 55, -51, 42, -65, -49, -89, -77, -86, -27, 17, -71, 45, 88, -60, 0, -18, -1, -42, 17, 33, 76, -38, 2, -20, -10, 51, -52, -61, -89, -99, -73, -93, 46, 58, -62, 40, 34, 66, -57, -85, 16, -31, -73, -42, 71, 15, 79, 74, 17, 10, 69, 8, 36, -40, -24, -80, -27, 50, -52, -45, 52, 57, -41],
         [77, -76, -22, -50, -90, 47, -92, 62, 70, -43, 33, -95, 85, 53, -86, 89, 32, -19, 20, 52, -25, 87, 43, -33, -91, 87, -78, -62, 98, 30, -2, -80, -7, -29, 57, 95, -94, -47, -42, -54, -43, 82, 61, 72, 98, -24, 4, -89, 23, 30, 13, -10, 58, 87, -28, 21, 36, -29, -56, 24, 77, -16, -80, 53, 36, -5, 28, -57, -78, -88, 48, 40, 78, 38, -9, 63, 71, 43, 55, 6, 29, -23, -25, 29, -95, 95, -6, -73, 18, -51, -64, -96, -38, 26, -57, 90, 89, -45, -21, 23],
         [-86, 17, 16, 70, 41, -63, 7, 27, -41, 66, -26, 39, 26, 6, -43, -47, -30, -95, -25, -24, 23, 2, -66, 4, 81, -19, 50, -46, 35, -34, -43, -32, -28, 23, -65, -40, -51, -99, 82, 67, 0, 17, 88, -19, 41, -13, 89, 66, 69, -52, -20, -98, -42, 50, 24, 35, 98, -50, 76, -85, 19, 24, -40, 87, 32, -51, -36, -32, 96, -64, 77, 72, 64, 28, -20, 91, 28, 25, 16, 6, 83, 39, 52, -44, 38, -16, -11, 19, -37, 96, -99, -5, -66, -37, -57, -12, 71, -37, 17, -78],
         [-67, -40, -23, -17, -46, 16, -8, -97, -80, -92, 17, -53, 6, -48, 80, 20, -87, 41, 11, -77, -76, 81, -23, 15, -96, 29, 28, 14, 2, -38, -45, 52, -28, -62, -73, 11, -2, -23, -67, 15, 100, 73, -86, -8, -67, 37, 77, -92, 59, 15, -78, -1, -90, -71, -81, -65, -75, -75, -24, 95, 93, 57, 22, 53, -13, -3, -66, -79, 13, -26, 80, 58, -85, -81, 39, 50, 93, 11, 80, 42, 25, -30, 45, -49, 34, -22, -14, 75, -83, -50, 56, 7, -97, -43, -91, 28, 88, -96, -74, -52],
         [85, -66, 99, 10, 13, 61, 35, 62, -63, 47, 44, 100, 25, -69, 98, 91, -66, -2, 59, 8, -11, 44, 12, 73, 23, 82, 14, -80, -76, -51, 70, -10, -34, -15, -89, 53, -54, -11, 50, -6, 72, 14, 86, -62, -37, 57, -1, 33, -90, 68, -39, -10, -39, 54, 12, -40, 67, -71, 41, 15, -40, 54, -51, -4, -12, 95, 57, 41, 49, -22, 68, 26, -20, 74, -96, 63, 86, 62, 85, 43, -75, -7, 5, -30, -27, -67, -94, -56, 89, 90, 1, -33, -65, -29, -22, -85, -34, 82, 74, 2],
         [6, -30, -38, 26, 63, 12, 84, 96, -40, -15, 79, -18, -2, -42, 8, 32, 34, 85, 59, 82, -41, -10, -37, 57, 36, 4, 47, 67, 78, 41, -99, -4, -3, 68, -19, 92, 31, 26, 92, -99, 33, -62, -97, 15, -20, 98, -11, -44, -5, 98, -38, 94, 66, 17, -77, 90, 40, -21, 64, -62, 92, 36, 25, -1, 41, 46, 91, 74, -34, 83, 45, -73, -83, -86, -54, -2, 10, -23, 37, 23, -42, 35, 90, 77, -84, -51, -78, 28, 4, -53, -77, -49, 24, 97, -40, -91, -6, 19, 61, 73],
         [41, -77, -96, 87, -40, 15, 53, -24, 69, 66, 7, -97, -9, 5, -68, 20, 99, 34, -90, 25, 25, -41, -73, 70, -77, 26, -86, 65, -94, -17, 52, 91, 86, -68, -27, -52, -11, 43, 28, 66, 12, 16, 89, 4, 74, -67, -18, -33, -70, -32, 69, 31, -40, -56, 16, 95, 45, 47, 72, 18, 95, -81, 5, -97, 75, -15, 86, -88, -7, -3, -3, 40, 93, -50, -12, -10, 18, -22, -60, 71, 17, 56, -23, -21, 42, 92, -71, -40, -43, 13, -11, -16, 47, -18, -27, 75, 60, 39, -37, 61],
         [57, -6, 25, 53, 97, 14, 15, 57, 40, -77, -6, -22, 67, -89, -43, 75, 7, -44, -29, -97, 17, -89, 49, -89, -11, 91, 63, -49, -82, 99, 69, -99, 81, -95, 43, 36, 42, -58, -1, -2, -65, -95, -33, -97, 68, -14, 62, 76, 39, 95, 93, -60, -17, -76, -23, 1, -55, 30, 79, -26, -56, -28, 45, 52, 27, 23, -59, -94, 38, 52, 31, 59, 36, 98, -42, 73, 11, 64, -17, -8, 37, 8, 70, -24, 65, 74, -54, -2, -56, -50, -17, -14, 75, -47, 2, 16, -95, -86, -28, -79],
         [-5, 26, -40, 59, 9, 15, -73, -53, -72, 59, 98, -45, 48, 41, 52, -51, 73, -58, 4, 60, 51, -37, -27, 42, 21, 2, -74, -20, -60, -16, -36, 29, 5, 61, 71, -100, -89, -53, 0, -90, -48, -48, 90, -34, -77, 9, 80, 56, 63, 23, -32, -6, 11, -81, 34, -29, -9, 74, 35, -78, -2, 19, -89, -72, -43, 93, 71, -95, 91, 91, 96, -54, -28, 21, -47, 54, -44, -11, 73, 77, 34, 76, -3, -36, 77, 47, -56, -22, -95, 36, 38, 55, 11, -13, -35, 39, -19, -19, -88, 76],
         [-63, -27, 78, 92, -42, 56, 68, 87, -89, -16, -27, -1, 94, -97, -86, 100, 4, 6, 22, -74, -48, -24, -33, -8, -79, 54, 13, 78, -46, -24, -96, 54, -90, -17, 81, 96, -43, 54, -13, -35, 88, -89, -83, -98, 72, -52, -68, 0, -92, -68, -83, 43, 98, 3, -37, 25, 60, 34, 0, 58, 62, -8, -79, 21, -55, -61, 76, 29, -100, 50, 67, -99, -75, -14, -55, -91, 91, 16, -44, 99, -62, 2, 20, -19, 80, -9, 99, 32, 70, 38, 45, -78, 5, 88, -32, 10, 31, -97, 14, -40],
         [-52, 65, -70, 22, 40, -6, 29, -11, -76, -5, 44, -83, 84, 32, -18, -4, 28, -55, -92, -98, 100, 43, 54, -2, 94, -14, 99, -80, 53, -36, 58, 12, -76, 88, 97, -69, 76, 17, 66, 47, 47, -29, -64, 74, -91, 85, -100, 65, -91, -37, -61, 64, 92, 11, 66, -87, -34, -33, -1, 53, 70, 20, 54, 33, 58, -80, 18, 28, 24, 30, -25, 46, -3, -34, -72, 10, -31, 6, -47, 99, -39, 53, 42, 62, -5, 26, 31, -39, -77, -96, 70, -5, -53, 74, 68, -77, -28, 63, 76, 21],
         [-95, -41, 6, 60, 41, 72, 60, 50, -43, 67, 23, -38, 6, 3, -63, -90, -87, 94, -33, -73, 2, 83, -69, 55, -59, -66, 91, -65, 58, -59, -28, 87, -44, 74, -77, 39, 7, 39, 16, 52, -53, 37, -57, -65, 23, 78, -41, 56, 75, 27, 17, 90, 88, -11, -32, -51, -65, -40, 47, -55, 40, 64, -98, 3, -10, -8, 43, -13, -68, 27, -65, 94, 80, -23, 82, 86, 91, -74, -97, 16, 80, -38, -84, 64, 94, -48, 79, 12, 55, 90, 81, 99, -59, 44, -9, -95, -36, 77, 35, 39],
         [-82, -41, -28, 82, 44, 89, 90, 22, -78, 39, 69, -81, -30, 69, 27, -73, -8, 16, 75, 61, 48, 92, -39, -79, -14, 82, -21, -91, -56, 97, -53, 54, -36, 49, -3, 96, 31, -34, -88, 94, 36, 11, 15, -51, -27, 63, -16, 59, -40, -73, -78, 98, 33, 23, -67, 85, 50, -13, -92, 37, 56, -30, 60, -84, 60, 55, 47, -97, -56, 93, -32, -32, 81, 7, -81, 60, -2, 20, 86, -41, 31, 40, 85, 21, 22, 40, -13, 79, 31, -92, 20, -98, 51, 38, 74, -77, -82, -21, 81, -25],
         [-4, -44, -4, -94, -78, -35, 47, 41, 89, 58, 43, 96, -12, 40, -69, -72, 8, 91, 36, 58, -94, 2, -64, 38, 93, -25, -80, 27, 81, -93, -14, -93, 24, -51, 22, 17, 49, 89, 47, 48, 70, 39, 3, 96, 35, 47, 37, 47, -36, -4, -64, -79, -23, -13, -16, -91, -19, -61, 1, 96, 36, 32, -77, 18, -98, -12, -50, 25, 39, 75, 90, -24, 70, -59, 44, -99, 8, 41, 66, 11, 47, 24, -61, 36, -88, 76, 41, 31, 70, 95, 30, -22, -5, 88, -32, -77, -82, 11, 30, 51],
         [81, 89, -82, 17, -3, 65, 15, 2, 41, -61, 37, 86, 53, -68, -33, -67, -87, -65, 6, -46, 45, 28, 28, 62, 52, -76, -5, -91, -91, -80, -25, 54, -22, 84, -32, -63, -79, -41, -34, 96, -41, 95, -30, 36, 52, 66, 26, -6, -95, 27, -89, -9, -25, 12, -67, 3, -52, 73, 4, 14, 49, 85, 77, 53, -79, 67, 49, -27, -80, 80, 15, -52, -21, 63, 31, 61, -31, 11, -72, 25, -4, 73, -3, 46, 1, 35, -66, 51, -53, -65, 13, -9, 36, 53, 34, -93, 5, 88, 63, -94],
         [26, 9, 87, -47, -3, 0, -92, 44, -19, -81, -94, -1, 73, 15, -91, 88, -2, 99, -53, -6, -45, 57, -51, 47, -57, 89, 19, 61, 9, -74, 23, -27, 55, -1, -67, -61, 61, 83, -75, 41, 63, 90, 33, -32, -67, 28, -82, -11, -53, 98, -15, 41, 50, 6, 51, -76, -78, -5, 10, 30, -67, -47, -98, -2, -69, -54, 94, 74, 81, -90, 58, 49, 49, -63, -57, 8, -46, -59, -41, -50, 28, -46, -13, 56, -45, -28, -2, -93, -95, 82, -54, 60, 3, -47, -12, -89, -35, -34, 58, 26],
         [22, -60, -14, 61, 88, -48, -79, -26, 41, 1, -2, -44, 36, -57, -48, -26, -94, 57, 42, -77, 26, 14, 75, -35, -6, -54, -48, 69, -23, 77, 94, -80, -29, 45, -37, 76, 43, 86, -36, 54, -22, 91, -53, 16, 78, -20, 42, 26, 10, 27, 19, 84, -74, 26, -94, 18, -54, -18, 59, 8, -97, 34, 60, 41, 10, -27, -83, -30, 13, -89, 67, -66, 52, 18, 81, -97, 18, 75, -49, -16, 94, -73, -29, 35, 87, -28, 14, 12, -60, 59, 95, -62, 55, 10, 24, -84, -71, -14, 17, -99],
         [50, -49, -64, -34, 53, -56, -31, -51, -23, 81, 83, 92, 4, -16, -96, 37, -79, -66, -58, 10, -22, -49, 45, 87, 14, 50, -29, 75, -72, -20, 92, 3, -83, 38, -46, -85, 33, 73, 1, -35, -50, 61, -23, 49, 45, 59, 89, -46, 94, 40, 37, -69, -29, -43, -35, -33, -91, -39, -76, 29, -87, 31, -19, -7, -98, 73, -48, -70, 92, -3, -90, 16, -65, 15, 74, 61, -4, 76, -96, 32, -2, -34, -11, -87, -65, -25, -59, -11, -47, -96, 22, -35, -64, -71, 85, 51, -54, 88, -86, 41],
         [-72, -18, -94, -6, 2, -81, -26, -76, -79, -69, 2, 18, -47, 97, -16, -30, -91, 75, 84, -23, -58, -1, -37, 94, 65, 94, 99, 82, 74, 67, 68, -35, -44, -71, -52, 20, -65, -30, 6, 23, -28, -64, -61, -68, -95, 100, 93, -33, 0, -58, -44, -79, -53, -52, 96, -83, -90, 2, -20, 83, 2, -95, -88, -22, 45, 2, 75, 17, -39, -35, 89, -75, 18, 35, 19, -80, 89, -81, -94, -3, 34, -45, 68, -54, 78, 74, -74, 34, -53, -80, -1, 33, 97, 77, 54, -6, 94, -58, 36, 45],
         [97, -93, -28, 87, -80, 50, 91, -27, 100, -17, -6, 10, -56, 34, 1, 78, -23, 5, -29, -8, 65, -49, 48, 34, -54, -4, -2, 4, -77, -55, 39, 32, 15, 58, 16, 38, 54, 89, 67, 45, -80, 25, 67, -14, 88, 22, -22, -53, -86, -90, 35, 96, -50, 79, 55, 91, 75, 74, 7, -96, 6, 85, 95, 12, -45, -16, -62, -10, -64, -48, -43, -29, -85, 49, 18, -12, 75, -69, 65, -99, 54, 48, -11, 31, 49, 67, 71, -75, -90, -19, 27, 79, -39, 50, -84, 18, -27, -78, -11, 32],
         [99, -3, -4, 58, -81, -80, -68, 58, -64, 16, 53, -93, 39, -68, -16, 96, 54, -76, 34, -95, 97, -46, -1, -59, -74, -77, -44, 96, 47, -8, -50, -13, 60, 34, -46, 27, -2, -45, 42, 54, 85, 37, 28, -31, -34, -47, 94, 16, 95, -90, 19, -87, -87, -45, 99, 6, -3, -34, -18, 66, -22, 60, -42, 6, -39, 89, -60, 65, -51, -66, -69, 79, 91, -35, -34, 21, 82, -18, 68, 71, -4, -90, -64, -12, 65, 35, 32, 55, 78, 45, -28, 61, -15, 74, -92, -1, 25, 45, -3, 16],
         [51, -30, 5, 26, 33, 38, -52, 69, -40, -80, -89, 97, 78, -77, -30, -83, -72, 51, 52, 24, -26, 54, -85, 42, -89, 77, 46, 61, 22, -63, 21, -35, 98, -51, 0, -36, -26, 71, -13, -99, -56, -77, 50, 35, 52, 9, 60, -29, -32, -43, 85, 65, 7, -38, -63, 90, -24, -12, 2, -44, 19, 13, -36, -59, -36, -57, 24, -78, -74, 16, 87, 47, 39, -31, 47, -97, -82, -95, 51, 62, -84, -72, -50, -23, -48, -49, 65, 76, -41, 16, -92, 83, 13, 39, 97, 95, 38, 72, 55, 98],
         [45, 92, 6, -79, 52, 90, -39, -37, 100, 83, -9, 53, -21, 14, 38, 50, -55, 11, -75, -81, -79, -31, -78, -67, 23, -24, 44, -28, -94, -65, 77, 62, 57, 50, -29, 90, 67, 53, -66, -40, -51, -25, 64, -50, 29, 83, -21, 40, -11, 1, 92, -55, -46, 26, 97, 33, -80, -93, -6, 54, 56, 14, 3, 12, 65, -99, -58, -85, 66, 75, 80, -27, -34, -37, 8, 98, 56, -55, 46, 4, 54, 78, -95, -52, -59, 14, -17, 28, -13, -75, 72, -31, -46, 89, 36, -88, 36, 49, 76, -48],
         [23, -47, -59, 12, 0, -49, -45, 49, -45, 54, 18, 80, -58, 14, 19, 98, -3, -1, 17, -71, -62, 50, 62, -76, 26, -71, -9, -70, -13, -7, 98, 67, 60, 74, 45, -58, -39, 63, 43, 52, 85, -88, -44, 28, 25, 51, 2, -95, 31, -24, 51, -57, 74, 94, 82, 5, 35, -49, 60, -26, 6, 26, -59, 88, -41, -43, 43, -48, 61, 3, -79, 85, -59, 76, 50, -50, 84, 90, -84, -78, 27, -60, -25, 31, 25, 98, 67, -62, -6, -22, 90, 33, 13, -69, 77, -71, -39, 90, -96, 3],
         [-25, 37, -72, 18, -74, 8, -8, 93, 69, 18, 61, -34, -3, 83, -67, 2, 65, 30, -98, 38, 19, -28, -57, -52, -5, 47, 28, -7, 51, 93, -64, -30, -39, 70, 63, 18, -26, -11, 37, 73, 42, -72, 98, 7, -97, 14, 24, 13, -85, 35, -30, -43, -17, -7, -55, -98, -4, 13, 22, -20, 89, 47, -59, -63, -31, 25, 10, 45, -5, 53, -79, -58, -44, -89, -22, -80, -4, -45, 5, -99, 84, 31, -62, 41, 43, 65, -98, -74, 28, -3, 13, -76, -50, 24, -84, -79, 89, -77, 17, -34],
         [-60, 61, 94, -46, 26, -3, 42, 70, 97, 67, -80, 85, -78, -9, 54, -29, 79, 74, 92, -75, -67, -60, -6, -7, 81, -66, 73, -17, 52, 20, 69, 74, -48, 46, -57, -87, 87, 64, 74, -29, 3, -87, -34, -22, 74, -21, -22, -20, -26, 60, -59, 13, 39, -92, 91, -68, 0, -66, -44, 52, -57, -91, -78, 9, -30, -33, 19, -68, 4, 42, 2, 77, 46, 26, 76, -61, -28, -14, 27, 77, 58, -23, 93, -31, -3, -7, -52, 70, -89, 4, -30, 70, -69, -30, -3, -97, 83, 70, 31, -65],
         [10, -34, 80, -68, 52, 95, -41, 79, -34, 60, -68, -31, -93, -18, -38, 15, 25, -87, 90, -6, 5, -19, -77, 19, -23, -78, 64, -2, 27, 55, 62, 43, 99, 28, -88, 74, -42, -47, -30, -64, -68, 57, -71, -3, -95, -70, 87, 93, 31, -54, -2, -27, 33, 31, 75, 6, 100, 67, -53, -43, 52, -76, 92, -97, -74, 92, 72, 99, -28, -75, -20, 30, 97, 17, -9, 98, 2, 84, -21, -68, -4, -41, 30, -64, 26, 14, 18, -60, -30, -18, 80, -100, -15, -17, -52, 64, 36, 12, -94, -53],
         [-33, 43, 50, -91, 93, 13, -85, -65, 52, 33, -14, -31, 18, -69, -31, 94, -53, 79, -92, 45, -27, 43, 26, 15, 79, -94, -85, -6, -100, 38, 34, -15, -59, 4, -68, -8, 9, -63, 66, 12, 9, 43, 15, -46, -51, -7, -22, -21, 34, 32, -62, 39, -10, -1, 66, -51, -27, -44, -44, -46, 57, 0, -5, -75, 62, -79, 19, 52, 89, -69, -42, -15, 51, -26, 5, -25, -3, 65, -59, -95, 50, 87, 11, -93, -38, 32, 55, -37, -83, 48, 8, -42, 80, 21, 7, 16, -71, -25, -81, -91],
         [-11, -56, 23, 62, 30, 1, -89, 96, 57, -41, -33, -22, -3, 17, -55, 68, -80, -63, -67, -63, 38, -90, 2, -97, -28, 50, -58, 80, 88, 53, -27, 99, -76, -64, -18, 26, 27, 26, 66, 64, 56, -79, -57, 42, -32, -83, 80, 91, 22, -91, 31, -41, 8, -86, 83, 20, -39, -95, -21, 69, -91, -96, 89, 18, -95, -90, 57, -71, 27, -100, -16, 24, -96, 49, 93, -80, 91, -80, 47, 74, -77, 35, 72, -56, 29, 19, -23, 8, -49, -50, 94, -59, -3, 16, -36, -1, -26, -88, 39, -66],
         [34, 0, -5, -41, -95, 87, 94, 52, 80, 74, -41, -73, 56, 60, 56, 90, 89, -67, 37, -66, -38, 17, 90, 37, 10, -68, 92, -97, -8, 27, 70, 49, 62, -93, -75, 72, 30, 33, -68, 51, 76, -56, 83, -36, -66, -4, 4, 54, -22, -1, -15, -72, -99, 8, 52, -23, -60, 17, 1, 29, 76, 59, 27, 59, -74, -50, -55, 9, -17, -65, -31, 6, -48, 33, 0, -84, -16, -75, -92, 70, 43, -37, 62, 21, -7, 85, 7, -24, 15, -9, 25, -56, 45, 99, -22, -32, 91, -61, -21, 77],
         [-98, -88, 73, 64, -65, 4, 68, -92, -97, -69, -98, -9, 83, -26, 19, -6, 60, 86, 59, -58, -61, 14, 59, -60, 93, -98, -88, 21, 71, -37, -22, 23, -83, 39, 8, -81, 79, 12, 80, 53, -3, -15, -98, 69, 51, -71, 24, -4, 9, -64, -29, 4, -72, 79, 49, 30, 84, 86, 63, -84, -82, 47, -66, -27, -21, -40, 72, 13, 34, 68, -13, 27, 29, -40, 85, -35, 1, -1, 71, -43, 34, -99, 24, 95, 5, 85, 79, 97, -91, 89, 99, -6, 39, 41, 42, 34, 45, -73, -46, 37],
         [-67, 22, 71, 52, -45, 1, 59, -6, 16, 79, 22, -75, -41, 21, -83, -58, -56, 67, -12, 83, 8, 0, 82, 76, 72, -43, 97, 46, 77, -80, -8, 93, -38, 19, 66, 30, -30, -33, -79, -58, -24, 19, 2, -15, -12, 90, -81, -45, -92, -65, -81, -83, -57, 0, 61, -76, -65, -7, 76, -72, -34, 30, 77, -37, 39, 51, -45, 81, -25, 3, 4, 21, 26, 35, -14, -41, 94, -32, 68, -61, -38, -44, 83, -47, 79, -24, 6, 46, 84, 47, 62, 79, -51, -76, 30, 73, 38, -34, 21, -79],
         [-77, -77, 92, 62, -59, -56, -13, 62, 16, -24, 77, 64, -33, 10, 69, -83, 86, -21, 53, -23, -91, 25, 87, 61, 35, -9, 64, 86, -76, -16, -60, -94, -34, 35, -78, -14, 84, 79, -98, 34, -98, -19, 40, 33, 16, 76, -20, 36, 96, 26, 38, -52, -11, -21, 64, -92, -18, 2, -17, 13, 0, 45, 74, -34, 76, -13, -45, -90, 40, -40, -49, 38, 1, -62, -7, -72, -79, 1, 25, 23, 37, -45, -31, 19, -6, 59, 57, 19, -72, 93, -63, -12, -2, 16, 47, -21, -49, -97, -14, -89],
         [-11, -26, 47, 60, -35, -68, -30, -17, 67, -68, -81, 72, -58, 67, -64, -64, 8, 46, 37, 17, 76, -5, 21, -53, 51, 25, 25, 22, 2, 1, -75, -47, 21, 4, -87, -93, 73, -58, -7, -82, -40, -68, -81, 52, 62, -40, 45, -64, -84, 33, -47, -68, -58, -73, -7, 78, -33, 25, -93, 18, 99, 72, 61, -63, 16, 61, -50, -89, 72, -21, -51, 74, -75, -57, 2, 36, -61, 8, 1, -67, -35, -51, -83, 100, 8, 85, -7, 70, -92, -48, 57, -35, 9, -60, -95, 97, -51, 61, 79, 41],
         [19, -72, -42, -20, -74, 69, 15, 51, -6, 15, -57, -68, 5, 68, -75, -84, 8, -24, 19, -77, -30, -44, -1, -71, -39, -30, 46, -86, 36, 33, -63, 44, -61, 42, 24, -22, 7, 24, 9, -95, -87, -18, -43, -52, 80, -3, 38, -84, 81, -87, 63, 86, -63, 13, -69, -76, 2, -67, 4, -29, -10, -43, -59, -75, 89, -30, 24, -91, -15, 50, -37, 84, 86, 78, -88, 77, -82, 92, -53, -30, 37, 7, -28, 70, -44, 5, -51, -57, -57, 90, -72, -59, 64, 79, 85, -80, 87, -18, -28, -2],
         [46, 21, 60, 71, 24, 80, -61, 3, -6, 39, 81, -12, -1, -38, 22, -41, 20, -70, 72, 8, -37, -27, -14, -2, -32, -29, -89, 84, -73, 66, -41, -35, -55, -29, -98, -61, -43, 71, 8, 88, -70, 92, -28, -8, 55, -22, -64, -62, 25, 18, -71, 74, -55, 43, -18, -51, -73, -45, 96, -42, 57, -80, -73, 88, -98, 20, -97, -49, 56, -71, -65, -37, 54, -82, -46, 86, -55, -68, -7, -30, 90, -55, -82, -45, -17, -51, -35, 24, -37, -25, 81, 42, 35, 21, -21, -4, 47, -32, 56, 72],
         [-50, -14, 45, -62, 28, -15, -6, -89, 38, -7, -55, 14, 8, 94, -71, 51, 46, 21, -23, 8, -71, 29, 51, 94, -30, -89, -87, 0, -70, -5, -83, 4, -24, -32, -96, -67, 71, 26, -10, 63, -95, -59, 95, 100, -93, 58, 46, -8, -87, 89, -71, 78, 71, 17, 87, -93, 80, -55, -98, 12, -56, -81, -80, -35, -77, 77, -54, -57, -32, -9, -42, -56, -76, 55, -69, -79, 26, 31, -10, -50, 82, -24, -28, -62, 40, 24, -50, -69, -62, -32, -64, -96, -20, 75, 60, -41, 98, -75, 87, 6],
         [72, 99, 29, 19, 46, 71, 100, -7, 85, 1, -43, 18, -46, 72, -46, -83, -90, -45, 51, -55, 49, -5, 20, -78, -99, 83, 67, -70, -64, -33, -58, 53, -37, -25, 23, 50, -74, -68, 19, 3, 91, 60, -26, -37, 26, 39, -33, 49, -53, -48, 43, -93, -75, -42, 49, -83, -15, 71, 26, 11, 87, -82, -36, -2, 79, -68, -82, -15, 60, -75, -24, 42, -51, 75, -29, 34, 10, 37, -18, 37, -72, 11, 42, 23, -85, 49, -13, -78, 29, 98, -26, 58, 32, 93, 78, 21, -32, -87, -77, -45],
         [15, 31, -14, -58, 34, -96, 40, 76, 52, -53, -4, -92, -49, 64, -64, 86, 82, -82, -48, -13, 58, 64, 1, -29, -61, 24, -58, -84, -86, -39, -70, 5, -18, 22, 51, 29, 17, 52, -5, -75, 21, -17, -95, -56, -92, -40, -52, 82, -64, -53, 81, 3, 64, -34, -76, -26, 73, -12, 74, -82, 17, 16, 88, -62, 75, -36, -71, -56, -67, -9, -69, 33, 62, -64, 84, 90, 38, 95, -13, 1, 3, 6, -34, 69, -82, 42, 13, 27, 36, 37, 86, -81, -29, -32, 82, 83, -36, 70, -67, -11],
         [-43, -52, -42, -40, 28, -82, -6, -9, 31, -58, 53, -56, -9, 37, 29, 11, 19, -88, -20, 74, 7, -22, -35, 56, 6, -40, 60, -21, 28, -7, 81, -26, 90, 81, 41, 49, -62, 74, 92, 58, -88, 85, -5, -74, 25, 5, 86, -43, -78, 5, -79, -40, 0, 90, -46, 84, 73, -87, -17, 79, -53, 89, 89, 69, -40, -59, 34, 29, -30, 81, 54, 36, 15, 70, -30, 99, -79, -67, -46, 1, 50, 8, 37, -4, 9, 24, -44, 58, -68, 23, -56, -27, -85, -95, -65, 40, -83, -74, -61, 98],
         [62, 73, -1, 21, 68, -55, -6, -50, 77, 24, 96, -42, 67, -97, -100, -10, -16, -20, -38, -5, 48, -62, -42, -10, 10, -69, -84, 65, 44, 39, -89, 13, -61, -89, 77, -77, 10, 92, -10, 86, -26, -7, 67, -55, 56, -22, 81, -49, -29, 57, 66, -35, -62, 5, -87, 41, -81, 44, 25, 73, -78, -42, 70, -65, -97, 74, -71, -46, 84, -86, 61, -63, -44, 51, 90, -77, 57, 44, 24, -89, -12, -90, 58, 97, 20, -100, -1, 38, 44, 31, 93, 55, 56, -91, -78, -61, -73, 17, 82, -2],
         [-46, 18, -26, 0, 3, -27, 38, 44, -59, 69, -16, 18, -68, -10, -9, -8, -13, 52, -63, 47, 20, 39, -84, -36, -99, -77, -48, 14, -23, 26, -32, 80, -4, -27, 31, -30, 38, -61, 29, 10, -42, -29, -1, -2, 94, 12, -50, -12, -10, -20, -6, 94, 18, -70, 17, 68, 35, -84, 73, -10, 16, -13, -100, 80, -74, -5, -3, 60, -68, -10, 9, -43, 13, 23, -27, -74, -84, -12, 64, -86, -44, 9, -36, -66, 75, -46, 4, 21, 99, -16, -77, 58, -76, -67, -46, -26, 90, 82, 41, -8],
         [-67, -65, 87, -46, 22, -53, 22, 56, 39, -85, 48, -79, -88, 79, 21, -40, 76, 26, -21, 10, 61, 58, 61, -6, 93, -21, -55, -57, -67, -77, 58, -12, -74, 22, 94, 33, -38, 90, 38, -23, 82, -39, -53, -33, 21, -26, 90, 86, 69, -14, -94, -57, -50, -71, 40, 95, 74, -48, 33, -87, 39, -82, -54, -73, -43, -29, -58, 18, 49, 58, 88, 49, -52, -31, -29, 36, -20, -64, -91, -23, -84, 93, -48, -70, -33, -7, -46, 37, -95, 91, -17, 30, 18, -68, -49, 98, -34, 34, 36, -12],
         [-63, 8, -65, -94, -39, -9, 98, 68, -97, -92, -14, 59, -22, -9, 79, 94, -86, -70, -13, -44, 21, 35, -58, -8, -32, -66, -10, 72, 6, -27, -41, 47, -100, 87, 82, -3, 62, 45, 50, -70, -85, 48, 61, 15, 33, -20, -28, -6, -71, -67, -40, -11, 91, -31, 95, -27, -95, 30, -23, 30, -14, -46, -95, 16, -13, -52, 0, -36, 38, 45, 26, 52, 41, -62, 23, -33, -38, 25, -28, 72, 41, -90, -7, 76, 17, 51, 3, -56, -38, -94, -73, -15, -26, -44, -79, 87, 9, -86, 86, -77],
         [-79, 58, 15, 48, -73, 24, 79, 1, -39, 83, -65, 8, 32, -6, 64, -74, 7, -35, 21, 34, 73, 61, -95, 30, 16, 78, -34, -28, 26, 84, -20, -86, -48, -12, 84, -58, 60, -66, -31, 39, 59, 69, -1, -91, -79, 42, 11, -35, 69, 13, 59, 66, -67, 61, 57, 34, -61, 87, -43, 44, 29, 80, -4, 31, 59, 29, 39, 91, -77, -97, -37, -90, 98, 41, 30, 5, 17, 80, 14, 56, 19, -91, 100, 62, -62, -74, -5, -9, 19, 30, -57, 45, -16, 61, -70, 80, -71, 94, -36, -1],
         [-89, 19, 1, 16, 43, -83, -50, -20, -33, 84, 91, -94, 67, 74, 30, 5, 19, -31, -79, 99, -43, 84, 7, -32, -80, -90, 65, -38, -85, -71, -38, 3, -9, 29, 74, -38, 10, 15, -28, -37, -79, 37, 55, 16, 88, 14, 22, 6, 72, 97, 97, -93, -100, -89, 82, 65, -25, 85, 6, 82, -88, 58, 10, 91, 79, 70, -16, -65, 6, 46, -4, 51, -36, 54, -50, 69, 57, 100, 70, 21, 54, 21, -6, 46, -86, 69, -32, 58, 12, -42, -55, -49, -53, -39, -52, -45, -43, 79, 60, 80],
         [-68, -70, 35, 66, -85, 84, -78, -12, 60, -15, 31, 51, 60, -100, -38, -31, -17, 3, -36, 41, 19, -83, 18, 22, -67, -84, -19, 68, 54, -13, -70, 30, -32, 56, 46, 84, 1, 67, -32, 78, 84, 95, 12, 37, 13, 97, -74, 51, 98, -89, -41, -39, 18, 9, 36, 29, -96, 100, 33, 0, 58, 57, -50, -38, 68, -86, 88, 46, -90, 29, -93, -92, -25, -81, 81, 17, -30, -7, -11, -43, -63, -31, 5, -97, 80, -4, -41, -86, -70, 87, -4, -40, 34, 87, 50, -64, -48, -39, -98, 45],
         [-93, -58, -42, -74, 31, 83, -89, 73, -86, 57, -71, -46, 60, -93, 6, -1, -77, -98, 26, 57, 28, -56, -91, -57, -48, -1, 41, -98, 55, 37, -53, 7, -19, 19, 73, -66, 78, -39, 40, 71, 25, 1, 37, 93, 54, -84, -25, -74, 38, 69, -59, 64, -71, -19, -30, -45, 22, -76, 33, 10, -5, 35, 63, 29, 15, 10, 75, -39, -10, -83, 82, 5, 68, 100, 22, 15, -49, 12, 74, 46, -32, -69, 72, 99, 59, -22, 85, -75, 65, 16, 98, -37, -9, -7, -7, 91, -39, -9, 21, 69],
         [36, -84, 81, 76, 49, 48, -6, -12, -72, -11, -55, -47, -94, 68, 2, 79, 54, -45, 96, 3, 43, -8, 53, 77, -46, 92, -12, 2, -50, -19, -78, -22, -33, -73, -13, 83, 74, -88, -91, 4, 61, -82, 99, 86, 98, -97, 23, 2, 11, -12, -17, -53, 2, -7, 71, -64, -43, 99, -43, -62, -6, -71, 10, -82, 30, -41, -41, -81, -61, 75, 39, 46, -46, -65, -8, -88, -7, 85, -6, -16, 82, -73, 29, 98, -22, -70, -8, -41, 42, 41, -81, -24, 8, -15, -57, 84, 27, -47, 11, -58],
         [-43, -45, 64, -40, -61, -25, 60, -77, 34, 49, -96, -69, 97, 45, 4, -45, 85, 51, 93, -83, 5, 89, 67, 5, -72, 31, 31, 9, -51, -41, -46, -69, 42, 23, -40, 39, 24, -50, 35, -60, 12, -36, 25, -67, -90, -84, 12, -85, 32, -33, -18, 60, 48, -5, -32, -32, -42, 87, 96, -98, 51, -32, 98, 90, -82, -68, 55, -60, 50, -14, -26, -40, 46, -54, 76, 32, -15, 33, 61, 36, 38, 40, 16, 0, 35, 81, -57, -43, -41, -13, 86, -83, 56, 50, -84, 79, -39, 51, -36, -99],
         [3, 48, -75, -42, -40, 48, -33, 51, 40, 25, 87, -63, -74, 19, -10, -52, -31, 2, 71, -78, -23, 45, 45, -60, -65, 87, 68, 25, 97, -53, 14, 11, 87, -96, -68, 96, -70, 51, 100, 21, -47, -45, 97, -43, -75, -8, -61, 97, -33, 37, 38, -31, -100, 33, -61, 65, -93, -86, 8, 15, -90, 85, 77, -48, -40, 15, -14, 36, 33, 63, -86, -36, 71, -9, 16, 7, -25, -61, -64, -39, -58, 46, -80, 51, -55, 28, 5, -22, -90, -62, -52, 60, 47, -3, 2, -29, 88, 89, -52, 82],
         [27, -32, -72, -3, 16, -56, -79, 27, -91, -1, -88, 13, 51, 47, -47, 32, -56, 36, -20, -96, 9, 91, -78, -29, -74, -45, 6, 56, -45, -7, 33, -74, 37, -74, 96, 94, 20, -26, -76, 93, -62, 55, -42, -4, 3, -98, -98, 63, 78, 76, 54, 50, -71, -85, -71, 8, -39, -68, 30, 32, -20, 33, 77, 86, -92, -37, 8, 17, -86, 63, 8, 40, 64, 11, -91, -23, 61, 70, 93, 58, 11, -92, 91, 61, 60, 5, -5, 65, -49, -33, -16, -61, 41, -23, -86, 92, -30, -81, 51, -47],
         [34, -83, 99, -55, -84, 52, 90, -30, 36, -99, 99, 99, 33, 3, -6, -88, -43, 92, -27, -72, -5, 7, -96, 90, 66, -90, 73, -55, -53, 55, -34, -40, 42, -51, 17, -93, 9, 27, -37, -5, 73, -53, -90, 60, -81, 1, 78, -77, -79, -81, -25, -17, -47, 10, -8, 34, -70, 88, -35, -33, 80, -20, 2, -10, 9, -3, -11, -53, -64, -85, 58, 40, 43, 69, -26, 77, 52, -70, -96, -41, 41, 26, -77, -8, -43, 27, -74, -41, 41, -54, -57, 94, 16, 33, 83, -51, 28, 19, -62, -51],
         [34, -39, -2, 6, -60, -15, -12, -79, 78, 78, -48, -54, -69, -51, 42, 83, 11, 18, 94, -68, -93, 93, -96, -13, 23, -96, -47, 94, 10, -85, 25, -91, -2, 13, 17, 13, -56, 4, 40, 12, -7, -48, -82, 28, 53, 83, 4, 90, 1, -6, -97, -43, 36, -43, 75, 26, -92, -58, 26, 87, -61, -58, -100, 14, -98, 43, 98, 7, 57, -62, -69, 14, -35, -30, 43, -15, -57, -39, 1, 74, 98, 94, 27, -98, 25, -49, -80, 78, -33, 6, 86, 92, 91, 38, -8, 86, -98, 16, 44, -57],
         [76, 70, -71, -36, -67, -26, 28, -23, -80, -16, 41, 30, -26, 0, 94, 55, -46, 65, -3, -23, -29, 68, -88, 30, 64, -12, -92, 29, -68, -57, 97, 23, -89, 90, 47, 41, 3, 19, 33, 58, -71, 83, -8, -96, 66, -90, -47, 100, -87, -42, 15, -55, -91, 14, -24, 45, -41, -90, -22, 66, -79, -60, 95, -93, -30, -60, 25, -60, -34, 10, -16, 31, -46, 40, 99, 60, -94, 45, 27, -30, 42, 81, 79, -70, -92, 75, 20, -12, 98, -83, 32, -38, 46, -6, 35, 65, -10, -34, -10, -66],
         [26, -29, -42, -9, 78, 0, 10, -50, -38, 32, 1, 13, -77, 81, 85, -57, -96, 96, -61, -29, 2, 82, 70, -48, -91, 31, -39, -26, 53, 24, -97, 82, 10, 44, -63, -40, -12, -94, 60, -33, -86, 11, -32, 13, -64, -14, 30, -50, -37, -56, 40, 31, -28, -56, -16, 72, 84, 99, -93, -86, -47, -13, -13, 25, -2, -84, 63, -76, -15, 47, -11, 56, 24, 58, 62, -57, 0, -16, 84, -51, 70, -75, -16, -41, 59, -67, 1, 76, 20, 12, 30, -70, -81, -15, 80, -13, 25, -84, -20, 68],
         [-73, -70, -6, -15, -61, 77, -55, 99, -7, 39, 9, 57, 15, 9, 3, 63, -97, -60, 82, 72, -65, -95, -59, 37, 94, 34, 15, -13, 77, -100, 57, -7, -5, -21, 77, 2, -87, 53, -81, -73, 11, -33, 20, 35, 36, -46, 77, -42, -63, -20, -19, 67, -78, -82, 16, -2, -97, 53, 78, 66, -84, 58, 0, -18, 67, 61, 82, 41, 8, 95, -66, 0, -38, 60, -48, -25, -28, 52, 78, 57, 27, 44, -35, 63, 9, -55, 65, 7, 15, -39, -16, 34, -18, 12, 55, -41, 57, -20, -92, 19],
         [-91, 27, -87, 70, 46, 77, -6, -45, -44, 6, 24, 24, 93, -51, 97, 19, 6, 16, -34, -47, -58, -63, 88, -79, -21, -94, 51, 53, -12, 95, 23, -44, -46, 45, 68, -40, 82, 70, 31, 70, 31, 62, 38, 30, 60, -35, 20, 77, 36, 22, -16, -15, 84, -74, 32, 58, 33, 80, 56, 31, 48, -18, 97, -25, 21, -48, 62, -2, 71, -90, 22, 56, 82, -78, 36, -87, 69, -52, -41, -24, 54, 39, 46, 8, -68, 24, -40, 42, 11, 18, 61, -62, -18, 74, -32, 52, -17, -22, -31, 26],
         [-98, -4, -5, 21, 100, -99, -69, 59, 93, 93, -60, 57, -26, -44, -71, -71, -66, 24, -92, -65, -89, 44, 32, 32, -32, 64, 2, 42, 11, -73, 26, 23, 51, -18, 87, 71, 10, -50, -60, -41, 73, -50, 33, -31, -39, 46, -79, 86, -33, 3, -55, -56, 51, 20, 7, -29, 19, 0, -98, -71, -79, 79, -46, -55, 11, -8, -33, 75, 68, 51, -46, -18, 1, -39, -79, -36, -45, 10, -10, -87, 94, 33, -80, 42, 73, 7, 9, -78, 95, 62, -49, 38, -43, 4, -87, -45, 27, 50, -47, -35],
         [96, -96, -41, -60, -39, -44, 17, -76, -27, -44, -39, 50, -30, -13, 72, -23, -19, 13, 99, -65, 45, -37, -83, 46, 83, 65, 14, -72, -25, -50, 76, 26, -54, 68, -42, 30, -88, 73, 9, 85, 38, 60, 7, -62, 43, 80, -92, -44, -38, -50, 80, 27, 21, -98, 67, -52, 60, -33, -77, -97, -44, 31, 57, 60, 75, -44, 3, -5, -49, 8, -39, -94, 58, 34, 7, 14, -32, 42, 78, -3, 70, 27, -8, -55, -24, 32, -79, 59, -24, 66, -27, -80, 62, 99, 94, 7, -66, -2, -84, 0],
         [-74, 56, 86, 29, 74, 91, 59, 36, 32, -79, -78, 66, 61, -93, 20, 29, -70, -47, 91, 41, -37, 3, 32, 94, -16, 90, 7, 67, 94, 31, 99, 3, -49, -37, -47, 71, 66, 43, 92, -36, -15, 41, 57, 98, 38, 59, -61, 80, 33, 92, -74, -100, 48, -83, -92, -26, -99, -81, -6, 6, 39, -17, 30, 10, -96, 33, -77, -91, -6, -48, 14, 4, -37, -98, 25, -18, -50, -83, -49, -87, -32, -88, 42, 16, 59, 26, 71, 99, 38, -57, -18, 27, 59, -69, -51, 29, 13, -47, -9, 37],
         [-32, -69, -27, -76, -88, 50, 71, -23, 89, -54, 65, 27, -52, -74, 50, -6, -68, -62, 68, 58, -93, 64, 47, 2, 39, -60, -91, 92, -58, 100, -61, 16, -3, 7, 80, 62, 28, 96, 88, -65, 76, 76, -61, 3, -6, 68, -47, -97, -14, -70, 13, 24, 3, 13, -78, -46, -14, -36, -92, 86, -36, -79, -34, -33, -49, 16, 90, -67, 96, -28, -96, 29, -83, 45, 84, 61, -69, -99, -14, 23, 73, 58, -5, 34, -48, 79, -93, -14, -29, -3, 16, 58, 31, -45, -24, 77, 69, -97, -91, 4],
         [-97, -96, -14, -40, 10, 18, -77, -9, -5, 37, 42, 27, 4, 41, -27, 75, 54, 38, -79, 4, -68, -49, -12, 86, 55, -70, 73, -82, 70, -17, -75, -48, 32, 6, -56, 2, 16, -95, 21, 13, -65, 85, -4, -62, -56, -88, 41, -11, -34, 16, 4, 2, 5, 17, 89, 30, 90, 62, 90, -10, -48, 99, 67, -5, -47, -9, -31, -10, -92, -31, 57, 31, 93, -21, -50, -71, -96, -93, 4, -100, -4, -9, 33, 59, 6, -41, -25, 43, 45, 94, -21, 47, 24, -30, -38, 95, 90, 85, -73, 52],
         [17, -89, 42, -10, 90, 94, 77, 14, 45, -79, 73, -91, -68, -56, -70, -97, -56, 45, 38, -40, 45, -97, -49, 83, 13, -38, 36, -33, -55, -95, -5, 50, -2, 65, 70, 5, -79, -42, 83, -68, -58, -1, 37, 51, 19, -13, -66, -40, 90, 38, 96, 29, 11, -35, -5, 71, -4, -92, 84, 27, -79, 10, 3, 30, 96, 15, 32, 10, 42, -19, -21, 0, -19, 44, 50, -11, -81, 36, 27, -34, -95, 58, -57, 23, -82, -41, -54, -17, -85, 76, -40, -55, -15, 47, -20, 0, 68, 51, 6, -6]]

    b = [[28, 57, -75, 95, 95, -95, -26, -68, 68, -7, 20, 46, 78, 10, 25, 71, 62, -32, -8, 96, -62, -90, 95, -17, 73, 49, -84, 29, -65, -39, 19, -67, 6, -32, 31, -56, -5, 93, 30, -43, -57, 17, 99, 71, 43, 7, 8, 73, 88, -4, 22, 44, -17, 20, -47, 6, -75, -93, 8, -29, 49, -44, -56, 29, 27, 69, 7, 6, 71, 20, 82, 12, 33, -3, -71, 73, 34, -86, -13, 70, 77, -72, -96, 7, 68, 39, -28, -30, 89, 100, 63, -14, -76, -70, -16, -36, -30, 95, -43, 6],
         [40, 94, 85, -69, -10, 34, 52, 96, -81, -41, -19, 81, 75, -36, 22, -97, -19, 14, 22, 18, 56, -26, -97, -11, -90, -52, -8, -78, -76, 36, 26, -80, 60, 48, -56, 52, -39, 15, -100, 46, -34, -24, -82, -18, 48, 72, -21, 54, 53, -65, -73, -58, -87, -21, 94, 90, -42, -4, -62, 28, 56, 78, 73, -55, 65, -5, -27, -21, 35, 94, -44, -24, -1, -41, -25, 11, -47, -10, 30, 54, -44, -12, -90, -51, 70, -88, 49, -34, -88, -59, 70, -35, 56, -40, 6, 44, 2, 59, -51, 75],
         [18, -27, 71, -72, 26, -9, -12, 99, 25, 89, -64, 27, 61, -23, -29, 77, -93, -69, 9, -70, -83, 98, -14, -39, 23, 57, 91, -85, -18, -44, 43, 42, 24, -19, 47, -28, 65, 25, -16, -57, -67, 14, 55, 46, 77, 83, 87, -11, 68, 74, 80, -88, -60, 51, -23, 78, -61, -1, 99, 79, -73, -88, 83, -19, 1, 15, -99, -15, 28, 67, -69, -62, -57, 14, 84, -80, 12, 37, 9, 15, -15, -84, 73, -57, 43, -51, 43, -1, -38, 33, 56, 93, 84, -63, -94, -2, -11, 99, -53, 5],
         [-34, 83, -94, 91, 84, -27, 36, 58, -43, 62, 35, -53, 71, 41, -80, -69, -58, -76, 100, -65, -96, 50, 94, -48, 28, -25, -12, 50, -23, -93, 29, 64, 20, -41, 28, 51, -45, -74, 43, 14, -45, -91, -59, -62, -86, 75, -97, -71, 22, -13, 23, 14, 67, 98, 88, -30, -36, 38, -57, 33, -6, 6, -59, -15, -33, 0, -40, -70, -11, -47, 41, -40, -33, 5, 87, -78, -62, 61, 92, 73, 93, -37, -6, 99, 53, 27, -43, -63, -81, -10, 26, -2, -95, -57, 44, 82, 15, -41, -17, -86],
         [-39, -96, -75, -52, 82, 100, -42, 7, 15, 86, 23, 78, -60, 59, -4, 54, 57, -80, -27, 12, 35, 83, 21, -62, -86, 68, 58, -74, 70, -89, -59, 69, 93, 98, -15, -80, -66, -27, -54, 41, 14, 93, -89, 59, -91, -12, 61, 51, 71, -10, 66, -94, 91, 53, 17, -33, 68, -31, -78, 30, -87, -81, 36, 70, -11, -22, 33, -94, -65, -11, -82, -86, -32, -74, 37, 27, 71, -57, 78, -96, -39, 83, 91, -90, 45, -83, -94, 72, -54, -1, 0, 66, 54, 91, -73, -86, 95, 87, 96, -44],
         [16, 24, 64, 92, 56, 65, 32, 45, -26, 98, -31, -76, 86, -96, 62, -5, -19, 44, -43, -54, -65, 85, 17, 83, -82, -33, 29, -81, 79, -27, -49, 7, 76, -66, -87, -9, 60, -29, 50, -5, 33, -34, 13, 68, 24, -92, -35, 64, -28, 39, -58, -86, -24, 25, 35, 9, 93, 1, -12, 9, -52, 56, 53, -86, 7, 99, -53, -64, -49, 96, 79, 17, -93, -93, -23, -27, -93, -56, 75, 97, 83, -43, -74, -68, -81, 17, -77, 27, -47, -70, -65, -79, 39, -88, -59, -64, -4, -71, 81, 48],
         [-41, 87, -60, 76, -61, -7, -39, -86, -53, 94, 75, 4, -40, 25, -98, 0, -55, 27, 49, -29, -22, 37, -54, 42, -45, 18, 97, 82, -98, 45, -63, -97, 98, 66, -37, 29, -10, 79, -75, 60, 2, -49, 0, -40, 57, -14, 5, -56, -12, -7, -94, 89, -60, 46, 91, 85, 5, -32, 83, -79, -45, 15, 76, -83, -76, -34, -9, -33, 79, -5, 45, -89, -66, 20, -6, -43, -28, -5, 42, -73, -8, 59, 2, -54, 62, 8, -70, -23, -44, 91, 2, -23, -63, -74, 5, -29, -62, -88, -23, 99],
         [73, 59, 86, -56, -12, -83, -7, 12, -6, -44, -44, 14, -33, 27, 82, 64, -5, -32, -10, 29, -25, -2, 13, 56, -58, -44, -83, 25, -86, 4, -61, -38, 50, -59, 51, -72, -73, -71, -83, 19, 70, 49, 96, -100, 42, 87, -71, 34, 39, 68, 23, 33, 19, 56, -73, 1, -69, -60, 9, 27, 36, 52, 100, -66, 63, -63, 65, 97, 62, -19, 70, -65, 95, -22, -84, -33, 76, 73, 52, 7, 90, 98, 21, 98, -73, 1, -16, 4, 12, -66, -87, -6, 76, -59, 16, -83, 56, -7, -83, -41],
         [31, -27, 28, 13, -84, 70, 85, -46, -15, -95, 14, -85, 9, -46, 37, -25, -64, 68, 59, -99, 43, 24, -56, -71, 59, -71, 40, -86, 78, 33, -67, -50, -3, -26, 56, 53, -18, 25, 87, -82, -32, 82, 46, 0, -48, 6, -83, 40, -60, -16, 35, 92, -72, -5, -38, 8, -2, 84, 75, 35, -71, -53, 3, 76, -86, -67, -25, -86, 82, -72, 67, -88, -38, 9, -52, 98, 36, -94, -86, 42, -7, -14, -73, 34, 15, 51, 8, 37, -97, 95, 97, 7, -11, 59, 6, -77, -91, 73, -97, 85],
         [-69, -49, -68, -73, -55, 58, 27, -91, -9, -53, -76, -32, -94, 17, 54, -69, -65, 43, -47, -21, -64, 70, -3, 63, 11, 94, 97, 44, -29, 22, 5, 85, 90, -15, 57, 88, 44, 2, -63, 67, 98, -53, -11, -42, 37, 60, -87, -69, 4, -33, -90, -49, -50, -87, 2, -71, -28, 32, 2, -23, 69, 74, 100, -77, -6, -82, -96, 60, 58, -54, 19, 19, -25, 76, 9, 96, 66, -63, 12, 56, 83, 16, -60, -61, 47, 40, -13, -77, -10, 69, -98, -72, -2, 64, -55, -84, 7, 62, -93, 78],
         [97, 69, 44, 17, -48, -2, -2, -75, -17, 84, -39, 51, 51, -16, -65, 17, -54, 19, 19, 52, -75, -22, 13, -83, -65, 33, 46, -20, 37, 39, -82, -13, 59, -13, 78, 71, 21, -48, -72, -9, -56, -27, 36, 37, 31, 31, 84, -25, 47, 58, 87, -96, -15, 19, -12, 42, -2, -94, -27, -27, -15, -47, 3, 44, 15, -4, 56, -90, -62, 27, -40, 71, 44, 75, 9, -94, 53, -88, -59, -54, -61, -40, 46, 15, -95, 83, 60, 64, -94, -91, 47, 45, -29, 7, 78, -47, -13, 51, 9, -65],
         [13, 87, 48, -83, 39, -36, 55, 50, 58, -54, -70, 95, -19, -50, -97, 39, -37, 79, -64, -8, 40, 92, 42, 74, 32, 26, -51, 53, -81, -21, -22, -98, -46, 64, -74, 12, -66, -73, -83, -30, 18, 66, 75, -7, -73, 19, 90, -24, -66, -29, 14, 72, -26, 79, 15, -80, 36, -38, -74, 8, -28, 38, 52, 64, -63, -37, 47, 74, -33, -82, 62, 93, -92, -88, 70, 36, 1, -81, 73, -5, 42, -66, 70, -49, -45, -90, -41, -86, 84, 50, -35, -59, -54, 16, 3, -31, -92, 72, -64, -81],
         [-53, -17, -35, 4, -67, -52, 85, -7, -66, 21, 30, 37, -61, 40, 50, -82, 72, -55, 75, 41, 90, 23, 12, -2, -46, 70, 69, 27, 88, 80, -51, 20, -87, 91, -70, 46, -32, -50, 26, -8, 57, 88, -28, 97, 51, 22, 83, -24, 8, 48, -31, 15, -14, -31, 86, 16, 84, 53, 31, -80, 41, -38, 76, -26, 82, -46, -37, 69, 21, 95, 80, -78, 80, -7, 59, 39, -63, 75, 2, 1, 56, -22, 60, 58, -33, -5, -35, 65, -53, -33, -94, -92, -15, 48, -23, -44, -97, -19, -79, -22],
         [-63, 98, 1, -81, -27, 21, 56, -45, 84, -83, 96, 10, 67, 87, -31, 69, -55, -39, -57, 78, 46, -76, 47, 44, -49, -29, -42, 84, -4, 11, -74, -92, -57, -78, -23, 76, -53, -5, 83, 100, 89, -49, -52, -42, 71, -20, 53, -55, 23, -100, 73, 27, 68, -90, -29, -77, 20, -48, -16, -15, -1, -21, 69, 89, 39, -94, 68, -39, 40, -53, -54, -4, 40, -54, -87, -83, -97, 25, -45, 98, 70, -36, -85, -51, -44, 14, -51, -43, -54, 19, -31, -82, 46, -98, -86, 25, -58, 45, -84, 80],
         [-98, 91, 39, -62, 1, -20, 14, 76, 55, 58, -6, -87, 28, 56, 4, 3, 5, 79, -51, 74, -20, -62, 50, -79, -96, 12, -30, -77, 43, -34, -91, 0, -48, -51, 96, 82, -14, -87, 93, 61, -24, -18, 7, 98, 80, 62, 62, 29, 1, -89, -80, -2, -34, -6, 44, 36, -22, 90, -86, -42, -4, 84, -84, -1, 19, 31, 82, 20, -44, -14, -58, -14, 79, 95, -21, 94, -17, 68, 51, -44, 78, 60, 92, -76, -96, 74, 8, 78, -8, 2, 6, -5, -43, 99, -100, -87, 12, -42, -38, -100],
         [79, -50, 31, -35, -12, 86, 39, -46, 93, -79, 48, -8, 28, 16, -58, 97, -1, 8, -62, 55, 3, 59, -54, -48, 58, -87, -89, -86, 84, 75, -89, -7, 97, 88, -5, 34, -27, -9, 40, 25, -81, -42, 23, -75, -13, -79, 77, -12, -55, -63, 18, -73, -6, -43, -53, -40, -89, -27, 24, -2, 14, 73, -7, 84, 38, -56, -98, -45, -76, 21, -26, -65, -17, 98, -26, 66, -92, 35, -13, -75, -70, -39, 18, -90, -7, 56, 29, -67, -67, -1, 44, -90, 99, -50, -40, -19, 67, -12, -56, -53],
         [-42, -13, -27, 63, -79, 70, 82, -59, 39, -19, -48, -98, -86, -21, -8, 5, -38, 25, -52, 4, 13, 25, 5, 14, 68, -42, 26, -6, 26, -98, -71, -90, 28, -42, -62, -9, -74, 14, -67, 54, 77, 83, -56, -73, -73, 14, 69, -22, 53, 45, -62, -55, -95, -70, -92, 21, -64, -48, 65, -42, 70, -88, -100, -31, -20, 8, -41, -47, -38, 20, 15, 7, 29, 74, -97, 6, 7, 9, 64, -13, 33, 23, -63, 8, -16, -96, 77, -57, -36, 79, 29, 63, -38, 30, 3, -40, -59, -35, -74, -31],
         [-23, 45, -37, 65, -24, -12, -17, 57, -57, 70, 61, -92, 92, 20, -87, 61, -68, 68, 28, 38, 20, -43, -62, 5, 66, -95, 60, 27, -19, -62, -73, -89, -52, 23, 7, 46, 13, -44, -55, -59, 90, 100, 45, -15, -88, 52, -6, 5, -43, 18, 56, -80, -47, -10, -38, -55, 20, 12, 85, 97, -35, 13, 52, -14, 1, -15, 47, 25, -66, 94, 90, 90, -74, -17, -19, 95, -58, -64, -84, 62, 19, -97, 74, 71, 42, -53, 96, 23, 96, -90, -44, -57, -1, 34, -88, -51, -64, -60, -63, -51],
         [90, -55, 13, -18, 87, 24, -16, 52, 31, -1, 49, 18, -87, -26, 98, -52, -10, 59, 63, -73, -98, 60, 48, 11, 26, -88, -45, 73, -44, -28, 67, -20, 19, -51, -20, -92, 61, 20, -53, 60, -53, -89, 13, 71, -93, 44, -33, 79, 20, 30, -29, -65, -6, 21, 38, -92, -5, 23, -42, -96, 99, -33, -54, -96, -9, 77, -71, 97, -28, 24, -25, -93, 63, 96, -14, -75, -25, 91, -60, 62, -49, 6, 61, 82, 65, -34, 37, 95, 12, -4, 37, 99, -62, -80, 10, 59, -3, -6, 37, -43],
         [-64, -87, -38, 95, -55, 23, -8, -53, -45, -98, -9, 19, -95, -70, 23, -89, -90, 61, -11, -98, 92, -63, -10, 39, 34, 95, -96, 65, -99, -86, -8, -74, 100, 83, 16, -63, 99, -65, 37, -78, -30, -86, 35, -3, -50, -39, 96, -76, -97, -28, -43, -98, -2, 69, -21, -5, 67, 6, 23, -18, -52, -7, 93, -82, -9, -6, -14, 88, -84, -50, -46, 16, -14, 12, 57, -59, -4, -96, -8, 65, 21, -52, -11, 54, -5, 8, 99, 70, -65, 65, -37, -76, -58, 90, 43, 25, 35, -22, 12, -59],
         [-37, 73, -26, 83, -65, 75, -49, 13, -46, -29, -94, -36, -33, -24, 57, 65, 61, -95, -45, 21, -17, -47, -48, 34, 64, -3, 43, -95, 66, -29, 64, 33, 73, -65, 88, 26, -8, -42, 78, -98, 28, 44, -35, 56, -96, -89, -23, -81, 16, -84, 71, -42, -59, 54, -22, 99, 100, 14, 80, -46, -66, -74, 99, -22, 94, -13, 5, 84, 72, 49, 41, -41, 81, 55, -41, 72, 1, 96, -30, 95, -90, 44, 30, -91, 71, 43, -91, 42, 49, -17, 82, -33, 42, -11, 32, 9, 72, -39, -72, -16],
         [-41, 22, 28, 93, -92, -73, -74, -3, 2, -18, -6, -32, 99, -21, 77, 83, 79, -43, -63, -10, 27, 89, 94, -30, -18, -37, 89, 34, -59, 75, -19, 23, -36, -93, -26, -55, 57, 33, -29, 44, 34, -49, -81, -82, 58, -72, 40, -40, -98, 0, -38, -95, -68, -2, -62, 82, 57, 23, 17, -15, -96, -51, -74, -41, 29, -14, 46, 94, 67, -86, -1, -65, -98, -31, 82, 75, -16, -88, 62, -98, -90, -65, 64, 37, 82, -63, -2, 82, 39, -83, 49, -99, 74, 43, -23, 52, -20, -43, 76, -64],
         [-47, 14, 95, 76, 22, 93, 97, 27, 92, 58, -41, -28, -63, -37, 21, -11, -32, -56, 52, 65, -88, -48, -61, -43, -26, 59, -50, 48, 77, -52, -62, 95, 15, 99, -39, 74, -29, 46, 52, 60, -40, 32, 74, -20, -29, 4, -29, -84, 63, 95, 34, 24, -34, 91, -80, -14, 56, -58, -82, 40, 91, 17, 22, 12, 12, 36, -31, 12, -50, 26, -73, -65, -6, -45, -82, -30, -57, 55, 63, -87, -15, 37, -65, -95, -38, 27, -24, -88, 1, -97, -13, -47, 44, -83, 43, 64, -92, -55, 70, 29],
         [54, 47, -69, 74, 6, -60, -17, 49, -59, 31, 20, 70, 92, -73, -47, -24, -97, 7, -55, 2, 51, 92, -22, 13, 0, 93, -69, 15, -6, -17, 79, 50, -24, 17, -60, -70, -13, 27, -30, 97, 8, -26, 92, 14, 58, 96, 24, -11, -1, 94, -13, -30, -33, -17, 28, -78, -1, -4, 54, -88, -97, 19, 81, 46, -65, 37, 96, -54, -52, 89, -12, 57, 83, 12, -38, 58, -37, -31, 89, -94, 98, 98, 30, 96, -7, 12, 91, -96, 21, -14, 96, 28, 73, 46, 62, -20, -57, -7, 41, 51],
         [-7, 48, -55, 47, -81, -27, 25, 14, 36, -31, -19, 5, 64, 46, -21, 63, -56, -99, 62, 53, 64, 46, 38, -45, -52, 92, -68, 33, -65, -84, -85, 59, 99, 44, -29, 59, 25, -1, -100, 68, -36, -89, -87, -56, -18, 25, 81, 84, -9, 28, -30, 99, -2, -81, 9, -20, 87, 32, -24, -48, -64, -98, 25, -27, -79, 56, 5, -46, -9, -94, 62, -42, 71, -32, -99, 98, 0, -34, -11, 11, 7, 73, 80, 13, 65, 40, 97, 38, 93, 5, -15, -17, -5, 86, 72, 17, -10, 98, -7, -46],
         [84, -12, -15, 29, -52, -89, 68, -93, -51, -99, 45, 43, 42, 2, 85, 96, -38, 6, 85, -21, -73, -33, -18, -13, -66, -65, 95, -40, 5, -64, -90, -56, -58, 41, -95, 49, 15, -5, -29, 22, 27, 97, 19, 17, -65, -76, -57, -38, 60, 96, -9, 76, 89, 99, -85, -49, -29, -33, -66, -40, -85, -7, -55, 66, 31, 7, 42, -38, 5, 7, 99, -94, 26, -58, -61, 3, 98, -75, 50, 3, 3, -80, -22, -38, 57, 50, -2, -74, 31, -82, -29, -95, -72, 16, 69, -32, -22, 29, 72, -56],
         [-11, 96, -69, 8, -4, 84, 1, 38, -55, -82, 39, -98, -42, 49, -23, -50, -84, 11, -77, 33, 90, -97, 83, 17, -96, 50, -59, 59, 52, -33, -8, 60, -45, -70, 27, 69, -26, -42, 52, 17, -44, -44, -31, 36, -71, -53, 55, -18, 24, 57, 77, -38, 3, 15, -21, -80, -84, 81, -3, -83, 0, 25, 39, -97, 57, 84, -10, -15, 98, -47, -2, -62, 80, 86, 81, 63, -16, 58, 30, 87, 36, 76, 14, 91, 38, -82, 55, -57, -78, 40, -69, -92, 28, 65, 4, 0, -91, 85, -57, 20],
         [16, -48, 24, 23, -23, 84, -87, -16, -6, 90, -96, -23, -50, 82, 18, -66, 64, -68, 15, 51, 22, 95, 22, 34, 30, -71, 69, -68, 31, -4, -65, -96, 1, -66, 96, 46, -81, 31, 62, -28, 60, -24, -99, -40, 67, -55, -68, 26, 77, 25, -96, -53, 61, 74, 38, -39, 94, -30, 11, -93, 89, 35, -61, 5, -94, -46, -88, 100, -84, -79, 75, -47, 43, 19, -47, 49, -25, 94, 57, 68, -11, -91, 69, -39, 12, 62, -16, 79, -18, 36, 67, 96, 20, -66, 63, 73, -77, 14, 97, 56],
         [75, -20, -41, -49, 81, 62, 39, -18, 1, -82, 63, -21, -22, 79, -21, 12, 58, -83, -27, -97, -61, 62, 62, -41, 12, -86, -11, -41, 33, -75, 39, 1, -24, 57, 1, 60, 3, -98, -73, -98, -61, -81, -59, 91, -67, 42, -72, 6, -73, -37, -89, -79, 63, 41, -11, 25, 94, 60, 85, 67, 15, -46, -35, -92, 39, 98, -89, -92, 77, -83, -2, -26, -2, 50, 17, -72, -45, -3, 44, -40, -29, -32, -79, 87, -30, 52, -52, 84, -24, -6, -88, 28, 3, -38, 64, 4, 42, 80, 70, 17],
         [91, -59, 32, -11, 14, 96, -97, -59, -15, 61, 64, -78, -77, 6, 51, -5, -64, -40, 95, 13, 66, 30, 62, -5, -100, -13, 35, 12, -93, 61, 73, -76, 72, -66, 36, 89, 26, 36, 59, -100, 53, 17, -43, 31, -34, -90, 46, -37, -7, 8, -47, -88, -21, 85, 54, 19, 39, -62, 48, -92, -75, 52, -55, -81, 19, 62, 20, -77, -60, -56, -16, -26, -95, 96, 1, 37, -52, 55, -2, 13, -37, 74, -96, 22, -14, -99, 96, 98, 78, -82, 42, -39, 97, -4, -96, 21, 43, -7, -98, 24],
         [29, -36, 13, 6, -85, 41, 76, -91, -24, 3, -61, 57, -48, 52, 87, 23, -69, 33, -84, 51, 47, -96, -66, -26, -71, 7, -78, -65, -87, -13, 32, 31, 60, 23, 81, -8, 99, -4, -27, -100, 89, 92, -14, -28, 37, 74, -38, 23, 75, -29, -48, 24, 52, 56, 19, 99, 60, 91, -58, -50, -7, 37, 4, 70, 24, -53, 78, -23, 8, 1, 60, 22, -5, 69, -38, -71, -54, -53, -34, -39, 62, -19, 25, -57, -67, 69, -19, 99, 85, 98, 98, -74, -72, 76, 84, -1, 25, -100, -92, -64],
         [30, -86, -73, 36, 46, -90, -60, 23, 58, -46, 32, -26, 13, 8, -27, -75, -65, -37, 9, -24, -48, -33, 24, 27, 9, -58, -15, -14, -44, -99, -77, -48, 32, -92, 63, -94, 10, 30, -1, -59, -38, -24, 95, -46, -81, -94, 79, -30, -50, -91, 72, 100, 43, 84, -45, -98, -23, -68, 2, 23, -54, -17, -26, 84, -78, 80, -9, 59, 48, -55, -12, 39, -89, 87, -66, 11, 73, 39, 80, 25, 6, -23, 53, 73, -2, 38, 70, 75, -87, -71, 63, -57, 15, -56, -19, 8, 39, 10, -55, 52],
         [42, 5, 17, 23, -46, -96, 26, 85, 94, 63, 43, -7, 74, -2, -93, -45, -5, -77, 73, -50, -80, -43, 74, 80, 87, -89, 29, 54, -64, 94, -11, -45, -89, -97, 35, -33, 84, -89, 8, 28, 52, -2, 90, -89, -51, 62, 27, -71, -33, 5, 17, -22, 70, 49, 58, -90, -4, 59, -68, 28, -91, 60, 38, -6, 69, -3, 3, -5, 48, -75, -65, 78, 67, 52, -60, -35, 66, 10, 1, 20, 34, -27, 68, 5, 82, -95, 96, 37, 59, -63, -34, -12, -98, -43, 81, -12, 67, -25, -4, -35],
         [-68, 38, -40, 56, 30, 40, 81, -26, 50, 71, -28, 17, -18, -49, 65, -8, 46, 54, 48, -63, -32, -72, 66, 91, 48, 43, -46, -58, -84, -86, -89, 75, 97, -1, 86, -49, 28, -84, -66, 31, 92, -76, -4, 78, 59, 79, 23, -49, 76, 0, -75, -80, 92, 69, -2, -39, -79, -94, -53, -58, 26, 98, 30, -13, 73, 54, -75, -86, 13, 95, 3, -72, 53, 61, -19, -44, -61, 23, -55, 90, 40, 76, 90, -2, -84, 52, -85, -18, -48, -26, -27, 2, -33, 55, 98, -46, -70, -66, -8, -19],
         [18, -62, -100, 19, -64, 44, -52, -87, -33, -10, -85, 83, 60, 11, 28, -35, 40, 5, -39, 28, 26, -68, 16, 46, -73, -28, -7, -72, 78, 52, -90, -26, -94, -21, -28, 27, -55, -27, 9, -15, 96, -73, -50, 0, 84, 7, -82, -40, 23, 99, -53, 66, -25, 36, 11, 54, 86, -89, -32, -69, 31, -97, 54, 56, -4, 19, 84, -63, -30, -17, 92, 93, 72, -68, 3, -100, -24, 8, 13, -37, -21, -72, -7, -44, -65, 14, 34, -22, 31, 5, -36, 85, 95, 26, 43, 2, -69, 2, 64, -25],
         [-15, -65, -7, 21, -48, 82, 10, -85, 92, 58, -94, 1, 87, 47, 49, 63, 66, -28, 15, 76, -60, 94, -35, 18, -71, 45, -42, -23, -21, 90, -48, -28, 98, -72, -25, 41, 41, 57, 86, 60, -35, -36, -82, 98, -3, 38, 25, 80, -31, -12, 32, 12, -26, -99, -17, -16, -87, -1, 80, -69, 9, -27, 37, 3, -85, 49, -6, -85, 89, 30, 51, 0, -74, 40, -83, -52, 90, -2, 78, -90, 39, -39, -20, 84, 32, -100, 46, -41, 98, -25, 60, -75, -78, -37, 70, -42, -52, 40, 55, 95],
         [-23, -8, -98, 100, 89, 92, 59, 34, -90, -16, -3, -59, 90, 36, 51, -3, -92, -16, 83, 85, -49, 63, 24, 68, 14, 61, -70, 66, 44, -29, 0, -99, -96, -64, 43, 63, -42, 63, 10, 24, 40, 70, 40, 67, -45, -24, 80, -26, 97, -36, -45, -4, -68, 19, -25, -76, -81, -2, -42, -77, -23, -18, -66, -23, 90, 96, 20, 48, 2, 43, 77, 80, -24, 88, -94, 98, 77, -68, 77, 45, -93, -78, -34, 35, 76, 77, -83, 69, 46, -50, -75, -18, 93, 49, 53, 92, 17, -74, -16, -42],
         [62, 76, 17, -75, 76, 100, -17, 37, 45, -64, -46, 5, -79, 57, -28, 20, -66, -83, 74, 86, 4, -88, 15, -72, -97, 45, 7, -71, 99, 58, 32, 7, 57, 40, -13, 93, -14, -77, -53, -77, 84, -19, 42, 78, -7, -34, -1, -56, -91, 30, 31, 55, 28, 13, 55, 33, -92, -17, 50, -93, 66, -100, 86, -53, -4, -1, -76, -43, -36, 70, -59, -23, 4, 8, -30, -13, -34, -53, -43, 46, 75, 49, -48, 86, 52, -38, -3, 99, 30, -75, -35, -82, 100, 23, 44, 75, 21, 91, -59, -95],
         [16, -70, 34, 21, 15, -36, 3, 75, -4, 8, 86, -71, 64, -60, -4, -44, 99, -10, 26, 97, -97, -85, 25, -85, -14, 86, -88, -39, 38, 53, 9, 90, -97, 28, 12, -19, -87, -42, 31, 0, 40, -34, 70, -62, -94, 69, 32, 68, -49, 2, -94, -25, -31, -15, 12, -95, -15, -18, -67, -47, -21, -91, -92, -12, 59, -55, 42, 61, -24, -17, -42, -19, 85, -13, 36, 16, -7, 33, -54, -9, -54, -28, 23, 71, -35, -90, -27, -75, -25, -81, 93, 0, 73, 50, -93, -68, -98, -30, -94, -1],
         [-38, -57, -68, -73, -72, 73, -55, 90, -54, -89, -26, 87, -99, -44, 24, -86, -59, -88, -27, -87, 7, 63, 74, -1, -10, -14, 60, 97, 90, 80, -89, -39, 26, -16, 79, 95, 14, 60, 52, 69, -58, 97, 13, 18, 5, 69, -22, 1, 18, -83, 54, -29, 53, -64, -93, -18, -85, 31, -54, 5, -88, -29, -74, -62, -49, 89, 12, 99, 82, 66, 7, -30, -82, -76, -32, 18, -38, -13, 28, 15, 79, 79, 30, 69, 50, -29, -89, 4, -12, -13, 79, 30, -10, -18, -16, 1, -76, 41, 98, -43],
         [21, -31, -37, -67, -13, -79, 97, -80, -56, -93, -28, 2, 14, -91, 5, 56, 5, 96, -81, -84, 34, -27, 92, 70, -1, 93, -77, 73, -32, -16, -21, 4, -64, 87, -36, -74, 52, 18, -3, -94, -43, 11, 81, -18, -64, 18, -33, -15, 41, -31, -52, -53, -57, 11, 61, 36, -73, 86, -68, 53, 32, -93, 43, -7, 94, 20, -41, 83, 40, 85, -56, -90, 65, -7, -41, -80, -48, 93, -44, -13, 20, -97, 60, -30, -82, -94, -97, -13, -66, 97, 84, 76, 29, -43, 20, -58, -49, 19, 86, 9],
         [23, -4, 0, -64, 38, 73, -85, 100, 98, -80, -20, 20, -54, 46, 46, -35, -65, -42, 15, -91, -12, 97, 66, -58, 34, 4, -1, 72, -79, 58, 86, 71, -75, -80, 33, 48, -19, 62, 43, -79, -49, 43, 18, -73, -12, -91, 32, 93, -4, 68, -52, 54, 39, -17, 77, 80, -96, 29, 23, 65, 21, 90, -81, 33, 22, 85, 57, -74, 8, 17, -42, 67, -89, -34, 70, -67, 90, 41, 61, -35, 32, -70, -92, 100, -15, -26, -92, -83, -60, -15, -76, 34, -28, 26, -94, 27, 94, 96, 99, 81],
         [32, -75, -61, -32, 96, 51, -25, 45, -45, 95, -27, -70, -30, -99, -35, 78, 52, -64, 28, -17, -85, -47, 60, -57, 70, 0, -95, -80, -32, 29, 38, -23, -3, -19, -56, -7, -81, 29, 27, -38, 50, 45, 95, -73, 57, 84, 49, -79, 72, -77, 65, -28, -44, 50, 67, -41, 64, -83, -6, -62, 5, 16, 6, -62, -78, 52, 48, -92, 20, -93, -29, -2, 73, -85, 15, 65, -67, -35, 47, -75, -4, -96, 51, -62, 20, 26, 84, -42, 95, 6, -46, -60, -2, -29, -13, 18, -50, 57, 1, -98],
         [-29, 48, 98, -80, 7, -62, 22, -46, 56, 47, 97, -61, 22, -63, -91, 72, 89, 10, 47, 97, -1, 17, -90, 55, 45, 6, 82, -83, 25, 48, 43, -27, 59, 45, -9, 5, -10, -89, -71, -42, 57, 7, -28, -22, 38, -16, -20, 94, 12, -12, -33, 16, 80, 36, -49, -56, -87, 72, -76, -51, -7, -49, -8, -22, 72, -80, 10, -72, 6, -69, 84, 97, -42, 48, -65, 41, 89, -91, -94, -34, 85, 53, 50, 97, -85, 74, 76, -7, 64, -7, -94, -56, -54, -17, -24, -36, 16, -6, 41, -36],
         [13, -35, -15, -41, 33, 32, -53, 75, 11, -56, 6, -23, -84, 25, 73, -12, -70, -21, -71, 13, -73, -98, 99, -59, -76, -90, -69, -71, 78, 89, -70, -35, 96, -34, -63, 39, 17, 31, 95, 68, 83, 97, 83, 18, -76, -85, -24, -71, -88, 71, 59, -55, -27, -1, -41, 74, 28, -69, -4, 75, 82, -98, -91, 22, -65, 52, 89, 57, -30, -28, 53, -52, 42, -64, 82, 66, 10, -49, 95, -94, 95, -25, 28, -71, 35, 89, -37, 66, 94, -57, -56, -63, 26, 69, -14, -1, -17, -86, -65, -50],
         [-41, 21, 16, 84, -73, 6, -38, -36, -53, 33, 75, 93, -10, 77, 37, -75, -76, 97, -91, 3, -8, -47, -100, 40, -86, 76, 79, -60, 79, 10, -45, -25, 55, -65, 13, -60, 87, 82, 46, -37, 52, -90, 43, -92, 67, -74, 95, -10, 72, -22, 49, 70, 26, 60, 58, -2, 30, -10, 54, -61, 39, -51, -9, -9, -41, -88, -89, 78, 22, 57, -54, -53, -74, -81, -99, -87, 72, 70, -91, 8, -33, 88, 34, 51, -31, -86, -6, -36, 68, -91, 59, -96, -52, 60, 77, -24, 72, 80, 52, -79],
         [-10, -71, 14, 83, 66, 15, 0, 99, 65, -48, 88, 23, 80, 28, -2, -38, 7, -71, -87, 88, 25, 18, -57, 10, -23, -46, -52, -37, 87, 22, 43, -25, 31, 54, -22, 10, -7, -87, 4, -69, -14, -83, -42, -65, 35, 59, 16, -41, 73, -2, 88, 2, -83, -67, -27, -75, 93, 13, -18, 48, 19, 22, 48, 18, 14, -20, -25, 5, -75, 43, -16, -29, 53, -72, -68, 78, -80, 3, 11, 87, -5, 21, -95, -1, 43, 11, -96, 93, -39, -67, 91, 68, 91, 96, -52, 54, 84, -29, -92, -86],
         [85, -32, 63, -68, -69, -96, -53, -3, 74, -62, 34, -13, 82, -15, -78, -80, -84, -49, -9, -73, 79, -47, 24, 93, -35, -46, 24, -21, 81, -32, 71, -30, 2, 89, -48, 2, 26, -51, 18, 99, 6, 17, 67, -80, -42, 44, -16, 4, 65, -67, 96, -18, -84, -7, -96, 20, -78, 6, -19, 91, 96, -51, -74, -2, 71, -39, 85, -2, 100, 44, 10, 47, -52, -85, 30, -8, -98, 81, -19, 48, 61, 28, 48, 57, -91, -61, 62, 100, 47, -57, -80, 67, -19, 12, -69, 94, -70, -51, 33, -9],
         [80, 87, -46, 78, 56, 48, 44, -32, -65, 8, 27, 25, 40, 53, -70, 60, -6, -77, -45, -76, 91, -90, 82, -49, -43, 10, -62, 65, 35, -27, -86, -54, 33, -54, -74, 29, -46, 72, 91, -14, -90, -2, 7, 49, 5, -77, -83, 21, 80, -75, -53, -46, 38, 77, -71, -73, -67, 4, -31, 98, -9, -26, -88, 13, 60, 45, -70, -84, 78, 72, -12, -23, -22, 89, -58, -71, -49, -53, 6, 19, -100, 25, 88, 39, -86, 44, 42, 94, 95, -80, -19, -48, -12, -82, -80, -67, -17, 55, 3, -51],
         [-13, 31, 13, -44, -55, 51, 72, 18, -68, -27, 100, 97, 32, -39, 57, 46, 55, 9, -79, 95, 45, 68, 88, 3, 40, 14, -7, -11, 82, -10, 61, 56, 26, 22, -79, -89, 21, 39, 57, -75, -55, -3, 46, -35, 79, 65, 80, -88, -18, -62, 75, -52, -25, -35, 48, -9, -79, -57, 45, -77, -86, -27, 27, -65, -13, 76, 84, 97, -76, -55, 18, -84, -49, -56, -50, 25, -75, 60, 89, -44, -19, -94, -82, 94, 64, 98, -78, 41, -61, -94, -96, 29, -39, 84, -33, -86, 60, 64, -21, -41],
         [72, -33, -32, -91, -91, -56, -21, -34, -62, -69, 15, -57, -94, -88, 44, -95, 91, -47, 28, 51, -96, 80, 41, -28, 86, 99, 79, 15, -29, -33, -75, -70, 15, 34, -10, -74, 76, 89, -51, 51, 39, -84, -59, -74, -61, 8, 69, 96, 82, 14, 58, -81, 77, -94, 43, 66, 99, -75, 14, -41, -7, 66, -2, 45, 71, -12, 48, 38, -66, 32, -57, -29, -72, 54, 52, -92, -87, -11, -13, -50, -42, 39, -72, -90, -90, -39, 4, -87, 26, 84, 0, -93, -84, -95, -24, 29, 52, 55, -52, 31],
         [-97, 7, -90, -54, -87, -6, 96, -65, 99, 82, 32, -9, -48, 92, -21, 90, -89, -90, -24, -52, -72, 39, 8, -94, -61, 44, 30, 95, 0, 86, 8, 100, 48, 10, -47, -71, 19, -54, 51, -63, 1, 92, 40, -48, -56, 41, -39, -84, 71, -18, -12, 55, -9, -85, -82, -71, 30, 57, 84, 59, 36, -3, -37, 23, -7, 28, 64, 1, -76, -9, -64, 26, -12, 56, -30, -27, 33, 93, 1, 100, 75, 78, -15, -90, 73, 57, -41, -54, -8, -17, -16, -59, -50, 33, 81, 93, 51, 32, -44, -98],
         [38, 84, -87, -12, -62, -99, -74, -90, 37, -90, 95, -86, 1, -51, -20, -77, -79, 1, -7, 74, -4, -97, -25, 56, -25, -25, 94, -58, -49, -62, -13, 50, -84, 59, -2, -45, 39, -49, 2, -10, 52, -92, -71, 91, -98, -44, -17, -95, 1, 71, 14, -6, -25, 56, 28, 68, 58, -79, 5, -89, 10, 21, -3, -50, -91, 11, 75, -48, -4, -18, 72, -95, -22, -60, -26, 68, -10, 15, 54, 100, 20, 82, 4, 25, -75, -90, 7, -2, 72, -24, -86, 29, 41, 81, 58, 87, 48, -59, 23, 93],
         [-64, 70, 74, -28, -26, 18, -45, -89, 22, -99, -5, 91, 67, -62, 83, -59, -86, 53, 2, -85, -14, 0, 21, -10, 84, 4, -61, 13, -48, -73, -64, -95, 82, 13, 73, 3, -83, -78, -84, -44, 40, -59, -43, 21, -46, 23, -33, -67, 44, 83, -37, -14, -99, 90, -26, 3, 71, 58, 23, -23, 22, 86, -62, 97, 42, -74, -39, 98, -61, -1, 45, 12, -74, 8, -94, -17, 97, -85, 51, 49, 77, 60, -33, 5, -25, -64, -17, 35, -64, 95, 3, 50, 45, 54, -26, -65, 18, -91, 25, -89],
         [-66, 15, -8, 21, -13, 70, -9, -37, 12, 75, -29, 17, -6, -63, 100, 30, 82, 30, -76, 81, -48, 66, -24, -30, 63, -30, -76, 44, 7, -29, 2, -6, 94, -89, -49, 63, -99, -63, 51, -73, -100, -54, 6, 62, -91, -14, -5, -73, -60, 0, 12, -50, -9, 23, 39, 63, -95, 67, 8, -77, 38, 3, -42, 89, -91, 25, 43, 68, 4, 37, 87, 63, -35, 84, 66, 44, 98, -2, 36, 79, 72, 58, -62, 10, 50, -64, -39, 66, -55, 57, 46, -29, -31, -32, 44, -96, -77, 1, -92, -37],
         [54, -35, -31, 62, 83, -51, 11, 35, 36, 85, 26, 48, 11, -5, 10, 36, -55, -15, -18, -43, -52, 31, -44, -2, 68, 2, -44, 24, 5, 36, -5, 18, 0, 62, 94, 21, 67, -96, -94, -11, 32, 96, -55, 96, 78, -39, -64, -28, -14, -92, -35, 11, 57, 88, -87, -44, -72, 75, -40, -24, -35, 10, -75, -65, 29, 51, 55, -81, 100, -83, -83, -54, 99, -13, 97, 64, -18, 36, -10, 71, 27, -49, 95, -37, 28, 83, 17, 55, -11, 95, 48, -3, -32, -52, 0, -1, -65, 99, -99, 58],
         [0, -3, 9, -92, -70, 5, -60, 70, -19, -25, 60, -54, 67, 97, -67, -100, -94, -95, -1, -86, -34, -61, 61, 85, -87, 84, 6, 5, 99, -95, 47, -59, -40, -64, 2, -81, -52, 25, 52, 97, -42, 49, -16, 4, 71, -92, 20, 53, 9, 53, -42, -76, -68, 34, 35, -64, -89, 45, -33, 8, 34, -47, 26, 98, 58, -38, -2, 23, -62, -18, -60, 3, 79, -39, 51, 65, -6, 77, -13, -5, -40, 18, 26, 11, -48, 73, 71, -28, -93, -90, 30, 73, 20, -39, -30, 60, 29, 42, -52, 32],
         [39, -60, 63, 7, -93, -54, -43, -43, 91, 94, 99, -47, 84, -42, -38, -63, 70, 52, 23, -98, -26, -32, 91, -69, 85, -80, 95, -92, 85, -1, 94, 28, 25, -65, -99, 86, -97, -13, 9, -15, 98, -80, -69, 49, 73, 71, -32, 79, -80, -50, -3, -21, -99, -91, 25, 98, 76, 11, 22, 55, 51, 14, -36, 8, 60, 98, -25, -25, 71, -78, 23, 11, 3, -65, 86, 43, -39, -18, 78, 27, 5, -5, -65, -41, -28, 13, 51, 36, 65, -74, 1, 80, -95, 56, -42, -79, 7, 82, 87, 70],
         [57, -87, -20, -90, -26, 19, 79, -19, -14, 73, 89, -43, 24, 79, 38, -60, 34, 19, -28, -24, 94, 19, -25, 15, 97, -20, 26, -96, 95, 70, 26, 6, 8, 50, 62, -11, 61, -72, -8, 91, -98, 22, 76, 86, -23, 20, 17, 70, -8, -42, -23, 91, -13, -19, -78, -81, 63, -18, -28, -58, 66, -22, 13, -83, 16, -13, 21, -97, -14, 87, 54, 99, 76, 94, -99, -32, 2, 97, 17, 5, 86, 41, -50, -47, -9, -73, -71, -16, -33, -80, -73, 31, -81, 86, 41, 29, 28, 91, 22, -68],
         [-99, -39, -46, -2, 81, 99, -26, 9, 46, -21, -29, -93, 84, -87, -82, 40, 20, -60, 25, 88, 98, -88, -95, -66, 49, -32, 3, -32, 99, -7, -70, 17, 16, 13, 9, -44, -97, -62, -87, -87, 97, -70, -77, -89, -34, 54, 47, 62, -29, 67, -15, -71, -50, 84, -25, 11, 28, -72, -2, 96, 90, -33, -26, -98, 97, 23, 93, -78, -72, 6, 71, 31, -27, -39, 68, 15, -19, 29, 71, 3, -97, -57, -18, 91, -1, -69, 45, 65, -44, 26, 10, -57, -11, -48, 58, -34, -41, 31, -45, 0],
         [87, 78, 48, 77, -68, -84, 13, 29, -28, -76, -7, -68, -18, -62, 64, -32, 14, 59, 87, 54, -79, 27, -78, 24, -44, 40, 55, -41, 85, -11, -93, -24, -79, 40, -3, 36, 80, 61, -13, -37, -15, -37, -24, -98, 88, -73, -72, 32, -72, -74, 97, -18, 41, 41, 23, 46, -77, 55, -33, 27, 7, 72, -30, 69, -62, 79, 91, 53, 2, 80, 43, -70, 38, 27, 44, 89, -13, 52, 53, -89, -20, -46, -94, 18, 23, -63, 2, 96, -36, 36, 56, 38, 81, -19, 25, 32, -36, 39, -79, 76],
         [32, -31, 80, 87, -82, -22, 7, 22, 85, -99, 23, 9, -99, 15, 37, -45, -94, -99, 49, -20, -61, -31, -76, -63, 88, 46, -70, -73, -94, 58, 69, -49, 80, 91, 8, -45, 93, -55, 92, 86, -46, -54, 81, 1, -11, 87, 31, 46, -31, -43, -41, -34, 43, -46, 5, 98, -19, -55, -5, 32, -86, -60, 92, -48, -80, 78, -63, -39, 0, 23, -37, -56, -23, 20, -25, 97, 60, 100, 46, -65, 24, 73, 84, 49, -27, 97, -82, -97, -12, -44, 53, 73, -4, 87, 31, -5, 40, -99, 66, -97],
         [-5, -70, -76, -28, 8, 14, 30, -9, -66, 40, 81, -41, -85, -63, -100, -63, 26, -83, -97, 91, 82, -98, -26, 58, -34, -59, -37, 22, 95, 73, 75, 74, 83, 17, -90, 22, 27, -85, -37, 52, -24, 32, -66, 19, 37, 88, -31, 10, 5, 8, 38, -47, 50, -91, -34, -16, 76, 63, 76, -90, 35, -71, 29, 65, -9, 31, 21, 76, 20, 80, -34, -16, 61, 26, 32, 16, -13, -61, -45, -48, 4, -20, 16, -16, -15, -4, 9, 31, -98, -3, 21, 61, 93, 9, -26, -36, 21, -37, -9, -92],
         [59, -52, -95, -45, -64, -28, 11, 31, 32, -36, 38, -88, 37, 49, 85, -77, -100, 44, 79, 94, 27, 13, -68, -3, -14, 45, 61, -34, 3, -32, 99, 30, 51, 29, 72, -26, 24, 92, -98, -67, -94, 44, 19, 37, 22, 3, 35, 99, 98, 21, 67, -56, -55, -55, 13, -86, -36, 38, -50, -2, 9, 72, -73, 87, 8, -83, -89, 80, -29, 58, -1, -10, 21, 70, -9, -40, 62, 16, -95, 52, 35, 93, 71, 97, -45, 64, -19, -38, 58, -52, -69, 95, -55, 97, -50, 30, 56, 78, 10, 51],
         [-3, -9, -42, -49, -27, -7, 52, -68, -80, 29, -14, -57, 19, 9, 16, -82, 67, 44, 71, -64, -15, 31, 8, -28, -68, 23, 6, -93, 68, 58, -24, -93, -41, -23, -90, 42, -1, -47, 81, -9, 64, 34, -20, 75, 61, -94, -95, 51, 53, 72, -60, -39, 59, 28, -98, 31, 34, 55, -47, 61, 11, 38, 19, -6, 42, -29, 77, 100, -93, -12, -22, 13, 5, 11, 66, -83, -93, -34, -24, -81, -83, -63, -10, -95, -55, 19, -13, 31, 43, -22, 82, -33, 34, 39, 40, -73, 79, -74, 20, 91],
         [48, 98, -70, 93, 44, 35, 58, -83, -21, -72, 9, 55, -76, 71, -45, 17, -28, -52, 89, 32, -1, -43, 31, -73, 12, -85, -55, -75, 71, -26, -14, -48, 18, -47, -68, -39, 33, 74, -8, 25, -41, 15, -38, -10, -26, -35, 78, -6, -16, -1, -58, 29, 94, -73, 54, 0, 39, -29, -13, -86, 35, 98, -69, -45, -80, -91, 96, -78, 44, -100, 17, 44, -48, -61, -66, -41, -72, -19, 19, -83, 2, -2, 37, 24, 25, -74, 52, 29, 23, 64, -97, 61, -77, 62, -38, -82, 17, 41, -14, 61],
         [86, -10, -1, -8, -51, -95, -66, -2, 31, 82, 63, 28, -31, 63, 7, 45, 26, 32, 66, 40, 97, 50, 75, 63, -8, 96, -67, -25, -62, 13, 71, -92, -74, 45, 12, 18, -60, -36, 30, -40, 67, -28, 56, -60, -41, 40, -67, 0, -52, -35, -62, 32, 45, 8, 89, -38, 77, 52, 29, -73, 28, 63, -67, -39, 82, 60, -86, 58, -71, -39, 91, 63, -60, -4, -35, 83, -37, -94, 89, 4, -49, -22, -23, 94, -56, 63, -10, 50, 17, 35, -77, 59, -100, 56, -69, 88, -95, -95, -32, -87],
         [89, 81, 82, 49, 81, -70, 97, 37, -48, 62, -3, -57, 48, -50, 55, -92, 14, 16, -67, -10, 54, -10, 62, -5, -31, -64, -13, -61, 77, -4, 14, 73, -82, -88, -77, 17, 20, -29, 6, -43, -59, -77, -7, -71, 36, 94, 89, 90, 9, 54, -80, -6, 99, 67, -57, -11, -96, -45, -19, 42, 22, 63, -33, 65, 75, -95, -18, -35, 91, -68, 64, -64, 34, -97, -52, 51, -97, 5, 32, 74, 95, -34, 1, 73, 7, 63, 19, -79, -77, -18, -28, 95, 3, 35, 38, -23, 78, -89, -50, -88],
         [-64, 3, 93, 8, -13, 8, -82, -93, -22, -30, -13, -73, 77, -53, -15, 100, 33, 65, 29, -81, 84, 40, 35, 1, -87, 31, -67, 37, 75, -14, -64, -7, -61, -79, 38, -8, 47, -34, -89, -26, -78, -50, 3, -16, -73, -99, 92, -56, -15, -75, -41, -16, -19, -22, 67, -12, -97, -12, 56, -29, 72, 82, 45, -67, 63, 85, 45, 88, -54, -16, 19, 28, 67, -89, -87, 53, -2, -91, 77, 9, 73, -20, 8, -94, -93, 93, 2, -58, 67, 2, 26, 88, -49, 33, -24, 61, 2, 18, -33, -92],
         [53, 3, 54, -54, -57, -30, 26, 30, 29, -90, -47, -76, -39, -41, -55, -67, -83, 58, -100, 71, -30, -59, 58, -97, -43, 89, 6, 34, -4, 29, 33, -5, 40, 66, 87, -19, 12, -49, 39, 43, 79, -9, 74, -11, -53, 26, 16, 9, -9, -94, 63, -99, 53, 93, -3, -42, 40, 73, -86, 9, 65, 27, 17, -17, -31, 92, -32, -65, -88, -92, -34, -57, 64, -78, 94, -5, -37, -67, 97, -14, -36, -14, -74, -56, 13, 54, -99, -17, -74, 74, -77, 73, 94, -64, 53, -64, 60, -62, -91, 69],
         [-39, -19, 94, 42, 76, 66, 7, 0, 1, -82, -31, -55, -69, 14, 2, 84, 3, -52, 62, 74, -65, -99, -26, -72, -73, -3, -96, 99, 66, 40, 82, -43, -59, -50, -29, 86, -21, -79, -42, 40, -62, 77, -29, -10, 95, -53, -25, 21, 17, 41, -82, -98, 44, 7, 89, 40, -38, -8, -23, -94, -60, -94, -54, 0, -53, -75, -86, 34, 74, 62, -41, -92, -39, 18, 69, -31, 50, 89, 64, -19, 79, 69, 89, 53, -49, -4, -61, 40, 69, 11, -54, -49, 87, -61, -30, 73, 87, -8, 86, 2],
         [-16, 42, 72, 77, 30, -43, -24, 64, 59, -55, 99, 97, 43, 45, -60, -22, -40, 61, -19, 17, 42, -95, -20, 25, 22, -47, 47, -11, 78, 77, -98, 23, -89, 13, -25, 85, -81, 21, 26, 44, 62, 67, -1, -27, -72, 25, -63, -17, -39, 85, 38, -92, 98, -63, -79, 97, -21, -99, 16, -75, 47, -3, -31, -86, -30, -90, 75, -3, 96, -12, 47, 4, 31, -69, 83, -13, 36, 4, 53, 97, -13, 72, 9, 74, 40, -26, 50, -83, 92, 99, -33, -96, 14, 92, -18, -93, -95, 77, -76, 45],
         [-18, -51, -39, -8, 3, -16, -22, 45, -87, 84, 9, -22, -66, -18, 53, -64, 89, 22, 61, 16, 95, -42, -84, 28, -26, -62, 36, -71, 61, 12, 65, 74, -13, 84, 18, -41, -64, -35, -27, -81, 56, -6, 36, 93, -19, -90, 44, -77, -71, 65, 81, -43, -10, 31, -74, 57, -13, -52, 18, -46, 85, 75, -100, -37, 50, -18, 14, -41, -92, -65, 14, -95, 89, 48, 36, 43, 13, 86, 55, -70, -87, 65, -27, 61, 32, -65, -48, -50, 45, 86, -18, 21, 78, 61, 79, -46, 6, 44, -72, 41],
         [-69, -66, -8, -18, -66, -7, 67, 82, 7, 97, -64, -73, -59, 91, -4, 77, -51, 52, 84, 94, -39, -21, 5, 56, -88, -1, 1, 70, -75, -64, 66, 43, -30, 12, -58, 13, -41, -90, -89, 83, -96, -89, -42, 14, 35, -18, -44, -32, 30, 32, -30, 90, 95, -42, -10, 67, -27, 29, 13, -58, -3, 31, 36, 49, -62, 81, 31, 67, -46, 16, -99, 60, 17, 47, -64, -67, 37, -88, -23, 7, 4, 52, 19, -31, -24, 85, -81, -88, 84, -90, 10, -72, 20, -82, -13, 26, -64, 33, 29, 93],
         [42, 19, -68, -76, -47, 35, 24, 18, 98, -3, 14, -96, 84, 75, 97, -94, 1, 7, 89, 71, 83, 34, -26, 24, 67, 26, 83, -78, -76, -20, -84, -2, 97, 24, 85, 72, -71, 52, 100, 21, 28, -60, -26, -37, -42, -67, 14, 22, 83, -3, -69, 3, -20, 5, -11, 42, 67, 52, -60, -17, 31, -80, -11, 54, 45, 85, -85, -98, 20, -82, 68, 19, 18, 8, 35, -33, -30, 100, 0, 33, -64, 43, 54, 70, -62, -48, -65, 75, -53, -55, -45, -48, 92, 87, -57, 11, -48, -84, 7, 30],
         [34, 51, 81, 79, -52, 35, -26, -67, 6, -49, 23, -55, 89, -61, -35, -13, 54, 100, -36, 54, -24, 27, -45, -79, -96, -89, -41, 18, 25, 1, 66, 43, 80, -6, -29, 40, -68, 91, 60, 4, 53, -40, -8, -66, -88, 81, 91, 97, -96, 64, 92, 79, -38, -6, -41, 74, -43, 96, 94, 36, -56, 46, -79, 74, 4, 95, 16, 26, 20, -91, 92, 33, -22, -20, -71, -69, -32, 38, -44, 88, 53, 31, -74, 44, -70, 37, 29, -46, 50, -60, 41, 5, 63, 27, -96, -42, 26, 62, -27, 77],
         [19, 71, 86, 72, -51, -33, 39, -50, 39, 64, -31, 75, 21, -45, -33, -39, 9, 50, -43, -58, 45, -8, 41, 73, 78, 11, -1, 22, 4, -65, 71, -17, 8, -7, 20, 26, 43, 51, -74, -41, 41, -47, -27, 9, 42, -60, -88, 88, 74, -34, 96, -68, -42, 60, -14, 90, 41, -42, 46, 42, 30, 84, -58, -82, -10, 74, 9, -3, 89, -78, 8, 61, -60, -46, -27, -84, 68, 84, 63, 64, 65, -72, -55, -71, 84, 44, -53, -65, 53, -39, 90, 17, -40, 15, 38, -77, -76, -1, -48, -71],
         [44, -93, 23, 44, 45, -14, -45, 53, -88, 85, 67, 31, -79, 35, -49, 46, -23, 73, 73, 34, -46, 61, -17, 65, 59, -68, -34, -55, -25, 4, -55, -54, 19, -22, 18, -65, -30, 7, -11, -6, 55, 82, -53, -72, -87, 92, 31, 14, 2, -8, 77, 38, 31, -70, -60, 9, 92, -14, 18, 36, -68, -64, 30, 5, -39, 64, 94, -15, 45, -24, -24, -55, -54, 44, 95, -37, -92, -96, -51, 99, -13, 4, 0, 62, 88, 32, -4, 21, -70, 79, 9, 80, -89, 52, 17, -38, -97, 97, -7, 44],
         [10, 95, -73, -53, -77, 12, -14, -83, -98, -3, 98, 92, 18, -36, -26, -25, 8, 3, -24, -49, -1, -78, 39, 26, -15, -36, -41, -43, 29, -93, 90, -33, 72, 31, -62, 89, -38, -17, -87, -60, -82, 78, -53, 91, -70, 21, 89, -2, 24, -82, 66, -39, -17, 10, 90, -54, -5, 75, 57, 36, 47, -70, 13, 91, -23, 3, -70, 57, -91, 43, 17, 47, 66, 84, 53, 50, 60, -87, -64, -84, 87, -44, -99, 82, -61, -33, 5, 95, 46, -31, 65, 2, 36, -98, -13, 51, -6, -35, -75, -75],
         [85, 35, 42, -64, -13, -87, 90, 86, 89, 25, 4, -77, -3, -96, -9, 47, -35, -100, 29, 33, -58, -30, -71, 53, -93, 20, 65, -96, -66, -51, 37, 85, 80, -44, 0, -57, -64, 38, 48, 25, 93, -91, -55, -41, 25, -22, -39, 53, -94, 46, -31, -57, -71, -97, -20, 44, -38, -63, -38, 23, -73, 49, -89, -41, 54, -28, 77, -53, -23, 57, -18, -54, 11, 47, 33, 61, 52, -50, -64, 62, -4, 92, 52, 13, -40, 15, 51, 74, 35, 38, -22, -9, 36, -63, 30, -50, -95, 57, -57, -49],
         [16, -68, -94, 69, 10, 42, 51, -97, 57, -20, -77, -67, 27, 86, 52, -25, 37, -78, -3, 3, 28, -55, -61, -5, -51, 39, 84, -35, -67, -4, 41, -13, -14, -54, -90, -11, 47, -34, 56, -82, -77, -97, -67, 83, 6, -59, -41, -97, -40, -75, 34, -64, -17, 56, -55, -45, -51, 27, -89, 77, 22, 19, 81, 56, 23, -35, -72, 54, 6, -78, 73, -90, 72, -23, -91, -75, 19, 98, 1, -4, 8, 44, -92, -7, 36, -8, -68, -7, 83, 62, 3, -83, 18, -68, -83, 7, -100, 59, 60, 80],
         [82, -72, -91, 71, 58, 34, 84, -41, 28, -92, -67, -18, 80, -76, 56, -41, -73, 58, 22, -92, -87, -46, -27, 32, -18, -6, 96, -97, 82, -35, 43, -57, 6, 74, 5, 91, 54, -65, -87, 46, 33, -17, -77, 16, -84, 48, -2, 40, -62, 10, 61, 38, -49, 10, -16, -25, 6, -54, -34, -37, -63, 50, 0, 1, -6, -17, 44, 45, -50, -81, 41, -57, 99, 55, 8, 87, 18, 90, -81, 2, -26, -6, 64, 88, 27, -13, 9, 23, 93, 81, -29, -60, -37, 22, -49, -75, 35, 21, -95, -43],
         [19, 37, -93, -84, 66, 11, -90, -90, -60, -21, 75, -66, 38, -19, -69, 79, 8, 1, -18, -5, -76, -81, -99, 22, -48, 32, 48, -78, 18, 26, -37, 98, 26, -40, 69, -77, 1, -87, -74, -86, 89, 23, -55, 14, 90, 17, -54, 27, -64, 54, 14, -9, 24, 33, 7, 6, -14, 33, 23, 7, 71, -55, -92, 14, -18, 37, -64, 78, 19, -42, -37, 7, -45, -18, -62, 66, -45, -42, 95, -45, -20, -24, 6, -70, 54, 29, 87, -38, -16, 29, -65, 49, 47, 6, 76, -67, 2, 68, 76, 46],
         [21, 50, -61, 97, 7, 2, -83, -4, 59, 78, 61, 69, 22, 61, -100, -9, -6, 88, 36, 12, -25, 78, 41, -68, 57, 16, -79, 85, -52, 39, -31, -15, 18, -5, 13, 62, 69, 82, 89, -77, 32, 9, -64, 72, -43, 13, 0, 26, 45, 31, -75, 48, 82, -29, 66, 1, -58, -82, 21, -63, -39, -92, 70, -66, 31, 26, 81, 75, 82, -83, 7, 55, 6, 21, -66, 39, -32, 58, 51, 48, -99, -59, -82, -94, -12, -29, 14, 94, 56, 95, 89, -59, 76, -76, 96, 16, -15, -99, 3, -94],
         [32, -83, 76, 36, 70, -74, 9, -26, -62, 2, 76, -98, -21, 95, -83, -99, 78, 53, 93, 77, -52, -99, -52, 83, -46, -92, -60, -73, -36, -65, -76, -81, -31, -41, 66, -33, -58, 38, 62, 46, -33, -13, -37, -35, 60, -23, -16, -64, -72, 29, -16, -94, -34, 15, -58, -99, 87, -23, 24, 92, -53, 70, 50, 56, -45, 21, 35, -82, -70, -84, -70, -26, 37, -30, -69, 55, -54, 88, 67, 84, -14, -85, -68, 29, -74, 23, -26, 14, 96, 47, 13, 77, 50, -71, -53, -16, 95, -62, -39, 77],
         [19, -86, 76, -35, 19, 80, -27, 16, -31, 42, -5, 79, -61, -82, -22, 54, 28, -94, -11, -64, 67, 64, -7, 96, -55, 60, -74, 80, 67, 10, -35, -83, -27, 78, -16, -11, -49, 30, 62, 8, -98, 0, -38, 80, -55, 96, 63, -34, -46, 2, -97, 6, -8, -78, -67, 24, 83, -22, -11, 76, 29, 0, -65, 71, -50, -52, 93, 47, -75, -100, -93, -61, -52, -50, -70, 15, -74, -91, -46, 27, 5, -55, -5, 98, -80, 44, 66, 82, 77, 90, -43, 28, -8, -8, 10, 20, 11, 97, 22, 73],
         [12, 30, 74, 74, 74, 52, -44, -37, 45, -78, 0, 4, 58, 87, 37, -7, -72, 13, -71, 29, -76, 39, -20, 82, -27, -44, 90, 80, 63, 41, 4, -12, -99, 32, 5, 85, -87, 14, -36, 92, 16, -89, -10, -10, 95, -15, -84, -64, 40, 73, 66, 24, -16, -83, 15, 72, 22, 37, 39, -39, -34, 93, 64, 90, -21, 23, -66, 53, 32, -36, -86, 30, -1, -44, -20, 40, -100, 80, -30, 7, -87, -42, 7, -91, 58, -72, -22, 14, 23, -69, 69, -76, -36, -2, -5, -5, 72, -59, -30, 88],
         [-72, 95, 73, 84, -93, 64, 52, -62, -55, -28, -25, -84, 95, 68, 96, -80, 17, -100, 53, 15, -64, 45, -24, -46, 55, -56, 46, -70, -90, 48, -44, -67, 78, 6, -59, 43, 0, 61, -55, -37, 63, -90, -89, -83, -92, -17, 92, -47, 60, -11, 22, -15, -28, 62, -10, 17, -68, -53, 98, -78, 30, 77, 46, -86, 77, -86, 75, 24, 95, -84, 19, 29, 82, 24, -21, -41, -76, -98, 43, 3, -29, 23, -66, -80, -63, 71, 31, 41, 41, -47, -92, -97, -93, -6, -21, 25, -34, 58, -19, 42],
         [-56, -9, 49, 35, -85, -77, -4, 32, -76, 42, -7, 65, 29, 80, 0, -81, 1, 22, 55, 82, -43, 13, 37, 33, -84, -79, -3, 3, 46, -49, 34, 66, 52, 2, -12, -52, 54, 35, -73, -37, -63, -8, 3, 80, -25, -13, -52, -54, -81, -61, -48, 41, -38, -83, 83, 80, -38, -5, -4, 13, 59, -6, 26, 5, -50, 57, 3, 18, -72, -5, -8, -48, 72, -22, -90, -76, -80, 74, 37, -41, 10, -39, 74, 7, -49, 79, -84, -54, -36, -85, 44, 35, -56, -95, 72, 39, -25, -3, 48, -64],
         [-17, 80, 57, -16, -69, 81, -12, -7, 65, -1, 29, -72, 63, 79, -6, 40, -18, 88, 67, -82, 65, 18, -91, -58, 29, 78, -98, 44, 33, -59, -4, -41, -81, -9, 46, 7, -34, 50, 51, 2, -60, 52, -91, 66, -47, 40, -43, -73, 35, 79, 75, 34, -46, -70, 95, -33, 96, 61, -66, 46, 9, 97, -90, 15, -98, 86, -35, 62, 63, -23, 51, 10, -67, 74, -85, -45, 25, 87, 19, -52, 39, 47, 20, -11, 35, -37, -22, 70, -63, 78, -62, 92, -85, -87, -30, 63, -6, 57, 64, 47],
         [82, 3, -71, 53, 43, -27, -47, 9, -23, -57, -26, -41, -54, 79, 21, -71, 88, -72, 52, 32, -26, -3, -12, 76, 82, 28, 73, -69, -72, 32, -80, 11, 21, -30, -11, -12, 23, 14, -77, 51, -79, 3, -18, -91, -21, -59, 100, 83, 7, -38, -79, -62, -51, 62, -62, -52, 93, -34, -15, -77, -33, -45, 40, -81, 71, 88, -63, -82, 67, -67, -32, -34, -80, -97, 2, -67, 30, -11, -52, -64, 37, 31, -88, -18, 19, 28, 12, -87, 7, -78, -17, 59, 83, -70, -39, -8, -13, 49, -75, -12],
         [39, -25, -18, 11, -86, 15, -86, 42, -4, 15, 69, 45, 72, -25, -8, 19, 68, -24, -9, -37, 9, -34, 26, -13, 49, 93, 21, 0, 20, 21, -52, 70, 29, 26, 79, 98, 14, 62, -84, -43, 34, -31, 66, 90, 92, -17, -37, 81, -70, -61, -76, 93, 50, -91, -18, 88, 76, 19, -91, 4, -60, -57, -72, -97, -54, -95, -84, -51, -20, 47, 69, -11, -84, -34, 15, 68, 52, 31, 13, 51, -56, -8, 8, 7, -34, -92, 6, -88, 17, 77, 68, 50, -56, -17, 35, -59, -93, 2, -26, 35],
         [-48, 90, 80, 18, -61, -22, 24, 62, -49, -72, 42, -43, 46, 45, -36, -35, 59, 6, 66, -24, -51, 37, -48, -92, 12, 98, 12, -45, -32, -77, 5, -56, 60, 93, 20, 91, 48, 36, 90, -49, 81, -65, 56, -46, 83, 39, 50, 91, 43, 72, 93, -33, 5, 25, -58, 58, 85, -79, 95, 96, -37, -56, -49, -9, 42, -41, -31, 20, -56, -86, -33, 77, -64, -77, 40, 43, 73, -31, -2, -1, -65, -36, -21, 9, 58, -79, -64, 22, 22, -24, 62, -52, 60, -47, 99, 69, -59, -78, -3, -3],
         [-77, 50, 4, -53, 19, 78, -6, -97, 18, -83, -67, 60, 7, -65, 86, 12, -30, 41, -62, 18, 73, 91, 11, 62, 60, -4, 52, 28, -33, 46, -67, -84, 52, -16, -86, 86, 91, 81, -85, -8, 94, 50, -39, 63, 53, -98, -80, -60, 3, -62, -47, 82, 16, 98, 68, -89, -94, -36, 18, -49, -25, -35, -17, -24, 81, -17, 6, -53, -45, 88, 73, 22, -1, -82, 78, 99, -83, 70, -74, 89, -89, -62, 93, 33, 3, 69, -89, 13, -78, -60, -19, -2, -27, -69, -81, 100, 21, -11, 14, -7],
         [0, -82, -80, -17, -62, -24, -81, -59, 14, 78, -86, -92, 31, -58, -22, 92, 67, 12, -54, -96, -8, 51, -19, -33, -76, -11, -2, 21, -6, -30, 81, -18, 92, -18, 78, -12, 20, 56, 6, 55, -13, 5, 86, 32, -75, 6, 56, 93, -31, -51, 0, 83, -3, 50, -23, -72, -25, -38, 35, 26, 47, 85, 41, 95, -48, 81, 67, -61, 100, -43, -35, 5, -29, -20, 81, -9, -14, -28, 34, 70, -17, 1, -12, -5, -53, 9, 76, -29, -11, -10, 61, 99, -99, -72, -20, 56, 100, 2, 42, 71],
         [-41, 8, -45, 95, -46, 50, 9, -74, -12, -27, -49, -25, 36, -37, 0, 8, 91, -60, 18, 75, -8, -77, 12, 59, -30, 82, 80, 68, 40, -78, -71, 36, 53, -11, -89, -29, 47, -52, 20, 2, -47, 56, 57, -41, -66, 71, 61, -45, -5, 20, 15, -48, -54, 15, 93, -28, -73, 47, 63, 39, -21, -34, 21, 88, -79, -72, 83, -91, -7, 29, 25, -81, -76, 3, -76, -88, -99, 22, -79, 64, -47, 71, 75, 12, -5, 42, -30, 58, 66, -50, -38, -51, -15, -63, -38, 45, 63, -61, -4, 68],
         [36, -53, -65, -35, 57, 16, 10, 38, 86, 3, -1, 26, -92, 91, 28, 91, 10, -79, 17, -67, 83, 67, -45, 98, -79, -37, -23, -74, -97, 20, -25, 52, 0, -18, 25, -99, -3, 54, 31, 56, 72, 53, 22, 90, -1, 30, 89, -2, 45, 60, 42, -95, 90, -22, 16, 47, 72, -32, 70, -66, -71, -95, 44, -61, 40, 17, -59, 16, -74, -92, 7, -12, 90, -20, 7, -35, 54, -20, 16, -89, -96, 50, 35, -23, 97, -100, -6, -48, 96, 49, 26, -62, 41, 79, -43, -72, 46, -97, -64, -88],
         [-67, 4, -21, -40, -50, 70, -2, 26, 27, 3, 73, 4, -94, -48, 16, -80, 26, 55, 20, -12, 74, 66, -83, -8, -96, 84, -78, 74, 81, 61, 32, 63, -70, 48, 9, -21, -65, 44, -46, -85, 60, 14, 64, -3, 91, 91, -13, -23, 95, 12, 50, 16, 7, -3, -24, 2, -15, 10, 43, 38, 43, -22, -88, 23, -26, 17, 83, -97, -13, -42, 75, -57, 81, 45, -32, 95, -9, -76, -33, -13, -97, -49, 63, -38, 40, -10, 17, 30, 64, -49, -6, -10, 31, -41, -72, -68, 20, -1, 46, -84],
         [-57, 49, 94, 3, 83, -65, 42, -39, -100, 72, 77, -56, 74, -83, 18, 53, 0, 100, 2, 39, -38, 94, 95, 87, 11, -83, -72, -36, 38, -25, 44, 2, 81, -18, 49, -93, -93, -58, 11, -56, 25, 78, 14, 43, 67, 87, 55, 9, -8, -94, -60, 18, 79, 94, 43, 80, 70, 42, 17, -35, 97, -70, 28, -8, 27, -6, -39, 32, -96, -98, -63, -59, 80, -10, -5, -44, -7, -72, -27, -76, 61, 92, -91, -17, -94, 99, 79, 22, -19, -35, 41, -83, 81, 50, -68, -38, 49, -17, -34, 83],
         [62, 54, -40, -16, -78, -63, 91, 83, -84, 92, 2, 19, -5, 99, -49, 23, -93, -74, -56, 42, 97, -31, -74, 79, -57, -80, 25, 40, -82, -53, 71, -72, 32, 87, -15, 83, -76, -39, 33, -62, 53, 52, -23, -61, 43, -70, -81, -66, 61, -37, 75, -97, 75, 21, -21, -17, 55, 51, -82, 38, 20, -58, 100, 31, 9, 60, 5, -15, 28, 1, -73, 50, -60, 9, 45, 94, -20, -91, -71, 10, -26, -88, 58, -76, 22, 20, 46, -8, 52, 20, 69, -16, -38, -13, 87, -22, 94, 55, -56, -79]]

    def __init__(self, num_variables, **kwargs):
        FletcherPowell.__init__(self, num_variables, **kwargs)
        self.alpha = F12.alpha[:num_variables]
        self.a = F12.a
        self.b = F12.b
        self.init_vector_e()


    def objective_function(self, phenome):
        obj_value = FletcherPowell.objective_function(self, phenome)
        obj_value += self.bias
        return obj_value
