"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EncryptionOption = exports.AthenaStartQueryExecution = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * Start an Athena Query as a Task
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-athena.html
 */
class AthenaStartQueryExecution extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_AthenaStartQueryExecutionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, AthenaStartQueryExecution);
            }
            throw error;
        }
        this.integrationPattern = props.integrationPattern ?? sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, AthenaStartQueryExecution.SUPPORTED_INTEGRATION_PATTERNS);
        this.taskPolicies = this.createPolicyStatements();
    }
    createPolicyStatements() {
        const policyStatements = [
            new iam.PolicyStatement({
                resources: [
                    cdk.Stack.of(this).formatArn({
                        service: 'athena',
                        resource: 'datacatalog',
                        resourceName: this.props.queryExecutionContext?.catalogName ?? 'AwsDataCatalog',
                    }),
                    cdk.Stack.of(this).formatArn({
                        service: 'athena',
                        resource: 'workgroup',
                        resourceName: this.props.workGroup ?? 'primary',
                    }),
                ],
                actions: ['athena:getDataCatalog', 'athena:startQueryExecution', 'athena:getQueryExecution'],
            }),
        ];
        policyStatements.push(new iam.PolicyStatement({
            actions: ['s3:CreateBucket',
                's3:ListBucket',
                's3:GetBucketLocation',
                's3:GetObject'],
            resources: ['*'],
        }));
        policyStatements.push(new iam.PolicyStatement({
            actions: ['s3:AbortMultipartUpload',
                's3:ListBucketMultipartUploads',
                's3:ListMultipartUploadParts',
                's3:PutObject'],
            resources: [this.props.resultConfiguration?.outputLocation?.bucketName ? `arn:aws:s3:::${this.props.resultConfiguration?.outputLocation?.bucketName}/${this.props.resultConfiguration?.outputLocation?.objectKey}/*` : '*'],
        }));
        policyStatements.push(new iam.PolicyStatement({
            actions: ['lakeformation:GetDataAccess'],
            resources: ['*'],
        }));
        policyStatements.push(new iam.PolicyStatement({
            actions: ['glue:BatchCreatePartition',
                'glue:BatchDeletePartition',
                'glue:BatchDeleteTable',
                'glue:BatchGetPartition',
                'glue:CreateDatabase',
                'glue:CreatePartition',
                'glue:CreateTable',
                'glue:DeleteDatabase',
                'glue:DeletePartition',
                'glue:DeleteTable',
                'glue:GetDatabase',
                'glue:GetDatabases',
                'glue:GetPartition',
                'glue:GetPartitions',
                'glue:GetTable',
                'glue:GetTables',
                'glue:UpdateDatabase',
                'glue:UpdatePartition',
                'glue:UpdateTable'],
            resources: [
                cdk.Stack.of(this).formatArn({
                    service: 'glue',
                    resource: 'catalog',
                }),
                cdk.Stack.of(this).formatArn({
                    service: 'glue',
                    resource: 'database',
                    resourceName: this.props.queryExecutionContext?.databaseName ?? 'default',
                }),
                cdk.Stack.of(this).formatArn({
                    service: 'glue',
                    resource: 'table',
                    resourceName: (this.props.queryExecutionContext?.databaseName ?? 'default') + '/*',
                }),
                cdk.Stack.of(this).formatArn({
                    service: 'glue',
                    resource: 'userDefinedFunction',
                    resourceName: (this.props.queryExecutionContext?.databaseName ?? 'default') + '/*',
                }),
            ],
        }));
        return policyStatements;
    }
    renderEncryption() {
        const encryptionConfiguration = this.props.resultConfiguration?.encryptionConfiguration !== undefined
            ? {
                EncryptionOption: this.props.resultConfiguration.encryptionConfiguration.encryptionOption,
                KmsKey: this.props.resultConfiguration.encryptionConfiguration.encryptionKey,
            }
            : undefined;
        return encryptionConfiguration;
    }
    /**
     * Provides the Athena start query execution service integration task configuration
     */
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('athena', 'startQueryExecution', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                QueryString: this.props.queryString,
                ClientRequestToken: this.props.clientRequestToken,
                QueryExecutionContext: (this.props.queryExecutionContext?.catalogName || this.props.queryExecutionContext?.databaseName) ? {
                    Catalog: this.props.queryExecutionContext?.catalogName,
                    Database: this.props.queryExecutionContext?.databaseName,
                } : undefined,
                ResultConfiguration: {
                    EncryptionConfiguration: this.renderEncryption(),
                    OutputLocation: this.props.resultConfiguration?.outputLocation ? `s3://${this.props.resultConfiguration.outputLocation.bucketName}/${this.props.resultConfiguration.outputLocation.objectKey}/` : undefined,
                },
                WorkGroup: this.props?.workGroup,
            }),
        };
    }
}
exports.AthenaStartQueryExecution = AthenaStartQueryExecution;
_a = JSII_RTTI_SYMBOL_1;
AthenaStartQueryExecution[_a] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.AthenaStartQueryExecution", version: "1.169.0" };
AthenaStartQueryExecution.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
/**
 * Encryption Options of the S3 bucket
 *
 * @see https://docs.aws.amazon.com/athena/latest/APIReference/API_EncryptionConfiguration.html#athena-Type-EncryptionConfiguration-EncryptionOption
 */
var EncryptionOption;
(function (EncryptionOption) {
    /**
     * Server side encryption (SSE) with an Amazon S3-managed key.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingServerSideEncryption.html
     */
    EncryptionOption["S3_MANAGED"] = "SSE_S3";
    /**
     * Server-side encryption (SSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html
     */
    EncryptionOption["KMS"] = "SSE_KMS";
    /**
     * Client-side encryption (CSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingClientSideEncryption.html
     */
    EncryptionOption["CLIENT_SIDE_KMS"] = "CSE_KMS";
})(EncryptionOption = exports.EncryptionOption || (exports.EncryptionOption = {}));
//# sourceMappingURL=data:application/json;base64,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