from __future__ import annotations

import os
import re
import shutil
import subprocess
import sys
from pathlib import Path

import pytest
import tomli
import tomli_w
from _pytest.capture import CaptureFixture
from tests import EXPECTED_TEST_DATA
from tests import INPUT_TEST_DATA
from tests import REPO_ROOT

from qt_dev_helper.config import Config
from qt_dev_helper.config import RccKwargs
from qt_dev_helper.config import UicKwargs
from qt_dev_helper.transpiler import build_all_assets
from qt_dev_helper.transpiler import build_resources
from qt_dev_helper.transpiler import build_uis
from qt_dev_helper.transpiler import compile_resource_file
from qt_dev_helper.transpiler import compile_ui_file
from qt_dev_helper.transpiler import transpile_sass

COMMENT_BLANK_LINE_PATTERN = re.compile(r"(\n\s*|\s*[/]{2}.+?)\n")
HEADER_PATTERN = re.compile(
    r"(/[*]{10,}.+[*]{10,}/|# -[*]- coding.+?WARNING! All changes.+?#{10,}\n)", re.DOTALL
)


def clean_text(text: str) -> str:
    """Helper function to clean autogenerated files before comparing."""
    text = re.sub(HEADER_PATTERN, "\n", text)
    return re.sub(COMMENT_BLANK_LINE_PATTERN, "\n", text)


def generated_files_equal(result: Path, expected: Path):
    """Compare generated files."""
    assert clean_text(result.read_text()) == clean_text(
        expected.read_text()
    ), f"{result=}\n{expected=}"


def test_transpile_sass(tmp_path: Path, dummy_config: Config):
    """Transpiling scss gives the expected result."""

    expected = (EXPECTED_TEST_DATA / "theme.qss").read_text()

    out_file = transpile_sass(INPUT_TEST_DATA / "styles/theme.scss", tmp_path / "theme.qss")

    assert out_file.read_text(encoding="utf8") == expected

    out_file = transpile_sass(*dummy_config.root_style_paths())

    assert out_file.read_text(encoding="utf8") == expected


def test_tranpile_ui_file(dummy_config: Config):
    """Create python or cpp header from ui file."""
    tmp_path = dummy_config.base_path

    ui_file = tmp_path / "assets/ui_files/minimal.ui"
    result1 = compile_ui_file(ui_file, tmp_path / "default.py")

    generated_files_equal(result1, EXPECTED_TEST_DATA / "Ui_minimal.py")

    result2 = compile_ui_file(ui_file, tmp_path / "default_no_from.py", form_import=False)

    generated_files_equal(result2, (EXPECTED_TEST_DATA / "Ui_minimal_no_from_imports.py"))

    result3 = compile_ui_file(ui_file, tmp_path / "Ui_minimal.h", generator="cpp")

    generated_files_equal(result3, EXPECTED_TEST_DATA / "Ui_minimal.h")


def test_tranpile_resource_file(dummy_config: Config):
    """Create python or cpp header from ui file."""
    tmp_path = dummy_config.base_path
    qrc_file = tmp_path / "assets/test_resource.qrc"
    result1 = compile_resource_file(qrc_file, tmp_path / "resource.py")

    assert "qt_resource_data" in result1.read_text()
    assert "qt_resource_name" in result1.read_text()
    assert "qt_resource_struct" in result1.read_text()

    result2 = compile_resource_file(qrc_file, tmp_path / "resource.h", generator="cpp")

    assert "static const unsigned char qt_resource_data[]" in result2.read_text()
    assert "static const unsigned char qt_resource_name[]" in result2.read_text()
    assert "static const unsigned char qt_resource_struct[]" in result2.read_text()


@pytest.mark.parametrize(
    "uic_kwargs, flatten_path, expected_rel_out_path, expected_file",
    (
        (
            {"generator": "python", "form_import": True},
            True,
            "Ui_minimal.py",
            "Ui_minimal.py",
        ),
        (
            None,
            True,
            "Ui_minimal.py",
            "Ui_minimal.py",
        ),
        (
            {"generator": "python", "form_import": True},
            False,
            "ui_files/Ui_minimal.py",
            "Ui_minimal.py",
        ),
        (
            {"generator": "cpp", "form_import": True},
            False,
            "ui_files/Ui_minimal.h",
            "Ui_minimal.h",
        ),
    ),
)
def test_build_uis(
    dummy_config: Config,
    capsys: CaptureFixture,
    uic_kwargs: UicKwargs | None,
    flatten_path: bool,
    expected_rel_out_path: str,
    expected_file: str,
):
    """Build ui files for whole folder."""
    base_path = dummy_config.base_path

    ui_files_folder, generated_ui_code_folder = dummy_config.ui_folder_paths()

    result_paths_flat = build_uis(
        ui_files_folder.parent,
        generated_ui_code_folder,
        flatten_path=flatten_path,
        uic_kwargs=uic_kwargs,
    )

    assert len(result_paths_flat) == 1
    assert base_path / f"outputs/ui_files/{expected_rel_out_path}" in result_paths_flat
    generated_files_equal(result_paths_flat[0], EXPECTED_TEST_DATA / expected_file)

    stdout, stderr = capsys.readouterr()

    assert stderr == ""
    assert stdout == f"Creating: {expected_rel_out_path}\n\n"


@pytest.mark.parametrize(
    "rcc_kwargs, flatten_path, expected_rel_out_path",
    (
        ({"generator": "python"}, True, "test_resource_rc.py"),
        (None, True, "test_resource_rc.py"),
        ({"generator": "python"}, False, "assets/test_resource_rc.py"),
        ({"generator": "cpp"}, False, "assets/test_resource_rc.h"),
    ),
)
def test_build_resources(
    dummy_config: Config,
    capsys: CaptureFixture,
    rcc_kwargs: RccKwargs | None,
    flatten_path: bool,
    expected_rel_out_path: str,
):
    """Build resource files for whole folder."""
    base_path = dummy_config.base_path

    resource_folder, generated_rc_code_folder = dummy_config.rc_folder_paths()

    result_paths_flat = build_resources(
        resource_folder.parent,
        generated_rc_code_folder,
        flatten_path=flatten_path,
        rcc_kwargs=rcc_kwargs,
    )

    assert len(result_paths_flat) == 1
    assert base_path / f"outputs/ui_files/{expected_rel_out_path}" in result_paths_flat

    stdout, stderr = capsys.readouterr()

    assert stderr == ""
    assert stdout == f"Creating: {expected_rel_out_path}\n\n"


@pytest.mark.parametrize("use_path", (True, False))
def test_build_all_assets_(dummy_config: Config, capsys: CaptureFixture, use_path: bool):
    """Files are build when definitions are in the config."""
    dummy_config.uic_args = []
    dummy_config.rcc_args = []

    result = build_all_assets(dummy_config.base_path if use_path is True else dummy_config)

    assert len(result) == 3

    stdout, stderr = capsys.readouterr()

    assert stderr == ""
    assert stdout == "\n".join(
        (
            "Creating: outputs/theme.qss",
            "Creating: Ui_minimal.py\n",
            "Creating: test_resource_rc.py\n\n",
        )
    )


def test_build_all_assets_no_config(tmp_path: Path, capsys: CaptureFixture):
    """No error if parts of the config are missing."""
    empty_config = Config(base_path=tmp_path)
    result = build_all_assets(empty_config)

    assert len(result) == 0

    stdout, stderr = capsys.readouterr()

    assert stderr == ""
    assert stdout == "\n".join(
        (
            "No style files to compile fund in config!",
            "No ui folders fund in config!",
            "No resource folders fund in config!\n",
        )
    )


@pytest.mark.skipif(
    "CI" not in os.environ,
    reason="This test takes very long and problems should be cover by different tests as well.",
)
def test_build_all_assets_as_packaging_buildsystem(dummy_config: Config):
    """All resources are in the wheel."""

    pyproject_toml_path = dummy_config.base_path / "pyproject.toml"

    qt_dev_helper_requirement = f"qt-dev-helper@{REPO_ROOT.resolve().as_uri()}"

    toml_config = tomli.loads(pyproject_toml_path.read_text())
    toml_config["build-system"]["requires"][-1] = qt_dev_helper_requirement
    toml_config["tool"]["qt-dev-helper"]["uic_args"] = []
    toml_config["tool"]["qt-dev-helper"]["rcc_args"] = []

    pyproject_toml_path.write_text(tomli_w.dumps(toml_config))

    subprocess.run(
        " ".join([sys.executable, "-m", "pip", "wheel", ".", "--no-deps"]),
        cwd=dummy_config.base_path,
        check=True,
        shell=True,
    )

    wheel_path = dummy_config.base_path / "outputs-0.0.1-py3-none-any.whl"
    dist_path = dummy_config.base_path / "dist"

    assert wheel_path.is_file() is True

    shutil.unpack_archive(wheel_path, dist_path, format="zip")

    qss_file = dist_path / "outputs/theme.qss"
    assert qss_file.is_file()
    assert qss_file.read_text() == (EXPECTED_TEST_DATA / "theme.qss").read_text()

    ui_file = dist_path / "outputs/ui_files/Ui_minimal.py"
    assert ui_file.is_file()
    generated_files_equal(ui_file, EXPECTED_TEST_DATA / "Ui_minimal.py")

    rc_file = dist_path / "outputs/ui_files/test_resource_rc.py"
    assert rc_file.is_file()
    assert "qt_resource_data" in rc_file.read_text()
    assert "qt_resource_name" in rc_file.read_text()
    assert "qt_resource_struct" in rc_file.read_text()
