from cProfile import Profile
import pstats


def profiling(sort_args=['cumulative'], print_args=[20], directory=None, name_global_directory=None):
    """
    Decorator profiling. Show in screen profile and generate dump stats
    Use package snakeviz for read file profile.prof generated by dump stats
    Snakeviz:  https://jiffyclub.github.io/snakeviz/

    command in terminal: snakeviz profile.prof

    # Example with profiling arguments:

    @profiling(sort_args=['cumulative', 'name'], print_args=[.5, 'init'],
                                name_global_directory='loggerProfilingDirectory', directory='loggerProfilingDirectory')
    def my_function(my_args):
        pass

    Only print screen:
    @profiling()
    def my_function(my_args):
        pass

    Save directory execution file
    @profiling(directory='profile.prof')
    def my_function():
        pass

    Save file using directory in global variable
    @profiling(name_global_directory='loggerProfilingDirectory')
    def my_function():
        pass

    Args:
         sort_args: sorted all the entries according to the standard module/line/name string that is printed.
         print_args: level of detail of method printed out all the statistics
         directory: Directory using for save file profile.prof
         name_global_directory: name global variable create in another context, for save file profile

    return: None

    """
    profiler = Profile()

    def decorator(fn):
        def inner(*args, **kwargs):

            try:
                result = profiler.runcall(fn, *args, **kwargs)
            finally:
                stats = pstats.Stats(profiler)
                stats.strip_dirs().sort_stats(*sort_args).print_stats(*print_args)

                if directory is not None:
                    stats.strip_dirs().sort_stats(*sort_args).dump_stats(directory)

                if name_global_directory is not None:
                    g = fn.__globals__  # use f.func_globals for py < 2.6
                    stats.strip_dirs().sort_stats(*sort_args).dump_stats(g[name_global_directory])

            return result

        return inner

    return decorator
