"""Prometheus metrics for pipeline telemetry."""

from __future__ import annotations

import time
from typing import Dict, Optional

from prometheus_client import Counter, Gauge, Histogram, Info, CollectorRegistry


class PipelineMetrics:
    """Prometheus metrics for pipeline monitoring."""

    def __init__(self, registry: Optional[CollectorRegistry] = None) -> None:
        """Initialize metrics with optional registry."""
        self.registry = registry or CollectorRegistry()

        # Pipeline info
        self.pipeline_info = Info(
            "pipeline_info", "Pipeline information", registry=self.registry
        )

        # Source metrics
        self.source_ticks_total = Counter(
            "source_ticks_total",
            "Total ticks received from source",
            ["tenant_id", "pipeline_id", "source"],
            registry=self.registry,
        )

        self.source_status = Gauge(
            "source_status",
            "Source connection status (1=connected, 0=disconnected)",
            ["tenant_id", "pipeline_id", "source"],
            registry=self.registry,
        )

        # Operator metrics
        self.operator_ticks_processed = Counter(
            "operator_ticks_processed_total",
            "Total ticks processed by operator",
            ["tenant_id", "pipeline_id", "operator"],
            registry=self.registry,
        )

        self.operator_bars_generated = Counter(
            "operator_bars_generated_total",
            "Total bars generated by operator",
            ["tenant_id", "pipeline_id", "operator"],
            registry=self.registry,
        )

        self.operator_processing_duration = Histogram(
            "operator_processing_duration_seconds",
            "Time spent processing ticks in operator",
            ["tenant_id", "pipeline_id", "operator"],
            buckets=[0.001, 0.005, 0.01, 0.025, 0.05, 0.1, 0.25, 0.5, 1.0, 2.5, 5.0],
            registry=self.registry,
        )

        # Batcher metrics
        self.batcher_items_added = Counter(
            "batcher_items_added_total",
            "Total items added to batcher",
            ["tenant_id", "pipeline_id", "batcher"],
            registry=self.registry,
        )

        self.batcher_batches_flushed = Counter(
            "batcher_batches_flushed_total",
            "Total batches flushed from batcher",
            ["tenant_id", "pipeline_id", "batcher"],
            registry=self.registry,
        )

        self.batcher_queue_depth = Gauge(
            "batcher_queue_depth",
            "Current queue depth in batcher",
            ["tenant_id", "pipeline_id", "batcher"],
            registry=self.registry,
        )

        self.batcher_items_dropped = Counter(
            "batcher_items_dropped_total",
            "Total items dropped due to queue limits",
            ["tenant_id", "pipeline_id", "batcher", "reason"],
            registry=self.registry,
        )

        # Sink metrics
        self.sink_batches_written = Counter(
            "sink_batches_written_total",
            "Total batches written to sink",
            ["tenant_id", "pipeline_id", "sink"],
            registry=self.registry,
        )

        self.sink_items_written = Counter(
            "sink_items_written_total",
            "Total items written to sink",
            ["tenant_id", "pipeline_id", "sink"],
            registry=self.registry,
        )

        self.sink_write_duration = Histogram(
            "sink_write_duration_seconds",
            "Time spent writing batches to sink",
            ["tenant_id", "pipeline_id", "sink"],
            buckets=[0.001, 0.005, 0.01, 0.025, 0.05, 0.1, 0.25, 0.5, 1.0, 2.5, 5.0],
            registry=self.registry,
        )

        self.sink_write_errors = Counter(
            "sink_write_errors_total",
            "Total sink write errors",
            ["tenant_id", "pipeline_id", "sink", "error_type"],
            registry=self.registry,
        )

        # Pacing metrics
        self.pacing_tokens_consumed = Counter(
            "pacing_tokens_consumed_total",
            "Total tokens consumed by pacing",
            ["tenant_id", "pipeline_id"],
            registry=self.registry,
        )

        self.pacing_blocked_total = Counter(
            "pacing_blocked_total",
            "Total time spent blocked by pacing",
            ["tenant_id", "pipeline_id"],
            registry=self.registry,
        )

        # Pipeline health
        self.pipeline_health = Gauge(
            "pipeline_health",
            "Pipeline health status (1=healthy, 0=unhealthy)",
            ["tenant_id", "pipeline_id"],
            registry=self.registry,
        )

        self.pipeline_uptime = Gauge(
            "pipeline_uptime_seconds",
            "Pipeline uptime in seconds",
            ["tenant_id", "pipeline_id"],
            registry=self.registry,
        )

    def set_pipeline_info(
        self, tenant_id: str, pipeline_id: str, **labels: str
    ) -> None:
        """Set pipeline information labels."""
        self.pipeline_info.info(
            {"tenant_id": tenant_id, "pipeline_id": pipeline_id, **labels}
        )

    def record_source_tick(self, tenant_id: str, pipeline_id: str, source: str) -> None:
        """Record a tick received from source."""
        self.source_ticks_total.labels(
            tenant_id=tenant_id, pipeline_id=pipeline_id, source=source
        ).inc()

    def set_source_status(
        self, tenant_id: str, pipeline_id: str, source: str, connected: bool
    ) -> None:
        """Set source connection status."""
        self.source_status.labels(
            tenant_id=tenant_id, pipeline_id=pipeline_id, source=source
        ).set(1 if connected else 0)

    def record_operator_processing(
        self,
        tenant_id: str,
        pipeline_id: str,
        operator: str,
        duration: float,
        bars_generated: int = 0,
    ) -> None:
        """Record operator processing metrics."""
        self.operator_ticks_processed.labels(
            tenant_id=tenant_id, pipeline_id=pipeline_id, operator=operator
        ).inc()

        if bars_generated > 0:
            self.operator_bars_generated.labels(
                tenant_id=tenant_id, pipeline_id=pipeline_id, operator=operator
            ).inc(bars_generated)

        self.operator_processing_duration.labels(
            tenant_id=tenant_id, pipeline_id=pipeline_id, operator=operator
        ).observe(duration)

    def record_batcher_activity(
        self,
        tenant_id: str,
        pipeline_id: str,
        batcher: str,
        items_added: int = 0,
        batches_flushed: int = 0,
        items_dropped: int = 0,
        drop_reason: str = "queue_full",
        queue_depth: int = 0,
    ) -> None:
        """Record batcher activity metrics."""
        if items_added > 0:
            self.batcher_items_added.labels(
                tenant_id=tenant_id, pipeline_id=pipeline_id, batcher=batcher
            ).inc(items_added)

        if batches_flushed > 0:
            self.batcher_batches_flushed.labels(
                tenant_id=tenant_id, pipeline_id=pipeline_id, batcher=batcher
            ).inc(batches_flushed)

        if items_dropped > 0:
            self.batcher_items_dropped.labels(
                tenant_id=tenant_id,
                pipeline_id=pipeline_id,
                batcher=batcher,
                reason=drop_reason,
            ).inc(items_dropped)

        self.batcher_queue_depth.labels(
            tenant_id=tenant_id, pipeline_id=pipeline_id, batcher=batcher
        ).set(queue_depth)

    def record_sink_write(
        self,
        tenant_id: str,
        pipeline_id: str,
        sink: str,
        batches_written: int,
        items_written: int,
        duration: float,
        errors: int = 0,
        error_type: str = "unknown",
    ) -> None:
        """Record sink write metrics."""
        if batches_written > 0:
            self.sink_batches_written.labels(
                tenant_id=tenant_id, pipeline_id=pipeline_id, sink=sink
            ).inc(batches_written)

        if items_written > 0:
            self.sink_items_written.labels(
                tenant_id=tenant_id, pipeline_id=pipeline_id, sink=sink
            ).inc(items_written)

        self.sink_write_duration.labels(
            tenant_id=tenant_id, pipeline_id=pipeline_id, sink=sink
        ).observe(duration)

        if errors > 0:
            self.sink_write_errors.labels(
                tenant_id=tenant_id,
                pipeline_id=pipeline_id,
                sink=sink,
                error_type=error_type,
            ).inc(errors)

    def record_pacing_activity(
        self,
        tenant_id: str,
        pipeline_id: str,
        tokens_consumed: int,
        blocked_time: float,
    ) -> None:
        """Record pacing activity metrics."""
        if tokens_consumed > 0:
            self.pacing_tokens_consumed.labels(
                tenant_id=tenant_id, pipeline_id=pipeline_id
            ).inc(tokens_consumed)

        if blocked_time > 0:
            self.pacing_blocked_total.labels(
                tenant_id=tenant_id, pipeline_id=pipeline_id
            ).inc(blocked_time)

    def set_pipeline_health(
        self, tenant_id: str, pipeline_id: str, healthy: bool
    ) -> None:
        """Set pipeline health status."""
        self.pipeline_health.labels(tenant_id=tenant_id, pipeline_id=pipeline_id).set(
            1 if healthy else 0
        )

    def set_pipeline_uptime(
        self, tenant_id: str, pipeline_id: str, uptime: float
    ) -> None:
        """Set pipeline uptime."""
        self.pipeline_uptime.labels(tenant_id=tenant_id, pipeline_id=pipeline_id).set(
            uptime
        )


# ── Phase 6.0B: KEDA Autoscaling Metrics ──────────────────────────────────────

# Graceful degradation if prometheus_client is not installed
try:
    from prometheus_client import Gauge as _Gauge  # type: ignore

    PIPELINE_RATE_SCALE_FACTOR = _Gauge(
        "pipeline_rate_scale_factor",
        "Current rate scale factor applied to provider (0.0..1.0).",
        ["provider"],
    )

    PIPELINE_BACKPRESSURE_STATE = _Gauge(
        "pipeline_backpressure_state",
        "Backpressure state: 0=ok, 1=soft, 2=hard.",
        ["provider"],
    )

    PIPELINE_FEEDBACK_QUEUE_DEPTH = _Gauge(
        "pipeline_feedback_queue_depth",
        "Queue depth reported by feedback source (echo of store).",
        ["source"],  # e.g. "store_coordinator"
    )

except Exception:  # pragma: no cover
    # prometheus_client not installed; provide no-op metrics
    class _NoopMetric:
        def labels(self, **_kw):  # type: ignore
            return self

        def set(self, *_a, **_kw):  # type: ignore # noqa: ARG002
            return None

        def inc(self, *_a, **_kw):  # type: ignore # noqa: ARG002
            return None

        def observe(self, *_a, **_kw):  # type: ignore # noqa: ARG002
            return None

    PIPELINE_RATE_SCALE_FACTOR = _NoopMetric()  # type: ignore
    PIPELINE_BACKPRESSURE_STATE = _NoopMetric()  # type: ignore
    PIPELINE_FEEDBACK_QUEUE_DEPTH = _NoopMetric()  # type: ignore


# ── Phase 10.1: Pulse Event Bus Metrics ───────────────────────────────────────

try:
    from prometheus_client import Counter as _Counter, Gauge as _PulseGauge  # type: ignore

    PULSE_CONSUME_TOTAL = _Counter(
        "pulse_consume_total",
        "Total Pulse events consumed",
        ["stream", "track", "outcome"],  # outcome: success|error|duplicate
    )

    PULSE_LAG_MS = _PulseGauge(
        "pulse_lag_ms",
        "Pulse consumer lag (ms) from event timestamp",
        ["stream"],
    )

except Exception:  # pragma: no cover
    # prometheus_client not installed; provide no-op metrics
    PULSE_CONSUME_TOTAL = _NoopMetric()  # type: ignore
    PULSE_LAG_MS = _NoopMetric()  # type: ignore


# ── Phase 11.0B: Schema Registry Metrics ──────────────────────────────────────

try:
    from prometheus_client import Counter as _RegCounter, Gauge as _RegGauge  # type: ignore

    SCHEMA_VALIDATION_TOTAL = _RegCounter(
        "schema_validation_total",
        "Total schema validations performed",
        ["schema", "outcome"],  # outcome: success|failure|error
    )

    SCHEMA_CACHE_HITS = _RegCounter(
        "schema_cache_hits_total",
        "Total schema cache hits",
        ["schema"],
    )

    SCHEMA_CACHE_MISSES = _RegCounter(
        "schema_cache_misses_total",
        "Total schema cache misses (fetches from registry)",
        ["schema"],
    )

    SCHEMA_REGISTRY_ERRORS = _RegCounter(
        "schema_registry_errors_total",
        "Total registry fetch/validation errors",
        ["schema", "error_type"],  # error_type: not_found|timeout|network
    )

    SCHEMA_CACHE_SIZE = _RegGauge(
        "schema_cache_size",
        "Current number of schemas in cache",
    )

except Exception:  # pragma: no cover
    # prometheus_client not installed; provide no-op metrics
    SCHEMA_VALIDATION_TOTAL = _NoopMetric()  # type: ignore
    SCHEMA_CACHE_HITS = _NoopMetric()  # type: ignore
    SCHEMA_CACHE_MISSES = _NoopMetric()  # type: ignore
    SCHEMA_REGISTRY_ERRORS = _NoopMetric()  # type: ignore
    SCHEMA_CACHE_SIZE = _NoopMetric()  # type: ignore


# ── Phase 11.1: Enforcement Mode Metrics ──────────────────────────────────────

try:
    from prometheus_client import Counter as _EnfCounter  # type: ignore

    SCHEMA_VALIDATION_FAILURES = _EnfCounter(
        "schema_validation_failures_total",
        "Total schema validation failures by enforcement mode",
        ["schema", "mode"],  # mode: warn|strict
    )

    SCHEMA_ENFORCEMENT_ACTIONS = _EnfCounter(
        "schema_enforcement_actions_total",
        "Total enforcement actions taken",
        ["schema", "severity", "action"],  # severity: warn|error, action: logged|rejected
    )

except Exception:  # pragma: no cover
    # prometheus_client not installed; provide no-op metrics
    SCHEMA_VALIDATION_FAILURES = _NoopMetric()  # type: ignore
    SCHEMA_ENFORCEMENT_ACTIONS = _NoopMetric()  # type: ignore