"""
This file provides a wrapper for the model object and functions.
Users will not need to know the names of the specific objects they need to create.
"""

import xarray as xr
from echopype.model.azfp import ModelAZFP
from echopype.model.ek60 import ModelEK60


def EchoData(nc_path):
    """
    Provides data analysis and computation tools for sonar data in netCDF form.

    Parameters
    ----------
    nc_path : str
        The path to a .nc file generated by `echopype`

    Returns
    -------
        Returns a specialized EchoData object depending on
        the type of echosounder the .nc file was produced with
    """

    # Open nc file in order to determine what echosounder produced the original dataset
    with xr.open_dataset(nc_path) as nc_file:
        try:
            echo_type = nc_file.keywords
        except AttributeError:
            raise ValueError("This file is incompatible with echopype functions.")

    # Returns specific EchoData object
    if echo_type == "EK60":
        return ModelEK60(nc_path)
    elif echo_type == "AZFP":
        return ModelAZFP(nc_path)
    else:
        raise ValueError("Unsupported file type")
