"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RepositoryEncryption = exports.TagMutability = exports.Repository = exports.RepositoryBase = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const os_1 = require("os");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const ecr_generated_1 = require("./ecr.generated");
const lifecycle_1 = require("./lifecycle");
/**
 * Base class for ECR repository. Reused between imported repositories and owned repositories.
 */
class RepositoryBase extends core_1.Resource {
    /**
     * The URI of this repository (represents the latest image):
     *
     *    ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY
     *
     */
    get repositoryUri() {
        return this.repositoryUriForTag();
    }
    /**
     * Returns the URL of the repository. Can be used in `docker push/pull`.
     *
     *    ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY[:TAG]
     *
     * @param tag Optional image tag
     */
    repositoryUriForTag(tag) {
        const tagSuffix = tag ? `:${tag}` : '';
        return this.repositoryUriWithSuffix(tagSuffix);
    }
    /**
     * Returns the URL of the repository. Can be used in `docker push/pull`.
     *
     *    ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY[@DIGEST]
     *
     * @param digest Optional image digest
     */
    repositoryUriForDigest(digest) {
        const digestSuffix = digest ? `@${digest}` : '';
        return this.repositoryUriWithSuffix(digestSuffix);
    }
    /**
     * Returns the repository URI, with an appended suffix, if provided.
     * @param suffix An image tag or an image digest.
     * @private
     */
    repositoryUriWithSuffix(suffix) {
        const parts = this.stack.splitArn(this.repositoryArn, core_1.ArnFormat.SLASH_RESOURCE_NAME);
        return `${parts.account}.dkr.ecr.${parts.region}.${this.stack.urlSuffix}/${this.repositoryName}${suffix}`;
    }
    /**
     * Define a CloudWatch event that triggers when something happens to this repository
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.ecr'],
            detailType: ['AWS API Call via CloudTrail'],
            detail: {
                requestParameters: {
                    repositoryName: [this.repositoryName],
                },
            },
        });
        return rule;
    }
    /**
     * Defines an AWS CloudWatch event rule that can trigger a target when an image is pushed to this
     * repository.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailImagePushed(id, options = {}) {
        jsiiDeprecationWarnings._aws_cdk_aws_ecr_OnCloudTrailImagePushedOptions(options);
        const rule = this.onCloudTrailEvent(id, options);
        rule.addEventPattern({
            detail: {
                eventName: ['PutImage'],
                requestParameters: {
                    imageTag: options.imageTag ? [options.imageTag] : undefined,
                },
            },
        });
        return rule;
    }
    /**
     * Defines an AWS CloudWatch event rule that can trigger a target when an image scan is completed
     *
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onImageScanCompleted(id, options = {}) {
        var _d;
        jsiiDeprecationWarnings._aws_cdk_aws_ecr_OnImageScanCompletedOptions(options);
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.ecr'],
            detailType: ['ECR Image Scan'],
            detail: {
                'repository-name': [this.repositoryName],
                'scan-status': ['COMPLETE'],
                'image-tags': (_d = options.imageTags) !== null && _d !== void 0 ? _d : undefined,
            },
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers for repository events. Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.ecr'],
            resources: [this.repositoryArn],
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * Grant the given principal identity permissions to perform the actions on this repository
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipalOrResource({
            grantee,
            actions,
            resourceArns: [this.repositoryArn],
            resourceSelfArns: [],
            resource: this,
        });
    }
    /**
     * Grant the given identity permissions to use the images in this repository
     */
    grantPull(grantee) {
        const ret = this.grant(grantee, 'ecr:BatchCheckLayerAvailability', 'ecr:GetDownloadUrlForLayer', 'ecr:BatchGetImage');
        iam.Grant.addToPrincipal({
            grantee,
            actions: ['ecr:GetAuthorizationToken'],
            resourceArns: ['*'],
            scope: this,
        });
        return ret;
    }
    /**
     * Grant the given identity permissions to pull and push images to this repository.
     */
    grantPullPush(grantee) {
        this.grantPull(grantee);
        return this.grant(grantee, 'ecr:PutImage', 'ecr:InitiateLayerUpload', 'ecr:UploadLayerPart', 'ecr:CompleteLayerUpload');
    }
}
exports.RepositoryBase = RepositoryBase;
_a = JSII_RTTI_SYMBOL_1;
RepositoryBase[_a] = { fqn: "@aws-cdk/aws-ecr.RepositoryBase", version: "1.145.0" };
/**
 * Define an ECR repository
 */
class Repository extends RepositoryBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.repositoryName,
        });
        this.lifecycleRules = new Array();
        jsiiDeprecationWarnings._aws_cdk_aws_ecr_RepositoryProps(props);
        Repository.validateRepositoryName(this.physicalName);
        const resource = new ecr_generated_1.CfnRepository(this, 'Resource', {
            repositoryName: this.physicalName,
            // It says "Text", but they actually mean "Object".
            repositoryPolicyText: core_1.Lazy.any({ produce: () => this.policyDocument }),
            lifecyclePolicy: core_1.Lazy.any({ produce: () => this.renderLifecyclePolicy() }),
            imageScanningConfiguration: !props.imageScanOnPush ? undefined : {
                scanOnPush: true,
            },
            imageTagMutability: props.imageTagMutability || undefined,
            encryptionConfiguration: this.parseEncryption(props),
        });
        resource.applyRemovalPolicy(props.removalPolicy);
        this.registryId = props.lifecycleRegistryId;
        if (props.lifecycleRules) {
            props.lifecycleRules.forEach(this.addLifecycleRule.bind(this));
        }
        this.repositoryName = this.getResourceNameAttribute(resource.ref);
        this.repositoryArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'ecr',
            resource: 'repository',
            resourceName: this.physicalName,
        });
    }
    /**
     * Import a repository
     */
    static fromRepositoryAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_ecr_RepositoryAttributes(attrs);
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryName = attrs.repositoryName;
                this.repositoryArn = attrs.repositoryArn;
            }
            addToResourcePolicy(_statement) {
                // dropped
                return { statementAdded: false };
            }
        }
        return new Import(scope, id);
    }
    static fromRepositoryArn(scope, id, repositoryArn) {
        // if repositoryArn is a token, the repository name is also required. this is because
        // repository names can include "/" (e.g. foo/bar/myrepo) and it is impossible to
        // parse the name from an ARN using CloudFormation's split/select.
        if (core_1.Token.isUnresolved(repositoryArn)) {
            throw new Error('"repositoryArn" is a late-bound value, and therefore "repositoryName" is required. Use `fromRepositoryAttributes` instead');
        }
        const repositoryName = repositoryArn.split('/').slice(1).join('/');
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryName = repositoryName;
                this.repositoryArn = repositoryArn;
            }
            addToResourcePolicy(_statement) {
                // dropped
                return { statementAdded: false };
            }
        }
        return new Import(scope, id, {
            environmentFromArn: repositoryArn,
        });
    }
    static fromRepositoryName(scope, id, repositoryName) {
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryName = repositoryName;
                this.repositoryArn = Repository.arnForLocalRepository(repositoryName, scope);
            }
            addToResourcePolicy(_statement) {
                // dropped
                return { statementAdded: false };
            }
        }
        return new Import(scope, id);
    }
    /**
     * Returns an ECR ARN for a repository that resides in the same account/region
     * as the current stack.
     */
    static arnForLocalRepository(repositoryName, scope, account) {
        return core_1.Stack.of(scope).formatArn({
            account,
            service: 'ecr',
            resource: 'repository',
            resourceName: repositoryName,
        });
    }
    static validateRepositoryName(physicalName) {
        const repositoryName = physicalName;
        if (!repositoryName || core_1.Token.isUnresolved(repositoryName)) {
            // the name is a late-bound value, not a defined string,
            // so skip validation
            return;
        }
        const errors = [];
        // Rules codified from https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecr-repository.html
        if (repositoryName.length < 2 || repositoryName.length > 256) {
            errors.push('Repository name must be at least 2 and no more than 256 characters');
        }
        const isPatternMatch = /^(?:[a-z0-9]+(?:[._-][a-z0-9]+)*\/)*[a-z0-9]+(?:[._-][a-z0-9]+)*$/.test(repositoryName);
        if (!isPatternMatch) {
            errors.push('Repository name must follow the specified pattern: (?:[a-z0-9]+(?:[._-][a-z0-9]+)*/)*[a-z0-9]+(?:[._-][a-z0-9]+)*');
        }
        if (errors.length > 0) {
            throw new Error(`Invalid ECR repository name (value: ${repositoryName})${os_1.EOL}${errors.join(os_1.EOL)}`);
        }
    }
    addToResourcePolicy(statement) {
        if (this.policyDocument === undefined) {
            this.policyDocument = new iam.PolicyDocument();
        }
        this.policyDocument.addStatements(statement);
        return { statementAdded: false, policyDependable: this.policyDocument };
    }
    validate() {
        var _d;
        const errors = super.validate();
        errors.push(...((_d = this.policyDocument) === null || _d === void 0 ? void 0 : _d.validateForResourcePolicy()) || []);
        return errors;
    }
    /**
     * Add a life cycle rule to the repository
     *
     * Life cycle rules automatically expire images from the repository that match
     * certain conditions.
     */
    addLifecycleRule(rule) {
        jsiiDeprecationWarnings._aws_cdk_aws_ecr_LifecycleRule(rule);
        // Validate rule here so users get errors at the expected location
        if (rule.tagStatus === undefined) {
            rule = { ...rule, tagStatus: rule.tagPrefixList === undefined ? lifecycle_1.TagStatus.ANY : lifecycle_1.TagStatus.TAGGED };
        }
        if (rule.tagStatus === lifecycle_1.TagStatus.TAGGED && (rule.tagPrefixList === undefined || rule.tagPrefixList.length === 0)) {
            throw new Error('TagStatus.Tagged requires the specification of a tagPrefixList');
        }
        if (rule.tagStatus !== lifecycle_1.TagStatus.TAGGED && rule.tagPrefixList !== undefined) {
            throw new Error('tagPrefixList can only be specified when tagStatus is set to Tagged');
        }
        if ((rule.maxImageAge !== undefined) === (rule.maxImageCount !== undefined)) {
            throw new Error(`Life cycle rule must contain exactly one of 'maxImageAge' and 'maxImageCount', got: ${JSON.stringify(rule)}`);
        }
        if (rule.tagStatus === lifecycle_1.TagStatus.ANY && this.lifecycleRules.filter(r => r.tagStatus === lifecycle_1.TagStatus.ANY).length > 0) {
            throw new Error('Life cycle can only have one TagStatus.Any rule');
        }
        this.lifecycleRules.push({ ...rule });
    }
    /**
     * Render the life cycle policy object
     */
    renderLifecyclePolicy() {
        const stack = core_1.Stack.of(this);
        let lifecyclePolicyText;
        if (this.lifecycleRules.length === 0 && !this.registryId) {
            return undefined;
        }
        if (this.lifecycleRules.length > 0) {
            lifecyclePolicyText = JSON.stringify(stack.resolve({
                rules: this.orderedLifecycleRules().map(renderLifecycleRule),
            }));
        }
        return {
            lifecyclePolicyText,
            registryId: this.registryId,
        };
    }
    /**
     * Return life cycle rules with automatic ordering applied.
     *
     * Also applies validation of the 'any' rule.
     */
    orderedLifecycleRules() {
        var _d;
        if (this.lifecycleRules.length === 0) {
            return [];
        }
        const prioritizedRules = this.lifecycleRules.filter(r => r.rulePriority !== undefined && r.tagStatus !== lifecycle_1.TagStatus.ANY);
        const autoPrioritizedRules = this.lifecycleRules.filter(r => r.rulePriority === undefined && r.tagStatus !== lifecycle_1.TagStatus.ANY);
        const anyRules = this.lifecycleRules.filter(r => r.tagStatus === lifecycle_1.TagStatus.ANY);
        if (anyRules.length > 0 && anyRules[0].rulePriority !== undefined && autoPrioritizedRules.length > 0) {
            // Supporting this is too complex for very little value. We just prohibit it.
            throw new Error("Cannot combine prioritized TagStatus.Any rule with unprioritized rules. Remove rulePriority from the 'Any' rule.");
        }
        const prios = prioritizedRules.map(r => r.rulePriority);
        let autoPrio = (prios.length > 0 ? Math.max(...prios) : 0) + 1;
        const ret = new Array();
        for (const rule of prioritizedRules.concat(autoPrioritizedRules).concat(anyRules)) {
            ret.push({
                ...rule,
                rulePriority: (_d = rule.rulePriority) !== null && _d !== void 0 ? _d : autoPrio++,
            });
        }
        // Do validation on the final array--might still be wrong because the user supplied all prios, but incorrectly.
        validateAnyRuleLast(ret);
        return ret;
    }
    /**
     * Set up key properties and return the Repository encryption property from the
     * user's configuration.
     */
    parseEncryption(props) {
        var _d, _e;
        // default based on whether encryptionKey is specified
        const encryptionType = (_d = props.encryption) !== null && _d !== void 0 ? _d : (props.encryptionKey ? RepositoryEncryption.KMS : RepositoryEncryption.AES_256);
        // if encryption key is set, encryption must be set to KMS.
        if (encryptionType !== RepositoryEncryption.KMS && props.encryptionKey) {
            throw new Error(`encryptionKey is specified, so 'encryption' must be set to KMS (value: ${encryptionType.value})`);
        }
        if (encryptionType === RepositoryEncryption.AES_256) {
            return undefined;
        }
        if (encryptionType === RepositoryEncryption.KMS) {
            return {
                encryptionType: 'KMS',
                kmsKey: (_e = props.encryptionKey) === null || _e === void 0 ? void 0 : _e.keyArn,
            };
        }
        throw new Error(`Unexpected 'encryptionType': ${encryptionType}`);
    }
}
exports.Repository = Repository;
_b = JSII_RTTI_SYMBOL_1;
Repository[_b] = { fqn: "@aws-cdk/aws-ecr.Repository", version: "1.145.0" };
function validateAnyRuleLast(rules) {
    const anyRules = rules.filter(r => r.tagStatus === lifecycle_1.TagStatus.ANY);
    if (anyRules.length === 1) {
        const maxPrio = Math.max(...rules.map(r => r.rulePriority));
        if (anyRules[0].rulePriority !== maxPrio) {
            throw new Error(`TagStatus.Any rule must have highest priority, has ${anyRules[0].rulePriority} which is smaller than ${maxPrio}`);
        }
    }
}
/**
 * Render the lifecycle rule to JSON
 */
function renderLifecycleRule(rule) {
    var _d, _e;
    return {
        rulePriority: rule.rulePriority,
        description: rule.description,
        selection: {
            tagStatus: rule.tagStatus || lifecycle_1.TagStatus.ANY,
            tagPrefixList: rule.tagPrefixList,
            countType: rule.maxImageAge !== undefined ? "sinceImagePushed" /* SINCE_IMAGE_PUSHED */ : "imageCountMoreThan" /* IMAGE_COUNT_MORE_THAN */,
            countNumber: (_e = (_d = rule.maxImageAge) === null || _d === void 0 ? void 0 : _d.toDays()) !== null && _e !== void 0 ? _e : rule.maxImageCount,
            countUnit: rule.maxImageAge !== undefined ? 'days' : undefined,
        },
        action: {
            type: 'expire',
        },
    };
}
/**
 * The tag mutability setting for your repository.
 */
var TagMutability;
(function (TagMutability) {
    /**
     * allow image tags to be overwritten.
     */
    TagMutability["MUTABLE"] = "MUTABLE";
    /**
     * all image tags within the repository will be immutable which will prevent them from being overwritten.
     */
    TagMutability["IMMUTABLE"] = "IMMUTABLE";
})(TagMutability = exports.TagMutability || (exports.TagMutability = {}));
/**
 * Indicates whether server-side encryption is enabled for the object, and whether that encryption is
 * from the AWS Key Management Service (AWS KMS) or from Amazon S3 managed encryption (SSE-S3).
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
class RepositoryEncryption {
    /**
     * @param value the string value of the encryption
     */
    constructor(value) {
        this.value = value;
    }
}
exports.RepositoryEncryption = RepositoryEncryption;
_c = JSII_RTTI_SYMBOL_1;
RepositoryEncryption[_c] = { fqn: "@aws-cdk/aws-ecr.RepositoryEncryption", version: "1.145.0" };
/**
 * 'AES256'
 */
RepositoryEncryption.AES_256 = new RepositoryEncryption('AES256');
/**
 * 'KMS'
 */
RepositoryEncryption.KMS = new RepositoryEncryption('KMS');
//# sourceMappingURL=data:application/json;base64,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