"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomRule = exports.ManagedRule = exports.MaximumExecutionFrequency = void 0;
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const config_generated_1 = require("./config.generated");
/**
 * A new or imported rule.
 */
class RuleBase extends core_1.Resource {
    /**
     * Defines a CloudWatch event rule which triggers for rule events. Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.config'],
            detail: {
                configRuleName: [this.configRuleName],
            },
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers for rule compliance events.
     */
    onComplianceChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['Config Rules Compliance Change'],
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers for rule re-evaluation status events.
     */
    onReEvaluationStatus(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['Config Rules Re-evaluation Status'],
        });
        return rule;
    }
}
/**
 * A new managed or custom rule.
 */
class RuleNew extends RuleBase {
    /**
     * Imports an existing rule.
     *
     * @param configRuleName the name of the rule
     */
    static fromConfigRuleName(scope, id, configRuleName) {
        class Import extends RuleBase {
            constructor() {
                super(...arguments);
                this.configRuleName = configRuleName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Restrict scope of changes to a specific resource.
     *
     * @see https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources
     *
     * @param type the resource type
     * @param identifier the resource identifier
     */
    scopeToResource(type, identifier) {
        this.scopeTo({
            complianceResourceId: identifier,
            complianceResourceTypes: [type],
        });
    }
    /**
     * Restrict scope of changes to specific resource types.
     *
     * @see https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources
     *
     * @param types resource types
     */
    scopeToResources(...types) {
        this.scopeTo({
            complianceResourceTypes: types,
        });
    }
    /**
     * Restrict scope of changes to a specific tag.
     *
     * @param key the tag key
     * @param value the tag value
     */
    scopeToTag(key, value) {
        this.scopeTo({
            tagKey: key,
            tagValue: value,
        });
    }
    scopeTo(scope) {
        if (!this.isManaged && !this.isCustomWithChanges) {
            throw new Error('Cannot scope rule when `configurationChanges` is set to false.');
        }
        this.scope = scope;
    }
}
/**
 * The maximum frequency at which the AWS Config rule runs evaluations.
 */
var MaximumExecutionFrequency;
(function (MaximumExecutionFrequency) {
    /**
     * 1 hour.
     */
    MaximumExecutionFrequency["ONE_HOUR"] = "One_Hour";
    /**
     * 3 hours.
     */
    MaximumExecutionFrequency["THREE_HOURS"] = "Three_Hours";
    /**
     * 6 hours.
     */
    MaximumExecutionFrequency["SIX_HOURS"] = "Six_Hours";
    /**
     * 12 hours.
     */
    MaximumExecutionFrequency["TWELVE_HOURS"] = "Twelve_Hours";
    /**
     * 24 hours.
     */
    MaximumExecutionFrequency["TWENTY_FOUR_HOURS"] = "TwentyFour_Hours";
})(MaximumExecutionFrequency = exports.MaximumExecutionFrequency || (exports.MaximumExecutionFrequency = {}));
/**
 * A new managed rule.
 *
 * @resource AWS::Config::ConfigRule
 */
class ManagedRule extends RuleNew {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.configRuleName,
        });
        const rule = new config_generated_1.CfnConfigRule(this, 'Resource', {
            configRuleName: this.physicalName,
            description: props.description,
            inputParameters: props.inputParameters,
            maximumExecutionFrequency: props.maximumExecutionFrequency,
            scope: core_1.Lazy.anyValue({ produce: () => this.scope }),
            source: {
                owner: 'AWS',
                sourceIdentifier: props.identifier,
            },
        });
        this.configRuleName = rule.ref;
        this.configRuleArn = rule.attrArn;
        this.configRuleId = rule.attrConfigRuleId;
        this.configRuleComplianceType = rule.attrComplianceType;
        this.isManaged = true;
    }
}
exports.ManagedRule = ManagedRule;
/**
 * A new custom rule.
 *
 * @resource AWS::Config::ConfigRule
 */
class CustomRule extends RuleNew {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.configRuleName,
        });
        if (!props.configurationChanges && !props.periodic) {
            throw new Error('At least one of `configurationChanges` or `periodic` must be set to true.');
        }
        const sourceDetails = [];
        if (props.configurationChanges) {
            sourceDetails.push({
                eventSource: 'aws.config',
                messageType: 'ConfigurationItemChangeNotification',
            });
            sourceDetails.push({
                eventSource: 'aws.config',
                messageType: 'OversizedConfigurationItemChangeNotification',
            });
        }
        if (props.periodic) {
            sourceDetails.push({
                eventSource: 'aws.config',
                maximumExecutionFrequency: props.maximumExecutionFrequency,
                messageType: 'ScheduledNotification',
            });
        }
        props.lambdaFunction.addPermission('Permission', {
            principal: new iam.ServicePrincipal('config.amazonaws.com'),
        });
        if (props.lambdaFunction.role) {
            props.lambdaFunction.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSConfigRulesExecutionRole'));
        }
        // The lambda permission must be created before the rule
        this.node.addDependency(props.lambdaFunction);
        const rule = new config_generated_1.CfnConfigRule(this, 'Resource', {
            configRuleName: this.physicalName,
            description: props.description,
            inputParameters: props.inputParameters,
            maximumExecutionFrequency: props.maximumExecutionFrequency,
            scope: core_1.Lazy.anyValue({ produce: () => this.scope }),
            source: {
                owner: 'CUSTOM_LAMBDA',
                sourceDetails,
                sourceIdentifier: props.lambdaFunction.functionArn,
            },
        });
        this.configRuleName = rule.ref;
        this.configRuleArn = rule.attrArn;
        this.configRuleId = rule.attrConfigRuleId;
        this.configRuleComplianceType = rule.attrComplianceType;
        if (props.configurationChanges) {
            this.isCustomWithChanges = true;
        }
    }
}
exports.CustomRule = CustomRule;
//# sourceMappingURL=data:application/json;base64,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