from dataclasses import dataclass
from typing import Sequence, TextIO, TypeVar

from pyass.event import Event
from pyass.section import (
    AegisubGarbageSection,
    EventsSection,
    ScriptInfoSection,
    Section,
    StylesSection,
)
from pyass.style import Style

Script = TypeVar("Script", bound="Script")
SectionT = TypeVar("SectionT", bound=Section)


@dataclass
class Script:
    sections: list[Section]

    def __init__(
        self,
        scriptInfo: Sequence[tuple[str, str]] = [],
        aegisubGarbage: Sequence[tuple[str, str]] = [],
        styles: Sequence[Style] = [],
        events: Sequence[Event] = [],
    ):
        self.sections = []

        if len(scriptInfo) == 0:
            scriptInfo = [
                ("", "Script generated by pyass"),
                ("Title", "Default Aegisub file"),
                ("Script Type", "v4.00+"),
                ("Wrap Style", "0"),
                ("ScaledBorderAndShadow", "yes"),
                ("YCbCr Matrix", "None"),
            ]

        self.sections.append(ScriptInfoSection(scriptInfo))
        self.sections.append(AegisubGarbageSection(aegisubGarbage))
        self.sections.append(StylesSection(styles))
        self.sections.append(EventsSection(events))

    def __str__(self) -> str:
        excludeIfEmptySections = [AegisubGarbageSection]
        sectionsToPrint = [
            section
            for section in self.sections
            if section.header()
            not in [SectionType.header() for SectionType in excludeIfEmptySections]
            or section
        ]
        return "\n".join([str(section) for section in sectionsToPrint])

    @staticmethod
    def parse(s: str) -> Script:
        lines = s.splitlines()
        currSectionLines = []

        ret = Script()
        ret.sections.clear()

        for line in lines:
            if line.startswith("[") and line.endswith("]"):
                if currSectionLines:
                    ret.sections.append(Section.parse("\n".join(currSectionLines)))
                    currSectionLines.clear()

            currSectionLines.append(line)

        if currSectionLines:
            ret.sections.append(Section.parse("\n".join(currSectionLines)))

        return ret

    @property
    def scriptInfo(self) -> ScriptInfoSection:
        return self._get_section_by_type(ScriptInfoSection)

    @scriptInfo.setter
    def scriptInfo(self, s: Sequence[tuple[str, str]]):
        self._set_section(ScriptInfoSection(s))

    @property
    def aegisubGarbage(self) -> AegisubGarbageSection:
        return self._get_section_by_type(AegisubGarbageSection)

    @aegisubGarbage.setter
    def aegisubGarbage(self, s: Sequence[tuple[str, str]]):
        self._set_section(AegisubGarbageSection(s))

    @property
    def styles(self) -> StylesSection:
        return self._get_section_by_type(StylesSection)

    @styles.setter
    def styles(self, s: Sequence[Style]):
        self._set_section(StylesSection(s))

    @property
    def events(self) -> EventsSection:
        return self._get_section_by_type(EventsSection)

    @events.setter
    def events(self, s: Sequence[Event]):
        self._set_section(EventsSection(s))

    def dump(self, fp: TextIO) -> None:
        fp.write(self.dumps())

    def dumps(self) -> str:
        return str(self)

    def _get_section_by_type(self, t: type[SectionT]) -> SectionT:
        for section in self.sections:
            if isinstance(section, t):
                return section

        raise AttributeError

    def _set_section(self, s: Section):
        for i, section in enumerate(self.sections):
            if isinstance(s, type(section)):
                self.sections[i] = s
                return

        raise AttributeError
