"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Canary = exports.Test = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const s3 = require("aws-cdk-lib/aws-s3");
const cdk = require("aws-cdk-lib");
const synthetics_canned_metrics_generated_1 = require("./synthetics-canned-metrics.generated");
const aws_synthetics_1 = require("aws-cdk-lib/aws-synthetics");
/**
 * Specify a test that the canary should run
 */
class Test {
    /**
     * Construct a Test property
     *
     * @param code The code that the canary should run
     * @param handler The handler of the canary
     */
    constructor(code, handler) {
        this.code = code;
        this.handler = handler;
    }
    /**
     * Specify a custom test with your own code
     *
     * @returns `Test` associated with the specified Code object
     * @param options The configuration options
     */
    static custom(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_synthetics_alpha_CustomTestOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.custom);
            }
            throw error;
        }
        Test.validateHandler(options.handler);
        return new Test(options.code, options.handler);
    }
    /**
     * Verifies that the given handler ends in '.handler'. Returns the handler if successful and
     * throws an error if not.
     *
     * @param handler - the handler given by the user
     */
    static validateHandler(handler) {
        if (!handler.endsWith('.handler')) {
            throw new Error(`Canary Handler must end in '.handler' (${handler})`);
        }
        if (handler.length > 21) {
            throw new Error(`Canary Handler must be less than 21 characters (${handler})`);
        }
    }
}
exports.Test = Test;
_a = JSII_RTTI_SYMBOL_1;
Test[_a] = { fqn: "@aws-cdk/aws-synthetics-alpha.Test", version: "2.68.0-alpha.0" };
/**
 * Define a new Canary
 */
class Canary extends cdk.Resource {
    constructor(scope, id, props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_synthetics_alpha_CanaryProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Canary);
            }
            throw error;
        }
        if (props.canaryName && !cdk.Token.isUnresolved(props.canaryName)) {
            validateName(props.canaryName);
        }
        super(scope, id, {
            physicalName: props.canaryName || cdk.Lazy.string({
                produce: () => this.generateUniqueName(),
            }),
        });
        this.artifactsBucket = props.artifactsBucketLocation?.bucket ?? new s3.Bucket(this, 'ArtifactsBucket', {
            encryption: s3.BucketEncryption.KMS_MANAGED,
            enforceSSL: true,
        });
        this.role = props.role ?? this.createDefaultRole(props);
        if (props.vpc) {
            // Security Groups are created and/or appended in `createVpcConfig`.
            this._connections = new ec2.Connections({});
        }
        const resource = new aws_synthetics_1.CfnCanary(this, 'Resource', {
            artifactS3Location: this.artifactsBucket.s3UrlForObject(props.artifactsBucketLocation?.prefix),
            executionRoleArn: this.role.roleArn,
            startCanaryAfterCreation: props.startAfterCreation ?? true,
            runtimeVersion: props.runtime.name,
            name: this.physicalName,
            schedule: this.createSchedule(props),
            failureRetentionPeriod: props.failureRetentionPeriod?.toDays(),
            successRetentionPeriod: props.successRetentionPeriod?.toDays(),
            code: this.createCode(props),
            runConfig: this.createRunConfig(props),
            vpcConfig: this.createVpcConfig(props),
            deleteLambdaResourcesOnCanaryDeletion: props.enableAutoDeleteLambdas,
        });
        this.canaryId = resource.attrId;
        this.canaryState = resource.attrState;
        this.canaryName = this.getResourceNameAttribute(resource.ref);
    }
    /**
     * Access the Connections object
     *
     * Will fail if not a VPC-enabled Canary
     */
    get connections() {
        if (!this._connections) {
            // eslint-disable-next-line max-len
            throw new Error('Only VPC-associated Canaries have security groups to manage. Supply the "vpc" parameter when creating the Canary.');
        }
        return this._connections;
    }
    /**
     * Measure the Duration of a single canary run, in seconds.
     *
     * @param options - configuration options for the metric
     *
     * @default avg over 5 minutes
     */
    metricDuration(options) {
        return new aws_cloudwatch_1.Metric({
            ...synthetics_canned_metrics_generated_1.CloudWatchSyntheticsMetrics.durationMaximum({ CanaryName: this.canaryName }),
            ...{ statistic: 'Average' },
            ...options,
        }).attachTo(this);
    }
    /**
     * Measure the percentage of successful canary runs.
     *
     * @param options - configuration options for the metric
     *
     * @default avg over 5 minutes
     */
    metricSuccessPercent(options) {
        return this.cannedMetric(synthetics_canned_metrics_generated_1.CloudWatchSyntheticsMetrics.successPercentAverage, options);
    }
    /**
     * Measure the number of failed canary runs over a given time period.
     *
     * Default: sum over 5 minutes
     *
     * @param options - configuration options for the metric
     */
    metricFailed(options) {
        return this.cannedMetric(synthetics_canned_metrics_generated_1.CloudWatchSyntheticsMetrics.failedSum, options);
    }
    /**
     * Returns a default role for the canary
     */
    createDefaultRole(props) {
        const prefix = props.artifactsBucketLocation?.prefix;
        // Created role will need these policies to run the Canary.
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-synthetics-canary.html#cfn-synthetics-canary-executionrolearn
        const policy = new iam.PolicyDocument({
            statements: [
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: ['s3:ListAllMyBuckets'],
                }),
                new iam.PolicyStatement({
                    resources: [this.artifactsBucket.bucketArn],
                    actions: ['s3:GetBucketLocation'],
                }),
                new iam.PolicyStatement({
                    resources: [this.artifactsBucket.arnForObjects(`${prefix ? prefix + '/*' : '*'}`)],
                    actions: ['s3:PutObject'],
                }),
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: ['cloudwatch:PutMetricData'],
                    conditions: { StringEquals: { 'cloudwatch:namespace': 'CloudWatchSynthetics' } },
                }),
                new iam.PolicyStatement({
                    resources: [this.logGroupArn()],
                    actions: ['logs:CreateLogStream', 'logs:CreateLogGroup', 'logs:PutLogEvents'],
                }),
            ],
        });
        const managedPolicies = [];
        if (props.vpc) {
            // Policy that will have ENI creation permissions
            managedPolicies.push(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaVPCAccessExecutionRole'));
        }
        return new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                canaryPolicy: policy,
            },
            managedPolicies,
        });
    }
    logGroupArn() {
        return cdk.Stack.of(this).formatArn({
            service: 'logs',
            resource: 'log-group',
            arnFormat: cdk.ArnFormat.COLON_RESOURCE_NAME,
            resourceName: '/aws/lambda/cwsyn-*',
        });
    }
    /**
     * Returns the code object taken in by the canary resource.
     */
    createCode(props) {
        const codeConfig = {
            handler: props.test.handler,
            ...props.test.code.bind(this, props.test.handler, props.runtime.family),
        };
        return {
            handler: codeConfig.handler,
            script: codeConfig.inlineCode,
            s3Bucket: codeConfig.s3Location?.bucketName,
            s3Key: codeConfig.s3Location?.objectKey,
            s3ObjectVersion: codeConfig.s3Location?.objectVersion,
        };
    }
    createRunConfig(props) {
        if (!props.environmentVariables) {
            return undefined;
        }
        return {
            environmentVariables: props.environmentVariables,
        };
    }
    /**
     * Returns a canary schedule object
     */
    createSchedule(props) {
        return {
            durationInSeconds: String(`${props.timeToLive?.toSeconds() ?? 0}`),
            expression: props.schedule?.expressionString ?? 'rate(5 minutes)',
        };
    }
    createVpcConfig(props) {
        if (!props.vpc) {
            if (props.vpcSubnets != null || props.securityGroups != null) {
                throw new Error("You must provide the 'vpc' prop when using VPC-related properties.");
            }
            return undefined;
        }
        const { subnetIds } = props.vpc.selectSubnets(props.vpcSubnets);
        if (subnetIds.length < 1) {
            throw new Error('No matching subnets found in the VPC.');
        }
        let securityGroups;
        if (props.securityGroups && props.securityGroups.length > 0) {
            securityGroups = props.securityGroups;
        }
        else {
            const securityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
                description: 'Automatic security group for Canary ' + cdk.Names.uniqueId(this),
            });
            securityGroups = [securityGroup];
        }
        this._connections.addSecurityGroup(...securityGroups);
        return {
            vpcId: props.vpc.vpcId,
            subnetIds,
            securityGroupIds: cdk.Lazy.list({ produce: () => this.connections.securityGroups.map(sg => sg.securityGroupId) }),
        };
    }
    /**
     * Creates a unique name for the canary. The generated name is the physical ID of the canary.
     */
    generateUniqueName() {
        const name = cdk.Names.uniqueId(this).toLowerCase().replace(' ', '-');
        if (name.length <= 21) {
            return name;
        }
        else {
            return name.substring(0, 15) + nameHash(name);
        }
    }
    cannedMetric(fn, props) {
        return new aws_cloudwatch_1.Metric({
            ...fn({ CanaryName: this.canaryName }),
            ...props,
        }).attachTo(this);
    }
}
exports.Canary = Canary;
_b = JSII_RTTI_SYMBOL_1;
Canary[_b] = { fqn: "@aws-cdk/aws-synthetics-alpha.Canary", version: "2.68.0-alpha.0" };
/**
 * Take a hash of the given name.
 *
 * @param name the name to be hashed
 */
function nameHash(name) {
    const md5 = crypto.createHash('sha256').update(name).digest('hex');
    return md5.slice(0, 6);
}
const nameRegex = /^[0-9a-z_\-]+$/;
/**
 * Verifies that the name fits the regex expression: ^[0-9a-z_\-]+$.
 *
 * @param name - the given name of the canary
 */
function validateName(name) {
    if (name.length > 21) {
        throw new Error(`Canary name is too large, must be between 1 and 21 characters, but is ${name.length} (got "${name}")`);
    }
    if (!nameRegex.test(name)) {
        throw new Error(`Canary name must be lowercase, numbers, hyphens, or underscores (got "${name}")`);
    }
}
//# sourceMappingURL=data:application/json;base64,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