"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Template = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const core_1 = require("@aws-cdk/core");
const fs = require("fs-extra");
const match_1 = require("./match");
const matcher_1 = require("./matcher");
const conditions_1 = require("./private/conditions");
const cyclic_1 = require("./private/cyclic");
const mappings_1 = require("./private/mappings");
const outputs_1 = require("./private/outputs");
const parameters_1 = require("./private/parameters");
const resources_1 = require("./private/resources");
/**
 * Suite of assertions that can be run on a CDK stack.
 * Typically used, as part of unit tests, to validate that the rendered
 * CloudFormation template has expected resources and properties.
 */
class Template {
    constructor(template) {
        this.template = template;
        cyclic_1.checkTemplateForCyclicDependencies(this.template);
    }
    /**
     * Base your assertions on the CloudFormation template synthesized by a CDK `Stack`.
     * @param stack the CDK Stack to run assertions on
     */
    static fromStack(stack) {
        return new Template(toTemplate(stack));
    }
    /**
     * Base your assertions from an existing CloudFormation template formatted as an in-memory
     * JSON object.
     * @param template the CloudFormation template formatted as a nested set of records
     */
    static fromJSON(template) {
        return new Template(template);
    }
    /**
     * Base your assertions from an existing CloudFormation template formatted as a
     * JSON string.
     * @param template the CloudFormation template in
     */
    static fromString(template) {
        return new Template(JSON.parse(template));
    }
    /**
     * The CloudFormation template deserialized into an object.
     */
    toJSON() {
        return this.template;
    }
    /**
     * Assert that the given number of resources of the given type exist in the
     * template.
     * @param type the resource type; ex: `AWS::S3::Bucket`
     * @param count number of expected instances
     */
    resourceCountIs(type, count) {
        const counted = resources_1.countResources(this.template, type);
        if (counted !== count) {
            throw new Error(`Expected ${count} resources of type ${type} but found ${counted}`);
        }
    }
    /**
     * Assert that a resource of the given type and properties exists in the
     * CloudFormation template.
     * By default, performs partial matching on the `Properties` key of the resource, via the
     * `Match.objectLike()`. To configure different behavour, use other matchers in the `Match` class.
     * @param type the resource type; ex: `AWS::S3::Bucket`
     * @param props the 'Properties' section of the resource as should be expected in the template.
     */
    hasResourceProperties(type, props) {
        const matchError = resources_1.hasResourceProperties(this.template, type, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Assert that a resource of the given type and given definition exists in the
     * CloudFormation template.
     * By default, performs partial matching on the resource, via the `Match.objectLike()`.
     * To configure different behavour, use other matchers in the `Match` class.
     * @param type the resource type; ex: `AWS::S3::Bucket`
     * @param props the entire defintion of the resource as should be expected in the template.
     */
    hasResource(type, props) {
        const matchError = resources_1.hasResource(this.template, type, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching resources of a given type and properties in the CloudFormation template.
     * @param type the type to match in the CloudFormation template
     * @param props by default, matches all resources with the given type.
     * When a literal is provided, performs a partial match via `Match.objectLike()`.
     * Use the `Match` APIs to configure a different behaviour.
     */
    findResources(type, props = {}) {
        return resources_1.findResources(this.template, type, props);
    }
    /**
     * Assert that a Parameter with the given properties exists in the CloudFormation template.
     * By default, performs partial matching on the parameter, via the `Match.objectLike()`.
     * To configure different behavior, use other matchers in the `Match` class.
     * @param logicalId the name of the parameter. Provide `'*'` to match all parameters in the template.
     * @param props the parameter as should be expected in the template.
     */
    hasParameter(logicalId, props) {
        const matchError = parameters_1.hasParameter(this.template, logicalId, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching Parameters that match the given properties in the CloudFormation template.
     * @param logicalId the name of the parameter. Provide `'*'` to match all parameters in the template.
     * @param props by default, matches all Parameters in the template.
     * When a literal object is provided, performs a partial match via `Match.objectLike()`.
     * Use the `Match` APIs to configure a different behaviour.
     */
    findParameters(logicalId, props = {}) {
        return parameters_1.findParameters(this.template, logicalId, props);
    }
    /**
     * Assert that an Output with the given properties exists in the CloudFormation template.
     * By default, performs partial matching on the resource, via the `Match.objectLike()`.
     * To configure different behavour, use other matchers in the `Match` class.
     * @param logicalId the name of the output. Provide `'*'` to match all outputs in the template.
     * @param props the output as should be expected in the template.
     */
    hasOutput(logicalId, props) {
        const matchError = outputs_1.hasOutput(this.template, logicalId, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching Outputs that match the given properties in the CloudFormation template.
     * @param logicalId the name of the output. Provide `'*'` to match all outputs in the template.
     * @param props by default, matches all Outputs in the template.
     * When a literal object is provided, performs a partial match via `Match.objectLike()`.
     * Use the `Match` APIs to configure a different behaviour.
     */
    findOutputs(logicalId, props = {}) {
        return outputs_1.findOutputs(this.template, logicalId, props);
    }
    /**
     * Assert that a Mapping with the given properties exists in the CloudFormation template.
     * By default, performs partial matching on the resource, via the `Match.objectLike()`.
     * To configure different behavour, use other matchers in the `Match` class.
     * @param logicalId the name of the mapping. Provide `'*'` to match all mappings in the template.
     * @param props the output as should be expected in the template.
     */
    hasMapping(logicalId, props) {
        const matchError = mappings_1.hasMapping(this.template, logicalId, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching Mappings that match the given properties in the CloudFormation template.
     * @param logicalId the name of the mapping. Provide `'*'` to match all mappings in the template.
     * @param props by default, matches all Mappings in the template.
     * When a literal object is provided, performs a partial match via `Match.objectLike()`.
     * Use the `Match` APIs to configure a different behaviour.
     */
    findMappings(logicalId, props = {}) {
        return mappings_1.findMappings(this.template, logicalId, props);
    }
    /**
     * Assert that a Condition with the given properties exists in the CloudFormation template.
     * By default, performs partial matching on the resource, via the `Match.objectLike()`.
     * To configure different behavour, use other matchers in the `Match` class.
     * @param logicalId the name of the mapping. Provide `'*'` to match all conditions in the template.
     * @param props the output as should be expected in the template.
     */
    hasCondition(logicalId, props) {
        const matchError = conditions_1.hasCondition(this.template, logicalId, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching Conditions that match the given properties in the CloudFormation template.
     * @param logicalId the name of the condition. Provide `'*'` to match all conditions in the template.
     * @param props by default, matches all Conditions in the template.
     * When a literal object is provided, performs a partial match via `Match.objectLike()`.
     * Use the `Match` APIs to configure a different behaviour.
     */
    findConditions(logicalId, props = {}) {
        return conditions_1.findConditions(this.template, logicalId, props);
    }
    /**
     * Assert that the CloudFormation template matches the given value
     * @param expected the expected CloudFormation template as key-value pairs.
     */
    templateMatches(expected) {
        const matcher = matcher_1.Matcher.isMatcher(expected) ? expected : match_1.Match.objectLike(expected);
        const result = matcher.test(this.template);
        if (result.hasFailed()) {
            throw new Error([
                'Template did not match as expected. The following mismatches were found:',
                ...result.toHumanStrings().map(s => `\t${s}`),
            ].join('\n'));
        }
    }
}
exports.Template = Template;
_a = JSII_RTTI_SYMBOL_1;
Template[_a] = { fqn: "@aws-cdk/assertions.Template", version: "1.163.0" };
function toTemplate(stack) {
    const root = stack.node.root;
    if (!core_1.Stage.isStage(root)) {
        throw new Error('unexpected: all stacks must be part of a Stage or an App');
    }
    const assembly = root.synth();
    if (stack.nestedStackParent) {
        // if this is a nested stack (it has a parent), then just read the template as a string
        return JSON.parse(fs.readFileSync(path.join(assembly.directory, stack.templateFile)).toString('utf-8'));
    }
    return assembly.getStackArtifact(stack.artifactId).template;
}
//# sourceMappingURL=data:application/json;base64,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