import json
from typing import List, Optional

from botocore.client import BaseClient

from dstack.aws import runners
from dstack.backend import Secret
from dstack.repo import RepoAddress, _repo_address_path


def list_secret_names(s3_client: BaseClient, bucket_name: str, repo_address: RepoAddress) -> List[str]:
    prefix = f"secrets/{_repo_address_path(repo_address)}"
    secret_head_prefix = f"{prefix}/l;"
    response = s3_client.list_objects_v2(Bucket=bucket_name, Prefix=secret_head_prefix)
    secret_names = []
    if "Contents" in response:
        for obj in response["Contents"]:
            secret_name = obj["Key"][len(secret_head_prefix):]
            secret_names.append(secret_name)
    return secret_names


def get_secret(secretsmanager_client: BaseClient, bucket_name: str, repo_address: RepoAddress, secret_name: str) \
        -> Optional[Secret]:
    try:
        return Secret(secret_name, secretsmanager_client.get_secret_value(
            SecretId=f"/dstack/{bucket_name}/secrets/{_repo_address_path(repo_address)}/{secret_name}"
        )["SecretString"])
    except Exception as e:
        if hasattr(e, "response") and e.response.get("Error") and e.response["Error"].get(
                "Code") in ["ResourceNotFoundException", "InvalidRequestException"]:
            return None
        else:
            raise e


def _secret_head_key(repo_address: RepoAddress, secret_name: str) -> str:
    prefix = f"secrets/{_repo_address_path(repo_address)}"
    key = f"{prefix}/l;{secret_name}"
    return key


def add_secret(sts_client: BaseClient, iam_client: BaseClient, secretsmanager_client: BaseClient, s3_client: BaseClient,
               bucket_name: str, repo_address: RepoAddress, secret: Secret):
    secret_id = f"/dstack/{bucket_name}/secrets/{_repo_address_path(repo_address)}/{secret.secret_name}"
    secretsmanager_client.create_secret(
        Name=secret_id,
        SecretString=secret.secret_value,
        Description="Generated by dstack",
        Tags=[
            {
                'Key': 'owner',
                'Value': 'dstack'
            },
            {
                'Key': 'dstack_bucket',
                'Value': bucket_name
            }
        ],
    )
    role_name = runners.role_name(iam_client, bucket_name)
    account_id = sts_client.get_caller_identity()["Account"]
    secretsmanager_client.put_resource_policy(
        SecretId=secret_id,
        ResourcePolicy=json.dumps(
            {
                "Version": "2012-10-17",
                "Statement": [
                    {
                        "Effect": "Allow",
                        "Principal": {"AWS": f"arn:aws:iam::{account_id}:role/{role_name}"},
                        "Action": [
                            "secretsmanager:GetSecretValue",
                            "secretsmanager:ListSecrets",
                        ],
                        "Resource": "*"
                    }
                ]
            }
        )
    )
    secret_head_key = _secret_head_key(repo_address, secret.secret_name)
    s3_client.put_object(Body="", Bucket=bucket_name, Key=secret_head_key)


def update_secret(secretsmanager_client: BaseClient, s3_client: BaseClient, bucket_name: str, repo_address: RepoAddress,
                  secret: Secret):
    secret_id = f"/dstack/{bucket_name}/secrets/{_repo_address_path(repo_address)}/{secret.secret_name}"
    secretsmanager_client.put_secret_value(
        SecretId=secret_id,
        SecretString=secret.secret_value,
    )
    secret_head_key = _secret_head_key(repo_address, secret.secret_name)
    s3_client.put_object(Body="", Bucket=bucket_name, Key=secret_head_key)


def delete_secret(secretsmanager_client: BaseClient, s3_client: BaseClient, bucket_name: str, repo_address: RepoAddress,
                  secret_name: str):
    secretsmanager_client.delete_secret(
        SecretId=f"/dstack/{bucket_name}/secrets/{_repo_address_path(repo_address)}/{secret_name}",
        ForceDeleteWithoutRecovery=True
    )
    secret_head_key = _secret_head_key(repo_address, secret_name)
    s3_client.delete_object(Bucket=bucket_name, Key=secret_head_key)
