# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/00_export_scala.ipynb (unless otherwise specified).

__all__ = ['read_nb', 'is_scala_nb', 'check_re', 'check_re_multi', 'is_export', 'is_non_def_export',
           'find_default_export', 'find_non_default_exports', 'export_names', 'reset_nbdev_module', 'get_nbdev_module',
           'save_nbdev_module', 'split_flags_and_code', 'create_mod_file', 'create_mod_files', 'update_baseurl',
           'nbglob', 'scala_notebook2script']

# Cell
from nbdev.imports import *
from fastcore.script import *
from fastcore.foundation import *
from keyword import iskeyword
import nbformat

# Cell
def read_nb(fname):
    "Read the notebook in `fname`."
    with open(Path(fname),'r', encoding='utf8') as f: return nbformat.reads(f.read(), as_version=4)

# Cell
def is_scala_nb(fname):
    kernel = read_nb(fname)['metadata']['kernelspec']
    is_scala = kernel['language'] == 'scala'
#     print(f'{fname} is {"" if is_scala else "not"} a scala notebook')
    return is_scala

# Cell
def check_re(cell, pat, code_only=True):
    "Check if `cell` contains a line with regex `pat`"
    if code_only and cell['cell_type'] != 'code': return
    if isinstance(pat, str): pat = re.compile(pat, re.IGNORECASE | re.MULTILINE)
    return pat.search(cell['source'])

# Cell
def check_re_multi(cell, pats, code_only=True):
    "Check if `cell` contains a line matching any regex in `pats`, returning the first match found"
    return L(pats).map_first(partial(check_re, cell, code_only=code_only))

# Cell
def _mk_flag_re(body, n_params, comment):
    "Compiles a regex for finding nbdev flags"
    assert body!=True, 'magics no longer supported'
    prefix = r"\s*\//\s*" # scala style comment "//" is prefix
    param_group = ""
    if n_params == -1: param_group = r"[ \t]+(.+)"
    if n_params == 1: param_group = r"[ \t]+(\S+)"
    if n_params == (0,1): param_group = r"(?:[ \t]+(\S+))?"
    return re.compile(rf"""
# {comment}:
^           # beginning of line (since re.MULTILINE is passed)
{prefix}
{body}
{param_group}
[ \t]*       # any number of spaces and/or tabs
$            # end of line (since re.MULTILINE is passed)
""", re.MULTILINE | re.VERBOSE)

# Cell
_re_blank_export = _mk_flag_re("export[si]?", 0,
    "Matches any line with //export, //exports or //exporti without any module name")

# Cell
_re_mod_export = _mk_flag_re("export[si]?", 1,
    "Matches any line with //export, //exports or //exporti with a module name and catches it in group 1")

# Cell
_re_internal_export = _mk_flag_re("exporti", (0,1),
    "Matches any line with //exporti with or without a module name")

# Internal Cell
def _is_external_export(tst):
    "Check if a cell is an external or internal export. `tst` is an re match"
    return _re_internal_export.search(tst.string) is None

# Cell
def is_export(cell, default):
    "Check if `cell` is to be exported and returns the name of the module to export it if provided"
    tst = check_re(cell, _re_blank_export)
    if tst:
        if default is None:
            print(f"No export destination, ignored:\n{cell['source']}")
        return default, _is_external_export(tst)
    tst = check_re(cell, _re_mod_export)
    if tst: return os.path.sep.join(tst.groups()[0].split('.')), _is_external_export(tst)
    else: return None

# Cell
def is_non_def_export(cell):
    "Check if `cell` is to be exported to special (non-defualt) module returns the name of the module to export"
    tst = check_re(cell, _re_mod_export)
    if tst: return os.path.sep.join(tst.groups()[0].split('.')), _is_external_export(tst)
    else: return None

# Cell
_re_default_exp = _mk_flag_re('default_exp', 1, "Matches any line with //default_exp with a module name")

# Cell
def find_default_export(cells):
    "Find in `cells` the default export module."
    res = L(cells).map_first(check_re, pat=_re_default_exp)
    return res.groups()[0] if res else None

# Cell
def find_non_default_exports(cells):
    "Find in `cells` all non default export modules: //export my_mod_name"
    res = L(cells).map(is_non_def_export).filter(lambda x: x != None)
    return [mod_name for mod_name, b in res]

# Cell
_re_class_func_def = re.compile(r"""
# Catches any function or class or trait or case class or object definition
(?:abstract\s+class|case\s+class|class|object|trait|sealed\s+trait|implicit\s+def|def)  # Non-catching group for def or class
\s+            # One whitespace or more
([^\(\s\[]*)     # Catching group with any character except an opening parenthesis or a whitespace (name) or a left brace for type parameter
\s*            # Any number of whitespace
(?:\(|:|{|extends|\[)       # Non-catching group with either an opening parenthesis or a : (classes don't need ())
""", re.MULTILINE | re.VERBOSE)

# Cell
def export_names(code):
    "Find the names of objects, functions or classes defined in `code` that are exported."
    names = _re_class_func_def.findall(code)
    return names

# Cell
_re_index_custom = re.compile(r'def custom_doc_links\(name\):(.*)$', re.DOTALL)

# Cell
def reset_nbdev_module():
    "Create a skeleton for <code>_nbdev</code>"
    fname = Config().path("lib_path")/'_nbdev.py'
    print(f'resetting {fname}')
    fname.parent.mkdir(parents=True, exist_ok=True)
    sep = '\n'* (int(Config().get('cell_spacing', '1'))+1)
    if fname.is_file():
        with open(fname, 'r') as f: search = _re_index_custom.search(f.read())
    else: search = None
    prev_code = search.groups()[0] if search is not None else ' return None\n'
    with open(fname, 'w') as f:
        f.write(f"# AUTOGENERATED BY NBDEV! DO NOT EDIT!")
        f.write('\n\n__all__ = ["index", "modules", "custom_doc_links", "git_url"]')
        f.write('\n\nindex = {}')
        f.write('\n\nmodules = []')
        f.write(f'\n\ndoc_url = "{Config().doc_host}{Config().doc_baseurl}"')
        f.write(f'\n\ngit_url = "{Config().git_url}"')
        f.write(f'{sep}def custom_doc_links(name):{prev_code}')

# Cell
class _EmptyModule():
    def __init__(self):
        self.index,self.modules = {},[]
        self.doc_url,self.git_url = f"{Config().doc_host}{Config().doc_baseurl}",Config().git_url

    def custom_doc_links(self, name): return None

# Cell
def get_nbdev_module():
    "Reads <code>_nbdev</code>"
    try:
        spec = importlib.util.spec_from_file_location(f"{Config().lib_name}._nbdev", Config().path("lib_path")/'_nbdev.py')
        mod = importlib.util.module_from_spec(spec)
        spec.loader.exec_module(mod)
        return mod
    except: return _EmptyModule()

# Cell
_re_index_idx = re.compile(r'index\s*=\s*{[^}]*}')
_re_index_mod = re.compile(r'modules\s*=\s*\[[^\]]*\]')

# Cell
def save_nbdev_module(mod):
    "Save `mod` inside <code>_nbdev</code>"
    fname = Config().path("lib_path")/'_nbdev.py'
    with open(fname, 'r') as f: code = f.read()
    t = r',\n         '.join([f'"{k}": "{v}"' for k,v in mod.index.items()])
    code = _re_index_idx.sub("index = {"+ t +"}", code)
    t = r',\n           '.join(['"' + f.replace('\\','/') + '"' for f in mod.modules])
    code = _re_index_mod.sub(f"modules = [{t}]", code)
    with open(fname, 'w') as f: f.write(code)

# Cell
def split_flags_and_code(cell, return_type=list):
    "Splits the `source` of a cell into 2 parts and returns (flags, code)"
    code_lines = cell['source'].split('\n')
    split_pos = 0 if code_lines[0].strip().startswith('//') else -1
    for i, line in enumerate(code_lines):
        if not line.startswith('//') and line.strip(): break
    split_pos+=1
    res = code_lines[:split_pos], code_lines[split_pos:]
    if return_type is list: return res
    return tuple('\n'.join(r) for r in res)

# Cell
def create_mod_file(fname, nb_path, bare=False):
    "Create a module file for `fname`."
    bare = str(Config().get('bare', bare)) == 'True'
    fname.parent.mkdir(parents=True, exist_ok=True)
    file_path = os.path.relpath(nb_path, Config().config_file.parent).replace('\\', '/')
#     print(f"create_mod_file: : {file_path}/{fname}")
    with open(fname, 'w') as f:
        if not bare: f.write(f"// AUTOGENERATED! DO NOT EDIT! File to edit: {file_path} (unless otherwise specified).")
#         f.write('\n\n__all__ = []')

# Cell
def create_mod_files(files, to_dict=False, bare=False):
    "Create mod files for default exports found in `files`"
    modules = []
    for f in files:
        fname = Path(f)
#         print(f"in create_mod_files, {f}")
        nb = read_nb(fname)
        if not is_scala_nb(f):
            print(f'Fail: Not creating mod file for non-scala notebook: {f}')
            # skip non-scala mod files
            continue
        default = find_default_export(nb['cells'])
        if default is not None:
            default = os.path.sep.join(default.split('.'))
            modules.append(default)
            if not to_dict:
                create_mod_file(Config().path("lib_path")/f'{default}.sc', Config().path("nbs_path")/f'{fname}', bare=bare)
        # add support for adding info on non-default models to _nvdev.py
        for non_def_mod in find_non_default_exports(nb['cells']):
            modules.append(non_def_mod)
            if not to_dict:
#                 print(f'creating mod file for {non_def_mod}')
                create_mod_file(Config().path("lib_path")/f'{non_def_mod}.sc', Config().path("nbs_path")/f'{fname}', bare=bare)
    return modules

# Cell
def _notebook2script(fname, modules, silent=False, to_dict=None, bare=False):
    "Finds cells starting with `//export` and puts them into a module created by `create_mod_files`"
    bare = str(Config().get('bare', bare)) == 'True'
    if os.environ.get('IN_TEST',0): return  # don't export if running tests
    sep = '\n'* (int(Config().get('cell_spacing', '1'))+1)
    nbname = fname
    fname = Path(fname)
    nb = read_nb(fname)
    if not is_scala_nb(nbname):
        print(f'Fail: {nbname} is not a Scala notebook', to_dict)
        if to_dict is None: return
        return to_dict
    default = find_default_export(nb['cells'])
    if default is not None:
        default = os.path.sep.join(default.split('.'))
    mod = get_nbdev_module()
    exports = [is_export(c, default) for c in nb['cells']]
    cells = [(i,c,e) for i,(c,e) in enumerate(zip(nb['cells'],exports)) if e is not None]
    for i,c,(e,a) in cells:
        if e not in modules:
            print(f'Warning: Exporting to "{e}.sc" but this module is not part of this build: {modules}')
#             create_mod_file(e, nb_path, bare=False)
        fname_out = Config().path("lib_path")/f'{e}.sc'
        if bare: orig = "\n"
        else: orig = (f'// {"" if a else "Internal "}C' if e==default else f'// Comes from {fname.name}, c') + 'ell\n'
        flag_lines,code_lines = split_flags_and_code(c)
#         code_lines = _deal_import(code_lines, fname_out)
        code = sep + orig + '\n'.join(code_lines)
        names = export_names(code)
        flags = '\n'.join(flag_lines)
#         extra,code = extra_add(flags, code)
#         code = _from_future_import(fname_out, flags, code, to_dict)
#         if a:
#             if to_dict is None: _add2all(fname_out, [f"'{f}'" for f in names if '.' not in f and len(f) > 0] + extra)
#         print(f"{fname} updating index with: {names}")

        # Handle conflicting names from different modules
        for f in names:
            if f in mod.index and mod.index.get(f) != fname.name:
                new_name = fname.name.split('.')[0] + '.' + f
                print(f"COLLISION, {f} : {mod.index.get(f)} already in index, renaming to '{new_name} : {fname.name}'")
                mod.index.update({new_name : fname.name})
            else:
                mod.index.update({f : fname.name})
#         mod.index.update({f: fname.name for f in names})
        code = re.sub(r' +$', '', code, flags=re.MULTILINE)
        if code != sep + orig[:-1]:
            if to_dict is not None: to_dict[e].append((i, fname, code))
            else:
                with open(fname_out, 'a', encoding='utf8') as f: f.write(code)
        if f'{e}.sc' not in mod.modules: mod.modules.append(f'{e}.sc')
    save_nbdev_module(mod)

    if not silent: print(f"Converted {fname.name}.")
    return to_dict

# Cell
_re_baseurl = re.compile('^baseurl\s*:.*$', re.MULTILINE)

# Cell
def update_baseurl():
    "Add or update `baseurl` in `_config.yml` for the docs"
    fname = Config().path("doc_path")/'_config.yml'
    if not fname.exists(): return
    with open(fname, 'r') as f: code = f.read()
    if _re_baseurl.search(code) is None: code = code + f"\nbaseurl: {Config().doc_baseurl}"
    else: code = _re_baseurl.sub(f"baseurl: {Config().doc_baseurl}", code)
    with open(fname, 'w') as f: f.write(code)

# Cell
#hide
def _get_paths(pth:str, names:list): return [Path(pth)/n for n in names if '/.' not in pth]

# Cell
def nbglob(fname=None, recursive=False) -> L:
    "Find all notebooks in a directory given a glob. Ignores hidden directories and filenames starting with `_`"
    fname = Path(fname) if fname else Config().path("nbs_path")
#     print(f'fname: {fname}')
    if recursive: fls = L(os.walk(fname)).map(lambda x: _get_paths(x[0], x[2])).concat()
    else: fls = fname.glob('*.ipynb')
#     for i, f in enumerate(fls):
#         print(f'file {i}: {f}')
    return L([f for f in fls if not f.name.startswith('_') and f.name.endswith('.ipynb')])

# Cell
def scala_notebook2script(fname=None, silent=False, to_dict=False, bare=False, recursive=False):
    "Convert notebooks matching `fname` to modules"
    # initial checks
    if os.environ.get('IN_TEST',0): return  # don't export if running tests
    if fname is None:
        reset_nbdev_module()
#         update_version()
        update_baseurl()
    files = nbglob(fname=fname, recursive=recursive)
    d = collections.defaultdict(list) if to_dict else None
    modules = create_mod_files(files, to_dict, bare=bare)
    for f in sorted(files): d = _notebook2script(f, modules, silent=silent, to_dict=d, bare=bare)
    return d