"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudExecutable = void 0;
const toolkit_lib_1 = require("@aws-cdk/toolkit-lib");
const cloud_assembly_1 = require("./cloud-assembly");
const api_private_1 = require("../../lib/api-private");
const singleton_plugin_host_1 = require("../cli/singleton-plugin-host");
const contextproviders = require("../context-providers");
/**
 * Represent the Cloud Executable and the synthesis we can do on it
 */
class CloudExecutable {
    constructor(props) {
        this.props = props;
    }
    async produce() {
        const synthesisResult = await this.synthesize(true);
        // We must return an `IReadableCloudAssembly` here, but this Cloud Assembly is only used in the context
        // of the CLI and `cli.ts` currently manages its own locking in the "synthesizer" callback function.
        //
        // All the lock-related functions are therefore no-ops.
        return new api_private_1.BorrowedAssembly(synthesisResult.assembly);
    }
    /**
     * Return whether there is an app command from the configuration
     */
    get hasApp() {
        return !!this.props.configuration.settings.get(['app']);
    }
    /**
     * Synthesize a set of stacks.
     *
     * @param cacheCloudAssembly whether to cache the Cloud Assembly after it has been first synthesized.
     *   This is 'true' by default, and only set to 'false' for 'cdk watch',
     *   which needs to re-synthesize the Assembly each time it detects a change to the project files
     */
    async synthesize(cacheCloudAssembly = true) {
        if (!this._cloudAssembly || !cacheCloudAssembly) {
            this._cloudAssembly = await this.doSynthesize();
        }
        return this._cloudAssembly;
    }
    async doSynthesize() {
        // We may need to run the cloud executable multiple times in order to satisfy all missing context
        // (When the executable runs, it will tell us about context it wants to use
        // but it missing. We'll then look up the context and run the executable again, and
        // again, until it doesn't complain anymore or we've stopped making progress).
        let previouslyMissingKeys;
        while (true) {
            const assembly = await this.props.synthesizer(this.props.sdkProvider, this.props.configuration);
            if (assembly.manifest.missing && assembly.manifest.missing.length > 0) {
                const missingKeys = missingContextKeys(assembly.manifest.missing);
                if (!this.canLookup) {
                    throw new toolkit_lib_1.ToolkitError('Context lookups have been disabled. '
                        + 'Make sure all necessary context is already in \'cdk.context.json\' by running \'cdk synth\' on a machine with sufficient AWS credentials and committing the result. '
                        + `Missing context keys: '${Array.from(missingKeys).join(', ')}'`);
                }
                let tryLookup = true;
                if (previouslyMissingKeys && setsEqual(missingKeys, previouslyMissingKeys)) {
                    await this.props.ioHelper.defaults.debug('Not making progress trying to resolve environmental context. Giving up.');
                    tryLookup = false;
                }
                previouslyMissingKeys = missingKeys;
                if (tryLookup) {
                    await this.props.ioHelper.defaults.debug('Some context information is missing. Fetching...');
                    const updates = await contextproviders.provideContextValues(assembly.manifest.missing, this.props.sdkProvider, singleton_plugin_host_1.GLOBAL_PLUGIN_HOST, this.props.ioHelper);
                    for (const [key, value] of Object.entries(updates)) {
                        this.props.configuration.context.set(key, value);
                    }
                    // Cache the new context to disk
                    await this.props.configuration.saveContext();
                    // Execute again
                    continue;
                }
            }
            return new cloud_assembly_1.CloudAssembly(assembly, this.props.ioHelper);
        }
    }
    get canLookup() {
        return !!(this.props.configuration.settings.get(['lookups']) ?? true);
    }
}
exports.CloudExecutable = CloudExecutable;
/**
 * Return all keys of missing context items
 */
function missingContextKeys(missing) {
    return new Set((missing || []).map(m => m.key));
}
function setsEqual(a, b) {
    if (a.size !== b.size) {
        return false;
    }
    for (const x of a) {
        if (!b.has(x)) {
            return false;
        }
    }
    return true;
}
//# sourceMappingURL=data:application/json;base64,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