"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdkToolkit = exports.AssetBuildTime = void 0;
exports.markTesting = markTesting;
const path = require("path");
const util_1 = require("util");
const cloud_assembly_schema_1 = require("@aws-cdk/cloud-assembly-schema");
const cxapi = require("@aws-cdk/cx-api");
const toolkit_lib_1 = require("@aws-cdk/toolkit-lib");
const chalk = require("chalk");
const chokidar = require("chokidar");
const fs = require("fs-extra");
const promptly = require("promptly");
const uuid = require("uuid");
const io_host_1 = require("./io-host");
const user_configuration_1 = require("./user-configuration");
const api_private_1 = require("../../lib/api-private");
const api_1 = require("../api");
const bootstrap_1 = require("../api/bootstrap");
const cloud_assembly_1 = require("../api/cloud-assembly");
const tags_1 = require("../api/tags");
const deploy_1 = require("../commands/deploy");
const list_stacks_1 = require("../commands/list-stacks");
const migrate_1 = require("../commands/migrate");
const cxapp_1 = require("../cxapp");
const logging_1 = require("../logging");
const util_2 = require("../util");
// Must use a require() otherwise esbuild complains about calling a namespace
// eslint-disable-next-line @typescript-eslint/no-require-imports,@typescript-eslint/consistent-type-imports
const pLimit = require('p-limit');
let TESTING = false;
function markTesting() {
    TESTING = true;
}
/**
 * When to build assets
 */
var AssetBuildTime;
(function (AssetBuildTime) {
    /**
     * Build all assets before deploying the first stack
     *
     * This is intended for expensive Docker image builds; so that if the Docker image build
     * fails, no stacks are unnecessarily deployed (with the attendant wait time).
     */
    AssetBuildTime["ALL_BEFORE_DEPLOY"] = "all-before-deploy";
    /**
     * Build assets just-in-time, before publishing
     */
    AssetBuildTime["JUST_IN_TIME"] = "just-in-time";
})(AssetBuildTime || (exports.AssetBuildTime = AssetBuildTime = {}));
/**
 * Custom implementation of the public Toolkit to integrate with the legacy CdkToolkit
 *
 * This overwrites how an sdkProvider is acquired
 * in favor of the one provided directly to CdkToolkit.
 */
class InternalToolkit extends toolkit_lib_1.Toolkit {
    constructor(sdkProvider, options) {
        super(options);
        this._sdkProvider = sdkProvider;
    }
    /**
     * Access to the AWS SDK
     * @internal
     */
    async sdkProvider(_action) {
        return this._sdkProvider;
    }
}
/**
 * Toolkit logic
 *
 * The toolkit runs the `cloudExecutable` to obtain a cloud assembly and
 * deploys applies them to `cloudFormation`.
 */
class CdkToolkit {
    constructor(props) {
        this.props = props;
        this.ioHost = props.ioHost ?? io_host_1.CliIoHost.instance();
        this.toolkitStackName = props.toolkitStackName ?? api_1.DEFAULT_TOOLKIT_STACK_NAME;
        this.toolkit = new InternalToolkit(props.sdkProvider, {
            assemblyFailureAt: this.validateMetadataFailAt(),
            color: true,
            emojis: true,
            ioHost: this.ioHost,
            toolkitStackName: this.toolkitStackName,
        });
    }
    async metadata(stackName, json) {
        const stacks = await this.selectSingleStackByName(stackName);
        printSerializedObject(stacks.firstStack.manifest.metadata ?? {}, json);
    }
    async acknowledge(noticeId) {
        const acks = this.props.configuration.context.get('acknowledged-issue-numbers') ?? [];
        acks.push(Number(noticeId));
        this.props.configuration.context.set('acknowledged-issue-numbers', acks);
        await this.props.configuration.saveContext();
    }
    async diff(options) {
        const stacks = await this.selectStacksForDiff(options.stackNames, options.exclusively);
        const strict = !!options.strict;
        const contextLines = options.contextLines || 3;
        const quiet = options.quiet || false;
        let diffs = 0;
        const parameterMap = buildParameterMap(options.parameters);
        if (options.templatePath !== undefined) {
            // Compare single stack against fixed template
            if (stacks.stackCount !== 1) {
                throw new toolkit_lib_1.ToolkitError('Can only select one stack when comparing to fixed template. Use --exclusively to avoid selecting multiple stacks.');
            }
            if (!(await fs.pathExists(options.templatePath))) {
                throw new toolkit_lib_1.ToolkitError(`There is no file at ${options.templatePath}`);
            }
            if (options.importExistingResources) {
                throw new toolkit_lib_1.ToolkitError('Can only use --import-existing-resources flag when comparing against deployed stacks.');
            }
            const template = (0, util_2.deserializeStructure)(await fs.readFile(options.templatePath, { encoding: 'UTF-8' }));
            const formatter = new api_1.DiffFormatter({
                templateInfo: {
                    oldTemplate: template,
                    newTemplate: stacks.firstStack,
                },
            });
            if (options.securityOnly) {
                const securityDiff = formatter.formatSecurityDiff();
                // Warn, count, and display the diff only if the reported changes are broadening permissions
                if (securityDiff.permissionChangeType === toolkit_lib_1.PermissionChangeType.BROADENING) {
                    (0, logging_1.warning)('This deployment will make potentially sensitive changes according to your current security approval level.\nPlease confirm you intend to make the following modifications:\n');
                    (0, logging_1.info)(securityDiff.formattedDiff);
                    diffs += 1;
                }
            }
            else {
                const diff = formatter.formatStackDiff({
                    strict,
                    contextLines,
                    quiet,
                });
                diffs = diff.numStacksWithChanges;
                (0, logging_1.info)(diff.formattedDiff);
            }
        }
        else {
            // Compare N stacks against deployed templates
            for (const stack of stacks.stackArtifacts) {
                const templateWithNestedStacks = await this.props.deployments.readCurrentTemplateWithNestedStacks(stack, options.compareAgainstProcessedTemplate);
                const currentTemplate = templateWithNestedStacks.deployedRootTemplate;
                const nestedStacks = templateWithNestedStacks.nestedStacks;
                const migrator = new api_1.ResourceMigrator({
                    deployments: this.props.deployments,
                    ioHelper: (0, api_private_1.asIoHelper)(this.ioHost, 'diff'),
                });
                const resourcesToImport = await migrator.tryGetResources(await this.props.deployments.resolveEnvironment(stack));
                if (resourcesToImport) {
                    (0, api_1.removeNonImportResources)(stack);
                }
                let changeSet = undefined;
                if (options.changeSet) {
                    let stackExists = false;
                    try {
                        stackExists = await this.props.deployments.stackExists({
                            stack,
                            deployName: stack.stackName,
                            tryLookupRole: true,
                        });
                    }
                    catch (e) {
                        (0, logging_1.debug)((0, util_2.formatErrorMessage)(e));
                        if (!quiet) {
                            (0, logging_1.info)(`Checking if the stack ${stack.stackName} exists before creating the changeset has failed, will base the diff on template differences (run again with -v to see the reason)\n`);
                        }
                        stackExists = false;
                    }
                    if (stackExists) {
                        changeSet = await api_private_1.cfnApi.createDiffChangeSet((0, api_private_1.asIoHelper)(this.ioHost, 'diff'), {
                            stack,
                            uuid: uuid.v4(),
                            deployments: this.props.deployments,
                            willExecute: false,
                            sdkProvider: this.props.sdkProvider,
                            parameters: Object.assign({}, parameterMap['*'], parameterMap[stack.stackName]),
                            resourcesToImport,
                            importExistingResources: options.importExistingResources,
                        });
                    }
                    else {
                        (0, logging_1.debug)(`the stack '${stack.stackName}' has not been deployed to CloudFormation or describeStacks call failed, skipping changeset creation.`);
                    }
                }
                const formatter = new api_1.DiffFormatter({
                    templateInfo: {
                        oldTemplate: currentTemplate,
                        newTemplate: stack,
                        changeSet,
                        isImport: !!resourcesToImport,
                        nestedStacks,
                    },
                });
                if (options.securityOnly) {
                    const securityDiff = formatter.formatSecurityDiff();
                    // Warn, count, and display the diff only if the reported changes are broadening permissions
                    if (securityDiff.permissionChangeType === toolkit_lib_1.PermissionChangeType.BROADENING) {
                        (0, logging_1.warning)('This deployment will make potentially sensitive changes according to your current security approval level.\nPlease confirm you intend to make the following modifications:\n');
                        (0, logging_1.info)(securityDiff.formattedDiff);
                        diffs += 1;
                    }
                }
                else {
                    const diff = formatter.formatStackDiff({
                        strict,
                        contextLines,
                        quiet,
                    });
                    (0, logging_1.info)(diff.formattedDiff);
                    diffs += diff.numStacksWithChanges;
                }
            }
        }
        (0, logging_1.info)((0, util_1.format)('\n✨  Number of stacks with differences: %s\n', diffs));
        return diffs && options.fail ? 1 : 0;
    }
    async deploy(options) {
        if (options.watch) {
            return this.watch(options);
        }
        // set progress from options, this includes user and app config
        if (options.progress) {
            this.ioHost.stackProgress = options.progress;
        }
        const startSynthTime = new Date().getTime();
        const stackCollection = await this.selectStacksForDeploy(options.selector, options.exclusively, options.cacheCloudAssembly, options.ignoreNoStacks);
        const elapsedSynthTime = new Date().getTime() - startSynthTime;
        (0, logging_1.info)(`\n✨  Synthesis time: ${(0, util_2.formatTime)(elapsedSynthTime)}s\n`);
        if (stackCollection.stackCount === 0) {
            (0, logging_1.error)('This app contains no stacks');
            return;
        }
        const migrator = new api_1.ResourceMigrator({
            deployments: this.props.deployments,
            ioHelper: (0, api_private_1.asIoHelper)(this.ioHost, 'deploy'),
        });
        await migrator.tryMigrateResources(stackCollection, {
            toolkitStackName: this.toolkitStackName,
            ...options,
        });
        const requireApproval = options.requireApproval ?? cloud_assembly_schema_1.RequireApproval.BROADENING;
        const parameterMap = buildParameterMap(options.parameters);
        if (options.deploymentMethod?.method === 'hotswap') {
            (0, logging_1.warning)('⚠️ The --hotswap and --hotswap-fallback flags deliberately introduce CloudFormation drift to speed up deployments');
            (0, logging_1.warning)('⚠️ They should only be used for development - never use them for your production Stacks!\n');
        }
        const stacks = stackCollection.stackArtifacts;
        const stackOutputs = {};
        const outputsFile = options.outputsFile;
        const buildAsset = async (assetNode) => {
            await this.props.deployments.buildSingleAsset(assetNode.assetManifestArtifact, assetNode.assetManifest, assetNode.asset, {
                stack: assetNode.parentStack,
                roleArn: options.roleArn,
                stackName: assetNode.parentStack.stackName,
            });
        };
        const publishAsset = async (assetNode) => {
            await this.props.deployments.publishSingleAsset(assetNode.assetManifest, assetNode.asset, {
                stack: assetNode.parentStack,
                roleArn: options.roleArn,
                stackName: assetNode.parentStack.stackName,
                forcePublish: options.force,
            });
        };
        const deployStack = async (stackNode) => {
            const stack = stackNode.stack;
            if (stackCollection.stackCount !== 1) {
                (0, logging_1.highlight)(stack.displayName);
            }
            if (!stack.environment) {
                // eslint-disable-next-line @stylistic/max-len
                throw new toolkit_lib_1.ToolkitError(`Stack ${stack.displayName} does not define an environment, and AWS credentials could not be obtained from standard locations or no region was configured.`);
            }
            if (Object.keys(stack.template.Resources || {}).length === 0) {
                // The generated stack has no resources
                if (!(await this.props.deployments.stackExists({ stack }))) {
                    (0, logging_1.warning)('%s: stack has no resources, skipping deployment.', chalk.bold(stack.displayName));
                }
                else {
                    (0, logging_1.warning)('%s: stack has no resources, deleting existing stack.', chalk.bold(stack.displayName));
                    await this.destroy({
                        selector: { patterns: [stack.hierarchicalId] },
                        exclusively: true,
                        force: true,
                        roleArn: options.roleArn,
                        fromDeploy: true,
                    });
                }
                return;
            }
            if (requireApproval !== cloud_assembly_schema_1.RequireApproval.NEVER) {
                const currentTemplate = await this.props.deployments.readCurrentTemplate(stack);
                const formatter = new api_1.DiffFormatter({
                    templateInfo: {
                        oldTemplate: currentTemplate,
                        newTemplate: stack,
                    },
                });
                const securityDiff = formatter.formatSecurityDiff();
                if (requiresApproval(requireApproval, securityDiff.permissionChangeType)) {
                    (0, logging_1.info)(securityDiff.formattedDiff);
                    await askUserConfirmation(this.ioHost, concurrency, '"--require-approval" is enabled and stack includes security-sensitive updates', 'Do you wish to deploy these changes');
                }
            }
            // Following are the same semantics we apply with respect to Notification ARNs (dictated by the SDK)
            //
            //  - undefined  =>  cdk ignores it, as if it wasn't supported (allows external management).
            //  - []:        =>  cdk manages it, and the user wants to wipe it out.
            //  - ['arn-1']  =>  cdk manages it, and the user wants to set it to ['arn-1'].
            const notificationArns = (!!options.notificationArns || !!stack.notificationArns)
                ? (options.notificationArns ?? []).concat(stack.notificationArns ?? [])
                : undefined;
            for (const notificationArn of notificationArns ?? []) {
                if (!(0, util_2.validateSnsTopicArn)(notificationArn)) {
                    throw new toolkit_lib_1.ToolkitError(`Notification arn ${notificationArn} is not a valid arn for an SNS topic`);
                }
            }
            const stackIndex = stacks.indexOf(stack) + 1;
            (0, logging_1.info)(`${chalk.bold(stack.displayName)}: deploying... [${stackIndex}/${stackCollection.stackCount}]`);
            const startDeployTime = new Date().getTime();
            let tags = options.tags;
            if (!tags || tags.length === 0) {
                tags = (0, tags_1.tagsForStack)(stack);
            }
            let elapsedDeployTime = 0;
            try {
                let deployResult;
                let rollback = options.rollback;
                let iteration = 0;
                while (!deployResult) {
                    if (++iteration > 2) {
                        throw new toolkit_lib_1.ToolkitError('This loop should have stabilized in 2 iterations, but didn\'t. If you are seeing this error, please report it at https://github.com/aws/aws-cdk/issues/new/choose');
                    }
                    const r = await this.props.deployments.deployStack({
                        stack,
                        deployName: stack.stackName,
                        roleArn: options.roleArn,
                        toolkitStackName: options.toolkitStackName,
                        reuseAssets: options.reuseAssets,
                        notificationArns,
                        tags,
                        execute: options.execute,
                        changeSetName: options.changeSetName,
                        deploymentMethod: options.deploymentMethod,
                        forceDeployment: options.force,
                        parameters: Object.assign({}, parameterMap['*'], parameterMap[stack.stackName]),
                        usePreviousParameters: options.usePreviousParameters,
                        rollback,
                        extraUserAgent: options.extraUserAgent,
                        assetParallelism: options.assetParallelism,
                        ignoreNoStacks: options.ignoreNoStacks,
                    });
                    switch (r.type) {
                        case 'did-deploy-stack':
                            deployResult = r;
                            break;
                        case 'failpaused-need-rollback-first': {
                            const motivation = r.reason === 'replacement'
                                ? `Stack is in a paused fail state (${r.status}) and change includes a replacement which cannot be deployed with "--no-rollback"`
                                : `Stack is in a paused fail state (${r.status}) and command line arguments do not include "--no-rollback"`;
                            if (options.force) {
                                (0, logging_1.warning)(`${motivation}. Rolling back first (--force).`);
                            }
                            else {
                                await askUserConfirmation(this.ioHost, concurrency, motivation, `${motivation}. Roll back first and then proceed with deployment`);
                            }
                            // Perform a rollback
                            await this.rollback({
                                selector: { patterns: [stack.hierarchicalId] },
                                toolkitStackName: options.toolkitStackName,
                                force: options.force,
                            });
                            // Go around through the 'while' loop again but switch rollback to true.
                            rollback = true;
                            break;
                        }
                        case 'replacement-requires-rollback': {
                            const motivation = 'Change includes a replacement which cannot be deployed with "--no-rollback"';
                            if (options.force) {
                                (0, logging_1.warning)(`${motivation}. Proceeding with regular deployment (--force).`);
                            }
                            else {
                                await askUserConfirmation(this.ioHost, concurrency, motivation, `${motivation}. Perform a regular deployment`);
                            }
                            // Go around through the 'while' loop again but switch rollback to true.
                            rollback = true;
                            break;
                        }
                        default:
                            throw new toolkit_lib_1.ToolkitError(`Unexpected result type from deployStack: ${JSON.stringify(r)}. If you are seeing this error, please report it at https://github.com/aws/aws-cdk/issues/new/choose`);
                    }
                }
                const message = deployResult.noOp
                    ? ' ✅  %s (no changes)'
                    : ' ✅  %s';
                (0, logging_1.success)('\n' + message, stack.displayName);
                elapsedDeployTime = new Date().getTime() - startDeployTime;
                (0, logging_1.info)(`\n✨  Deployment time: ${(0, util_2.formatTime)(elapsedDeployTime)}s\n`);
                if (Object.keys(deployResult.outputs).length > 0) {
                    (0, logging_1.info)('Outputs:');
                    stackOutputs[stack.stackName] = deployResult.outputs;
                }
                for (const name of Object.keys(deployResult.outputs).sort()) {
                    const value = deployResult.outputs[name];
                    (0, logging_1.info)(`${chalk.cyan(stack.id)}.${chalk.cyan(name)} = ${chalk.underline(chalk.cyan(value))}`);
                }
                (0, logging_1.info)('Stack ARN:');
                (0, logging_1.result)(deployResult.stackArn);
            }
            catch (e) {
                // It has to be exactly this string because an integration test tests for
                // "bold(stackname) failed: ResourceNotReady: <error>"
                throw new toolkit_lib_1.ToolkitError([`❌  ${chalk.bold(stack.stackName)} failed:`, ...(e.name ? [`${e.name}:`] : []), (0, util_2.formatErrorMessage)(e)].join(' '));
            }
            finally {
                if (options.cloudWatchLogMonitor) {
                    const foundLogGroupsResult = await (0, api_1.findCloudWatchLogGroups)(this.props.sdkProvider, (0, api_private_1.asIoHelper)(this.ioHost, 'deploy'), stack);
                    options.cloudWatchLogMonitor.addLogGroups(foundLogGroupsResult.env, foundLogGroupsResult.sdk, foundLogGroupsResult.logGroupNames);
                }
                // If an outputs file has been specified, create the file path and write stack outputs to it once.
                // Outputs are written after all stacks have been deployed. If a stack deployment fails,
                // all of the outputs from successfully deployed stacks before the failure will still be written.
                if (outputsFile) {
                    fs.ensureFileSync(outputsFile);
                    await fs.writeJson(outputsFile, stackOutputs, {
                        spaces: 2,
                        encoding: 'utf8',
                    });
                }
            }
            (0, logging_1.info)(`\n✨  Total time: ${(0, util_2.formatTime)(elapsedSynthTime + elapsedDeployTime)}s\n`);
        };
        const assetBuildTime = options.assetBuildTime ?? AssetBuildTime.ALL_BEFORE_DEPLOY;
        const prebuildAssets = assetBuildTime === AssetBuildTime.ALL_BEFORE_DEPLOY;
        const concurrency = options.concurrency || 1;
        if (concurrency > 1) {
            // always force "events" progress output when we have concurrency
            this.ioHost.stackProgress = deploy_1.StackActivityProgress.EVENTS;
            // ...but only warn if the user explicitly requested "bar" progress
            if (options.progress && options.progress != deploy_1.StackActivityProgress.EVENTS) {
                (0, logging_1.warning)('⚠️ The --concurrency flag only supports --progress "events". Switching to "events".');
            }
        }
        const stacksAndTheirAssetManifests = stacks.flatMap((stack) => [
            stack,
            ...stack.dependencies.filter(x => cxapi.AssetManifestArtifact.isAssetManifestArtifact(x)),
        ]);
        const workGraph = new api_1.WorkGraphBuilder((0, api_private_1.asIoHelper)(this.ioHost, 'deploy'), prebuildAssets).build(stacksAndTheirAssetManifests);
        // Unless we are running with '--force', skip already published assets
        if (!options.force) {
            await this.removePublishedAssets(workGraph, options);
        }
        const graphConcurrency = {
            'stack': concurrency,
            'asset-build': 1, // This will be CPU-bound/memory bound, mostly matters for Docker builds
            'asset-publish': (options.assetParallelism ?? true) ? 8 : 1, // This will be I/O-bound, 8 in parallel seems reasonable
        };
        await workGraph.doParallel(graphConcurrency, {
            deployStack,
            buildAsset,
            publishAsset,
        });
    }
    /**
     * Detect infrastructure drift for the given stack(s)
     */
    async drift(options) {
        const driftResults = await this.toolkit.drift(this.props.cloudExecutable, {
            stacks: {
                patterns: options.selector.patterns,
                strategy: options.selector.patterns.length > 0 ? toolkit_lib_1.StackSelectionStrategy.PATTERN_MATCH : toolkit_lib_1.StackSelectionStrategy.ALL_STACKS,
            },
        });
        const totalDrifts = Object.values(driftResults).reduce((total, current) => total + (current.numResourcesWithDrift ?? 0), 0);
        return totalDrifts > 0 && options.fail ? 1 : 0;
    }
    /**
     * Roll back the given stack or stacks.
     */
    async rollback(options) {
        const startSynthTime = new Date().getTime();
        const stackCollection = await this.selectStacksForDeploy(options.selector, true);
        const elapsedSynthTime = new Date().getTime() - startSynthTime;
        (0, logging_1.info)(`\n✨  Synthesis time: ${(0, util_2.formatTime)(elapsedSynthTime)}s\n`);
        if (stackCollection.stackCount === 0) {
            (0, logging_1.error)('No stacks selected');
            return;
        }
        let anyRollbackable = false;
        for (const stack of stackCollection.stackArtifacts) {
            (0, logging_1.info)('Rolling back %s', chalk.bold(stack.displayName));
            const startRollbackTime = new Date().getTime();
            try {
                const result = await this.props.deployments.rollbackStack({
                    stack,
                    roleArn: options.roleArn,
                    toolkitStackName: options.toolkitStackName,
                    orphanFailedResources: options.force,
                    validateBootstrapStackVersion: options.validateBootstrapStackVersion,
                    orphanLogicalIds: options.orphanLogicalIds,
                });
                if (!result.notInRollbackableState) {
                    anyRollbackable = true;
                }
                const elapsedRollbackTime = new Date().getTime() - startRollbackTime;
                (0, logging_1.info)(`\n✨  Rollback time: ${(0, util_2.formatTime)(elapsedRollbackTime).toString()}s\n`);
            }
            catch (e) {
                (0, logging_1.error)('\n ❌  %s failed: %s', chalk.bold(stack.displayName), (0, util_2.formatErrorMessage)(e));
                throw new toolkit_lib_1.ToolkitError('Rollback failed (use --force to orphan failing resources)');
            }
        }
        if (!anyRollbackable) {
            throw new toolkit_lib_1.ToolkitError('No stacks were in a state that could be rolled back');
        }
    }
    async watch(options) {
        const rootDir = path.dirname(path.resolve(user_configuration_1.PROJECT_CONFIG));
        const ioHelper = (0, api_private_1.asIoHelper)(this.ioHost, 'watch');
        (0, logging_1.debug)("root directory used for 'watch' is: %s", rootDir);
        const watchSettings = this.props.configuration.settings.get(['watch']);
        if (!watchSettings) {
            throw new toolkit_lib_1.ToolkitError("Cannot use the 'watch' command without specifying at least one directory to monitor. " +
                'Make sure to add a "watch" key to your cdk.json');
        }
        // For the "include" subkey under the "watch" key, the behavior is:
        // 1. No "watch" setting? We error out.
        // 2. "watch" setting without an "include" key? We default to observing "./**".
        // 3. "watch" setting with an empty "include" key? We default to observing "./**".
        // 4. Non-empty "include" key? Just use the "include" key.
        const watchIncludes = this.patternsArrayForWatch(watchSettings.include, {
            rootDir,
            returnRootDirIfEmpty: true,
        });
        (0, logging_1.debug)("'include' patterns for 'watch': %s", watchIncludes);
        // For the "exclude" subkey under the "watch" key,
        // the behavior is to add some default excludes in addition to the ones specified by the user:
        // 1. The CDK output directory.
        // 2. Any file whose name starts with a dot.
        // 3. Any directory's content whose name starts with a dot.
        // 4. Any node_modules and its content (even if it's not a JS/TS project, you might be using a local aws-cli package)
        const outputDir = this.props.configuration.settings.get(['output']);
        const watchExcludes = this.patternsArrayForWatch(watchSettings.exclude, {
            rootDir,
            returnRootDirIfEmpty: false,
        }).concat(`${outputDir}/**`, '**/.*', '**/.*/**', '**/node_modules/**');
        (0, logging_1.debug)("'exclude' patterns for 'watch': %s", watchExcludes);
        // Since 'cdk deploy' is a relatively slow operation for a 'watch' process,
        // introduce a concurrency latch that tracks the state.
        // This way, if file change events arrive when a 'cdk deploy' is still executing,
        // we will batch them, and trigger another 'cdk deploy' after the current one finishes,
        // making sure 'cdk deploy's  always execute one at a time.
        // Here's a diagram showing the state transitions:
        // --------------                --------    file changed     --------------    file changed     --------------  file changed
        // |            |  ready event   |      | ------------------> |            | ------------------> |            | --------------|
        // | pre-ready  | -------------> | open |                     | deploying  |                     |   queued   |               |
        // |            |                |      | <------------------ |            | <------------------ |            | <-------------|
        // --------------                --------  'cdk deploy' done  --------------  'cdk deploy' done  --------------
        let latch = 'pre-ready';
        const cloudWatchLogMonitor = options.traceLogs ? new api_1.CloudWatchLogEventMonitor({
            ioHelper,
        }) : undefined;
        const deployAndWatch = async () => {
            latch = 'deploying';
            await cloudWatchLogMonitor?.deactivate();
            await this.invokeDeployFromWatch(options, cloudWatchLogMonitor);
            // If latch is still 'deploying' after the 'await', that's fine,
            // but if it's 'queued', that means we need to deploy again
            while (latch === 'queued') {
                // TypeScript doesn't realize latch can change between 'awaits',
                // and thinks the above 'while' condition is always 'false' without the cast
                latch = 'deploying';
                (0, logging_1.info)("Detected file changes during deployment. Invoking 'cdk deploy' again");
                await this.invokeDeployFromWatch(options, cloudWatchLogMonitor);
            }
            latch = 'open';
            await cloudWatchLogMonitor?.activate();
        };
        chokidar
            .watch(watchIncludes, {
            ignored: watchExcludes,
            cwd: rootDir,
        })
            .on('ready', async () => {
            latch = 'open';
            (0, logging_1.debug)("'watch' received the 'ready' event. From now on, all file changes will trigger a deployment");
            (0, logging_1.info)("Triggering initial 'cdk deploy'");
            await deployAndWatch();
        })
            .on('all', async (event, filePath) => {
            if (latch === 'pre-ready') {
                (0, logging_1.info)(`'watch' is observing ${event === 'addDir' ? 'directory' : 'the file'} '%s' for changes`, filePath);
            }
            else if (latch === 'open') {
                (0, logging_1.info)("Detected change to '%s' (type: %s). Triggering 'cdk deploy'", filePath, event);
                await deployAndWatch();
            }
            else {
                // this means latch is either 'deploying' or 'queued'
                latch = 'queued';
                (0, logging_1.info)("Detected change to '%s' (type: %s) while 'cdk deploy' is still running. " +
                    'Will queue for another deployment after this one finishes', filePath, event);
            }
        });
    }
    async import(options) {
        const stacks = await this.selectStacksForDeploy(options.selector, true, true, false);
        // set progress from options, this includes user and app config
        if (options.progress) {
            this.ioHost.stackProgress = options.progress;
        }
        if (stacks.stackCount > 1) {
            throw new toolkit_lib_1.ToolkitError(`Stack selection is ambiguous, please choose a specific stack for import [${stacks.stackArtifacts.map((x) => x.id).join(', ')}]`);
        }
        if (!process.stdout.isTTY && !options.resourceMappingFile) {
            throw new toolkit_lib_1.ToolkitError('--resource-mapping is required when input is not a terminal');
        }
        const stack = stacks.stackArtifacts[0];
        (0, logging_1.highlight)(stack.displayName);
        const resourceImporter = new api_1.ResourceImporter(stack, {
            deployments: this.props.deployments,
            ioHelper: (0, api_private_1.asIoHelper)(this.ioHost, 'import'),
        });
        const { additions, hasNonAdditions } = await resourceImporter.discoverImportableResources(options.force);
        if (additions.length === 0) {
            (0, logging_1.warning)('%s: no new resources compared to the currently deployed stack, skipping import.', chalk.bold(stack.displayName));
            return;
        }
        // Prepare a mapping of physical resources to CDK constructs
        const actualImport = !options.resourceMappingFile
            ? await resourceImporter.askForResourceIdentifiers(additions)
            : await resourceImporter.loadResourceIdentifiers(additions, options.resourceMappingFile);
        if (actualImport.importResources.length === 0) {
            (0, logging_1.warning)('No resources selected for import.');
            return;
        }
        // If "--create-resource-mapping" option was passed, write the resource mapping to the given file and exit
        if (options.recordResourceMapping) {
            const outputFile = options.recordResourceMapping;
            fs.ensureFileSync(outputFile);
            await fs.writeJson(outputFile, actualImport.resourceMap, {
                spaces: 2,
                encoding: 'utf8',
            });
            (0, logging_1.info)('%s: mapping file written.', outputFile);
            return;
        }
        // Import the resources according to the given mapping
        (0, logging_1.info)('%s: importing resources into stack...', chalk.bold(stack.displayName));
        const tags = (0, tags_1.tagsForStack)(stack);
        await resourceImporter.importResourcesFromMap(actualImport, {
            roleArn: options.roleArn,
            tags,
            deploymentMethod: options.deploymentMethod,
            usePreviousParameters: true,
            rollback: options.rollback,
        });
        // Notify user of next steps
        (0, logging_1.info)(`Import operation complete. We recommend you run a ${chalk.blueBright('drift detection')} operation ` +
            'to confirm your CDK app resource definitions are up-to-date. Read more here: ' +
            chalk.underline.blueBright('https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/detect-drift-stack.html'));
        if (actualImport.importResources.length < additions.length) {
            (0, logging_1.info)('');
            (0, logging_1.warning)(`Some resources were skipped. Run another ${chalk.blueBright('cdk import')} or a ${chalk.blueBright('cdk deploy')} to bring the stack up-to-date with your CDK app definition.`);
        }
        else if (hasNonAdditions) {
            (0, logging_1.info)('');
            (0, logging_1.warning)(`Your app has pending updates or deletes excluded from this import operation. Run a ${chalk.blueBright('cdk deploy')} to bring the stack up-to-date with your CDK app definition.`);
        }
    }
    async destroy(options) {
        let stacks = await this.selectStacksForDestroy(options.selector, options.exclusively);
        // The stacks will have been ordered for deployment, so reverse them for deletion.
        stacks = stacks.reversed();
        if (!options.force) {
            // eslint-disable-next-line @stylistic/max-len
            const confirmed = await promptly.confirm(`Are you sure you want to delete: ${chalk.blue(stacks.stackArtifacts.map((s) => s.hierarchicalId).join(', '))} (y/n)?`);
            if (!confirmed) {
                return;
            }
        }
        const action = options.fromDeploy ? 'deploy' : 'destroy';
        for (const [index, stack] of stacks.stackArtifacts.entries()) {
            (0, logging_1.success)('%s: destroying... [%s/%s]', chalk.blue(stack.displayName), index + 1, stacks.stackCount);
            try {
                await this.props.deployments.destroyStack({
                    stack,
                    deployName: stack.stackName,
                    roleArn: options.roleArn,
                });
                (0, logging_1.success)(`\n ✅  %s: ${action}ed`, chalk.blue(stack.displayName));
            }
            catch (e) {
                (0, logging_1.error)(`\n ❌  %s: ${action} failed`, chalk.blue(stack.displayName), e);
                throw e;
            }
        }
    }
    async list(selectors, options = {}) {
        const stacks = await (0, list_stacks_1.listStacks)(this, {
            selectors: selectors,
        });
        if (options.long && options.showDeps) {
            printSerializedObject(stacks, options.json ?? false);
            return 0;
        }
        if (options.showDeps) {
            const stackDeps = [];
            for (const stack of stacks) {
                stackDeps.push({
                    id: stack.id,
                    dependencies: stack.dependencies,
                });
            }
            printSerializedObject(stackDeps, options.json ?? false);
            return 0;
        }
        if (options.long) {
            const long = [];
            for (const stack of stacks) {
                long.push({
                    id: stack.id,
                    name: stack.name,
                    environment: stack.environment,
                });
            }
            printSerializedObject(long, options.json ?? false);
            return 0;
        }
        // just print stack IDs
        for (const stack of stacks) {
            (0, logging_1.result)(stack.id);
        }
        return 0; // exit-code
    }
    /**
     * Synthesize the given set of stacks (called when the user runs 'cdk synth')
     *
     * INPUT: Stack names can be supplied using a glob filter. If no stacks are
     * given, all stacks from the application are implicitly selected.
     *
     * OUTPUT: If more than one stack ends up being selected, an output directory
     * should be supplied, where the templates will be written.
     */
    async synth(stackNames, exclusively, quiet, autoValidate, json) {
        const stacks = await this.selectStacksForDiff(stackNames, exclusively, autoValidate);
        // if we have a single stack, print it to STDOUT
        if (stacks.stackCount === 1) {
            if (!quiet) {
                printSerializedObject((0, util_2.obscureTemplate)(stacks.firstStack.template), json ?? false);
            }
            return undefined;
        }
        // not outputting template to stdout, let's explain things to the user a little bit...
        (0, logging_1.success)(`Successfully synthesized to ${chalk.blue(path.resolve(stacks.assembly.directory))}`);
        (0, logging_1.info)(`Supply a stack id (${stacks.stackArtifacts.map((s) => chalk.green(s.hierarchicalId)).join(', ')}) to display its template.`);
        return undefined;
    }
    /**
     * Bootstrap the CDK Toolkit stack in the accounts used by the specified stack(s).
     *
     * @param userEnvironmentSpecs environment names that need to have toolkit support
     *             provisioned, as a glob filter. If none is provided, all stacks are implicitly selected.
     * @param options The name, role ARN, bootstrapping parameters, etc. to be used for the CDK Toolkit stack.
     */
    async bootstrap(userEnvironmentSpecs, options) {
        const bootstrapper = new bootstrap_1.Bootstrapper(options.source, (0, api_private_1.asIoHelper)(this.ioHost, 'bootstrap'));
        // If there is an '--app' argument and an environment looks like a glob, we
        // select the environments from the app. Otherwise, use what the user said.
        const environments = await this.defineEnvironments(userEnvironmentSpecs);
        const limit = pLimit(20);
        // eslint-disable-next-line @cdklabs/promiseall-no-unbounded-parallelism
        await Promise.all(environments.map((environment) => limit(async () => {
            (0, logging_1.success)(' ⏳  Bootstrapping environment %s...', chalk.blue(environment.name));
            try {
                const result = await bootstrapper.bootstrapEnvironment(environment, this.props.sdkProvider, options);
                const message = result.noOp
                    ? ' ✅  Environment %s bootstrapped (no changes).'
                    : ' ✅  Environment %s bootstrapped.';
                (0, logging_1.success)(message, chalk.blue(environment.name));
            }
            catch (e) {
                (0, logging_1.error)(' ❌  Environment %s failed bootstrapping: %s', chalk.blue(environment.name), e);
                throw e;
            }
        })));
    }
    /**
     * Garbage collects assets from a CDK app's environment
     * @param options Options for Garbage Collection
     */
    async garbageCollect(userEnvironmentSpecs, options) {
        const environments = await this.defineEnvironments(userEnvironmentSpecs);
        for (const environment of environments) {
            (0, logging_1.success)(' ⏳  Garbage Collecting environment %s...', chalk.blue(environment.name));
            const gc = new api_1.GarbageCollector({
                sdkProvider: this.props.sdkProvider,
                ioHelper: (0, api_private_1.asIoHelper)(this.ioHost, 'gc'),
                resolvedEnvironment: environment,
                bootstrapStackName: options.bootstrapStackName,
                rollbackBufferDays: options.rollbackBufferDays,
                createdBufferDays: options.createdBufferDays,
                action: options.action ?? 'full',
                type: options.type ?? 'all',
                confirm: options.confirm ?? true,
            });
            await gc.garbageCollect();
        }
    }
    async defineEnvironments(userEnvironmentSpecs) {
        // By default, glob for everything
        const environmentSpecs = userEnvironmentSpecs.length > 0 ? [...userEnvironmentSpecs] : ['**'];
        // Partition into globs and non-globs (this will mutate environmentSpecs).
        const globSpecs = (0, util_2.partition)(environmentSpecs, cxapp_1.looksLikeGlob);
        if (globSpecs.length > 0 && !this.props.cloudExecutable.hasApp) {
            if (userEnvironmentSpecs.length > 0) {
                // User did request this glob
                throw new toolkit_lib_1.ToolkitError(`'${globSpecs}' is not an environment name. Specify an environment name like 'aws://123456789012/us-east-1', or run in a directory with 'cdk.json' to use wildcards.`);
            }
            else {
                // User did not request anything
                throw new toolkit_lib_1.ToolkitError("Specify an environment name like 'aws://123456789012/us-east-1', or run in a directory with 'cdk.json'.");
            }
        }
        const environments = [...(0, cxapp_1.environmentsFromDescriptors)(environmentSpecs)];
        // If there is an '--app' argument, select the environments from the app.
        if (this.props.cloudExecutable.hasApp) {
            environments.push(...(await (0, cxapp_1.globEnvironmentsFromStacks)(await this.selectStacksForList([]), globSpecs, this.props.sdkProvider)));
        }
        return environments;
    }
    /**
     * Migrates a CloudFormation stack/template to a CDK app
     * @param options Options for CDK app creation
     */
    async migrate(options) {
        (0, logging_1.warning)('This command is an experimental feature.');
        const language = options.language?.toLowerCase() ?? 'typescript';
        const environment = (0, migrate_1.setEnvironment)(options.account, options.region);
        let generateTemplateOutput;
        let cfn;
        let templateToDelete;
        try {
            // if neither fromPath nor fromStack is provided, generate a template using cloudformation
            const scanType = (0, migrate_1.parseSourceOptions)(options.fromPath, options.fromStack, options.stackName).source;
            if (scanType == migrate_1.TemplateSourceOptions.SCAN) {
                generateTemplateOutput = await (0, migrate_1.generateTemplate)({
                    stackName: options.stackName,
                    filters: options.filter,
                    fromScan: options.fromScan,
                    sdkProvider: this.props.sdkProvider,
                    environment: environment,
                });
                templateToDelete = generateTemplateOutput.templateId;
            }
            else if (scanType == migrate_1.TemplateSourceOptions.PATH) {
                const templateBody = (0, migrate_1.readFromPath)(options.fromPath);
                const parsedTemplate = (0, util_2.deserializeStructure)(templateBody);
                const templateId = parsedTemplate.Metadata?.TemplateId?.toString();
                if (templateId) {
                    // if we have a template id, we can call describe generated template to get the resource identifiers
                    // resource metadata, and template source to generate the template
                    cfn = new migrate_1.CfnTemplateGeneratorProvider(await (0, migrate_1.buildCfnClient)(this.props.sdkProvider, environment));
                    const generatedTemplateSummary = await cfn.describeGeneratedTemplate(templateId);
                    generateTemplateOutput = (0, migrate_1.buildGenertedTemplateOutput)(generatedTemplateSummary, templateBody, generatedTemplateSummary.GeneratedTemplateId);
                }
                else {
                    generateTemplateOutput = {
                        migrateJson: {
                            templateBody: templateBody,
                            source: 'localfile',
                        },
                    };
                }
            }
            else if (scanType == migrate_1.TemplateSourceOptions.STACK) {
                const template = await (0, migrate_1.readFromStack)(options.stackName, this.props.sdkProvider, environment);
                if (!template) {
                    throw new toolkit_lib_1.ToolkitError(`No template found for stack-name: ${options.stackName}`);
                }
                generateTemplateOutput = {
                    migrateJson: {
                        templateBody: template,
                        source: options.stackName,
                    },
                };
            }
            else {
                // We shouldn't ever get here, but just in case.
                throw new toolkit_lib_1.ToolkitError(`Invalid source option provided: ${scanType}`);
            }
            const stack = (0, migrate_1.generateStack)(generateTemplateOutput.migrateJson.templateBody, options.stackName, language);
            (0, logging_1.success)(' ⏳  Generating CDK app for %s...', chalk.blue(options.stackName));
            await (0, migrate_1.generateCdkApp)(options.stackName, stack, language, options.outputPath, options.compress);
            if (generateTemplateOutput) {
                (0, migrate_1.writeMigrateJsonFile)(options.outputPath, options.stackName, generateTemplateOutput.migrateJson);
            }
            if ((0, migrate_1.isThereAWarning)(generateTemplateOutput)) {
                (0, logging_1.warning)(' ⚠️  Some resources could not be migrated completely. Please review the README.md file for more information.');
                (0, migrate_1.appendWarningsToReadme)(`${path.join(options.outputPath ?? process.cwd(), options.stackName)}/README.md`, generateTemplateOutput.resources);
            }
        }
        catch (e) {
            (0, logging_1.error)(' ❌  Migrate failed for `%s`: %s', options.stackName, e.message);
            throw e;
        }
        finally {
            if (templateToDelete) {
                if (!cfn) {
                    cfn = new migrate_1.CfnTemplateGeneratorProvider(await (0, migrate_1.buildCfnClient)(this.props.sdkProvider, environment));
                }
                if (!process.env.MIGRATE_INTEG_TEST) {
                    await cfn.deleteGeneratedTemplate(templateToDelete);
                }
            }
        }
    }
    async refactor(options) {
        if (options.mappingFile && options.excludeFile) {
            throw new toolkit_lib_1.ToolkitError('Cannot use both --exclude-file and mapping-file.');
        }
        if (options.revert && !options.mappingFile) {
            throw new toolkit_lib_1.ToolkitError('The --revert option can only be used with the --mapping-file option.');
        }
        try {
            await this.toolkit.refactor(this.props.cloudExecutable, {
                dryRun: options.dryRun,
                stacks: {
                    patterns: options.selector.patterns,
                    strategy: options.selector.patterns.length > 0 ? toolkit_lib_1.StackSelectionStrategy.PATTERN_MATCH : toolkit_lib_1.StackSelectionStrategy.ALL_STACKS,
                },
                exclude: await readExcludeFile(options.excludeFile),
                mappings: await readMappingFile(options.mappingFile),
                revert: options.revert,
            });
        }
        catch (e) {
            (0, logging_1.error)(e.message);
            return 1;
        }
        return 0;
        async function readMappingFile(filePath) {
            if (filePath == null) {
                return undefined;
            }
            if (!(await fs.pathExists(filePath))) {
                throw new toolkit_lib_1.ToolkitError(`The mapping file ${filePath} does not exist`);
            }
            const content = JSON.parse(fs.readFileSync(filePath).toString('utf-8'));
            if (content.environments) {
                return content.environments;
            }
            else {
                throw new toolkit_lib_1.ToolkitError(`The mapping file ${filePath} does not contain an \`environments\` array`);
            }
        }
        async function readExcludeFile(filePath) {
            if (filePath != null) {
                if (!(await fs.pathExists(filePath))) {
                    throw new toolkit_lib_1.ToolkitError(`The exclude file '${filePath}' does not exist`);
                }
                return fs.readFileSync(filePath).toString('utf-8').split('\n');
            }
            return undefined;
        }
    }
    async selectStacksForList(patterns) {
        const assembly = await this.assembly();
        const stacks = await assembly.selectStacks({ patterns }, { defaultBehavior: cxapp_1.DefaultSelection.AllStacks });
        // No validation
        return stacks;
    }
    async selectStacksForDeploy(selector, exclusively, cacheCloudAssembly, ignoreNoStacks) {
        const assembly = await this.assembly(cacheCloudAssembly);
        const stacks = await assembly.selectStacks(selector, {
            extend: exclusively ? cloud_assembly_1.ExtendedStackSelection.None : cloud_assembly_1.ExtendedStackSelection.Upstream,
            defaultBehavior: cxapp_1.DefaultSelection.OnlySingle,
            ignoreNoStacks,
        });
        this.validateStacksSelected(stacks, selector.patterns);
        await this.validateStacks(stacks);
        return stacks;
    }
    async selectStacksForDiff(stackNames, exclusively, autoValidate) {
        const assembly = await this.assembly();
        const selectedForDiff = await assembly.selectStacks({ patterns: stackNames }, {
            extend: exclusively ? cloud_assembly_1.ExtendedStackSelection.None : cloud_assembly_1.ExtendedStackSelection.Upstream,
            defaultBehavior: cxapp_1.DefaultSelection.MainAssembly,
        });
        const allStacks = await this.selectStacksForList([]);
        const autoValidateStacks = autoValidate
            ? allStacks.filter((art) => art.validateOnSynth ?? false)
            : new cloud_assembly_1.StackCollection(assembly, []);
        this.validateStacksSelected(selectedForDiff.concat(autoValidateStacks), stackNames);
        await this.validateStacks(selectedForDiff.concat(autoValidateStacks));
        return selectedForDiff;
    }
    async selectStacksForDestroy(selector, exclusively) {
        const assembly = await this.assembly();
        const stacks = await assembly.selectStacks(selector, {
            extend: exclusively ? cloud_assembly_1.ExtendedStackSelection.None : cloud_assembly_1.ExtendedStackSelection.Downstream,
            defaultBehavior: cxapp_1.DefaultSelection.OnlySingle,
        });
        // No validation
        return stacks;
    }
    /**
     * Validate the stacks for errors and warnings according to the CLI's current settings
     */
    async validateStacks(stacks) {
        const failAt = this.validateMetadataFailAt();
        await stacks.validateMetadata(failAt, stackMetadataLogger(this.props.verbose));
    }
    validateMetadataFailAt() {
        let failAt = 'error';
        if (this.props.ignoreErrors) {
            failAt = 'none';
        }
        if (this.props.strict) {
            failAt = 'warn';
        }
        return failAt;
    }
    /**
     * Validate that if a user specified a stack name there exists at least 1 stack selected
     */
    validateStacksSelected(stacks, stackNames) {
        if (stackNames.length != 0 && stacks.stackCount == 0) {
            throw new toolkit_lib_1.ToolkitError(`No stacks match the name(s) ${stackNames}`);
        }
    }
    /**
     * Select a single stack by its name
     */
    async selectSingleStackByName(stackName) {
        const assembly = await this.assembly();
        const stacks = await assembly.selectStacks({ patterns: [stackName] }, {
            extend: cloud_assembly_1.ExtendedStackSelection.None,
            defaultBehavior: cxapp_1.DefaultSelection.None,
        });
        // Could have been a glob so check that we evaluated to exactly one
        if (stacks.stackCount > 1) {
            throw new toolkit_lib_1.ToolkitError(`This command requires exactly one stack and we matched more than one: ${stacks.stackIds}`);
        }
        return assembly.stackById(stacks.firstStack.id);
    }
    assembly(cacheCloudAssembly) {
        return this.props.cloudExecutable.synthesize(cacheCloudAssembly);
    }
    patternsArrayForWatch(patterns, options) {
        const patternsArray = patterns !== undefined ? (Array.isArray(patterns) ? patterns : [patterns]) : [];
        return patternsArray.length > 0 ? patternsArray : options.returnRootDirIfEmpty ? [options.rootDir] : [];
    }
    async invokeDeployFromWatch(options, cloudWatchLogMonitor) {
        const deployOptions = {
            ...options,
            requireApproval: cloud_assembly_schema_1.RequireApproval.NEVER,
            // if 'watch' is called by invoking 'cdk deploy --watch',
            // we need to make sure to not call 'deploy' with 'watch' again,
            // as that would lead to a cycle
            watch: false,
            cloudWatchLogMonitor,
            cacheCloudAssembly: false,
            extraUserAgent: `cdk-watch/hotswap-${options.deploymentMethod?.method === 'hotswap' ? 'on' : 'off'}`,
            concurrency: options.concurrency,
        };
        try {
            await this.deploy(deployOptions);
        }
        catch {
            // just continue - deploy will show the error
        }
    }
    /**
     * Remove the asset publishing and building from the work graph for assets that are already in place
     */
    async removePublishedAssets(graph, options) {
        await graph.removeUnnecessaryAssets(assetNode => this.props.deployments.isSingleAssetPublished(assetNode.assetManifest, assetNode.asset, {
            stack: assetNode.parentStack,
            roleArn: options.roleArn,
            stackName: assetNode.parentStack.stackName,
        }));
    }
}
exports.CdkToolkit = CdkToolkit;
/**
 * Print a serialized object (YAML or JSON) to stdout.
 */
function printSerializedObject(obj, json) {
    (0, logging_1.result)((0, util_2.serializeStructure)(obj, json));
}
function buildParameterMap(parameters) {
    const parameterMap = { '*': {} };
    for (const key in parameters) {
        if (parameters.hasOwnProperty(key)) {
            const [stack, parameter] = key.split(':', 2);
            if (!parameter) {
                parameterMap['*'][stack] = parameters[key];
            }
            else {
                if (!parameterMap[stack]) {
                    parameterMap[stack] = {};
                }
                parameterMap[stack][parameter] = parameters[key];
            }
        }
    }
    return parameterMap;
}
/**
 * Ask the user for a yes/no confirmation
 *
 * Automatically fail the confirmation in case we're in a situation where the confirmation
 * cannot be interactively obtained from a human at the keyboard.
 */
async function askUserConfirmation(ioHost, concurrency, motivation, question) {
    await ioHost.withCorkedLogging(async () => {
        // only talk to user if STDIN is a terminal (otherwise, fail)
        if (!TESTING && !process.stdin.isTTY) {
            throw new toolkit_lib_1.ToolkitError(`${motivation}, but terminal (TTY) is not attached so we are unable to get a confirmation from the user`);
        }
        // only talk to user if concurrency is 1 (otherwise, fail)
        if (concurrency > 1) {
            throw new toolkit_lib_1.ToolkitError(`${motivation}, but concurrency is greater than 1 so we are unable to get a confirmation from the user`);
        }
        const confirmed = await promptly.confirm(`${chalk.cyan(question)} (y/n)?`);
        if (!confirmed) {
            throw new toolkit_lib_1.ToolkitError('Aborted by user');
        }
    });
}
/**
 * Logger for processing stack metadata
 */
function stackMetadataLogger(verbose) {
    const makeLogger = (level) => {
        switch (level) {
            case 'error':
                return [logging_1.error, 'Error'];
            case 'warn':
                return [logging_1.warning, 'Warning'];
            default:
                return [logging_1.info, 'Info'];
        }
    };
    return async (level, msg) => {
        const [logFn, prefix] = makeLogger(level);
        logFn(`[${prefix} at ${msg.id}] ${msg.entry.data}`);
        if (verbose && msg.entry.trace) {
            logFn(`  ${msg.entry.trace.join('\n  ')}`);
        }
    };
}
/**
 * Determine if manual approval is required or not. Requires approval for
 * - RequireApproval.ANY_CHANGE
 * - RequireApproval.BROADENING and the changes are indeed broadening permissions
 */
function requiresApproval(requireApproval, permissionChangeType) {
    return requireApproval === cloud_assembly_schema_1.RequireApproval.ANYCHANGE ||
        requireApproval === cloud_assembly_schema_1.RequireApproval.BROADENING && permissionChangeType === toolkit_lib_1.PermissionChangeType.BROADENING;
}
//# sourceMappingURL=data:application/json;base64,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