from dataclasses import dataclass
from typing import Any, Dict, List, Literal, Optional

from .user import UserShortDetails


@dataclass
class DomainInInference:
    protocol: str
    name: str
    id: str


@dataclass
class InferenceHardwareInstanceDetails:
    id: str
    name: str


@dataclass
class InferenceAutogeneratedHFModelDetails:
    id: str


@dataclass
class InferenceAutogeneratedOutpostModelDetails:
    fullName: str


class InferenceAutogeneratedTemplateConfig:
    config: Dict[str, Any]
    revision: Optional[str]
    modelSource: Literal["huggingface", "outpost", "none"]
    huggingfaceModel: Optional[InferenceAutogeneratedHFModelDetails]
    outpostModel: Optional[InferenceAutogeneratedOutpostModelDetails]

    def __init__(self, *args, **kwargs) -> None:
        for field in self.__annotations__:
            if field == "outpostModel" and kwargs.get("outpostModel") is not None:
                self.outpostModel = InferenceAutogeneratedOutpostModelDetails(
                    **kwargs.get("outpostModel")
                )
            elif (
                field == "huggingfaceModel"
                and kwargs.get("huggingfaceModel") is not None
            ):
                self.huggingfaceModel = InferenceAutogeneratedHFModelDetails(
                    **kwargs.get("huggingfaceModel")
                )
            else:
                setattr(self, field, kwargs.get(field))


@dataclass
class InferenceCustomTemplateConfig:
    type: Literal["file", "url"]
    url: Optional[str]
    className: str


@dataclass
class InferenceDeployment:
    id: str
    status: str
    createdAt: str
    concludedAt: Optional[str]
    updatedAt: str
    timeTakenS: Optional[int]
    creator: Optional[UserShortDetails]

    def __init__(self, *args, **kwargs) -> None:
        for field in self.__annotations__:
            if field == "creator" and kwargs.get("creator") is not None:
                self.creator = UserShortDetails(**kwargs.get("creator"))
            else:
                setattr(self, field, kwargs.get(field))


@dataclass
class ReplicaScalingConfig:
    id: str
    min: int
    max: int
    scaledownPeriod: int
    targetPendingRequests: int


@dataclass
class EndpointResource:
    """
    A Inference Service on Outpost.
    """

    fullName: str
    """The fullName used to identify the inference service."""

    name: str
    """Name of the inference service."""

    visibility: Literal["public", "private", "internal"]
    """Name of the inference service."""

    id: str
    """ID of the inference service."""

    ownerId: str
    """Owner of the inference service."""

    containerType: Literal["custom", "prebuilt"]
    """Container type of the inference service."""

    templateType: Literal["autogenerated", "custom"]
    """type of template to be used for the inference server."""

    autogeneratedTemplateConfig: Optional[InferenceAutogeneratedTemplateConfig]
    """configs to autogenerate template."""

    customTemplateConfig: Optional[InferenceCustomTemplateConfig]
    """custom template details."""

    taskType: str
    """Task type of the inference service."""

    config: dict
    """Config of the inference service."""

    predictionPath: str
    """Relative path used for prediction and target for scaling."""

    healthcheckPath: str
    """Relative path used for healthcheck and readiness probes"""

    primaryDomain: Optional[DomainInInference]

    createdAt: str

    updatedAt: str

    status: str

    hardwareInstance: InferenceHardwareInstanceDetails

    port: int

    internalDomains: List[Dict[str, Any]]

    # creatorId: Optional[str]=None

    # currentDeploymentId: Optional[str]=None

    currentDeployment: Optional[InferenceDeployment] = None

    # thirdPartyKeyId: Optional[str] =None

    # configSchema: Optional[str] =None

    replicaScalingConfig: Optional[ReplicaScalingConfig] = None

    def __init__(self, *args, **kwargs) -> None:
        for field in self.__annotations__:
            if (
                field == "autogeneratedTemplateConfig"
                and kwargs.get("autogeneratedTemplateConfig") is not None
            ):
                self.autogeneratedTemplateConfig = InferenceAutogeneratedTemplateConfig(
                    **kwargs.get("autogeneratedTemplateConfig")
                )
            elif (
                field == "customTemplateConfig"
                and kwargs.get("customTemplateConfig") is not None
            ):
                self.customTemplateConfig = InferenceCustomTemplateConfig(
                    **kwargs.get("customTemplateConfig")
                )
            elif field == "primaryDomain" and kwargs.get("primaryDomain") is not None:
                self.primaryDomain = DomainInInference(**kwargs.get("primaryDomain"))
            elif field == "hardwareInstance":
                self.hardwareInstance = InferenceHardwareInstanceDetails(
                    **kwargs.get("hardwareInstance")
                )
            elif (
                field == "currentDeployment"
                and kwargs.get("currentDeployment") is not None
            ):
                self.currentDeployment = InferenceDeployment(
                    **kwargs.get("currentDeployment")
                )
            elif (
                field == "replicaScalingConfig"
                and kwargs.get("replicaScalingConfig") is not None
            ):
                self.replicaScalingConfig = ReplicaScalingConfig(
                    **kwargs.get("replicaScalingConfig")
                )
            else:
                setattr(self, field, kwargs.get(field))
