"""Fundamental physical constants.
"""


class Constants:
    """The `Constants` class contains physical constants and data
    related to chemical elements.

    References
    ----------
    .. [Mohr2016] P. J. Mohr; D. B. Newell; B. N. Taylor; CODATA
        recommended values of the fundamental physical constants:
        2014., Journal of Physical and Chemical Reference Data, Vol.
        45, No. 4, 2016, 043102.

    .. [Lide2010] D. Lide; CRC Handbook of Chemistry and Physics, 90th
        ed., CRC Press, 2010.

    .. [Newell2018] D. B. Newell; et al.; The CODATA 2017 values of
        `h`, `e`, `k`, and `N`\ :sub:`A` for the revision of the SI,
        Metrologia, Vol. 55, No. 1, 2018, L13-L16.

    .. [Cordero2008] B. Cordero; V. Gómez; A. E. Platero-Prats; M.
        Revés; J. Echeverrí; E. Cremades; F. Barragán; S. Alvarez;
        Covalent radii revisited, Dalton Trans., 2008, 2832-2838.
    """

    # Fundamental constants according to [Mohr2016].
    # Speed of light in vacuum (m/s).
    c = 2.99792458*10**8
    # Bohr radius (m).
    a = 0.52917721067*10**-10
    # Thermochemical calorie (J). [Lide2010]
    cal = 4.184
    # Hartree energy (J).
    e_h = 4.359744650*10**-18
    # Planck constant (J*s). [Newell2018]
    h = 6.62607015*10**-34
    # Boltzmann constant (J/K). [Newell2018]
    k = 1.380649*10**-23
    # Avogadro constant (1/mol). [Newell2018]
    n_a = 6.02214076*10**23
    # Atomic mass unit, amu (kg).
    u = 1.660539040*10**-27
    # Electron mass (kg).
    m_e = 9.10938356*10**-31
    # Proton mass (kg).
    m_p = 1.672621898*10**-27
    # Angstrom (m). [Lide2010]
    ang = 10**-10
    # Elementary charge (C).
    e = 1.6021766208*10**-19


    # Conversion ratios.
    # A Bohr radius to angstroms.
    a_to_ang = 0.52917721067
    # Proton-electron mass ratio.
    m_p_to_m_e = 1.83615267389*10**3

    # Atomic numbers of chemical elements.
    n_dict = {"H": 1,
              "He": 2,
              "Li": 3,
              "Be": 4,
              "B": 5,
              "C": 6,
              "N": 7,
              "O": 8,
              "F": 9,
              "Ne": 10,
              "Na": 11,
              "Mg": 12,
              "Al": 13,
              "Si": 14,
              "P": 15,
              "S": 16,
              "Cl": 17,
              "Ar": 18,
              "K": 19,
              "Ca": 20,
              "Sc": 21,
              "Ti": 22,
              "V": 23,
              "Cr": 24,
              "Mn": 25,
              "Fe": 26,
              "Co": 27,
              "Ni": 28,
              "Cu": 29,
              "Zn": 30,
              "Ga": 31,
              "Ge": 32,
              "As": 33,
              "Se": 34,
              "Br": 35,
              "Kr": 36,
              "Rb": 37,
              "Sr": 38,
              "Y": 39,
              "Zr": 40,
              "Nb": 41,
              "Mo": 42,
              "Tc": 43,
              "Ru": 44,
              "Rh": 45,
              "Pd": 46,
              "Ag": 47,
              "Cd": 48,
              "In": 49,
              "Sn": 50,
              "Sb": 51,
              "Te": 52,
              "I": 53,
              "Xe": 54,
              "Cs": 55,
              "Ba": 56,
              "La": 57,
              "Ce": 58,
              "Pr": 59,
              "Nd": 60,
              "Pm": 61,
              "Sm": 62,
              "Eu": 63,
              "Gd": 64,
              "Tb": 65,
              "Dy": 66,
              "Ho": 67,
              "Er": 68,
              "Tm": 69,
              "Yb": 70,
              "Lu": 71,
              "Hf": 72,
              "Ta": 73,
              "W": 74,
              "Re": 75,
              "Os": 76,
              "Ir": 77,
              "Pt": 78,
              "Au": 79,
              "Hg": 80,
              "Tl": 81,
              "Pb": 82,
              "Bi": 83,
              "Po": 84,
              "At": 85,
              "Rn": 86,
              "Fr": 87,
              "Ra": 88,
              "Ac": 89,
              "Th": 90,
              "Pa": 91,
              "U": 92,
              "Np": 93,
              "Pu": 94,
              "Am": 95,
              "Cm": 96}

    # Average atomic masses of the chemical elements in amu.
    # [Lide2010].
    m_dict = {"H": 1.007947,
              "He": 4.0026022,
              "Li": 6.9412,
              "Be": 9.0121823,
              "B": 10.8117,
              "C": 12.01078,
              "N": 14.00672,
              "O": 15.99943,
              "F": 18.99840325,
              "Ne": 20.17976,
              "Na": 22.989769282,
              "Mg": 24.30506,
              "Al": 26.98153868,
              "Si": 28.08553,
              "P": 30.9737622,
              "S": 32.0655,
              "Cl": 35.4532,
              "Ar": 39.9481,
              "K": 39.09831,
              "Ca": 40.0784,
              "Sc": 44.9559126,
              "Ti": 47.8671,
              "V": 50.94151,
              "Cr": 51.99616,
              "Mn": 54.9380455,
              "Fe": 55.8452,
              "Co": 58.9331955,
              "Ni": 58.69344,
              "Cu": 63.5463,
              "Zn": 65.382,
              "Ga": 69.7231,
              "Ge": 72.641,
              "As": 74.921602,
              "Se": 78.963,
              "Br": 79.9041,
              "Kr": 83.7982,
              "Rb": 85.46783,
              "Sr": 87.621,
              "Y": 88.905852,
              "Zr": 91.2242,
              "Nb": 92.906382,
              "Mo": 95.962,
              "Tc": 97.9072,
              "Ru": 101.072,
              "Rh": 102.905502,
              "Pd": 106.421,
              "Ag": 107.86822,
              "Cd": 112.4118,
              "In": 114.8183,
              "Sn": 118.7107,
              "Sb": 121.7601,
              "Te": 127.603,
              "I": 126.904473,
              "Xe": 131.2936,
              "Cs": 132.90545192,
              "Ba": 137.3277,
              "La": 138.905477,
              "Ce": 140.1161,
              "Pr": 140.907652,
              "Nd": 144.2423,
              "Pm": 144.9127,
              "Sm": 150.362,
              "Eu": 151.9641,
              "Gd": 157.253,
              "Tb": 158.925352,
              "Dy": 162.5001,
              "Ho": 164.930322,
              "Er": 167.2593,
              "Tm": 168.934212,
              "Yb": 173.0545,
              "Lu": 174.96681,
              "Hf": 178.492,
              "Ta": 180.947882,
              "W": 183.841,
              "Re": 186.2071,
              "Os": 190.233,
              "Ir": 192.2173,
              "Pt": 195.0849,
              "Au": 196.9665694,
              "Hg": 200.592,
              "Tl": 204.38332,
              "Pb": 207.21,
              "Bi": 208.980401,
              "Po": 208.9824,
              "At": 209.9871,
              "Rn": 222.0176,
              "Fr": 223.0197,
              "Ra": 226.0254,
              "Ac": 227.0277,
              "Th": 232.038062,
              "Pa": 231.035882,
              "U": 238.028913,
              "Np": 237.0482,
              "Pu": 244.0642,
              "Am": 243.0614,
              "Cm": 247.0704}

    # Covalent radii of the chemical elements in angstroms.
    # [Cordero2008].
    r_dict = {"H": 0.31,
              "He": 0.28,
              "Li": 1.28,
              "Be": 0.96,
              "B": 0.84,
              "C": 0.76,
              "N": 0.71,
              "O": 0.66,
              "F": 0.57,
              "Ne": 0.58,
              "Na": 1.66,
              "Mg": 1.41,
              "Al": 1.21,
              "Si": 1.11,
              "P": 1.07,
              "S": 1.05,
              "Cl": 1.02,
              "Ar": 1.06,
              "K": 2.03,
              "Ca": 1.76,
              "Sc": 1.70,
              "Ti": 1.60,
              "V": 1.53,
              "Cr": 1.39,
              "Mn": 1.61,
              "Fe": 1.52,
              "Co": 1.50,
              "Ni": 1.24,
              "Cu": 1.32,
              "Zn": 1.22,
              "Ga": 1.22,
              "Ge": 1.20,
              "As": 1.19,
              "Se": 1.20,
              "Br": 1.20,
              "Kr": 1.16,
              "Rb": 2.20,
              "Sr": 1.95,
              "Y": 1.90,
              "Zr": 1.75,
              "Nb": 1.64,
              "Mo": 1.54,
              "Tc": 1.47,
              "Ru": 1.46,
              "Rh": 1.42,
              "Pd": 1.39,
              "Ag": 1.45,
              "Cd": 1.44,
              "In": 1.42,
              "Sn": 1.39,
              "Sb": 1.39,
              "Te": 1.38,
              "I": 1.39,
              "Xe": 1.40,
              "Cs": 2.44,
              "Ba": 2.15,
              "La": 2.07,
              "Ce": 2.04,
              "Pr": 2.03,
              "Nd": 2.01,
              "Pm": 1.99,
              "Sm": 1.98,
              "Eu": 1.98,
              "Gd": 1.96,
              "Tb": 1.94,
              "Dy": 1.92,
              "Ho": 1.92,
              "Er": 1.89,
              "Tm": 1.90,
              "Yb": 1.87,
              "Lu": 1.87,
              "Hf": 1.75,
              "Ta": 1.70,
              "W": 1.62,
              "Re": 1.51,
              "Os": 1.44,
              "Ir": 1.41,
              "Pt": 1.36,
              "Au": 1.36,
              "Hg": 1.32,
              "Tl": 1.45,
              "Pb": 1.46,
              "Bi": 1.48,
              "Po": 1.40,
              "At": 1.50,
              "Rn": 1.50,
              "Fr": 2.60,
              "Ra": 2.21,
              "Ac": 2.15,
              "Th": 2.06,
              "Pa": 2.00,
              "U": 1.96,
              "Np": 1.90,
              "Pu": 1.87,
              "Am": 1.80,
              "Cm": 1.69}
