import logging
import os
from pathlib import Path
import re
import shutil
import subprocess
import tempfile
import time
from typing import Dict, List, Sequence, Tuple, Union

from jinja2 import Template

from madplot.utils import Convert


logger = logging.getLogger(__name__)

FilePath = Union[str, Path]


class Runner:
    def __init__(self, madx: FilePath = None):
        self.madx = madx or os.environ.get('MADX')
        if not self.madx:
            raise EnvironmentError(
                'Could not determine the path to MADX executable. Specify either by setting the '
                '`MADX` environment variable or via the `madx` argument'
            )

    def run(self, *args, **kwargs):
        raise NotImplementedError


class FileRunner(Runner):
    def run(self, f_path: FilePath, results: List[str]) -> list:
        if not isinstance(f_path, Path):
            f_path = Path(f_path)
        completed_process = subprocess.run([self.madx, f_path.name], cwd=f_path.parent,
                                           stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        stdout = completed_process.stdout.decode('utf-8')
        stderr = completed_process.stderr.decode('utf-8')
        f_out = f_path.parent / 'stdout'
        f_err = f_path.parent / 'stderr'
        with open(f_out, 'w') as f1, open(f_err, 'w') as f2:
            f1.write(stdout)
            f2.write(stderr)
        logger.debug(f'Simulation finished with return code {completed_process.returncode}')
        if completed_process.returncode != 0:
            raise RuntimeError(
                f'Simulation did not finish successfully (return code {completed_process.returncode}). '
                f'See {f_out} and {f_err} for the stdout and stderr respectively.'
            )
        results = map(lambda x: x if isinstance(x, (tuple, list)) else (x, {}), results)
        results = map(lambda x: (f_path.parent / x[0], x[1]), results)
        return list(map(lambda x: Convert.by_file_name(x[0], **x[1]), results))


class MADXEngine(FileRunner):
    """Main class for running MADX scripts.
    
    Example
    -------
    
    ```python
    engine = MADXEngine(['example.madx'])
    twiss, = engine.run(['example.twiss'])
    ```
    """

    def __init__(self,
                 templates : Union[FilePath, Sequence[FilePath]],
                 madx : FilePath = None,
                 working_directory : FilePath = None):
        """Creates a new MADX engine.
        
        Parameters
        ----------
        templates: list or tuple
            List containing the file paths of all MADX scripts (or templates) that are required for
            running the main script. The main script is the first element of the template list.
            A template is a MADX scripts that contains substitution keys of the form `{key}`
            (following the Python formatting specification). This can be interpolated when running
            the scripts.
        madx: str, optional
            File path to the MADX executable. If not given, then an attempt to retrieve the path from
            the `MADX` environment variable is made. If this attempt fails then a `EnvironmentError`
            is raised.
        working_directory: str, optional
            If specified, the engine will run the script(s) in this directory (and hence place all
            required files there as well). If not specified, a temporary directory is created automatically.
            
        Raises
        ------
        EnvironmentError
            If no `madx` executable path is specified and the `MADX` environment variable is not set.
        """
        super().__init__(madx)
        if isinstance(templates, (str, Path)):
            self.templates = [Path(templates)]
        else:
            self.templates = list(map(Path, templates))
        self.working_directory = Path(working_directory or tempfile.mkdtemp())

    def run(self, results: list, configuration: dict = None, job_id: Union[int, str] = None, overwrite: bool = False) -> list:
        """Run the specified scripts (templates).
        
        Parameters
        ----------
        results : list or tuple
            List containing the file names of output files that should be retrieved. These output
            files need to be generated by the scripts that are run. The resulting TFS files will be
            read and converted to `pandas.DataFrame` objects.
            For fine-tuning of how an output should be handled it can alternatively be specified as
            a tuple of `(file_name, options)` where `options` is a dictionary containing additional
            keyword arguments for the conversion function (which is `madplot.utils.tfs_to_pd`).
        configuration : dict, optional
            A dictionary containing the configuration for one or more of the template scripts that
            have been specified during instantiation of that engine. The keys are file names (not
            file paths) of the template scripts and the values are dictionaries containing the
            substitution keys and values that occur with the particular scripts.
        job_id : str, optional
            If a `job_id` is specified it will be appended to the `working_directory` and the scripts
            will be run in the resulting sub-directory.
        overwrite : bool, optional, default = `False`
            If set to true, overwrites templates that are to be interpolated where the working directory
            is similar to the template directory. If false, raises a UserWarning when overwriting would
            be necessary.
            
        Returns
        -------
        results: list
            A list containing the data frames corresponding to the specified result names.
            If a result was specified as `(file_name, {'return_meta': True})` then the corresponding
            element is a tuple of `(data_frame, meta_data)` where `meta_data` is a dictionary
            containing the `@`-prefixed data of the corresponding output TFS file.
            
        Raises
        ------
        RuntimeError
            If the MADX executable did not finish with return code 0.
        UserWarning
            If a template would be overwritten by its interpolated version. This can happen if the
            `working_directory` is the same directory where the templates are placed. This behavior
            can be overridden by setting ``overwrite=True``.
        """
        if configuration is None:
            configuration = {}
        if job_id is not None:
            working_directory = self.working_directory / str(job_id)
            os.mkdir(working_directory)
        else:
            working_directory = self.working_directory
        logger.debug(f'Working directory is {working_directory}')

        for template in self.templates:
            new_path = working_directory / template.name
            if new_path == template and template.name in configuration and not overwrite:
                raise UserWarning(f'Refusing to overwrite template by interpolation '
                                  f'(target path is identical with template path): {template}')
                continue
            if template.name in configuration:
                with open(template) as f1:
                    formatted = f1.read().format(**configuration[template.name])
                with open(new_path, 'w') as f2:
                    f2.write(formatted)
            elif new_path != template:
                shutil.copyfile(template, new_path)

        return super().run(working_directory / self.templates[0], results=results)


class MADXPipe(Runner):
    """Main class for running MADX commands where input and output is piped via stdin/stdout."""

    def run(self, script: str) -> Tuple[str, str]:
        """Run the given script and return the (stdout, stdin) as a tuple.

        Raises
        ------
        CalledProcessError
            See :class:`subprocess.CalledProcessError`.
        """
        result = subprocess.run([self.madx], input=script.encode(), stdout=subprocess.PIPE, stderr=subprocess.PIPE, check=True)
        return result.stdout.decode(), result.stderr.decode()


class MADXSession(Runner):
    """Main class for creating an interactive MADX session.

    MADX sessions are similar to the MADX interactive interpreter.

    Example
    -------

    ```python
    session = MADXSession()
    twiss, = session.run(commands=['twiss, file="example.twiss"'], results=['example.twiss'])
    ```
    """

    def __init__(self,
                 start_up_scripts: Sequence[str] = (),
                 madx : FilePath = None,
                 stderr = None, stdout = None,
                 buffer_time: float = 0.05, timeout: float = 1):
        """Create a new MADX session.

        Parameters
        ----------
        start_up_scripts: list or tuple
            Sequence of scripts that are to be executed once after the session is started. The specified scripts
            will be invoked via the "call" command. The absolute or relative path for each script should be given.
        madx: str, optional
            File path to the MADX executable. If not given, then an attempt to retrieve the path from
            the `MADX` environment variable is made. If this attempt fails then a `EnvironmentError`
            is raised.
        stderr, stdout: file handle
            Can be used to capture the stderr and stdout of the session respectively.
            See `subprocess.Popen` for more information.
        buffer_time, timeout: float
            Default values to be used when waiting for the session to finish writing files.
            See `MADXSession.sentry` for more information.

        Raises
        ------
        EnvironmentError
            If no `madx` executable path is specified and the `MADX` environment variable is not set.
        """
        super().__init__(madx)
        self.buffer_time = buffer_time
        self.timeout = timeout
        self.process = subprocess.Popen(self.madx, stdin=subprocess.PIPE, stderr=stderr, stdout=stdout)
        if isinstance(start_up_scripts, str):
            start_up_scripts = (start_up_scripts,)
        for f_path in start_up_scripts:
            f_path = f_path if os.path.isabs(f_path) else os.path.join(os.getcwd(), f_path)
            logging.info('Call: %s', f_path)
            self.process.stdin.write(f'call, file = "{f_path}";\n'.encode('ascii'))
            self.process.stdin.flush()
        if start_up_scripts:
            self.sentry(self.buffer_time, self.timeout)

    def run(self, commands: Sequence[str] = (), results: Sequence[str] = (),
            sentry: bool = True, buffer_time: float = None, timeout: float = None):
        """Run a sequence of commands in the interactive session.

        Optionally returns a list of specified results. Because the subprocess needs to write these results to disk
        we need to wait a grace period for it to finish. If that period is exceeded an exception is raised.

        Parameters
        ----------
        commands: list or tuple
            Sequence of commands to be executed. The commands will be executed in the given order.
        results: list or tuple
            Sequence of paths pointing to files that were by the session before. The corresponding data frames
            will be returned.
        sentry: bool
            Indicates whether a sentry should be placed in order to make sure all specified commands have been executed.
        buffer_time: float
            Time between checks whether the session has finished writing all files.
        timeout: float
            Maximum grace period for waiting for the session to finish writing files. If this time is exceeded
            an exception is raised.

        Returns
        -------
        results: list
            List of results corresponding to the specification.

        Raises
        ------
        subprocess.TimeoutExpired:
            In case the grace period specified by `timeout` is exceeded.
        """
        results = [x if isinstance(x, (tuple, list)) else (x, {}) for x in results]
        results = [(os.path.join(os.getcwd(), x[0]), x[1]) for x in results]
        for command in commands:
            command = command.rstrip('\n').rstrip(';')
            self.process.stdin.write(f'{command};\n'.encode('ascii'))
            self.process.stdin.flush()
        if commands and sentry:
            self.sentry(buffer_time or self.buffer_time, timeout or self.timeout)
        return list(map(lambda x: Convert.by_file_name(x[0], **x[1]), results))

    def sentry(self, buffer_time=0.05, timeout=1):
        """Execute sentry command which creates a file on the hard disk. We then check for existence of that file
        in order to make sure all previous commands in the session have been executed. This is useful for example
        when requesting a table as a result (e.g. from twiss command), in order to make sure this table has been
        written completely to the disk.

        Parameters
        ----------
        buffer_time: float
            Time between checks for existence of the sentry file.
        timeout: float
            Maximum grace period for waiting for the session to create the sentry file. If this time is exceeded
            an exception is raised.

        Raises
        ------
        subprocess.TimeoutExpired:
            In case the grace period specified by `timeout` is exceeded.
        """
        fd, sentry_name = tempfile.mkstemp()
        os.close(fd)
        os.remove(sentry_name)
        command = (
            'create, table="sentry", column=s;'
            f'write, table="sentry", file="{sentry_name}";'
            'delete, table="sentry"'
        )
        self.process.stdin.write(f'{command};\n'.encode('ascii'))
        self.process.stdin.flush()
        sleep_time = 0
        while sleep_time < timeout and not os.path.exists(sentry_name):
            time.sleep(buffer_time)  # Wait for process to finish writing files, etc.
            sleep_time += buffer_time
        if not os.path.exists(sentry_name):
            raise subprocess.TimeoutExpired(self.madx, timeout)
        os.remove(sentry_name)


class JinjaRunner(Runner):
    def __init__(self, template: FilePath, *, madx: FilePath = None):
        super().__init__(madx)
        with open(template) as fh:
            self.template = Template(fh.read())


class JinjaEngine(JinjaRunner, FileRunner):
    """Running MADX script templates using the Jinja2 templating engine by creating intermediary script files.

    This engine saves the rendered scripts as files before running them with MADX. MADX outputs can also be files,
    as a dedicated working directory can be created.

    Parameters
    ----------
    template : str or Path
        A MADX script, optionally template'ized according to Jinja2 syntax. This template is used as the basis for
        running various configurations.
    madx, working_directory
        See :class:`MADXEngine`.
    """

    def __init__(self, template: FilePath, *, madx: FilePath = None, working_directory: FilePath = None):
        super().__init__(template, madx=madx)
        self.working_directory = Path(working_directory or tempfile.mkdtemp())
        self.script_name = re.sub('[.](template|j2|jinja|jinja2)', '', Path(template).name)

    def run(self, results: List[str], configuration: Dict = None, *, job_id: Union[int, str] = None) -> list:
        """Run a single configuration against the template.

        Parameters
        ----------
        configuration : dict
            Gets passed directly to `template.render`.
        results, job_id
            See :class:`MADXEngine`.

        Returns
        -------
        See :class:`MADXEngine`.
        """
        if job_id is not None:
            working_directory = self.working_directory / str(job_id)
            os.mkdir(working_directory)
        else:
            working_directory = self.working_directory
        script_path = working_directory / self.script_name
        with open(script_path, 'w') as fh:
            fh.write(self.template.render(**configuration or {}))
        return super().run(script_path, results=results)


class JinjaPipe(JinjaRunner, MADXPipe):
    """Running MADX script templates using the Jinja2 templating engine without creating intermediary script files.

    The engine pipes input and output to and from the underlying subprocess, hence no intermediary script files will
    be created.

    Parameters
    ----------
    template : str or Path
        A MADX script, optionally template'ized according to Jinja2 syntax. This template is used as the basis for
        running various configurations.
    madx
        See :class:`MADXPipe`.
    """
    def __init__(self, template: FilePath, *, madx: FilePath = None):
        super().__init__(template, madx=madx)

    def run(self, **configuration) -> Tuple[str, str]:
        """Run a single configuration against the template.

        Parameters
        ----------
        configuration
            Gets passed directly to `template.render`.

        Returns
        -------
        See :class:`MADXPipe`.
        """
        return super().run(self.template.render(**configuration))
