# -*- coding: utf-8 -*-
import sys

import typeguard
from outflow.core.exceptions import IOCheckerError
from outflow.core.logging import logger
from outflow.core.target import Target
from outflow.core.types import Skipped
from typing_extensions import TypedDict

from .base_task import BaseTask


class Task(BaseTask):
    def __init__(self, *args, auto_outputs=True, **kwargs):

        super().__init__(*args, **kwargs)
        self.skip = False
        self.skip_if_upstream_skip = True
        self._memo = None
        if not self.outputs and auto_outputs:
            # run automatic output detection if none were declared
            Target.output("__auto__")(self)

    def _run(self, *args, **kwargs):
        return self.run(*args, **kwargs)

    def __call__(self, *args, **kwargs):
        super().__call__(*args, **kwargs)

        if self.skip_if_upstream_skip and any(parent.skip for parent in self.parents):
            self.skip = True

        if self.skip:
            return Skipped()

        self.db_task.start()

        self.resolve_remote_inputs(kwargs)

        task_kwargs = {**kwargs, **self.bind_kwargs, **self.parameterized_kwargs}

        memo = self.generate_memo(task_kwargs)
        self.check_argument_types(memo)

        try:
            return_value = self._run(**task_kwargs)
        except Exception as e:
            e.__dict__["db_task"] = self.db_task
            self.db_task.fail()
            raise e

        self.check_return_type(return_value)

        self.db_task.success()

        return return_value

    def resolve_remote_inputs(self, task_kwargs):
        """Transform 'ray' ObjectIDs into Python object

        From main actor: get the object in the object store
        If remote: get in the remote object store and transfer by TCP
        """
        try:
            import ray

            for key, val in task_kwargs.items():
                if isinstance(val, ray._raylet.ObjectID):
                    task_kwargs[key] = ray.get(val)
        except ImportError:
            logger.debug(
                "The 'ray' package is not available, usage of remote objects is therefore impossible"
            )

    def generate_memo(self, task_kwargs):

        # Find the frame in which the function was declared, for resolving forward references later
        _localns = sys._getframe(1).f_locals

        # stored needed info to check the task input/output types
        try:
            return typeguard._CallMemo(self.run, _localns, args=[], kwargs=task_kwargs)
        except TypeError as e:
            raise IOCheckerError(f"In task '{self.name}' - {e}") from e

    def check_argument_types(self, memo):
        try:
            typeguard.check_argument_types(memo)
        except TypeError as e:
            raise IOCheckerError(f"In task '{self.name}' - {e}") from e

    def check_return_type(self, return_value):
        """A custom version of typeguard.check_return_type(return_value, memo)

        This checks the type of each output (type being Any if declared with __auto__)

        Note: we have to skip both None and Skipped tasks to improve the TypeError
        generated by 'typeguard.check_type'

        """
        if return_value is not None and not isinstance(return_value, Skipped):
            # check either against a typed dict or a single type
            if isinstance(return_value, dict):
                typed_dict = {}
                for output_target in self.outputs.values():
                    typed_dict.update({output_target.name: output_target.type})

                return_type = TypedDict("return_type", typed_dict)
            else:
                return_type = next(iter(self.outputs.values())).type

            typeguard.check_type("return_value", return_value, return_type)


task = Task.as_task
