"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualGateway = void 0;
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
const gateway_route_1 = require("./gateway-route");
const mesh_1 = require("./mesh");
const virtual_gateway_listener_1 = require("./virtual-gateway-listener");
class VirtualGatewayBase extends cdk.Resource {
    /**
     * (experimental) Utility method to add a new GatewayRoute to the VirtualGateway.
     *
     * @experimental
     */
    addGatewayRoute(id, props) {
        return new gateway_route_1.GatewayRoute(this, id, {
            ...props,
            virtualGateway: this,
        });
    }
}
/**
 * (experimental) VirtualGateway represents a newly defined App Mesh Virtual Gateway.
 *
 * A virtual gateway allows resources that are outside of your mesh to communicate to resources that
 * are inside of your mesh.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_gateways.html
 * @experimental
 */
class VirtualGateway extends VirtualGatewayBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id, {
            physicalName: props.virtualGatewayName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        /**
         * @experimental
         */
        this.listeners = new Array();
        this.mesh = props.mesh;
        if (!props.listeners) {
            // Use listener default of http listener port 8080 if no listener is defined
            this.listeners.push(virtual_gateway_listener_1.VirtualGatewayListener.http().bind(this));
        }
        else {
            props.listeners.forEach(listener => this.listeners.push(listener.bind(this)));
        }
        const accessLogging = (_a = props.accessLog) === null || _a === void 0 ? void 0 : _a.bind(this);
        const node = new appmesh_generated_1.CfnVirtualGateway(this, 'Resource', {
            virtualGatewayName: this.physicalName,
            meshName: this.mesh.meshName,
            spec: {
                listeners: this.listeners.map(listener => listener.listener),
                backendDefaults: (_b = props.backendsDefaultClientPolicy) === null || _b === void 0 ? void 0 : _b.bind(this),
                logging: accessLogging !== undefined ? {
                    accessLog: accessLogging.virtualGatewayAccessLog,
                } : undefined,
            },
        });
        this.virtualGatewayName = this.getResourceNameAttribute(node.attrVirtualGatewayName);
        this.virtualGatewayArn = this.getResourceArnAttribute(node.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualGateway`,
            resourceName: this.physicalName,
        });
    }
    /**
     * (experimental) Import an existing VirtualGateway given an ARN.
     *
     * @experimental
     */
    static fromVirtualGatewayArn(scope, id, virtualGatewayArn) {
        return new class extends VirtualGatewayBase {
            constructor() {
                super(...arguments);
                this.parsedArn = cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(virtualGatewayArn).resourceName);
                this.mesh = mesh_1.Mesh.fromMeshName(this, 'Mesh', cdk.Fn.select(0, this.parsedArn));
                this.virtualGatewayArn = virtualGatewayArn;
                this.virtualGatewayName = cdk.Fn.select(2, this.parsedArn);
            }
        }(scope, id);
    }
    /**
     * (experimental) Import an existing VirtualGateway given its attributes.
     *
     * @experimental
     */
    static fromVirtualGatewayAttributes(scope, id, attrs) {
        return new class extends VirtualGatewayBase {
            constructor() {
                super(...arguments);
                this.mesh = attrs.mesh;
                this.virtualGatewayName = attrs.virtualGatewayName;
                this.virtualGatewayArn = cdk.Stack.of(this).formatArn({
                    service: 'appmesh',
                    resource: `mesh/${attrs.mesh.meshName}/virtualGateway`,
                    resourceName: this.virtualGatewayName,
                });
            }
        }(scope, id);
    }
}
exports.VirtualGateway = VirtualGateway;
//# sourceMappingURL=data:application/json;base64,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