"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const secretsmanager = require("../lib");
module.exports = {
    'default secret'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new secretsmanager.Secret(stack, 'Secret');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::Secret', {
            GenerateSecretString: {},
        }));
        test.done();
    },
    'secret with kms'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const key = new kms.Key(stack, 'KMS');
        // WHEN
        new secretsmanager.Secret(stack, 'Secret', { encryptionKey: key });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::KMS::Key', {
            KeyPolicy: {
                Statement: [
                    {},
                    {
                        Effect: 'Allow',
                        Resource: '*',
                        Action: [
                            'kms:Decrypt',
                            'kms:Encrypt',
                            'kms:ReEncrypt*',
                            'kms:GenerateDataKey*',
                        ],
                        Principal: {
                            AWS: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':iam::',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':root',
                                    ],
                                ],
                            },
                        },
                        Condition: {
                            StringEquals: {
                                'kms:ViaService': {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'secretsmanager.',
                                            {
                                                Ref: 'AWS::Region',
                                            },
                                            '.amazonaws.com',
                                        ],
                                    ],
                                },
                            },
                        },
                    },
                    {
                        Effect: 'Allow',
                        Resource: '*',
                        Action: [
                            'kms:CreateGrant',
                            'kms:DescribeKey',
                        ],
                        Principal: {
                            AWS: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':iam::',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':root',
                                    ],
                                ],
                            },
                        },
                        Condition: {
                            StringEquals: {
                                'kms:ViaService': {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'secretsmanager.',
                                            {
                                                Ref: 'AWS::Region',
                                            },
                                            '.amazonaws.com',
                                        ],
                                    ],
                                },
                            },
                        },
                    },
                ],
                Version: '2012-10-17',
            },
        }));
        test.done();
    },
    'secret with generate secret string options'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new secretsmanager.Secret(stack, 'Secret', {
            generateSecretString: {
                excludeUppercase: true,
                passwordLength: 20,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::Secret', {
            GenerateSecretString: {
                ExcludeUppercase: true,
                PasswordLength: 20,
            },
        }));
        test.done();
    },
    'templated secret string'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new secretsmanager.Secret(stack, 'Secret', {
            generateSecretString: {
                secretStringTemplate: JSON.stringify({ username: 'username' }),
                generateStringKey: 'password',
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::Secret', {
            GenerateSecretString: {
                SecretStringTemplate: '{"username":"username"}',
                GenerateStringKey: 'password',
            },
        }));
        test.done();
    },
    'grantRead'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const key = new kms.Key(stack, 'KMS');
        const secret = new secretsmanager.Secret(stack, 'Secret', { encryptionKey: key });
        const role = new iam.Role(stack, 'Role', { assumedBy: new iam.AccountRootPrincipal() });
        // WHEN
        secret.grantRead(role);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [{
                        Action: 'secretsmanager:GetSecretValue',
                        Effect: 'Allow',
                        Resource: { Ref: 'SecretA720EF05' },
                    }],
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::KMS::Key', {
            KeyPolicy: {
                Statement: [
                    {},
                    {},
                    {},
                    {
                        Action: 'kms:Decrypt',
                        Condition: {
                            StringEquals: {
                                'kms:ViaService': {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'secretsmanager.',
                                            {
                                                Ref: 'AWS::Region',
                                            },
                                            '.amazonaws.com',
                                        ],
                                    ],
                                },
                            },
                        },
                        Effect: 'Allow',
                        Principal: {
                            AWS: {
                                'Fn::GetAtt': [
                                    'Role1ABCC5F0',
                                    'Arn',
                                ],
                            },
                        },
                        Resource: '*',
                    },
                ],
                Version: '2012-10-17',
            },
        }));
        test.done();
    },
    'grantRead with version label constraint'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const key = new kms.Key(stack, 'KMS');
        const secret = new secretsmanager.Secret(stack, 'Secret', { encryptionKey: key });
        const role = new iam.Role(stack, 'Role', { assumedBy: new iam.AccountRootPrincipal() });
        // WHEN
        secret.grantRead(role, ['FOO', 'bar']);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [{
                        Action: 'secretsmanager:GetSecretValue',
                        Effect: 'Allow',
                        Resource: { Ref: 'SecretA720EF05' },
                        Condition: {
                            'ForAnyValue:StringEquals': {
                                'secretsmanager:VersionStage': ['FOO', 'bar'],
                            },
                        },
                    }],
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::KMS::Key', {
            KeyPolicy: {
                Statement: [
                    {},
                    {},
                    {},
                    {
                        Action: 'kms:Decrypt',
                        Condition: {
                            StringEquals: {
                                'kms:ViaService': {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'secretsmanager.',
                                            {
                                                Ref: 'AWS::Region',
                                            },
                                            '.amazonaws.com',
                                        ],
                                    ],
                                },
                            },
                        },
                        Effect: 'Allow',
                        Principal: {
                            AWS: {
                                'Fn::GetAtt': [
                                    'Role1ABCC5F0',
                                    'Arn',
                                ],
                            },
                        },
                        Resource: '*',
                    },
                ],
                Version: '2012-10-17',
            },
        }));
        test.done();
    },
    'grantWrite'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const secret = new secretsmanager.Secret(stack, 'Secret', {});
        const role = new iam.Role(stack, 'Role', { assumedBy: new iam.AccountRootPrincipal() });
        // WHEN
        secret.grantWrite(role);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [{
                        Action: 'secretsmanager:PutSecretValue',
                        Effect: 'Allow',
                        Resource: { Ref: 'SecretA720EF05' },
                    }],
            },
        }));
        test.done();
    },
    'grantWrite with kms'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const key = new kms.Key(stack, 'KMS');
        const secret = new secretsmanager.Secret(stack, 'Secret', { encryptionKey: key });
        const role = new iam.Role(stack, 'Role', { assumedBy: new iam.AccountRootPrincipal() });
        // WHEN
        secret.grantWrite(role);
        // THEN
        const expectStack = assert_1.expect(stack);
        expectStack.to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [{
                        Action: 'secretsmanager:PutSecretValue',
                        Effect: 'Allow',
                        Resource: { Ref: 'SecretA720EF05' },
                    }],
            },
        }));
        expectStack.to(assert_1.haveResourceLike('AWS::KMS::Key', {
            KeyPolicy: {
                Statement: [
                    {},
                    {},
                    {},
                    {
                        Action: [
                            'kms:Encrypt',
                            'kms:ReEncrypt*',
                            'kms:GenerateDataKey*',
                        ],
                        Condition: {
                            StringEquals: {
                                'kms:ViaService': {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'secretsmanager.',
                                            {
                                                Ref: 'AWS::Region',
                                            },
                                            '.amazonaws.com',
                                        ],
                                    ],
                                },
                            },
                        },
                        Effect: 'Allow',
                        Principal: {
                            AWS: {
                                'Fn::GetAtt': [
                                    'Role1ABCC5F0',
                                    'Arn',
                                ],
                            },
                        },
                        Resource: '*',
                    },
                ],
            },
        }));
        test.done();
    },
    'secretValue'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const key = new kms.Key(stack, 'KMS');
        const secret = new secretsmanager.Secret(stack, 'Secret', { encryptionKey: key });
        // WHEN
        new cdk.CfnResource(stack, 'FakeResource', {
            type: 'CDK::Phony::Resource',
            properties: {
                value: secret.secretValue,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('CDK::Phony::Resource', {
            value: {
                'Fn::Join': ['', [
                        '{{resolve:secretsmanager:',
                        { Ref: 'SecretA720EF05' },
                        ':SecretString:::}}',
                    ]],
            },
        }));
        test.done();
    },
    'import'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const encryptionKey = new kms.Key(stack, 'KMS');
        const secretArn = 'arn::of::a::secret';
        // WHEN
        const secret = secretsmanager.Secret.fromSecretAttributes(stack, 'Secret', {
            secretArn, encryptionKey,
        });
        // THEN
        test.equals(secret.secretArn, secretArn);
        test.same(secret.encryptionKey, encryptionKey);
        test.deepEqual(stack.resolve(secret.secretValue), '{{resolve:secretsmanager:arn::of::a::secret:SecretString:::}}');
        test.deepEqual(stack.resolve(secret.secretValueFromJson('password')), '{{resolve:secretsmanager:arn::of::a::secret:SecretString:password::}}');
        test.done();
    },
    'can attach a secret with attach()'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const secret = new secretsmanager.Secret(stack, 'Secret');
        // WHEN
        secret.attach({
            asSecretAttachmentTarget: () => ({
                targetId: 'target-id',
                targetType: 'target-type',
            }),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::SecretTargetAttachment', {
            SecretId: {
                Ref: 'SecretA720EF05',
            },
            TargetId: 'target-id',
            TargetType: 'target-type',
        }));
        test.done();
    },
    'throws when trying to attach a target multiple times to a secret'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const secret = new secretsmanager.Secret(stack, 'Secret');
        const target = {
            asSecretAttachmentTarget: () => ({
                targetId: 'target-id',
                targetType: 'target-type',
            }),
        };
        secret.attach(target);
        // THEN
        test.throws(() => secret.attach(target), /Secret is already attached to a target/);
        test.done();
    },
    'add a rotation schedule to an attached secret'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const secret = new secretsmanager.Secret(stack, 'Secret');
        const attachedSecret = secret.attach({
            asSecretAttachmentTarget: () => ({
                targetId: 'target-id',
                targetType: 'target-type',
            }),
        });
        const rotationLambda = new lambda.Function(stack, 'Lambda', {
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromInline('export.handler = event => event;'),
            handler: 'index.handler',
        });
        // WHEN
        attachedSecret.addRotationSchedule('RotationSchedule', {
            rotationLambda,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::RotationSchedule', {
            SecretId: {
                Ref: 'SecretAttachment2E1B7C3B',
            },
        }));
        test.done();
    },
    'throws when specifying secretStringTemplate but not generateStringKey'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // THEN
        test.throws(() => new secretsmanager.Secret(stack, 'Secret', {
            generateSecretString: {
                secretStringTemplate: JSON.stringify({ username: 'username' }),
            },
        }), /`secretStringTemplate`.+`generateStringKey`/);
        test.done();
    },
    'throws when specifying generateStringKey but not secretStringTemplate'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // THEN
        test.throws(() => new secretsmanager.Secret(stack, 'Secret', {
            generateSecretString: {
                generateStringKey: 'password',
            },
        }), /`secretStringTemplate`.+`generateStringKey`/);
        test.done();
    },
    'equivalence of SecretValue and Secret.fromSecretAttributes'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const imported = secretsmanager.Secret.fromSecretAttributes(stack, 'Imported', { secretArn: 'my-secret-arn' }).secretValueFromJson('password');
        const value = cdk.SecretValue.secretsManager('my-secret-arn', { jsonField: 'password' });
        // THEN
        test.deepEqual(stack.resolve(imported), stack.resolve(value));
        test.done();
    },
    'can add to the resource policy of a secret'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const secret = new secretsmanager.Secret(stack, 'Secret');
        // WHEN
        secret.addToResourcePolicy(new iam.PolicyStatement({
            actions: ['secretsmanager:GetSecretValue'],
            resources: ['*'],
            principals: [new iam.ArnPrincipal('arn:aws:iam::123456789012:user/cool-user')],
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::ResourcePolicy', {
            ResourcePolicy: {
                Statement: [
                    {
                        Action: 'secretsmanager:GetSecretValue',
                        Effect: 'Allow',
                        Principal: {
                            AWS: 'arn:aws:iam::123456789012:user/cool-user',
                        },
                        Resource: '*',
                    },
                ],
                Version: '2012-10-17',
            },
            SecretId: {
                Ref: 'SecretA720EF05',
            },
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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