"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SdkQuery = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("./assertions");
const hash_1 = require("./private/hash");
const providers_1 = require("./providers");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const constructs_1 = require("constructs");
class SdkQuery extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.flattenResponse = 'false';
        const provider = new providers_1.AssertionsProvider(this, 'SdkProvider');
        provider.addPolicyStatementFromSdkCall(props.service, props.api);
        this.sdkCallResource = new aws_cdk_lib_1.CustomResource(this, 'Default', {
            serviceToken: provider.serviceToken,
            properties: {
                service: props.service,
                api: props.api,
                parameters: provider.encode(props.parameters),
                flattenResponse: aws_cdk_lib_1.Lazy.string({ produce: () => this.flattenResponse }),
            },
            resourceType: `${providers_1.SDK_RESOURCE_TYPE_PREFIX}${props.service}${props.api}`,
        });
        // Needed so that all the policies set up by the provider should be available before the custom resource is provisioned.
        this.sdkCallResource.node.addDependency(provider);
    }
    /**
     * Returns the value of an attribute of the custom resource of an arbitrary
     * type. Attributes are returned from the custom resource provider through the
     * `Data` map where the key is the attribute name.
     *
     * @param attributeName the name of the attribute
     * @returns a token for `Fn::GetAtt`. Use `Token.asXxx` to encode the returned `Reference` as a specific type or
     * use the convenience `getAttString` for string attributes.
     */
    getAtt(attributeName) {
        this.flattenResponse = 'true';
        return this.sdkCallResource.getAtt(`apiCallResponse.${attributeName}`);
    }
    /**
     * Returns the value of an attribute of the custom resource of type string.
     * Attributes are returned from the custom resource provider through the
     * `Data` map where the key is the attribute name.
     *
     * @param attributeName the name of the attribute
     * @returns a token for `Fn::GetAtt` encoded as a string.
     */
    getAttString(attributeName) {
        this.flattenResponse = 'true';
        return this.sdkCallResource.getAttString(`apiCallResponse.${attributeName}`);
    }
    /**
     * Creates an assertion custom resource that will assert that the response
     * from the SDKQuery equals the 'expected' value
     */
    assertEqual(expected, actualAttr) {
        const hash = hash_1.md5hash(expected);
        let inputResourceAtt = 'apiCallResponse';
        if (actualAttr) {
            this.flattenResponse = 'true';
            inputResourceAtt = `apiCallResponse.${actualAttr}`;
        }
        return new assertions_1.EqualsAssertion(this, `AssertEquals${hash}`, {
            expected,
            inputResource: this.sdkCallResource,
            inputResourceAtt,
        });
    }
}
exports.SdkQuery = SdkQuery;
//# sourceMappingURL=data:application/json;base64,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