"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeployAssert = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const hash_1 = require("./private/hash");
const providers_1 = require("./providers");
const sdk_1 = require("./sdk");
const DEPLOY_ASSERT_SYMBOL = Symbol.for('@aws-cdk/integ-tests.DeployAssert');
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const constructs_2 = require("constructs");
/**
 * Construct that allows for registering a list of assertions
 * that should be performed on a construct
 */
class DeployAssert extends constructs_2.Construct {
    constructor(scope) {
        super(scope, 'DeployAssert');
        Object.defineProperty(this, DEPLOY_ASSERT_SYMBOL, { value: true });
        this._assertions = [];
        const provider = new providers_1.AssertionsProvider(this, 'ResultsProvider');
        const resource = new aws_cdk_lib_1.CustomResource(this, 'ResultsCollection', {
            serviceToken: provider.serviceToken,
            properties: {
                assertionResults: aws_cdk_lib_1.Lazy.list({
                    produce: () => this._assertions.map(a => a.result),
                }),
            },
            resourceType: providers_1.RESULTS_RESOURCE_TYPE,
        });
        // TODO: need to show/store this information
        new aws_cdk_lib_1.CfnOutput(this, 'Results', {
            value: `\n${resource.getAttString('message')}`,
        }).overrideLogicalId('Results');
    }
    /**
     * Returns whether the construct is a DeployAssert construct
     */
    static isDeployAssert(x) {
        return x !== null && typeof (x) === 'object' && DEPLOY_ASSERT_SYMBOL in x;
    }
    /**
     * Finds a DeployAssert construct in the given scope
     */
    static of(construct) {
        const scopes = constructs_1.Node.of(construct).scopes.reverse();
        const deployAssert = scopes.find(s => DeployAssert.isDeployAssert(s));
        if (!deployAssert) {
            throw new Error('No DeployAssert construct found in scopes');
        }
        return deployAssert;
    }
    /**
     * Query AWS using JavaScript SDK V2 API calls
     */
    queryAws(options) {
        const id = hash_1.md5hash(options);
        return new sdk_1.SdkQuery(this, `SdkQuery${id}`, options);
    }
    /**
     * Register an assertion that should be run as part of the
     * deployment
     */
    registerAssertion(assertion) {
        this._assertions.push(assertion);
    }
}
exports.DeployAssert = DeployAssert;
//# sourceMappingURL=data:application/json;base64,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