"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecureBucket = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const _ = require("lodash");
const ez_construct_1 = require("../ez-construct");
const utils_1 = require("../lib/utils");
/**
 * Will create a secure bucket with the following features:
 * - Bucket name will be modified to include account and region.
 * - Access limited to the owner
 * - Object Versioning
 * - Encryption at rest
 * - Object expiration max limit to 10 years
 * - Object will transition to IA after 60 days and later to deep archive after 365 days
 *
 * Example:
 *
 * ```ts
 *    let aBucket = new SecureBucket(mystack, 'secureBucket', {
 *      bucketName: 'mybucket',
 *      objectsExpireInDays: 500,
 *      enforceSSL: false,
 *     });
 * ```
 */
class SecureBucket extends ez_construct_1.EzConstruct {
    /**
     * Creates the SecureBucket
     * @param scope - the stack in which the construct is defined.
     * @param id - a unique identifier for the construct.
     */
    constructor(scope, id) {
        super(scope, id);
        this._moveToGlacierDeepArchive = false;
        this._objectsExpireInDays = 3650;
        this._restrictToIpOrCidrs = [];
        this._restrictToVpcIds = [];
        this.id = id;
        this.scope = scope;
    }
    /**
     * The underlying S3 bucket created by this construct.
     */
    get bucket() {
        return this._bucket;
    }
    /**
     * Adds access restrictions so that the access is allowed from the following IP ranges
     * @param ipsOrCidrs
     */
    restrictAccessToIpOrCidrs(ipsOrCidrs) {
        this._restrictToIpOrCidrs.push(...ipsOrCidrs);
        return this;
    }
    /**
     * Adds access restrictions so that the access is allowed from the following VPCs
     * @param vpcIds
     */
    restrictAccessToVpcs(vpcIds) {
        this._restrictToVpcIds.push(...vpcIds);
        return this;
    }
    /**
     * The name of the bucket. Internally the bucket name will be modified to include the account and region.
     * @param name - the name of the bucket to use
     */
    bucketName(name) {
        this._bucketName = name;
        return this;
    }
    /**
     * Use only for buckets that have archiving data.
     * CAUTION, once the object is archived, a temporary bucket to store the data.
     * @default false
     * @returns SecureBucket
     */
    moveToGlacierDeepArchive(move) {
        this._moveToGlacierDeepArchive = move !== null && move !== void 0 ? move : false;
        return this;
    }
    /**
     * The number of days that object will be kept.
     * @default 3650 - 10 years
     * @returns SecureBucket
     */
    objectsExpireInDays(expiryInDays) {
        this._objectsExpireInDays = expiryInDays;
        return this;
    }
    /**
     * Adds restriction to the bucket based on IP/CIDRs or VPC IDs specified.
     * @param bucket - the bucket
     */
    addRestrictionPolicy(bucket) {
        let conditions = {};
        if (this._restrictToIpOrCidrs.length > 0) {
            conditions.NotIpAddress = { 'aws:SourceIp': this._restrictToIpOrCidrs };
        }
        if (this._restrictToVpcIds.length > 0) {
            conditions.StringNotEquals = { 'aws:SourceVpce': this._restrictToVpcIds };
        }
        if (_.isEmpty(conditions))
            return;
        bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.DENY,
            principals: [new aws_iam_1.StarPrincipal()],
            actions: ['s3:*'],
            resources: [`${bucket.bucketArn}/*`],
            conditions: conditions,
        }));
    }
    /**
     * This function allows users to override the defaults calculated by this construct and is only recommended for advanced usecases.
     * The values supplied via props superseeds the defaults that are calculated.
     * @param props - The customized set of properties
     * @returns SecureBucket
     */
    overrideBucketProperties(props) {
        var _b, _c;
        // canonicalize the bucket name
        let currentScope = this.scope;
        const stack = aws_cdk_lib_1.Stack.of(currentScope);
        let bucketName = utils_1.Utils.appendIfNecessary(_.toLower(this._bucketName), stack.account, stack.region);
        // create the defaults
        let encryption = aws_s3_1.BucketEncryption.S3_MANAGED;
        let versioned = true;
        let enforceSSL = true;
        let publicReadAccess = false;
        let objectsExpireInDays = (_b = this._objectsExpireInDays) !== null && _b !== void 0 ? _b : 3650; // 10 years
        let moveToGlacierDeepArchive = (_c = this._moveToGlacierDeepArchive) !== null && _c !== void 0 ? _c : false;
        // block access if necessary
        let blockPublicAccess = (!publicReadAccess ? aws_s3_1.BlockPublicAccess.BLOCK_ALL : undefined);
        // will add transitions if expiry set on object is >90 days.
        let transitions = [];
        if (objectsExpireInDays >= 60) {
            transitions.push({
                storageClass: aws_s3_1.StorageClass.INFREQUENT_ACCESS,
                transitionAfter: aws_cdk_lib_1.Duration.days(30),
            });
        }
        if (objectsExpireInDays >= 90) {
            transitions.push({
                storageClass: aws_s3_1.StorageClass.INTELLIGENT_TIERING,
                transitionAfter: aws_cdk_lib_1.Duration.days(60),
            });
        }
        if (moveToGlacierDeepArchive && objectsExpireInDays >= 365) {
            transitions.push({
                storageClass: aws_s3_1.StorageClass.DEEP_ARCHIVE,
                transitionAfter: aws_cdk_lib_1.Duration.days(90),
            });
        }
        let lifecycleRules = [
            {
                expiration: aws_cdk_lib_1.Duration.days(objectsExpireInDays),
                abortIncompleteMultipartUploadAfter: aws_cdk_lib_1.Duration.days(30),
                transitions: transitions,
            },
        ];
        // override bucket props with defaults
        let bucketProps = Object.assign({}, {
            bucketName,
            encryption,
            versioned,
            enforceSSL,
            publicReadAccess,
            blockPublicAccess,
            lifecycleRules,
        }, props);
        this._props = bucketProps;
        return this;
    }
    /**
     * Creates the underlying S3 bucket.
     */
    assemble() {
        if (this._props === undefined) {
            this.overrideBucketProperties({});
        }
        let newBucket = new aws_s3_1.Bucket(this, 'Bucket', this._props);
        // add restrictions as needed.
        this.addRestrictionPolicy(newBucket);
        this._bucket = newBucket;
        // disable s3 access log enfocement
        utils_1.Utils.suppressNagRule(this._bucket, 'AwsSolutions-S1');
        return this;
    }
}
exports.SecureBucket = SecureBucket;
_a = JSII_RTTI_SYMBOL_1;
SecureBucket[_a] = { fqn: "ez-constructs.SecureBucket", version: "0.0.20" };
//# sourceMappingURL=data:application/json;base64,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