import { Bucket, BucketProps } from 'aws-cdk-lib/aws-s3';
import { Construct } from 'constructs';
import { EzConstruct } from '../ez-construct';
/**
 * Will create a secure bucket with the following features:
 * - Bucket name will be modified to include account and region.
 * - Access limited to the owner
 * - Object Versioning
 * - Encryption at rest
 * - Object expiration max limit to 10 years
 * - Object will transition to IA after 60 days and later to deep archive after 365 days
 *
 * Example:
 *
 * ```ts
 *    let aBucket = new SecureBucket(mystack, 'secureBucket', {
 *      bucketName: 'mybucket',
 *      objectsExpireInDays: 500,
 *      enforceSSL: false,
 *     });
 * ```
 */
export declare class SecureBucket extends EzConstruct {
    private _bucket;
    private _bucketName;
    private _props;
    private _moveToGlacierDeepArchive;
    private _objectsExpireInDays;
    private readonly scope;
    private readonly id;
    private _restrictToIpOrCidrs;
    private _restrictToVpcIds;
    /**
     * Creates the SecureBucket
     * @param scope - the stack in which the construct is defined.
     * @param id - a unique identifier for the construct.
     */
    constructor(scope: Construct, id: string);
    /**
     * The underlying S3 bucket created by this construct.
     */
    get bucket(): Bucket | undefined;
    /**
     * Adds access restrictions so that the access is allowed from the following IP ranges
     * @param ipsOrCidrs
     */
    restrictAccessToIpOrCidrs(ipsOrCidrs: Array<string>): SecureBucket;
    /**
     * Adds access restrictions so that the access is allowed from the following VPCs
     * @param vpcIds
     */
    restrictAccessToVpcs(vpcIds: Array<string>): SecureBucket;
    /**
     * The name of the bucket. Internally the bucket name will be modified to include the account and region.
     * @param name - the name of the bucket to use
     */
    bucketName(name: string): SecureBucket;
    /**
     * Use only for buckets that have archiving data.
     * CAUTION, once the object is archived, a temporary bucket to store the data.
     * @default false
     * @returns SecureBucket
     */
    moveToGlacierDeepArchive(move?: boolean): SecureBucket;
    /**
     * The number of days that object will be kept.
     * @default 3650 - 10 years
     * @returns SecureBucket
     */
    objectsExpireInDays(expiryInDays: number): SecureBucket;
    /**
     * Adds restriction to the bucket based on IP/CIDRs or VPC IDs specified.
     * @param bucket - the bucket
     */
    private addRestrictionPolicy;
    /**
     * This function allows users to override the defaults calculated by this construct and is only recommended for advanced usecases.
     * The values supplied via props superseeds the defaults that are calculated.
     * @param props - The customized set of properties
     * @returns SecureBucket
     */
    overrideBucketProperties(props: BucketProps): SecureBucket;
    /**
     * Creates the underlying S3 bucket.
     */
    assemble(): SecureBucket;
}
//# sourceMappingURL=index.d.ts.map