"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Utils = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const assertions_1 = require("aws-cdk-lib/assertions");
const cdk_nag_1 = require("cdk-nag");
const _ = require("lodash");
/**
 * A utility class that have common functions.
 */
class Utils {
    /**
     * Will append the suffix to the given name if the name do not contain the suffix
     * @param name - a string
     * @param suffixes - the string to append
     * @returns the name with the suffix appended if necessary delimited by a hyphen
     */
    static appendIfNecessary(name, ...suffixes) {
        let newName = name;
        suffixes.forEach(suffix => {
            newName = _.includes(newName, suffix) ? newName : `${newName}-${suffix}`;
        });
        return newName;
    }
    /**
     * Will wrap the given string using the given delimiter.
     * @param str - the string to wrap
     * @param delimiter - the delimiter to use
     * @returns the wrapped string
     */
    static wrap(str, delimiter) {
        let newStr = str;
        if (!Utils.startsWith(str, delimiter))
            newStr = `${delimiter}${newStr}`;
        if (!Utils.endsWith(str, delimiter))
            newStr = `${newStr}${delimiter}`;
        return newStr;
    }
    /**
     * Will check if the given string starts with the given prefix.
     * @param str - a string
     * @param s - the prefix to check
     */
    static startsWith(str, s) {
        return _.startsWith(str, s);
    }
    /**
     * Will check if the given string ends with the given suffix.
     * @param str - a string
     * @param s - suffix to check
     */
    static endsWith(str, s) {
        return _.endsWith(str, s);
    }
    /**
     * Will check if the given object is empty
     * @param value
     */
    static isEmpty(value) {
        return _.isEmpty(value);
    }
    /**
     * Will convert the given string to lower case and transform any spaces to hyphens
     * @param str - a string
     */
    static kebabCase(str) {
        return _.kebabCase(_.toLower(str));
    }
    /**
     * Splits a given Github URL and extracts the owner and repo name
     * @param url
     */
    static parseGithubUrl(url) {
        if (Utils.isEmpty(url))
            throw new TypeError('Invalid URL');
        if (!(Utils.startsWith(url, 'https://github.cms.gov/') || Utils.startsWith(url, 'https://github.com')))
            throw new TypeError('Invalid URL');
        if (!Utils.endsWith(url, '.git'))
            throw new TypeError('Invalid URL');
        // find the details from url
        let u = new URL(url);
        let owner = u.pathname.split('/')[1];
        let repo = _.replace(u.pathname.split('/')[2], '.git', '');
        let enterprise = u.hostname == 'github.cms.gov';
        let github = u.hostname == 'github.com';
        if (_.isEmpty(owner) || _.isEmpty(repo))
            throw new TypeError('Invalid URL');
        return {
            owner,
            repo,
            github,
            enterprise,
        };
    }
    /**
     * A utility function that will print the content of a CDK stack.
     * @warning This function is only used for debugging purpose.
     * @param stack - a valid stack
     */
    static prettyPrintStack(stack, persist = true) {
        let t = assertions_1.Template.fromStack(stack);
        console.log(JSON.stringify(t.toJSON(), null, 2));
        if (persist) {
            fs.writeFileSync(`tmp/${stack.stackName}.json`, JSON.stringify(t.toJSON(), null, 2));
        }
    }
    /**
     * Will disable the CDK NAG rule for the given construct and its children.
     * @param scope - the scope to disable the rule for
     * @param ruleId - the rule id to disable
     * @param reason - reason for disabling the rule.
     */
    static suppressNagRule(scope, ruleId, reason) {
        cdk_nag_1.NagSuppressions.addResourceSuppressions(scope, [{
                id: ruleId,
                reason: reason || `${ruleId} is not needed in this context (${scope.node.id}).`,
            }], true);
    }
}
exports.Utils = Utils;
_a = JSII_RTTI_SYMBOL_1;
Utils[_a] = { fqn: "ez-constructs.Utils", version: "0.0.20" };
//# sourceMappingURL=data:application/json;base64,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