"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SimpleCodebuildProject = exports.GitEvent = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const ez_construct_1 = require("../ez-construct");
const utils_1 = require("../lib/utils");
const secure_bucket_1 = require("../secure-bucket");
/**
 * The Github events which should trigger this build.
 */
var GitEvent;
(function (GitEvent) {
    GitEvent["PULL_REQUEST"] = "pull_request";
    GitEvent["PUSH"] = "push";
    GitEvent["ALL"] = "all";
})(GitEvent = exports.GitEvent || (exports.GitEvent = {}));
/**
 * Most of the cases,a developer will use CodeBuild setup to perform simple CI tasks such as:
 * - Build and test your code on a PR
 * - Run a specific script based on a cron schedule.
 * Also, they might want:
 * - artifacts like testcase reports to be available via Reports UI and/or S3.
 * - logs to be available via CloudWatch Logs.
 *
 * However, there can be additional organizational retention policies, for example retaining logs for a particular period of time.
 * With this construct, you can easily create a basic CodeBuild project with many opinated defaults that are compliant with FISMA and NIST.
 *
 * Example, creates a project named `my-project`, with artifacts going to my-project-artifacts-<accountId>-<region>
 *  and logs going to `/aws/codebuild/my-project` log group with a retention period of 90 days and 14 months respectively.
 *
 * ```ts
 *    new SimpleCodebuildProject(stack, 'MyProject')
 *      .projectName('myproject')
 *      .gitRepoUrl('https://github.com/bijujoseph/cloudbiolinux.git')
 *      .gitBaseBranch('main')
 *      .triggerEvent(GitEvent.PULL_REQUEST)
 *      .buildSpecPath('buildspecs/my-pr-checker.yml')
 *      .assemble();
 * ```
 *
 */
class SimpleCodebuildProject extends ez_construct_1.EzConstruct {
    constructor(scope, id) {
        super(scope, id);
        this._gitBaseBranch = 'develop';
        this._grantReportGroupPermissions = true;
        this._privileged = false;
        this._computType = aws_codebuild_1.ComputeType.MEDIUM;
        this._envVariables = {};
        this.scope = scope;
        this.id = id;
    }
    /**
     * The underlying codebuild project that is created by this construct.
     */
    get project() {
        return this._project;
    }
    /**
     * A convenient way to set the project environment variables.
     * The values set here will be presnted on the UI when build with overriding is used.
     * @param name - The environment variable name
     * @param envVar - The environment variable value
     *Example:
     *
     * ```ts
     *  project
     *    .addEnvironmentVariable('MY_ENV_VAR', {value: 'abcd})
     *    .addEnvironmentVariable('MY_ENV_VAR', {value: '/dev/thatkey, type: BuildEnvironmentVariableType.PARAMETER_STORE})
     *    .addEnvironmentVariable('MY_ENV_VAR', {value: 'arn:of:secret, type: BuildEnvironmentVariableType.SECRETS_MANAGER});
     * ```
     */
    addEnv(name, envVar) {
        this._envVariables[name] = envVar;
        return this;
    }
    /**
     * The name of the codebuild project
     * @param projectName - a valid name string
     */
    projectName(projectName) {
        this._projectName = projectName;
        return this;
    }
    /**
     * The description of the codebuild project
     * @param projectDescription - a valid description string
     */
    projectDescription(projectDescription) {
        this._projectDescription = projectDescription;
        return this;
    }
    /**
     * The compute type to use
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-compute-types.html
     * @param computeType
     */
    computeType(computeType) {
        this._computType = computeType;
        return this;
    }
    /**
     * Set privileged mode of execution. Usually needed if this project builds Docker images,
     * and the build environment image you chose is not provided by CodeBuild with Docker support.
     * By default, Docker containers do not allow access to any devices.
     * Privileged mode grants a build project's Docker container access to all devices
     * https://docs.aws.amazon.com/codebuild/latest/userguide/change-project-console.html#change-project-console-environment
     * @param p - true/false
     */
    privileged(p) {
        this._privileged = p;
        return this;
    }
    /**
     * The build spec file path
     * @param buildSpecPath - relative location of the build spec file
     */
    buildSpecPath(buildSpecPath) {
        this._buildSpecPath = buildSpecPath;
        return this;
    }
    /**
     * The github or enterprise github repository url
     * @param gitRepoUrl
     */
    gitRepoUrl(gitRepoUrl) {
        this._gitRepoUrl = gitRepoUrl;
        return this;
    }
    /**
     * The main branch of the github project.
     * @param branch
     */
    gitBaseBranch(branch) {
        this._gitBaseBranch = branch;
        return this;
    }
    /**
     * The Github events that can trigger this build.
     * @param event
     */
    triggerBuildOnGitEvent(event) {
        this._triggerOnGitEvent = event;
        return this;
    }
    /**
     * The cron schedule on which this build gets triggerd.
     * @param schedule
     */
    triggerBuildOnSchedule(schedule) {
        this._triggerOnSchedule = schedule;
        return this;
    }
    /**
     * The name of the bucket to store the artifacts.
     * By default the buckets will get stored in `<project-name>-artifacts` bucket.
     * This function can be used to ovrride the default behavior.
     * @param artifactBucket - a valid existing Bucket reference or bucket name to use.
     */
    artifactBucket(artifactBucket) {
        this._artifactBucket = artifactBucket;
        return this;
    }
    overrideProjectProps(props) {
        var _b;
        let projectName = this._projectName ? this._projectName : utils_1.Utils.kebabCase(this.id);
        let description = this._projectDescription ? this._projectDescription : `Codebuild description for ${projectName}`;
        let defaults = {
            projectName,
            description,
            grantReportGroupPermissions: this._grantReportGroupPermissions,
        };
        // set the default environment if not provided.
        if (utils_1.Utils.isEmpty(props.environment)) {
            // @ts-ignore
            defaults.environment = {
                buildImage: aws_codebuild_1.LinuxBuildImage.STANDARD_5_0,
                privileged: this._privileged,
                computeType: this._computType,
                environmentVariables: this._envVariables,
            };
        }
        // add default logging if not provided
        if (utils_1.Utils.isEmpty(props.logging)) {
            // @ts-ignore
            defaults.logging = {
                cloudWatch: {
                    logGroup: new aws_logs_1.LogGroup(this, 'ProjectLogGroup', {
                        logGroupName: `/aws/codebuild/${utils_1.Utils.kebabCase(projectName)}`,
                        retention: aws_logs_1.RetentionDays.THIRTEEN_MONTHS,
                    }),
                },
            };
        }
        // create source if not provided in props
        if (utils_1.Utils.isEmpty(props.source)) {
            // @ts-ignore
            defaults.source = this.createSource(this._gitRepoUrl, this._gitBaseBranch, this._triggerOnGitEvent);
        }
        // create artifact bucket if one was not provided
        if (utils_1.Utils.isEmpty(props.artifacts)) {
            // @ts-ignore
            defaults.artifacts = this.createArtifacts((_b = this._artifactBucket) !== null && _b !== void 0 ? _b : `${this._projectName}-artifacts`);
        }
        // create the build spec if one was not provided
        if (utils_1.Utils.isEmpty(props.buildSpec) && !utils_1.Utils.isEmpty(this._buildSpecPath)) {
            // @ts-ignore
            defaults.buildSpec = this.createBuildSpec(this._buildSpecPath);
        }
        this._props = Object.assign({}, defaults, props);
        return this;
    }
    assemble(defaultProps) {
        // create the default project properties
        this.overrideProjectProps(defaultProps !== null && defaultProps !== void 0 ? defaultProps : {});
        // create a codebuild project
        let project = new aws_codebuild_1.Project(this.scope, 'Project', this._props);
        // run above project on a schedule ?
        if (this._triggerOnSchedule) {
            new aws_events_1.Rule(this.scope, 'ScheduleRule', {
                schedule: this._triggerOnSchedule,
                targets: [new aws_events_targets_1.CodeBuildProject(project)],
            });
        }
        this._project = project;
        utils_1.Utils.suppressNagRule(this._project, 'AwsSolutions-CB4', 'Artifacts produced by this project are encrypted using `aws/s3` key, and also at rest by S3.');
        utils_1.Utils.suppressNagRule(this._project, 'AwsSolutions-IAM5', 'There is an artifact bucket per project. The wild card IAM policies are scoped to this project and only used to manage artifacts produced by the project.');
        return this;
    }
    /**
     * Create an S3 bucket for storing artifacts produced by the codebuild project
     * @param bucketName - the s3 bucket
     * @private
     */
    createBucket(bucketName) {
        return new secure_bucket_1.SecureBucket(this, 'ProjectArtifactBucket')
            .bucketName(bucketName)
            .objectsExpireInDays(90)
            .assemble()
            .bucket;
    }
    /**
     * Creates an S3 artifact store for storing the objects produced by the codebuild project
     * @param artifactBucket - a bucket object or bucket name
     * @private
     */
    createArtifacts(artifactBucket) {
        let theBucket = (typeof artifactBucket === 'string') ? this.createBucket(artifactBucket) : artifactBucket;
        return aws_codebuild_1.Artifacts.s3({
            bucket: theBucket,
            includeBuildId: true,
            packageZip: true,
        });
    }
    /**
     * Creates a Github or Enterprise Githb repo source object
     * @param repoUrl - the url of the repo
     * @param base - the main or base branch used by the repo
     * @param gitEvent - the github events that can trigger builds
     * @private
     */
    createSource(repoUrl, base, gitEvent) {
        let webhook = gitEvent && true;
        let repoDetails = utils_1.Utils.parseGithubUrl(repoUrl);
        let webhookFilter = this.createWebHookFilters(base, gitEvent);
        if (repoDetails.enterprise == true) {
            return aws_codebuild_1.Source.gitHubEnterprise({
                httpsCloneUrl: repoUrl,
                webhook,
                webhookFilters: webhookFilter,
            });
        }
        return aws_codebuild_1.Source.gitHub({
            owner: repoDetails.owner,
            repo: repoDetails.repo,
            webhook,
            webhookFilters: webhookFilter,
        });
    }
    /**
     * Creates a webhook filter object
     * @param base - the base branch
     * @param gitEvent - the github event
     * @private
     */
    createWebHookFilters(base, gitEvent) {
        // @ts-ignore
        let fg = null;
        if (!gitEvent)
            return undefined;
        if (gitEvent == GitEvent.PULL_REQUEST) {
            fg = aws_codebuild_1.FilterGroup.inEventOf(aws_codebuild_1.EventAction.PULL_REQUEST_CREATED, aws_codebuild_1.EventAction.PULL_REQUEST_UPDATED, aws_codebuild_1.EventAction.PULL_REQUEST_REOPENED);
            if (base) {
                fg = fg.andBaseBranchIs(base);
            }
        }
        if (gitEvent == GitEvent.PUSH) {
            fg = aws_codebuild_1.FilterGroup.inEventOf(aws_codebuild_1.EventAction.PUSH);
        }
        if (gitEvent == GitEvent.ALL) {
            fg = aws_codebuild_1.FilterGroup.inEventOf(aws_codebuild_1.EventAction.PUSH, aws_codebuild_1.EventAction.PULL_REQUEST_CREATED, aws_codebuild_1.EventAction.PULL_REQUEST_UPDATED, aws_codebuild_1.EventAction.PULL_REQUEST_REOPENED, aws_codebuild_1.EventAction.PULL_REQUEST_MERGED);
        }
        return [fg];
    }
    /**
     * Loads the build spec associated with the given codebuild project
     * @param buildSpecPath - location of the build spec file.
     * @private
     */
    createBuildSpec(buildSpecPath) {
        return aws_codebuild_1.BuildSpec.fromSourceFilename(buildSpecPath);
    }
}
exports.SimpleCodebuildProject = SimpleCodebuildProject;
_a = JSII_RTTI_SYMBOL_1;
SimpleCodebuildProject[_a] = { fqn: "ez-constructs.SimpleCodebuildProject", version: "0.0.20" };
//# sourceMappingURL=data:application/json;base64,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