import { ComputeType, Project, ProjectProps } from 'aws-cdk-lib/aws-codebuild';
import { BuildEnvironmentVariable } from 'aws-cdk-lib/aws-codebuild/lib/project';
import { Schedule } from 'aws-cdk-lib/aws-events';
import { IBucket } from 'aws-cdk-lib/aws-s3';
import { Construct } from 'constructs';
import { EzConstruct } from '../ez-construct';
/**
 * The Github events which should trigger this build.
 */
export declare enum GitEvent {
    PULL_REQUEST = "pull_request",
    PUSH = "push",
    ALL = "all"
}
/**
 * Most of the cases,a developer will use CodeBuild setup to perform simple CI tasks such as:
 * - Build and test your code on a PR
 * - Run a specific script based on a cron schedule.
 * Also, they might want:
 * - artifacts like testcase reports to be available via Reports UI and/or S3.
 * - logs to be available via CloudWatch Logs.
 *
 * However, there can be additional organizational retention policies, for example retaining logs for a particular period of time.
 * With this construct, you can easily create a basic CodeBuild project with many opinated defaults that are compliant with FISMA and NIST.
 *
 * Example, creates a project named `my-project`, with artifacts going to my-project-artifacts-<accountId>-<region>
 *  and logs going to `/aws/codebuild/my-project` log group with a retention period of 90 days and 14 months respectively.
 *
 * ```ts
 *    new SimpleCodebuildProject(stack, 'MyProject')
 *      .projectName('myproject')
 *      .gitRepoUrl('https://github.com/bijujoseph/cloudbiolinux.git')
 *      .gitBaseBranch('main')
 *      .triggerEvent(GitEvent.PULL_REQUEST)
 *      .buildSpecPath('buildspecs/my-pr-checker.yml')
 *      .assemble();
 * ```
 *
 */
export declare class SimpleCodebuildProject extends EzConstruct {
    private _project;
    private _projectName?;
    private _projectDescription?;
    private _gitRepoUrl?;
    private _gitBaseBranch;
    private _buildSpecPath?;
    private _grantReportGroupPermissions;
    private _privileged;
    private _triggerOnGitEvent?;
    private _triggerOnSchedule?;
    private _artifactBucket?;
    private _computType;
    private _envVariables;
    private _props;
    private readonly scope;
    private readonly id;
    constructor(scope: Construct, id: string);
    /**
     * The underlying codebuild project that is created by this construct.
     */
    get project(): Project | undefined;
    /**
     * A convenient way to set the project environment variables.
     * The values set here will be presnted on the UI when build with overriding is used.
     * @param name - The environment variable name
     * @param envVar - The environment variable value
     *Example:
     *
     * ```ts
     *  project
     *    .addEnvironmentVariable('MY_ENV_VAR', {value: 'abcd})
     *    .addEnvironmentVariable('MY_ENV_VAR', {value: '/dev/thatkey, type: BuildEnvironmentVariableType.PARAMETER_STORE})
     *    .addEnvironmentVariable('MY_ENV_VAR', {value: 'arn:of:secret, type: BuildEnvironmentVariableType.SECRETS_MANAGER});
     * ```
     */
    addEnv(name: string, envVar: BuildEnvironmentVariable): SimpleCodebuildProject;
    /**
     * The name of the codebuild project
     * @param projectName - a valid name string
     */
    projectName(projectName: string): SimpleCodebuildProject;
    /**
     * The description of the codebuild project
     * @param projectDescription - a valid description string
     */
    projectDescription(projectDescription: string): SimpleCodebuildProject;
    /**
     * The compute type to use
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-compute-types.html
     * @param computeType
     */
    computeType(computeType: ComputeType): SimpleCodebuildProject;
    /**
     * Set privileged mode of execution. Usually needed if this project builds Docker images,
     * and the build environment image you chose is not provided by CodeBuild with Docker support.
     * By default, Docker containers do not allow access to any devices.
     * Privileged mode grants a build project's Docker container access to all devices
     * https://docs.aws.amazon.com/codebuild/latest/userguide/change-project-console.html#change-project-console-environment
     * @param p - true/false
     */
    privileged(p: boolean): SimpleCodebuildProject;
    /**
     * The build spec file path
     * @param buildSpecPath - relative location of the build spec file
     */
    buildSpecPath(buildSpecPath: string): SimpleCodebuildProject;
    /**
     * The github or enterprise github repository url
     * @param gitRepoUrl
     */
    gitRepoUrl(gitRepoUrl: string): SimpleCodebuildProject;
    /**
     * The main branch of the github project.
     * @param branch
     */
    gitBaseBranch(branch: string): SimpleCodebuildProject;
    /**
     * The Github events that can trigger this build.
     * @param event
     */
    triggerBuildOnGitEvent(event: GitEvent): SimpleCodebuildProject;
    /**
     * The cron schedule on which this build gets triggerd.
     * @param schedule
     */
    triggerBuildOnSchedule(schedule: Schedule): SimpleCodebuildProject;
    /**
     * The name of the bucket to store the artifacts.
     * By default the buckets will get stored in `<project-name>-artifacts` bucket.
     * This function can be used to ovrride the default behavior.
     * @param artifactBucket - a valid existing Bucket reference or bucket name to use.
     */
    artifactBucket(artifactBucket: IBucket | string): SimpleCodebuildProject;
    overrideProjectProps(props: ProjectProps): SimpleCodebuildProject;
    assemble(defaultProps?: ProjectProps): SimpleCodebuildProject;
    /**
     * Create an S3 bucket for storing artifacts produced by the codebuild project
     * @param bucketName - the s3 bucket
     * @private
     */
    private createBucket;
    /**
     * Creates an S3 artifact store for storing the objects produced by the codebuild project
     * @param artifactBucket - a bucket object or bucket name
     * @private
     */
    private createArtifacts;
    /**
     * Creates a Github or Enterprise Githb repo source object
     * @param repoUrl - the url of the repo
     * @param base - the main or base branch used by the repo
     * @param gitEvent - the github events that can trigger builds
     * @private
     */
    private createSource;
    /**
     * Creates a webhook filter object
     * @param base - the base branch
     * @param gitEvent - the github event
     * @private
     */
    private createWebHookFilters;
    /**
     * Loads the build spec associated with the given codebuild project
     * @param buildSpecPath - location of the build spec file.
     * @private
     */
    private createBuildSpec;
}
//# sourceMappingURL=index.d.ts.map