"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecurityGroup = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const peer_1 = require("./peer");
const port_1 = require("./port");
const SECURITY_GROUP_SYMBOL = Symbol.for('@aws-cdk/iam.SecurityGroup');
const SECURITY_GROUP_DISABLE_INLINE_RULES_CONTEXT_KEY = '@aws-cdk/aws-ec2.securityGroupDisableInlineRules';
/**
 * A SecurityGroup that is not created in this template
 */
class SecurityGroupBase extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, props);
        /**
         * Whether the rule can be inlined into a SecurityGroup or not.
         *
         * @stability stable
         */
        this.canInlineRule = false;
        /**
         * The network connections associated with this resource.
         *
         * @stability stable
         */
        this.connections = new connections_1.Connections({ securityGroups: [this] });
        this.peerAsTokenCount = 0;
        Object.defineProperty(this, SECURITY_GROUP_SYMBOL, { value: true });
    }
    /**
     * Return whether the indicated object is a security group.
     *
     * @stability stable
     */
    static isSecurityGroup(x) {
        return SECURITY_GROUP_SYMBOL in x;
    }
    /**
     * A unique identifier for this connection peer.
     *
     * @stability stable
     */
    get uniqueId() {
        return core_1.Names.nodeUniqueId(this.node);
    }
    addIngressRule(peer, connection, description, remoteRule) {
        if (description === undefined) {
            description = `from ${peer.uniqueId}:${connection}`;
        }
        const [scope, id] = this.determineRuleScope(peer, connection, 'from', remoteRule);
        // Skip duplicates
        if (scope.node.tryFindChild(id) === undefined) {
            new ec2_generated_1.CfnSecurityGroupIngress(scope, id, {
                groupId: this.securityGroupId,
                ...peer.toIngressRuleConfig(),
                ...connection.toRuleJson(),
                description,
            });
        }
    }
    addEgressRule(peer, connection, description, remoteRule) {
        if (description === undefined) {
            description = `to ${peer.uniqueId}:${connection}`;
        }
        const [scope, id] = this.determineRuleScope(peer, connection, 'to', remoteRule);
        // Skip duplicates
        if (scope.node.tryFindChild(id) === undefined) {
            new ec2_generated_1.CfnSecurityGroupEgress(scope, id, {
                groupId: this.securityGroupId,
                ...peer.toEgressRuleConfig(),
                ...connection.toRuleJson(),
                description,
            });
        }
    }
    /**
     * Produce the ingress rule JSON for the given connection.
     *
     * @stability stable
     */
    toIngressRuleConfig() {
        return { sourceSecurityGroupId: this.securityGroupId };
    }
    /**
     * Produce the egress rule JSON for the given connection.
     *
     * @stability stable
     */
    toEgressRuleConfig() {
        return { destinationSecurityGroupId: this.securityGroupId };
    }
    /**
     * Determine where to parent a new ingress/egress rule.
     *
     * A SecurityGroup rule is parented under the group it's related to, UNLESS
     * we're in a cross-stack scenario with another Security Group. In that case,
     * we respect the 'remoteRule' flag and will parent under the other security
     * group.
     *
     * This is necessary to avoid cyclic dependencies between stacks, since both
     * ingress and egress rules will reference both security groups, and a naive
     * parenting will lead to the following situation:
     *
     *    ╔════════════════════╗         ╔════════════════════╗
     *    ║  ┌───────────┐     ║         ║    ┌───────────┐   ║
     *    ║  │  GroupA   │◀────╬─┐   ┌───╬───▶│  GroupB   │   ║
     *    ║  └───────────┘     ║ │   │   ║    └───────────┘   ║
     *    ║        ▲           ║ │   │   ║          ▲         ║
     *    ║        │           ║ │   │   ║          │         ║
     *    ║        │           ║ │   │   ║          │         ║
     *    ║  ┌───────────┐     ║ └───┼───╬────┌───────────┐   ║
     *    ║  │  EgressA  │─────╬─────┘   ║    │ IngressB  │   ║
     *    ║  └───────────┘     ║         ║    └───────────┘   ║
     *    ║                    ║         ║                    ║
     *    ╚════════════════════╝         ╚════════════════════╝
     *
     * By having the ability to switch the parent, we avoid the cyclic reference by
     * keeping all rules in a single stack.
     *
     * If this happens, we also have to change the construct ID, because
     * otherwise we might have two objects with the same ID if we have
     * multiple reversed security group relationships.
     *
     *    ╔═══════════════════════════════════╗
     *    ║┌───────────┐                      ║
     *    ║│  GroupB   │                      ║
     *    ║└───────────┘                      ║
     *    ║      ▲                            ║
     *    ║      │              ┌───────────┐ ║
     *    ║      ├────"from A"──│ IngressB  │ ║
     *    ║      │              └───────────┘ ║
     *    ║      │              ┌───────────┐ ║
     *    ║      ├─────"to B"───│  EgressA  │ ║
     *    ║      │              └───────────┘ ║
     *    ║      │              ┌───────────┐ ║
     *    ║      └─────"to B"───│  EgressC  │ ║  <-- oops
     *    ║                     └───────────┘ ║
     *    ╚═══════════════════════════════════╝
     *
     * @stability stable
     */
    determineRuleScope(peer, connection, fromTo, remoteRule) {
        if (remoteRule && SecurityGroupBase.isSecurityGroup(peer) && differentStacks(this, peer)) {
            // Reversed
            const reversedFromTo = fromTo === 'from' ? 'to' : 'from';
            return [peer, `${this.uniqueId}:${connection} ${reversedFromTo}`];
        }
        else {
            // Regular (do old ID escaping to in order to not disturb existing deployments)
            return [this, `${fromTo} ${this.renderPeer(peer)}:${connection}`.replace('/', '_')];
        }
    }
    renderPeer(peer) {
        if (core_1.Token.isUnresolved(peer.uniqueId)) {
            // Need to return a unique value each time a peer
            // is an unresolved token, else the duplicate skipper
            // in `sg.addXxxRule` can detect unique rules as duplicates
            return this.peerAsTokenCount++ ? `'{IndirectPeer${this.peerAsTokenCount}}'` : '{IndirectPeer}';
        }
        else {
            return peer.uniqueId;
        }
    }
}
function differentStacks(group1, group2) {
    return core_1.Stack.of(group1) !== core_1.Stack.of(group2);
}
/**
 * Creates an Amazon EC2 security group within a VPC.
 *
 * Security Groups act like a firewall with a set of rules, and are associated
 * with any AWS resource that has or creates Elastic Network Interfaces (ENIs).
 * A typical example of a resource that has a security group is an Instance (or
 * Auto Scaling Group of instances)
 *
 * If you are defining new infrastructure in CDK, there is a good chance you
 * won't have to interact with this class at all. Like IAM Roles, Security
 * Groups need to exist to control access between AWS resources, but CDK will
 * automatically generate and populate them with least-privilege permissions
 * for you so you can concentrate on your business logic.
 *
 * All Constructs that require Security Groups will create one for you if you
 * don't specify one at construction. After construction, you can selectively
 * allow connections to and between constructs via--for example-- the `instance.connections`
 * object. Think of it as "allowing connections to your instance", rather than
 * "adding ingress rules a security group". See the [Allowing
 * Connections](https://docs.aws.amazon.com/cdk/api/latest/docs/aws-ec2-readme.html#allowing-connections)
 * section in the library documentation for examples.
 *
 * Direct manipulation of the Security Group through `addIngressRule` and
 * `addEgressRule` is possible, but mutation through the `.connections` object
 * is recommended. If you peer two constructs with security groups this way,
 * appropriate rules will be created in both.
 *
 * If you have an existing security group you want to use in your CDK application,
 * you would import it like this:
 *
 * ```ts
 * const securityGroup = ec2.SecurityGroup.fromSecurityGroupId(this, 'SG', 'sg-12345', {
 *    mutable: false
 * });
 * ```
 *
 * @stability stable
 */
class SecurityGroup extends SecurityGroupBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.securityGroupName,
        });
        this.directIngressRules = [];
        this.directEgressRules = [];
        const groupDescription = props.description || this.node.path;
        this.allowAllOutbound = props.allowAllOutbound !== false;
        this.disableInlineRules = props.disableInlineRules !== undefined ?
            !!props.disableInlineRules :
            !!this.node.tryGetContext(SECURITY_GROUP_DISABLE_INLINE_RULES_CONTEXT_KEY);
        this.securityGroup = new ec2_generated_1.CfnSecurityGroup(this, 'Resource', {
            groupName: this.physicalName,
            groupDescription,
            securityGroupIngress: core_1.Lazy.any({ produce: () => this.directIngressRules }, { omitEmptyArray: true }),
            securityGroupEgress: core_1.Lazy.any({ produce: () => this.directEgressRules }, { omitEmptyArray: true }),
            vpcId: props.vpc.vpcId,
        });
        this.securityGroupId = this.securityGroup.attrGroupId;
        this.securityGroupVpcId = this.securityGroup.attrVpcId;
        this.securityGroupName = this.securityGroup.ref;
        this.addDefaultEgressRule();
    }
    /**
     * Look up a security group by id.
     *
     * @stability stable
     */
    static fromLookup(scope, id, securityGroupId) {
        if (core_1.Token.isUnresolved(securityGroupId)) {
            throw new Error('All arguments to look up a security group must be concrete (no Tokens)');
        }
        const attributes = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.SECURITY_GROUP_PROVIDER,
            props: { securityGroupId },
            dummyValue: {
                securityGroupId: 'sg-12345',
                allowAllOutbound: true,
            },
        }).value;
        return SecurityGroup.fromSecurityGroupId(scope, id, attributes.securityGroupId, {
            allowAllOutbound: attributes.allowAllOutbound,
            mutable: true,
        });
    }
    /**
     * Import an existing security group into this app.
     *
     * This method will assume that the Security Group has a rule in it which allows
     * all outbound traffic, and so will not add egress rules to the imported Security
     * Group (only ingress rules).
     *
     * If your existing Security Group needs to have egress rules added, pass the
     * `allowAllOutbound: false` option on import.
     *
     * @stability stable
     */
    static fromSecurityGroupId(scope, id, securityGroupId, options = {}) {
        class MutableImport extends SecurityGroupBase {
            constructor() {
                var _b;
                super(...arguments);
                this.securityGroupId = securityGroupId;
                this.allowAllOutbound = (_b = options.allowAllOutbound) !== null && _b !== void 0 ? _b : true;
            }
            addEgressRule(peer, connection, description, remoteRule) {
                // Only if allowAllOutbound has been disabled
                if (options.allowAllOutbound === false) {
                    super.addEgressRule(peer, connection, description, remoteRule);
                }
            }
        }
        class ImmutableImport extends SecurityGroupBase {
            constructor() {
                var _b;
                super(...arguments);
                this.securityGroupId = securityGroupId;
                this.allowAllOutbound = (_b = options.allowAllOutbound) !== null && _b !== void 0 ? _b : true;
            }
            addEgressRule(_peer, _connection, _description, _remoteRule) {
                // do nothing
            }
            addIngressRule(_peer, _connection, _description, _remoteRule) {
                // do nothing
            }
        }
        return options.mutable !== false
            ? new MutableImport(scope, id)
            : new ImmutableImport(scope, id);
    }
    /**
     * Add an ingress rule for the current security group.
     *
     * `remoteRule` controls where the Rule object is created if the peer is also a
     * securityGroup and they are in different stack. If false (default) the
     * rule object is created under the current SecurityGroup object. If true and the
     * peer is also a SecurityGroup, the rule object is created under the remote
     * SecurityGroup object.
     *
     * @stability stable
     */
    addIngressRule(peer, connection, description, remoteRule) {
        if (!peer.canInlineRule || !connection.canInlineRule || this.disableInlineRules) {
            super.addIngressRule(peer, connection, description, remoteRule);
            return;
        }
        if (description === undefined) {
            description = `from ${peer.uniqueId}:${connection}`;
        }
        this.addDirectIngressRule({
            ...peer.toIngressRuleConfig(),
            ...connection.toRuleJson(),
            description,
        });
    }
    /**
     * Add an egress rule for the current security group.
     *
     * `remoteRule` controls where the Rule object is created if the peer is also a
     * securityGroup and they are in different stack. If false (default) the
     * rule object is created under the current SecurityGroup object. If true and the
     * peer is also a SecurityGroup, the rule object is created under the remote
     * SecurityGroup object.
     *
     * @stability stable
     */
    addEgressRule(peer, connection, description, remoteRule) {
        if (this.allowAllOutbound) {
            // In the case of "allowAllOutbound", we don't add any more rules. There
            // is only one rule which allows all traffic and that subsumes any other
            // rule.
            if (!remoteRule) { // Warn only if addEgressRule() was explicitely called
                core_1.Annotations.of(this).addWarning('Ignoring Egress rule since \'allowAllOutbound\' is set to true; To add customize rules, set allowAllOutbound=false on the SecurityGroup');
            }
            return;
        }
        else {
            // Otherwise, if the bogus rule exists we can now remove it because the
            // presence of any other rule will get rid of EC2's implicit "all
            // outbound" rule anyway.
            this.removeNoTrafficRule();
        }
        if (!peer.canInlineRule || !connection.canInlineRule || this.disableInlineRules) {
            super.addEgressRule(peer, connection, description, remoteRule);
            return;
        }
        if (description === undefined) {
            description = `from ${peer.uniqueId}:${connection}`;
        }
        const rule = {
            ...peer.toEgressRuleConfig(),
            ...connection.toRuleJson(),
            description,
        };
        if (isAllTrafficRule(rule)) {
            // We cannot allow this; if someone adds the rule in this way, it will be
            // removed again if they add other rules. We also can't automatically switch
            // to "allOutbound=true" mode, because we might have already emitted
            // EgressRule objects (which count as rules added later) and there's no way
            // to recall those. Better to prevent this for now.
            throw new Error('Cannot add an "all traffic" egress rule in this way; set allowAllOutbound=true on the SecurityGroup instead.');
        }
        this.addDirectEgressRule(rule);
    }
    /**
     * Add a direct ingress rule
     */
    addDirectIngressRule(rule) {
        if (!this.hasIngressRule(rule)) {
            this.directIngressRules.push(rule);
        }
    }
    /**
     * Return whether the given ingress rule exists on the group
     */
    hasIngressRule(rule) {
        return this.directIngressRules.findIndex(r => ingressRulesEqual(r, rule)) > -1;
    }
    /**
     * Add a direct egress rule
     */
    addDirectEgressRule(rule) {
        if (!this.hasEgressRule(rule)) {
            this.directEgressRules.push(rule);
        }
    }
    /**
     * Return whether the given egress rule exists on the group
     */
    hasEgressRule(rule) {
        return this.directEgressRules.findIndex(r => egressRulesEqual(r, rule)) > -1;
    }
    /**
     * Add the default egress rule to the securityGroup
     *
     * This depends on allowAllOutbound:
     *
     * - If allowAllOutbound is true, we *TECHNICALLY* don't need to do anything, because
     *   EC2 is going to create this default rule anyway. But, for maximum readability
     *   of the template, we will add one anyway.
     * - If allowAllOutbound is false, we add a bogus rule that matches no traffic in
     *   order to get rid of the default "all outbound" rule that EC2 creates by default.
     *   If other rules happen to get added later, we remove the bogus rule again so
     *   that it doesn't clutter up the template too much (even though that's not
     *   strictly necessary).
     */
    addDefaultEgressRule() {
        if (this.disableInlineRules) {
            const peer = this.allowAllOutbound ? ALL_TRAFFIC_PEER : NO_TRAFFIC_PEER;
            const port = this.allowAllOutbound ? ALL_TRAFFIC_PORT : NO_TRAFFIC_PORT;
            const description = this.allowAllOutbound ? ALLOW_ALL_RULE.description : MATCH_NO_TRAFFIC.description;
            super.addEgressRule(peer, port, description, false);
        }
        else {
            const rule = this.allowAllOutbound ? ALLOW_ALL_RULE : MATCH_NO_TRAFFIC;
            this.directEgressRules.push(rule);
        }
    }
    /**
     * Remove the bogus rule if it exists
     */
    removeNoTrafficRule() {
        if (this.disableInlineRules) {
            const [scope, id] = this.determineRuleScope(NO_TRAFFIC_PEER, NO_TRAFFIC_PORT, 'to', false);
            scope.node.tryRemoveChild(id);
        }
        else {
            const i = this.directEgressRules.findIndex(r => egressRulesEqual(r, MATCH_NO_TRAFFIC));
            if (i > -1) {
                this.directEgressRules.splice(i, 1);
            }
        }
    }
}
exports.SecurityGroup = SecurityGroup;
_a = JSII_RTTI_SYMBOL_1;
SecurityGroup[_a] = { fqn: "@aws-cdk/aws-ec2.SecurityGroup", version: "1.132.0" };
/**
 * Egress rule that purposely matches no traffic
 *
 * This is used in order to disable the "all traffic" default of Security Groups.
 *
 * No machine can ever actually have the 255.255.255.255 IP address, but
 * in order to lock it down even more we'll restrict to a nonexistent
 * ICMP traffic type.
 */
const MATCH_NO_TRAFFIC = {
    cidrIp: '255.255.255.255/32',
    description: 'Disallow all traffic',
    ipProtocol: 'icmp',
    fromPort: 252,
    toPort: 86,
};
const NO_TRAFFIC_PEER = peer_1.Peer.ipv4(MATCH_NO_TRAFFIC.cidrIp);
const NO_TRAFFIC_PORT = port_1.Port.icmpTypeAndCode(MATCH_NO_TRAFFIC.fromPort, MATCH_NO_TRAFFIC.toPort);
/**
 * Egress rule that matches all traffic
 */
const ALLOW_ALL_RULE = {
    cidrIp: '0.0.0.0/0',
    description: 'Allow all outbound traffic by default',
    ipProtocol: '-1',
};
const ALL_TRAFFIC_PEER = peer_1.Peer.anyIpv4();
const ALL_TRAFFIC_PORT = port_1.Port.allTraffic();
/**
 * Compare two ingress rules for equality the same way CloudFormation would (discarding description)
 */
function ingressRulesEqual(a, b) {
    return a.cidrIp === b.cidrIp
        && a.cidrIpv6 === b.cidrIpv6
        && a.fromPort === b.fromPort
        && a.toPort === b.toPort
        && a.ipProtocol === b.ipProtocol
        && a.sourceSecurityGroupId === b.sourceSecurityGroupId
        && a.sourceSecurityGroupName === b.sourceSecurityGroupName
        && a.sourceSecurityGroupOwnerId === b.sourceSecurityGroupOwnerId;
}
/**
 * Compare two egress rules for equality the same way CloudFormation would (discarding description)
 */
function egressRulesEqual(a, b) {
    return a.cidrIp === b.cidrIp
        && a.cidrIpv6 === b.cidrIpv6
        && a.fromPort === b.fromPort
        && a.toPort === b.toPort
        && a.ipProtocol === b.ipProtocol
        && a.destinationPrefixListId === b.destinationPrefixListId
        && a.destinationSecurityGroupId === b.destinationSecurityGroupId;
}
/**
 * Whether this rule refers to all traffic
 */
function isAllTrafficRule(rule) {
    return rule.cidrIp === '0.0.0.0/0' && rule.ipProtocol === '-1';
}
//# sourceMappingURL=data:application/json;base64,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