import { UserData } from './user-data';
import { WindowsVersion } from './windows-versions';
import { Construct } from '@aws-cdk/core';
/**
 * Interface for classes that can select an appropriate machine image to use.
 *
 * @stability stable
 */
export interface IMachineImage {
    /**
     * Return the image to use in the given context.
     *
     * @stability stable
     */
    getImage(scope: Construct): MachineImageConfig;
}
/**
 * Factory functions for standard Amazon Machine Image objects.
 *
 * @stability stable
 */
export declare abstract class MachineImage {
    /**
     * A Windows image that is automatically kept up-to-date.
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     *
     * @stability stable
     */
    static latestWindows(version: WindowsVersion, props?: WindowsImageProps): IMachineImage;
    /**
     * An Amazon Linux image that is automatically kept up-to-date.
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     *
     * @stability stable
     */
    static latestAmazonLinux(props?: AmazonLinuxImageProps): IMachineImage;
    /**
     * A Linux image where you specify the AMI ID for every region.
     *
     * @param amiMap For every region where you are deploying the stack, specify the AMI ID for that region.
     * @param props Customize the image by supplying additional props.
     * @stability stable
     */
    static genericLinux(amiMap: Record<string, string>, props?: GenericLinuxImageProps): IMachineImage;
    /**
     * A Windows image where you specify the AMI ID for every region.
     *
     * @param amiMap For every region where you are deploying the stack, specify the AMI ID for that region.
     * @param props Customize the image by supplying additional props.
     * @stability stable
     */
    static genericWindows(amiMap: Record<string, string>, props?: GenericWindowsImageProps): IMachineImage;
    /**
     * (deprecated) An image specified in SSM parameter store that is automatically kept up-to-date.
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     *
     * @param parameterName The name of SSM parameter containing the AMi id.
     * @param os The operating system type of the AMI.
     * @param userData optional user data for the given image.
     * @deprecated Use `MachineImage.fromSsmParameter()` instead
     */
    static fromSSMParameter(parameterName: string, os: OperatingSystemType, userData?: UserData): IMachineImage;
    /**
     * An image specified in SSM parameter store.
     *
     * By default, the SSM parameter is refreshed at every deployment,
     * causing your instances to be replaced whenever a new version of the AMI
     * is released.
     *
     * Pass `{ cachedInContext: true }` to keep the AMI ID stable. If you do, you
     * will have to remember to periodically invalidate the context to refresh
     * to the newest AMI ID.
     *
     * @stability stable
     */
    static fromSsmParameter(parameterName: string, options?: SsmParameterImageOptions): IMachineImage;
    /**
     * Look up a shared Machine Image using DescribeImages.
     *
     * The most recent, available, launchable image matching the given filter
     * criteria will be used. Looking up AMIs may take a long time; specify
     * as many filter criteria as possible to narrow down the search.
     *
     * The AMI selected will be cached in `cdk.context.json` and the same value
     * will be used on future runs. To refresh the AMI lookup, you will have to
     * evict the value from the cache using the `cdk context` command. See
     * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
     *
     * This function can not be used in environment-agnostic stacks.
     *
     * @stability stable
     */
    static lookup(props: LookupMachineImageProps): IMachineImage;
}
/**
 * Configuration for a machine image.
 *
 * @stability stable
 */
export interface MachineImageConfig {
    /**
     * The AMI ID of the image to use.
     *
     * @stability stable
     */
    readonly imageId: string;
    /**
     * Operating system type for this image.
     *
     * @stability stable
     */
    readonly osType: OperatingSystemType;
    /**
     * Initial UserData for this image.
     *
     * @stability stable
     */
    readonly userData: UserData;
}
/**
 * Select the image based on a given SSM parameter.
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 *
 * @stability stable
 */
export declare class GenericSSMParameterImage implements IMachineImage {
    private readonly os;
    private readonly userData?;
    /**
     * Name of the SSM parameter we're looking up.
     *
     * @stability stable
     */
    readonly parameterName: string;
    /**
     * @stability stable
     */
    constructor(parameterName: string, os: OperatingSystemType, userData?: UserData | undefined);
    /**
     * Return the image to use in the given context.
     *
     * @stability stable
     */
    getImage(scope: Construct): MachineImageConfig;
}
/**
 * Properties for GenericSsmParameterImage.
 *
 * @stability stable
 */
export interface SsmParameterImageOptions {
    /**
     * Operating system.
     *
     * @default OperatingSystemType.LINUX
     * @stability stable
     */
    readonly os?: OperatingSystemType;
    /**
     * Custom UserData.
     *
     * @default - UserData appropriate for the OS
     * @stability stable
     */
    readonly userData?: UserData;
    /**
     * Whether the AMI ID is cached to be stable between deployments.
     *
     * By default, the newest image is used on each deployment. This will cause
     * instances to be replaced whenever a new version is released, and may cause
     * downtime if there aren't enough running instances in the AutoScalingGroup
     * to reschedule the tasks on.
     *
     * If set to true, the AMI ID will be cached in `cdk.context.json` and the
     * same value will be used on future runs. Your instances will not be replaced
     * but your AMI version will grow old over time. To refresh the AMI lookup,
     * you will have to evict the value from the cache using the `cdk context`
     * command. See https://docs.aws.amazon.com/cdk/latest/guide/context.html for
     * more information.
     *
     * Can not be set to `true` in environment-agnostic stacks.
     *
     * @default false
     * @stability stable
     */
    readonly cachedInContext?: boolean;
}
/**
 * Configuration options for WindowsImage.
 *
 * @stability stable
 */
export interface WindowsImageProps {
    /**
     * Initial user data.
     *
     * @default - Empty UserData for Windows machines
     * @stability stable
     */
    readonly userData?: UserData;
}
/**
 * Select the latest version of the indicated Windows version.
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 *
 * https://aws.amazon.com/blogs/mt/query-for-the-latest-windows-ami-using-systems-manager-parameter-store/
 *
 * @stability stable
 */
export declare class WindowsImage extends GenericSSMParameterImage {
    /**
     * @stability stable
     */
    constructor(version: WindowsVersion, props?: WindowsImageProps);
}
/**
 * CPU type.
 *
 * @stability stable
 */
export declare enum AmazonLinuxCpuType {
    /**
     * arm64 CPU type.
     *
     * @stability stable
     */
    ARM_64 = "arm64",
    /**
     * x86_64 CPU type.
     *
     * @stability stable
     */
    X86_64 = "x86_64"
}
/**
 * Amazon Linux image properties.
 *
 * @stability stable
 */
export interface AmazonLinuxImageProps {
    /**
     * What generation of Amazon Linux to use.
     *
     * @default AmazonLinux
     * @stability stable
     */
    readonly generation?: AmazonLinuxGeneration;
    /**
     * What edition of Amazon Linux to use.
     *
     * @default Standard
     * @stability stable
     */
    readonly edition?: AmazonLinuxEdition;
    /**
     * Virtualization type.
     *
     * @default HVM
     * @stability stable
     */
    readonly virtualization?: AmazonLinuxVirt;
    /**
     * What storage backed image to use.
     *
     * @default GeneralPurpose
     * @stability stable
     */
    readonly storage?: AmazonLinuxStorage;
    /**
     * Initial user data.
     *
     * @default - Empty UserData for Linux machines
     * @stability stable
     */
    readonly userData?: UserData;
    /**
     * CPU Type.
     *
     * @default X86_64
     * @stability stable
     */
    readonly cpuType?: AmazonLinuxCpuType;
    /**
     * Whether the AMI ID is cached to be stable between deployments.
     *
     * By default, the newest image is used on each deployment. This will cause
     * instances to be replaced whenever a new version is released, and may cause
     * downtime if there aren't enough running instances in the AutoScalingGroup
     * to reschedule the tasks on.
     *
     * If set to true, the AMI ID will be cached in `cdk.context.json` and the
     * same value will be used on future runs. Your instances will not be replaced
     * but your AMI version will grow old over time. To refresh the AMI lookup,
     * you will have to evict the value from the cache using the `cdk context`
     * command. See https://docs.aws.amazon.com/cdk/latest/guide/context.html for
     * more information.
     *
     * Can not be set to `true` in environment-agnostic stacks.
     *
     * @default false
     * @stability stable
     */
    readonly cachedInContext?: boolean;
}
/**
 * Selects the latest version of Amazon Linux.
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 *
 * @stability stable
 */
export declare class AmazonLinuxImage extends GenericSSMParameterImage {
    private readonly props;
    /**
     * Return the SSM parameter name that will contain the Amazon Linux image with the given attributes.
     *
     * @stability stable
     */
    static ssmParameterName(props?: AmazonLinuxImageProps): string;
    private readonly cachedInContext;
    /**
     * @stability stable
     */
    constructor(props?: AmazonLinuxImageProps);
    /**
     * Return the image to use in the given context.
     *
     * @stability stable
     */
    getImage(scope: Construct): MachineImageConfig;
}
/**
 * What generation of Amazon Linux to use.
 *
 * @stability stable
 */
export declare enum AmazonLinuxGeneration {
    /**
     * Amazon Linux.
     *
     * @stability stable
     */
    AMAZON_LINUX = "amzn",
    /**
     * Amazon Linux 2.
     *
     * @stability stable
     */
    AMAZON_LINUX_2 = "amzn2"
}
/**
 * Amazon Linux edition.
 *
 * @stability stable
 */
export declare enum AmazonLinuxEdition {
    /**
     * Standard edition.
     *
     * @stability stable
     */
    STANDARD = "standard",
    /**
     * Minimal edition.
     *
     * @stability stable
     */
    MINIMAL = "minimal"
}
/**
 * Virtualization type for Amazon Linux.
 *
 * @stability stable
 */
export declare enum AmazonLinuxVirt {
    /**
     * HVM virtualization (recommended).
     *
     * @stability stable
     */
    HVM = "hvm",
    /**
     * PV virtualization.
     *
     * @stability stable
     */
    PV = "pv"
}
/**
 * @stability stable
 */
export declare enum AmazonLinuxStorage {
    /**
     * EBS-backed storage.
     *
     * @stability stable
     */
    EBS = "ebs",
    /**
     * S3-backed storage
     */
    S3 = "ebs",
    /**
     * General Purpose-based storage (recommended).
     *
     * @stability stable
     */
    GENERAL_PURPOSE = "gp2"
}
/**
 * Configuration options for GenericLinuxImage.
 *
 * @stability stable
 */
export interface GenericLinuxImageProps {
    /**
     * Initial user data.
     *
     * @default - Empty UserData for Linux machines
     * @stability stable
     */
    readonly userData?: UserData;
}
/**
 * Configuration options for GenericWindowsImage.
 *
 * @stability stable
 */
export interface GenericWindowsImageProps {
    /**
     * Initial user data.
     *
     * @default - Empty UserData for Windows machines
     * @stability stable
     */
    readonly userData?: UserData;
}
/**
 * Construct a Linux machine image from an AMI map.
 *
 * Linux images IDs are not published to SSM parameter store yet, so you'll have to
 * manually specify an AMI map.
 *
 * @stability stable
 */
export declare class GenericLinuxImage implements IMachineImage {
    private readonly amiMap;
    private readonly props;
    /**
     * @stability stable
     */
    constructor(amiMap: {
        [region: string]: string;
    }, props?: GenericLinuxImageProps);
    /**
     * Return the image to use in the given context.
     *
     * @stability stable
     */
    getImage(scope: Construct): MachineImageConfig;
}
/**
 * Construct a Windows machine image from an AMI map.
 *
 * Allows you to create a generic Windows EC2 , manually specify an AMI map.
 *
 * @stability stable
 */
export declare class GenericWindowsImage implements IMachineImage {
    private readonly amiMap;
    private readonly props;
    /**
     * @stability stable
     */
    constructor(amiMap: {
        [region: string]: string;
    }, props?: GenericWindowsImageProps);
    /**
     * Return the image to use in the given context.
     *
     * @stability stable
     */
    getImage(scope: Construct): MachineImageConfig;
}
/**
 * The OS type of a particular image.
 *
 * @stability stable
 */
export declare enum OperatingSystemType {
    /**
     * @stability stable
     */
    LINUX = 0,
    /**
     * @stability stable
     */
    WINDOWS = 1,
    /**
     * Used when the type of the operating system is not known (for example, for imported Auto-Scaling Groups).
     *
     * @stability stable
     */
    UNKNOWN = 2
}
/**
 * A machine image whose AMI ID will be searched using DescribeImages.
 *
 * The most recent, available, launchable image matching the given filter
 * criteria will be used. Looking up AMIs may take a long time; specify
 * as many filter criteria as possible to narrow down the search.
 *
 * The AMI selected will be cached in `cdk.context.json` and the same value
 * will be used on future runs. To refresh the AMI lookup, you will have to
 * evict the value from the cache using the `cdk context` command. See
 * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
 *
 * @stability stable
 */
export declare class LookupMachineImage implements IMachineImage {
    private readonly props;
    /**
     * @stability stable
     */
    constructor(props: LookupMachineImageProps);
    /**
     * Return the image to use in the given context.
     *
     * @stability stable
     */
    getImage(scope: Construct): MachineImageConfig;
}
/**
 * Properties for looking up an image.
 *
 * @stability stable
 */
export interface LookupMachineImageProps {
    /**
     * Name of the image (may contain wildcards).
     *
     * @stability stable
     */
    readonly name: string;
    /**
     * Owner account IDs or aliases.
     *
     * @default - All owners
     * @stability stable
     */
    readonly owners?: string[];
    /**
     * Additional filters on the AMI.
     *
     * @default - No additional filters
     * @see https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImages.html
     * @stability stable
     */
    readonly filters?: {
        [key: string]: string[];
    };
    /**
     * Look for Windows images.
     *
     * @default false
     * @stability stable
     */
    readonly windows?: boolean;
    /**
     * Custom userdata for this image.
     *
     * @default - Empty user data appropriate for the platform type
     * @stability stable
     */
    readonly userData?: UserData;
}
