"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PullRequestCheck = void 0;
const path = require("path");
const core_1 = require("@aws-cdk/core");
const aws_codebuild_1 = require("@aws-cdk/aws-codebuild");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
/**
 * Represents a reference to a PullRequestCheck.
 */
class PullRequestCheck extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { repository, buildSpec, buildImage = aws_codebuild_1.LinuxBuildImage.STANDARD_4_0, computeType = buildImage.defaultComputeType, privileged = false, updateApprovalState = true, postComment = true, projectName = `${repository.repositoryName}-pull-request`, role, } = props;
        this.pullRequestProject = new aws_codebuild_1.Project(this, 'PullRequestProject', {
            projectName,
            source: aws_codebuild_1.Source.codeCommit({
                repository,
            }),
            environment: {
                buildImage,
                computeType,
                privileged,
            },
            buildSpec,
            role,
        });
        if (updateApprovalState || postComment) {
            const codeBuildResultFunction = new aws_lambda_1.Function(this, 'CodeBuildResultFunction', {
                runtime: aws_lambda_1.Runtime.NODEJS_12_X,
                code: aws_lambda_1.Code.asset(path.join(__dirname, 'lambdas', 'code-build-result')),
                handler: 'index.handler',
                environment: {
                    UPDATE_APPROVAL_STATE: updateApprovalState ? 'TRUE' : 'FALSE',
                    POST_COMMENT: postComment ? 'TRUE' : 'FALSE',
                },
            });
            codeBuildResultFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                resources: [repository.repositoryArn],
                actions: [
                    'codecommit:PostCommentForPullRequest',
                    'codecommit:UpdatePullRequestApprovalState',
                ],
            }));
            this.pullRequestProject.onStateChange('PullRequestValidationRule', {
                target: new aws_events_targets_1.LambdaFunction(codeBuildResultFunction),
            });
        }
        const rule = repository.onPullRequestStateChange(`${this.node.uniqueId}Rule`, {
            eventPattern: {
                detail: {
                    event: ['pullRequestSourceBranchUpdated', 'pullRequestCreated'],
                },
            },
        });
        rule.addTarget(new aws_events_targets_1.CodeBuildProject(this.pullRequestProject, {
            event: aws_events_1.RuleTargetInput.fromObject({
                sourceVersion: aws_events_1.EventField.fromPath('$.detail.sourceCommit'),
                artifactsOverride: { type: 'NO_ARTIFACTS' },
                environmentVariablesOverride: [
                    {
                        name: 'pullRequestId',
                        value: aws_events_1.EventField.fromPath('$.detail.pullRequestId'),
                        type: 'PLAINTEXT',
                    },
                    {
                        name: 'repositoryName',
                        value: aws_events_1.EventField.fromPath('$.detail.repositoryNames[0]'),
                        type: 'PLAINTEXT',
                    },
                    {
                        name: 'sourceCommit',
                        value: aws_events_1.EventField.fromPath('$.detail.sourceCommit'),
                        type: 'PLAINTEXT',
                    },
                    {
                        name: 'destinationCommit',
                        value: aws_events_1.EventField.fromPath('$.detail.destinationCommit'),
                        type: 'PLAINTEXT',
                    },
                    {
                        name: 'revisionId',
                        value: aws_events_1.EventField.fromPath('$.detail.revisionId'),
                        type: 'PLAINTEXT',
                    },
                ],
            }),
        }));
    }
    /**
     * Defines an event rule which triggers when a check fails.
     */
    onCheckFailed(id, options) {
        return this.pullRequestProject.onBuildFailed(id, options);
    }
    /**
     * Defines an event rule which triggers when a check starts.
     */
    onCheckStarted(id, options) {
        return this.pullRequestProject.onBuildStarted(id, options);
    }
    /**
     * Defines an event rule which triggers when a check complets successfully.
     */
    onCheckSucceeded(id, options) {
        return this.pullRequestProject.onBuildSucceeded(id, options);
    }
    /**
     * Add a permission only if there's a policy attached.
     * @param statement The permissions statement to add
     */
    addToRolePolicy(statement) {
        this.pullRequestProject.addToRolePolicy(statement);
    }
}
exports.PullRequestCheck = PullRequestCheck;
//# sourceMappingURL=data:application/json;base64,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