"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StepScalingPolicy = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_autoscaling_common_1 = require("@aws-cdk/aws-autoscaling-common");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const step_scaling_action_1 = require("./step-scaling-action");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("@aws-cdk/core");
/**
 * Define a scaling strategy which scales depending on absolute values of some metric.
 *
 * You can specify the scaling behavior for various values of the metric.
 *
 * Implemented using one or more CloudWatch alarms and Step Scaling Policies.
 */
class StepScalingPolicy extends core_1.Construct {
    constructor(scope, id, props) {
        var _b, _c, _d, _e;
        super(scope, id);
        jsiiDeprecationWarnings._aws_cdk_aws_applicationautoscaling_StepScalingPolicyProps(props);
        if (props.scalingSteps.length < 2) {
            throw new Error('You must supply at least 2 intervals for autoscaling');
        }
        if (props.datapointsToAlarm !== undefined && props.datapointsToAlarm < 1) {
            throw new RangeError(`datapointsToAlarm cannot be less than 1, got: ${props.datapointsToAlarm}`);
        }
        const adjustmentType = props.adjustmentType || step_scaling_action_1.AdjustmentType.CHANGE_IN_CAPACITY;
        const changesAreAbsolute = adjustmentType === step_scaling_action_1.AdjustmentType.EXACT_CAPACITY;
        const intervals = aws_autoscaling_common_1.normalizeIntervals(props.scalingSteps, changesAreAbsolute);
        const alarms = aws_autoscaling_common_1.findAlarmThresholds(intervals);
        if (alarms.lowerAlarmIntervalIndex !== undefined) {
            const threshold = intervals[alarms.lowerAlarmIntervalIndex].upper;
            this.lowerAction = new step_scaling_action_1.StepScalingAction(this, 'LowerPolicy', {
                adjustmentType,
                cooldown: props.cooldown,
                metricAggregationType: (_b = props.metricAggregationType) !== null && _b !== void 0 ? _b : aggregationTypeFromMetric(props.metric),
                minAdjustmentMagnitude: props.minAdjustmentMagnitude,
                scalingTarget: props.scalingTarget,
            });
            for (let i = alarms.lowerAlarmIntervalIndex; i >= 0; i--) {
                this.lowerAction.addAdjustment({
                    adjustment: intervals[i].change,
                    lowerBound: i !== 0 ? intervals[i].lower - threshold : undefined,
                    upperBound: intervals[i].upper - threshold,
                });
            }
            this.lowerAlarm = new cloudwatch.Alarm(this, 'LowerAlarm', {
                // Recommended by AutoScaling
                metric: props.metric,
                alarmDescription: 'Lower threshold scaling alarm',
                comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_OR_EQUAL_TO_THRESHOLD,
                evaluationPeriods: (_c = props.evaluationPeriods) !== null && _c !== void 0 ? _c : 1,
                datapointsToAlarm: props.datapointsToAlarm,
                threshold,
            });
            this.lowerAlarm.addAlarmAction(new StepScalingAlarmAction(this.lowerAction));
        }
        if (alarms.upperAlarmIntervalIndex !== undefined) {
            const threshold = intervals[alarms.upperAlarmIntervalIndex].lower;
            this.upperAction = new step_scaling_action_1.StepScalingAction(this, 'UpperPolicy', {
                adjustmentType,
                cooldown: props.cooldown,
                metricAggregationType: (_d = props.metricAggregationType) !== null && _d !== void 0 ? _d : aggregationTypeFromMetric(props.metric),
                minAdjustmentMagnitude: props.minAdjustmentMagnitude,
                scalingTarget: props.scalingTarget,
            });
            for (let i = alarms.upperAlarmIntervalIndex; i < intervals.length; i++) {
                this.upperAction.addAdjustment({
                    adjustment: intervals[i].change,
                    lowerBound: intervals[i].lower - threshold,
                    upperBound: i !== intervals.length - 1 ? intervals[i].upper - threshold : undefined,
                });
            }
            this.upperAlarm = new cloudwatch.Alarm(this, 'UpperAlarm', {
                // Recommended by AutoScaling
                metric: props.metric,
                alarmDescription: 'Upper threshold scaling alarm',
                comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
                evaluationPeriods: (_e = props.evaluationPeriods) !== null && _e !== void 0 ? _e : 1,
                datapointsToAlarm: props.datapointsToAlarm,
                threshold,
            });
            this.upperAlarm.addAlarmAction(new StepScalingAlarmAction(this.upperAction));
        }
    }
}
exports.StepScalingPolicy = StepScalingPolicy;
_a = JSII_RTTI_SYMBOL_1;
StepScalingPolicy[_a] = { fqn: "@aws-cdk/aws-applicationautoscaling.StepScalingPolicy", version: "1.150.0" };
function aggregationTypeFromMetric(metric) {
    var _b;
    const statistic = (_b = metric.toMetricConfig().metricStat) === null || _b === void 0 ? void 0 : _b.statistic;
    if (statistic == null) {
        return undefined;
    } // Math expression, don't know aggregation, leave default
    switch (statistic) {
        case 'Average':
            return step_scaling_action_1.MetricAggregationType.AVERAGE;
        case 'Minimum':
            return step_scaling_action_1.MetricAggregationType.MINIMUM;
        case 'Maximum':
            return step_scaling_action_1.MetricAggregationType.MAXIMUM;
        default:
            return step_scaling_action_1.MetricAggregationType.AVERAGE;
    }
}
/**
 * Use a StepScalingAction as an Alarm Action
 *
 * This class is here and not in aws-cloudwatch-actions because this library
 * needs to use the class, and otherwise we'd have a circular dependency:
 *
 * aws-autoscaling -> aws-cloudwatch-actions (for using the Action)
 * aws-cloudwatch-actions -> aws-autoscaling (for the definition of IStepScalingAction)
 */
class StepScalingAlarmAction {
    constructor(stepScalingAction) {
        this.stepScalingAction = stepScalingAction;
    }
    bind(_scope, _alarm) {
        return { alarmActionArn: this.stepScalingAction.scalingPolicyArn };
    }
}
//# sourceMappingURL=data:application/json;base64,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