#!/usr/bin/env python

"""Tests for `rapid_models.doe` module."""

import pytest

import rapid_models.doe as doe

import numpy as np

# 1. Arrange
# 2. Act
# 3. Assert
# 4. Cleanup

# Tests related to rapid_models.doe.fullfact_with_bounds()


def test_fullfact_with_bounds_pos_dim():
    """
    Test ValueError raised for negative dimensions
    """
    with pytest.raises(ValueError):
        doe.fullfact_with_bounds([0.1, 0.1], [0.9, 0.9], [-1, 5])


def test_fullfact_with_bounds_zero_dim():
    """
    Test ValueError raised for zero dimension
    """
    with pytest.raises(ValueError):
        doe.fullfact_with_bounds([0.1, 0.1], [0.9, 0.9], [3, 0])


def test_fullfact_with_bounds_input_shape1():
    """
    Test ValueError raised for different first dimension of LB
    """
    with pytest.raises(ValueError):
        doe.fullfact_with_bounds([0.1], [0.9, 0.9], [3, 3])


def test_fullfact_with_bounds_input_shape2():
    """
    Test ValueError raised for different first dimension of UB
    """
    with pytest.raises(ValueError):
        doe.fullfact_with_bounds([0.1, 0.1], [0.9, 0.9, 1.2], [3, 3])


def test_fullfact_with_bounds_input_shape3():
    """
    Test ValueError raised for different first dimension of N_xi
    """
    with pytest.raises(ValueError):
        doe.fullfact_with_bounds([0.1, 0.1], [0.9, 0.9], [3, 1, 4])


def test_fullfact_with_bounds_input_shape4():
    """
    Test ValueError raised for input dimensions > 1
    """
    with pytest.raises(ValueError):
        doe.fullfact_with_bounds([[0.1, 0.9]], [[0.1, 0.9]], [[3, 1]])


def test_fullfact_with_bounds_output_shape1():
    """
    Test output shape 1
    """
    assert (3 * 5, 2) == doe.fullfact_with_bounds([0.1, 0.1], [0.9, 0.9], [3, 5]).shape


def test_fullfact_with_bounds_1():
    """
    Test output 1
    """
    # 1. + 2.
    X = np.asarray(doe.fullfact_with_bounds([0.1, 0.1], [0.9, 0.9], [3, 5]))
    X = X[X[:, 0].argsort()]
    # 3.
    assert np.allclose(
        X,
        np.array(
            [
                [0.1, 0.1],
                [0.1, 0.3],
                [0.1, 0.5],
                [0.1, 0.7],
                [0.1, 0.9],
                [0.5, 0.1],
                [0.5, 0.3],
                [0.5, 0.5],
                [0.5, 0.7],
                [0.5, 0.9],
                [0.9, 0.1],
                [0.9, 0.3],
                [0.9, 0.5],
                [0.9, 0.7],
                [0.9, 0.9],
            ]
        ),
    )


def test_fullfact_with_bounds_2():
    """
    Test output 2
    """
    # 1. + 2.
    X = np.asarray(doe.fullfact_with_bounds([-3, 4], [5, 12], [5, 3]))
    X = X[X[:, 0].argsort()]
    # 3.
    assert np.allclose(
        X,
        np.array(
            [
                [-3.0, 4.0],
                [-3.0, 8.0],
                [-3.0, 12.0],
                [-1.0, 4.0],
                [-1.0, 8.0],
                [-1.0, 12.0],
                [1.0, 4.0],
                [1.0, 8.0],
                [1.0, 12.0],
                [3.0, 4.0],
                [3.0, 8.0],
                [3.0, 12.0],
                [5.0, 4.0],
                [5.0, 8.0],
                [5.0, 12.0],
            ]
        ),
    )


def test_fullfact_with_bounds_3():
    """
    Test output 3
    """
    # 1. + 2.
    X = np.asarray(
        doe.fullfact_with_bounds([0.1, 0.1, 0.1], [0.9, 0.9, 1.2], [3, 3, 2])
    )
    # X = X[X[:, 0].argsort()]
    # 3.
    assert np.allclose(
        X,
        np.array(
            [
                [0.1, 0.1, 0.1],
                [0.5, 0.1, 0.1],
                [0.9, 0.1, 0.1],
                [0.1, 0.5, 0.1],
                [0.5, 0.5, 0.1],
                [0.9, 0.5, 0.1],
                [0.1, 0.9, 0.1],
                [0.5, 0.9, 0.1],
                [0.9, 0.9, 0.1],
                [0.1, 0.1, 1.2],
                [0.5, 0.1, 1.2],
                [0.9, 0.1, 1.2],
                [0.1, 0.5, 1.2],
                [0.5, 0.5, 1.2],
                [0.9, 0.5, 1.2],
                [0.1, 0.9, 1.2],
                [0.5, 0.9, 1.2],
                [0.9, 0.9, 1.2],
            ]
        ),
    )


# Tests related to rapid_models.doe.lhs_with_bounds()


def test_lhs_with_bounds_input_shape1():
    """
    Test ValueError raised for different first dimension of LB
    """
    with pytest.raises(ValueError):
        doe.lhs_with_bounds(2, 15, [0.1], [0.9, 0.9])


def test_lhs_with_bounds_input_shape2():
    """
    Test ValueError raised for different first dimension of UB
    """
    with pytest.raises(ValueError):
        doe.lhs_with_bounds(2, 15, [0.1, 0.1], [0.9, 0.9, 1.2])


def test_lhs_with_bounds_input_shape3():
    """
    Test ValueError raised for different nDim than dimension of LB and UB
    """
    with pytest.raises(ValueError):
        doe.lhs_with_bounds(3, 15, [0.1, 0.1], [0.9, 0.9])


def test_lhs_with_bounds_output_shape1():
    """
    Test output shape
    """
    assert (15, 2) == doe.lhs_with_bounds(2, 15, [0.1, 0.1], [0.9, 0.9]).shape


def test_lhs_with_bounds_1():
    """
    Test output 1
    """
    X = doe.lhs_with_bounds(2, 15, [0.1, 0.1], [0.9, 0.9], 42)
    assert np.allclose(
        X,
        np.array(
            [
                [0.87826211, 0.64077301],
                [0.66596549, 0.55465368],
                [0.48303066, 0.84914402],
                [0.76432373, 0.59553222],
                [0.60303707, 0.82075917],
                [0.36776451, 0.78187605],
                [0.70224771, 0.15070476],
                [0.11997547, 0.41839519],
                [0.26309779, 0.21498637],
                [0.19237301, 0.48311491],
                [0.8039826, 0.43132475],
                [0.54289292, 0.70620596],
                [0.3453928, 0.30619606],
                [0.46439694, 0.18526179],
                [0.21498766, 0.3510972],
            ]
        ),
    )


# TEST rapid_models.doe.in_hull()


@pytest.fixture
def lhs_2d_n50():
    return np.array(
        [
            [4.02440765, 2.41848545],
            [6.62818484, 0.62123391],
            [8.17262069, 4.06232981],
            [9.35215701, 1.40464504],
            [2.63993476, 1.35142344],
            [3.12150897, 0.71834045],
            [1.886389, 2.04951769],
            [3.21301032, 1.2785176],
            [8.2661796, 0.92912291],
            [3.59312641, 2.81959829],
            [7.87169315, 4.57132448],
            [0.920223, 2.19093204],
            [3.66092275, 3.49868869],
            [5.38789979, 4.91078914],
            [1.43636499, 0.85247564],
            [6.4561869, 3.51987157],
            [4.66234222, 1.78083973],
            [0.43120373, 1.51705241],
            [1.66084845, 2.68948274],
            [8.46219646, 4.74937956],
            [6.07773546, 3.3802197],
            [4.90934206, 1.94401525],
            [0.07490802, 0.47080726],
            [8.97744255, 3.0271349],
            [8.74592124, 4.36375575],
            [5.19391693, 2.3520068],
            [2.12237058, 1.01394939],
            [3.93684661, 2.93253303],
            [9.02391885, 0.38661761],
            [2.25842893, 3.91158691],
            [7.54137147, 0.59699099],
            [6.81491013, 3.77290072],
            [1.0041169, 2.57751328],
            [0.34639879, 4.10635584],
            [5.51958, 1.80976721],
            [4.2068777, 0.09507143],
            [1.36648853, 3.25426961],
            [9.55419344, 3.13567533],
            [2.491214, 4.44722149],
            [7.15444895, 2.26625223],
            [5.6176985, 2.79218742],
            [2.91848291, 4.23251833],
            [9.70454657, 4.65612772],
            [4.451756, 1.69488855],
            [7.20110442, 3.68154614],
            [5.80904546, 0.21559945],
            [0.61161672, 0.15986585],
            [9.80508383, 3.80740447],
            [7.75425407, 1.13663618],
            [6.3657475, 4.8427541],
        ]
    )


@pytest.fixture
def lhs_2d_n200():
    return np.array(
        [
            [9.17024459e00, 3.54440302e-01],
            [3.43189827e00, 4.15160920e00],
            [9.10804122e00, 3.76052883e00],
            [4.36265991e00, 1.24152286e00],
            [5.17995926e00, 4.22095018e-01],
            [9.87045136e00, 4.48124757e00],
            [8.84427895e00, 8.53649902e-02],
            [8.36996590e00, 4.37211782e00],
            [4.95951606e00, 2.41931146e00],
            [2.52752497e00, 3.23209575e00],
            [2.32817871e00, 3.37500663e00],
            [5.82264650e00, 1.11362930e00],
            [7.78340495e00, 3.64451186e00],
            [1.79469117e00, 1.83055199e-01],
            [6.10617835e00, 3.40754830e00],
            [4.78986111e00, 2.53177864e-01],
            [2.44285832e00, 4.48556399e-01],
            [2.56573879e00, 2.75345999e00],
            [1.92214639e00, 1.49941900e00],
            [5.52572016e00, 3.85101071e00],
            [1.02734425e00, 1.87072561e00],
            [3.82042784e-01, 1.02713440e00],
            [8.41947411e00, 4.30829135e00],
            [2.12441454e00, 8.48177263e-01],
            [8.13980524e00, 4.18425786e00],
            [8.82820532e-01, 2.87441730e00],
            [2.66399596e00, 1.55096421e00],
            [6.78425010e00, 4.58518627e00],
            [7.44003391e00, 4.62416966e00],
            [9.34504880e00, 1.52538514e00],
            [6.04856234e00, 3.30905362e00],
            [7.99140703e00, 2.95078279e00],
            [1.98040723e00, 6.12856368e-01],
            [9.00450269e00, 1.37648567e00],
            [8.92174519e00, 2.04378082e00],
            [1.35380257e-01, 4.56680261e-01],
            [2.94914543e00, 4.41176948e00],
            [2.49315686e00, 3.19899482e00],
            [1.64401031e00, 3.79003482e00],
            [1.20727837e00, 2.68407597e00],
            [3.82943515e00, 4.11623667e00],
            [9.67968496e00, 1.95879668e00],
            [5.94876044e00, 1.75244007e00],
            [1.35518644e00, 5.59726966e-01],
            [1.82278093e-01, 2.81722660e00],
            [3.62637520e00, 5.06466758e-01],
            [3.96629780e00, 2.27895888e00],
            [3.36287640e00, 3.27423553e00],
            [5.87363242e-01, 4.57249514e00],
            [5.66407835e00, 4.24082585e00],
            [2.62799622e-01, 2.49700283e00],
            [6.82351426e00, 4.64009090e00],
            [7.25631176e-01, 1.69561069e00],
            [1.10700911e00, 2.46743477e00],
            [6.40506106e00, 1.43696988e00],
            [9.84898018e00, 4.84857398e00],
            [6.54057989e00, 2.23576849e00],
            [8.00530103e-01, 2.89330152e00],
            [4.82455257e00, 4.34623160e00],
            [1.72223753e00, 3.06748333e00],
            [8.70407683e00, 1.45487491e00],
            [1.53309670e00, 3.58952151e00],
            [7.08786904e00, 1.31990137e00],
            [3.71409895e00, 1.37383578e00],
            [7.64779754e00, 3.33020115e00],
            [9.46541447e00, 2.66232289e00],
            [4.54814365e00, 3.93869781e00],
            [9.68093144e-01, 1.77924928e00],
            [1.87407412e00, 1.99078746e00],
            [7.49628941e00, 3.70633859e00],
            [4.85037644e00, 2.27209711e00],
            [6.59352330e00, 2.12647498e00],
            [5.77595712e00, 1.06166235e-01],
            [1.47809996e00, 4.80533993e00],
            [3.56444987e00, 7.54731070e-01],
            [2.64041678e00, 2.75130949e-01],
            [3.27574437e00, 2.21646026e00],
            [5.47372049e00, 1.87854800e00],
            [8.28174558e00, 4.01390478e00],
            [1.30798374e00, 2.71193896e00],
            [2.36567942e00, 4.09020381e00],
            [4.09274707e-01, 1.64754615e00],
            [7.20900753e00, 4.13968007e00],
            [2.86726909e00, 2.57809154e00],
            [3.05398561e00, 3.97080338e00],
            [4.40866836e00, 4.95468166e00],
            [7.39094867e00, 3.08613061e00],
            [7.67129525e00, 3.67308534e00],
            [9.97001376e00, 2.92547878e00],
            [2.22003106e00, 4.38949583e00],
            [2.28209611e00, 3.51151163e00],
            [2.04889841e00, 1.21191245e00],
            [8.20001987e00, 2.37062271e00],
            [5.23341705e00, 3.53835453e00],
            [8.59828459e00, 3.48974955e00],
            [4.65043425e00, 4.71735780e00],
            [2.19585279e00, 4.53467153e00],
            [1.41931532e00, 3.91342010e00],
            [9.78393447e00, 4.86184750e00],
            [8.01487336e00, 1.84839758e00],
            [3.86213784e00, 3.37866951e-02],
            [5.04129842e00, 3.69991451e00],
            [6.20787217e00, 2.61004337e00],
            [4.90426719e00, 1.26999486e00],
            [5.05859622e00, 3.03462157e00],
            [2.96621309e00, 3.13166011e00],
            [8.33909779e00, 2.52395838e00],
            [9.53530332e00, 9.63203321e-01],
            [4.63546040e-01, 6.99901150e-02],
            [5.95556319e00, 1.74490403e00],
            [5.47720634e-01, 4.75737678e00],
            [8.09905078e00, 3.73104132e00],
            [3.04936298e-01, 2.54459297e00],
            [3.30739840e00, 1.16473209e00],
            [2.35301551e-01, 6.89199152e-01],
            [6.61555946e00, 4.89267445e00],
            [4.64294317e00, 8.59428783e-01],
            [5.67442359e-03, 2.31838697e-01],
            [1.17359442e00, 1.08042209e00],
            [6.08943115e00, 3.46775422e00],
            [4.11998313e00, 1.91353664e00],
            [6.74774438e00, 4.02593388e00],
            [5.70957910e00, 1.41529084e00],
            [1.05873182e00, 1.27596883e00],
            [9.94454376e00, 3.83024232e00],
            [2.71988246e00, 4.26931093e00],
            [9.37247265e-01, 6.32631063e-01],
            [7.55767204e00, 2.32169377e00],
            [7.53198780e00, 3.61000736e00],
            [9.26088205e00, 2.16929281e00],
            [4.00109917e00, 3.99877681e00],
            [3.10842371e00, 4.78449971e00],
            [7.89721580e00, 2.90541434e00],
            [9.07457235e00, 1.93233631e00],
            [7.29421078e00, 6.64922788e-01],
            [7.10221731e00, 1.60378355e00],
            [6.91402768e00, 4.45716213e00],
            [8.65850825e00, 2.22929839e-01],
            [6.45405345e00, 4.06438307e00],
            [7.90817375e00, 3.42754474e00],
            [6.42839530e-01, 1.66999675e00],
            [5.44958370e00, 8.19593954e-01],
            [9.74790193e00, 9.32572518e-01],
            [5.12697252e00, 1.19395046e00],
            [5.88592957e00, 2.78623427e00],
            [3.18491292e00, 4.65766367e00],
            [6.39589382e00, 4.69984361e00],
            [2.76748409e00, 3.82336495e-01],
            [3.54009871e00, 3.35552619e00],
            [6.65712997e00, 1.07250783e00],
            [7.31243567e00, 3.15345464e00],
            [4.23386237e00, 1.13684076e00],
            [9.35746857e00, 9.88131124e-01],
            [1.27638259e00, 3.00520070e00],
            [5.32828260e00, 4.93068862e00],
            [5.56248916e00, 1.57510911e-01],
            [3.69323024e00, 5.99220673e-01],
            [2.09176480e00, 2.57430883e00],
            [3.93396871e00, 4.88160097e-01],
            [1.66262387e00, 7.41201792e-01],
            [5.27742585e00, 2.09037224e00],
            [8.17335369e00, 3.82009752e00],
            [9.40232480e00, 1.81815568e00],
            [6.99113883e00, 3.88691707e00],
            [8.50858222e00, 2.39547314e00],
            [5.64595865e00, 3.48380895e-01],
            [3.47458949e00, 3.22129810e00],
            [6.19957116e00, 1.25600284e-01],
            [9.24662439e00, 1.71233038e00],
            [4.70395678e00, 4.72629870e00],
            [6.77651285e-01, 2.05883892e00],
            [8.63412989e00, 1.01505100e00],
            [5.39680570e00, 2.19862416e00],
            [7.81225449e00, 5.34017756e-01],
            [6.33210734e00, 3.28315243e00],
            [4.58641336e00, 1.51886703e00],
            [1.83872290e00, 1.58912048e00],
            [8.89799769e00, 7.76475611e-01],
            [4.06252230e00, 4.44310428e00],
            [4.28451203e00, 2.73420616e00],
            [6.87549208e00, 4.97663850e00],
            [6.28110796e00, 7.02118051e-01],
            [8.64367317e-02, 4.92043118e00],
            [4.34167375e00, 3.57214961e00],
            [4.49304815e00, 2.43607983e00],
            [9.58258399e00, 3.17398905e-01],
            [9.63572816e00, 1.33638599e00],
            [4.17736464e00, 2.63720839e00],
            [7.18131740e00, 2.00888285e00],
            [1.59770545e00, 4.29796370e00],
            [8.45951440e00, 4.20562678e00],
            [7.01272774e00, 3.10088748e00],
            [8.75938224e00, 8.93851347e-01],
            [8.99517113e00, 2.33491155e00],
            [7.74509119e00, 9.21976213e-01],
            [2.80016746e00, 2.99625564e00],
            [7.76843290e-01, 2.84784887e00],
            [3.78917234e00, 2.12362563e00],
            [3.23213893e00, 2.43620774e-02],
            [3.03557510e00, 4.50343871e00],
        ]
    )


@pytest.fixture
def pts_def_hull():
    return np.array(
        [
            [4.02440765, 2.41848545],
            [2.63993476, 1.35142344],
            [3.12150897, 0.71834045],
            [3.21301032, 1.2785176],
            [8.2661796, 0.92912291],
            [3.59312641, 2.81959829],
            [4.66234222, 1.78083973],
            [4.90934206, 1.94401525],
            [5.19391693, 2.3520068],
            [3.93684661, 2.93253303],
            [5.51958, 1.80976721],
            [7.15444895, 2.26625223],
            [5.6176985, 2.79218742],
            [4.451756, 1.69488855],
            [7.75425407, 1.13663618],
        ]
    )


@pytest.fixture
def pts_def_circ():
    return np.array(
        [
            [4.02440765, 2.41848545],
            [2.63993476, 1.35142344],
            [3.12150897, 0.71834045],
            [1.886389, 2.04951769],
            [3.21301032, 1.2785176],
            [3.59312641, 2.81959829],
            [3.66092275, 3.49868869],
            [4.66234222, 1.78083973],
            [1.66084845, 2.68948274],
            [4.90934206, 1.94401525],
            [5.19391693, 2.3520068],
            [2.12237058, 1.01394939],
            [3.93684661, 2.93253303],
            [2.25842893, 3.91158691],
            [2.91848291, 4.23251833],
            [4.451756, 1.69488855],
        ]
    )


@pytest.fixture
def sample_inhull():
    return np.array(
        [
            [4.36265991, 1.24152286],
            [4.95951606, 2.41931146],
            [5.8226465, 1.1136293],
            [5.94876044, 1.75244007],
            [3.9662978, 2.27895888],
            [6.40506106, 1.43696988],
            [6.54057989, 2.23576849],
            [7.08786904, 1.31990137],
            [3.71409895, 1.37383578],
            [4.85037644, 2.27209711],
            [6.5935233, 2.12647498],
            [3.56444987, 0.75473107],
            [3.27574437, 2.21646026],
            [5.47372049, 1.878548],
            [4.90426719, 1.26999486],
            [5.95556319, 1.74490403],
            [3.3073984, 1.16473209],
            [4.64294317, 0.85942878],
            [4.11998313, 1.91353664],
            [5.7095791, 1.41529084],
            [7.10221731, 1.60378355],
            [5.4495837, 0.81959395],
            [5.12697252, 1.19395046],
            [6.65712997, 1.07250783],
            [4.23386237, 1.13684076],
            [5.27742585, 2.09037224],
            [5.3968057, 2.19862416],
            [4.58641336, 1.51886703],
            [4.28451203, 2.73420616],
            [4.49304815, 2.43607983],
            [4.17736464, 2.63720839],
            [7.1813174, 2.00888285],
            [7.74509119, 0.92197621],
            [3.78917234, 2.12362563],
        ]
    )


@pytest.fixture
def sample_inhull_circ():
    return np.array(
        [
            [4.95951606, 2.41931146],
            [2.52752497, 3.23209575],
            [2.32817871, 3.37500663],
            [2.56573879, 2.75345999],
            [2.66399596, 1.55096421],
            [2.49315686, 3.19899482],
            [3.9662978, 2.27895888],
            [3.3628764, 3.27423553],
            [3.71409895, 1.37383578],
            [1.87407412, 1.99078746],
            [4.85037644, 2.27209711],
            [3.27574437, 2.21646026],
            [2.86726909, 2.57809154],
            [3.05398561, 3.97080338],
            [2.28209611, 3.51151163],
            [2.96621309, 3.13166011],
            [3.3073984, 1.16473209],
            [4.11998313, 1.91353664],
            [3.54009871, 3.35552619],
            [2.0917648, 2.57430883],
            [3.47458949, 3.2212981],
            [4.28451203, 2.73420616],
            [4.49304815, 2.43607983],
            [4.17736464, 2.63720839],
            [2.80016746, 2.99625564],
            [3.78917234, 2.12362563],
        ]
    )


def test_in_hull_shape2():
    """
    Test ValueError raised for different second dimension of points and hull
    (i.e. different number of dimensions). The size of the first dimension is
    the sample size.
    """
    with pytest.raises(ValueError):
        doe.in_hull(np.ones((2, 3)), np.ones((2, 2)))


def test_in_hull_non_array1():
    """
    Test ValueError raised for non-array input
    """
    with pytest.raises(ValueError):
        doe.in_hull(2, np.ones((2, 2)))


def test_in_hull_non_array2():
    """
    Test ValueError raised for non-array input
    """
    with pytest.raises(ValueError):
        doe.in_hull(np.ones((2, 3)), 15)


def test_in_hull_output_shape(lhs_2d_n200, pts_def_hull):
    """
    Test output shape of a polygon hull
    """
    # 2. Act
    b_inhull = doe.in_hull(lhs_2d_n200, pts_def_hull)
    # 3. Assert
    assert b_inhull.shape == (len(lhs_2d_n200),)
    assert isinstance(b_inhull, np.ndarray)


def test_in_hull_1(lhs_2d_n200, pts_def_hull, sample_inhull):
    """
    Test output 1 of a polygon hull
    """
    # 2. Act
    b_inhull = doe.in_hull(lhs_2d_n200, pts_def_hull)
    # 3. Assert
    assert np.allclose(lhs_2d_n200[b_inhull], sample_inhull)


def test_in_hull_2(lhs_2d_n200, pts_def_circ, sample_inhull_circ):
    """
    Test output 2 of a circle hull
    """
    # 2. Act
    b_inhull = doe.in_hull(lhs_2d_n200, pts_def_circ)
    # 3. Assert
    assert np.allclose(lhs_2d_n200[b_inhull], sample_inhull_circ)


# TEST rapid_models.doe.sample_N_kmeans_cluster()


def test_kmeans_sample_N_int(pts_def_hull):
    """
    Test TypeError raised for non-int input of ´N´
    """
    with pytest.raises(TypeError):
        doe.kmeans_sample(
            points=pts_def_hull, N="1", values=None, mode="center", random_state=42
        )


def test_kmeans_sample_mode_str(pts_def_hull):
    """
    Test TypeError raised for non-string input of ´type´
    """
    with pytest.raises(TypeError):
        doe.kmeans_sample(
            points=pts_def_hull, N=10, values=None, mode=34, random_state=42
        )


def test_kmeans_sample_mode_val(pts_def_hull):
    """
    Test ValueError raised for input of ´type´ not supported
    """
    with pytest.raises(ValueError):
        doe.kmeans_sample(
            points=pts_def_hull, N=10, values=None, mode="whatever", random_state=42
        )


def test_kmeans_sample_values_None(pts_def_hull):
    """
    Test ValueError raised for input of ´values´=None when ´type´ require values
    """
    with pytest.raises(TypeError):
        doe.kmeans_sample(
            points=pts_def_hull, N=10, values=None, mode="min", random_state=42
        )


def test_kmeans_sample_values_shape(pts_def_hull):
    """
    Test ValueError raised for input of ´values´ with different number of values
    than the corresponding points.
    """
    with pytest.raises(ValueError):
        doe.kmeans_sample(
            points=pts_def_hull, N=10, values=[1, 2], mode="min", random_state=42
        )


def test_kmeans_sample_1_center(sample_inhull):
    """
    Test output 1: kmeans_sample in a polygon hull
    """
    # 2. Act
    kms = doe.kmeans_sample(
        points=sample_inhull, N=8, values=None, mode="center", random_state=42
    )
    # 3. Assert
    assert np.allclose(
        kms,
        np.array(
            [
                [4.31830828, 2.60249813],
                [6.28869758, 1.85931149],
                [3.52864907, 1.09776631],
                [4.5460292, 1.20533086],
                [3.78779941, 2.13314535],
                [7.15472498, 1.38541036],
                [5.19156891, 2.1717906],
                [5.52719545, 1.13561614],
            ]
        ),
    )


def test_kmeans_sample_2_center(sample_inhull_circ):
    """
    Test output 2: kmeans_sample in a circular hull
    """
    # 2. Act
    kms = doe.kmeans_sample(
        points=sample_inhull_circ, N=8, values=None, mode="center", random_state=42
    )
    # 3. Assert
    assert np.allclose(
        kms,
        np.array(
            [
                [2.66399596, 1.55096421],
                [4.55296347, 2.49978059],
                [3.27955266, 3.39070466],
                [2.74439178, 2.77593572],
                [3.51074868, 1.26928394],
                [1.98291946, 2.28254814],
                [2.40773916, 3.32940221],
                [3.78779941, 2.13314535],
            ]
        ),
    )


# TODO: make tests for mode='max','min','center_closest_point'
