import sys
import typing
from dataclasses import dataclass

if sys.version_info < (3, 8):
    from typing_extensions import Protocol
else:
    from typing import Protocol

from starlette.routing import BaseRoute, Mount
from starlette.routing import Router as StarletteRouter

from xpresso.routing.pathitem import Path
from xpresso.routing.websockets import WebSocketRoute


class App(Protocol):
    @property
    def router(self) -> StarletteRouter:
        ...


AppType = typing.TypeVar("AppType", bound=App)


@dataclass(frozen=True)
class VisitedRoute(typing.Generic[AppType]):
    path: str
    nodes: typing.List[typing.Union[StarletteRouter, AppType]]
    route: BaseRoute


def visit_routes(
    app_type: typing.Type[AppType],
    router: StarletteRouter,
    nodes: typing.List[typing.Union[StarletteRouter, AppType]],
    path: str,
) -> typing.Generator[VisitedRoute[AppType], None, None]:
    for route in typing.cast(typing.Iterable[BaseRoute], router.routes):  # type: ignore  # for Pylance
        if isinstance(route, Mount):
            app: typing.Any = route.app
            mount_path: str = route.path  # type: ignore  # for Pylance
            if isinstance(app, StarletteRouter):
                yield VisitedRoute(
                    path=path,
                    nodes=nodes + [app],
                    route=route,
                )
                yield from visit_routes(
                    app_type=app_type,
                    router=app,
                    nodes=nodes + [app],
                    path=path + mount_path,
                )
            elif isinstance(app, app_type):
                yield VisitedRoute(
                    path=path,
                    nodes=nodes + [app, app.router],
                    route=route,
                )
                yield from visit_routes(
                    app_type=app_type,
                    router=app.router,
                    nodes=nodes + [app, app.router],
                    path=path + mount_path,
                )
        elif isinstance(route, (Path, WebSocketRoute)):
            yield VisitedRoute(
                path=path + route.path,
                nodes=nodes,
                route=route,
            )
