"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OriginBase = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
/**
 * Represents a distribution origin, that describes the Amazon S3 bucket, HTTP server (for example, a web server),
 * Amazon MediaStore, or other server from which CloudFront gets your files.
 */
class OriginBase {
    constructor(domainName, props = {}) {
        var _b;
        jsiiDeprecationWarnings._aws_cdk_aws_cloudfront_OriginProps(props);
        validateIntInRangeOrUndefined('connectionTimeout', 1, 10, (_b = props.connectionTimeout) === null || _b === void 0 ? void 0 : _b.toSeconds());
        validateIntInRangeOrUndefined('connectionAttempts', 1, 3, props.connectionAttempts, false);
        validateCustomHeaders(props.customHeaders);
        this.domainName = domainName;
        this.originPath = this.validateOriginPath(props.originPath);
        this.connectionTimeout = props.connectionTimeout;
        this.connectionAttempts = props.connectionAttempts;
        this.customHeaders = props.customHeaders;
        this.originShieldRegion = props.originShieldRegion;
    }
    /**
     * Binds the origin to the associated Distribution. Can be used to grant permissions, create dependent resources, etc.
     */
    bind(_scope, options) {
        var _b;
        jsiiDeprecationWarnings._aws_cdk_aws_cloudfront_OriginBindOptions(options);
        const s3OriginConfig = this.renderS3OriginConfig();
        const customOriginConfig = this.renderCustomOriginConfig();
        if (!s3OriginConfig && !customOriginConfig) {
            throw new Error('Subclass must override and provide either s3OriginConfig or customOriginConfig');
        }
        return {
            originProperty: {
                domainName: this.domainName,
                id: options.originId,
                originPath: this.originPath,
                connectionAttempts: this.connectionAttempts,
                connectionTimeout: (_b = this.connectionTimeout) === null || _b === void 0 ? void 0 : _b.toSeconds(),
                originCustomHeaders: this.renderCustomHeaders(),
                s3OriginConfig,
                customOriginConfig,
                originShield: this.renderOriginShield(this.originShieldRegion),
            },
        };
    }
    // Overridden by sub-classes to provide S3 origin config.
    renderS3OriginConfig() {
        return undefined;
    }
    // Overridden by sub-classes to provide custom origin config.
    renderCustomOriginConfig() {
        return undefined;
    }
    renderCustomHeaders() {
        if (!this.customHeaders || Object.entries(this.customHeaders).length === 0) {
            return undefined;
        }
        return Object.entries(this.customHeaders).map(([headerName, headerValue]) => {
            return { headerName, headerValue };
        });
    }
    /**
     * If the path is defined, it must start with a '/' and not end with a '/'.
     * This method takes in the originPath, and returns it back (if undefined) or adds/removes the '/' as appropriate.
     */
    validateOriginPath(originPath) {
        if (core_1.Token.isUnresolved(originPath)) {
            return originPath;
        }
        if (originPath === undefined) {
            return undefined;
        }
        let path = originPath;
        if (!path.startsWith('/')) {
            path = '/' + path;
        }
        if (path.endsWith('/')) {
            path = path.substr(0, path.length - 1);
        }
        return path;
    }
    /**
     * Takes origin shield region and converts to CfnDistribution.OriginShieldProperty
     */
    renderOriginShield(originShieldRegion) {
        return originShieldRegion
            ? { enabled: true, originShieldRegion }
            : undefined;
    }
}
exports.OriginBase = OriginBase;
_a = JSII_RTTI_SYMBOL_1;
OriginBase[_a] = { fqn: "@aws-cdk/aws-cloudfront.OriginBase", version: "1.151.0" };
/**
 * Throws an error if a value is defined and not an integer or not in a range.
 */
function validateIntInRangeOrUndefined(name, min, max, value, isDuration = true) {
    if (value === undefined) {
        return;
    }
    if (!Number.isInteger(value) || value < min || value > max) {
        const seconds = isDuration ? ' seconds' : '';
        throw new Error(`${name}: Must be an int between ${min} and ${max}${seconds} (inclusive); received ${value}.`);
    }
}
/**
 * Throws an error if custom header assignment is prohibited by CloudFront.
 * @link: https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/add-origin-custom-headers.html#add-origin-custom-headers-denylist
 */
function validateCustomHeaders(customHeaders) {
    if (!customHeaders || Object.entries(customHeaders).length === 0) {
        return;
    }
    const customHeaderKeys = Object.keys(customHeaders);
    const prohibitedHeaderKeys = [
        'Cache-Control', 'Connection', 'Content-Length', 'Cookie', 'Host', 'If-Match', 'If-Modified-Since', 'If-None-Match', 'If-Range', 'If-Unmodified-Since',
        'Max-Forwards', 'Pragma', 'Proxy-Authorization', 'Proxy-Connection', 'Range', 'Request-Range', 'TE', 'Trailer', 'Transfer-Encoding', 'Upgrade', 'Via',
        'X-Real-Ip',
    ];
    const prohibitedHeaderKeyPrefixes = [
        'X-Amz-', 'X-Edge-',
    ];
    const prohibitedHeadersKeysMatches = customHeaderKeys.filter(customKey => {
        return prohibitedHeaderKeys.map((prohibitedKey) => prohibitedKey.toLowerCase()).includes(customKey.toLowerCase());
    });
    const prohibitedHeaderPrefixMatches = customHeaderKeys.filter(customKey => {
        return prohibitedHeaderKeyPrefixes.some(prohibitedKeyPrefix => customKey.toLowerCase().startsWith(prohibitedKeyPrefix.toLowerCase()));
    });
    if (prohibitedHeadersKeysMatches.length !== 0) {
        throw new Error(`The following headers cannot be configured as custom origin headers: ${prohibitedHeadersKeysMatches.join(', ')}`);
    }
    if (prohibitedHeaderPrefixMatches.length !== 0) {
        throw new Error(`The following headers cannot be used as prefixes for custom origin headers: ${prohibitedHeaderPrefixMatches.join(', ')}`);
    }
}
//# sourceMappingURL=data:application/json;base64,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