"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EdgeFunction = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const ssm = require("@aws-cdk/aws-ssm");
const core_1 = require("@aws-cdk/core");
/**
 * A Lambda@Edge function.
 *
 * Convenience resource for requesting a Lambda function in the 'us-east-1' region for use with Lambda@Edge.
 * Implements several restrictions enforced by Lambda@Edge.
 *
 * Note that this construct requires that the 'us-east-1' region has been bootstrapped.
 * See https://docs.aws.amazon.com/cdk/latest/guide/bootstrapping.html or 'cdk bootstrap --help' for options.
 *
 * @resource AWS::Lambda::Function
 */
class EdgeFunction extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.isBoundToVpc = false;
        jsiiDeprecationWarnings._aws_cdk_aws_cloudfront_experimental_EdgeFunctionProps(props);
        // Create a simple Function if we're already in us-east-1; otherwise create a cross-region stack.
        const regionIsUsEast1 = !core_1.Token.isUnresolved(this.stack.region) && this.stack.region === 'us-east-1';
        const { edgeFunction, edgeArn } = regionIsUsEast1
            ? this.createInRegionFunction(props)
            : this.createCrossRegionFunction(id, props);
        this.edgeArn = edgeArn;
        this.functionArn = edgeArn;
        this._edgeFunction = edgeFunction;
        this.functionName = this._edgeFunction.functionName;
        this.grantPrincipal = this._edgeFunction.role;
        this.permissionsNode = this._edgeFunction.permissionsNode;
        this.version = lambda.extractQualifierFromArn(this.functionArn);
        this.architecture = this._edgeFunction.architecture;
        this.resourceArnsForGrantInvoke = this._edgeFunction.resourceArnsForGrantInvoke;
        this.node.defaultChild = this._edgeFunction;
    }
    get lambda() {
        return this._edgeFunction;
    }
    /**
     * Convenience method to make `EdgeFunction` conform to the same interface as `Function`.
     */
    get currentVersion() {
        return this;
    }
    addAlias(aliasName, options = {}) {
        return new lambda.Alias(this._edgeFunction, `Alias${aliasName}`, {
            aliasName,
            version: this._edgeFunction.currentVersion,
            ...options,
        });
    }
    /**
     * Not supported. Connections are only applicable to VPC-enabled functions.
     */
    get connections() {
        throw new Error('Lambda@Edge does not support connections');
    }
    get latestVersion() {
        throw new Error('$LATEST function version cannot be used for Lambda@Edge');
    }
    addEventSourceMapping(id, options) {
        return this.lambda.addEventSourceMapping(id, options);
    }
    addPermission(id, permission) {
        return this.lambda.addPermission(id, permission);
    }
    addToRolePolicy(statement) {
        return this.lambda.addToRolePolicy(statement);
    }
    grantInvoke(identity) {
        return this.lambda.grantInvoke(identity);
    }
    metric(metricName, props) {
        return this.lambda.metric(metricName, { ...props, region: EdgeFunction.EDGE_REGION });
    }
    metricDuration(props) {
        return this.lambda.metricDuration({ ...props, region: EdgeFunction.EDGE_REGION });
    }
    metricErrors(props) {
        return this.lambda.metricErrors({ ...props, region: EdgeFunction.EDGE_REGION });
    }
    metricInvocations(props) {
        return this.lambda.metricInvocations({ ...props, region: EdgeFunction.EDGE_REGION });
    }
    metricThrottles(props) {
        return this.lambda.metricThrottles({ ...props, region: EdgeFunction.EDGE_REGION });
    }
    /** Adds an event source to this function. */
    addEventSource(source) {
        return this.lambda.addEventSource(source);
    }
    configureAsyncInvoke(options) {
        return this.lambda.configureAsyncInvoke(options);
    }
    /** Create a function in-region */
    createInRegionFunction(props) {
        const edgeFunction = new lambda.Function(this, 'Fn', props);
        addEdgeLambdaToRoleTrustStatement(edgeFunction.role);
        return { edgeFunction, edgeArn: edgeFunction.currentVersion.edgeArn };
    }
    /** Create a support stack and function in us-east-1, and a SSM reader in-region */
    createCrossRegionFunction(id, props) {
        const parameterNamePrefix = 'cdk/EdgeFunctionArn';
        if (core_1.Token.isUnresolved(this.env.region)) {
            throw new Error('stacks which use EdgeFunctions must have an explicitly set region');
        }
        // SSM parameter names must only contain letters, numbers, ., _, -, or /.
        const sanitizedPath = this.node.path.replace(/[^\/\w.-]/g, '_');
        const parameterName = `/${parameterNamePrefix}/${this.env.region}/${sanitizedPath}`;
        const functionStack = this.edgeStack(props.stackId);
        const edgeFunction = new lambda.Function(functionStack, id, props);
        addEdgeLambdaToRoleTrustStatement(edgeFunction.role);
        // Store the current version's ARN to be retrieved by the cross region reader below.
        const version = edgeFunction.currentVersion;
        new ssm.StringParameter(edgeFunction, 'Parameter', {
            parameterName,
            stringValue: version.edgeArn,
        });
        const edgeArn = this.createCrossRegionArnReader(parameterNamePrefix, parameterName, version);
        return { edgeFunction, edgeArn };
    }
    createCrossRegionArnReader(parameterNamePrefix, parameterName, version) {
        // Prefix of the parameter ARN that applies to all EdgeFunctions.
        // This is necessary because the `CustomResourceProvider` is a singleton, and the `policyStatement`
        // must work for multiple EdgeFunctions.
        const parameterArnPrefix = this.stack.formatArn({
            service: 'ssm',
            region: EdgeFunction.EDGE_REGION,
            resource: 'parameter',
            resourceName: parameterNamePrefix + '/*',
        });
        const resourceType = 'Custom::CrossRegionStringParameterReader';
        const serviceToken = core_1.CustomResourceProvider.getOrCreate(this, resourceType, {
            codeDirectory: path.join(__dirname, 'edge-function'),
            runtime: core_1.CustomResourceProviderRuntime.NODEJS_12_X,
            policyStatements: [{
                    Effect: 'Allow',
                    Resource: parameterArnPrefix,
                    Action: ['ssm:GetParameter'],
                }],
        });
        const resource = new core_1.CustomResource(this, 'ArnReader', {
            resourceType: resourceType,
            serviceToken,
            properties: {
                Region: EdgeFunction.EDGE_REGION,
                ParameterName: parameterName,
                // This is used to determine when the function has changed, to refresh the ARN from the custom resource.
                //
                // Use the logical id of the function version. Whenever a function version changes, the logical id must be
                // changed for it to take effect - a good candidate for RefreshToken.
                RefreshToken: core_1.Lazy.uncachedString({
                    produce: () => {
                        const cfn = version.node.defaultChild;
                        return this.stack.resolve(cfn.logicalId);
                    },
                }),
            },
        });
        return resource.getAttString('FunctionArn');
    }
    edgeStack(stackId) {
        const stage = core_1.Stage.of(this);
        if (!stage) {
            throw new Error('stacks which use EdgeFunctions must be part of a CDK app or stage');
        }
        const edgeStackId = stackId !== null && stackId !== void 0 ? stackId : `edge-lambda-stack-${this.stack.node.addr}`;
        let edgeStack = stage.node.tryFindChild(edgeStackId);
        if (!edgeStack) {
            edgeStack = new core_1.Stack(stage, edgeStackId, {
                env: {
                    region: EdgeFunction.EDGE_REGION,
                    account: core_1.Stack.of(this).account,
                },
            });
        }
        this.stack.addDependency(edgeStack);
        return edgeStack;
    }
}
exports.EdgeFunction = EdgeFunction;
_a = JSII_RTTI_SYMBOL_1;
EdgeFunction[_a] = { fqn: "@aws-cdk/aws-cloudfront.experimental.EdgeFunction", version: "1.151.0" };
EdgeFunction.EDGE_REGION = 'us-east-1';
function addEdgeLambdaToRoleTrustStatement(role) {
    if (role instanceof iam.Role && role.assumeRolePolicy) {
        const statement = new iam.PolicyStatement();
        const edgeLambdaServicePrincipal = new iam.ServicePrincipal('edgelambda.amazonaws.com');
        statement.addPrincipals(edgeLambdaServicePrincipal);
        statement.addActions(edgeLambdaServicePrincipal.assumeRoleAction);
        role.assumeRolePolicy.addStatements(statement);
    }
}
//# sourceMappingURL=data:application/json;base64,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