"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.accountIfDifferentFromStack = exports.regionIfDifferentFromStack = void 0;
const cdk = require("@aws-cdk/core");
/**
 * Make a Token that renders to given region if used in a different stack, otherwise undefined
 */
function regionIfDifferentFromStack(region) {
    return cdk.Token.asString(new StackDependentToken(region, stack => stack.region));
}
exports.regionIfDifferentFromStack = regionIfDifferentFromStack;
/**
 * Make a Token that renders to given account if used in a different stack, otherwise undefined
 */
function accountIfDifferentFromStack(account) {
    return cdk.Token.asString(new StackDependentToken(account, stack => stack.account));
}
exports.accountIfDifferentFromStack = accountIfDifferentFromStack;
/**
 * A lazy token that requires an instance of Stack to evaluate
 */
class StackDependentToken {
    constructor(originalValue, fn) {
        this.originalValue = originalValue;
        this.fn = fn;
        this.creationStack = cdk.captureStackTrace();
    }
    resolve(context) {
        const stackValue = this.fn(cdk.Stack.of(context.scope));
        // Don't render if the values are definitely the same. If the stack
        // is unresolved we don't know, better output the value.
        if (!cdk.Token.isUnresolved(stackValue) && stackValue === this.originalValue) {
            return undefined;
        }
        return this.originalValue;
    }
    toString() {
        return cdk.Token.asString(this);
    }
    toJSON() {
        return this.originalValue;
    }
}
//# sourceMappingURL=data:application/json;base64,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