from marshmallow import Schema, fields

from restit.open_api import reusable_schema


@reusable_schema
class RFC7807Schema(Schema):
    """See https://tools.ietf.org/html/rfc7807

    HTTP [RFC7230] status codes are sometimes not sufficient to convey
    enough information about an error to be helpful.  While humans behind
    Web browsers can be informed about the nature of the problem with an
    HTML [W3C.REC-html5-20141028] response body, non-human consumers of
    so-called "HTTP APIs" are usually not.
    """

    type = fields.String(default="about:blank")
    type.__doc__ = (
        "A URI reference [RFC3986] that identifies the "
        "problem type.  This specification encourages that, when "
        "dereferenced, it provide human-readable documentation for the "
        "problem type (e.g., using HTML [W3C.REC-html5-20141028]).  When "
        "this member is not present, its value is assumed to be "
        "'about:blank'."
    )

    title = fields.String()
    title.__doc__ = (
        "A short, human-readable summary of the problem "
        "type.  It SHOULD NOT change from occurrence to occurrence of the "
        "problem, except for purposes of localization (e.g., using "
        "proactive content negotiation; see [RFC7231]"
    )

    status = fields.Number()
    status.__doc__ = (
        "The HTTP status code ([RFC7231], Section 6) "
        "generated by the origin server for this occurrence of the problem."
    )

    detail = fields.String()
    detail.__doc__ = "A human-readable explanation specific to this occurrence of the problem."

    instance = fields.String()
    instance.__doc__ = (
        "A URI reference that identifies the specific "
        "occurrence of the problem.  It may or may not yield further "
        "information if dereferenced."
    )
