# -*- coding: utf-8 -*-
from distutils.core import setup

packages = \
['bson-stubs', 'mongoengine-stubs', 'pymongo-stubs']

package_data = \
{'': ['*'], 'mongoengine-stubs': ['queryset/*']}

setup_kwargs = {
    'name': 'mongo-types',
    'version': '0.7.5',
    'description': 'Type stubs for mongoengine w/ basic support for bson and pymongo',
    'long_description': '# mongo-types [![PyPI](https://img.shields.io/pypi/v/mongo-types.svg)](https://pypi.org/project/mongo-types/)\n\nType stubs for [`mongoengine`][0] with some basic type stubs for [`pymongo`][1]\nand [`bson`][2].\n\nAllows for autocomplete and static typing.\n\n## install\n\n```shell\npip install mongo-types\n```\n\nMonkey patch mongoengine\'s `QuerySet` so we can type it with a generic\nargument at runtime:\n\n```python\nimport types\nfrom mongoengine.queryset.queryset import QuerySet\n\ndef no_op(self, x):\n    return self\n\nQuerySet.__class_getitem__ = types.MethodType(no_op, QuerySet)\n```\n\n## usage\n\nAfter installing and monkey patching, the types should work for the most\npart,\nbut you\'ll probably need to change how you write some things.\n\n### getting `objects` to work\n\nBy default, the base document is typed to not have an `objects` property so\nthat each document can type it properly.\n\nHere\'s a helper class that\'s useful for simple cases which don\'t modify the\n`QuerySet`.\n\n```python\nfrom typing import Generic, Type, TypeVar\nfrom mongoengine import QuerySet, Document\n\nU = TypeVar("U", bound=Document)\n\nclass QuerySetManager(Generic[U]):\n    def __get__(self, instance: object, cls: Type[U]) -> QuerySet[U]:\n        return QuerySet(cls, cls._get_collection())\n\nclass Page(Document):\n    meta = {\n        "collection": "pages",\n    }\n\n    objects = QuerySetManager["Page"]()\n\n    organization = fields.StringField()\n```\n\n### replacing usages of `queryset_class`\n\nbefore:\n\n```python\nfrom typing import Type\nfrom mongoengine import QuerySet, Document\n\nclass PostQuerySet(QuerySet):\n    def for_org(self, *, org: str) -> QuerySet:\n        return self.filter(organization=org)\n\n    def exists(self) -> bool:\n        return self.count() > 0\n\nclass Post(Document):\n    meta = {\n        "collection": "posts",\n        "queryset_class": SMSLogQuerySet,\n    }\n\n    organization = fields.StringField()\n    # --snip--\n```\n\nafter:\n\n```python\nfrom typing import Type\nfrom mongoengine import QuerySet, Document\n\nclass PostQuerySet(QuerySet["Post"]):\n    def for_org(self, *, org: str) -> QuerySet["Post"]:\n        return self.filter(organization=org)\n\n    def exists(self) -> bool:\n        return self.count() > 0\n\n\nclass QuerySetManager:\n    def __get__(self, instance: object, cls: Type[Post]) -> PostQuerySet:\n        return PostQuerySet(cls, cls._get_collection())\n\n\nclass Post(Document):\n    meta = {\n        "collection": "posts",\n    }\n\n    objects = QuerySetManager()\n\n    organization = fields.StringField()\n    # --snip--\n```\n\n### replicating `@queryset_manager` behavior\n\nbefore:\n\n```python\nfrom mongoengine import Document, QuerySet, queryset_manager, fields\n\nclass UserQuerySet(QuerySet):\n    def for_org(self, *, org: str) -> QuerySet:\n        return self.filter(organization=org)\n\nclass User(Document):\n    meta = {\n        "collection": "users",\n        "queryset_class": UserQuerySet,\n    }\n\n    is_active = fields.BooleanField()\n\n    # --snip--\n\n    @queryset_manager\n    def objects(self, queryset: QuerySet) -> QuerySet:\n        return queryset.filter(is_active=True)\n\n    @queryset_manager\n    def all_objects(self, queryset: QuerySet) -> QuerySet:\n        return queryset\n\nmaybe_user = User.all_objects.first()\n```\n\nafter:\n\n```python\nfrom __future__ import annotations\nfrom typing import Type\nfrom mongoengine import QuerySet, Document\n\nclass UserQuerySet(QuerySet["User"]):\n    def for_org(self, *, org: str) -> UserQuerySet:\n        return self.filter(organization=org)\n\n\nclass QuerySetManager:\n    def __get__(self, instance: object, cls: Type[User]) -> UserQuerySet:\n        return UserQuerySet(cls, cls._get_collection()).filter(is_active=True)\n\n\nclass User(Document):\n    meta = {\n        "collection": "users",\n    }\n\n    is_active = fields.BooleanField()\n\n    # --snip--\n\n    objects = QuerySetManager()\n\n    @classmethod\n    def all_objects(cls) -> UserQuerySet:\n        return UserQuerySet(cls, cls._get_collection())\n\nmaybe_user = User.all_objects().first()\n```\n\n### fixing "Model" has no attribute "id"\n\nMongoengine will define an `id` field for you automatically.\nMongo-types require you specify your `id` explicitly so that\nthe types can be more strict.\n\n```python\nclass User(Document):\n    meta = {\n        "collection": "users",\n    }\n\n# becomes\n\nclass User(Document):\n    meta = {\n        "collection": "users",\n    }\n    id = fields.StringField(name="_id", primary_key=True, default=default_id)\n\n# or if you prefer ObjectIds\n\nclass User(Document):\n    meta = {\n        "collection": "users",\n    }\n    id = fields.ObjectIdField(name="_id", primary_key=True, default=ObjectId)\n```\n\n## dev\n\n```shell\npoetry install\n\n# run formatting, linting, and typechecking\ns/lint\n\n# build\npoetry build -f wheel\n\n# build and publish\npoetry publish --build\n```\n\n[0]: https://github.com/MongoEngine/mongoengine\n[1]: https://github.com/mongodb/mongo-python-driver/tree/master/pymongo\n[2]: https://github.com/mongodb/mongo-python-driver/tree/master/bson\n',
    'author': 'Steve Dignam',
    'author_email': 'steve@dignam.xyz',
    'url': 'https://github.com/sbdchd/mongo-types',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
