import { IBucket } from '@aws-cdk/aws-s3';
import { Resource } from '@aws-cdk/core';
import { OperatingSystemType } from './machine-image';
/**
 * Options when constructing UserData for Linux.
 *
 * @stability stable
 */
export interface LinuxUserDataOptions {
    /**
     * Shebang for the UserData script.
     *
     * @default "#!/bin/bash"
     * @stability stable
     */
    readonly shebang?: string;
}
/**
 * Options when downloading files from S3.
 *
 * @stability stable
 */
export interface S3DownloadOptions {
    /**
     * Name of the S3 bucket to download from.
     *
     * @stability stable
     */
    readonly bucket: IBucket;
    /**
     * The key of the file to download.
     *
     * @stability stable
     */
    readonly bucketKey: string;
    /**
     * The name of the local file.
     *
     * @default Linux   - /tmp/bucketKey
     * Windows - %TEMP%/bucketKey
     * @stability stable
     */
    readonly localFile?: string;
    /**
     * The region of the S3 Bucket (needed for access via VPC Gateway).
     *
     * @default none
     * @stability stable
     */
    readonly region?: string;
}
/**
 * Options when executing a file.
 *
 * @stability stable
 */
export interface ExecuteFileOptions {
    /**
     * The path to the file.
     *
     * @stability stable
     */
    readonly filePath: string;
    /**
     * The arguments to be passed to the file.
     *
     * @default No arguments are passed to the file.
     * @stability stable
     */
    readonly arguments?: string;
}
/**
 * Instance User Data.
 *
 * @stability stable
 */
export declare abstract class UserData {
    /**
     * Create a userdata object for Linux hosts.
     *
     * @stability stable
     */
    static forLinux(options?: LinuxUserDataOptions): UserData;
    /**
     * Create a userdata object for Windows hosts.
     *
     * @stability stable
     */
    static forWindows(): UserData;
    /**
     * Create a userdata object with custom content.
     *
     * @stability stable
     */
    static custom(content: string): UserData;
    /**
     * @stability stable
     */
    static forOperatingSystem(os: OperatingSystemType): UserData;
    /**
     * Add one or more commands to the user data.
     *
     * @stability stable
     */
    abstract addCommands(...commands: string[]): void;
    /**
     * Add one or more commands to the user data that will run when the script exits.
     *
     * @stability stable
     */
    abstract addOnExitCommands(...commands: string[]): void;
    /**
     * Render the UserData for use in a construct.
     *
     * @stability stable
     */
    abstract render(): string;
    /**
     * Adds commands to download a file from S3.
     *
     * @returns : The local path that the file will be downloaded to
     * @stability stable
     */
    abstract addS3DownloadCommand(params: S3DownloadOptions): string;
    /**
     * Adds commands to execute a file.
     *
     * @stability stable
     */
    abstract addExecuteFileCommand(params: ExecuteFileOptions): void;
    /**
     * Adds a command which will send a cfn-signal when the user data script ends.
     *
     * @stability stable
     */
    abstract addSignalOnExitCommand(resource: Resource): void;
}
/**
 * Options when creating `MultipartBody`.
 *
 * @stability stable
 */
export interface MultipartBodyOptions {
    /**
     * `Content-Type` header of this part.
     *
     * Some examples of content types:
     * * `text/x-shellscript; charset="utf-8"` (shell script)
     * * `text/cloud-boothook; charset="utf-8"` (shell script executed during boot phase)
     *
     * For Linux shell scripts use `text/x-shellscript`.
     *
     * @stability stable
     */
    readonly contentType: string;
    /**
     * `Content-Transfer-Encoding` header specifying part encoding.
     *
     * @default undefined - body is not encoded
     * @stability stable
     */
    readonly transferEncoding?: string;
    /**
     * The body of message.
     *
     * @default undefined - body will not be added to part
     * @stability stable
     */
    readonly body?: string;
}
/**
 * The base class for all classes which can be used as {@link MultipartUserData}.
 *
 * @stability stable
 */
export declare abstract class MultipartBody {
    /**
     * Content type for shell scripts.
     *
     * @stability stable
     */
    static readonly SHELL_SCRIPT = "text/x-shellscript; charset=\"utf-8\"";
    /**
     * Content type for boot hooks.
     *
     * @stability stable
     */
    static readonly CLOUD_BOOTHOOK = "text/cloud-boothook; charset=\"utf-8\"";
    /**
     * Constructs the new `MultipartBody` wrapping existing `UserData`. Modification to `UserData` are reflected in subsequent renders of the part.
     *
     * For more information about content types see {@link MultipartBodyOptions.contentType}.
     *
     * @param userData user data to wrap into body part.
     * @param contentType optional content type, if default one should not be used.
     * @stability stable
     */
    static fromUserData(userData: UserData, contentType?: string): MultipartBody;
    /**
     * Constructs the raw `MultipartBody` using specified body, content type and transfer encoding.
     *
     * When transfer encoding is specified (typically as Base64), it's caller responsibility to convert body to
     * Base64 either by wrapping with `Fn.base64` or by converting it by other converters.
     *
     * @stability stable
     */
    static fromRawBody(opts: MultipartBodyOptions): MultipartBody;
    /**
     * @stability stable
     */
    constructor();
    /**
     * Render body part as the string.
     *
     * Subclasses should not add leading nor trailing new line characters (\r \n)
     *
     * @stability stable
     */
    abstract renderBodyPart(): string[];
}
/**
 * Options for creating {@link MultipartUserData}.
 *
 * @stability stable
 */
export interface MultipartUserDataOptions {
    /**
     * The string used to separate parts in multipart user data archive (it's like MIME boundary).
     *
     * This string should contain [a-zA-Z0-9()+,-./:=?] characters only, and should not be present in any part, or in text content of archive.
     *
     * @default `+AWS+CDK+User+Data+Separator==`
     * @stability stable
     */
    readonly partsSeparator?: string;
}
/**
 * Mime multipart user data.
 *
 * This class represents MIME multipart user data, as described in.
 * [Specifying Multiple User Data Blocks Using a MIME Multi Part Archive](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/bootstrap_container_instance.html#multi-part_user_data)
 *
 * @stability stable
 */
export declare class MultipartUserData extends UserData {
    private static readonly USE_PART_ERROR;
    private static readonly BOUNDRY_PATTERN;
    private parts;
    private opts;
    private defaultUserData?;
    /**
     * @stability stable
     */
    constructor(opts?: MultipartUserDataOptions);
    /**
     * Adds a part to the list of parts.
     *
     * @stability stable
     */
    addPart(part: MultipartBody): void;
    /**
     * Adds a multipart part based on a UserData object.
     *
     * If `makeDefault` is true, then the UserData added by this method
     * will also be the target of calls to the `add*Command` methods on
     * this MultipartUserData object.
     *
     * If `makeDefault` is false, then this is the same as calling:
     *
     * ```ts
     * declare const multiPart: ec2.MultipartUserData;
     * declare const userData: ec2.UserData;
     * declare const contentType: string;
     *
     * multiPart.addPart(ec2.MultipartBody.fromUserData(userData, contentType));
     * ```
     *
     * An undefined `makeDefault` defaults to either:
     * - `true` if no default UserData has been set yet; or
     * - `false` if there is no default UserData set.
     *
     * @stability stable
     */
    addUserDataPart(userData: UserData, contentType?: string, makeDefault?: boolean): void;
    /**
     * Render the UserData for use in a construct.
     *
     * @stability stable
     */
    render(): string;
    /**
     * Adds commands to download a file from S3.
     *
     * @stability stable
     */
    addS3DownloadCommand(params: S3DownloadOptions): string;
    /**
     * Adds commands to execute a file.
     *
     * @stability stable
     */
    addExecuteFileCommand(params: ExecuteFileOptions): void;
    /**
     * Adds a command which will send a cfn-signal when the user data script ends.
     *
     * @stability stable
     */
    addSignalOnExitCommand(resource: Resource): void;
    /**
     * Add one or more commands to the user data.
     *
     * @stability stable
     */
    addCommands(...commands: string[]): void;
    /**
     * Add one or more commands to the user data that will run when the script exits.
     *
     * @stability stable
     */
    addOnExitCommands(...commands: string[]): void;
}
