"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecurityGroup = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const peer_1 = require("./peer");
const port_1 = require("./port");
const SECURITY_GROUP_SYMBOL = Symbol.for('@aws-cdk/iam.SecurityGroup');
const SECURITY_GROUP_DISABLE_INLINE_RULES_CONTEXT_KEY = '@aws-cdk/aws-ec2.securityGroupDisableInlineRules';
/**
 * A SecurityGroup that is not created in this template
 */
class SecurityGroupBase extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, props);
        /**
         * Whether the rule can be inlined into a SecurityGroup or not.
         *
         * @stability stable
         */
        this.canInlineRule = false;
        /**
         * The network connections associated with this resource.
         *
         * @stability stable
         */
        this.connections = new connections_1.Connections({ securityGroups: [this] });
        this.peerAsTokenCount = 0;
        Object.defineProperty(this, SECURITY_GROUP_SYMBOL, { value: true });
    }
    /**
     * Return whether the indicated object is a security group.
     *
     * @stability stable
     */
    static isSecurityGroup(x) {
        return SECURITY_GROUP_SYMBOL in x;
    }
    /**
     * A unique identifier for this connection peer.
     *
     * @stability stable
     */
    get uniqueId() {
        return core_1.Names.nodeUniqueId(this.node);
    }
    addIngressRule(peer, connection, description, remoteRule) {
        if (description === undefined) {
            description = `from ${peer.uniqueId}:${connection}`;
        }
        const [scope, id] = this.determineRuleScope(peer, connection, 'from', remoteRule);
        // Skip duplicates
        if (scope.node.tryFindChild(id) === undefined) {
            new ec2_generated_1.CfnSecurityGroupIngress(scope, id, {
                groupId: this.securityGroupId,
                ...peer.toIngressRuleConfig(),
                ...connection.toRuleJson(),
                description,
            });
        }
    }
    addEgressRule(peer, connection, description, remoteRule) {
        if (description === undefined) {
            description = `to ${peer.uniqueId}:${connection}`;
        }
        const [scope, id] = this.determineRuleScope(peer, connection, 'to', remoteRule);
        // Skip duplicates
        if (scope.node.tryFindChild(id) === undefined) {
            new ec2_generated_1.CfnSecurityGroupEgress(scope, id, {
                groupId: this.securityGroupId,
                ...peer.toEgressRuleConfig(),
                ...connection.toRuleJson(),
                description,
            });
        }
    }
    /**
     * Produce the ingress rule JSON for the given connection.
     *
     * @stability stable
     */
    toIngressRuleConfig() {
        return { sourceSecurityGroupId: this.securityGroupId };
    }
    /**
     * Produce the egress rule JSON for the given connection.
     *
     * @stability stable
     */
    toEgressRuleConfig() {
        return { destinationSecurityGroupId: this.securityGroupId };
    }
    /**
     * Determine where to parent a new ingress/egress rule.
     *
     * A SecurityGroup rule is parented under the group it's related to, UNLESS
     * we're in a cross-stack scenario with another Security Group. In that case,
     * we respect the 'remoteRule' flag and will parent under the other security
     * group.
     *
     * This is necessary to avoid cyclic dependencies between stacks, since both
     * ingress and egress rules will reference both security groups, and a naive
     * parenting will lead to the following situation:
     *
     *    ╔════════════════════╗         ╔════════════════════╗
     *    ║  ┌───────────┐     ║         ║    ┌───────────┐   ║
     *    ║  │  GroupA   │◀────╬─┐   ┌───╬───▶│  GroupB   │   ║
     *    ║  └───────────┘     ║ │   │   ║    └───────────┘   ║
     *    ║        ▲           ║ │   │   ║          ▲         ║
     *    ║        │           ║ │   │   ║          │         ║
     *    ║        │           ║ │   │   ║          │         ║
     *    ║  ┌───────────┐     ║ └───┼───╬────┌───────────┐   ║
     *    ║  │  EgressA  │─────╬─────┘   ║    │ IngressB  │   ║
     *    ║  └───────────┘     ║         ║    └───────────┘   ║
     *    ║                    ║         ║                    ║
     *    ╚════════════════════╝         ╚════════════════════╝
     *
     * By having the ability to switch the parent, we avoid the cyclic reference by
     * keeping all rules in a single stack.
     *
     * If this happens, we also have to change the construct ID, because
     * otherwise we might have two objects with the same ID if we have
     * multiple reversed security group relationships.
     *
     *    ╔═══════════════════════════════════╗
     *    ║┌───────────┐                      ║
     *    ║│  GroupB   │                      ║
     *    ║└───────────┘                      ║
     *    ║      ▲                            ║
     *    ║      │              ┌───────────┐ ║
     *    ║      ├────"from A"──│ IngressB  │ ║
     *    ║      │              └───────────┘ ║
     *    ║      │              ┌───────────┐ ║
     *    ║      ├─────"to B"───│  EgressA  │ ║
     *    ║      │              └───────────┘ ║
     *    ║      │              ┌───────────┐ ║
     *    ║      └─────"to B"───│  EgressC  │ ║  <-- oops
     *    ║                     └───────────┘ ║
     *    ╚═══════════════════════════════════╝
     *
     * @stability stable
     */
    determineRuleScope(peer, connection, fromTo, remoteRule) {
        if (remoteRule && SecurityGroupBase.isSecurityGroup(peer) && differentStacks(this, peer)) {
            // Reversed
            const reversedFromTo = fromTo === 'from' ? 'to' : 'from';
            return [peer, `${this.uniqueId}:${connection} ${reversedFromTo}`];
        }
        else {
            // Regular (do old ID escaping to in order to not disturb existing deployments)
            return [this, `${fromTo} ${this.renderPeer(peer)}:${connection}`.replace('/', '_')];
        }
    }
    renderPeer(peer) {
        if (core_1.Token.isUnresolved(peer.uniqueId)) {
            // Need to return a unique value each time a peer
            // is an unresolved token, else the duplicate skipper
            // in `sg.addXxxRule` can detect unique rules as duplicates
            return this.peerAsTokenCount++ ? `'{IndirectPeer${this.peerAsTokenCount}}'` : '{IndirectPeer}';
        }
        else {
            return peer.uniqueId;
        }
    }
}
function differentStacks(group1, group2) {
    return core_1.Stack.of(group1) !== core_1.Stack.of(group2);
}
/**
 * Creates an Amazon EC2 security group within a VPC.
 *
 * Security Groups act like a firewall with a set of rules, and are associated
 * with any AWS resource that has or creates Elastic Network Interfaces (ENIs).
 * A typical example of a resource that has a security group is an Instance (or
 * Auto Scaling Group of instances)
 *
 * If you are defining new infrastructure in CDK, there is a good chance you
 * won't have to interact with this class at all. Like IAM Roles, Security
 * Groups need to exist to control access between AWS resources, but CDK will
 * automatically generate and populate them with least-privilege permissions
 * for you so you can concentrate on your business logic.
 *
 * All Constructs that require Security Groups will create one for you if you
 * don't specify one at construction. After construction, you can selectively
 * allow connections to and between constructs via--for example-- the `instance.connections`
 * object. Think of it as "allowing connections to your instance", rather than
 * "adding ingress rules a security group". See the [Allowing
 * Connections](https://docs.aws.amazon.com/cdk/api/latest/docs/aws-ec2-readme.html#allowing-connections)
 * section in the library documentation for examples.
 *
 * Direct manipulation of the Security Group through `addIngressRule` and
 * `addEgressRule` is possible, but mutation through the `.connections` object
 * is recommended. If you peer two constructs with security groups this way,
 * appropriate rules will be created in both.
 *
 * If you have an existing security group you want to use in your CDK application,
 * you would import it like this:
 *
 * ```ts
 * const securityGroup = ec2.SecurityGroup.fromSecurityGroupId(this, 'SG', 'sg-12345', {
 *    mutable: false
 * });
 * ```
 *
 * @stability stable
 */
class SecurityGroup extends SecurityGroupBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.securityGroupName,
        });
        this.directIngressRules = [];
        this.directEgressRules = [];
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_SecurityGroupProps(props);
        const groupDescription = props.description || this.node.path;
        this.allowAllOutbound = props.allowAllOutbound !== false;
        this.disableInlineRules = props.disableInlineRules !== undefined ?
            !!props.disableInlineRules :
            !!this.node.tryGetContext(SECURITY_GROUP_DISABLE_INLINE_RULES_CONTEXT_KEY);
        this.securityGroup = new ec2_generated_1.CfnSecurityGroup(this, 'Resource', {
            groupName: this.physicalName,
            groupDescription,
            securityGroupIngress: core_1.Lazy.any({ produce: () => this.directIngressRules }, { omitEmptyArray: true }),
            securityGroupEgress: core_1.Lazy.any({ produce: () => this.directEgressRules }, { omitEmptyArray: true }),
            vpcId: props.vpc.vpcId,
        });
        this.securityGroupId = this.securityGroup.attrGroupId;
        this.securityGroupVpcId = this.securityGroup.attrVpcId;
        this.securityGroupName = this.securityGroup.ref;
        this.addDefaultEgressRule();
    }
    /**
     * (deprecated) Look up a security group by id.
     *
     * @deprecated Use `fromLookupById()` instead
     */
    static fromLookup(scope, id, securityGroupId) {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-ec2.SecurityGroup#fromLookup", "Use `fromLookupById()` instead");
        return this.fromLookupAttributes(scope, id, { securityGroupId });
    }
    /**
     * Look up a security group by id.
     *
     * @stability stable
     */
    static fromLookupById(scope, id, securityGroupId) {
        return this.fromLookupAttributes(scope, id, { securityGroupId });
    }
    /**
     * Look up a security group by name.
     *
     * @stability stable
     */
    static fromLookupByName(scope, id, securityGroupName, vpc) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_IVpc(vpc);
        return this.fromLookupAttributes(scope, id, { securityGroupName, vpc });
    }
    /**
     * Import an existing security group into this app.
     *
     * This method will assume that the Security Group has a rule in it which allows
     * all outbound traffic, and so will not add egress rules to the imported Security
     * Group (only ingress rules).
     *
     * If your existing Security Group needs to have egress rules added, pass the
     * `allowAllOutbound: false` option on import.
     *
     * @stability stable
     */
    static fromSecurityGroupId(scope, id, securityGroupId, options = {}) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_SecurityGroupImportOptions(options);
        class MutableImport extends SecurityGroupBase {
            constructor() {
                var _b;
                super(...arguments);
                this.securityGroupId = securityGroupId;
                this.allowAllOutbound = (_b = options.allowAllOutbound) !== null && _b !== void 0 ? _b : true;
            }
            addEgressRule(peer, connection, description, remoteRule) {
                // Only if allowAllOutbound has been disabled
                if (options.allowAllOutbound === false) {
                    super.addEgressRule(peer, connection, description, remoteRule);
                }
            }
        }
        class ImmutableImport extends SecurityGroupBase {
            constructor() {
                var _b;
                super(...arguments);
                this.securityGroupId = securityGroupId;
                this.allowAllOutbound = (_b = options.allowAllOutbound) !== null && _b !== void 0 ? _b : true;
            }
            addEgressRule(_peer, _connection, _description, _remoteRule) {
                // do nothing
            }
            addIngressRule(_peer, _connection, _description, _remoteRule) {
                // do nothing
            }
        }
        return options.mutable !== false
            ? new MutableImport(scope, id)
            : new ImmutableImport(scope, id);
    }
    /**
     * Look up a security group.
     */
    static fromLookupAttributes(scope, id, options) {
        var _b, _c;
        if (core_1.Token.isUnresolved(options.securityGroupId) || core_1.Token.isUnresolved(options.securityGroupName) || core_1.Token.isUnresolved((_b = options.vpc) === null || _b === void 0 ? void 0 : _b.vpcId)) {
            throw new Error('All arguments to look up a security group must be concrete (no Tokens)');
        }
        const attributes = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.SECURITY_GROUP_PROVIDER,
            props: {
                securityGroupId: options.securityGroupId,
                securityGroupName: options.securityGroupName,
                vpcId: (_c = options.vpc) === null || _c === void 0 ? void 0 : _c.vpcId,
            },
            dummyValue: {
                securityGroupId: 'sg-12345',
                allowAllOutbound: true,
            },
        }).value;
        return SecurityGroup.fromSecurityGroupId(scope, id, attributes.securityGroupId, {
            allowAllOutbound: attributes.allowAllOutbound,
            mutable: true,
        });
    }
    /**
     * Add an ingress rule for the current security group.
     *
     * `remoteRule` controls where the Rule object is created if the peer is also a
     * securityGroup and they are in different stack. If false (default) the
     * rule object is created under the current SecurityGroup object. If true and the
     * peer is also a SecurityGroup, the rule object is created under the remote
     * SecurityGroup object.
     *
     * @stability stable
     */
    addIngressRule(peer, connection, description, remoteRule) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_IPeer(peer);
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_Port(connection);
        if (!peer.canInlineRule || !connection.canInlineRule || this.disableInlineRules) {
            super.addIngressRule(peer, connection, description, remoteRule);
            return;
        }
        if (description === undefined) {
            description = `from ${peer.uniqueId}:${connection}`;
        }
        this.addDirectIngressRule({
            ...peer.toIngressRuleConfig(),
            ...connection.toRuleJson(),
            description,
        });
    }
    /**
     * Add an egress rule for the current security group.
     *
     * `remoteRule` controls where the Rule object is created if the peer is also a
     * securityGroup and they are in different stack. If false (default) the
     * rule object is created under the current SecurityGroup object. If true and the
     * peer is also a SecurityGroup, the rule object is created under the remote
     * SecurityGroup object.
     *
     * @stability stable
     */
    addEgressRule(peer, connection, description, remoteRule) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_IPeer(peer);
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_Port(connection);
        if (this.allowAllOutbound) {
            // In the case of "allowAllOutbound", we don't add any more rules. There
            // is only one rule which allows all traffic and that subsumes any other
            // rule.
            if (!remoteRule) { // Warn only if addEgressRule() was explicitely called
                core_1.Annotations.of(this).addWarning('Ignoring Egress rule since \'allowAllOutbound\' is set to true; To add customize rules, set allowAllOutbound=false on the SecurityGroup');
            }
            return;
        }
        else {
            // Otherwise, if the bogus rule exists we can now remove it because the
            // presence of any other rule will get rid of EC2's implicit "all
            // outbound" rule anyway.
            this.removeNoTrafficRule();
        }
        if (!peer.canInlineRule || !connection.canInlineRule || this.disableInlineRules) {
            super.addEgressRule(peer, connection, description, remoteRule);
            return;
        }
        if (description === undefined) {
            description = `from ${peer.uniqueId}:${connection}`;
        }
        const rule = {
            ...peer.toEgressRuleConfig(),
            ...connection.toRuleJson(),
            description,
        };
        if (isAllTrafficRule(rule)) {
            // We cannot allow this; if someone adds the rule in this way, it will be
            // removed again if they add other rules. We also can't automatically switch
            // to "allOutbound=true" mode, because we might have already emitted
            // EgressRule objects (which count as rules added later) and there's no way
            // to recall those. Better to prevent this for now.
            throw new Error('Cannot add an "all traffic" egress rule in this way; set allowAllOutbound=true on the SecurityGroup instead.');
        }
        this.addDirectEgressRule(rule);
    }
    /**
     * Add a direct ingress rule
     */
    addDirectIngressRule(rule) {
        if (!this.hasIngressRule(rule)) {
            this.directIngressRules.push(rule);
        }
    }
    /**
     * Return whether the given ingress rule exists on the group
     */
    hasIngressRule(rule) {
        return this.directIngressRules.findIndex(r => ingressRulesEqual(r, rule)) > -1;
    }
    /**
     * Add a direct egress rule
     */
    addDirectEgressRule(rule) {
        if (!this.hasEgressRule(rule)) {
            this.directEgressRules.push(rule);
        }
    }
    /**
     * Return whether the given egress rule exists on the group
     */
    hasEgressRule(rule) {
        return this.directEgressRules.findIndex(r => egressRulesEqual(r, rule)) > -1;
    }
    /**
     * Add the default egress rule to the securityGroup
     *
     * This depends on allowAllOutbound:
     *
     * - If allowAllOutbound is true, we *TECHNICALLY* don't need to do anything, because
     *   EC2 is going to create this default rule anyway. But, for maximum readability
     *   of the template, we will add one anyway.
     * - If allowAllOutbound is false, we add a bogus rule that matches no traffic in
     *   order to get rid of the default "all outbound" rule that EC2 creates by default.
     *   If other rules happen to get added later, we remove the bogus rule again so
     *   that it doesn't clutter up the template too much (even though that's not
     *   strictly necessary).
     */
    addDefaultEgressRule() {
        if (this.disableInlineRules) {
            const peer = this.allowAllOutbound ? ALL_TRAFFIC_PEER : NO_TRAFFIC_PEER;
            const port = this.allowAllOutbound ? ALL_TRAFFIC_PORT : NO_TRAFFIC_PORT;
            const description = this.allowAllOutbound ? ALLOW_ALL_RULE.description : MATCH_NO_TRAFFIC.description;
            super.addEgressRule(peer, port, description, false);
        }
        else {
            const rule = this.allowAllOutbound ? ALLOW_ALL_RULE : MATCH_NO_TRAFFIC;
            this.directEgressRules.push(rule);
        }
    }
    /**
     * Remove the bogus rule if it exists
     */
    removeNoTrafficRule() {
        if (this.disableInlineRules) {
            const [scope, id] = this.determineRuleScope(NO_TRAFFIC_PEER, NO_TRAFFIC_PORT, 'to', false);
            scope.node.tryRemoveChild(id);
        }
        else {
            const i = this.directEgressRules.findIndex(r => egressRulesEqual(r, MATCH_NO_TRAFFIC));
            if (i > -1) {
                this.directEgressRules.splice(i, 1);
            }
        }
    }
}
exports.SecurityGroup = SecurityGroup;
_a = JSII_RTTI_SYMBOL_1;
SecurityGroup[_a] = { fqn: "@aws-cdk/aws-ec2.SecurityGroup", version: "1.141.0" };
/**
 * Egress rule that purposely matches no traffic
 *
 * This is used in order to disable the "all traffic" default of Security Groups.
 *
 * No machine can ever actually have the 255.255.255.255 IP address, but
 * in order to lock it down even more we'll restrict to a nonexistent
 * ICMP traffic type.
 */
const MATCH_NO_TRAFFIC = {
    cidrIp: '255.255.255.255/32',
    description: 'Disallow all traffic',
    ipProtocol: 'icmp',
    fromPort: 252,
    toPort: 86,
};
const NO_TRAFFIC_PEER = peer_1.Peer.ipv4(MATCH_NO_TRAFFIC.cidrIp);
const NO_TRAFFIC_PORT = port_1.Port.icmpTypeAndCode(MATCH_NO_TRAFFIC.fromPort, MATCH_NO_TRAFFIC.toPort);
/**
 * Egress rule that matches all traffic
 */
const ALLOW_ALL_RULE = {
    cidrIp: '0.0.0.0/0',
    description: 'Allow all outbound traffic by default',
    ipProtocol: '-1',
};
const ALL_TRAFFIC_PEER = peer_1.Peer.anyIpv4();
const ALL_TRAFFIC_PORT = port_1.Port.allTraffic();
/**
 * Compare two ingress rules for equality the same way CloudFormation would (discarding description)
 */
function ingressRulesEqual(a, b) {
    return a.cidrIp === b.cidrIp
        && a.cidrIpv6 === b.cidrIpv6
        && a.fromPort === b.fromPort
        && a.toPort === b.toPort
        && a.ipProtocol === b.ipProtocol
        && a.sourceSecurityGroupId === b.sourceSecurityGroupId
        && a.sourceSecurityGroupName === b.sourceSecurityGroupName
        && a.sourceSecurityGroupOwnerId === b.sourceSecurityGroupOwnerId;
}
/**
 * Compare two egress rules for equality the same way CloudFormation would (discarding description)
 */
function egressRulesEqual(a, b) {
    return a.cidrIp === b.cidrIp
        && a.cidrIpv6 === b.cidrIpv6
        && a.fromPort === b.fromPort
        && a.toPort === b.toPort
        && a.ipProtocol === b.ipProtocol
        && a.destinationPrefixListId === b.destinationPrefixListId
        && a.destinationSecurityGroupId === b.destinationSecurityGroupId;
}
/**
 * Whether this rule refers to all traffic
 */
function isAllTrafficRule(rule) {
    return rule.cidrIp === '0.0.0.0/0' && rule.ipProtocol === '-1';
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2VjdXJpdHktZ3JvdXAuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJzZWN1cml0eS1ncm91cC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7QUFBQSwyREFBMkQ7QUFDM0Qsd0NBQTRIO0FBRzVILCtDQUE0QztBQUM1QyxtREFBb0c7QUFDcEcsaUNBQXFDO0FBQ3JDLGlDQUE4QjtBQUc5QixNQUFNLHFCQUFxQixHQUFHLE1BQU0sQ0FBQyxHQUFHLENBQUMsNEJBQTRCLENBQUMsQ0FBQztBQUV2RSxNQUFNLCtDQUErQyxHQUFHLGtEQUFrRCxDQUFDO0FBaUIzRzs7R0FFRztBQUNILE1BQWUsaUJBQWtCLFNBQVEsZUFBUTtJQWUvQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXFCO1FBQzdELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7UUFQVixrQkFBYSxHQUFHLEtBQUssQ0FBQzs7Ozs7O1FBQ3RCLGdCQUFXLEdBQWdCLElBQUkseUJBQVcsQ0FBQyxFQUFFLGNBQWMsRUFBRSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQztRQUcvRSxxQkFBZ0IsR0FBVyxDQUFDLENBQUM7UUFLbkMsTUFBTSxDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUscUJBQXFCLEVBQUUsRUFBRSxLQUFLLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztLQUNyRTs7Ozs7O0lBakJNLE1BQU0sQ0FBQyxlQUFlLENBQUMsQ0FBTTtRQUNsQyxPQUFPLHFCQUFxQixJQUFJLENBQUMsQ0FBQztLQUNuQzs7Ozs7O0lBaUJELElBQVcsUUFBUTtRQUNqQixPQUFPLFlBQUssQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO0tBQ3RDO0lBRU0sY0FBYyxDQUFDLElBQVcsRUFBRSxVQUFnQixFQUFFLFdBQW9CLEVBQUUsVUFBb0I7UUFDN0YsSUFBSSxXQUFXLEtBQUssU0FBUyxFQUFFO1lBQzdCLFdBQVcsR0FBRyxRQUFRLElBQUksQ0FBQyxRQUFRLElBQUksVUFBVSxFQUFFLENBQUM7U0FDckQ7UUFFRCxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxHQUFHLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFLE1BQU0sRUFBRSxVQUFVLENBQUMsQ0FBQztRQUVsRixrQkFBa0I7UUFDbEIsSUFBSSxLQUFLLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQUMsS0FBSyxTQUFTLEVBQUU7WUFDN0MsSUFBSSx1Q0FBdUIsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO2dCQUNyQyxPQUFPLEVBQUUsSUFBSSxDQUFDLGVBQWU7Z0JBQzdCLEdBQUcsSUFBSSxDQUFDLG1CQUFtQixFQUFFO2dCQUM3QixHQUFHLFVBQVUsQ0FBQyxVQUFVLEVBQUU7Z0JBQzFCLFdBQVc7YUFDWixDQUFDLENBQUM7U0FDSjtLQUNGO0lBRU0sYUFBYSxDQUFDLElBQVcsRUFBRSxVQUFnQixFQUFFLFdBQW9CLEVBQUUsVUFBb0I7UUFDNUYsSUFBSSxXQUFXLEtBQUssU0FBUyxFQUFFO1lBQzdCLFdBQVcsR0FBRyxNQUFNLElBQUksQ0FBQyxRQUFRLElBQUksVUFBVSxFQUFFLENBQUM7U0FDbkQ7UUFFRCxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxHQUFHLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFLElBQUksRUFBRSxVQUFVLENBQUMsQ0FBQztRQUVoRixrQkFBa0I7UUFDbEIsSUFBSSxLQUFLLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQUMsS0FBSyxTQUFTLEVBQUU7WUFDN0MsSUFBSSxzQ0FBc0IsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO2dCQUNwQyxPQUFPLEVBQUUsSUFBSSxDQUFDLGVBQWU7Z0JBQzdCLEdBQUcsSUFBSSxDQUFDLGtCQUFrQixFQUFFO2dCQUM1QixHQUFHLFVBQVUsQ0FBQyxVQUFVLEVBQUU7Z0JBQzFCLFdBQVc7YUFDWixDQUFDLENBQUM7U0FDSjtLQUNGOzs7Ozs7SUFFTSxtQkFBbUI7UUFDeEIsT0FBTyxFQUFFLHFCQUFxQixFQUFFLElBQUksQ0FBQyxlQUFlLEVBQUUsQ0FBQztLQUN4RDs7Ozs7O0lBRU0sa0JBQWtCO1FBQ3ZCLE9BQU8sRUFBRSwwQkFBMEIsRUFBRSxJQUFJLENBQUMsZUFBZSxFQUFFLENBQUM7S0FDN0Q7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztJQUlTLGtCQUFrQixDQUMxQixJQUFXLEVBQ1gsVUFBZ0IsRUFDaEIsTUFBcUIsRUFDckIsVUFBb0I7UUFFcEIsSUFBSSxVQUFVLElBQUksaUJBQWlCLENBQUMsZUFBZSxDQUFDLElBQUksQ0FBQyxJQUFJLGVBQWUsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLEVBQUU7WUFDeEYsV0FBVztZQUNYLE1BQU0sY0FBYyxHQUFHLE1BQU0sS0FBSyxNQUFNLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDO1lBQ3pELE9BQU8sQ0FBQyxJQUFJLEVBQUUsR0FBRyxJQUFJLENBQUMsUUFBUSxJQUFJLFVBQVUsSUFBSSxjQUFjLEVBQUUsQ0FBQyxDQUFDO1NBQ25FO2FBQU07WUFDTCwrRUFBK0U7WUFDL0UsT0FBTyxDQUFDLElBQUksRUFBRSxHQUFHLE1BQU0sSUFBSSxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxJQUFJLFVBQVUsRUFBRSxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQztTQUNyRjtLQUNGO0lBRU8sVUFBVSxDQUFDLElBQVc7UUFDNUIsSUFBSSxZQUFLLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsRUFBRTtZQUNyQyxpREFBaUQ7WUFDakQscURBQXFEO1lBQ3JELDJEQUEyRDtZQUMzRCxPQUFPLElBQUksQ0FBQyxnQkFBZ0IsRUFBRSxDQUFDLENBQUMsQ0FBQyxpQkFBaUIsSUFBSSxDQUFDLGdCQUFnQixJQUFJLENBQUMsQ0FBQyxDQUFDLGdCQUFnQixDQUFDO1NBQ2hHO2FBQU07WUFDTCxPQUFPLElBQUksQ0FBQyxRQUFRLENBQUM7U0FDdEI7S0FDRjtDQUNGO0FBRUQsU0FBUyxlQUFlLENBQUMsTUFBeUIsRUFBRSxNQUF5QjtJQUMzRSxPQUFPLFlBQUssQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssWUFBSyxDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUMsQ0FBQztBQUMvQyxDQUFDOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7QUE2QkQsTUFBYSxhQUFjLFNBQVEsaUJBQWlCOzs7O0lBZ0dsRCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXlCO1FBQ2pFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQ2YsWUFBWSxFQUFFLEtBQUssQ0FBQyxpQkFBaUI7U0FDdEMsQ0FBQyxDQUFDO1FBWFksdUJBQWtCLEdBQXVDLEVBQUUsQ0FBQztRQUM1RCxzQkFBaUIsR0FBc0MsRUFBRSxDQUFDOztRQVl6RSxNQUFNLGdCQUFnQixHQUFHLEtBQUssQ0FBQyxXQUFXLElBQUksSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUM7UUFFN0QsSUFBSSxDQUFDLGdCQUFnQixHQUFHLEtBQUssQ0FBQyxnQkFBZ0IsS0FBSyxLQUFLLENBQUM7UUFFekQsSUFBSSxDQUFDLGtCQUFrQixHQUFHLEtBQUssQ0FBQyxrQkFBa0IsS0FBSyxTQUFTLENBQUMsQ0FBQztZQUNoRSxDQUFDLENBQUMsS0FBSyxDQUFDLGtCQUFrQixDQUFDLENBQUM7WUFDNUIsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLCtDQUErQyxDQUFDLENBQUM7UUFFN0UsSUFBSSxDQUFDLGFBQWEsR0FBRyxJQUFJLGdDQUFnQixDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDMUQsU0FBUyxFQUFFLElBQUksQ0FBQyxZQUFZO1lBQzVCLGdCQUFnQjtZQUNoQixvQkFBb0IsRUFBRSxXQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxrQkFBa0IsRUFBRSxFQUFFLEVBQUUsY0FBYyxFQUFFLElBQUksRUFBRSxDQUFFO1lBQ3JHLG1CQUFtQixFQUFFLFdBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLGlCQUFpQixFQUFFLEVBQUUsRUFBRSxjQUFjLEVBQUUsSUFBSSxFQUFFLENBQUU7WUFDbkcsS0FBSyxFQUFFLEtBQUssQ0FBQyxHQUFHLENBQUMsS0FBSztTQUN2QixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsZUFBZSxHQUFHLElBQUksQ0FBQyxhQUFhLENBQUMsV0FBVyxDQUFDO1FBQ3RELElBQUksQ0FBQyxrQkFBa0IsR0FBRyxJQUFJLENBQUMsYUFBYSxDQUFDLFNBQVMsQ0FBQztRQUN2RCxJQUFJLENBQUMsaUJBQWlCLEdBQUcsSUFBSSxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUM7UUFFaEQsSUFBSSxDQUFDLG9CQUFvQixFQUFFLENBQUM7S0FDN0I7Ozs7OztJQXhITSxNQUFNLENBQUMsVUFBVSxDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLGVBQXVCOztRQUM1RSxPQUFPLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEVBQUUsZUFBZSxFQUFFLENBQUMsQ0FBQztLQUNsRTs7Ozs7O0lBR00sTUFBTSxDQUFDLGNBQWMsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxlQUF1QjtRQUNoRixPQUFPLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEVBQUUsZUFBZSxFQUFFLENBQUMsQ0FBQztLQUNsRTs7Ozs7O0lBR00sTUFBTSxDQUFDLGdCQUFnQixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLGlCQUF5QixFQUFFLEdBQVM7O1FBQy9GLE9BQU8sSUFBSSxDQUFDLG9CQUFvQixDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsRUFBRSxpQkFBaUIsRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDO0tBQ3pFOzs7Ozs7Ozs7Ozs7O0lBR00sTUFBTSxDQUFDLG1CQUFtQixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLGVBQXVCLEVBQUUsVUFBc0MsRUFBRTs7UUFDL0gsTUFBTSxhQUFjLFNBQVEsaUJBQWlCO1lBQTdDOzs7Z0JBQ1Msb0JBQWUsR0FBRyxlQUFlLENBQUM7Z0JBQ2xDLHFCQUFnQixTQUFHLE9BQU8sQ0FBQyxnQkFBZ0IsbUNBQUksSUFBSSxDQUFDO1lBUTdELENBQUM7WUFOUSxhQUFhLENBQUMsSUFBVyxFQUFFLFVBQWdCLEVBQUUsV0FBb0IsRUFBRSxVQUFvQjtnQkFDNUYsNkNBQTZDO2dCQUM3QyxJQUFJLE9BQU8sQ0FBQyxnQkFBZ0IsS0FBSyxLQUFLLEVBQUU7b0JBQ3RDLEtBQUssQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRSxXQUFXLEVBQUUsVUFBVSxDQUFDLENBQUM7aUJBQ2hFO1lBQ0gsQ0FBQztTQUNGO1FBRUQsTUFBTSxlQUFnQixTQUFRLGlCQUFpQjtZQUEvQzs7O2dCQUNTLG9CQUFlLEdBQUcsZUFBZSxDQUFDO2dCQUNsQyxxQkFBZ0IsU0FBRyxPQUFPLENBQUMsZ0JBQWdCLG1DQUFJLElBQUksQ0FBQztZQVM3RCxDQUFDO1lBUFEsYUFBYSxDQUFDLEtBQVksRUFBRSxXQUFpQixFQUFFLFlBQXFCLEVBQUUsV0FBcUI7Z0JBQ2hHLGFBQWE7WUFDZixDQUFDO1lBRU0sY0FBYyxDQUFDLEtBQVksRUFBRSxXQUFpQixFQUFFLFlBQXFCLEVBQUUsV0FBcUI7Z0JBQ2pHLGFBQWE7WUFDZixDQUFDO1NBQ0Y7UUFFRCxPQUFPLE9BQU8sQ0FBQyxPQUFPLEtBQUssS0FBSztZQUM5QixDQUFDLENBQUMsSUFBSSxhQUFhLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQztZQUM5QixDQUFDLENBQUMsSUFBSSxlQUFlLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0tBQ3BDO0lBRUQ7O09BRUc7SUFDSyxNQUFNLENBQUMsb0JBQW9CLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsT0FBbUM7O1FBQ25HLElBQUksWUFBSyxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUMsZUFBZSxDQUFDLElBQUksWUFBSyxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUMsaUJBQWlCLENBQUMsSUFBSSxZQUFLLENBQUMsWUFBWSxPQUFDLE9BQU8sQ0FBQyxHQUFHLDBDQUFFLEtBQUssQ0FBQyxFQUFFO1lBQzFJLE1BQU0sSUFBSSxLQUFLLENBQUMsd0VBQXdFLENBQUMsQ0FBQztTQUMzRjtRQUVELE1BQU0sVUFBVSxHQUF1QyxzQkFBZSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUU7WUFDckYsUUFBUSxFQUFFLFFBQVEsQ0FBQyxlQUFlLENBQUMsdUJBQXVCO1lBQzFELEtBQUssRUFBRTtnQkFDTCxlQUFlLEVBQUUsT0FBTyxDQUFDLGVBQWU7Z0JBQ3hDLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxpQkFBaUI7Z0JBQzVDLEtBQUssUUFBRSxPQUFPLENBQUMsR0FBRywwQ0FBRSxLQUFLO2FBQzFCO1lBQ0QsVUFBVSxFQUFFO2dCQUNWLGVBQWUsRUFBRSxVQUFVO2dCQUMzQixnQkFBZ0IsRUFBRSxJQUFJO2FBQ2U7U0FDeEMsQ0FBQyxDQUFDLEtBQUssQ0FBQztRQUVULE9BQU8sYUFBYSxDQUFDLG1CQUFtQixDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsVUFBVSxDQUFDLGVBQWUsRUFBRTtZQUM5RSxnQkFBZ0IsRUFBRSxVQUFVLENBQUMsZ0JBQWdCO1lBQzdDLE9BQU8sRUFBRSxJQUFJO1NBQ2QsQ0FBQyxDQUFDO0tBQ0o7Ozs7Ozs7Ozs7OztJQW1ETSxjQUFjLENBQUMsSUFBVyxFQUFFLFVBQWdCLEVBQUUsV0FBb0IsRUFBRSxVQUFvQjs7O1FBQzdGLElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsSUFBSSxJQUFJLENBQUMsa0JBQWtCLEVBQUU7WUFDL0UsS0FBSyxDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFLFdBQVcsRUFBRSxVQUFVLENBQUMsQ0FBQztZQUNoRSxPQUFPO1NBQ1I7UUFFRCxJQUFJLFdBQVcsS0FBSyxTQUFTLEVBQUU7WUFDN0IsV0FBVyxHQUFHLFFBQVEsSUFBSSxDQUFDLFFBQVEsSUFBSSxVQUFVLEVBQUUsQ0FBQztTQUNyRDtRQUVELElBQUksQ0FBQyxvQkFBb0IsQ0FBQztZQUN4QixHQUFHLElBQUksQ0FBQyxtQkFBbUIsRUFBRTtZQUM3QixHQUFHLFVBQVUsQ0FBQyxVQUFVLEVBQUU7WUFDMUIsV0FBVztTQUNaLENBQUMsQ0FBQztLQUNKOzs7Ozs7Ozs7Ozs7SUFFTSxhQUFhLENBQUMsSUFBVyxFQUFFLFVBQWdCLEVBQUUsV0FBb0IsRUFBRSxVQUFvQjs7O1FBQzVGLElBQUksSUFBSSxDQUFDLGdCQUFnQixFQUFFO1lBQ3pCLHdFQUF3RTtZQUN4RSx3RUFBd0U7WUFDeEUsUUFBUTtZQUNSLElBQUksQ0FBQyxVQUFVLEVBQUUsRUFBRSxzREFBc0Q7Z0JBQ3ZFLGtCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFVBQVUsQ0FBQyx5SUFBeUksQ0FBQyxDQUFDO2FBQzVLO1lBQ0QsT0FBTztTQUNSO2FBQU07WUFDTCx1RUFBdUU7WUFDdkUsaUVBQWlFO1lBQ2pFLHlCQUF5QjtZQUN6QixJQUFJLENBQUMsbUJBQW1CLEVBQUUsQ0FBQztTQUM1QjtRQUVELElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsSUFBSSxJQUFJLENBQUMsa0JBQWtCLEVBQUU7WUFDL0UsS0FBSyxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFLFdBQVcsRUFBRSxVQUFVLENBQUMsQ0FBQztZQUMvRCxPQUFPO1NBQ1I7UUFFRCxJQUFJLFdBQVcsS0FBSyxTQUFTLEVBQUU7WUFDN0IsV0FBVyxHQUFHLFFBQVEsSUFBSSxDQUFDLFFBQVEsSUFBSSxVQUFVLEVBQUUsQ0FBQztTQUNyRDtRQUVELE1BQU0sSUFBSSxHQUFHO1lBQ1gsR0FBRyxJQUFJLENBQUMsa0JBQWtCLEVBQUU7WUFDNUIsR0FBRyxVQUFVLENBQUMsVUFBVSxFQUFFO1lBQzFCLFdBQVc7U0FDWixDQUFDO1FBRUYsSUFBSSxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsRUFBRTtZQUMxQix5RUFBeUU7WUFDekUsNEVBQTRFO1lBQzVFLG9FQUFvRTtZQUNwRSwyRUFBMkU7WUFDM0UsbURBQW1EO1lBQ25ELE1BQU0sSUFBSSxLQUFLLENBQUMsOEdBQThHLENBQUMsQ0FBQztTQUNqSTtRQUVELElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLENBQUMsQ0FBQztLQUNoQztJQUVEOztPQUVHO0lBQ0ssb0JBQW9CLENBQUMsSUFBc0M7UUFDakUsSUFBSSxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLEVBQUU7WUFDOUIsSUFBSSxDQUFDLGtCQUFrQixDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUNwQztLQUNGO0lBRUQ7O09BRUc7SUFDSyxjQUFjLENBQUMsSUFBc0M7UUFDM0QsT0FBTyxJQUFJLENBQUMsa0JBQWtCLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7S0FDaEY7SUFFRDs7T0FFRztJQUNLLG1CQUFtQixDQUFDLElBQXFDO1FBQy9ELElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxFQUFFO1lBQzdCLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7U0FDbkM7S0FDRjtJQUVEOztPQUVHO0lBQ0ssYUFBYSxDQUFDLElBQXFDO1FBQ3pELE9BQU8sSUFBSSxDQUFDLGlCQUFpQixDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO0tBQzlFO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNLLG9CQUFvQjtRQUMxQixJQUFJLElBQUksQ0FBQyxrQkFBa0IsRUFBRTtZQUMzQixNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQyxlQUFlLENBQUM7WUFDeEUsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUMsZUFBZSxDQUFDO1lBQ3hFLE1BQU0sV0FBVyxHQUFHLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsZ0JBQWdCLENBQUMsV0FBVyxDQUFDO1lBQ3RHLEtBQUssQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLElBQUksRUFBRSxXQUFXLEVBQUUsS0FBSyxDQUFDLENBQUM7U0FDckQ7YUFBTTtZQUNMLE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQSxDQUFDLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxnQkFBZ0IsQ0FBQztZQUN0RSxJQUFJLENBQUMsaUJBQWlCLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1NBQ25DO0tBQ0Y7SUFFRDs7T0FFRztJQUNLLG1CQUFtQjtRQUN6QixJQUFJLElBQUksQ0FBQyxrQkFBa0IsRUFBRTtZQUMzQixNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxHQUFHLElBQUksQ0FBQyxrQkFBa0IsQ0FDekMsZUFBZSxFQUNmLGVBQWUsRUFDZixJQUFJLEVBQ0osS0FBSyxDQUNOLENBQUM7WUFDRixLQUFLLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxFQUFFLENBQUMsQ0FBQztTQUMvQjthQUFNO1lBQ0wsTUFBTSxDQUFDLEdBQUcsSUFBSSxDQUFDLGlCQUFpQixDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUMsRUFBRSxnQkFBZ0IsQ0FBQyxDQUFDLENBQUM7WUFDdkYsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLEVBQUU7Z0JBQ1YsSUFBSSxDQUFDLGlCQUFpQixDQUFDLE1BQU0sQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7YUFDckM7U0FDRjtLQUNGOztBQXBRSCxzQ0FxUUM7OztBQUVEOzs7Ozs7OztHQVFHO0FBQ0gsTUFBTSxnQkFBZ0IsR0FBRztJQUN2QixNQUFNLEVBQUUsb0JBQW9CO0lBQzVCLFdBQVcsRUFBRSxzQkFBc0I7SUFDbkMsVUFBVSxFQUFFLE1BQU07SUFDbEIsUUFBUSxFQUFFLEdBQUc7SUFDYixNQUFNLEVBQUUsRUFBRTtDQUNYLENBQUM7QUFFRixNQUFNLGVBQWUsR0FBRyxXQUFJLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLE1BQU0sQ0FBQyxDQUFDO0FBQzNELE1BQU0sZUFBZSxHQUFHLFdBQUksQ0FBQyxlQUFlLENBQUMsZ0JBQWdCLENBQUMsUUFBUSxFQUFFLGdCQUFnQixDQUFDLE1BQU0sQ0FBQyxDQUFDO0FBRWpHOztHQUVHO0FBQ0gsTUFBTSxjQUFjLEdBQUc7SUFDckIsTUFBTSxFQUFFLFdBQVc7SUFDbkIsV0FBVyxFQUFFLHVDQUF1QztJQUNwRCxVQUFVLEVBQUUsSUFBSTtDQUNqQixDQUFDO0FBRUYsTUFBTSxnQkFBZ0IsR0FBRyxXQUFJLENBQUMsT0FBTyxFQUFFLENBQUM7QUFDeEMsTUFBTSxnQkFBZ0IsR0FBRyxXQUFJLENBQUMsVUFBVSxFQUFFLENBQUM7QUFnQjNDOztHQUVHO0FBQ0gsU0FBUyxpQkFBaUIsQ0FBQyxDQUFtQyxFQUFFLENBQW1DO0lBQ2pHLE9BQU8sQ0FBQyxDQUFDLE1BQU0sS0FBSyxDQUFDLENBQUMsTUFBTTtXQUN2QixDQUFDLENBQUMsUUFBUSxLQUFLLENBQUMsQ0FBQyxRQUFRO1dBQ3pCLENBQUMsQ0FBQyxRQUFRLEtBQUssQ0FBQyxDQUFDLFFBQVE7V0FDekIsQ0FBQyxDQUFDLE1BQU0sS0FBSyxDQUFDLENBQUMsTUFBTTtXQUNyQixDQUFDLENBQUMsVUFBVSxLQUFLLENBQUMsQ0FBQyxVQUFVO1dBQzdCLENBQUMsQ0FBQyxxQkFBcUIsS0FBSyxDQUFDLENBQUMscUJBQXFCO1dBQ25ELENBQUMsQ0FBQyx1QkFBdUIsS0FBSyxDQUFDLENBQUMsdUJBQXVCO1dBQ3ZELENBQUMsQ0FBQywwQkFBMEIsS0FBSyxDQUFDLENBQUMsMEJBQTBCLENBQUM7QUFDckUsQ0FBQztBQUVEOztHQUVHO0FBQ0gsU0FBUyxnQkFBZ0IsQ0FBQyxDQUFrQyxFQUFFLENBQWtDO0lBQzlGLE9BQU8sQ0FBQyxDQUFDLE1BQU0sS0FBSyxDQUFDLENBQUMsTUFBTTtXQUN2QixDQUFDLENBQUMsUUFBUSxLQUFLLENBQUMsQ0FBQyxRQUFRO1dBQ3pCLENBQUMsQ0FBQyxRQUFRLEtBQUssQ0FBQyxDQUFDLFFBQVE7V0FDekIsQ0FBQyxDQUFDLE1BQU0sS0FBSyxDQUFDLENBQUMsTUFBTTtXQUNyQixDQUFDLENBQUMsVUFBVSxLQUFLLENBQUMsQ0FBQyxVQUFVO1dBQzdCLENBQUMsQ0FBQyx1QkFBdUIsS0FBSyxDQUFDLENBQUMsdUJBQXVCO1dBQ3ZELENBQUMsQ0FBQywwQkFBMEIsS0FBSyxDQUFDLENBQUMsMEJBQTBCLENBQUM7QUFDckUsQ0FBQztBQUVEOztHQUVHO0FBQ0gsU0FBUyxnQkFBZ0IsQ0FBQyxJQUFTO0lBQ2pDLE9BQU8sSUFBSSxDQUFDLE1BQU0sS0FBSyxXQUFXLElBQUksSUFBSSxDQUFDLFVBQVUsS0FBSyxJQUFJLENBQUM7QUFDakUsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGN4c2NoZW1hIGZyb20gJ0Bhd3MtY2RrL2Nsb3VkLWFzc2VtYmx5LXNjaGVtYSc7XG5pbXBvcnQgeyBBbm5vdGF0aW9ucywgQ29udGV4dFByb3ZpZGVyLCBJUmVzb3VyY2UsIExhenksIE5hbWVzLCBSZXNvdXJjZSwgUmVzb3VyY2VQcm9wcywgU3RhY2ssIFRva2VuIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgKiBhcyBjeGFwaSBmcm9tICdAYXdzLWNkay9jeC1hcGknO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBDb25uZWN0aW9ucyB9IGZyb20gJy4vY29ubmVjdGlvbnMnO1xuaW1wb3J0IHsgQ2ZuU2VjdXJpdHlHcm91cCwgQ2ZuU2VjdXJpdHlHcm91cEVncmVzcywgQ2ZuU2VjdXJpdHlHcm91cEluZ3Jlc3MgfSBmcm9tICcuL2VjMi5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgSVBlZXIsIFBlZXIgfSBmcm9tICcuL3BlZXInO1xuaW1wb3J0IHsgUG9ydCB9IGZyb20gJy4vcG9ydCc7XG5pbXBvcnQgeyBJVnBjIH0gZnJvbSAnLi92cGMnO1xuXG5jb25zdCBTRUNVUklUWV9HUk9VUF9TWU1CT0wgPSBTeW1ib2wuZm9yKCdAYXdzLWNkay9pYW0uU2VjdXJpdHlHcm91cCcpO1xuXG5jb25zdCBTRUNVUklUWV9HUk9VUF9ESVNBQkxFX0lOTElORV9SVUxFU19DT05URVhUX0tFWSA9ICdAYXdzLWNkay9hd3MtZWMyLnNlY3VyaXR5R3JvdXBEaXNhYmxlSW5saW5lUnVsZXMnO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIElTZWN1cml0eUdyb3VwIGV4dGVuZHMgSVJlc291cmNlLCBJUGVlciB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgc2VjdXJpdHlHcm91cElkOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGFsbG93QWxsT3V0Ym91bmQ6IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIGFkZEluZ3Jlc3NSdWxlKHBlZXI6IElQZWVyLCBjb25uZWN0aW9uOiBQb3J0LCBkZXNjcmlwdGlvbj86IHN0cmluZywgcmVtb3RlUnVsZT86IGJvb2xlYW4pOiB2b2lkO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIGFkZEVncmVzc1J1bGUocGVlcjogSVBlZXIsIGNvbm5lY3Rpb246IFBvcnQsIGRlc2NyaXB0aW9uPzogc3RyaW5nLCByZW1vdGVSdWxlPzogYm9vbGVhbik6IHZvaWQ7XG59XG5cbi8qKlxuICogQSBTZWN1cml0eUdyb3VwIHRoYXQgaXMgbm90IGNyZWF0ZWQgaW4gdGhpcyB0ZW1wbGF0ZVxuICovXG5hYnN0cmFjdCBjbGFzcyBTZWN1cml0eUdyb3VwQmFzZSBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSVNlY3VyaXR5R3JvdXAge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgaXNTZWN1cml0eUdyb3VwKHg6IGFueSk6IHggaXMgU2VjdXJpdHlHcm91cEJhc2Uge1xuICAgIHJldHVybiBTRUNVUklUWV9HUk9VUF9TWU1CT0wgaW4geDtcbiAgfVxuXG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBzZWN1cml0eUdyb3VwSWQ6IHN0cmluZztcbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGFsbG93QWxsT3V0Ym91bmQ6IGJvb2xlYW47XG5cbiAgcHVibGljIHJlYWRvbmx5IGNhbklubGluZVJ1bGUgPSBmYWxzZTtcbiAgcHVibGljIHJlYWRvbmx5IGNvbm5lY3Rpb25zOiBDb25uZWN0aW9ucyA9IG5ldyBDb25uZWN0aW9ucyh7IHNlY3VyaXR5R3JvdXBzOiBbdGhpc10gfSk7XG4gIHB1YmxpYyByZWFkb25seSBkZWZhdWx0UG9ydD86IFBvcnQ7XG5cbiAgcHJpdmF0ZSBwZWVyQXNUb2tlbkNvdW50OiBudW1iZXIgPSAwO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzPzogUmVzb3VyY2VQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwgcHJvcHMpO1xuXG4gICAgT2JqZWN0LmRlZmluZVByb3BlcnR5KHRoaXMsIFNFQ1VSSVRZX0dST1VQX1NZTUJPTCwgeyB2YWx1ZTogdHJ1ZSB9KTtcbiAgfVxuXG4gIHB1YmxpYyBnZXQgdW5pcXVlSWQoKSB7XG4gICAgcmV0dXJuIE5hbWVzLm5vZGVVbmlxdWVJZCh0aGlzLm5vZGUpO1xuICB9XG5cbiAgcHVibGljIGFkZEluZ3Jlc3NSdWxlKHBlZXI6IElQZWVyLCBjb25uZWN0aW9uOiBQb3J0LCBkZXNjcmlwdGlvbj86IHN0cmluZywgcmVtb3RlUnVsZT86IGJvb2xlYW4pIHtcbiAgICBpZiAoZGVzY3JpcHRpb24gPT09IHVuZGVmaW5lZCkge1xuICAgICAgZGVzY3JpcHRpb24gPSBgZnJvbSAke3BlZXIudW5pcXVlSWR9OiR7Y29ubmVjdGlvbn1gO1xuICAgIH1cblxuICAgIGNvbnN0IFtzY29wZSwgaWRdID0gdGhpcy5kZXRlcm1pbmVSdWxlU2NvcGUocGVlciwgY29ubmVjdGlvbiwgJ2Zyb20nLCByZW1vdGVSdWxlKTtcblxuICAgIC8vIFNraXAgZHVwbGljYXRlc1xuICAgIGlmIChzY29wZS5ub2RlLnRyeUZpbmRDaGlsZChpZCkgPT09IHVuZGVmaW5lZCkge1xuICAgICAgbmV3IENmblNlY3VyaXR5R3JvdXBJbmdyZXNzKHNjb3BlLCBpZCwge1xuICAgICAgICBncm91cElkOiB0aGlzLnNlY3VyaXR5R3JvdXBJZCxcbiAgICAgICAgLi4ucGVlci50b0luZ3Jlc3NSdWxlQ29uZmlnKCksXG4gICAgICAgIC4uLmNvbm5lY3Rpb24udG9SdWxlSnNvbigpLFxuICAgICAgICBkZXNjcmlwdGlvbixcbiAgICAgIH0pO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBhZGRFZ3Jlc3NSdWxlKHBlZXI6IElQZWVyLCBjb25uZWN0aW9uOiBQb3J0LCBkZXNjcmlwdGlvbj86IHN0cmluZywgcmVtb3RlUnVsZT86IGJvb2xlYW4pIHtcbiAgICBpZiAoZGVzY3JpcHRpb24gPT09IHVuZGVmaW5lZCkge1xuICAgICAgZGVzY3JpcHRpb24gPSBgdG8gJHtwZWVyLnVuaXF1ZUlkfToke2Nvbm5lY3Rpb259YDtcbiAgICB9XG5cbiAgICBjb25zdCBbc2NvcGUsIGlkXSA9IHRoaXMuZGV0ZXJtaW5lUnVsZVNjb3BlKHBlZXIsIGNvbm5lY3Rpb24sICd0bycsIHJlbW90ZVJ1bGUpO1xuXG4gICAgLy8gU2tpcCBkdXBsaWNhdGVzXG4gICAgaWYgKHNjb3BlLm5vZGUudHJ5RmluZENoaWxkKGlkKSA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICBuZXcgQ2ZuU2VjdXJpdHlHcm91cEVncmVzcyhzY29wZSwgaWQsIHtcbiAgICAgICAgZ3JvdXBJZDogdGhpcy5zZWN1cml0eUdyb3VwSWQsXG4gICAgICAgIC4uLnBlZXIudG9FZ3Jlc3NSdWxlQ29uZmlnKCksXG4gICAgICAgIC4uLmNvbm5lY3Rpb24udG9SdWxlSnNvbigpLFxuICAgICAgICBkZXNjcmlwdGlvbixcbiAgICAgIH0pO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyB0b0luZ3Jlc3NSdWxlQ29uZmlnKCk6IGFueSB7XG4gICAgcmV0dXJuIHsgc291cmNlU2VjdXJpdHlHcm91cElkOiB0aGlzLnNlY3VyaXR5R3JvdXBJZCB9O1xuICB9XG5cbiAgcHVibGljIHRvRWdyZXNzUnVsZUNvbmZpZygpOiBhbnkge1xuICAgIHJldHVybiB7IGRlc3RpbmF0aW9uU2VjdXJpdHlHcm91cElkOiB0aGlzLnNlY3VyaXR5R3JvdXBJZCB9O1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuXG4gIHByb3RlY3RlZCBkZXRlcm1pbmVSdWxlU2NvcGUoXG4gICAgcGVlcjogSVBlZXIsXG4gICAgY29ubmVjdGlvbjogUG9ydCxcbiAgICBmcm9tVG86ICdmcm9tJyB8ICd0bycsXG4gICAgcmVtb3RlUnVsZT86IGJvb2xlYW4pOiBbU2VjdXJpdHlHcm91cEJhc2UsIHN0cmluZ10ge1xuXG4gICAgaWYgKHJlbW90ZVJ1bGUgJiYgU2VjdXJpdHlHcm91cEJhc2UuaXNTZWN1cml0eUdyb3VwKHBlZXIpICYmIGRpZmZlcmVudFN0YWNrcyh0aGlzLCBwZWVyKSkge1xuICAgICAgLy8gUmV2ZXJzZWRcbiAgICAgIGNvbnN0IHJldmVyc2VkRnJvbVRvID0gZnJvbVRvID09PSAnZnJvbScgPyAndG8nIDogJ2Zyb20nO1xuICAgICAgcmV0dXJuIFtwZWVyLCBgJHt0aGlzLnVuaXF1ZUlkfToke2Nvbm5lY3Rpb259ICR7cmV2ZXJzZWRGcm9tVG99YF07XG4gICAgfSBlbHNlIHtcbiAgICAgIC8vIFJlZ3VsYXIgKGRvIG9sZCBJRCBlc2NhcGluZyB0byBpbiBvcmRlciB0byBub3QgZGlzdHVyYiBleGlzdGluZyBkZXBsb3ltZW50cylcbiAgICAgIHJldHVybiBbdGhpcywgYCR7ZnJvbVRvfSAke3RoaXMucmVuZGVyUGVlcihwZWVyKX06JHtjb25uZWN0aW9ufWAucmVwbGFjZSgnLycsICdfJyldO1xuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgcmVuZGVyUGVlcihwZWVyOiBJUGVlcikge1xuICAgIGlmIChUb2tlbi5pc1VucmVzb2x2ZWQocGVlci51bmlxdWVJZCkpIHtcbiAgICAgIC8vIE5lZWQgdG8gcmV0dXJuIGEgdW5pcXVlIHZhbHVlIGVhY2ggdGltZSBhIHBlZXJcbiAgICAgIC8vIGlzIGFuIHVucmVzb2x2ZWQgdG9rZW4sIGVsc2UgdGhlIGR1cGxpY2F0ZSBza2lwcGVyXG4gICAgICAvLyBpbiBgc2cuYWRkWHh4UnVsZWAgY2FuIGRldGVjdCB1bmlxdWUgcnVsZXMgYXMgZHVwbGljYXRlc1xuICAgICAgcmV0dXJuIHRoaXMucGVlckFzVG9rZW5Db3VudCsrID8gYCd7SW5kaXJlY3RQZWVyJHt0aGlzLnBlZXJBc1Rva2VuQ291bnR9fSdgIDogJ3tJbmRpcmVjdFBlZXJ9JztcbiAgICB9IGVsc2Uge1xuICAgICAgcmV0dXJuIHBlZXIudW5pcXVlSWQ7XG4gICAgfVxuICB9XG59XG5cbmZ1bmN0aW9uIGRpZmZlcmVudFN0YWNrcyhncm91cDE6IFNlY3VyaXR5R3JvdXBCYXNlLCBncm91cDI6IFNlY3VyaXR5R3JvdXBCYXNlKSB7XG4gIHJldHVybiBTdGFjay5vZihncm91cDEpICE9PSBTdGFjay5vZihncm91cDIpO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIFNlY3VyaXR5R3JvdXBQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHNlY3VyaXR5R3JvdXBOYW1lPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZGVzY3JpcHRpb24/OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB2cGM6IElWcGM7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGFsbG93QWxsT3V0Ym91bmQ/OiBib29sZWFuO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZGlzYWJsZUlubGluZVJ1bGVzPzogYm9vbGVhbjtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBTZWN1cml0eUdyb3VwSW1wb3J0T3B0aW9ucyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBhbGxvd0FsbE91dGJvdW5kPzogYm9vbGVhbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBtdXRhYmxlPzogYm9vbGVhbjtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgU2VjdXJpdHlHcm91cCBleHRlbmRzIFNlY3VyaXR5R3JvdXBCYXNlIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUxvb2t1cChzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBzZWN1cml0eUdyb3VwSWQ6IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmZyb21Mb29rdXBBdHRyaWJ1dGVzKHNjb3BlLCBpZCwgeyBzZWN1cml0eUdyb3VwSWQgfSk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUxvb2t1cEJ5SWQoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgc2VjdXJpdHlHcm91cElkOiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5mcm9tTG9va3VwQXR0cmlidXRlcyhzY29wZSwgaWQsIHsgc2VjdXJpdHlHcm91cElkIH0pO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUxvb2t1cEJ5TmFtZShzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBzZWN1cml0eUdyb3VwTmFtZTogc3RyaW5nLCB2cGM6IElWcGMpIHtcbiAgICByZXR1cm4gdGhpcy5mcm9tTG9va3VwQXR0cmlidXRlcyhzY29wZSwgaWQsIHsgc2VjdXJpdHlHcm91cE5hbWUsIHZwYyB9KTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBmcm9tU2VjdXJpdHlHcm91cElkKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHNlY3VyaXR5R3JvdXBJZDogc3RyaW5nLCBvcHRpb25zOiBTZWN1cml0eUdyb3VwSW1wb3J0T3B0aW9ucyA9IHt9KTogSVNlY3VyaXR5R3JvdXAge1xuICAgIGNsYXNzIE11dGFibGVJbXBvcnQgZXh0ZW5kcyBTZWN1cml0eUdyb3VwQmFzZSB7XG4gICAgICBwdWJsaWMgc2VjdXJpdHlHcm91cElkID0gc2VjdXJpdHlHcm91cElkO1xuICAgICAgcHVibGljIGFsbG93QWxsT3V0Ym91bmQgPSBvcHRpb25zLmFsbG93QWxsT3V0Ym91bmQgPz8gdHJ1ZTtcblxuICAgICAgcHVibGljIGFkZEVncmVzc1J1bGUocGVlcjogSVBlZXIsIGNvbm5lY3Rpb246IFBvcnQsIGRlc2NyaXB0aW9uPzogc3RyaW5nLCByZW1vdGVSdWxlPzogYm9vbGVhbikge1xuICAgICAgICAvLyBPbmx5IGlmIGFsbG93QWxsT3V0Ym91bmQgaGFzIGJlZW4gZGlzYWJsZWRcbiAgICAgICAgaWYgKG9wdGlvbnMuYWxsb3dBbGxPdXRib3VuZCA9PT0gZmFsc2UpIHtcbiAgICAgICAgICBzdXBlci5hZGRFZ3Jlc3NSdWxlKHBlZXIsIGNvbm5lY3Rpb24sIGRlc2NyaXB0aW9uLCByZW1vdGVSdWxlKTtcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cblxuICAgIGNsYXNzIEltbXV0YWJsZUltcG9ydCBleHRlbmRzIFNlY3VyaXR5R3JvdXBCYXNlIHtcbiAgICAgIHB1YmxpYyBzZWN1cml0eUdyb3VwSWQgPSBzZWN1cml0eUdyb3VwSWQ7XG4gICAgICBwdWJsaWMgYWxsb3dBbGxPdXRib3VuZCA9IG9wdGlvbnMuYWxsb3dBbGxPdXRib3VuZCA/PyB0cnVlO1xuXG4gICAgICBwdWJsaWMgYWRkRWdyZXNzUnVsZShfcGVlcjogSVBlZXIsIF9jb25uZWN0aW9uOiBQb3J0LCBfZGVzY3JpcHRpb24/OiBzdHJpbmcsIF9yZW1vdGVSdWxlPzogYm9vbGVhbikge1xuICAgICAgICAvLyBkbyBub3RoaW5nXG4gICAgICB9XG5cbiAgICAgIHB1YmxpYyBhZGRJbmdyZXNzUnVsZShfcGVlcjogSVBlZXIsIF9jb25uZWN0aW9uOiBQb3J0LCBfZGVzY3JpcHRpb24/OiBzdHJpbmcsIF9yZW1vdGVSdWxlPzogYm9vbGVhbikge1xuICAgICAgICAvLyBkbyBub3RoaW5nXG4gICAgICB9XG4gICAgfVxuXG4gICAgcmV0dXJuIG9wdGlvbnMubXV0YWJsZSAhPT0gZmFsc2VcbiAgICAgID8gbmV3IE11dGFibGVJbXBvcnQoc2NvcGUsIGlkKVxuICAgICAgOiBuZXcgSW1tdXRhYmxlSW1wb3J0KHNjb3BlLCBpZCk7XG4gIH1cblxuICAvKipcbiAgICogTG9vayB1cCBhIHNlY3VyaXR5IGdyb3VwLlxuICAgKi9cbiAgcHJpdmF0ZSBzdGF0aWMgZnJvbUxvb2t1cEF0dHJpYnV0ZXMoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgb3B0aW9uczogU2VjdXJpdHlHcm91cExvb2t1cE9wdGlvbnMpIHtcbiAgICBpZiAoVG9rZW4uaXNVbnJlc29sdmVkKG9wdGlvbnMuc2VjdXJpdHlHcm91cElkKSB8fMKgVG9rZW4uaXNVbnJlc29sdmVkKG9wdGlvbnMuc2VjdXJpdHlHcm91cE5hbWUpIHx8IFRva2VuLmlzVW5yZXNvbHZlZChvcHRpb25zLnZwYz8udnBjSWQpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0FsbCBhcmd1bWVudHMgdG8gbG9vayB1cCBhIHNlY3VyaXR5IGdyb3VwIG11c3QgYmUgY29uY3JldGUgKG5vIFRva2VucyknKTtcbiAgICB9XG5cbiAgICBjb25zdCBhdHRyaWJ1dGVzOiBjeGFwaS5TZWN1cml0eUdyb3VwQ29udGV4dFJlc3BvbnNlID0gQ29udGV4dFByb3ZpZGVyLmdldFZhbHVlKHNjb3BlLCB7XG4gICAgICBwcm92aWRlcjogY3hzY2hlbWEuQ29udGV4dFByb3ZpZGVyLlNFQ1VSSVRZX0dST1VQX1BST1ZJREVSLFxuICAgICAgcHJvcHM6IHtcbiAgICAgICAgc2VjdXJpdHlHcm91cElkOiBvcHRpb25zLnNlY3VyaXR5R3JvdXBJZCxcbiAgICAgICAgc2VjdXJpdHlHcm91cE5hbWU6IG9wdGlvbnMuc2VjdXJpdHlHcm91cE5hbWUsXG4gICAgICAgIHZwY0lkOiBvcHRpb25zLnZwYz8udnBjSWQsXG4gICAgICB9LFxuICAgICAgZHVtbXlWYWx1ZToge1xuICAgICAgICBzZWN1cml0eUdyb3VwSWQ6ICdzZy0xMjM0NScsXG4gICAgICAgIGFsbG93QWxsT3V0Ym91bmQ6IHRydWUsXG4gICAgICB9IGFzIGN4YXBpLlNlY3VyaXR5R3JvdXBDb250ZXh0UmVzcG9uc2UsXG4gICAgfSkudmFsdWU7XG5cbiAgICByZXR1cm4gU2VjdXJpdHlHcm91cC5mcm9tU2VjdXJpdHlHcm91cElkKHNjb3BlLCBpZCwgYXR0cmlidXRlcy5zZWN1cml0eUdyb3VwSWQsIHtcbiAgICAgIGFsbG93QWxsT3V0Ym91bmQ6IGF0dHJpYnV0ZXMuYWxsb3dBbGxPdXRib3VuZCxcbiAgICAgIG11dGFibGU6IHRydWUsXG4gICAgfSk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IHNlY3VyaXR5R3JvdXBOYW1lOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgc2VjdXJpdHlHcm91cElkOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBzZWN1cml0eUdyb3VwVnBjSWQ6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGFsbG93QWxsT3V0Ym91bmQ6IGJvb2xlYW47XG5cbiAgcHJpdmF0ZSByZWFkb25seSBzZWN1cml0eUdyb3VwOiBDZm5TZWN1cml0eUdyb3VwO1xuICBwcml2YXRlIHJlYWRvbmx5IGRpcmVjdEluZ3Jlc3NSdWxlczogQ2ZuU2VjdXJpdHlHcm91cC5JbmdyZXNzUHJvcGVydHlbXSA9IFtdO1xuICBwcml2YXRlIHJlYWRvbmx5IGRpcmVjdEVncmVzc1J1bGVzOiBDZm5TZWN1cml0eUdyb3VwLkVncmVzc1Byb3BlcnR5W10gPSBbXTtcblxuICAvKipcbiAgICogV2hldGhlciB0byBkaXNhYmxlIG9wdGltaXphdGlvbiBmb3IgaW5saW5lIHNlY3VyaXR5IGdyb3VwIHJ1bGVzLlxuICAgKi9cbiAgcHJpdmF0ZSByZWFkb25seSBkaXNhYmxlSW5saW5lUnVsZXM6IGJvb2xlYW47XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFNlY3VyaXR5R3JvdXBQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwge1xuICAgICAgcGh5c2ljYWxOYW1lOiBwcm9wcy5zZWN1cml0eUdyb3VwTmFtZSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGdyb3VwRGVzY3JpcHRpb24gPSBwcm9wcy5kZXNjcmlwdGlvbiB8fCB0aGlzLm5vZGUucGF0aDtcblxuICAgIHRoaXMuYWxsb3dBbGxPdXRib3VuZCA9IHByb3BzLmFsbG93QWxsT3V0Ym91bmQgIT09IGZhbHNlO1xuXG4gICAgdGhpcy5kaXNhYmxlSW5saW5lUnVsZXMgPSBwcm9wcy5kaXNhYmxlSW5saW5lUnVsZXMgIT09IHVuZGVmaW5lZCA/XG4gICAgICAhIXByb3BzLmRpc2FibGVJbmxpbmVSdWxlcyA6XG4gICAgICAhIXRoaXMubm9kZS50cnlHZXRDb250ZXh0KFNFQ1VSSVRZX0dST1VQX0RJU0FCTEVfSU5MSU5FX1JVTEVTX0NPTlRFWFRfS0VZKTtcblxuICAgIHRoaXMuc2VjdXJpdHlHcm91cCA9IG5ldyBDZm5TZWN1cml0eUdyb3VwKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIGdyb3VwTmFtZTogdGhpcy5waHlzaWNhbE5hbWUsXG4gICAgICBncm91cERlc2NyaXB0aW9uLFxuICAgICAgc2VjdXJpdHlHcm91cEluZ3Jlc3M6IExhenkuYW55KHsgcHJvZHVjZTogKCkgPT4gdGhpcy5kaXJlY3RJbmdyZXNzUnVsZXMgfSwgeyBvbWl0RW1wdHlBcnJheTogdHJ1ZSB9ICksXG4gICAgICBzZWN1cml0eUdyb3VwRWdyZXNzOiBMYXp5LmFueSh7IHByb2R1Y2U6ICgpID0+IHRoaXMuZGlyZWN0RWdyZXNzUnVsZXMgfSwgeyBvbWl0RW1wdHlBcnJheTogdHJ1ZSB9ICksXG4gICAgICB2cGNJZDogcHJvcHMudnBjLnZwY0lkLFxuICAgIH0pO1xuXG4gICAgdGhpcy5zZWN1cml0eUdyb3VwSWQgPSB0aGlzLnNlY3VyaXR5R3JvdXAuYXR0ckdyb3VwSWQ7XG4gICAgdGhpcy5zZWN1cml0eUdyb3VwVnBjSWQgPSB0aGlzLnNlY3VyaXR5R3JvdXAuYXR0clZwY0lkO1xuICAgIHRoaXMuc2VjdXJpdHlHcm91cE5hbWUgPSB0aGlzLnNlY3VyaXR5R3JvdXAucmVmO1xuXG4gICAgdGhpcy5hZGREZWZhdWx0RWdyZXNzUnVsZSgpO1xuICB9XG5cbiAgcHVibGljIGFkZEluZ3Jlc3NSdWxlKHBlZXI6IElQZWVyLCBjb25uZWN0aW9uOiBQb3J0LCBkZXNjcmlwdGlvbj86IHN0cmluZywgcmVtb3RlUnVsZT86IGJvb2xlYW4pIHtcbiAgICBpZiAoIXBlZXIuY2FuSW5saW5lUnVsZSB8fCAhY29ubmVjdGlvbi5jYW5JbmxpbmVSdWxlIHx8IHRoaXMuZGlzYWJsZUlubGluZVJ1bGVzKSB7XG4gICAgICBzdXBlci5hZGRJbmdyZXNzUnVsZShwZWVyLCBjb25uZWN0aW9uLCBkZXNjcmlwdGlvbiwgcmVtb3RlUnVsZSk7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgaWYgKGRlc2NyaXB0aW9uID09PSB1bmRlZmluZWQpIHtcbiAgICAgIGRlc2NyaXB0aW9uID0gYGZyb20gJHtwZWVyLnVuaXF1ZUlkfToke2Nvbm5lY3Rpb259YDtcbiAgICB9XG5cbiAgICB0aGlzLmFkZERpcmVjdEluZ3Jlc3NSdWxlKHtcbiAgICAgIC4uLnBlZXIudG9JbmdyZXNzUnVsZUNvbmZpZygpLFxuICAgICAgLi4uY29ubmVjdGlvbi50b1J1bGVKc29uKCksXG4gICAgICBkZXNjcmlwdGlvbixcbiAgICB9KTtcbiAgfVxuXG4gIHB1YmxpYyBhZGRFZ3Jlc3NSdWxlKHBlZXI6IElQZWVyLCBjb25uZWN0aW9uOiBQb3J0LCBkZXNjcmlwdGlvbj86IHN0cmluZywgcmVtb3RlUnVsZT86IGJvb2xlYW4pIHtcbiAgICBpZiAodGhpcy5hbGxvd0FsbE91dGJvdW5kKSB7XG4gICAgICAvLyBJbiB0aGUgY2FzZSBvZiBcImFsbG93QWxsT3V0Ym91bmRcIiwgd2UgZG9uJ3QgYWRkIGFueSBtb3JlIHJ1bGVzLiBUaGVyZVxuICAgICAgLy8gaXMgb25seSBvbmUgcnVsZSB3aGljaCBhbGxvd3MgYWxsIHRyYWZmaWMgYW5kIHRoYXQgc3Vic3VtZXMgYW55IG90aGVyXG4gICAgICAvLyBydWxlLlxuICAgICAgaWYgKCFyZW1vdGVSdWxlKSB7IC8vIFdhcm4gb25seSBpZiBhZGRFZ3Jlc3NSdWxlKCkgd2FzIGV4cGxpY2l0ZWx5IGNhbGxlZFxuICAgICAgICBBbm5vdGF0aW9ucy5vZih0aGlzKS5hZGRXYXJuaW5nKCdJZ25vcmluZyBFZ3Jlc3MgcnVsZSBzaW5jZSBcXCdhbGxvd0FsbE91dGJvdW5kXFwnIGlzIHNldCB0byB0cnVlOyBUbyBhZGQgY3VzdG9taXplIHJ1bGVzLCBzZXQgYWxsb3dBbGxPdXRib3VuZD1mYWxzZSBvbiB0aGUgU2VjdXJpdHlHcm91cCcpO1xuICAgICAgfVxuICAgICAgcmV0dXJuO1xuICAgIH0gZWxzZSB7XG4gICAgICAvLyBPdGhlcndpc2UsIGlmIHRoZSBib2d1cyBydWxlIGV4aXN0cyB3ZSBjYW4gbm93IHJlbW92ZSBpdCBiZWNhdXNlIHRoZVxuICAgICAgLy8gcHJlc2VuY2Ugb2YgYW55IG90aGVyIHJ1bGUgd2lsbCBnZXQgcmlkIG9mIEVDMidzIGltcGxpY2l0IFwiYWxsXG4gICAgICAvLyBvdXRib3VuZFwiIHJ1bGUgYW55d2F5LlxuICAgICAgdGhpcy5yZW1vdmVOb1RyYWZmaWNSdWxlKCk7XG4gICAgfVxuXG4gICAgaWYgKCFwZWVyLmNhbklubGluZVJ1bGUgfHwgIWNvbm5lY3Rpb24uY2FuSW5saW5lUnVsZSB8fCB0aGlzLmRpc2FibGVJbmxpbmVSdWxlcykge1xuICAgICAgc3VwZXIuYWRkRWdyZXNzUnVsZShwZWVyLCBjb25uZWN0aW9uLCBkZXNjcmlwdGlvbiwgcmVtb3RlUnVsZSk7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgaWYgKGRlc2NyaXB0aW9uID09PSB1bmRlZmluZWQpIHtcbiAgICAgIGRlc2NyaXB0aW9uID0gYGZyb20gJHtwZWVyLnVuaXF1ZUlkfToke2Nvbm5lY3Rpb259YDtcbiAgICB9XG5cbiAgICBjb25zdCBydWxlID0ge1xuICAgICAgLi4ucGVlci50b0VncmVzc1J1bGVDb25maWcoKSxcbiAgICAgIC4uLmNvbm5lY3Rpb24udG9SdWxlSnNvbigpLFxuICAgICAgZGVzY3JpcHRpb24sXG4gICAgfTtcblxuICAgIGlmIChpc0FsbFRyYWZmaWNSdWxlKHJ1bGUpKSB7XG4gICAgICAvLyBXZSBjYW5ub3QgYWxsb3cgdGhpczsgaWYgc29tZW9uZSBhZGRzIHRoZSBydWxlIGluIHRoaXMgd2F5LCBpdCB3aWxsIGJlXG4gICAgICAvLyByZW1vdmVkIGFnYWluIGlmIHRoZXkgYWRkIG90aGVyIHJ1bGVzLiBXZSBhbHNvIGNhbid0IGF1dG9tYXRpY2FsbHkgc3dpdGNoXG4gICAgICAvLyB0byBcImFsbE91dGJvdW5kPXRydWVcIiBtb2RlLCBiZWNhdXNlIHdlIG1pZ2h0IGhhdmUgYWxyZWFkeSBlbWl0dGVkXG4gICAgICAvLyBFZ3Jlc3NSdWxlIG9iamVjdHMgKHdoaWNoIGNvdW50IGFzIHJ1bGVzIGFkZGVkIGxhdGVyKSBhbmQgdGhlcmUncyBubyB3YXlcbiAgICAgIC8vIHRvIHJlY2FsbCB0aG9zZS4gQmV0dGVyIHRvIHByZXZlbnQgdGhpcyBmb3Igbm93LlxuICAgICAgdGhyb3cgbmV3IEVycm9yKCdDYW5ub3QgYWRkIGFuIFwiYWxsIHRyYWZmaWNcIiBlZ3Jlc3MgcnVsZSBpbiB0aGlzIHdheTsgc2V0IGFsbG93QWxsT3V0Ym91bmQ9dHJ1ZSBvbiB0aGUgU2VjdXJpdHlHcm91cCBpbnN0ZWFkLicpO1xuICAgIH1cblxuICAgIHRoaXMuYWRkRGlyZWN0RWdyZXNzUnVsZShydWxlKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgYSBkaXJlY3QgaW5ncmVzcyBydWxlXG4gICAqL1xuICBwcml2YXRlIGFkZERpcmVjdEluZ3Jlc3NSdWxlKHJ1bGU6IENmblNlY3VyaXR5R3JvdXAuSW5ncmVzc1Byb3BlcnR5KSB7XG4gICAgaWYgKCF0aGlzLmhhc0luZ3Jlc3NSdWxlKHJ1bGUpKSB7XG4gICAgICB0aGlzLmRpcmVjdEluZ3Jlc3NSdWxlcy5wdXNoKHJ1bGUpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm4gd2hldGhlciB0aGUgZ2l2ZW4gaW5ncmVzcyBydWxlIGV4aXN0cyBvbiB0aGUgZ3JvdXBcbiAgICovXG4gIHByaXZhdGUgaGFzSW5ncmVzc1J1bGUocnVsZTogQ2ZuU2VjdXJpdHlHcm91cC5JbmdyZXNzUHJvcGVydHkpOiBib29sZWFuIHtcbiAgICByZXR1cm4gdGhpcy5kaXJlY3RJbmdyZXNzUnVsZXMuZmluZEluZGV4KHIgPT4gaW5ncmVzc1J1bGVzRXF1YWwociwgcnVsZSkpID4gLTE7XG4gIH1cblxuICAvKipcbiAgICogQWRkIGEgZGlyZWN0IGVncmVzcyBydWxlXG4gICAqL1xuICBwcml2YXRlIGFkZERpcmVjdEVncmVzc1J1bGUocnVsZTogQ2ZuU2VjdXJpdHlHcm91cC5FZ3Jlc3NQcm9wZXJ0eSkge1xuICAgIGlmICghdGhpcy5oYXNFZ3Jlc3NSdWxlKHJ1bGUpKSB7XG4gICAgICB0aGlzLmRpcmVjdEVncmVzc1J1bGVzLnB1c2gocnVsZSk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybiB3aGV0aGVyIHRoZSBnaXZlbiBlZ3Jlc3MgcnVsZSBleGlzdHMgb24gdGhlIGdyb3VwXG4gICAqL1xuICBwcml2YXRlIGhhc0VncmVzc1J1bGUocnVsZTogQ2ZuU2VjdXJpdHlHcm91cC5FZ3Jlc3NQcm9wZXJ0eSk6IGJvb2xlYW4ge1xuICAgIHJldHVybiB0aGlzLmRpcmVjdEVncmVzc1J1bGVzLmZpbmRJbmRleChyID0+IGVncmVzc1J1bGVzRXF1YWwociwgcnVsZSkpID4gLTE7XG4gIH1cblxuICAvKipcbiAgICogQWRkIHRoZSBkZWZhdWx0IGVncmVzcyBydWxlIHRvIHRoZSBzZWN1cml0eUdyb3VwXG4gICAqXG4gICAqIFRoaXMgZGVwZW5kcyBvbiBhbGxvd0FsbE91dGJvdW5kOlxuICAgKlxuICAgKiAtIElmIGFsbG93QWxsT3V0Ym91bmQgaXMgdHJ1ZSwgd2UgKlRFQ0hOSUNBTExZKiBkb24ndCBuZWVkIHRvIGRvIGFueXRoaW5nLCBiZWNhdXNlXG4gICAqICAgRUMyIGlzIGdvaW5nIHRvIGNyZWF0ZSB0aGlzIGRlZmF1bHQgcnVsZSBhbnl3YXkuIEJ1dCwgZm9yIG1heGltdW0gcmVhZGFiaWxpdHlcbiAgICogICBvZiB0aGUgdGVtcGxhdGUsIHdlIHdpbGwgYWRkIG9uZSBhbnl3YXkuXG4gICAqIC0gSWYgYWxsb3dBbGxPdXRib3VuZCBpcyBmYWxzZSwgd2UgYWRkIGEgYm9ndXMgcnVsZSB0aGF0IG1hdGNoZXMgbm8gdHJhZmZpYyBpblxuICAgKiAgIG9yZGVyIHRvIGdldCByaWQgb2YgdGhlIGRlZmF1bHQgXCJhbGwgb3V0Ym91bmRcIiBydWxlIHRoYXQgRUMyIGNyZWF0ZXMgYnkgZGVmYXVsdC5cbiAgICogICBJZiBvdGhlciBydWxlcyBoYXBwZW4gdG8gZ2V0IGFkZGVkIGxhdGVyLCB3ZSByZW1vdmUgdGhlIGJvZ3VzIHJ1bGUgYWdhaW4gc29cbiAgICogICB0aGF0IGl0IGRvZXNuJ3QgY2x1dHRlciB1cCB0aGUgdGVtcGxhdGUgdG9vIG11Y2ggKGV2ZW4gdGhvdWdoIHRoYXQncyBub3RcbiAgICogICBzdHJpY3RseSBuZWNlc3NhcnkpLlxuICAgKi9cbiAgcHJpdmF0ZSBhZGREZWZhdWx0RWdyZXNzUnVsZSgpIHtcbiAgICBpZiAodGhpcy5kaXNhYmxlSW5saW5lUnVsZXMpIHtcbiAgICAgIGNvbnN0IHBlZXIgPSB0aGlzLmFsbG93QWxsT3V0Ym91bmQgPyBBTExfVFJBRkZJQ19QRUVSIDogTk9fVFJBRkZJQ19QRUVSO1xuICAgICAgY29uc3QgcG9ydCA9IHRoaXMuYWxsb3dBbGxPdXRib3VuZCA/IEFMTF9UUkFGRklDX1BPUlQgOiBOT19UUkFGRklDX1BPUlQ7XG4gICAgICBjb25zdCBkZXNjcmlwdGlvbiA9IHRoaXMuYWxsb3dBbGxPdXRib3VuZCA/IEFMTE9XX0FMTF9SVUxFLmRlc2NyaXB0aW9uIDogTUFUQ0hfTk9fVFJBRkZJQy5kZXNjcmlwdGlvbjtcbiAgICAgIHN1cGVyLmFkZEVncmVzc1J1bGUocGVlciwgcG9ydCwgZGVzY3JpcHRpb24sIGZhbHNlKTtcbiAgICB9IGVsc2Uge1xuICAgICAgY29uc3QgcnVsZSA9IHRoaXMuYWxsb3dBbGxPdXRib3VuZD8gQUxMT1dfQUxMX1JVTEUgOiBNQVRDSF9OT19UUkFGRklDO1xuICAgICAgdGhpcy5kaXJlY3RFZ3Jlc3NSdWxlcy5wdXNoKHJ1bGUpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBSZW1vdmUgdGhlIGJvZ3VzIHJ1bGUgaWYgaXQgZXhpc3RzXG4gICAqL1xuICBwcml2YXRlIHJlbW92ZU5vVHJhZmZpY1J1bGUoKSB7XG4gICAgaWYgKHRoaXMuZGlzYWJsZUlubGluZVJ1bGVzKSB7XG4gICAgICBjb25zdCBbc2NvcGUsIGlkXSA9IHRoaXMuZGV0ZXJtaW5lUnVsZVNjb3BlKFxuICAgICAgICBOT19UUkFGRklDX1BFRVIsXG4gICAgICAgIE5PX1RSQUZGSUNfUE9SVCxcbiAgICAgICAgJ3RvJyxcbiAgICAgICAgZmFsc2UsXG4gICAgICApO1xuICAgICAgc2NvcGUubm9kZS50cnlSZW1vdmVDaGlsZChpZCk7XG4gICAgfSBlbHNlIHtcbiAgICAgIGNvbnN0IGkgPSB0aGlzLmRpcmVjdEVncmVzc1J1bGVzLmZpbmRJbmRleChyID0+IGVncmVzc1J1bGVzRXF1YWwociwgTUFUQ0hfTk9fVFJBRkZJQykpO1xuICAgICAgaWYgKGkgPiAtMSkge1xuICAgICAgICB0aGlzLmRpcmVjdEVncmVzc1J1bGVzLnNwbGljZShpLCAxKTtcbiAgICAgIH1cbiAgICB9XG4gIH1cbn1cblxuLyoqXG4gKiBFZ3Jlc3MgcnVsZSB0aGF0IHB1cnBvc2VseSBtYXRjaGVzIG5vIHRyYWZmaWNcbiAqXG4gKiBUaGlzIGlzIHVzZWQgaW4gb3JkZXIgdG8gZGlzYWJsZSB0aGUgXCJhbGwgdHJhZmZpY1wiIGRlZmF1bHQgb2YgU2VjdXJpdHkgR3JvdXBzLlxuICpcbiAqIE5vIG1hY2hpbmUgY2FuIGV2ZXIgYWN0dWFsbHkgaGF2ZSB0aGUgMjU1LjI1NS4yNTUuMjU1IElQIGFkZHJlc3MsIGJ1dFxuICogaW4gb3JkZXIgdG8gbG9jayBpdCBkb3duIGV2ZW4gbW9yZSB3ZSdsbCByZXN0cmljdCB0byBhIG5vbmV4aXN0ZW50XG4gKiBJQ01QIHRyYWZmaWMgdHlwZS5cbiAqL1xuY29uc3QgTUFUQ0hfTk9fVFJBRkZJQyA9IHtcbiAgY2lkcklwOiAnMjU1LjI1NS4yNTUuMjU1LzMyJyxcbiAgZGVzY3JpcHRpb246ICdEaXNhbGxvdyBhbGwgdHJhZmZpYycsXG4gIGlwUHJvdG9jb2w6ICdpY21wJyxcbiAgZnJvbVBvcnQ6IDI1MixcbiAgdG9Qb3J0OiA4Nixcbn07XG5cbmNvbnN0IE5PX1RSQUZGSUNfUEVFUiA9IFBlZXIuaXB2NChNQVRDSF9OT19UUkFGRklDLmNpZHJJcCk7XG5jb25zdCBOT19UUkFGRklDX1BPUlQgPSBQb3J0LmljbXBUeXBlQW5kQ29kZShNQVRDSF9OT19UUkFGRklDLmZyb21Qb3J0LCBNQVRDSF9OT19UUkFGRklDLnRvUG9ydCk7XG5cbi8qKlxuICogRWdyZXNzIHJ1bGUgdGhhdCBtYXRjaGVzIGFsbCB0cmFmZmljXG4gKi9cbmNvbnN0IEFMTE9XX0FMTF9SVUxFID0ge1xuICBjaWRySXA6ICcwLjAuMC4wLzAnLFxuICBkZXNjcmlwdGlvbjogJ0FsbG93IGFsbCBvdXRib3VuZCB0cmFmZmljIGJ5IGRlZmF1bHQnLFxuICBpcFByb3RvY29sOiAnLTEnLFxufTtcblxuY29uc3QgQUxMX1RSQUZGSUNfUEVFUiA9IFBlZXIuYW55SXB2NCgpO1xuY29uc3QgQUxMX1RSQUZGSUNfUE9SVCA9IFBvcnQuYWxsVHJhZmZpYygpO1xuXG5leHBvcnQgaW50ZXJmYWNlIENvbm5lY3Rpb25SdWxlIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBwcm90b2NvbD86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZnJvbVBvcnQ6IG51bWJlcjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHRvUG9ydD86IG51bWJlcjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGRlc2NyaXB0aW9uPzogc3RyaW5nO1xufVxuXG4vKipcbiAqIENvbXBhcmUgdHdvIGluZ3Jlc3MgcnVsZXMgZm9yIGVxdWFsaXR5IHRoZSBzYW1lIHdheSBDbG91ZEZvcm1hdGlvbiB3b3VsZCAoZGlzY2FyZGluZyBkZXNjcmlwdGlvbilcbiAqL1xuZnVuY3Rpb24gaW5ncmVzc1J1bGVzRXF1YWwoYTogQ2ZuU2VjdXJpdHlHcm91cC5JbmdyZXNzUHJvcGVydHksIGI6IENmblNlY3VyaXR5R3JvdXAuSW5ncmVzc1Byb3BlcnR5KSB7XG4gIHJldHVybiBhLmNpZHJJcCA9PT0gYi5jaWRySXBcbiAgICAmJiBhLmNpZHJJcHY2ID09PSBiLmNpZHJJcHY2XG4gICAgJiYgYS5mcm9tUG9ydCA9PT0gYi5mcm9tUG9ydFxuICAgICYmIGEudG9Qb3J0ID09PSBiLnRvUG9ydFxuICAgICYmIGEuaXBQcm90b2NvbCA9PT0gYi5pcFByb3RvY29sXG4gICAgJiYgYS5zb3VyY2VTZWN1cml0eUdyb3VwSWQgPT09IGIuc291cmNlU2VjdXJpdHlHcm91cElkXG4gICAgJiYgYS5zb3VyY2VTZWN1cml0eUdyb3VwTmFtZSA9PT0gYi5zb3VyY2VTZWN1cml0eUdyb3VwTmFtZVxuICAgICYmIGEuc291cmNlU2VjdXJpdHlHcm91cE93bmVySWQgPT09IGIuc291cmNlU2VjdXJpdHlHcm91cE93bmVySWQ7XG59XG5cbi8qKlxuICogQ29tcGFyZSB0d28gZWdyZXNzIHJ1bGVzIGZvciBlcXVhbGl0eSB0aGUgc2FtZSB3YXkgQ2xvdWRGb3JtYXRpb24gd291bGQgKGRpc2NhcmRpbmcgZGVzY3JpcHRpb24pXG4gKi9cbmZ1bmN0aW9uIGVncmVzc1J1bGVzRXF1YWwoYTogQ2ZuU2VjdXJpdHlHcm91cC5FZ3Jlc3NQcm9wZXJ0eSwgYjogQ2ZuU2VjdXJpdHlHcm91cC5FZ3Jlc3NQcm9wZXJ0eSkge1xuICByZXR1cm4gYS5jaWRySXAgPT09IGIuY2lkcklwXG4gICAgJiYgYS5jaWRySXB2NiA9PT0gYi5jaWRySXB2NlxuICAgICYmIGEuZnJvbVBvcnQgPT09IGIuZnJvbVBvcnRcbiAgICAmJiBhLnRvUG9ydCA9PT0gYi50b1BvcnRcbiAgICAmJiBhLmlwUHJvdG9jb2wgPT09IGIuaXBQcm90b2NvbFxuICAgICYmIGEuZGVzdGluYXRpb25QcmVmaXhMaXN0SWQgPT09IGIuZGVzdGluYXRpb25QcmVmaXhMaXN0SWRcbiAgICAmJiBhLmRlc3RpbmF0aW9uU2VjdXJpdHlHcm91cElkID09PSBiLmRlc3RpbmF0aW9uU2VjdXJpdHlHcm91cElkO1xufVxuXG4vKipcbiAqIFdoZXRoZXIgdGhpcyBydWxlIHJlZmVycyB0byBhbGwgdHJhZmZpY1xuICovXG5mdW5jdGlvbiBpc0FsbFRyYWZmaWNSdWxlKHJ1bGU6IGFueSkge1xuICByZXR1cm4gcnVsZS5jaWRySXAgPT09ICcwLjAuMC4wLzAnICYmIHJ1bGUuaXBQcm90b2NvbCA9PT0gJy0xJztcbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBsb29raW5nIHVwIGFuIGV4aXN0aW5nIFNlY3VyaXR5R3JvdXAuXG4gKlxuICogRWl0aGVyIGBzZWN1cml0eUdyb3VwTmFtZWAgb3IgYHNlY3VyaXR5R3JvdXBJZGAgaGFzIHRvIGJlIHNwZWNpZmllZC5cbiAqL1xuaW50ZXJmYWNlIFNlY3VyaXR5R3JvdXBMb29rdXBPcHRpb25zIHtcbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBzZWN1cml0eSBncm91cFxuICAgKlxuICAgKiBJZiBnaXZlbiwgd2lsbCBpbXBvcnQgdGhlIFNlY3VyaXR5R3JvdXAgd2l0aCB0aGlzIG5hbWUuXG4gICAqXG4gICAqIEBkZWZhdWx0IERvbid0IGZpbHRlciBvbiBzZWN1cml0eUdyb3VwTmFtZVxuICAgKi9cbiAgcmVhZG9ubHkgc2VjdXJpdHlHcm91cE5hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBJRCBvZiB0aGUgc2VjdXJpdHkgZ3JvdXBcbiAgICpcbiAgICogSWYgZ2l2ZW4sIHdpbGwgaW1wb3J0IHRoZSBTZWN1cml0eUdyb3VwIHdpdGggdGhpcyBJRC5cbiAgICpcbiAgICogQGRlZmF1bHQgRG9uJ3QgZmlsdGVyIG9uIHNlY3VyaXR5R3JvdXBJZFxuICAgKi9cbiAgcmVhZG9ubHkgc2VjdXJpdHlHcm91cElkPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgVlBDIG9mIHRoZSBzZWN1cml0eSBncm91cFxuICAgKlxuICAgKiBJZiBnaXZlbiwgd2lsbCBmaWx0ZXIgdGhlIFNlY3VyaXR5R3JvdXAgYmFzZWQgb24gdGhlIFZQQy5cbiAgICpcbiAgICogQGRlZmF1bHQgRG9uJ3QgZmlsdGVyIG9uIFZQQ1xuICAgKi9cbiAgcmVhZG9ubHkgdnBjPzogSVZwYyxcbn1cbiJdfQ==