from typing import TYPE_CHECKING, Any, NamedTuple, Optional, Union
from urllib.parse import SplitResult, urlencode, urlsplit, urlunsplit

from starlite.datastructures import Headers
from starlite.datastructures.multi_dicts import QueryMultiDict

if TYPE_CHECKING:
    from starlite.types import Scope


_DEFAULT_SCHEME_PORTS = {"http": 80, "https": 443, "ftp": 21, "ws": 80, "wss": 443}


class Address(NamedTuple):
    """Just a network address."""

    host: str
    """Address host"""
    port: int
    """Address port"""


def make_absolute_url(path: Union[str, "URL"], base: Union[str, "URL"]) -> str:
    """Create an absolute URL.

    Args:
        path: URL path to make absolute
        base: URL to use as a base

    Returns:
        A string representing the new, absolute URL
    """
    if isinstance(base, str):
        base = URL(base)
    netloc = base.netloc
    path = base.path.rstrip("/") + str(path)
    return str(URL.from_components(scheme=base.scheme, netloc=netloc, path=path))


class URL:
    """Representation and modification utilities of a URL."""

    __slots__ = (
        "_query_params",
        "_url",
        "fragment",
        "hostname",
        "netloc",
        "password",
        "path",
        "port",
        "query",
        "scheme",
        "username",
    )

    scheme: str
    """URL scheme"""
    netloc: str
    """Network location"""
    path: str
    """Hierarchical path"""
    fragment: str
    """Fragment component"""
    query: str
    """Query string"""
    username: Optional[str]
    """Username if specified"""
    password: Optional[str]
    """Password if specified"""
    port: Optional[int]
    """Port if specified"""
    hostname: Optional[str]
    """Hostname if specified"""

    def __init__(self, url: Union[str, SplitResult]) -> None:
        """Initialize `URL` from a string or a.

        [SplitResult][urllib.parse.SplitResult].

        Args:
            url: URL, either as a string or a [SplitResult][urllib.parse.SplitResult] as
                returned by [urlsplit][urllib.parse.urlsplit]
        """
        if isinstance(url, str):
            result = urlsplit(url)
            self._url = url
        else:
            result = url
            self._url = urlunsplit(url)

        self.scheme = result.scheme
        self.netloc = result.netloc
        self.path = result.path
        self.fragment = result.fragment
        self.query = result.query
        self.username = result.username
        self.password = result.password
        self.port = result.port
        self.hostname = result.hostname
        self._query_params: Optional[QueryMultiDict] = None

    @classmethod
    def from_components(
        cls,
        scheme: str = "",
        netloc: str = "",
        path: str = "",
        fragment: str = "",
        query: str = "",
    ) -> "URL":
        """Create a new URL from components.

        Args:
            scheme: URL scheme
            netloc: Network location
            path: Hierarchical path
            query: Query component
            fragment: Fragment identifier

        Returns:
            A new URL with the given components
        """
        return cls(
            SplitResult(
                scheme=scheme,
                netloc=netloc,
                path=path,
                fragment=fragment,
                query=query,
            )
        )

    @classmethod
    def from_scope(cls, scope: "Scope") -> "URL":
        """Construct a URL from a [Scope][starlite.types.Scope]

        Args:
            scope: A scope

        Returns:
            A URL
        """
        scheme = scope.get("scheme", "http")
        server = scope.get("server")
        path = scope.get("root_path", "") + scope["path"]
        query_string = scope.get("query_string", b"")

        host = Headers.from_scope(scope).get("host", "")
        if server and not host:
            host, port = server
            default_port = _DEFAULT_SCHEME_PORTS[scheme]
            if port != default_port:
                host = f"{host}:{port}"

        return cls.from_components(
            scheme=scheme if server else "",
            query=query_string.decode(),
            netloc=host,
            path=path,
        )

    def with_replacements(
        self,
        scheme: str = "",
        netloc: str = "",
        path: str = "",
        query: Optional[Union[str, QueryMultiDict]] = None,
        fragment: str = "",
    ) -> "URL":
        """Create a new URL, replacing the given components.

        Args:
            scheme: URL scheme
            netloc: Network location
            path: Hierarchical path
            query: Raw query string
            fragment: Fragment identifier

        Returns:
            A new URL with the given components replaced
        """
        if isinstance(query, QueryMultiDict):
            query = urlencode(query=query)

        return URL.from_components(
            scheme=scheme or self.scheme,
            netloc=netloc or self.netloc,
            path=path or self.path,
            query=query or self.query,
            fragment=fragment or self.fragment,
        )

    @property
    def query_params(self) -> QueryMultiDict:
        """Query parameters of a URL as a [MultiDict][multidict.MultiDict]

        Returns:
            A [MultiDict][multidict.MultiDict] with query parameters

        Notes:
            - While the returned `MultiDict` is mutable, [URL][starlite.datastructures.URL]
                itself is *immutable*, therefore mutating the query parameters will not
                directly mutate the `URL`. If you want to modify query parameters, make
                modifications in the multidict and pass them back to
                [with_replacements][starlite.datastructures.URL.with_replacements]
        """
        if self._query_params is None:
            self._query_params = QueryMultiDict.from_query_string(self.query)
        return self._query_params

    def __str__(self) -> str:
        return self._url

    def __eq__(self, other: Any) -> bool:
        if isinstance(other, (str, URL)):
            return str(self) == str(other)
        return NotImplemented

    def __repr__(self) -> str:
        return f"{type(self).__name__}({self._url!r})"
