# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pandas_alive']

package_data = \
{'': ['*']}

install_requires = \
['attrs>=19.3.0,<20.0.0',
 'matplotlib>=3.2.1,<4.0.0',
 'pandas>=1.0.3,<2.0.0',
 'pillow>=7.1.2,<8.0.0']

setup_kwargs = {
    'name': 'pandas-alive',
    'version': '0.2.2.dev2',
    'description': 'Animated plotting extension for Pandas with Matplotlib',
    'long_description': '# Pandas_Alive\n\nAnimated plotting extension for Pandas with Matplotlib\n\n[![Inline docs](http://inch-ci.org/github/dwyl/hapi-auth-jwt2.svg?branch=master)](https://jackmckew.github.io/pandas_alive/) ![Interrogate Coverage](badges/interrogate_badge.svg) [![Downloads](https://pepy.tech/badge/pandas-alive/month)](https://pepy.tech/project/pandas-alive/month) [![PyPI version shields.io](https://img.shields.io/pypi/v/pandas_alive.svg)](https://pypi.python.org/pypi/pandas_alive/) [![PyPI license](https://img.shields.io/pypi/l/pandas_alive.svg)](https://pypi.python.org/pypi/pandas_alive/) [![saythanks](https://img.shields.io/badge/say-thanks-ff69b4.svg)](https://www.buymeacoffee.com/jackmckew)\n\n**Pandas_Alive** is intended to provide a plotting backend for animated [matplotlib](https://matplotlib.org/) charts for [Pandas](https://pandas.pydata.org/) DataFrames, similar to the already [existing Visualization feature of Pandas](https://pandas.pydata.org/pandas-docs/stable/visualization.html).\n\nWith **Pandas_Alive**, creating stunning, animated visualisations is as easy as calling:\n\n`df.plot_animated()`\n\n![Example Bar Chart](examples/example-barh-chart.gif)\n\n## Table of Contents\n<!-- START doctoc generated TOC please keep comment here to allow auto update -->\n<!-- DON\'T EDIT THIS SECTION, INSTEAD RE-RUN doctoc TO UPDATE -->\n\n\n- [Installation](#installation)\n- [Usage](#usage)\n  - [Currently Supported Chart Types](#currently-supported-chart-types)\n    - [Horizontal Bar Chart Races](#horizontal-bar-chart-races)\n    - [Vertical Bar Chart Races](#vertical-bar-chart-races)\n    - [Line Charts](#line-charts)\n    - [Bar Charts](#bar-charts)\n    - [Scatter Charts](#scatter-charts)\n    - [Pie Charts](#pie-charts)\n    - [Bubble Charts](#bubble-charts)\n    - [GeoSpatial Charts](#geospatial-charts)\n      - [GeoSpatial Point Charts](#geospatial-point-charts)\n      - [Polygon GeoSpatial Charts](#polygon-geospatial-charts)\n  - [Multiple Charts](#multiple-charts)\n    - [Urban Population](#urban-population)\n    - [Life Expectancy in G7 Countries](#life-expectancy-in-g7-countries)\n    - [NSW COVID Visualisation](#nsw-covid-visualisation)\n    - [Italy COVID Visualisation](#italy-covid-visualisation)\n  - [HTML 5 Videos](#html-5-videos)\n  - [Progress Bars!](#progress-bars)\n- [Future Features](#future-features)\n- [Inspiration](#inspiration)\n- [Requirements](#requirements)\n- [Documentation](#documentation)\n- [Contributing](#contributing)\n  - [Development](#development)\n- [Changelog](#changelog)\n\n<!-- END doctoc generated TOC please keep comment here to allow auto update -->\n\n## Installation\n\nInstall with `pip install pandas_alive`\n\n## Usage\n\nAs this package was inspired by [`bar_chart_race`](https://github.com/dexplo/bar_chart_race), the example data set is sourced from there.\n\nMust begin with a pandas DataFrame containing \'wide\' data where:\n\n- Every row represents a single period of time\n- Each column holds the value for a particular category\n- The index contains the time component (optional)\n\nThe data below is an example of properly formatted data. It shows total deaths from COVID-19 for the highest 20 countries by date.\n\n![Example Data Table](https://raw.githubusercontent.com/dexplo/bar_chart_race/master/docs/images/wide_data.png)\n\nTo produce the above visualisation:\n\n- Check [Requirements](#requirements) first to ensure you have the tooling installed!\n- Call `plot_animated()` on the DataFrame\n    - Either specify a file name to write to with `df.plot_animated(filename=\'example.mp4\')` or use `df.plot_animated().get_html5_video` to return a HTML5 video\n- Done!\n\n``` python\nimport pandas_alive\n\ncovid_df = pandas_alive.load_dataset()\n\ncovid_df.plot_animated(filename=\'examples/example-barh-chart.gif\')\n```\n\n### Currently Supported Chart Types\n\n#### Horizontal Bar Chart Races\n\n``` python\nimport pandas as pd\nimport pandas_alive\n\nelec_df = pd.read_csv("data/Aus_Elec_Gen_1980_2018.csv",index_col=0,parse_dates=[0],thousands=\',\')\n\nelec_df.fillna(0).plot_animated(\'examples/example-electricity-generated-australia.gif\',period_fmt="%Y",title=\'Australian Electricity Generation Sources 1980-2018\')\n```\n\n![Electricity Example Line Chart](examples/example-electricity-generated-australia.gif)\n\n``` python\nimport pandas_alive\n\ncovid_df = pandas_alive.load_dataset()\n\ndef current_total(values):\n    total = values.sum()\n    s = f\'Total : {int(total)}\'\n    return {\'x\': .85, \'y\': .2, \'s\': s, \'ha\': \'right\', \'size\': 11}\n\ncovid_df.plot_animated(filename=\'examples/summary-func-example.gif\',period_summary_func=current_total)\n```\n\n![Summary Func Example](examples/summary-func-example.gif)\n\n``` python\nimport pandas as pd\nimport pandas_alive\n\nelec_df = pd.read_csv("data/Aus_Elec_Gen_1980_2018.csv",index_col=0,parse_dates=[0],thousands=\',\')\n\nelec_df.fillna(0).plot_animated(\'examples/fixed-example.gif\',period_fmt="%Y",title=\'Australian Electricity Generation Sources 1980-2018\',fixed_max=True,fixed_order=True)\n```\n\n![Fixed Example](examples/fixed-example.gif)\n\n``` python\nimport pandas_alive\n\ncovid_df = pandas_alive.load_dataset()\n\ncovid_df.plot_animated(filename=\'examples/perpendicular-example.gif\',perpendicular_bar_func=\'mean\')\n```\n\n![Perpendicular Example](examples/perpendicular-example.gif)\n\n#### Vertical Bar Chart Races\n\n``` python\nimport pandas_alive\n\ncovid_df = pandas_alive.load_dataset()\n\ncovid_df.plot_animated(filename=\'examples/example-barv-chart.gif\',orientation=\'v\')\n```\n\n![Example Barv Chart](examples/example-barv-chart.gif)\n\n#### Line Charts\n\nWith as many lines as data columns in the DataFrame.\n\n``` python\nimport pandas_alive\n\ncovid_df = pandas_alive.load_dataset()\n\ncovid_df.diff().fillna(0).plot_animated(filename=\'examples/example-line-chart.gif\',kind=\'line\',period_label={\'x\':0.1,\'y\':0.9})\n```\n\n![Example Line Chart](examples/example-line-chart.gif)\n\n\n#### Bar Charts\n\nSimilar to line charts with time as the x-axis\n\n``` python\nimport pandas_alive\n\ncovid_df = pandas_alive.load_dataset()\n\ncovid_df.sum(axis=1).fillna(0).plot_animated(filename=\'examples/example-bar-chart.gif\',kind=\'bar\',period_label={\'x\':0.1,\'y\':0.9})\n```\n\n![Example Bar Chart](examples/example-bar-chart.gif)\n\n#### Scatter Charts\n\n``` python\nimport pandas as pd\nimport pandas_alive\n\nmax_temp_df = pd.read_csv(\n    "data/Newcastle_Australia_Max_Temps.csv",\n    parse_dates={"Timestamp": ["Year", "Month", "Day"]},\n)\nmin_temp_df = pd.read_csv(\n    "data/Newcastle_Australia_Min_Temps.csv",\n    parse_dates={"Timestamp": ["Year", "Month", "Day"]},\n)\n\nmerged_temp_df = pd.merge_asof(max_temp_df, min_temp_df, on="Timestamp")\n\nmerged_temp_df.index = pd.to_datetime(merged_temp_df["Timestamp"].dt.strftime(\'%Y/%m/%d\'))\n\nkeep_columns = ["Minimum temperature (Degree C)", "Maximum temperature (Degree C)"]\n\nmerged_temp_df[keep_columns].resample("Y").mean().plot_animated(filename=\'examples/example-scatter-chart.gif\',kind="scatter",title=\'Max & Min Temperature Newcastle, Australia\')\n\n```\n\n![Example Scatter Chart](examples/example-scatter-chart.gif)\n\n#### Pie Charts\n\n``` python\nimport pandas_alive\n\ncovid_df = pandas_alive.load_dataset()\n\ncovid_df.plot_animated(filename=\'examples/example-pie-chart.gif\',kind="pie",rotatelabels=True,period_label={\'x\':0,\'y\':0})\n```\n\n![Example Pie Chart](examples/example-pie-chart.gif)\n\n#### Bubble Charts\n\nBubble charts are generated from a multi-indexed dataframes. Where the index is the time period (optional) and the axes are defined with `x_data_label` & `y_data_label` which should be passed a string in the level 0 column labels.\n\nSee an example multi-indexed dataframe at: <https://github.com/JackMcKew/pandas_alive/tree/master/data/multi.csv>\n\n``` python\nimport pandas_alive\n\nmulti_index_df = pd.read_csv("data/multi.csv", header=[0, 1], index_col=0)\n\nmulti_index_df.index = pd.to_datetime(multi_index_df.index,dayfirst=True)\n\nmap_chart = multi_index_df.plot_animated(\n    kind="bubble",\n    filename="examples/example-bubble-chart.gif",\n    x_data_label="Longitude",\n    y_data_label="Latitude",\n    size_data_label="Cases",\n)\n```\n\n![Bubble Chart Example](examples/example-bubble-chart.gif)\n\n#### GeoSpatial Charts\n\nGeoSpatial charts can now be animated easily using [`geopandas`](https://geopandas.org/index.html)!\n\n> If using Windows, [anaconda](https://www.anaconda.com/) is the easiest way to install with all GDAL dependancies.\n\nMust begin with a `geopandas` GeoDataFrame containing \'wide\' data where:\n\n- Every row represents a single geometry (Point or Polygon).\n    - The index contains the geometry label (optional)\n- Each column represents a single period in time.\n\n> These can be easily composed by transposing data compatible with the rest of the charts using `df = df.T`.\n\n##### GeoSpatial Point Charts\n\n``` python\nimport geopandas\nimport pandas_alive\nimport contextily\n\ngdf = geopandas.read_file(\'data/nsw-covid19-cases-by-postcode.gpkg\')\ngdf.index = gdf.postcode\ngdf = gdf.drop(\'postcode\',axis=1)\n\nmap_chart = gdf.plot_animated(filename=\'examples/example-geo-point-chart.gif\',basemap_format={\'source\':contextily.providers.Stamen.Terrain})\n```\n\n![Example Point GeoSpatialChart](examples/example-geo-point-chart.gif)\n\n##### Polygon GeoSpatial Charts\n\nSupports GeoDataFrames containing Polygons!\n\n``` python\nimport geopandas\nimport pandas_alive\nimport contextily\n\ngdf = geopandas.read_file(\'data/italy-covid-region.gpkg\')\ngdf.index = gdf.region\ngdf = gdf.drop(\'region\',axis=1)\n\nmap_chart = gdf.plot_animated(filename=\'examples/example-geo-polygon-chart.gif\',basemap_format={\'source\':contextily.providers.Stamen.Terrain})\n```\n\n![Example Polygon GeoSpatialChart](examples/example-geo-polygon-chart.gif)\n\n### Multiple Charts\n\n`pandas_alive` supports multiple animated charts in a single visualisation.\n\n- Create a list of all charts to include in animation\n- Use `animate_multiple_plots` with a `filename` and the list of charts (this will use `matplotlib.subplots`)\n- Done!\n\n``` python\nimport pandas_alive\n\ncovid_df = pandas_alive.load_dataset()\n\nanimated_line_chart = covid_df.diff().fillna(0).plot_animated(kind=\'line\',period_label=False)\n\nanimated_bar_chart = covid_df.plot_animated(n_visible=10)\n\npandas_alive.animate_multiple_plots(\'examples/example-bar-and-line-chart.gif\',[animated_bar_chart,animated_line_chart])\n```\n\n![Example Bar & Line Chart](examples/example-bar-and-line-chart.gif)\n\n#### Urban Population\n\n``` python\nimport pandas_alive\n\nurban_df = pandas_alive.load_dataset("urban_pop")\n\nanimated_line_chart = (\n    urban_df.sum(axis=1)\n    .pct_change()\n    .fillna(method=\'bfill\')\n    .mul(100)\n    .plot_animated(kind="line", title="Total % Change in Population",period_label=False)\n)\n\nanimated_bar_chart = urban_df.plot_animated(n_visible=10,title=\'Top 10 Populous Countries\',period_fmt="%Y")\n\npandas_alive.animate_multiple_plots(\'examples/example-bar-and-line-urban-chart.gif\',[animated_bar_chart,animated_line_chart],title=\'Urban Population 1977 - 2018\',adjust_subplot_top=0.85)\n```\n\n![Urban Population Bar & Line Chart](examples/example-bar-and-line-urban-chart.gif)\n\n\n#### Life Expectancy in G7 Countries\n\n``` python\nimport pandas_alive\nimport pandas as pd\n\ndata_raw = pd.read_csv(\n    "https://raw.githubusercontent.com/owid/owid-datasets/master/datasets/Long%20run%20life%20expectancy%20-%20Gapminder%2C%20UN/Long%20run%20life%20expectancy%20-%20Gapminder%2C%20UN.csv"\n)\n\nlist_G7 = [\n    "Canada",\n    "France",\n    "Germany",\n    "Italy",\n    "Japan",\n    "United Kingdom",\n    "United States",\n]\n\ndata_raw = data_raw.pivot(\n    index="Year", columns="Entity", values="Life expectancy (Gapminder, UN)"\n)\n\ndata = pd.DataFrame()\ndata["Year"] = data_raw.reset_index()["Year"]\nfor country in list_G7:\n    data[country] = data_raw[country].values\n\ndata = data.fillna(method="pad")\ndata = data.fillna(0)\ndata = data.set_index("Year").loc[1900:].reset_index()\n\ndata["Year"] = pd.to_datetime(data.reset_index()["Year"].astype(str))\n\ndata = data.set_index("Year")\n\nanimated_bar_chart = data.plot_animated(\n    period_fmt="%Y",perpendicular_bar_func="mean", period_length=200,fixed_max=True\n)\n\nanimated_line_chart = data.plot_animated(\n    kind="line", period_fmt="%Y", period_length=200,fixed_max=True\n)\n\npandas_alive.animate_multiple_plots(\n    "examples/life-expectancy.gif",\n    plots=[animated_bar_chart, animated_line_chart],\n    title="Life expectancy in G7 countries up to 2015",\n    adjust_subplot_left=0.2,\n)\n\n```\n\n![Life Expectancy Chart](examples/life-expectancy.gif)\n\n#### NSW COVID Visualisation\n\n``` python\nimport geopandas\nimport pandas as pd\nimport pandas_alive\nimport contextily\nimport matplotlib.pyplot as plt\n\nimport urllib.request, json\n\nwith urllib.request.urlopen(\n    "https://data.nsw.gov.au/data/api/3/action/package_show?id=aefcde60-3b0c-4bc0-9af1-6fe652944ec2"\n) as url:\n    data = json.loads(url.read().decode())\n\n# Extract url to csv component\ncovid_nsw_data_url = data["result"]["resources"][0]["url"]\n\n# Read csv from data API url\nnsw_covid = pd.read_csv(covid_nsw_data_url)\npostcode_dataset = pd.read_csv("data/postcode-data.csv")\n\n# Prepare data from NSW health dataset\n\n\nnsw_covid = nsw_covid.fillna(9999)\nnsw_covid["postcode"] = nsw_covid["postcode"].astype(int)\n\ngrouped_df = nsw_covid.groupby(["notification_date", "postcode"]).size()\ngrouped_df = pd.DataFrame(grouped_df).unstack()\ngrouped_df.columns = grouped_df.columns.droplevel().astype(str)\n\ngrouped_df = grouped_df.fillna(0)\ngrouped_df.index = pd.to_datetime(grouped_df.index)\n\ncases_df = grouped_df\n\n# Clean data in postcode dataset prior to matching\n\ngrouped_df = grouped_df.T\npostcode_dataset = postcode_dataset[postcode_dataset[\'Longitude\'].notna()]\npostcode_dataset = postcode_dataset[postcode_dataset[\'Longitude\'] != 0]\npostcode_dataset = postcode_dataset[postcode_dataset[\'Latitude\'].notna()]\npostcode_dataset = postcode_dataset[postcode_dataset[\'Latitude\'] != 0]\npostcode_dataset[\'Postcode\'] = postcode_dataset[\'Postcode\'].astype(str)\n\n# Build GeoDataFrame from Lat Long dataset and make map chart\ngrouped_df[\'Longitude\'] = grouped_df.index.map(postcode_dataset.set_index(\'Postcode\')[\'Longitude\'].to_dict())\ngrouped_df[\'Latitude\'] = grouped_df.index.map(postcode_dataset.set_index(\'Postcode\')[\'Latitude\'].to_dict())\ngdf = geopandas.GeoDataFrame(\n    grouped_df, geometry=geopandas.points_from_xy(grouped_df.Longitude, grouped_df.Latitude),crs="EPSG:4326")\ngdf = gdf.dropna()\n\n# Prepare GeoDataFrame for writing to geopackage\ngdf = gdf.drop([\'Longitude\',\'Latitude\'],axis=1)\ngdf.columns = gdf.columns.astype(str)\ngdf[\'postcode\'] = gdf.index\ngdf.to_file("data/nsw-covid19-cases-by-postcode.gpkg", layer=\'nsw-postcode-covid\', driver="GPKG")\n\n# Prepare GeoDataFrame for plotting\ngdf.index = gdf.postcode\ngdf = gdf.drop(\'postcode\',axis=1)\ngdf = gdf.to_crs("EPSG:3857") #Web Mercator\n\nmap_chart = gdf.plot_animated(basemap_format={\'source\':contextily.providers.Stamen.Terrain},cmap=\'cool\')\n\ncases_df.to_csv(\'data/nsw-covid-cases-by-postcode.csv\')\n\nfrom datetime import datetime\n\nbar_chart = cases_df.sum(axis=1).plot_animated(\n    kind=\'line\',\n    label_events={\n        \'Ruby Princess Disembark\':datetime.strptime("19/03/2020", "%d/%m/%Y"),\n        \'Lockdown\':datetime.strptime("31/03/2020", "%d/%m/%Y")\n    },\n    fill_under_line_color="blue",\n    enable_progress_bar=True\n)\n\nmap_chart.ax.set_title(\'Cases by Location\')\n\ngrouped_df = pd.read_csv(\'data/nsw-covid-cases-by-postcode.csv\', index_col=0, parse_dates=[0])\n\nline_chart = (\n    grouped_df.sum(axis=1)\n    .cumsum()\n    .fillna(0)\n    .plot_animated(kind="line", period_label=False, title="Cumulative Total Cases")\n)\n\n\ndef current_total(values):\n    total = values.sum()\n    s = f\'Total : {int(total)}\'\n    return {\'x\': .85, \'y\': .2, \'s\': s, \'ha\': \'right\', \'size\': 11}\n\nrace_chart = grouped_df.cumsum().plot_animated(\n    n_visible=5, title="Cases by Postcode", period_label=False,period_summary_func=current_total\n)\n\nimport time\n\ntimestr = time.strftime("%d/%m/%Y")\n\nplots = [bar_chart, line_chart, map_chart, race_chart]\n\nfrom matplotlib import rcParams\n\nrcParams.update({"figure.autolayout": False})\n\nfigs = plt.figure()\ngs = figs.add_gridspec(2, 3, hspace=0.5)\nf3_ax1 = figs.add_subplot(gs[0, :])\nf3_ax1.set_title(bar_chart.title)\nbar_chart.ax = f3_ax1\n\nf3_ax2 = figs.add_subplot(gs[1, 0])\nf3_ax2.set_title(line_chart.title)\nline_chart.ax = f3_ax2\n\nf3_ax3 = figs.add_subplot(gs[1, 1])\nf3_ax3.set_title(map_chart.title)\nmap_chart.ax = f3_ax3\n\nf3_ax4 = figs.add_subplot(gs[1, 2])\nf3_ax4.set_title(race_chart.title)\nrace_chart.ax = f3_ax4\n\ntimestr = cases_df.index.max().strftime("%d/%m/%Y")\nfigs.suptitle(f"NSW COVID-19 Confirmed Cases up to {timestr}")\n\npandas_alive.animate_multiple_plots(\n    \'examples/nsw-covid.gif\',\n    plots,\n    figs\n)\n```\n\n![NSW COVID](examples/nsw-covid.gif)\n\n#### Italy COVID Visualisation\n\n``` python\nimport geopandas\nimport pandas as pd\nimport pandas_alive\nimport contextily\nimport matplotlib.pyplot as plt\n\n\nregion_gdf = geopandas.read_file(\'data\\geo-data\\italy-with-regions\')\nregion_gdf.NOME_REG = region_gdf.NOME_REG.str.lower().str.title()\nregion_gdf = region_gdf.replace(\'Trentino-Alto Adige/Sudtirol\',\'Trentino-Alto Adige\')\nregion_gdf = region_gdf.replace("Valle D\'Aosta/VallÃ©e D\'Aoste\\r\\nValle D\'Aosta/VallÃ©e D\'Aoste","Valle d\'Aosta")\n\nitaly_df = pd.read_csv(\'data\\Regional Data - Sheet1.csv\',index_col=0,header=1,parse_dates=[0])\n\nitaly_df = italy_df[italy_df[\'Region\'] != \'NA\']\n\ncases_df = italy_df.iloc[:,:3]\ncases_df[\'Date\'] = cases_df.index\npivoted = cases_df.pivot(values=\'New positives\',index=\'Date\',columns=\'Region\')\npivoted.columns = pivoted.columns.astype(str)\npivoted = pivoted.rename(columns={\'nan\':\'Unknown Region\'})\n\ncases_gdf = pivoted.T\ncases_gdf[\'geometry\'] = cases_gdf.index.map(region_gdf.set_index(\'NOME_REG\')[\'geometry\'].to_dict())\n\ncases_gdf = cases_gdf[cases_gdf[\'geometry\'].notna()]\n\ncases_gdf = geopandas.GeoDataFrame(cases_gdf, crs=region_gdf.crs, geometry=cases_gdf.geometry)\n\ngdf = cases_gdf\n\nmap_chart = gdf.plot_animated(basemap_format={\'source\':contextily.providers.Stamen.Terrain},cmap=\'viridis\')\n\ncases_df = pivoted\n\nfrom datetime import datetime\n\nbar_chart = cases_df.sum(axis=1).plot_animated(\n    kind=\'line\',\n    label_events={\n        \'Schools Close\':datetime.strptime("4/03/2020", "%d/%m/%Y"),\n        \'Phase I Lockdown\':datetime.strptime("11/03/2020", "%d/%m/%Y"),\n        \'1M Global Cases\':datetime.strptime("02/04/2020", "%d/%m/%Y"),\n        \'100k Global Deaths\':datetime.strptime("10/04/2020", "%d/%m/%Y"),\n        \'Manufacturing Reopens\':datetime.strptime("26/04/2020", "%d/%m/%Y"),\n        \'Phase II Lockdown\':datetime.strptime("4/05/2020", "%d/%m/%Y"),\n\n    },\n    fill_under_line_color="blue",\n    enable_progress_bar=True\n)\n\nmap_chart.ax.set_title(\'Cases by Location\')\n\nline_chart = (\n    cases_df.sum(axis=1)\n    .cumsum()\n    .fillna(0)\n    .plot_animated(kind="line", period_label=False, title="Cumulative Total Cases")\n)\n\n\ndef current_total(values):\n    total = values.sum()\n    s = f\'Total : {int(total)}\'\n    return {\'x\': .85, \'y\': .2, \'s\': s, \'ha\': \'right\', \'size\': 11}\n\nrace_chart = cases_df.cumsum().plot_animated(\n    n_visible=5, title="Cases by Region", period_label=False,period_summary_func=current_total\n)\n\nimport time\n\ntimestr = time.strftime("%d/%m/%Y")\n\nplots = [bar_chart, race_chart, map_chart, line_chart]\n\n# Otherwise titles overlap and adjust_subplot does nothing\nfrom matplotlib import rcParams\nfrom matplotlib.animation import FuncAnimation\n\nrcParams.update({"figure.autolayout": False})\n\nfigs = plt.figure()\ngs = figs.add_gridspec(2, 3, hspace=0.5)\nf3_ax1 = figs.add_subplot(gs[0, :])\nf3_ax1.set_title(bar_chart.title)\nbar_chart.ax = f3_ax1\n\nf3_ax2 = figs.add_subplot(gs[1, 0])\nf3_ax2.set_title(race_chart.title)\nrace_chart.ax = f3_ax2\n\nf3_ax3 = figs.add_subplot(gs[1, 1])\nf3_ax3.set_title(map_chart.title)\nmap_chart.ax = f3_ax3\n\nf3_ax4 = figs.add_subplot(gs[1, 2])\nf3_ax4.set_title(line_chart.title)\nline_chart.ax = f3_ax4\n\naxes = [f3_ax1, f3_ax2, f3_ax3, f3_ax4]\ntimestr = cases_df.index.max().strftime("%d/%m/%Y")\nfigs.suptitle(f"Italy COVID-19 Confirmed Cases up to {timestr}")\n\npandas_alive.animate_multiple_plots(\n    \'examples/italy-covid.gif\',\n    plots,\n    figs\n)\n```\n\n![Italy COVID](examples/italy-covid.gif)\n\n### HTML 5 Videos\n\n`Pandas_Alive` supports rendering HTML5 videos through the use of `df.plot_animated().get_html5_video()`. `.get_html5_video` saves the animation as an h264 video, encoded in base64 directly into the HTML5 video tag. This respects the rc parameters for the writer as well as the bitrate. This also makes use of the interval to control the speed, and uses the repeat parameter to decide whether to loop.\n\nThis is typically used in Jupyter notebooks.\n\n``` python\nimport pandas_alive\nfrom IPython.display import HTML\n\ncovid_df = pandas_alive.load_dataset()\n\nanimated_html = covid_df.plot_animated().get_html5_video()\n\nHTML(animated_html)\n```\n\n### Progress Bars!\n\nGenerating animations can take some time, so enable progress bars by installing [tqdm](https://github.com/tqdm/tqdm) with `pip install tqdm` and using the keyword `enable_progress_bar=True`.\n\nBy default Pandas_Alive will create a `tqdm` progress bar for the number of frames to animate, and update the progres bar after each frame.\n\n``` python\nimport pandas_alive\n\ncovid_df = pandas_alive.load_dataset()\n\ncovid_df.plot_animated(enable_progress_bar=True)\n```\n\nExample of TQDM in action:\n\n![TQDM Example](https://raw.githubusercontent.com/tqdm/tqdm/master/images/tqdm.gif)\n\n## Future Features\n\nA list of future features that may/may not be developed is:\n\n- ~~Geographic charts (currently using OSM export image, potential [geopandas](https://geopandas.org/))~~\n- ~~Loading bar support (potential [tqdm](https://github.com/tqdm/tqdm) or [alive-progress](https://github.com/rsalmei/alive-progress))~~\n- ~~Potentially support writing to GIF in memory with <https://github.com/maxhumber/gif>~~\n- ~~Support custom figures & axes for multiple plots (eg, gridspec)~~\n\n## Inspiration\n\nThe inspiration for this project comes from:\n\n- [bar_chart_race](https://github.com/dexplo/bar_chart_race) by [Ted Petrou](https://github.com/tdpetrou)\n- [Pandas-Bokeh](https://github.com/PatrikHlobil/Pandas-Bokeh) by [Patrik Hlobil](https://github.com/PatrikHlobil)\n\n## Requirements\n\nIf you get an error such as `TypeError: \'MovieWriterRegistry\' object is not an iterator`, this signals there isn\'t a writer library installed on your machine.\n\nThis package utilises the [matplotlib.animation function](https://matplotlib.org/3.2.1/api/animation_api.html), thus requiring a writer library.\n\nEnsure to have one of the supported tooling software installed prior to use!\n\n- [ffmpeg](https://ffmpeg.org/)\n- [ImageMagick](https://imagemagick.org/index.php)\n- See more at <https://matplotlib.org/3.2.1/api/animation_api.html#writer-classes>\n\n> If the output file name has an extension of `.gif`, `pandas_alive` will write this with `PIL` in memory\n\n## Documentation\n\nDocumentation is provided at <https://jackmckew.github.io/pandas_alive/>\n\n## Contributing\n\nPull requests are welcome! Please help to cover more and more chart types!\n\n### Development\n\nTo get started in development, clone a copy of this repository to your PC. This will now enable you to create a Jupyter notebook or a standalone `.py` file, and import `pandas_alive` as a local module. Now you can create new chart types in `pandas_alive/charts.py` or `pandas_alive/geocharts.py` to build to your hearts content!\n\n## [Changelog](CHANGELOG.md)\n',
    'author': 'JackMcKew',
    'author_email': 'jackmckew2@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/JackMcKew/pandas_alive',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6.1,<4.0.0',
}


setup(**setup_kwargs)
