# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['proteingraphml',
 'proteingraphml.Analysis',
 'proteingraphml.DataAdapter',
 'proteingraphml.GraphTools',
 'proteingraphml.MLTools',
 'proteingraphml.MLTools.Data',
 'proteingraphml.MLTools.MetapathFeatures',
 'proteingraphml.MLTools.Models',
 'proteingraphml.MLTools.Procedures',
 'proteingraphml.MLTools.StaticFeatures']

package_data = \
{'': ['*']}

install_requires = \
['PyYAML',
 'XlsxWriter',
 'colorama',
 'matplotlib',
 'mysqlclient',
 'networkx',
 'numpy',
 'openpyxl',
 'pandas',
 'pony',
 'pyreadr',
 'raven[flask]',
 'scikit-learn',
 'xgboost',
 'xlrd']

setup_kwargs = {
    'name': 'proteingraphml',
    'version': '0.1.0',
    'description': 'Knowledge Graphs for Protein Science',
    'long_description': '# ProteinGraphML\n\nThis software is designed to to predict disease-to-protein (protein-coding\ngene) associations, from a biomedical knowledge graph, via machine learning (ML).\nThis codebase abstracts the ML from the domain knowledge and data sources, to\nallow reuse for other applications. The input PostgreSQL relational database is\nconverted to a knowledge graph, then converted to feature vectors by metapath\nmatching, based on an input disease, defining a training set of proteins. Then\nXGBoost is used to generate and optimize a predictive model.\n\n## Table of Contents  \n\n* [Dependencies](#Dependencies)\n* [How to Run Workflow](#Howto)\n   * [Build KG](#HowtoBuildKG)\n   * [Generate Static Features](#HowtoStaticFeatures)\n   * [Prepare Training and Test Sets](#HowtoPrep)\n   * [Generate Metapath Features](#HowtoMetapathFeatures)\n   * [Train ML Model](#HowtoTrainML)\n   * [Predict Using Trained Model](#HowtoPredictML)\n* Also see\n   * [ComparingResults (Python vs R)](doc/ComparingResults.md)\n   * [Visualization](doc/Visualization.md) _(In development.)_\n\n## <a name="Dependencies"/>Dependencies\n\n* R 3.5+\n* R packages: `data.table`, `Matrix`, `RPostgreSQL`\n* Python 3.4+\n* Python packages: `xgboost`, `scikit-learn`, `networkx`, `pandas`, `pony`, `matplotlib`, `xlrd`, `XlsxWriter`\n* PostgreSQL database `metap`.\n   * Edit `DBcreds.yaml` with valid db credentials. Needed throughout workflow.\n* Hardware: Some of the programs need 20+ GB RAM and Xgboost takes several minutes to train and test the data if the number of CPU cores is less than 8. So, a machine with 20+ GB RAM and 8+ CPU core is preferable for running these codes.\n\n## <a name="Howto"/>How to run the Workflow:\n\nThe command-line programs of ProteinGraphML must be executed in the following order.\nHowever, the __BuildKG__ and __StaticFeatures__ are one-time steps, re-useable for\nmultiple ML models. Re-run only required if database updated.\n\n### <a name="HowtoBuildKG"/>Build KG\n\n`BuildKG.py`, from the relational db, generates a knowledge graph,\na `ProteinDiseaseAssociationGraph`, saved as a pickled `networkX` graph. \nVia the adapter and [`pony`](https://docs.ponyorm.org)\nobject-relational model (ORM), nodes and edges are queried from the db to comprise the\ngraph.\n\nCommand line parameters:\n\n* `--o` : Pickled KG file.\n* `--db` : database (olegdb or tcrd) to use to build KG (default: tcrd).\n* `--logfile` : KG log file (optional).\n* `--cyjsfile`: Save KG as CYJS file. (optional).\n* `--graphmlfile` : Save KG as graphML. (optional).\n* `--tsvfile` : Save KG as TSV file. (optional).\n\nExample commands:\n\n```\nBuildKG.py -h\nBuildKG.py --db olegdb --o ProteinDisease_GRAPH.pkl\nBuildKG.py --db olegdb --o ProteinDisease_GRAPH.pkl --logfile ProteinDisease_GRAPH.log --cyjsfile ProteinDisease_GRAPH.cyjs --tsvfile ProteinDisease_GRAPH.tsv\n```\n\n### <a name="HowtoStaticFeatures"/>Static Features\n\n`GenStaticFeatures.py` generates files for static features: lincs, hpa, gtex, and ccle for use by `GenTrainingAndTestFeatures.py`. Static features are _not_ dependent on training set labels, only the database, \nso the same TSV files can be reused for all models, and only needs to be re-run if\nthe database changes.  ___Note: Requires large memory server, approximately 100GB+,\n80GB for this process.___\n\nCommand line parameters:\n\n* `--db` : database (olegdb or tcrd) to use to generate static features (default: tcrd)\n* `--outputdir` : output folder to save TSV files.\n* `--sources` : static features (default: ["gtex", "lincs", "ccle", "hpa"]).\n* `--decimals` : decimal place for the values (default:3)\n```\nGenStaticFeatures.py --db olegdb --source "gtex,hpa" --outputdir static_olegdb  (only \ngtex and hpa)\nGenStaticFeatures.py --db olegdb --outputdir static_olegdb  (for all 4 static features)\n```\n\n### <a name="HowtoPrep"/>Prepare Training and Test Sets\n\n`PrepTrainingAndTestSets.py` generates two files:\n\n1.  A `pickle`ed Python dictionary that\ncontains protein_ids for both class \'True\' and \'False\'. This training set file is needed\nfor running ML for a disease defined by a custom labeled training set,\nrather than a Mammalian Phenotype (MP) term ID. The custome labeled training set may\nreference proteins via `protein_id`s or gene symbols; if gene symbols, this code fetches\nthe corresponding `protein_id` for each symbol from the database. The prepared,\npicked training set uses `protein_id`s.\n1. A `pickle`ed test set of protein_ids, unlabeled, defining the predictions of\ninterest, for testing the trained model.\n\nCommand line parameters:\n\n* `--i` : Input file that contains protein_ids/symbols and labels for a given disease, with extension (csv|txt|xlsx|rds).\n* `--symbol_or_pid` : "symbol" or "pid" (default: symbol).\n* `--use_default_negatives` : Use default negatives, ~3500 genes with known associations but not with query disease. If false, input training set must include negatives.\n* `--db` : database (olegdb or tcrd) (default: tcrd)\n\nIf the file is a spreadsheet, the header should have "Protein_id Label" or "Symbol Label" and the sheet name should be "Sheet1".\nIf the file is a text file, the Protein_id/symbol and\nLabel should be comma-separated. There should not be any header in the text file. \nIf a disease is not present in the graph, use the corresponding RDS file in this program to generate sets of training and predict protein ids. E.g. 104300.rds, PS168600.rds\n\nExample commands:\n\n```\nPrepTrainingAndTestSets.py -h\nPrepTrainingAndTestSets.py --i data/diabetes_pid.txt --symbol_or_pid \'pid\' --db olegdb\nPrepTrainingAndTestSets.py --i data/autophagy.xlsx --db olegdb\nPrepTrainingAndTestSets.py --i data/diabetes.xlsx --use_default_negatives --db olegdb\nPrepTrainingAndTestSets.py --i data/Asthma.rds --db olegdb\n```\n\n### <a name="HowtoMetapathFeatures"/>Metapath Features\n\nTo generate metapath features from the KG, use `GenTrainingAndTestFeatures.py`. From the KG\nand hard coded metapath patterns, plus the positively labeled proteins in the\ntraining set, feature vectors are generated for all training cases and optionally\npredict cases. Normally, any human proteins not in the labeled training set \nwill be in the predict set.  Metapath-based features\nmust be generated for each model (unlike static features), since how metapath \nsemantic patterns match the KG depends on the query disease. Mammalian Phenotype ID can also be used\nwith this program to generate training and predict data sets for ML models.\n\nCommand line parameters:\n\n*  `--disease` : Mammalian Phenotype ID, e.g. MP_0000180 (Diseases without MP_TERM_ID may give error. So, use their \nRDS files to create sets of training and predict protein ids using `PrepTrainingAndTestSets.py`)\n*  `--trainingfile` : pickled training set, e.g. "diabetes.pkl"\n*  `--predictfile` : pickled predict set, e.g. "diabetes_test.pkl"\n*  `--outputdir` : directory where train and test data with features will be saved, e.g. "diabetes_no_lincs"\n*  `--kgfile` : input pickled KG (default: "ProteinDisease_GRAPH.pkl")\n*  `--static_data` : (default: "gtex,lincs,ccle,hpa")\n*  `--static_dir` : directory of static features files: lincs.tsv, hpa.tsv, gtex.tsv, and ccle.tsv\n*  `--db` : database (olegdb or tcrd) (default: tcrd)\n\nExample commands:\n\n```\nGenTrainingAndTestFeatures.py -h\nGenTrainingAndTestFeatures.py --trainingfile data/ATG.pkl --predictfile data/ATG_predict.pkl --outputdir results/ATG --kgfile ProteinDisease_GRAPH.pkl --static_data "gtex" --static_dir static_olegdb --db olegdb\nGenTrainingAndTestFeatures.py --disease MP_0000180 --outputdir results/MP_0000180 --kgfile ProteinDisease_GRAPH.pkl --static_data "gtex,lincs,ccle,hpa" --static_dir static_olegdb --db olegdb\nGenTrainingAndTestFeatures.py --trainingfile data/PS118220.pkl --predictfile data/PS118220_predict.pkl --outputdir results/PS118220 --kgfile ProteinDisease_GRAPH.pkl --static_data "gtex,lincs,ccle,hpa" --static_dir static_olegdb --db olegdb\n```\n\n### <a name="HowtoTrainML"/>Train ML Model\n\n`TrainModelML.py`, from the training set feature vectors, or a training set\nimplicated by specified disease (Mammalian Phenotype ID), \nexecutes the specified ML procedure, training a predictive model, then saved to a\nreusable file (.model).  The procedure `XGBGridSearch` uses\nXGBoost, trains a model with cross-validation and grid-search parameter optimization,\ngenerates a list of important features used by the classification model.\n\nCommand line parameters:\n\n* `PROCEDURE` (positional parameter):\n   * `XGBGridSearch` :  Grid search for optimal XGBoost parameters.\n   * `XGBCrossValPred` :  5-fold cross-validation, one iteration. \n   * `XGBKfoldsRunPred` : 5-fold cross-validation, multiple iterations. In each iteration, data is randomly divided into\n   train and test set (80:20). Model trained on train set is tested on test set. The average, min and max AUC are computed\n   using the classification results of test data. \n* `--trainingfile` : Training set file, produced by `GenTrainingAndTestFeatures.py`.\n* `--resultdir` : directory for output results\n* `--nrounds_for_avg` : number of iterations to compute average AUC, accuracy, and MCC. This is used for procedure `XGBKfoldsRunPred`.  \n* `--rseed` : random seed that XGBoost should use for procedure `XGBGridSearch` (default:1234)\n* `--nthreds` : number of CPU threads for procedure `XGBGridSearch` (default:1).\n* `--xgboost_param_file` : XGBoost configuration parameter file (e.g. XGBparams.txt). This is used for `XGBCrossValPred` and `XGBKfoldsRunPred`. XGBparams.txt created by GridSearch can be used for this parameter. Modify XGBparams.txt if any parameter needs to be changed.\n*  `--db` : database (olegdb or tcrd) (default: tcrd)\n*  `--static_data` : (default: "gtex,lincs,ccle,hpa")\n*  `--static_dir` : directory of static features files: lincs.tsv, hpa.tsv, gtex.tsv, and ccle.tsv\n\nExample commands:\n\n```\nTrainModelML.py -h\nTrainModelML.py XGBGridSearch --trainingfile results/ATG/ATG_TrainingData.pkl --rseed 1234 --nthreads 32 --resultdir results/ATG --db olegdb --static_data "gtex,lincs,ccle,hpa" --static_dir static_olegdb\nTrainModelML.py XGBCrossValPred --trainingfile results/ATG/ATG_TrainingData.pkl --resultdir results/ATG --xgboost_param_file XGBparams.txt --db olegdb --static_data "gtex,lincs,ccle,hpa" --static_dir static_olegdb\nTrainModelML.py XGBKfoldsRunPred --trainingfile results/ATG/ATG_TrainingData.pkl --resultdir results/ATG --xgboost_param_file XGBparams.txt --nrounds_for_avg 5 --db olegdb --static_data "gtex,lincs,ccle,hpa" --static_dir static_olegdb\n```\n\nResults will be saved in the specified --resultsdir. See logs for specific\nsubdirectories and output files, including:\n\n* Saved XGBoost model (.model).\n* Feature importance lists (.tsv, .xlsx).\n\n### <a name="HowtoPredictML"/>Test Trained ML Model\n\n`PredictML.py`, Using the model trained on the training set and KG, predicts the probability of True class for\nproteins. The procedure `XGBPredict` uses the saved XGBoost model, and generates results for predictions on all proteins in the test set. \n\nCommand line parameters:\n\n* `PROCEDURE` (positional parameter):\n   * `XGBPredict` :  load the saved model\n* `--modelfile` : trained model (e.g. results/autophagy_test20191003/XGBCrossVal.model).\n* `--predictfile` : predict data file, produced by `PrepTrainingAndTestSets.py` (e.g.  "diabetesPredictData.pkl")\n* `--resultdir` : directory for output results\n* `--infofile` : protein information file with full path. The file should contain tdl, fam, uniprot data.\n* `--db` : database (olegdb or tcrd) (default: tcrd)\n\nExample commands:\n\n```\nPredictML.py -h\nPredictML.py XGBPredict --predictfile results/ATG/ATG_predict_PredictData.pkl --model results/ATG/XGBCrossValPred.model --resultdir results/ATG --db olegdb --infofile data/plotDT.xlsx\n```\n\nResults will be saved in the specified --resultsdir. See logs for specific\nsubdirectories and output files, including:\n\n* Predictions with probabilities for all proteins (.tsv, .xlsx).\n\n## <a name="Notes"/>Notes\n\n* The code currently assumes that all nodes are unique, that proteins are integer IDs, and the only ints in the graph. \n* New data sources can be supported by adding new Adapter class in `ProteinGraphML/DataAdapter/`.\n* New ML procedures may be added to `ProteinGraphML/MLTools/Procedures/`.\n\nWorkflow overview diagram:\n\n> <img src="doc/MPML_Workflow_Overview.png" height="400">\n',
    'author': 'UNM Translational Informatics',
    'author_email': 'datascience.software@salud.unm.edu',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/unmtransinfo/ProteinGraphML',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8.1,<3.10',
}


setup(**setup_kwargs)
