''':mod:`streamcorpus_pipeline` tagger stage for OpenSextant

.. This software is released under an MIT/X11 open source license.
   Copyright 2014 Diffeo, Inc.

This provides a connector to use the OpenSextantToolbox
https://github.com/OpenSextant/OpenSextantToolbox/ as a tagger in
:mod:`streamcorpus_pipeline`.  Typical configuration looks like:

.. code-block:: yaml

    streamcorpus_pipeline:
      reader: from_local_chunks
      incremental_transforms: [language, guess_media_type, clean_html,
                               title, hyperlink_labels, clean_visible,
                               opensextant]
      batch_transforms: [multi_token_match_align_labels]
      writers: [to_local_chunks]
      opensextant:
        add_geo_selectors: true
        annotate_sentences: false
        path_in_third: opensextant/opensextant-current
      multi_token_match_align_labels:
        annotator_id: author
        tagger_id: opensextant

The ``opensextant`` stage is an incremental transform.  Failures on
individual stream items will result in those stream items remaining in
the stream, but without any tagging.

Note that this stage does *not* run its own aligner, unlike older
tagger stages.  If desired, you must explicitly include an aligner in
``batch_transforms`` to convert document-level
:class:`streamcorpus.Rating` objects to token-level
:class:`streamcorpus.Token` objects.

For all stages that expect a tagger ID, this uses a tagger ID of
``opensextant``.  The stage has no configuration beyond `rest_url`

.. autoclass:: OpenSextantTagger
   :show-inheritance:

'''
from __future__ import absolute_import
import itertools
import json
import logging
import time

import geojson
from geojson import Point, Feature
import requests
from requests.auth import HTTPBasicAuth
from sortedcollection import SortedCollection

from streamcorpus import Tagging, make_stream_time, \
    OffsetType, EntityType, MentionType
from streamcorpus_pipeline.stages import IncrementalTransform
from streamcorpus.ttypes import Selector, Offset


logger = logging.getLogger('streamcorpus_pipeline' + '.' + __name__)


class OpenSextantTagger(IncrementalTransform):
    ''':mod:`streamcorpus_pipeline` tagger stage for OpenSextant.

    This *requires* that the `nltk_tokenizer` transform run before it,
    and it removes the `sentences` entry generated by that transform.
    This modifies the :class:`streamcorpus.Token` objects from that
    transform and puts them back into
    :attr:`streamcorpus.StreamItem.body.sentences`

    This is an incremental transform, and needs to be included in the
    ``incremental_transforms`` list to run within
    :mod:`streamcorpus_pipeline`.

    .. automethod:: __init__
    .. automethod:: process_path
    .. automethod:: shutdown

    '''

    config_name = 'opensextant'
    tagger_id = 'opensextant'

    default_config = {
        'scheme': 'http',
        'network_address': 'localhost:8182',
        'service_path': '/opensextant/extract/',
        'confidence_threshold': 0,
        'timeout': 40,
        'retries': 3,
        'verify_ssl': False,
        'username': None,
        'password': None,
        'cert': None,
        'annotate_sentences': True,
        'add_geo_selectors': True,
    }

    def __init__(self, config, *args, **kwargs):
        '''Create a new tagger.

        `config` should provides ``scheme``, ``network_address``, and
        ``service_path``, which are assembled into a URL for POSTing
        :attr:`~streamcorpus.StreamItem.body.clean_visible` to obtain
        JSON.  The defaults provide this URL:
        `http://localhost:8182/opensextant/extract/general/json`.

        Optionally, `config` can also contain `verify_ssl` with a path
        to a cert.ca-bundle file to verify the remote server's SSL
        cert.  This is useful if the OpenSextant tagger is proxied
        behind an SSL gateway.  By default, `verify_ssl` is False.

        Optionally, `config` can also contain `username` and
        `password` for BasicAuth to access the OpenSextent end point.

        Per the python `requests` documentation, you can also specify
        a local cert to use as client side certificate, as a single
        file (containing the private key and the certificate) or as a
        tuple of both file's path `cert=('cert.crt', 'cert.key')`

        :param dict config: local configuration dictionary

        '''
        super(OpenSextantTagger, self).__init__(config, *args, **kwargs)
        kwargs = {}
        self.rest_url = (
            config.get('scheme', 'http') +
            '://' + config.get('network_address', 'localhost:8182') +
            config.get('service_path', '/opensextant/extract/')
        )
        if config.get('annotate_sentences'):
            # use full NER models from GATE, library upon which
            # opensextant is built: https://gate.ac.uk/projects.html
            self.rest_url += 'general/json'
        else:
            # only use the GEO models from GATE
            self.rest_url += 'geo/json'

        self.verify_ssl = config.get('verify_ssl', False)

        # Session carries connection pools that automatically provide
        # HTTP keep-alive, so we can send many documents over one
        # connection.
        self.session = requests.Session()
        username = config.get('username')
        password = config.get('password')
        if username and password:
            self.session.auth = HTTPBasicAuth(username, password)

        cert = config.get('cert')
        if cert and isinstance(cert, (list, tuple)):
            self.session.cert = tuple(cert)
        elif cert:
            self.session.cert = cert

    def shutdown(self):
        '''Try to stop processing.

        Does nothing, since all of the work is done in-process.

        '''
        pass

    def request_json(self, si):
        # clean_visible will be UTF-8 encoded
        logger.debug('POST %d bytes of clean_visible to %s',
                     len(si.body.clean_visible), self.rest_url)
        headers = {
            'content-encoding': 'UTF-8',
            'content-type': 'text/plain; charset=UTF-8',
        }
        retries = int(self.config.get('retries', 1))
        tries = 0
        while tries < retries:
            tries += 1
            try:
                response = self.session.post(
                    self.rest_url,
                    data=si.body.clean_visible,
                    verify=self.verify_ssl,
                    headers=headers,
                    timeout=int(self.config.get('timeout', 10)),
                )
                break
            except Exception, exc:
                if str(exc).startswith('ReadTimeout'):
                    logger.info('retrying OpenSextant connection: %d of %d',
                                tries, retries)
                    time.sleep(2**tries)
                    continue
                raise
        # save JSON for testing; make file names based on length of
        # clean_visible
        # fname = 'query-%d.json' % len(si.body.clean_visible)
        # fpath = os.path.join(os.path.dirname(__file__), 'tests', fname)
        # open(fpath, 'wb').write(response.content)
        return response

    def get_geo_selectors(self, results):
        '''Given a JSON result from opensextant, create Selectors
        '''
        features = results["annoList"]

        # For each feature, if it is a PLACE, yield a Selector
        for feature in features:
            if(feature['type'] == 'PLACE'):
                lat = feature['features']['place']['latitude']
                lng = feature['features']['place']['longitude']
                raw = feature['features']['place']['placeName']
                pid = feature['features']['place']['placeID']
                conf = feature['features']['place']['nameBias']
                span = (feature['start'], feature['end'])

                point = Point((lng, lat))
                properties = {'name': raw}
                feature = Feature(geometry=point,
                                  properties=properties, id=pid)

                logger.info('emitting Selector(name=%r, ... conf=%f',
                            raw, conf)

                # Set the offset
                o = Offset(
                    type=OffsetType.CHARS,
                    content_form='clean_visible',
                    first=span[0], length=span[1] - span[0])

                yield Selector(
                    # selector_type is allowed to be any string, but
                    # downstream code depends on knowing what it is
                    selector_type='GEOJSON',
                    raw_selector=raw.encode('utf-8'),
                    canonical_selector=geojson.dumps(feature),
                    offsets={OffsetType.CHARS: o})

    def filter(self, results):
        '''Boosting precision will naturally degrade recall.  The two ways to
        get better precision. (Notes adapted from dlutz comments.)

        1. Use the small gazetteer. It contains a small subset of the
        big gazetteer but contains the most common place names
        (countries, capitals provinces ...)  It is also much faster.

        2. Filter by the placeconfidence score. The PlaceCandidate
        object attached to each Place annotation has a
        getPlaceConfidenceScore() method which will return a score
        between -1 to +1.  See the geotagger example in
        org.opensextant.examples.GeotaggerExample around line 155
        where the results are being written out. The comment shows how
        to get the PlaceCandidate object.  If you are using the REST
        interfaces, there is an element named "nameBias" in the JSON
        returned for each place which is the same score as described
        above.  The model currently generates weakly useful confidence
        scores in that filtering out the low scores does increase the
        precision a bit but at a large cost in recall.

        '''
        confidence_threshold = self.config.get('confidence_threshold', 0)

        def confidence_filter(result):
            if result['type'] != 'PLACE':
                return True
            conf = result['features']['place']['nameBias']
            return conf >= confidence_threshold

        results['annoList'] = filter(confidence_filter, results['annoList'])
        return results

    def process_item(self, si, context=None):
        '''Run OpenSextant over a single stream item.

        This ignores the `context`, and always returns the input
        stream item `si`.  Its sole action is to add a ``opensextant``
        value to the tagger-keyed fields in `si.body`, provided that
        `si` in fact has a
        :attr:`~streamcorpus.ContentItem.clean_visible` part.

        :param si: stream item to process
        :paramtype si: :class:`streamcorpus.StreamItem`
        :param dict context: additional shared context data
        :return: `si`

        '''
        if si.body and si.body.clean_visible:
            response = self.request_json(si)

            results = json.loads(response.content)

            results = self.filter(results)

            # remove a Tagging entry from nltk_tokenizer
            # si.body.taggings.pop('nltk_tokenizer')
            tagging = Tagging(
                tagger_id=self.tagger_id,
                tagger_version='2.1',
                generation_time=make_stream_time(time.time()),
                raw_tagging=response.content
            )
            si.body.taggings[self.tagger_id] = tagging

            if self.config.get('annotate_sentences') is True:
                self.annotate_sentences(si, results)

            if self.config.get('add_geo_selectors') is True:
                selectors = list(self.get_geo_selectors(results))
                logger.info('opensextant added %d selectors', len(selectors))
                si.body.selectors[self.tagger_id] = selectors

            # si.body.relations[self.tagger_id] = make_relations(result)
            # si.body.attributes[self.tagger_id] = make_attributes(result)

        return si

    def annotate_sentences(self, si, result):
        logger.info(json.dumps(result, indent=4, sort_keys=4))

        sentences = si.body.sentences.pop('nltk_tokenizer')
        si.body.sentences[self.tagger_id] = sentences

        toks = SortedCollection(
            itertools.chain(*[sent.tokens for sent in sentences]),
            key=lambda tok: tok.offsets[OffsetType.CHARS].first
            )

        cv = si.body.clean_visible.decode('utf8')
        mention_id = 0
        for anno in result.get('annoList', []):
            # if not anno.get('features', {}).get('isEntity'):
            #     logger.debug('skipping isEntity=False: %s',
            #                  json.dumps(anno, indent=4, sort_keys=True))
            #     continue
            start = anno['start']
            end = anno['end']
            if not cv[start:end] == anno['matchText']:
                # these appear to typically be spaces collapsed by OpenSextant
                pre = 30
                post = 30
                logger.debug(
                    'alignment failure:\n\t%s\n\t%s%s%s',
                    si.body.clean_visible.decode('utf8')[start-pre:end+post],
                    ' ' * pre,
                    anno['matchText'],
                    ' ' * post)

            for tok in toks.find_range(start, end):
                fhierarchy = anno['features']['hierarchy']
                fh_parts = fhierarchy.split('.')
                if entity_types.get(fhierarchy):
                    e_type, m_type = entity_types[fhierarchy]
                elif entity_types.get(fh_parts[0]):
                    e_type, m_type = entity_types[fh_parts[0]]
                else:
                    e_type, m_type = None, None

                if e_type is not None:
                    tok.entity_type = e_type
                    tok.mention_type = m_type
                    tok.mention_id = mention_id
                    # too bad no coref chains, so nominals are not connected
                    # to names:
                    tok.equiv_id = mention_id

            mention_id += 1


entity_types = {
    # most events are unnamed, so default to NOM
    'Action': (EntityType.EVENT, MentionType.NOM),

    'Attribute.attribute.measurableCharacteristic': None,
    'Attribute.weight': None,

    # descriptive attributes --> nominatives
    'Geo.area': (EntityType.LOC, MentionType.NOM),
    'Geo.distance': (EntityType.LOC, MentionType.NOM),
    'Geo.weather': (EntityType.LOC, MentionType.NOM),

    # most GEO area named locations
    'Geo.featureType.AdminRegion': (EntityType.LOC, MentionType.NAME),
    'Geo.featureType.Area': (EntityType.LOC, MentionType.NAME),
    'Geo.featureType.Hydro': (EntityType.LOC, MentionType.NAME),
    'Geo.featureType.Hypso': (EntityType.LOC, MentionType.NAME),
    'Geo.featureType.Misc': (EntityType.LOC, MentionType.NAME),
    'Geo.featureType.PopulatedPlace': (EntityType.LOC, MentionType.NAME),
    'Geo.featureType.Street': (EntityType.LOC, MentionType.NAME),
    'Geo.featureType.Undersea': (EntityType.LOC, MentionType.NAME),
    'Geo.featureType.Vegetation': (EntityType.LOC, MentionType.NAME),

    'Geo.place.geocoordinate': (EntityType.LOC, MentionType.NAME),
    'Geo.place.namedPlace': (EntityType.LOC, MentionType.NAME),

    # these are usually named facilities
    'Geo.featureType.SpotFeature': (EntityType.FAC, MentionType.NAME),
    'Geo.facilityComponents': (EntityType.FAC, MentionType.NAME),

    'Idea': None,
    'Information': None,
    'Object': None,

    'Organization': (EntityType.ORG, MentionType.NAME),
    'Person': (EntityType.PER, MentionType.NAME),

    # would be nice to map these into relations
    'Person.attitude.emotion': None,
    'Person.attitude.emotion.negativeEmotion': None,
    'Person.attitude.emotion.positiveEmotion': None,
    'Person.bodyPart': None,
    'Person.ethnicity': None,
    'Person.health': None,
    'Person.health.disease': None,
    'Person.health.injury': None,
    'Person.jobOrRole': None,
    'Person.language': None,

    'Person.name.personName': (EntityType.PER, MentionType.NAME),

    'Person.name.title.corporateTitle': (EntityType.PER, MentionType.NOM),
    'Person.name.title.governmentTitle': (EntityType.PER, MentionType.NOM),
    'Person.name.title.hereditaryTitle': (EntityType.PER, MentionType.NOM),
    'Person.name.title.militaryTitle': (EntityType.PER, MentionType.NOM),
    'Person.name.title.personalTitle': (EntityType.PER, MentionType.NOM),
    'Person.name.title.religiousTitle': (EntityType.PER, MentionType.NOM),

    'Person.relative': None,

    'Substance': None,
    'Time': None,
}


# this list of hierarchical entity types is copied from
# https://github.com/OpenSextant/OpenSextantToolbox/blob/master/LanguageResources/docs/
entity_hierarchy = {
    'Action.event': None,
    'Action.event.crime': None,
    'Action.event.disaster': None,
    'Action.event.legalEvent': None,
    'Action.event.meetingEvent': None,
    'Action.event.militaryEvent': None,
    'Action.event.movement': None,
    'Action.event.politicalEvent': None,
    'Action.event.socialEvent': None,
    'Action.event.violentEvent': None,
    'Attribute.attribute.measurableCharacteristic': None,
    'Attribute.weight': None,
    'Geo.area': None,
    'Geo.distance': None,
    'Geo.facilityComponents': None,
    'Geo.featureType.AdminRegion': None,
    'Geo.featureType.Area': None,
    'Geo.featureType.Hydro': None,
    'Geo.featureType.Hypso': None,
    'Geo.featureType.Misc': None,
    'Geo.featureType.PopulatedPlace': None,
    'Geo.featureType.SpotFeature': None,
    'Geo.featureType.Street': None,
    'Geo.featureType.Undersea': None,
    'Geo.featureType.Vegetation': None,
    'Geo.place.geocoordinate': None,
    'Geo.place.namedPlace': None,
    'Geo.weather': None,
    'Idea.fieldOfStudy': None,
    'Idea.idea': None,
    'Idea.ideology.politicalIdealogy': None,
    'Idea.ideology.socialIdealogy': None,
    'Information': None,
    'Information.identifier': None,
    'Information.identifier.documentTitle': None,
    'Information.identifier.MACAddress': None,
    'Information.identifier.telephoneNumber': None,
    'Information.informationArtifact': None,
    'Information.software': None,
    'Information.web.emailAddress': None,
    'Information.web.IPAddress': None,
    'Information.web.url': None,
    'Information.web.webSite': None,
    'Object': None,
    'Object.animal': None,
    'Object.clothing': None,
    'Object.container': None,
    'Object.debris': None,
    'Object.electronics': None,
    'Object.equipment': None,
    'Object.equipment.constructionEquipment': None,
    'Object.equipment.tool': None,
    'Object.finance.financialInstrument': None,
    'Object.finance.money': None,
    'Object.finance.money': None,
    'Object.food': None,
    'Object.vehicle': None,
    'Object.vehicle.aircraft': None,
    'Object.vehicle.aircraft.combatAircraft': None,
    'Object.vehicle.aircraft.combatSupportAircraft': None,
    'Object.vehicle.aircraft.helicopter': None,
    'Object.vehicle.emergencyVehicle': None,
    'Object.vehicle.militaryVehicle': None,
    'Object.vehicle.militaryVehicle.armoredVehicle': None,
    'Object.vehicle.ship': None,
    'Object.vehicle.spacecraft': None,
    'Object.vehicle.submarine': None,
    'Object.weapon': None,
    'Object.weapon.explosive': None,
    'Object.weapon.firearm': None,
    'Object.weapon.weaponOfMassDestruction': None,
    'Organization': None,
    'Organization.corporateOrganization': None,
    'Organization.criminalOrganization': None,
    'Organization.governmentOrganization': None,
    'Organization.governmentOrganization.politicalParty': None,
    'Organization.governmentOrganization.USGovernmentOrganization': None,
    'Organization.informalOrganization': None,
    'Organization.internationalOrganization': None,
    'Organization.media.newspaper': None,
    'Organization.militantGroup': None,
    'Organization.militaryOrganization': None,
    'Organization.religion': None,
    'Organization.terroristGroup': None,
    'Person': None,
    'Person.attitude.emotion': None,
    'Person.attitude.emotion.negativeEmotion': None,
    'Person.attitude.emotion.positiveEmotion': None,
    'Person.bodyPart': None,
    'Person.ethnicity': None,
    'Person.health': None,
    'Person.health.disease': None,
    'Person.health.injury': None,
    'Person.jobOrRole': None,
    'Person.language': None,
    'Person.name.personName': None,
    'Person.name.personName': None,
    'Person.name.title.corporateTitle': None,
    'Person.name.title.governmentTitle': None,
    'Person.name.title.hereditaryTitle': None,
    'Person.name.title.militaryTitle': None,
    'Person.name.title.personalTitle': None,
    'Person.name.title.religiousTitle': None,
    'Person.relative': None,
    'Substance': None,
    'Substance.chemical': None,
    'Substance.drug': None,
    'Substance.material': None,
    'Time.date': None,
    'Time.date': None,
    'Time.dayOfTheWeek': None,
    'Time.holiday': None,
    'Time.lengthOfTime': None,
    'Time.month': None,
    'Time.season': None,
    'Time.time': None,
    'Time.timePhrase': None,
}
