# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/core.ipynb.

# %% ../nbs/core.ipynb 3
from __future__ import annotations
import numpy as np
from typing import List, Tuple, Dict, Any, Optional, Iterable, Sequence, Iterator
import jax
from jax import vmap, grad, jit, numpy as jnp
from jax.random import PRNGKey
from abc import ABC
from dataclasses import dataclass
import collections
import warnings

# %% auto 0
__all__ = ['PRNGSequence', 'Dataset', 'ArrayDataset', 'TorchDataset', 'HFDataset', 'BaseDataLoader', 'DataLoaderJax',
           'DataLoaderPytorch', 'DataloaderBackends', 'DataLoader']

# %% ../nbs/core.ipynb 4
try: 
    import torch.utils.data as torch_data
    import torch
except ModuleNotFoundError: 
    torch_data = None
    torch = None

try: import haiku as hk 
except ModuleNotFoundError: hk = None

try: import datasets as hf_datasets
except ModuleNotFoundError: hf_datasets = None

# %% ../nbs/core.ipynb 5
@dataclass
class Config:
    rng_reserve_size: int
    global_seed: int

    @classmethod
    def default(cls) -> Config:
        return cls(rng_reserve_size=1, global_seed=42)

# %% ../nbs/core.ipynb 6
main_config = Config.default()

# %% ../nbs/core.ipynb 7
def get_config() -> Config:
    return main_config

# %% ../nbs/core.ipynb 8
class PRNGSequence(Iterator[PRNGKey]):
    """An Interator of Jax PRNGKey (minimal version of haiku.PRNGSequence)."""

    def __init__(self, seed: int):
        self._key = jax.random.PRNGKey(seed)
        self._subkeys = collections.deque()

    def reserve(self, num):
        """Splits additional ``num`` keys for later use."""
        if num > 0:
            new_keys = tuple(jax.random.split(self._key, num + 1))
            self._key = new_keys[0]
            self._subkeys.extend(new_keys[1:])
            
    def __next__(self):
        if not self._subkeys:
            self.reserve(get_config().rng_reserve_size)
        return self._subkeys.popleft()

# %% ../nbs/core.ipynb 9
def _check_pytorch_installed():
    if torch_data is None:
        raise ModuleNotFoundError("`pytorch` library needs to be installed. "
            "Try `pip install torch`. Please refer to pytorch documentation for details: "
            "https://pytorch.org/get-started/.")
    
def _check_hf_installed():
    if hf_datasets is None:
        raise ModuleNotFoundError("`datasets` library needs to be installed. "
            "Try `pip install datasets`. Please refer to huggingface documentation for details: "
            "https://huggingface.co/docs/datasets/installation.html.")

# %% ../nbs/core.ipynb 11
class Dataset:
    """A pytorch-like Dataset class."""

    def __len__(self):
        raise NotImplementedError

    def __getitem__(self, index):
        raise NotImplementedError

# %% ../nbs/core.ipynb 12
class ArrayDataset(Dataset):
    """Dataset wrapping numpy arrays."""

    def __init__(
        self, 
        *arrays: jnp.DeviceArray # Numpy array with same first dimension
    ):
        assert all(arrays[0].shape[0] == arr.shape[0] for arr in arrays), \
            "All arrays must have the same dimension."
        self.arrays = arrays

    def __len__(self):
        return self.arrays[0].shape[0]

    def __getitem__(self, index):
        return tuple(arr[index] for arr in self.arrays)


# %% ../nbs/core.ipynb 17
def _has_tensor(batch) -> bool:
    if isinstance(batch[0], torch.Tensor):
        return True
    elif isinstance(batch[0], (tuple, list)):
        transposed = zip(*batch)
        return any([_has_tensor(samples) for samples in transposed])
    else:
        return False

# %% ../nbs/core.ipynb 18
class TorchDataset(Dataset):
    """(Deprecated) A Dataset class that wraps a pytorch Dataset."""
    
    def __init__(
        self, 
        dataset: torch_data.Dataset # Pytorch Dataset
    ):
        _check_pytorch_installed()
        if not isinstance(dataset, torch_data.Dataset):
            raise TypeError(f"`dataset` must be a torch Dataset, but got {type(dataset)}")
        # Give a warning if the dataset is not in numpy format
        if _has_tensor(dataset[0]):
            warnings.warn("The dataset contains `torch.Tensor`. "
                "Please make sure the dataset is in numpy format.")
        self._ds = dataset

    def __len__(self):
        return len(self._ds)

    def __getitem__(self, index):
        return self._ds[index]

# %% ../nbs/core.ipynb 28
class HFDataset(Dataset):
    """(Deprecated) A Dataset class that wraps a huggingface Dataset."""
    
    def __init__(
        self, 
        dataset: hf_datasets.Dataset # Huggingface Dataset
    ):
        _check_hf_installed()
        # if not isinstance(dataset, hf_datasets.Dataset):
        #     raise TypeError(f"`dataset` must be a huggingface Dataset, "
        #                     f"but got {type(dataset)}")
        # Ensure the dataset is in jax format
        self._ds = dataset.with_format("jax")

    def __len__(self):
        return len(self._ds)

    def __getitem__(self, index):
        return self._ds[index]

# %% ../nbs/core.ipynb 35
class BaseDataLoader:
    """Dataloader Interface"""
    def __init__(
        self, 
        dataset: Dataset, 
        batch_size: int = 1,  # batch size
        shuffle: bool = False,  # if true, dataloader shuffles before sampling each batch
        drop_last: bool = False,
        **kwargs
    ):
        pass

    def __len__(self):
        raise NotImplementedError
    
    def __next__(self):
        raise NotImplementedError
    
    def __iter__(self):
        raise NotImplementedError

# %% ../nbs/core.ipynb 37
class DataLoaderJax(BaseDataLoader):
    """Dataloder in Vanilla Jax"""

    def __init__(
        self, 
        dataset,
        batch_size: int = 1,  # batch size
        shuffle: bool = False,  # if true, dataloader shuffles before sampling each batch
        drop_last: bool = False, # drop last batches or not
        **kwargs
    ):
        self.dataset = dataset
        self.batch_size = batch_size
        self.shuffle = shuffle
        self.drop_last = drop_last

        self.keys = PRNGSequence(seed=get_config().global_seed) \
            if hk is None else hk.PRNGSequence(get_config().global_seed)
        self.data_len = len(dataset)  # Length of the dataset
        self.indices = jnp.arange(self.data_len) # available indices in the dataset
        self.pose = 0  # record the current position in the dataset
        self._shuffle()

    def _shuffle(self):
        if self.shuffle:
            self.indices = jax.random.permutation(next(self.keys), self.indices)
        
    def _stop_iteration(self):
        self.pose = 0
        self._shuffle()
        raise StopIteration

    def __len__(self):
        if self.drop_last:
            batches = len(self.dataset) // self.batch_size  # get the floor of division
        else:
            batches = -(len(self.dataset) // -self.batch_size)  # get the ceil of division
        return batches

    def __next__(self):
        if self.pose + self.batch_size <= self.data_len:
            batch_indices = self.indices[self.pose: self.pose + self.batch_size]
            batch_data = self.dataset[batch_indices]
            self.pose += self.batch_size
            return batch_data
        elif self.pose < self.data_len and not self.drop_last:
            batch_indices = self.indices[self.pose:]
            batch_data = self.dataset[batch_indices]
            self.pose += self.batch_size
            return batch_data
        else:
            self._stop_iteration()

    def __iter__(self):
        return self

# %% ../nbs/core.ipynb 41
# adapted from https://jax.readthedocs.io/en/latest/notebooks/Neural_Network_and_Data_Loading.html#data-loading-with-pytorch
def _numpy_collate(batch):
    if isinstance(batch[0], (np.ndarray, jax.Array)):
        return np.stack(batch)
    elif isinstance(batch[0], (tuple, list)):
        transposed = zip(*batch)
        return [_numpy_collate(samples) for samples in transposed]
    elif isinstance(batch[0], dict):
        return {key: _numpy_collate([d[key] for d in batch]) for key in batch[0]}
    else:
        return np.array(batch)

def _convert_dataset_pytorch(dataset: Dataset):
    class DatasetPytorch(torch_data.Dataset):
        def __init__(self, dataset: Dataset): self.dataset = dataset
        def __len__(self): return len(self.dataset)
        def __getitem__(self, idx): return self.dataset[idx]
    
    return DatasetPytorch(dataset)

# %% ../nbs/core.ipynb 42
class DataLoaderPytorch(BaseDataLoader):
    """Pytorch Dataloader"""
    def __init__(
        self, 
        dataset,
        batch_size: int = 1,  # Batch size
        shuffle: bool = False,  # If true, dataloader shuffles before sampling each batch
        drop_last: bool = False, # Drop last batch or not
        **kwargs
    ):
        super().__init__(dataset, batch_size, shuffle, drop_last)
        _check_pytorch_installed()
        
        # dataset = _convert_dataset_pytorch(dataset)
        self.dataloader = torch_data.DataLoader(
            dataset, 
            batch_size=batch_size, 
            shuffle=shuffle, 
            drop_last=drop_last,
            collate_fn=_numpy_collate,
            **kwargs
        ) 

    def __len__(self):
        return len(self.dataloader)

    def __next__(self):
        return next(self.dataloader)

    def __iter__(self):
        return self.dataloader.__iter__()

# %% ../nbs/core.ipynb 45
def _is_hf_dataset(dataset):
    return hf_datasets and (
        isinstance(dataset, hf_datasets.Dataset) 
        or isinstance(dataset, hf_datasets.DatasetDict)
    )

def _dispatch_dataset(
    dataset, # Dataset or Pytorch Dataset or HuggingFace Dataset
) -> Dataset:
    if isinstance(dataset, Dataset):
        return dataset
    elif torch_data and isinstance(dataset, torch_data.Dataset):
        # return TorchDataset(dataset)
        # Give a warning if the dataset is not in numpy format
        if _has_tensor(dataset[0]):
            warnings.warn("The dataset contains `torch.Tensor`. "
                "Please make sure the dataset is in numpy format.")
        return dataset
    elif _is_hf_dataset(dataset):
        # return HFDataset(dataset)
        return dataset.with_format("jax")
    else:
        raise ValueError(f"dataset must be one of `jax_loader.core.Dataset`, "
                         "`torch.utils.data.Dataset`, `datasets.Dataset`, "
                         f"but got {type(dataset)}")

# %% ../nbs/core.ipynb 46
@dataclass(frozen=True)
class DataloaderBackends:
    jax: BaseDataLoader = DataLoaderJax
    pytorch: BaseDataLoader = DataLoaderPytorch
    tensorflow: BaseDataLoader = None
    merlin: BaseDataLoader = None

    __all__ = dict(
        jax=jax, pytorch=pytorch, tensorflow=tensorflow, merlin=merlin
    )

    def __getitem__(self, key):
        return self.__all__[key]

    @property
    def supported(self) -> List[str]:
        return [
            backend for backend, dl_cls in self.__all__.items() if dl_cls is not None
        ]

# %% ../nbs/core.ipynb 47
def _dispatch_dataloader(
    backend: str # dataloader backend
) -> BaseDataLoader:
    """Return Dataloader class based on given `backend`"""
    backends = DataloaderBackends()
    if not backend in backends.supported:
        raise ValueError(f"backend=`{backend}` is either an invalid backend or not supported yet. "
            f"Should be one of {backends.supported}.")
    
    dl_cls = backends[backend]
    return dl_cls


def _dispatch_dataset_and_backend(
    dataset, # Dataset or Pytorch Dataset or HuggingFace Dataset
    backend: str # dataloader backend
) -> Tuple[Dataset, BaseDataLoader]:
    """Return Dataset and Dataloader class based on given `dataset` and `backend`"""

    if isinstance(dataset, torch_data.Dataset) and backend != "pytorch":
        raise ValueError(f"dataset (type={type(dataset)}) is a pytorch dataset, "
                         "which is only supported by 'pytorch' backend."
                         f"However, we got `backend={backend}`, which is not 'pytorch'.")

    dataset = _dispatch_dataset(dataset)
    
    dl_cls = _dispatch_dataloader(backend)
    return dataset, dl_cls
        

# %% ../nbs/core.ipynb 49
class DataLoader:
    """Main Dataloader class to load Numpy data batches"""

    def __init__(
        self,
        dataset, # Dataset or Pytorch Dataset or HuggingFace Dataset
        backend: str, # Dataloader backend
        batch_size: int = 1,  # batch size
        shuffle: bool = False,  # if true, dataloader shuffles before sampling each batch
        drop_last: bool = False, # drop last batches or not
        **kwargs
    ):
        dataset, dl_cls = _dispatch_dataset_and_backend(dataset, backend)
        self.dataloader = dl_cls(
            dataset=dataset, 
            batch_size=batch_size, 
            shuffle=shuffle, 
            drop_last=drop_last,
            **kwargs
        )

    def __len__(self):
        return len(self.dataloader)

    def __next__(self):
        return next(self.dataloader)

    def __iter__(self):
        return self.dataloader.__iter__()
